/**
 ** mgrxcolr.h ---- some useful color definitions
 **
 ** Copyright (C) 2019,2022 Mariano Alvarez Fernandez
 ** [e-mail: malfer at telefonica.net]
 **
 ** This file is part of the GRX graphics library.
 **
 ** The GRX graphics library is free software; you can redistribute it
 ** and/or modify it under some conditions; see the "copying.grx" file
 ** for details.
 **
 ** This library is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 **
 **/

#ifndef __GRCOLORS_H_INCLUDED__
#define __GRCOLORS_H_INCLUDED__

#ifndef __MGRX_H_INCLUDED__
#include <mgrx.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

typedef struct {
    char *name;
    long rgbcolor;
    GrColor color;
} GrColorTableDef;

//
// EGA colors
//

#define NUM_EGA_COLORS 16

// RGB color definitions, valid to init vars,
// but them you need GrAllocColor2(EGAR_xxx)

#define EGAR_BLACK          0x000000
#define EGAR_BLUE           0x0000AA
#define EGAR_GREEN          0x00AA00
#define EGAR_CYAN           0x00AAAA
#define EGAR_RED            0xAA0000
#define EGAR_MAGENTA        0xAA00AA
#define EGAR_BROWN          0xAA5500
#define EGAR_LIGHTGRAY      0xAAAAAA
#define EGAR_DARKGRAY       0x555555
#define EGAR_LIGHTBLUE      0x5555FF
#define EGAR_LIGHTGREEN     0x55FF55
#define EGAR_LIGHTCYAN      0x55FFFF
#define EGAR_LIGHTRED       0xFF5555
#define EGAR_LIGHTMAGENTA   0xFF55FF
#define EGAR_YELLOW         0xFFFF55
#define EGAR_WHITE          0xFFFFFF

// a convenient table of WEB colors

extern GrColorTableDef GrEgaColorTable[];

int GrGenEgaColorTable(void);

// GrColors you can use after calling GrGenEgaColorTable,
// but they can not be used to init vars

#define EGAC_BLACK          GrEgaColorTable[0].color
#define EGAC_BLUE           GrEgaColorTable[1].color
#define EGAC_GREEN          GrEgaColorTable[2].color
#define EGAC_CYAN           GrEgaColorTable[3].color
#define EGAC_RED            GrEgaColorTable[4].color
#define EGAC_MAGENTA        GrEgaColorTable[5].color
#define EGAC_BROWN          GrEgaColorTable[6].color
#define EGAC_LIGHTGRAY      GrEgaColorTable[7].color
#define EGAC_DARKGRAY       GrEgaColorTable[8].color
#define EGAC_LIGHTBLUE      GrEgaColorTable[9].color
#define EGAC_LIGHTGREEN     GrEgaColorTable[10].color
#define EGAC_LIGHTCYAN      GrEgaColorTable[11].color
#define EGAC_LIGHTRED       GrEgaColorTable[12].color
#define EGAC_LIGHTMAGENTA   GrEgaColorTable[13].color
#define EGAC_YELLOW         GrEgaColorTable[14].color
#define EGAC_WHITE          GrEgaColorTable[15].color

// Indexes to Ega table, valid to init vars,
// but after calling GrGenEgaColorTable you must use
// GrEgaColorTable[EGAI_xxx].color or the macro EgaColor

#define EGAI_BLACK          0
#define EGAI_BLUE           1
#define EGAI_GREEN          2
#define EGAI_CYAN           3
#define EGAI_RED            4
#define EGAI_MAGENTA        5
#define EGAI_BROWN          6
#define EGAI_LIGHTGRAY      7
#define EGAI_DARKGRAY       8
#define EGAI_LIGHTBLUE      9
#define EGAI_LIGHTGREEN     10
#define EGAI_LIGHTCYAN      11
#define EGAI_LIGHTRED       12
#define EGAI_LIGHTMAGENTA   13
#define EGAI_YELLOW         14
#define EGAI_WHITE          15

#define EgaColor(i) (GrEgaColorTable[i].color)

//
// WEB colors, as defined by CSS and SVG specifications
//

#define NUM_WEB_COLORS 138

// RGB color definitions, valid to init vars,
// but them you need GrAllocColor2(WEBR_xxx)

#define WEBR_ALICEBLUE              0xF0F8FF 
#define WEBR_ANTIQUEWHITE           0xFAEBD7 
#define WEBR_AQUAMARINE             0x7FFFD4 
#define WEBR_AZURE                  0xF0FFFF 
#define WEBR_BEIGE                  0xF5F5DC 
#define WEBR_BISQUE                 0xFFE4C4 
#define WEBR_BLACK                  0x000000 
#define WEBR_BLANCHEDALMOND         0xFFEBCD 
#define WEBR_BLUE                   0x0000FF 
#define WEBR_BLUEVIOLET             0x8A2BE2 
#define WEBR_BROWN                  0xA52A2A 
#define WEBR_BURLYWOOD              0xDEB887 
#define WEBR_CADETBLUE              0x5F9EA0 
#define WEBR_CHARTREUSE             0x7FFF00 
#define WEBR_CHOCOLATE              0xD2691E 
#define WEBR_CORAL                  0xFF7F50 
#define WEBR_CORNFLOWERBLUE         0x6495ED 
#define WEBR_CORNSILK               0xFFF8DC 
#define WEBR_CRIMSON                0xDC143C 
#define WEBR_CYAN                   0x00FFFF 
#define WEBR_DARKBLUE               0x00008B 
#define WEBR_DARKCYAN               0x008B8B 
#define WEBR_DARKGOLDENROD          0xB8860B 
#define WEBR_DARKGRAY               0xA9A9A9 
#define WEBR_DARKGREEN              0x006400 
#define WEBR_DARKKHAKI              0xBDB76B 
#define WEBR_DARKMAGENTA            0x8B008B 
#define WEBR_DARKOLIVEGREEN         0x556B2F 
#define WEBR_DARKORANGE             0xFF8C00 
#define WEBR_DARKORCHID             0x9932CC 
#define WEBR_DARKRED                0x8B0000 
#define WEBR_DARKSALMON             0xE9967A 
#define WEBR_DARKSEAGREEN           0x8FBC8F 
#define WEBR_DARKSLATEBLUE          0x483D8B 
#define WEBR_DARKSLATEGRAY          0x2F4F4F 
#define WEBR_DARKTURQUOISE          0x00CED1 
#define WEBR_DARKVIOLET             0x9400D3 
#define WEBR_DEEPPINK               0xFF1493 
#define WEBR_DEEPSKYBLUE            0x00BFFF 
#define WEBR_DIMGRAY                0x696969 
#define WEBR_DODGERBLUE             0x1E90FF 
#define WEBR_FIREBRICK              0xB22222 
#define WEBR_FLORALWHITE            0xFFFAF0 
#define WEBR_FORESTGREEN            0x228B22 
#define WEBR_GAINSBORO              0xDCDCDC 
#define WEBR_GHOSTWHITE             0xF8F8FF 
#define WEBR_GOLD                   0xFFD700 
#define WEBR_GOLDENROD              0xDAA520 
#define WEBR_GRAY                   0x808080 
#define WEBR_GREEN                  0x008000 
#define WEBR_GREENYELLOW            0xADFF2F 
#define WEBR_HONEYDEW               0xF0FFF0 
#define WEBR_HOTPINK                0xFF69B4 
#define WEBR_INDIANRED              0xCD5C5C 
#define WEBR_INDIGO                 0x4B0082 
#define WEBR_IVORY                  0xFFFFF0 
#define WEBR_KHAKI                  0xF0E68C 
#define WEBR_LAVENDER               0xE6E6FA 
#define WEBR_LAVENDERBLUSH          0xFFF0F5 
#define WEBR_LAWNGREEN              0x7CFC00 
#define WEBR_LEMONCHIFFON           0xFFFACD 
#define WEBR_LIGHTBLUE              0xADD8E6 
#define WEBR_LIGHTCORAL             0xF08080 
#define WEBR_LIGHTCYAN              0xE0FFFF 
#define WEBR_LIGHTGOLDENRODYELLOW   0xFAFAD2 
#define WEBR_LIGHTGRAY              0xD3D3D3 
#define WEBR_LIGHTGREEN             0x90EE90 
#define WEBR_LIGHTPINK              0xFFB6C1 
#define WEBR_LIGHTSALMON            0xFFA07A 
#define WEBR_LIGHTSEAGREEN          0x20B2AA 
#define WEBR_LIGHTSKYBLUE           0x87CEFA 
#define WEBR_LIGHTSLATEGRAY         0x778899 
#define WEBR_LIGHTSTEELBLUE         0xB0C4DE 
#define WEBR_LIGHTYELLOW            0xFFFFE0 
#define WEBR_LIME                   0x00FF00 
#define WEBR_LIMEGREEN              0x32CD32 
#define WEBR_LINEN                  0xFAF0E6 
#define WEBR_MAGENTA                0xFF00FF 
#define WEBR_MAROON                 0x800000 
#define WEBR_MEDIUMAQUAMARINE       0x66CDAA 
#define WEBR_MEDIUMBLUE             0x0000CD 
#define WEBR_MEDIUMORCHID           0xBA55D3 
#define WEBR_MEDIUMPURPLE           0x9370DB 
#define WEBR_MEDIUMSEAGREEN         0x3CB371 
#define WEBR_MEDIUMSLATEBLUE        0x7B68EE 
#define WEBR_MEDIUMSPRINGGREEN      0x00FA9A 
#define WEBR_MEDIUMTURQUOISE        0x48D1CC 
#define WEBR_MEDIUMVIOLETRED        0xC71585 
#define WEBR_MIDNIGHTBLUE           0x191970 
#define WEBR_MINTCREAM              0xF5FFFA 
#define WEBR_MISTYROSE              0xFFE4E1 
#define WEBR_MOCCASIN               0xFFE4B5 
#define WEBR_NAVAJOWHITE            0xFFDEAD 
#define WEBR_NAVY                   0x000080 
#define WEBR_OLDLACE                0xFDF5E6 
#define WEBR_OLIVE                  0x808000 
#define WEBR_OLIVEDRAB              0x6B8E23 
#define WEBR_ORANGE                 0xFFA500 
#define WEBR_ORANGERED              0xFF4500 
#define WEBR_ORCHID                 0xDA70D6 
#define WEBR_PALEGOLDENROD          0xEEE8AA 
#define WEBR_PALEGREEN              0x98FB98 
#define WEBR_PALETURQUOISE          0xAFEEEE 
#define WEBR_PALEVIOLETRED          0xDB7093 
#define WEBR_PAPAYAWHIP             0xFFEFD5 
#define WEBR_PEACHPUFF              0xFFDAB9 
#define WEBR_PERU                   0xCD853F 
#define WEBR_PINK                   0xFFC0CB 
#define WEBR_PLUM                   0xDDA0DD 
#define WEBR_POWDERBLUE             0xB0E0E6 
#define WEBR_PURPLE                 0x800080 
#define WEBR_RED                    0xFF0000 
#define WEBR_ROSYBROWN              0xBC8F8F 
#define WEBR_ROYALBLUE              0x4169E1 
#define WEBR_SADDLEBROWN            0x8B4513 
#define WEBR_SALMON                 0xFA8072 
#define WEBR_SANDYBROWN             0xF4A460 
#define WEBR_SEAGREEN               0x2E8B57 
#define WEBR_SEASHELL               0xFFF5EE 
#define WEBR_SIENNA                 0xA0522D 
#define WEBR_SILVER                 0xC0C0C0 
#define WEBR_SKYBLUE                0x87CEEB 
#define WEBR_SLATEBLUE              0x6A5ACD 
#define WEBR_SLATEGRAY              0x708090 
#define WEBR_SNOW                   0xFFFAFA 
#define WEBR_SPRINGGREEN            0x00FF7F 
#define WEBR_STEELBLUE              0x4682B4 
#define WEBR_TAN                    0xD2B48C 
#define WEBR_TEAL                   0x008080 
#define WEBR_THISTLE                0xD8BFD8 
#define WEBR_TOMATO                 0xFF6347 
#define WEBR_TURQUOISE              0x40E0D0 
#define WEBR_VIOLET                 0xEE82EE 
#define WEBR_WHEAT                  0xF5DEB3 
#define WEBR_WHITE                  0xFFFFFF 
#define WEBR_WHITESMOKE             0xF5F5F5 
#define WEBR_YELLOW                 0xFFFF00 
#define WEBR_YELLOWGREEN            0x9ACD32 

//  some common synonyms

#define WEBR_AQUA                   0x00FFFF // CYAN
#define WEBR_DARKGREY               0xA9A9A9 // DARKGRAY
#define WEBR_DARKSLATEGREY          0x2F4F4F // DARKSLATEGRAY
#define WEBR_DIMGREY                0x696969 // DIMGRAY
#define WEBR_FUCHSIA                0xFF00FF // MAGENTA
#define WEBR_GREY                   0x808080 // GRAY
#define WEBR_LIGHTGREY              0xD3D3D3 // LIGHTGRAY
#define WEBR_LIGHTSLATEGREY         0x778899 // LIGHTSLATEGRAEY
#define WEBR_SLATEGREY              0x708090 // SLATEGRAY

// a convenient table of WEB colors

extern GrColorTableDef GrWebColorTable[];

int GrGenWebColorTable(void);

// GrColors you can use after calling GrGenWebColorTable,
// but they can not be used to init vars

#define WEBC_ALICEBLUE              GrWebColorTable[0].color 
#define WEBC_ANTIQUEWHITE           GrWebColorTable[1].color 
#define WEBC_AQUAMARINE             GrWebColorTable[2].color 
#define WEBC_AZURE                  GrWebColorTable[3].color 
#define WEBC_BEIGE                  GrWebColorTable[4].color 
#define WEBC_BISQUE                 GrWebColorTable[5].color 
#define WEBC_BLACK                  GrWebColorTable[6].color 
#define WEBC_BLANCHEDALMOND         GrWebColorTable[7].color 
#define WEBC_BLUE                   GrWebColorTable[8].color 
#define WEBC_BLUEVIOLET             GrWebColorTable[9].color 
#define WEBC_BROWN                  GrWebColorTable[10].color 
#define WEBC_BURLYWOOD              GrWebColorTable[11].color 
#define WEBC_CADETBLUE              GrWebColorTable[12].color 
#define WEBC_CHARTREUSE             GrWebColorTable[13].color 
#define WEBC_CHOCOLATE              GrWebColorTable[14].color 
#define WEBC_CORAL                  GrWebColorTable[15].color 
#define WEBC_CORNFLOWERBLUE         GrWebColorTable[16].color 
#define WEBC_CORNSILK               GrWebColorTable[17].color 
#define WEBC_CRIMSON                GrWebColorTable[18].color 
#define WEBC_CYAN                   GrWebColorTable[19].color 
#define WEBC_DARKBLUE               GrWebColorTable[20].color 
#define WEBC_DARKCYAN               GrWebColorTable[21].color 
#define WEBC_DARKGOLDENROD          GrWebColorTable[22].color 
#define WEBC_DARKGRAY               GrWebColorTable[23].color 
#define WEBC_DARKGREEN              GrWebColorTable[24].color 
#define WEBC_DARKKHAKI              GrWebColorTable[25].color 
#define WEBC_DARKMAGENTA            GrWebColorTable[26].color 
#define WEBC_DARKOLIVEGREEN         GrWebColorTable[27].color 
#define WEBC_DARKORANGE             GrWebColorTable[28].color 
#define WEBC_DARKORCHID             GrWebColorTable[29].color 
#define WEBC_DARKRED                GrWebColorTable[30].color 
#define WEBC_DARKSALMON             GrWebColorTable[31].color 
#define WEBC_DARKSEAGREEN           GrWebColorTable[32].color 
#define WEBC_DARKSLATEBLUE          GrWebColorTable[33].color 
#define WEBC_DARKSLATEGRAY          GrWebColorTable[34].color 
#define WEBC_DARKTURQUOISE          GrWebColorTable[35].color 
#define WEBC_DARKVIOLET             GrWebColorTable[36].color 
#define WEBC_DEEPPINK               GrWebColorTable[37].color 
#define WEBC_DEEPSKYBLUE            GrWebColorTable[38].color 
#define WEBC_DIMGRAY                GrWebColorTable[39].color 
#define WEBC_DODGERBLUE             GrWebColorTable[40].color 
#define WEBC_FIREBRICK              GrWebColorTable[41].color 
#define WEBC_FLORALWHITE            GrWebColorTable[42].color 
#define WEBC_FORESTGREEN            GrWebColorTable[43].color 
#define WEBC_GAINSBORO              GrWebColorTable[44].color 
#define WEBC_GHOSTWHITE             GrWebColorTable[45].color 
#define WEBC_GOLD                   GrWebColorTable[46].color 
#define WEBC_GOLDENROD              GrWebColorTable[47].color 
#define WEBC_GRAY                   GrWebColorTable[48].color 
#define WEBC_GREEN                  GrWebColorTable[49].color 
#define WEBC_GREENYELLOW            GrWebColorTable[50].color 
#define WEBC_HONEYDEW               GrWebColorTable[51].color 
#define WEBC_HOTPINK                GrWebColorTable[52].color 
#define WEBC_INDIANRED              GrWebColorTable[53].color 
#define WEBC_INDIGO                 GrWebColorTable[54].color 
#define WEBC_IVORY                  GrWebColorTable[55].color 
#define WEBC_KHAKI                  GrWebColorTable[56].color 
#define WEBC_LAVENDER               GrWebColorTable[57].color 
#define WEBC_LAVENDERBLUSH          GrWebColorTable[58].color 
#define WEBC_LAWNGREEN              GrWebColorTable[59].color 
#define WEBC_LEMONCHIFFON           GrWebColorTable[60].color 
#define WEBC_LIGHTBLUE              GrWebColorTable[61].color 
#define WEBC_LIGHTCORAL             GrWebColorTable[62].color 
#define WEBC_LIGHTCYAN              GrWebColorTable[63].color 
#define WEBC_LIGHTGOLDENRODYELLOW   GrWebColorTable[64].color 
#define WEBC_LIGHTGRAY              GrWebColorTable[65].color 
#define WEBC_LIGHTGREEN             GrWebColorTable[66].color 
#define WEBC_LIGHTPINK              GrWebColorTable[67].color 
#define WEBC_LIGHTSALMON            GrWebColorTable[68].color 
#define WEBC_LIGHTSEAGREEN          GrWebColorTable[69].color 
#define WEBC_LIGHTSKYBLUE           GrWebColorTable[70].color 
#define WEBC_LIGHTSLATEGRAY         GrWebColorTable[71].color 
#define WEBC_LIGHTSTEELBLUE         GrWebColorTable[72].color 
#define WEBC_LIGHTYELLOW            GrWebColorTable[73].color 
#define WEBC_LIME                   GrWebColorTable[74].color 
#define WEBC_LIMEGREEN              GrWebColorTable[75].color 
#define WEBC_LINEN                  GrWebColorTable[76].color 
#define WEBC_MAGENTA                GrWebColorTable[77].color 
#define WEBC_MAROON                 GrWebColorTable[78].color 
#define WEBC_MEDIUMAQUAMARINE       GrWebColorTable[79].color 
#define WEBC_MEDIUMBLUE             GrWebColorTable[80].color 
#define WEBC_MEDIUMORCHID           GrWebColorTable[81].color 
#define WEBC_MEDIUMPURPLE           GrWebColorTable[82].color 
#define WEBC_MEDIUMSEAGREEN         GrWebColorTable[83].color 
#define WEBC_MEDIUMSLATEBLUE        GrWebColorTable[84].color 
#define WEBC_MEDIUMSPRINGGREEN      GrWebColorTable[85].color 
#define WEBC_MEDIUMTURQUOISE        GrWebColorTable[86].color 
#define WEBC_MEDIUMVIOLETRED        GrWebColorTable[87].color 
#define WEBC_MIDNIGHTBLUE           GrWebColorTable[88].color 
#define WEBC_MINTCREAM              GrWebColorTable[89].color 
#define WEBC_MISTYROSE              GrWebColorTable[90].color 
#define WEBC_MOCCASIN               GrWebColorTable[91].color 
#define WEBC_NAVAJOWHITE            GrWebColorTable[92].color 
#define WEBC_NAVY                   GrWebColorTable[93].color 
#define WEBC_OLDLACE                GrWebColorTable[94].color 
#define WEBC_OLIVE                  GrWebColorTable[95].color 
#define WEBC_OLIVEDRAB              GrWebColorTable[96].color 
#define WEBC_ORANGE                 GrWebColorTable[97].color 
#define WEBC_ORANGERED              GrWebColorTable[98].color 
#define WEBC_ORCHID                 GrWebColorTable[99].color 
#define WEBC_PALEGOLDENROD          GrWebColorTable[100].color 
#define WEBC_PALEGREEN              GrWebColorTable[101].color 
#define WEBC_PALETURQUOISE          GrWebColorTable[102].color 
#define WEBC_PALEVIOLETRED          GrWebColorTable[103].color 
#define WEBC_PAPAYAWHIP             GrWebColorTable[104].color 
#define WEBC_PEACHPUFF              GrWebColorTable[105].color 
#define WEBC_PERU                   GrWebColorTable[106].color 
#define WEBC_PINK                   GrWebColorTable[107].color 
#define WEBC_PLUM                   GrWebColorTable[108].color 
#define WEBC_POWDERBLUE             GrWebColorTable[109].color 
#define WEBC_PURPLE                 GrWebColorTable[110].color 
#define WEBC_RED                    GrWebColorTable[111].color 
#define WEBC_ROSYBROWN              GrWebColorTable[112].color 
#define WEBC_ROYALBLUE              GrWebColorTable[113].color 
#define WEBC_SADDLEBROWN            GrWebColorTable[114].color 
#define WEBC_SALMON                 GrWebColorTable[115].color 
#define WEBC_SANDYBROWN             GrWebColorTable[116].color 
#define WEBC_SEAGREEN               GrWebColorTable[117].color 
#define WEBC_SEASHELL               GrWebColorTable[118].color 
#define WEBC_SIENNA                 GrWebColorTable[119].color 
#define WEBC_SILVER                 GrWebColorTable[120].color 
#define WEBC_SKYBLUE                GrWebColorTable[121].color 
#define WEBC_SLATEBLUE              GrWebColorTable[122].color 
#define WEBC_SLATEGRAY              GrWebColorTable[123].color 
#define WEBC_SNOW                   GrWebColorTable[124].color 
#define WEBC_SPRINGGREEN            GrWebColorTable[125].color 
#define WEBC_STEELBLUE              GrWebColorTable[126].color 
#define WEBC_TAN                    GrWebColorTable[127].color 
#define WEBC_TEAL                   GrWebColorTable[128].color 
#define WEBC_THISTLE                GrWebColorTable[129].color 
#define WEBC_TOMATO                 GrWebColorTable[130].color 
#define WEBC_TURQUOISE              GrWebColorTable[131].color 
#define WEBC_VIOLET                 GrWebColorTable[132].color 
#define WEBC_WHEAT                  GrWebColorTable[133].color 
#define WEBC_WHITE                  GrWebColorTable[134].color 
#define WEBC_WHITESMOKE             GrWebColorTable[135].color 
#define WEBC_YELLOW                 GrWebColorTable[136].color 
#define WEBC_YELLOWGREEN            GrWebColorTable[137].color 

//  some common synonyms

#define WEBC_AQUA                   WEBC_CYAN
#define WEBC_DARKGREY               WEBC_DARKGRAY
#define WEBC_DARKSLATEGREY          WEBC_DARKSLATEGRAY
#define WEBC_DIMGREY                WEBC_DIMGRAY
#define WEBC_FUCHSIA                WEBC_MAGENTA
#define WEBC_GREY                   WEBC_GRAY
#define WEBC_LIGHTGREY              WEBC_LIGHTGRAY
#define WEBC_LIGHTSLATEGREY         WEBC_LIGHTSLATEGRAEY
#define WEBC_SLATEGREY              WEBC_SLATEGRAY

// Indexes to Web table, valid to init vars,
// but after calling GrGenEgaColorTable you must use
// GrWebColorTable[WEBI_xxx].color or the macro WebColor

#define WEBI_ALICEBLUE              0 
#define WEBI_ANTIQUEWHITE           1 
#define WEBI_AQUAMARINE             2 
#define WEBI_AZURE                  3 
#define WEBI_BEIGE                  4 
#define WEBI_BISQUE                 5 
#define WEBI_BLACK                  6 
#define WEBI_BLANCHEDALMOND         7 
#define WEBI_BLUE                   8 
#define WEBI_BLUEVIOLET             9 
#define WEBI_BROWN                  10 
#define WEBI_BURLYWOOD              11 
#define WEBI_CADETBLUE              12 
#define WEBI_CHARTREUSE             13 
#define WEBI_CHOCOLATE              14 
#define WEBI_CORAL                  15 
#define WEBI_CORNFLOWERBLUE         16 
#define WEBI_CORNSILK               17 
#define WEBI_CRIMSON                18 
#define WEBI_CYAN                   19 
#define WEBI_DARKBLUE               20 
#define WEBI_DARKCYAN               21 
#define WEBI_DARKGOLDENROD          22 
#define WEBI_DARKGRAY               23 
#define WEBI_DARKGREEN              24 
#define WEBI_DARKKHAKI              25 
#define WEBI_DARKMAGENTA            26 
#define WEBI_DARKOLIVEGREEN         27 
#define WEBI_DARKORANGE             28 
#define WEBI_DARKORCHID             29 
#define WEBI_DARKRED                30 
#define WEBI_DARKSALMON             31 
#define WEBI_DARKSEAGREEN           32 
#define WEBI_DARKSLATEBLUE          33 
#define WEBI_DARKSLATEGRAY          34 
#define WEBI_DARKTURQUOISE          35 
#define WEBI_DARKVIOLET             36 
#define WEBI_DEEPPINK               37 
#define WEBI_DEEPSKYBLUE            38 
#define WEBI_DIMGRAY                39 
#define WEBI_DODGERBLUE             40 
#define WEBI_FIREBRICK              41 
#define WEBI_FLORALWHITE            42 
#define WEBI_FORESTGREEN            43 
#define WEBI_GAINSBORO              44 
#define WEBI_GHOSTWHITE             45 
#define WEBI_GOLD                   46 
#define WEBI_GOLDENROD              47 
#define WEBI_GRAY                   48 
#define WEBI_GREEN                  49 
#define WEBI_GREENYELLOW            50 
#define WEBI_HONEYDEW               51 
#define WEBI_HOTPINK                52 
#define WEBI_INDIANRED              53 
#define WEBI_INDIGO                 54 
#define WEBI_IVORY                  55 
#define WEBI_KHAKI                  56 
#define WEBI_LAVENDER               57 
#define WEBI_LAVENDERBLUSH          58 
#define WEBI_LAWNGREEN              59 
#define WEBI_LEMONCHIFFON           60 
#define WEBI_LIGHTBLUE              61 
#define WEBI_LIGHTCORAL             62 
#define WEBI_LIGHTCYAN              63 
#define WEBI_LIGHTGOLDENRODYELLOW   64 
#define WEBI_LIGHTGRAY              65 
#define WEBI_LIGHTGREEN             66 
#define WEBI_LIGHTPINK              67 
#define WEBI_LIGHTSALMON            68 
#define WEBI_LIGHTSEAGREEN          69 
#define WEBI_LIGHTSKYBLUE           70 
#define WEBI_LIGHTSLATEGRAY         71 
#define WEBI_LIGHTSTEELBLUE         72 
#define WEBI_LIGHTYELLOW            73 
#define WEBI_LIME                   74 
#define WEBI_LIMEGREEN              75 
#define WEBI_LINEN                  76 
#define WEBI_MAGENTA                77 
#define WEBI_MAROON                 78 
#define WEBI_MEDIUMAQUAMARINE       79 
#define WEBI_MEDIUMBLUE             80 
#define WEBI_MEDIUMORCHID           81 
#define WEBI_MEDIUMPURPLE           82 
#define WEBI_MEDIUMSEAGREEN         83 
#define WEBI_MEDIUMSLATEBLUE        84 
#define WEBI_MEDIUMSPRINGGREEN      85 
#define WEBI_MEDIUMTURQUOISE        86 
#define WEBI_MEDIUMVIOLETRED        87 
#define WEBI_MIDNIGHTBLUE           88 
#define WEBI_MINTCREAM              89 
#define WEBI_MISTYROSE              90 
#define WEBI_MOCCASIN               91 
#define WEBI_NAVAJOWHITE            92 
#define WEBI_NAVY                   93 
#define WEBI_OLDLACE                94 
#define WEBI_OLIVE                  95 
#define WEBI_OLIVEDRAB              96 
#define WEBI_ORANGE                 97 
#define WEBI_ORANGERED              98 
#define WEBI_ORCHID                 99 
#define WEBI_PALEGOLDENROD          100 
#define WEBI_PALEGREEN              101 
#define WEBI_PALETURQUOISE          102 
#define WEBI_PALEVIOLETRED          103 
#define WEBI_PAPAYAWHIP             104 
#define WEBI_PEACHPUFF              105 
#define WEBI_PERU                   106 
#define WEBI_PINK                   107 
#define WEBI_PLUM                   108 
#define WEBI_POWDERBLUE             109 
#define WEBI_PURPLE                 110 
#define WEBI_RED                    111 
#define WEBI_ROSYBROWN              112 
#define WEBI_ROYALBLUE              113 
#define WEBI_SADDLEBROWN            114 
#define WEBI_SALMON                 115 
#define WEBI_SANDYBROWN             116 
#define WEBI_SEAGREEN               117 
#define WEBI_SEASHELL               118 
#define WEBI_SIENNA                 119 
#define WEBI_SILVER                 120 
#define WEBI_SKYBLUE                121 
#define WEBI_SLATEBLUE              122 
#define WEBI_SLATEGRAY              123 
#define WEBI_SNOW                   124 
#define WEBI_SPRINGGREEN            125 
#define WEBI_STEELBLUE              126 
#define WEBI_TAN                    127 
#define WEBI_TEAL                   128 
#define WEBI_THISTLE                129 
#define WEBI_TOMATO                 130 
#define WEBI_TURQUOISE              131 
#define WEBI_VIOLET                 132 
#define WEBI_WHEAT                  133 
#define WEBI_WHITE                  134 
#define WEBI_WHITESMOKE             135 
#define WEBI_YELLOW                 136 
#define WEBI_YELLOWGREEN            137 

//  some common synonyms

#define WEBI_AQUA                   WEBI_CYAN
#define WEBI_DARKGREY               WEBI_DARKGRAY
#define WEBI_DARKSLATEGREY          WEBI_DARKSLATEGRAY
#define WEBI_DIMGREY                WEBI_DIMGRAY
#define WEBI_FUCHSIA                WEBI_MAGENTA
#define WEBI_GREY                   WEBI_GRAY
#define WEBI_LIGHTGREY              WEBI_LIGHTGRAY
#define WEBI_LIGHTSLATEGREY         WEBI_LIGHTSLATEGRAEY
#define WEBI_SLATEGREY              WEBI_SLATEGRAY

#define WebColor(i) (GrWebColorTable[i].color)
//
// ANSI colors, usally used in xterm an other consoles
//

#define NUM_ANS_COLORS 256

// RGB color definitions, valid to init vars,
// but them you need GrAllocColor2(ANSR_xxx)

#define  ANSR_BLACK                 0x000000
#define  ANSR_MAROON                0x800000
#define  ANSR_GREEN                 0x008000
#define  ANSR_OLIVE                 0x808000
#define  ANSR_NAVY                  0x000080
#define  ANSR_PURPLE                0x800080
#define  ANSR_TEAL                  0x008080
#define  ANSR_SILVER                0xC0C0C0
#define  ANSR_GREY                  0x808080
#define  ANSR_RED                   0xFF0000
#define  ANSR_LIME                  0x00FF00
#define  ANSR_YELLOW                0xFFFF00
#define  ANSR_BLUE                  0x0000FF
#define  ANSR_FUCHSIA               0xFF00FF
#define  ANSR_AQUA                  0x00FFFF
#define  ANSR_WHITE                 0xFFFFFF
#define  ANSR_GREY0                 0x000000
#define  ANSR_NAVYBLUE              0x00005F
#define  ANSR_DARKBLUE              0x000087
#define  ANSR_BLUE2                 0x0000AF
#define  ANSR_BLUE3                 0x0000D7
#define  ANSR_BLUE4                 0x0000FF
#define  ANSR_DARKGREEN             0x005F00
#define  ANSR_DEEPSKYBLUE           0x005F5F
#define  ANSR_DEEPSKYBLUE2          0x005F87
#define  ANSR_DEEPSKYBLUE3          0x005FAF
#define  ANSR_DODGERBLUE            0x005FD7
#define  ANSR_DODGERBLUE2           0x005FFF
#define  ANSR_GREEN2                0x008700
#define  ANSR_SPRINGGREEN           0x00875F
#define  ANSR_TURQUOISE             0x008787
#define  ANSR_DEEPSKYBLUE4          0x0087AF
#define  ANSR_DEEPSKYBLUE5          0x0087D7
#define  ANSR_DODGERBLUE3           0x0087FF
#define  ANSR_GREEN3                0x00AF00
#define  ANSR_SPRINGGREEN2          0x00AF5F
#define  ANSR_DARKCYAN              0x00AF87
#define  ANSR_LIGHTSEAGREEN         0x00AFAF
#define  ANSR_DEEPSKYBLUE6          0x00AFD7
#define  ANSR_DEEPSKYBLUE7          0x00AFFF
#define  ANSR_GREEN4                0x00D700
#define  ANSR_SPRINGGREEN3          0x00D75F
#define  ANSR_SPRINGGREEN4          0x00D787
#define  ANSR_CYAN                  0x00D7AF
#define  ANSR_DARKTURQUOISE         0x00D7D7
#define  ANSR_TURQUOISE2            0x00D7FF
#define  ANSR_GREEN5                0x00FF00
#define  ANSR_SPRINGGREEN5          0x00FF5F
#define  ANSR_SPRINGGREEN6          0x00FF87
#define  ANSR_MEDIUMSPRINGGREEN     0x00FFAF
#define  ANSR_CYAN2                 0x00FFD7
#define  ANSR_CYAN3                 0x00FFFF
#define  ANSR_DARKRED               0x5F0000
#define  ANSR_DEEPPINK              0x5F005F
#define  ANSR_PURPLE2               0x5F0087
#define  ANSR_PURPLE3               0x5F00AF
#define  ANSR_PURPLE4               0x5F00D7
#define  ANSR_BLUEVIOLET            0x5F00FF
#define  ANSR_ORANGE                0x5F5F00
#define  ANSR_GREY37                0x5F5F5F
#define  ANSR_MEDIUMPURPLE          0x5F5F87
#define  ANSR_SLATEBLUE             0x5F5FAF
#define  ANSR_SLATEBLUE2            0x5F5FD7
#define  ANSR_ROYALBLUE             0x5F5FFF
#define  ANSR_CHARTREUSE            0x5F8700
#define  ANSR_DARKSEAGREEN          0x5F875F
#define  ANSR_PALETURQUOISE         0x5F8787
#define  ANSR_STEELBLUE             0x5F87AF
#define  ANSR_STEELBLUE2            0x5F87D7
#define  ANSR_CORNFLOWERBLUE        0x5F87FF
#define  ANSR_CHARTREUSE2           0x5FAF00
#define  ANSR_DARKSEAGREEN2         0x5FAF5F
#define  ANSR_CADETBLUE             0x5FAF87
#define  ANSR_CADETBLUE2            0x5FAFAF
#define  ANSR_SKYBLUE               0x5FAFD7
#define  ANSR_STEELBLUE3            0x5FAFFF
#define  ANSR_CHARTREUSE3           0x5FD700
#define  ANSR_PALEGREEN             0x5FD75F
#define  ANSR_SEAGREEN              0x5FD787
#define  ANSR_AQUAMARINE            0x5FD7AF
#define  ANSR_MEDIUMTURQUOISE       0x5FD7D7
#define  ANSR_STEELBLUE4            0x5FD7FF
#define  ANSR_CHARTREUSE4           0x5FFF00
#define  ANSR_SEAGREEN2             0x5FFF5F
#define  ANSR_SEAGREEN3             0x5FFF87
#define  ANSR_SEAGREEN4             0x5FFFAF
#define  ANSR_AQUAMARINE2           0x5FFFD7
#define  ANSR_DARKSLATEGRAY         0x5FFFFF
#define  ANSR_DARKRED2              0x870000
#define  ANSR_DEEPPINK2             0x87005F
#define  ANSR_DARKMAGENTA           0x870087
#define  ANSR_DARKMAGENTA2          0x8700AF
#define  ANSR_DARKVIOLET            0x8700D7
#define  ANSR_PURPLE5               0x8700FF
#define  ANSR_ORANGE2               0x875F00
#define  ANSR_LIGHTPINK             0x875F5F
#define  ANSR_PLUM                  0x875F87
#define  ANSR_MEDIUMPURPLE2         0x875FAF
#define  ANSR_MEDIUMPURPLE3         0x875FD7
#define  ANSR_SLATEBLUE3            0x875FFF
#define  ANSR_YELLOW2               0x878700
#define  ANSR_WHEAT                 0x87875F
#define  ANSR_GREY53                0x878787
#define  ANSR_LIGHTSLATEGREY        0x8787AF
#define  ANSR_MEDIUMPURPLE4         0x8787D7
#define  ANSR_LIGHTSLATEBLUE        0x8787FF
#define  ANSR_YELLOW3               0x87AF00
#define  ANSR_DARKOLIVEGREEN        0x87AF5F
#define  ANSR_DARKSEAGREEN3         0x87AF87
#define  ANSR_LIGHTSKYBLUE          0x87AFAF
#define  ANSR_LIGHTSKYBLUE2         0x87AFD7
#define  ANSR_SKYBLUE2              0x87AFFF
#define  ANSR_CHARTREUSE5           0x87D700
#define  ANSR_DARKOLIVEGREEN2       0x87D75F
#define  ANSR_PALEGREEN2            0x87D787
#define  ANSR_DARKSEAGREEN4         0x87D7AF
#define  ANSR_DARKSLATEGRAY2        0x87D7D7
#define  ANSR_SKYBLUE3              0x87D7FF
#define  ANSR_CHARTREUSE6           0x87FF00
#define  ANSR_LIGHTGREEN            0x87FF5F
#define  ANSR_LIGHTGREEN2           0x87FF87
#define  ANSR_PALEGREEN3            0x87FFAF
#define  ANSR_AQUAMARINE3           0x87FFD7
#define  ANSR_DARKSLATEGRAY3        0x87FFFF
#define  ANSR_RED2                  0xAF0000
#define  ANSR_DEEPPINK3             0xAF005F
#define  ANSR_MEDIUMVIOLETRED       0xAF0087
#define  ANSR_MAGENTA               0xAF00AF
#define  ANSR_DARKVIOLET2           0xAF00D7
#define  ANSR_PURPLE6               0xAF00FF
#define  ANSR_DARKORANGE            0xAF5F00
#define  ANSR_INDIANRED             0xAF5F5F
#define  ANSR_HOTPINK               0xAF5F87
#define  ANSR_MEDIUMORCHID          0xAF5FAF
#define  ANSR_MEDIUMORCHID2         0xAF5FD7
#define  ANSR_MEDIUMPURPLE5         0xAF5FFF
#define  ANSR_DARKGOLDENROD         0xAF8700
#define  ANSR_LIGHTSALMON           0xAF875F
#define  ANSR_ROSYBROWN             0xAF8787
#define  ANSR_GREY63                0xAF87AF
#define  ANSR_MEDIUMPURPLE6         0xAF87D7
#define  ANSR_MEDIUMPURPLE7         0xAF87FF
#define  ANSR_GOLD                  0xAFAF00
#define  ANSR_DARKKHAKI             0xAFAF5F
#define  ANSR_NAVAJOWHITE           0xAFAF87
#define  ANSR_GREY69                0xAFAFAF
#define  ANSR_LIGHTSTEELBLUE        0xAFAFD7
#define  ANSR_LIGHTSTEELBLUE2       0xAFAFFF
#define  ANSR_YELLOW4               0xAFD700
#define  ANSR_DARKOLIVEGREEN3       0xAFD75F
#define  ANSR_DARKSEAGREEN5         0xAFD787
#define  ANSR_DARKSEAGREEN6         0xAFD7AF
#define  ANSR_LIGHTCYAN             0xAFD7D7
#define  ANSR_LIGHTSKYBLUE3         0xAFD7FF
#define  ANSR_GREENYELLOW           0xAFFF00
#define  ANSR_DARKOLIVEGREEN4       0xAFFF5F
#define  ANSR_PALEGREEN4            0xAFFF87
#define  ANSR_DARKSEAGREEN7         0xAFFFAF
#define  ANSR_DARKSEAGREEN8         0xAFFFD7
#define  ANSR_PALETURQUOISE2        0xAFFFFF
#define  ANSR_RED3                  0xD70000
#define  ANSR_DEEPPINK4             0xD7005F
#define  ANSR_DEEPPINK5             0xD70087
#define  ANSR_MAGENTA2              0xD700AF
#define  ANSR_MAGENTA3              0xD700D7
#define  ANSR_MAGENTA4              0xD700FF
#define  ANSR_DARKORANGE2           0xD75F00
#define  ANSR_INDIANRED2            0xD75F5F
#define  ANSR_HOTPINK2              0xD75F87
#define  ANSR_HOTPINK3              0xD75FAF
#define  ANSR_ORCHID                0xD75FD7
#define  ANSR_MEDIUMORCHID3         0xD75FFF
#define  ANSR_ORANGE3               0xD78700
#define  ANSR_LIGHTSALMON2          0xD7875F
#define  ANSR_LIGHTPINK2            0xD78787
#define  ANSR_PINK                  0xD787AF
#define  ANSR_PLUM2                 0xD787D7
#define  ANSR_VIOLET                0xD787FF
#define  ANSR_GOLD2                 0xD7AF00
#define  ANSR_LIGHTGOLDENROD        0xD7AF5F
#define  ANSR_TAN                   0xD7AF87
#define  ANSR_MISTYROSE             0xD7AFAF
#define  ANSR_THISTLE               0xD7AFD7
#define  ANSR_PLUM3                 0xD7AFFF
#define  ANSR_YELLOW5               0xD7D700
#define  ANSR_KHAKI                 0xD7D75F
#define  ANSR_LIGHTGOLDENROD2       0xD7D787
#define  ANSR_LIGHTYELLOW           0xD7D7AF
#define  ANSR_GREY84                0xD7D7D7
#define  ANSR_LIGHTSTEELBLUE3       0xD7D7FF
#define  ANSR_YELLOW6               0xD7FF00
#define  ANSR_DARKOLIVEGREEN5       0xD7FF5F
#define  ANSR_DARKOLIVEGREEN6       0xD7FF87
#define  ANSR_DARKSEAGREEN9         0xD7FFAF
#define  ANSR_HONEYDEW              0xD7FFD7
#define  ANSR_LIGHTCYAN2            0xD7FFFF
#define  ANSR_RED4                  0xFF0000
#define  ANSR_DEEPPINK6             0xFF005F
#define  ANSR_DEEPPINK7             0xFF0087
#define  ANSR_DEEPPINK8             0xFF00AF
#define  ANSR_MAGENTA5              0xFF00D7
#define  ANSR_MAGENTA6              0xFF00FF
#define  ANSR_ORANGERED             0xFF5F00
#define  ANSR_INDIANRED3            0xFF5F5F
#define  ANSR_INDIANRED4            0xFF5F87
#define  ANSR_HOTPINK4              0xFF5FAF
#define  ANSR_HOTPINK5              0xFF5FD7
#define  ANSR_MEDIUMORCHID4         0xFF5FFF
#define  ANSR_DARKORANGE3           0xFF8700
#define  ANSR_SALMON                0xFF875F
#define  ANSR_LIGHTCORAL            0xFF8787
#define  ANSR_PALEVIOLETRED         0xFF87AF
#define  ANSR_ORCHID2               0xFF87D7
#define  ANSR_ORCHID3               0xFF87FF
#define  ANSR_ORANGE4               0xFFAF00
#define  ANSR_SANDYBROWN            0xFFAF5F
#define  ANSR_LIGHTSALMON3          0xFFAF87
#define  ANSR_LIGHTPINK3            0xFFAFAF
#define  ANSR_PINK2                 0xFFAFD7
#define  ANSR_PLUM4                 0xFFAFFF
#define  ANSR_GOLD3                 0xFFD700
#define  ANSR_LIGHTGOLDENROD3       0xFFD75F
#define  ANSR_LIGHTGOLDENROD4       0xFFD787
#define  ANSR_NAVAJOWHITE2          0xFFD7AF
#define  ANSR_MISTYROSE2            0xFFD7D7
#define  ANSR_THISTLE2              0xFFD7FF
#define  ANSR_YELLOW7               0xFFFF00
#define  ANSR_LIGHTGOLDENROD5       0xFFFF5F
#define  ANSR_KHAKI2                0xFFFF87
#define  ANSR_WHEAT2                0xFFFFAF
#define  ANSR_CORNSILK              0xFFFFD7
#define  ANSR_GREY100               0xFFFFFF
#define  ANSR_GREY3                 0x080808
#define  ANSR_GREY7                 0x121212
#define  ANSR_GREY11                0x1C1C1C
#define  ANSR_GREY15                0x262626
#define  ANSR_GREY19                0x303030
#define  ANSR_GREY23                0x3A3A3A
#define  ANSR_GREY27                0x444444
#define  ANSR_GREY30                0x4E4E4E
#define  ANSR_GREY35                0x585858
#define  ANSR_GREY39                0x626262
#define  ANSR_GREY42                0x6C6C6C
#define  ANSR_GREY46                0x767676
#define  ANSR_GREY50                0x808080
#define  ANSR_GREY54                0x8A8A8A
#define  ANSR_GREY58                0x949494
#define  ANSR_GREY62                0x9E9E9E
#define  ANSR_GREY66                0xA8A8A8
#define  ANSR_GREY70                0xB2B2B2
#define  ANSR_GREY74                0xBCBCBC
#define  ANSR_GREY78                0xC6C6C6
#define  ANSR_GREY82                0xD0D0D0
#define  ANSR_GREY85                0xDADADA
#define  ANSR_GREY89                0xE4E4E4
#define  ANSR_GREY93                0xEEEEEE

// a convenient table of ANSI colors

extern GrColorTableDef GrAnsColorTable[];

int GrGenAnsColorTable(void);

// GrColors you can use after calling GrGenAnsColorTable,
// but they can not be used to init vars

#define  ANSC_BLACK                GrAnsColorTable[0].color
#define  ANSC_MAROON               GrAnsColorTable[1].color
#define  ANSC_GREEN                GrAnsColorTable[2].color
#define  ANSC_OLIVE                GrAnsColorTable[3].color
#define  ANSC_NAVY                 GrAnsColorTable[4].color
#define  ANSC_PURPLE               GrAnsColorTable[5].color
#define  ANSC_TEAL                 GrAnsColorTable[6].color
#define  ANSC_SILVER               GrAnsColorTable[7].color
#define  ANSC_GREY                 GrAnsColorTable[8].color
#define  ANSC_RED                  GrAnsColorTable[9].color
#define  ANSC_LIME                 GrAnsColorTable[10].color
#define  ANSC_YELLOW               GrAnsColorTable[11].color
#define  ANSC_BLUE                 GrAnsColorTable[12].color
#define  ANSC_FUCHSIA              GrAnsColorTable[13].color
#define  ANSC_AQUA                 GrAnsColorTable[14].color
#define  ANSC_WHITE                GrAnsColorTable[15].color
#define  ANSC_GREY0                GrAnsColorTable[16].color
#define  ANSC_NAVYBLUE             GrAnsColorTable[17].color
#define  ANSC_DARKBLUE             GrAnsColorTable[18].color
#define  ANSC_BLUE2                GrAnsColorTable[19].color
#define  ANSC_BLUE3                GrAnsColorTable[20].color
#define  ANSC_BLUE4                GrAnsColorTable[21].color
#define  ANSC_DARKGREEN            GrAnsColorTable[22].color
#define  ANSC_DEEPSKYBLUE          GrAnsColorTable[23].color
#define  ANSC_DEEPSKYBLUE2         GrAnsColorTable[24].color
#define  ANSC_DEEPSKYBLUE3         GrAnsColorTable[25].color
#define  ANSC_DODGERBLUE           GrAnsColorTable[26].color
#define  ANSC_DODGERBLUE2          GrAnsColorTable[27].color
#define  ANSC_GREEN2               GrAnsColorTable[28].color
#define  ANSC_SPRINGGREEN          GrAnsColorTable[29].color
#define  ANSC_TURQUOISE            GrAnsColorTable[30].color
#define  ANSC_DEEPSKYBLUE4         GrAnsColorTable[31].color
#define  ANSC_DEEPSKYBLUE5         GrAnsColorTable[32].color
#define  ANSC_DODGERBLUE3          GrAnsColorTable[33].color
#define  ANSC_GREEN3               GrAnsColorTable[34].color
#define  ANSC_SPRINGGREEN2         GrAnsColorTable[35].color
#define  ANSC_DARKCYAN             GrAnsColorTable[36].color
#define  ANSC_LIGHTSEAGREEN        GrAnsColorTable[37].color
#define  ANSC_DEEPSKYBLUE6         GrAnsColorTable[38].color
#define  ANSC_DEEPSKYBLUE7         GrAnsColorTable[39].color
#define  ANSC_GREEN4               GrAnsColorTable[40].color
#define  ANSC_SPRINGGREEN3         GrAnsColorTable[41].color
#define  ANSC_SPRINGGREEN4         GrAnsColorTable[42].color
#define  ANSC_CYAN                 GrAnsColorTable[43].color
#define  ANSC_DARKTURQUOISE        GrAnsColorTable[44].color
#define  ANSC_TURQUOISE2           GrAnsColorTable[45].color
#define  ANSC_GREEN5               GrAnsColorTable[46].color
#define  ANSC_SPRINGGREEN5         GrAnsColorTable[47].color
#define  ANSC_SPRINGGREEN6         GrAnsColorTable[48].color
#define  ANSC_MEDIUMSPRINGGREEN    GrAnsColorTable[49].color
#define  ANSC_CYAN2                GrAnsColorTable[50].color
#define  ANSC_CYAN3                GrAnsColorTable[51].color
#define  ANSC_DARKRED              GrAnsColorTable[52].color
#define  ANSC_DEEPPINK             GrAnsColorTable[53].color
#define  ANSC_PURPLE2              GrAnsColorTable[54].color
#define  ANSC_PURPLE3              GrAnsColorTable[55].color
#define  ANSC_PURPLE4              GrAnsColorTable[56].color
#define  ANSC_BLUEVIOLET           GrAnsColorTable[57].color
#define  ANSC_ORANGE               GrAnsColorTable[58].color
#define  ANSC_GREY37               GrAnsColorTable[59].color
#define  ANSC_MEDIUMPURPLE         GrAnsColorTable[60].color
#define  ANSC_SLATEBLUE            GrAnsColorTable[61].color
#define  ANSC_SLATEBLUE2           GrAnsColorTable[62].color
#define  ANSC_ROYALBLUE            GrAnsColorTable[63].color
#define  ANSC_CHARTREUSE           GrAnsColorTable[64].color
#define  ANSC_DARKSEAGREEN         GrAnsColorTable[65].color
#define  ANSC_PALETURQUOISE        GrAnsColorTable[66].color
#define  ANSC_STEELBLUE            GrAnsColorTable[67].color
#define  ANSC_STEELBLUE2           GrAnsColorTable[68].color
#define  ANSC_CORNFLOWERBLUE       GrAnsColorTable[69].color
#define  ANSC_CHARTREUSE2          GrAnsColorTable[70].color
#define  ANSC_DARKSEAGREEN2        GrAnsColorTable[71].color
#define  ANSC_CADETBLUE            GrAnsColorTable[72].color
#define  ANSC_CADETBLUE2           GrAnsColorTable[73].color
#define  ANSC_SKYBLUE              GrAnsColorTable[74].color
#define  ANSC_STEELBLUE3           GrAnsColorTable[75].color
#define  ANSC_CHARTREUSE3          GrAnsColorTable[76].color
#define  ANSC_PALEGREEN            GrAnsColorTable[77].color
#define  ANSC_SEAGREEN             GrAnsColorTable[78].color
#define  ANSC_AQUAMARINE           GrAnsColorTable[79].color
#define  ANSC_MEDIUMTURQUOISE      GrAnsColorTable[80].color
#define  ANSC_STEELBLUE4           GrAnsColorTable[81].color
#define  ANSC_CHARTREUSE4          GrAnsColorTable[82].color
#define  ANSC_SEAGREEN2            GrAnsColorTable[83].color
#define  ANSC_SEAGREEN3            GrAnsColorTable[84].color
#define  ANSC_SEAGREEN4            GrAnsColorTable[85].color
#define  ANSC_AQUAMARINE2          GrAnsColorTable[86].color
#define  ANSC_DARKSLATEGRAY        GrAnsColorTable[87].color
#define  ANSC_DARKRED2             GrAnsColorTable[88].color
#define  ANSC_DEEPPINK2            GrAnsColorTable[89].color
#define  ANSC_DARKMAGENTA          GrAnsColorTable[90].color
#define  ANSC_DARKMAGENTA2         GrAnsColorTable[91].color
#define  ANSC_DARKVIOLET           GrAnsColorTable[92].color
#define  ANSC_PURPLE5              GrAnsColorTable[93].color
#define  ANSC_ORANGE2              GrAnsColorTable[94].color
#define  ANSC_LIGHTPINK            GrAnsColorTable[95].color
#define  ANSC_PLUM                 GrAnsColorTable[96].color
#define  ANSC_MEDIUMPURPLE2        GrAnsColorTable[97].color
#define  ANSC_MEDIUMPURPLE3        GrAnsColorTable[98].color
#define  ANSC_SLATEBLUE3           GrAnsColorTable[99].color
#define  ANSC_YELLOW2              GrAnsColorTable[100].color
#define  ANSC_WHEAT                GrAnsColorTable[101].color
#define  ANSC_GREY53               GrAnsColorTable[102].color
#define  ANSC_LIGHTSLATEGREY       GrAnsColorTable[103].color
#define  ANSC_MEDIUMPURPLE4        GrAnsColorTable[104].color
#define  ANSC_LIGHTSLATEBLUE       GrAnsColorTable[105].color
#define  ANSC_YELLOW3              GrAnsColorTable[106].color
#define  ANSC_DARKOLIVEGREEN       GrAnsColorTable[107].color
#define  ANSC_DARKSEAGREEN3        GrAnsColorTable[108].color
#define  ANSC_LIGHTSKYBLUE         GrAnsColorTable[109].color
#define  ANSC_LIGHTSKYBLUE2        GrAnsColorTable[110].color
#define  ANSC_SKYBLUE2             GrAnsColorTable[111].color
#define  ANSC_CHARTREUSE5          GrAnsColorTable[112].color
#define  ANSC_DARKOLIVEGREEN2      GrAnsColorTable[113].color
#define  ANSC_PALEGREEN2           GrAnsColorTable[114].color
#define  ANSC_DARKSEAGREEN4        GrAnsColorTable[115].color
#define  ANSC_DARKSLATEGRAY2       GrAnsColorTable[116].color
#define  ANSC_SKYBLUE3             GrAnsColorTable[117].color
#define  ANSC_CHARTREUSE6          GrAnsColorTable[118].color
#define  ANSC_LIGHTGREEN           GrAnsColorTable[119].color
#define  ANSC_LIGHTGREEN2          GrAnsColorTable[120].color
#define  ANSC_PALEGREEN3           GrAnsColorTable[121].color
#define  ANSC_AQUAMARINE3          GrAnsColorTable[122].color
#define  ANSC_DARKSLATEGRAY3       GrAnsColorTable[123].color
#define  ANSC_RED2                 GrAnsColorTable[124].color
#define  ANSC_DEEPPINK3            GrAnsColorTable[125].color
#define  ANSC_MEDIUMVIOLETRED      GrAnsColorTable[126].color
#define  ANSC_MAGENTA              GrAnsColorTable[127].color
#define  ANSC_DARKVIOLET2          GrAnsColorTable[128].color
#define  ANSC_PURPLE6              GrAnsColorTable[129].color
#define  ANSC_DARKORANGE           GrAnsColorTable[130].color
#define  ANSC_INDIANRED            GrAnsColorTable[131].color
#define  ANSC_HOTPINK              GrAnsColorTable[132].color
#define  ANSC_MEDIUMORCHID         GrAnsColorTable[133].color
#define  ANSC_MEDIUMORCHID2        GrAnsColorTable[134].color
#define  ANSC_MEDIUMPURPLE5        GrAnsColorTable[135].color
#define  ANSC_DARKGOLDENROD        GrAnsColorTable[136].color
#define  ANSC_LIGHTSALMON          GrAnsColorTable[137].color
#define  ANSC_ROSYBROWN            GrAnsColorTable[138].color
#define  ANSC_GREY63               GrAnsColorTable[139].color
#define  ANSC_MEDIUMPURPLE6        GrAnsColorTable[140].color
#define  ANSC_MEDIUMPURPLE7        GrAnsColorTable[141].color
#define  ANSC_GOLD                 GrAnsColorTable[142].color
#define  ANSC_DARKKHAKI            GrAnsColorTable[143].color
#define  ANSC_NAVAJOWHITE          GrAnsColorTable[144].color
#define  ANSC_GREY69               GrAnsColorTable[145].color
#define  ANSC_LIGHTSTEELBLUE       GrAnsColorTable[146].color
#define  ANSC_LIGHTSTEELBLUE2      GrAnsColorTable[147].color
#define  ANSC_YELLOW4              GrAnsColorTable[148].color
#define  ANSC_DARKOLIVEGREEN3      GrAnsColorTable[149].color
#define  ANSC_DARKSEAGREEN5        GrAnsColorTable[150].color
#define  ANSC_DARKSEAGREEN6        GrAnsColorTable[151].color
#define  ANSC_LIGHTCYAN            GrAnsColorTable[152].color
#define  ANSC_LIGHTSKYBLUE3        GrAnsColorTable[153].color
#define  ANSC_GREENYELLOW          GrAnsColorTable[154].color
#define  ANSC_DARKOLIVEGREEN4      GrAnsColorTable[155].color
#define  ANSC_PALEGREEN4           GrAnsColorTable[156].color
#define  ANSC_DARKSEAGREEN7        GrAnsColorTable[157].color
#define  ANSC_DARKSEAGREEN8        GrAnsColorTable[158].color
#define  ANSC_PALETURQUOISE2       GrAnsColorTable[159].color
#define  ANSC_RED3                 GrAnsColorTable[160].color
#define  ANSC_DEEPPINK4            GrAnsColorTable[161].color
#define  ANSC_DEEPPINK5            GrAnsColorTable[162].color
#define  ANSC_MAGENTA2             GrAnsColorTable[163].color
#define  ANSC_MAGENTA3             GrAnsColorTable[164].color
#define  ANSC_MAGENTA4             GrAnsColorTable[165].color
#define  ANSC_DARKORANGE2          GrAnsColorTable[166].color
#define  ANSC_INDIANRED2           GrAnsColorTable[167].color
#define  ANSC_HOTPINK2             GrAnsColorTable[168].color
#define  ANSC_HOTPINK3             GrAnsColorTable[169].color
#define  ANSC_ORCHID               GrAnsColorTable[170].color
#define  ANSC_MEDIUMORCHID3        GrAnsColorTable[171].color
#define  ANSC_ORANGE3              GrAnsColorTable[172].color
#define  ANSC_LIGHTSALMON2         GrAnsColorTable[173].color
#define  ANSC_LIGHTPINK2           GrAnsColorTable[174].color
#define  ANSC_PINK                 GrAnsColorTable[175].color
#define  ANSC_PLUM2                GrAnsColorTable[176].color
#define  ANSC_VIOLET               GrAnsColorTable[177].color
#define  ANSC_GOLD2                GrAnsColorTable[178].color
#define  ANSC_LIGHTGOLDENROD       GrAnsColorTable[179].color
#define  ANSC_TAN                  GrAnsColorTable[180].color
#define  ANSC_MISTYROSE            GrAnsColorTable[181].color
#define  ANSC_THISTLE              GrAnsColorTable[182].color
#define  ANSC_PLUM3                GrAnsColorTable[183].color
#define  ANSC_YELLOW5              GrAnsColorTable[184].color
#define  ANSC_KHAKI                GrAnsColorTable[185].color
#define  ANSC_LIGHTGOLDENROD2      GrAnsColorTable[186].color
#define  ANSC_LIGHTYELLOW          GrAnsColorTable[187].color
#define  ANSC_GREY84               GrAnsColorTable[188].color
#define  ANSC_LIGHTSTEELBLUE3      GrAnsColorTable[189].color
#define  ANSC_YELLOW6              GrAnsColorTable[190].color
#define  ANSC_DARKOLIVEGREEN5      GrAnsColorTable[191].color
#define  ANSC_DARKOLIVEGREEN6      GrAnsColorTable[192].color
#define  ANSC_DARKSEAGREEN9        GrAnsColorTable[193].color
#define  ANSC_HONEYDEW             GrAnsColorTable[194].color
#define  ANSC_LIGHTCYAN2           GrAnsColorTable[195].color
#define  ANSC_RED4                 GrAnsColorTable[196].color
#define  ANSC_DEEPPINK6            GrAnsColorTable[197].color
#define  ANSC_DEEPPINK7            GrAnsColorTable[198].color
#define  ANSC_DEEPPINK8            GrAnsColorTable[199].color
#define  ANSC_MAGENTA5             GrAnsColorTable[200].color
#define  ANSC_MAGENTA6             GrAnsColorTable[201].color
#define  ANSC_ORANGERED            GrAnsColorTable[202].color
#define  ANSC_INDIANRED3           GrAnsColorTable[203].color
#define  ANSC_INDIANRED4           GrAnsColorTable[204].color
#define  ANSC_HOTPINK4             GrAnsColorTable[205].color
#define  ANSC_HOTPINK5             GrAnsColorTable[206].color
#define  ANSC_MEDIUMORCHID4        GrAnsColorTable[207].color
#define  ANSC_DARKORANGE3          GrAnsColorTable[208].color
#define  ANSC_SALMON               GrAnsColorTable[209].color
#define  ANSC_LIGHTCORAL           GrAnsColorTable[210].color
#define  ANSC_PALEVIOLETRED        GrAnsColorTable[211].color
#define  ANSC_ORCHID2              GrAnsColorTable[212].color
#define  ANSC_ORCHID3              GrAnsColorTable[213].color
#define  ANSC_ORANGE4              GrAnsColorTable[214].color
#define  ANSC_SANDYBROWN           GrAnsColorTable[215].color
#define  ANSC_LIGHTSALMON3         GrAnsColorTable[216].color
#define  ANSC_LIGHTPINK3           GrAnsColorTable[217].color
#define  ANSC_PINK2                GrAnsColorTable[218].color
#define  ANSC_PLUM4                GrAnsColorTable[219].color
#define  ANSC_GOLD3                GrAnsColorTable[220].color
#define  ANSC_LIGHTGOLDENROD3      GrAnsColorTable[221].color
#define  ANSC_LIGHTGOLDENROD4      GrAnsColorTable[222].color
#define  ANSC_NAVAJOWHITE2         GrAnsColorTable[223].color
#define  ANSC_MISTYROSE2           GrAnsColorTable[224].color
#define  ANSC_THISTLE2             GrAnsColorTable[225].color
#define  ANSC_YELLOW7              GrAnsColorTable[226].color
#define  ANSC_LIGHTGOLDENROD5      GrAnsColorTable[227].color
#define  ANSC_KHAKI2               GrAnsColorTable[228].color
#define  ANSC_WHEAT2               GrAnsColorTable[229].color
#define  ANSC_CORNSILK             GrAnsColorTable[230].color
#define  ANSC_GREY100              GrAnsColorTable[231].color
#define  ANSC_GREY3                GrAnsColorTable[232].color
#define  ANSC_GREY7                GrAnsColorTable[233].color
#define  ANSC_GREY11               GrAnsColorTable[234].color
#define  ANSC_GREY15               GrAnsColorTable[235].color
#define  ANSC_GREY19               GrAnsColorTable[236].color
#define  ANSC_GREY23               GrAnsColorTable[237].color
#define  ANSC_GREY27               GrAnsColorTable[238].color
#define  ANSC_GREY30               GrAnsColorTable[239].color
#define  ANSC_GREY35               GrAnsColorTable[240].color
#define  ANSC_GREY39               GrAnsColorTable[241].color
#define  ANSC_GREY42               GrAnsColorTable[242].color
#define  ANSC_GREY46               GrAnsColorTable[243].color
#define  ANSC_GREY50               GrAnsColorTable[244].color
#define  ANSC_GREY54               GrAnsColorTable[245].color
#define  ANSC_GREY58               GrAnsColorTable[246].color
#define  ANSC_GREY62               GrAnsColorTable[247].color
#define  ANSC_GREY66               GrAnsColorTable[248].color
#define  ANSC_GREY70               GrAnsColorTable[249].color
#define  ANSC_GREY74               GrAnsColorTable[250].color
#define  ANSC_GREY78               GrAnsColorTable[251].color
#define  ANSC_GREY82               GrAnsColorTable[252].color
#define  ANSC_GREY85               GrAnsColorTable[253].color
#define  ANSC_GREY89               GrAnsColorTable[254].color
#define  ANSC_GREY93               GrAnsColorTable[255].color

// Indexes to ANSI table, valid to init vars,
// but after calling GrGenAnsColorTable you must use
// GrAnsColorTable[ANSI_xxx].color or the macro AnsColor

#define  ANSI_BLACK                0
#define  ANSI_MAROON               1
#define  ANSI_GREEN                2
#define  ANSI_OLIVE                3
#define  ANSI_NAVY                 4
#define  ANSI_PURPLE               5
#define  ANSI_TEAL                 6
#define  ANSI_SILVER               7
#define  ANSI_GREY                 8
#define  ANSI_RED                  9
#define  ANSI_LIME                 10
#define  ANSI_YELLOW               11
#define  ANSI_BLUE                 12
#define  ANSI_FUCHSIA              13
#define  ANSI_AQUA                 14
#define  ANSI_WHITE                15
#define  ANSI_GREY0                16
#define  ANSI_NAVYBLUE             17
#define  ANSI_DARKBLUE             18
#define  ANSI_BLUE2                19
#define  ANSI_BLUE3                20
#define  ANSI_BLUE4                21
#define  ANSI_DARKGREEN            22
#define  ANSI_DEEPSKYBLUE          23
#define  ANSI_DEEPSKYBLUE2         24
#define  ANSI_DEEPSKYBLUE3         25
#define  ANSI_DODGERBLUE           26
#define  ANSI_DODGERBLUE2          27
#define  ANSI_GREEN2               28
#define  ANSI_SPRINGGREEN          29
#define  ANSI_TURQUOISE            30
#define  ANSI_DEEPSKYBLUE4         31
#define  ANSI_DEEPSKYBLUE5         32
#define  ANSI_DODGERBLUE3          33
#define  ANSI_GREEN3               34
#define  ANSI_SPRINGGREEN2         35
#define  ANSI_DARKCYAN             36
#define  ANSI_LIGHTSEAGREEN        37
#define  ANSI_DEEPSKYBLUE6         38
#define  ANSI_DEEPSKYBLUE7         39
#define  ANSI_GREEN4               40
#define  ANSI_SPRINGGREEN3         41
#define  ANSI_SPRINGGREEN4         42
#define  ANSI_CYAN                 43
#define  ANSI_DARKTURQUOISE        44
#define  ANSI_TURQUOISE2           45
#define  ANSI_GREEN5               46
#define  ANSI_SPRINGGREEN5         47
#define  ANSI_SPRINGGREEN6         48
#define  ANSI_MEDIUMSPRINGGREEN    49
#define  ANSI_CYAN2                50
#define  ANSI_CYAN3                51
#define  ANSI_DARKRED              52
#define  ANSI_DEEPPINK             53
#define  ANSI_PURPLE2              54
#define  ANSI_PURPLE3              55
#define  ANSI_PURPLE4              56
#define  ANSI_BLUEVIOLET           57
#define  ANSI_ORANGE               58
#define  ANSI_GREY37               59
#define  ANSI_MEDIUMPURPLE         60
#define  ANSI_SLATEBLUE            61
#define  ANSI_SLATEBLUE2           62
#define  ANSI_ROYALBLUE            63
#define  ANSI_CHARTREUSE           64
#define  ANSI_DARKSEAGREEN         65
#define  ANSI_PALETURQUOISE        66
#define  ANSI_STEELBLUE            67
#define  ANSI_STEELBLUE2           68
#define  ANSI_CORNFLOWERBLUE       69
#define  ANSI_CHARTREUSE2          70
#define  ANSI_DARKSEAGREEN2        71
#define  ANSI_CADETBLUE            72
#define  ANSI_CADETBLUE2           73
#define  ANSI_SKYBLUE              74
#define  ANSI_STEELBLUE3           75
#define  ANSI_CHARTREUSE3          76
#define  ANSI_PALEGREEN            77
#define  ANSI_SEAGREEN             78
#define  ANSI_AQUAMARINE           79
#define  ANSI_MEDIUMTURQUOISE      80
#define  ANSI_STEELBLUE4           81
#define  ANSI_CHARTREUSE4          82
#define  ANSI_SEAGREEN2            83
#define  ANSI_SEAGREEN3            84
#define  ANSI_SEAGREEN4            85
#define  ANSI_AQUAMARINE2          86
#define  ANSI_DARKSLATEGRAY        87
#define  ANSI_DARKRED2             88
#define  ANSI_DEEPPINK2            89
#define  ANSI_DARKMAGENTA          90
#define  ANSI_DARKMAGENTA2         91
#define  ANSI_DARKVIOLET           92
#define  ANSI_PURPLE5              93
#define  ANSI_ORANGE2              94
#define  ANSI_LIGHTPINK            95
#define  ANSI_PLUM                 96
#define  ANSI_MEDIUMPURPLE2        97
#define  ANSI_MEDIUMPURPLE3        98
#define  ANSI_SLATEBLUE3           99
#define  ANSI_YELLOW2              100
#define  ANSI_WHEAT                101
#define  ANSI_GREY53               102
#define  ANSI_LIGHTSLATEGREY       103
#define  ANSI_MEDIUMPURPLE4        104
#define  ANSI_LIGHTSLATEBLUE       105
#define  ANSI_YELLOW3              106
#define  ANSI_DARKOLIVEGREEN       107
#define  ANSI_DARKSEAGREEN3        108
#define  ANSI_LIGHTSKYBLUE         109
#define  ANSI_LIGHTSKYBLUE2        110
#define  ANSI_SKYBLUE2             111
#define  ANSI_CHARTREUSE5          112
#define  ANSI_DARKOLIVEGREEN2      113
#define  ANSI_PALEGREEN2           114
#define  ANSI_DARKSEAGREEN4        115
#define  ANSI_DARKSLATEGRAY2       116
#define  ANSI_SKYBLUE3             117
#define  ANSI_CHARTREUSE6          118
#define  ANSI_LIGHTGREEN           119
#define  ANSI_LIGHTGREEN2          120
#define  ANSI_PALEGREEN3           121
#define  ANSI_AQUAMARINE3          122
#define  ANSI_DARKSLATEGRAY3       123
#define  ANSI_RED2                 124
#define  ANSI_DEEPPINK3            125
#define  ANSI_MEDIUMVIOLETRED      126
#define  ANSI_MAGENTA              127
#define  ANSI_DARKVIOLET2          128
#define  ANSI_PURPLE6              129
#define  ANSI_DARKORANGE           130
#define  ANSI_INDIANRED            131
#define  ANSI_HOTPINK              132
#define  ANSI_MEDIUMORCHID         133
#define  ANSI_MEDIUMORCHID2        134
#define  ANSI_MEDIUMPURPLE5        135
#define  ANSI_DARKGOLDENROD        136
#define  ANSI_LIGHTSALMON          137
#define  ANSI_ROSYBROWN            138
#define  ANSI_GREY63               139
#define  ANSI_MEDIUMPURPLE6        140
#define  ANSI_MEDIUMPURPLE7        141
#define  ANSI_GOLD                 142
#define  ANSI_DARKKHAKI            143
#define  ANSI_NAVAJOWHITE          144
#define  ANSI_GREY69               145
#define  ANSI_LIGHTSTEELBLUE       146
#define  ANSI_LIGHTSTEELBLUE2      147
#define  ANSI_YELLOW4              148
#define  ANSI_DARKOLIVEGREEN3      149
#define  ANSI_DARKSEAGREEN5        150
#define  ANSI_DARKSEAGREEN6        151
#define  ANSI_LIGHTCYAN            152
#define  ANSI_LIGHTSKYBLUE3        153
#define  ANSI_GREENYELLOW          154
#define  ANSI_DARKOLIVEGREEN4      155
#define  ANSI_PALEGREEN4           156
#define  ANSI_DARKSEAGREEN7        157
#define  ANSI_DARKSEAGREEN8        158
#define  ANSI_PALETURQUOISE2       159
#define  ANSI_RED3                 160
#define  ANSI_DEEPPINK4            161
#define  ANSI_DEEPPINK5            162
#define  ANSI_MAGENTA2             163
#define  ANSI_MAGENTA3             164
#define  ANSI_MAGENTA4             165
#define  ANSI_DARKORANGE2          166
#define  ANSI_INDIANRED2           167
#define  ANSI_HOTPINK2             168
#define  ANSI_HOTPINK3             169
#define  ANSI_ORCHID               170
#define  ANSI_MEDIUMORCHID3        171
#define  ANSI_ORANGE3              172
#define  ANSI_LIGHTSALMON2         173
#define  ANSI_LIGHTPINK2           174
#define  ANSI_PINK                 175
#define  ANSI_PLUM2                176
#define  ANSI_VIOLET               177
#define  ANSI_GOLD2                178
#define  ANSI_LIGHTGOLDENROD       179
#define  ANSI_TAN                  180
#define  ANSI_MISTYROSE            181
#define  ANSI_THISTLE              182
#define  ANSI_PLUM3                183
#define  ANSI_YELLOW5              184
#define  ANSI_KHAKI                185
#define  ANSI_LIGHTGOLDENROD2      186
#define  ANSI_LIGHTYELLOW          187
#define  ANSI_GREY84               188
#define  ANSI_LIGHTSTEELBLUE3      189
#define  ANSI_YELLOW6              190
#define  ANSI_DARKOLIVEGREEN5      191
#define  ANSI_DARKOLIVEGREEN6      192
#define  ANSI_DARKSEAGREEN9        193
#define  ANSI_HONEYDEW             194
#define  ANSI_LIGHTCYAN2           195
#define  ANSI_RED4                 196
#define  ANSI_DEEPPINK6            197
#define  ANSI_DEEPPINK7            198
#define  ANSI_DEEPPINK8            199
#define  ANSI_MAGENTA5             200
#define  ANSI_MAGENTA6             201
#define  ANSI_ORANGERED            202
#define  ANSI_INDIANRED3           203
#define  ANSI_INDIANRED4           204
#define  ANSI_HOTPINK4             205
#define  ANSI_HOTPINK5             206
#define  ANSI_MEDIUMORCHID4        207
#define  ANSI_DARKORANGE3          208
#define  ANSI_SALMON               209
#define  ANSI_LIGHTCORAL           210
#define  ANSI_PALEVIOLETRED        211
#define  ANSI_ORCHID2              212
#define  ANSI_ORCHID3              213
#define  ANSI_ORANGE4              214
#define  ANSI_SANDYBROWN           215
#define  ANSI_LIGHTSALMON3         216
#define  ANSI_LIGHTPINK3           217
#define  ANSI_PINK2                218
#define  ANSI_PLUM4                219
#define  ANSI_GOLD3                220
#define  ANSI_LIGHTGOLDENROD3      221
#define  ANSI_LIGHTGOLDENROD4      222
#define  ANSI_NAVAJOWHITE2         223
#define  ANSI_MISTYROSE2           224
#define  ANSI_THISTLE2             225
#define  ANSI_YELLOW7              226
#define  ANSI_LIGHTGOLDENROD5      227
#define  ANSI_KHAKI2               228
#define  ANSI_WHEAT2               229
#define  ANSI_CORNSILK             230
#define  ANSI_GREY100              231
#define  ANSI_GREY3                232
#define  ANSI_GREY7                233
#define  ANSI_GREY11               234
#define  ANSI_GREY15               235
#define  ANSI_GREY19               236
#define  ANSI_GREY23               237
#define  ANSI_GREY27               238
#define  ANSI_GREY30               239
#define  ANSI_GREY35               240
#define  ANSI_GREY39               241
#define  ANSI_GREY42               242
#define  ANSI_GREY46               243
#define  ANSI_GREY50               244
#define  ANSI_GREY54               245
#define  ANSI_GREY58               246
#define  ANSI_GREY62               247
#define  ANSI_GREY66               248
#define  ANSI_GREY70               249
#define  ANSI_GREY74               250
#define  ANSI_GREY78               251
#define  ANSI_GREY82               252
#define  ANSI_GREY85               253
#define  ANSI_GREY89               254
#define  ANSI_GREY93               255

#define AnsColor(i) (GrAnsColorTable[i].color)

#ifdef __cplusplus
}
#endif

#endif /* whole file */
