/*
    Copyright (c) 2014, Lukas Holecek <hluk@email.cz>

    This file is part of CopyQ.

    CopyQ is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    CopyQ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with CopyQ.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "itemimage.h"
#include "ui_itemimagesettings.h"

#include "common/contenttype.h"
#include "item/itemeditor.h"

#include <QHBoxLayout>
#include <QModelIndex>
#include <QPixmap>
#include <QtPlugin>
#include <QVariant>

namespace {

QString findImageFormat(const QList<QString> &formats)
{
    // Check formats in this order.
    static const QStringList imageFormats = QStringList()
            << QString("image/svg+xml")
            << QString("image/png")
            << QString("image/bmp")
            << QString("image/jpeg")
            << QString("image/gif");

    foreach (const QString &format, imageFormats) {
        if ( formats.contains(format) )
            return format;
    }

    return QString();
}

bool getImageData(const QModelIndex &index, QByteArray *data, QString *mime)
{
    QVariantMap dataMap = index.data(contentType::data).toMap();

    *mime = findImageFormat(dataMap.keys());
    if ( mime->isEmpty() )
        return false;

    *data = dataMap[*mime].toByteArray();

    return true;
}

bool getPixmapFromData(const QModelIndex &index, QPixmap *pix)
{
    QString mime;
    QByteArray data;
    if ( !getImageData(index, &data, &mime) )
        return false;

    pix->loadFromData( data, mime.toLatin1() );

    return true;
}

} // namespace

ItemImage::ItemImage(const QPixmap &pix, const QString &imageEditor, const QString &svgEditor,
                     QWidget *parent)
    : QLabel(parent)
    , ItemWidget(this)
    , m_editor(imageEditor)
    , m_svgEditor(svgEditor)
{
    setMargin(4);
    setPixmap(pix);
}

QObject *ItemImage::createExternalEditor(const QModelIndex &index, QWidget *parent) const
{
    QString mime;
    QByteArray data;
    if ( !getImageData(index, &data, &mime) )
        return NULL;

    const QString &cmd = mime.contains("svg") ? m_svgEditor : m_editor;

    return cmd.isEmpty() ? NULL : new ItemEditor(data, mime, cmd, parent);
}

ItemImageLoader::ItemImageLoader()
{
}

ItemImageLoader::~ItemImageLoader()
{
}

ItemWidget *ItemImageLoader::create(const QModelIndex &index, QWidget *parent) const
{
    // TODO: Just check if image provided and load it in different thread.
    QPixmap pix;
    if ( !getPixmapFromData(index, &pix) )
        return NULL;

    // scale pixmap
    const int w = m_settings.value("max_image_width", 320).toInt();
    const int h = m_settings.value("max_image_height", 240).toInt();
    if ( w > 0 && pix.width() > w && (h <= 0 || pix.width()/w > pix.height()/h) ) {
        pix = pix.scaledToWidth(w);
    } else if (h > 0 && pix.height() > h) {
        pix = pix.scaledToHeight(h);
    }

    return new ItemImage(pix, m_settings.value("image_editor").toString(),
                         m_settings.value("svg_editor").toString(), parent);
}

QStringList ItemImageLoader::formatsToSave() const
{
    return QStringList("image/svg+xml") << QString("image/bmp") << QString("image/png")
                                        << QString("image/jpeg") << QString("image/gif");
}

QVariantMap ItemImageLoader::applySettings()
{
    m_settings["max_image_width"] = ui->spinBoxImageWidth->value();
    m_settings["max_image_height"] = ui->spinBoxImageHeight->value();
    m_settings["image_editor"] = ui->lineEditImageEditor->text();
    m_settings["svg_editor"] = ui->lineEditSvgEditor->text();
    return m_settings;
}

QWidget *ItemImageLoader::createSettingsWidget(QWidget *parent)
{
    ui.reset(new Ui::ItemImageSettings);
    QWidget *w = new QWidget(parent);
    ui->setupUi(w);
    ui->spinBoxImageWidth->setValue( m_settings.value("max_image_width", 320).toInt() );
    ui->spinBoxImageHeight->setValue( m_settings.value("max_image_height", 240).toInt() );
    ui->lineEditImageEditor->setText( m_settings.value("image_editor", "").toString() );
    ui->lineEditSvgEditor->setText( m_settings.value("svg_editor", "").toString() );
    return w;
}

Q_EXPORT_PLUGIN2(itemimage, ItemImageLoader)
