/*
 * @(#)MouseEvent.java	1.34 06/10/10
 *
 * Copyright  1990-2008 Sun Microsystems, Inc. All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 only, as published by the Free Software Foundation. 
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License version 2 for more details (a copy is
 * included at /legal/license.txt). 
 * 
 * You should have received a copy of the GNU General Public License
 * version 2 along with this work; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA 
 * 
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 or visit www.sun.com if you need additional
 * information or have any questions. 
 *
 */

package jail.java.awt.event;

import jail.java.awt.Component;
import jail.java.awt.Point;
import java.io.IOException;
import java.io.ObjectInputStream;

/**
 * An event which indicates that a mouse action occurred in a component. A mouse
 * action is considered to occur in a particular component if and only if the
 * mouse cursor is over the unobscured part of the component's bounds when the
 * action happens. Component bounds can be obscurred by the visible component's
 * children or by a menu or by a top-level window. This event is used both for
 * mouse events (click, enter, exit) and mouse motion events (moves and drags).
 * <P>
 * This low-level event is generated by a component object for:
 * <ul>
 * <li>Mouse Events
 * <ul>
 * <li>a mouse button is pressed
 * <li>a mouse button is released
 * <li>a mouse button is clicked (pressed and released)
 * <li>the mouse cursor enters the unobscured part of component's geometry
 * <li>the mouse cursor exits the unobscured part of component's geometry
 * </ul>
 * <li>Mouse Motion Events
 * <ul>
 * <li>the mouse is moved
 * <li>the mouse is dragged
 * </ul>
 * </ul>
 * <P>
 * A <code>MouseEvent</code> object is passed to every
 * <code>MouseListener</code> or <code>MouseAdapter</code> object which is
 * registered to receive the "interesting" mouse events using the component's
 * <code>addMouseListener</code> method. (<code>MouseAdapter</code> objects
 * implement the <code>MouseListener</code> interface.) Each such listener
 * object gets a <code>MouseEvent</code> containing the mouse event.
 * <P>
 * A <code>MouseEvent</code> object is also passed to every
 * <code>MouseMotionListener</code> or <code>MouseMotionAdapter</code> object
 * which is registered to receive mouse motion events using the component's
 * <code>addMouseMotionListener</code> method. (<code>MouseMotionAdapter</code>
 * objects implement the <code>MouseMotionListener</code> interface.) Each such
 * listener object gets a <code>MouseEvent</code> containing the mouse motion
 * event.
 * <P>
 * When a mouse button is clicked, events are generated and sent to the
 * registered <code>MouseListener</code>s. The state of modal keys can be
 * retrieved using {@link InputEvent#getModifiers} and
 * {@link InputEvent#getModifiersEx}. The button mask returned by
 * {@link InputEvent#getModifiers} reflects only the button that changed state,
 * not the current state of all buttons. To get state of all buttons use
 * {@link InputEvent#getModifiersEx} instead. The button which has changed state
 * is returned by {@link MouseEvent#getButton}
 * <P>
 * For example, if the first mouse button is pressed, events are sent in the
 * following order:
 * 
 * <PRE>
 *    <b   >id           </b   >   <b   >modifiers   </b   > <b   >button </b   >
 *    <code>MOUSE_PRESSED</code>:  <code>BUTTON1_MASK</code> <code>BUTTON1</code>
 *    <code>MOUSE_RELEASED</code>: <code>BUTTON1_MASK</code> <code>BUTTON1</code>
 *    <code>MOUSE_CLICKED</code>:  <code>BUTTON1_MASK</code> <code>BUTTON1</code>
 * </PRE>
 * 
 * When multiple mouse buttons are pressed, each press, release, and click
 * results in a separate event.
 * <P>
 * For example, if the user presses <b>button 1</b> followed by <b>button 2</b>,
 * and then releases them in the same order, the following sequence of events is
 * generated:
 * 
 * <PRE>
 *    <b   >id           </b   >   <b   >modifiers   </b   > <b   >button </b   >
 *    <code>MOUSE_PRESSED</code>:  <code>BUTTON1_MASK</code> <code>BUTTON1</code>
 *    <code>MOUSE_PRESSED</code>:  <code>BUTTON2_MASK</code> <code>BUTTON2</code>
 *    <code>MOUSE_RELEASED</code>: <code>BUTTON1_MASK</code> <code>BUTTON1</code>
 *    <code>MOUSE_CLICKED</code>:  <code>BUTTON1_MASK</code> <code>BUTTON1</code>
 *    <code>MOUSE_RELEASED</code>: <code>BUTTON2_MASK</code> <code>BUTTON2</code>
 *    <code>MOUSE_CLICKED</code>:  <code>BUTTON2_MASK</code> <code>BUTTON2</code>
 * </PRE>
 * 
 * If <b>button 2</b> is released first, the <code>MOUSE_RELEASED</code>/
 * <code>MOUSE_CLICKED</code> pair for <code>BUTTON2_MASK</code> arrives first,
 * followed by the pair for <code>BUTTON1_MASK</code>.
 * <p>
 *
 * <code>MOUSE_DRAGGED</code> events are delivered to the <code>Component</code>
 * in which the mouse button was pressed until the mouse button is released
 * (regardless of whether the mouse position is within the bounds of the
 * <code>Component</code>). Due to platform-dependent Drag&Drop implementations,
 * <code>MOUSE_DRAGGED</code> events may not be delivered during a native
 * Drag&Drop operation.
 *
 * In a multi-screen environment mouse drag events are delivered to the
 * <code>Component</code> even if the mouse position is outside the bounds of
 * the <code>GraphicsConfiguration</code> associated with that
 * <code>Component</code>. However, the reported position for mouse drag events
 * in this case may differ from the actual mouse position:
 * <ul>
 * <li>In a multi-screen environment without a virtual device: <br>
 * The reported coordinates for mouse drag events are clipped to fit within the
 * bounds of the <code>GraphicsConfiguration</code> associated with the
 * <code>Component</code>.
 * <li>In a multi-screen environment with a virtual device: <br>
 * The reported coordinates for mouse drag events are clipped to fit within the
 * bounds of the virtual device associated with the <code>Component</code>.
 * </ul>
 *
 * @author Carl Quinn 1.21 02/20/02
 *
 * @see MouseAdapter
 * @see MouseListener
 * @see MouseMotionAdapter
 * @see MouseMotionListener
 * @see MouseWheelListener
 * @see <a href=
 *      "http://java.sun.com/docs/books/tutorial/post1.0/ui/mouselistener.html">
 *      Tutorial: Writing a Mouse Listener</a>
 * @see <a href=
 *      "http://java.sun.com/docs/books/tutorial/post1.0/ui/mousemotionlistener.html">
 *      Tutorial: Writing a Mouse Motion Listener</a>
 * @see <a href="http://www.awl.com/cp/javaseries/jcl1_2.html">Reference: The
 *      Java Class Libraries (update file)</a>
 *
 * @since 1.1
 */
public class MouseEvent extends InputEvent {
    /**
     * The first number in the range of ids used for mouse events.
     */
    public static final int MOUSE_FIRST = 500;
    /**
     * The last number in the range of ids used for mouse events.
     */
    public static final int MOUSE_LAST = 507;
    /**
     * The "mouse clicked" event. This <code>MouseEvent</code> occurs when a
     * mouse button is pressed and released.
     */
    public static final int MOUSE_CLICKED = MOUSE_FIRST;
    /**
     * The "mouse pressed" event. This <code>MouseEvent</code> occurs when a
     * mouse button is pushed down.
     */
    public static final int MOUSE_PRESSED = 1 + MOUSE_FIRST; //Event.MOUSE_DOWN
    /**
     * The "mouse released" event. This <code>MouseEvent</code> occurs when a
     * mouse button is let up.
     */
    public static final int MOUSE_RELEASED = 2 + MOUSE_FIRST; //Event.MOUSE_UP
    /**
     * The "mouse moved" event. This <code>MouseMotionEvent</code> occurs when
     * the mouse position changes.
     */
    public static final int MOUSE_MOVED = 3 + MOUSE_FIRST; //Event.MOUSE_MOVE
    /**
     * The "mouse entered" event. This <code>MouseEvent</code> occurs when the
     * mouse cursor enters the unobscured part of component's geometry.
     */
    public static final int MOUSE_ENTERED = 4 + MOUSE_FIRST; //Event.MOUSE_ENTER
    /**
     * The "mouse exited" event. This <code>MouseEvent</code> occurs when the
     * mouse cursor exits the unobscured part of component's geometry.
     */
    public static final int MOUSE_EXITED = 5 + MOUSE_FIRST; //Event.MOUSE_EXIT

    /**
     * The "mouse dragged" event. This <code>MouseMotionEvent</code> occurs when
     * the mouse position changes while a mouse button is pressed.
     */
    public static final int MOUSE_DRAGGED = 6 + MOUSE_FIRST; //Event.MOUSE_DRAG

    /**
     * The mouse event's x coordinate. The x value is relative to the component
     * that fired the event.
     *
     * @serial
     * @see #getX()
     */
    int x;
    /**
     * The mouse event's y coordinate. The y value is relative to the component
     * that fired the event.
     *
     * @serial
     * @see #getY()
     */
    int y;
    /**
     * Indicates the number of quick consecutive clicks of a mouse button.
     * clickCount will be valid for only three mouse events :<BR>
     * <code>MOUSE_CLICKED</code>, <code>MOUSE_PRESSED</code> and
     * <code>MOUSE_RELEASED</code>. For the above, the <code>clickCount</code>
     * will be at least 1. For all other events the count will be 0.
     *
     * @serial
     * @see #getClickCount().
     */
    int clickCount;
    /**
     * Indicates which, if any, of the mouse buttons has changed state.
     *
     * The only legal values are the following constants: <code>NOBUTTON</code>,
     * <code>BUTTON1</code>, <code>BUTTON2</code> or <code>BUTTON3</code>.
     * 
     * @serial
     * @see #getButton().
     */
    int button;
    /**
     * A property used to indicate whether a Popup Menu should appear with a
     * certain gestures. If <code>popupTrigger</code> = <code>false</code>, no
     * popup menu should appear. If it is <code>true</code> then a popup menu
     * should appear.
     *
     * @serial
     * @see java.awt.PopupMenu
     * @see #isPopupTrigger()
     */
    boolean popupTrigger = false;
    /*
     * JDK 1.1 serialVersionUID
     */
    private static final long serialVersionUID = -991214153494842848L;

    /**
     * Constructs a <code>MouseEvent</code> object with the specified source
     * component, type, modifiers, coordinates, and click count.
     * <p>
     * Note that passing in an invalid <code>id</code> results in unspecified
     * behavior.
     *
     * @param source
     *            the <code>Component</code> that originated the event
     * @param id
     *            the integer that identifies the event
     * @param when
     *            a long int that gives the time the event occurred
     * @param modifiers
     *            the modifier keys down during event (e.g. shift, ctrl, alt,
     *            meta) Either extended _DOWN_MASK or old _MASK modifiers should
     *            be used, but both models should not be mixed in one event. Use
     *            of the extended modifiers is preferred.
     * @param x
     *            the horizontal x coordinate for the mouse location
     * @param y
     *            the vertical y coordinate for the mouse location
     * @param clickCount
     *            the number of mouse clicks associated with event
     * @param popupTrigger
     *            a boolean, true if this event is a trigger for a popup menu
     */
    public MouseEvent(Component source, int id, long when, int modifiers, int x, int y, int clickCount,
            boolean popupTrigger) {
        super(source, id, when, modifiers);
        this.x = x;
        this.y = y;
        this.clickCount = clickCount;
        this.popupTrigger = popupTrigger;
    }

    /**
     * Returns the horizontal x position of the event relative to the source
     * component.
     *
     * @return x an integer indicating horizontal position relative to the
     *         component
     */
    public int getX() {
        return x;
    }

    /**
     * Returns the vertical y position of the event relative to the source
     * component.
     *
     * @return y an integer indicating vertical position relative to the
     *         component
     */
    public int getY() {
        return y;
    }

    /**
     * Returns the x,y position of the event relative to the source component.
     *
     * @return a <code>Point</code> object containing the x and y coordinates
     *         relative to the source component
     *
     */
    public Point getPoint() {
        int x;
        int y;
        synchronized (this) {
            x = this.x;
            y = this.y;
        }
        return new Point(x, y);
    }

    /**
     * Translates the event's coordinates to a new position by adding specified
     * <code>x</code> (horizontal) and <code>y</code> (vertical) offsets.
     *
     * @param x
     *            the horizontal x value to add to the current x coordinate
     *            position
     * @param y
     *            the vertical y value to add to the current y coordinate
     *            position
     */
    public synchronized void translatePoint(int x, int y) {
        this.x += x;
        this.y += y;
    }

    /**
     * Returns the number of mouse clicks associated with this event.
     *
     * @return integer value for the number of clicks
     */
    public int getClickCount() {
        return clickCount;
    }

    /**
     * Returns whether or not this mouse event is the popup menu trigger event
     * for the platform.
     * <p>
     * <b>Note</b>: Popup menus are triggered differently on different systems.
     * Therefore, <code>isPopupTrigger</code> should be checked in both
     * <code>mousePressed</code> and <code>mouseReleased</code> for proper
     * cross-platform functionality.
     *
     * @return boolean, true if this event is the popup menu trigger for this
     *         platform
     */
    public boolean isPopupTrigger() {
        return popupTrigger;
    }

    /**
     * Returns a parameter string identifying this event. This method is useful
     * for event-logging and for debugging.
     *
     * @return a string identifying the event and its attributes
     */
    public String paramString() {
        String str;
        switch (id) {
        case MOUSE_PRESSED:
            str = "MOUSE_PRESSED";
            break;

        case MOUSE_RELEASED:
            str = "MOUSE_RELEASED";
            break;

        case MOUSE_CLICKED:
            str = "MOUSE_CLICKED";
            break;

        case MOUSE_ENTERED:
            str = "MOUSE_ENTERED";
            break;

        case MOUSE_EXITED:
            str = "MOUSE_EXITED";
            break;

        case MOUSE_MOVED:
            str = "MOUSE_MOVED";
            break;

        case MOUSE_DRAGGED:
            str = "MOUSE_DRAGGED";
            break;

        default:
            str = "unknown type";
        }
        str += ",(" + x + "," + y + ")" + modifiers;
        return str + ",clickCount=" + clickCount;
    }

    /**
     * Sets new modifiers by the old ones. The mouse modifiers override
     * overlaping key modifiers.
     * 
     * @serial
     */
    private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException {
        s.defaultReadObject();
    }
}
