/*
 *      NiceShaper - Dynamic Traffic Management
 *
 *      Copyright 2004 Mariusz Jedwabny <mariusz@jedwabny.net>
 *
 *      This file is subject to the terms and conditions of the GNU General Public
 *      License.  See the file COPYING in the main directory of this archive for
 *      more details.
 */

#include "class.h"

#include <cstdlib>

#include <iostream>
#include <string>
#include <vector>

#include "main.h"
#include "log.h"
#include "filter.h"
#include "sys.h"
#include "ifaces.h"
#include "tests.h"
#include "aux.h"

using std::string;
using std::vector;

using namespace aux;

NsClass::NsClass(string ipt_my_chain, string ipt_section_hook)
{
    Dev = "";
    Name = "";
    EsfqHash = "classic";
    IptMyChain = ipt_my_chain;
    IptSectionHook = ipt_section_hook;
    TcClassAdd = "";
    TcQdiscAdd = "";
    TcClassDel = "";
    TcQdiscDel = "";
    DevId = 0;
    ClassId = 0;
    Alive = 0;
    Hold = 30;
    NsLow = 0;
    NsCeil = 0;
    HtbParentId = g_section_id;
    HtbRate = 0;
    HtbCeil = 0;
    HtbPrio = 5;
    HtbBurst = 0;
    HtbCBurst = 0;
    OldHtbCeil = 0;
    OldHtbRate = 0;
    MeasureRate = 0;
    Rate = 0;
    SfqPerturb = 10;
    EsfqPerturb = 10;
    HardLimit = 0;
    SoftLimit = 0;
    SectionShape = 0;
    Strict = 0.70;
    UseIpt = true;
    UseTcClass = true;
    UseTcFilter = true;
    SumInSection = true;
    StatsShowHtbCeil = true;
    StatsShowRate = true;
    Work = false;
    Qdisc = SFQ;
}

NsClass::~NsClass()
{
    for ( unsigned int n=0; n<TcFilters.size(); n++ ) delete TcFilters[n];

    TcFilters.clear();
}

int NsClass::store(string buf)
{
    string option, param, value;
    
    if (buf.empty()) return 0;      

    option = awk( buf, 1 );  
    param = awk( buf, 2 );  
    value = awk( buf, 3 );  
    
    if (option == "class") 
    {
        Dev = trim_dev(awk(buf, 3));
        Name = awk(buf, 4);

        if (!ifaces->isValidSysDev(Dev)) { log.error (16, buf); return -1; }

        DevId = ifaces->index(Dev);

        if (!Name.size() || awk(buf, 5).size()) { log.error (24, buf); return -1; }
    }
    else if (option == "match") TcFilters.push_back (new TcFilter (buf, Dev, IptMyChain, IptSectionHook));
    else if (option == "classid") { ClassId = str_to_uint (param); }
    else if (option == "low") NsLow = unit_convert (param, BITS );
    else if (option == "ceil") HtbCeil = NsCeil = unit_convert (param, BITS );
    else if (option == "rate") { 
        NsLow = unit_convert (param, BITS); 
        HtbCeil = NsCeil = unit_convert (param, BITS);
    } 
    else if (option == "strict")
        Strict = (double) (str_to_uint (param))/100;
    else if (option == "hold") 
        Hold = str_to_uint (param);
    else if (option == "htb") {
        if (param == "prio") {
            HtbPrio = str_to_uint (value);
        }    
        else if (param == "scheduler") {
            if (value == "sfq") Qdisc = SFQ;
            else if (value == "esfq") Qdisc = ESFQ;
            else if (value == "no") { 
                Qdisc = NOQDISC;
            }
            else if ((value == "none") || (value == "false")) { 
                Qdisc = NOQDISC;
                log.warning(9, buf);
            }
            else { log.error (17, buf); return -1; }
        }   
        else if (param == "burst") {
            HtbBurst = unit_convert (value, BYTES);
        }
        else if (param == "cburst") {
            HtbCBurst = unit_convert (value, BYTES);
        }
        else {
            log.error (11, buf);
            return -1;
        }        
    }
    else if (option == "sfq")
    {
        if (param == "perturb") SfqPerturb = str_to_uint (value);  
        else { log.error (11, buf); return -1; }    
    }
    else if (option == "esfq") {
        if (param == "perturb") {
            EsfqPerturb = str_to_uint (value);  
        }
        else if (param == "hash") {
            if (value == "classic") EsfqHash = "classic";
            else if (value == "dst") EsfqHash = "dst";
            else if (value == "src") EsfqHash = "src";
            else { log.error (11, buf); return -1; }
        }
        else { log.error (11, buf); return -1; }
    }
    else if (option == "alter") {
        Alter.store(buf);
    }
    else if (option == "quota") {
        Quota.store(buf);
    }
    //    else if (option == "show")
    //    {
    //  if (value == "stats") 
    //  {
    //      stats_show_htb_ceil = false; stats_show_rate = false;
    //  }
    //    }
    else if (option == "type") {
        if (param == "standard-class")  
        {
            StatsShowHtbCeil = true; StatsShowRate = true;
            UseTcClass = true; UseTcFilter = true; SumInSection = true;
        }
        else if (param == "wrapper")
        {
            StatsShowHtbCeil = true; StatsShowRate = true;
            UseTcClass = true; UseTcFilter = true; SumInSection = false;
            NsLow = NsCeil;
            HtbParentId = ifaces->htbDNWrapperId();
        }
        else if (param == "do-not-shape")
        {
            StatsShowHtbCeil = false; StatsShowRate = true;
            UseTcClass = false; UseTcFilter = true; SumInSection = false;
            NsLow = 0; NsCeil = 0; HtbRate = 0; HtbCeil = 0;
        } 
        else if (param == "virtual") 
        {
            StatsShowHtbCeil = false; StatsShowRate = true;
            UseTcClass = false; UseTcFilter = false; SumInSection = false;
            NsLow = 0; NsCeil = 0; HtbRate = 0; HtbCeil = 0;
        } 
        else { log.error (11, buf); return -1; }
    }
    else if (option == "set-mark") 
    { } 

    if (option != "match") {
        for (unsigned int i=0; i<TcFilters.size(); i++) {
            if (TcFilters[i]->store(buf) == -1) return -1;
        }
    }

    return 1;
}

int NsClass::validateParams(unsigned int section_shape)
{
    if ((NsLow > MAX_RATE) || (NsLow && (NsLow < MIN_RATE))) { log.error (806); return -1; }
    if ((NsCeil > MAX_RATE) || (NsCeil && (NsCeil < MIN_RATE))) { log.error (806); return -1; }
    if ((Strict > 1) || (Strict < 0)) { log.error(807, ""); return -1; }

    if (!NsLow) NsLow = MIN_RATE;
    if (!NsCeil ) NsCeil = section_shape; 
    if ((HtbParentId == g_section_id) && (NsCeil > section_shape)) NsCeil = section_shape;
    if (NsLow > NsCeil ) NsLow = NsCeil;
    
    return 0;
}

int NsClass::prepareIptRules(bool apply, vector <string> &ipt_rules)
{
    for (unsigned int i=0; i < TcFilters.size(); i++) 
    {
        if (TcFilters[i]->prepareIptFilter() == -1) return -1;
        if (TcFilters[i]->prepareIptRules(apply, ipt_rules) == -1) return -1;
    }
    return 0;
}

int NsClass::prepareTcClass( unsigned int section_shape, unsigned int section_classes )
{
    unsigned int htb_rate = section_shape / section_classes; 

    SectionShape = section_shape;

    if ( NsLow > NsCeil ) NsCeil = NsLow;
    if ( UseTcClass ) 
    {
        if ( !SumInSection ) htb_rate = HtbCeil;
        TcClassAdd = "tc class add dev " + Dev 
            + " parent 1:" + int_to_hex(HtbParentId) + " classid 1:" + int_to_hex(ClassId) 
            + " htb rate " + int_to_str(htb_rate) + " ceil " + int_to_str(htb_rate) 
            + " prio " +  int_to_str(HtbPrio) + " quantum " + int_to_str(compute_quantum(htb_rate));
        if (HtbBurst) TcClassAdd += (" burst " + int_to_str(HtbBurst));
        if (HtbCBurst) TcClassAdd += (" cburst " + int_to_str(HtbCBurst));
        TcClassDel = "tc class del dev " + Dev + " classid 1:" + int_to_hex(ClassId);
        if ( Qdisc == SFQ )
        {
            TcQdiscAdd = "tc qdisc add dev " + Dev + " parent 1:" + int_to_hex(ClassId) + " handle " + int_to_hex(ClassId) + ": sfq perturb " + int_to_str(SfqPerturb);
            TcQdiscDel = "tc qdisc del dev " + Dev + " parent 1:" + int_to_hex(ClassId);
        }
        if ( Qdisc == ESFQ )
        {
            TcQdiscAdd = "tc qdisc add dev " + Dev + " parent 1:" + int_to_hex(ClassId) + " handle " + int_to_hex(ClassId) + ": esfq perturb " + int_to_str(EsfqPerturb) + " hash " + EsfqHash;
            TcQdiscDel = "tc qdisc del dev " + Dev + " parent 1:" + int_to_hex(ClassId);
        }
        else if ( Qdisc == NOQDISC )
        {
            TcQdiscAdd = "";
            TcQdiscDel = "";
        }
    }
    if ( UseTcFilter ) 
    {
        for ( unsigned int i = 0; i < TcFilters.size(); i++ )
        {
            if ( TcFilters[i]->prepareTcFilter() == -1 ) return -1;
        }
    }

    return 1;
}

unsigned int NsClass::check(vector < string > &iptables_summary, struct timeval now, double cycle_time)
{
    unsigned int cycle_bytes = 0;

    Rate = MeasureRate = 0;

    for (int i = TcFilters.size()-1 ; i >= 0 ; i--) cycle_bytes += TcFilters[i]->check(iptables_summary);
    Rate = MeasureRate = (unsigned int)((double) cycle_bytes / cycle_time);

    if (Work) {
        if (Rate) Alive = now.tv_sec;
        else if (((now.tv_sec - Alive) >= Hold) && (Hold)) del();
    }
    else if (Rate) { 
        add(); 
        Alive = now.tv_sec; 
    }

    if (SumInSection) {
        if (Rate > HtbCeil) Rate = HtbCeil;
        Quota.totalize(cycle_bytes);
        proceedTriggers (now);
        return Rate;
    }
    else return 0;
}

int NsClass::proceedTriggers (struct timeval now)
{
    unsigned int dmin;
    unsigned int wday;
    unsigned int mday;
    bool mday_last = false;
    int trigger_state;
    struct tm *ltime;

    ltime = localtime(&now.tv_sec);
    dmin = ltime->tm_hour*60+ltime->tm_min;
    wday = ltime->tm_wday;
    mday = ltime->tm_mday;
    now.tv_sec += 86400;
    ltime = localtime(&now.tv_sec);
    if (ltime->tm_mday == 1) mday_last = true;

    // Check alter trigger
    trigger_state = Alter.check (dmin);
    if ((trigger_state == 1) || (trigger_state == 2)) {
        // Replace (A<=>Q);
        if (Quota.isActive() && Alter.isUseNsLow() && Quota.isUseNsLow()) shift (Alter.getTriggerNsLowRef(), Quota.getTriggerNsLowRef());
        else if (Quota.isActive() && Alter.isUseNsCeil() && Quota.isUseNsCeil()) shift (Alter.getTriggerNsCeilRef(), Quota.getTriggerNsCeilRef());
        else {
            // Replace (A<=>0);
            if (Alter.isUseNsLow()) shift (Alter.getTriggerNsLowRef(), NsLow);
            if (Alter.isUseNsCeil()) shift (Alter.getTriggerNsCeilRef(), NsCeil);
        }
    }

    // Check quota trigger
    trigger_state = Quota.check (dmin, wday, mday, mday_last);
    if ((trigger_state == 1) || (trigger_state == 2)) {
        // Replace (Q<=>0)
        if (Quota.isUseNsLow()) shift (Quota.getTriggerNsLowRef(), NsLow);
        if (Quota.isUseNsCeil()) shift (Quota.getTriggerNsCeilRef(), NsCeil);
    }

    return 0;
}

/*unsigned int NsClass::expect_rate( unsigned int expect_rate )
  {
  if ( sum_in_section ) 
  { 
  if ( rate < expect_rate ) 
  {
  unsigned int inc = 4;
  return ( rate + (power(100*rate/htb_ceil,inc) / power(100,inc-1)) * ((expect_rate-htb_ceil)/100) );
  }
  else 
  {
  return expect_rate; 
  }
  }
  else return 0;
  }*/

int NsClass::add()
{
    Work = true;
    HtbRate = 0;
    HtbCeil = NsCeil;
    OldHtbCeil = HtbCeil;
    OldHtbRate = 0;

    if (UseTcClass) {
        if (!g_fallback_to_tc) {
            sys->tcClass(TC_ADD, DevId, HtbParentId, ClassId, HtbCeil, HtbCeil, HtbPrio, compute_quantum(HtbCeil), HtbBurst, HtbCBurst);
            if (Qdisc == ESFQ) sys->tc(TcQdiscAdd);
            else if (Qdisc != NOQDISC) sys->tcQdisc(TC_ADD, DevId, ClassId, ClassId, Qdisc, SfqPerturb);
        }
        else {
            sys->tc(TcClassAdd);
            if (Qdisc != NOQDISC) sys->tc(TcQdiscAdd);
        }
    }

    if (UseTcFilter) for (unsigned int i = 0; i < TcFilters.size(); i++) TcFilters[i]->add();

    return 1;
}

int NsClass::del()
{
    Work = false;

    if (UseTcFilter) for (unsigned int i = 0; i < TcFilters.size(); i++) TcFilters[i]->del();

    if (UseTcClass) {
        if (!g_fallback_to_tc) {
            if (Qdisc != NOQDISC) sys->tcQdisc(TC_DEL, DevId, ClassId, ClassId, Qdisc, 0);
            sys->tcClass(TC_DEL, DevId, HtbParentId, ClassId, HtbRate, HtbCeil, HtbPrio, compute_quantum(HtbCeil), HtbBurst, HtbCBurst);
        }
        else {
            if (Qdisc != NOQDISC) sys->tc(TcQdiscDel);
            sys->tc(TcClassDel);
        }
    }

    return 1;
}

int NsClass::computeTcChanges(unsigned int working)
{
    if ( !Work ) return 0;
    if ( !SumInSection ) return 0;

    if ( HtbCeil > NsCeil ) HtbCeil = NsCeil;
    if ( HtbCeil < NsLow ) HtbCeil = NsLow;

    HtbRate = SectionShape / working; 

    if ( HtbCeil < HtbRate ) HtbRate = HtbCeil;

    if (( OldHtbCeil == HtbCeil ) && ( OldHtbRate == HtbRate )) return 0;

    OldHtbRate = HtbRate;
    OldHtbCeil = HtbCeil;

    return 1;
}

int NsClass::applyChanges(unsigned int working)
{
    std::string buf = "";

    if (!computeTcChanges(working)) return 0;

    if (!g_fallback_to_tc) {
        sys->tcClass(TC_MOD, DevId, HtbParentId, ClassId, HtbRate, HtbCeil, HtbPrio, compute_quantum(HtbCeil), HtbBurst, HtbCBurst);
    } 
    else {
        buf = "tc class change dev " + Dev + " classid 1:" + int_to_hex(ClassId) 
                + " htb rate " + int_to_str(HtbRate) + " ceil " + int_to_str(HtbCeil)
                + " prio " + int_to_str(HtbPrio) + "  quantum "+ int_to_str(compute_quantum(HtbCeil));
        if (HtbBurst) buf += (" burst " + int_to_str(HtbBurst));
        if (HtbCBurst) buf += (" cburst " + int_to_str(HtbCBurst));
        sys->tc(buf);
    }

    return 1;
}

string NsClass::stats()
{
    string result;

    if (( g_stats_classes == SC_ALL ) || 
            (( g_stats_classes == SC_ACTIVE ) && MeasureRate ) ||
            (( g_stats_classes == SC_WORKING ) && Work )) {
        //
    }   
    else return "";

    if ( StatsShowHtbCeil && StatsShowRate ) result = Name + " " + int_to_str(HtbCeil) + " " + int_to_str(OldHtbCeil) + " " + int_to_str(MeasureRate);
    else if ( StatsShowHtbCeil ) result = Name + " " + int_to_str(HtbCeil) + " " + int_to_str(OldHtbCeil) + " -";
    else if ( StatsShowRate ) result = Name + " - - " + int_to_str(MeasureRate);
    else result = Name + " - - -";

    return result;
}

std::string NsClass::dumpQuotaCounters()
{
    std::string counters = Quota.dumpCounters();

    if (counters.size()) return (Name + " " + counters);
    return "";
}

void NsClass::setQuotaCounters(unsigned int counter_day, unsigned int counter_week, unsigned int counter_month)
{
    Quota.setCounters (counter_day, counter_week, counter_month);

    return;
}

