/*
 *      NiceShaper - Dynamic Traffic Management
 *
 *      Copyright 2004 Mariusz Jedwabny <mariusz@jedwabny.net>
 *
 *      This file is subject to the terms and conditions of the GNU General Public
 *      License.  See the file COPYING in the main directory of this archive for
 *      more details.
 */   

#include "instance.h"

#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>

#include "main.h"
#include "aux.h"
#include "niceshaper.h"
#include "talk.h"

using std::string;
using std::vector;

using namespace aux;

Instance::Instance(bool &i_am_supervisor)
{
    int fd1[2], fd2[2];

    ResultsCounter = 0;

    pipe(fd1);
    pipe(fd2);

    if ((Pid = fork())) {
        i_am_supervisor = true;
        close (fd1[1]);
        close (fd2[0]);
        PipeRead = fd1[0];
        PipeWrite = fd2[1];
        fcntl (PipeRead, F_SETFL, 0);
        fcntl (PipeWrite, F_SETFL, 0);
        fcntl (fd1[0], F_SETFL, 0);
        fcntl (fd2[1], F_SETFL, 0);
    } 
    else {
        i_am_supervisor = false;
        Pid = getpid();
        close (fd2[1]);
        close (fd1[0]);
        PipeRead = fd2[0];
        PipeWrite = fd1[1];
        fcntl (PipeRead, F_SETFL, 0);
        fcntl (PipeWrite, F_SETFL, 0);
        fcntl (fd1[1], F_SETFL, 0);
        fcntl (fd2[0], F_SETFL, 0);
    }
}

Instance::~Instance()
{
    close (PipeRead);
    close (PipeWrite);    
}

void Instance::killChild()
{
    kill(Pid, SIGTERM);
    waitpid(Pid, NULL, 0);
}

int Instance::sendResult( NiceShaper *ns )
{
    vector <string> stats_table;

    stats_table = ns->stats();

    talk->sendMsgv(pipeWrite(), stats_table);

    return 0; 
}

int Instance::getResult()
{
    if ( talk->getMsgv(pipeRead(), StatsTable) == -1 ) return -1;

    return 0; 
}

int Instance::formatStats( vector <string> &stats_vector, UnitType stats_unit)
{
    const int max_rate_size = int_to_str(MAX_RATE).size() + 4;
    string stat_line;
    string buf;

    for ( unsigned int n=0; n<StatsTable.size(); n++ ) {
        stat_line = StatsTable[n];
        // Name column
        buf = undent(awk( stat_line, 1 ), MAX_CLASS_NAME_SIZE) + "  ";
        if (n)
        {
            // Ceil column
            if ( awk( stat_line, 2 ) != "-" ) {
                buf += indent((int_to_str(unit_convert( awk( stat_line, 2 ), stats_unit )) + unit_to_str(stats_unit, 0)), max_rate_size ) + " - "; 
            }
            else { 
                buf += string(max_rate_size , ' ') + "   ";
            }

            // Last-Ceil column
            if ( awk( stat_line, 3 ) != "-" ) {
                buf += indent((int_to_str(unit_convert( awk( stat_line, 3 ), stats_unit )) + unit_to_str(stats_unit, 0)), max_rate_size ) + " "; 
            }
            else {
                buf += string(max_rate_size, ' ') + " ";
            }

            // Last-Utilize column
            if ( awk(stat_line, 4) != "-" ) {
                buf += "( " + indent((int_to_str(unit_convert( awk( stat_line, 4 ), stats_unit )) + unit_to_str(stats_unit, 0)), max_rate_size ) + " )";
            }
            else { 
                buf += "( " + string(max_rate_size, ' ') + " )";
            }   
        } 
        else 
        {
            buf += indent(awk( stat_line, 2 ), max_rate_size) + " - ";
            buf += indent(awk( stat_line, 3 ), max_rate_size);
            buf += " ( " + indent(awk( stat_line, 4 ), max_rate_size) + " )";
        }

        stats_vector.push_back(buf);
    }

    buf = "";
    stats_vector.push_back(buf);

    return 0;
}

string Instance::undent(string arg, unsigned int count)
{
    string res = "";

    res.assign(arg, 0 , count);
    if (count > res.size()) {
        res.append(count-res.size(), ' ');
    }

    return res;
}

string Instance::indent(string arg, unsigned int count)
{
    string res = "";

    if (count > arg.size()) {
        res.append(count-arg.size(), ' ');
    }

    res.append(arg, 0 , count);

    return res;
}

