/*
 *      NiceShaper - Dynamic Traffic Management
 *
 *      Copyright 2004 Mariusz Jedwabny <mariusz@jedwabny.net>
 *
 *      This file is subject to the terms and conditions of the GNU General Public
 *      License.  See the file COPYING in the main directory of this archive for
 *      more details.
 */

#include "main.h"

#include <sys/time.h>
#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/wait.h>
#include <sys/prctl.h>
#include <unistd.h>

#include <iostream>
#include <fstream>
#include <string>
#include <vector>

#include "instance.h"
#include "conv.h"
#include "aux.h"
#include "log.h"
#include "talk.h"
#include "ifaces.h"
#include "tests.h"
#include "sys.h"
#include "niceshaper.h"

using namespace aux;

int controller(std::string, std::string, std::string, std::string, int);
int supervisor(int, char *[], bool, std::vector <unsigned int> &);
int supervisor_loop();
void handle_controller(int);
void dump_stats_to_file();
void worker_loop(std::vector <unsigned int > &);
void converter(std::vector <std::string>);
int read_cmdline_params (std::vector <std::string>, bool &, std::string &, std::string &, std::string &, int &);
int proceed_global_config ();
int prepare_env();
int save_quota_counters();
int restore_quota_counters();
void sig_exit_controller(int);
void sig_exit_supervisor(int);
void sig_exit_worker(int);
void sig_sigchld(int);
int get_rid_of_unused(int unused) { return unused; }
void quit_supervisor_init();
void set_proc_title_init (int argc, char *argv[]);
void set_proc_title (std::string);

// Externs 
class Sys *sys;
class IfacesMap *ifaces;
class Tests *test;
class Log log;
class Talk *talk;

// Init extern globals
std::string g_section_name = "niceshaper";
unsigned int g_section_id = 16; // first section id value
bool g_fallback_to_tc = false;
bool g_fallback_to_ipt = false;
EnumStatsClassesType g_stats_classes = SC_WORKING;
EnumStatsSumType g_stats_sum_position = SS_BOTTOM;

// Declare and init non extern globals
std::string pidfile = "/var/run/niceshaper.pid";
std::string confdir = "/etc/niceshaper";
std::string conffile = confdir + "/config.conf";
std::string classfile = confdir + "/class.conf";
std::string usersfile = confdir + "/users.conf";
std::string vardir = "/var/lib/niceshaper";
std::string quotafile = "";
std::string iptfile = "/tmp/niceshaper";
std::string svinfofile = vardir + "/supervisor.info";

int supervisor_socket;
int supervisor_connections = 0;

std::vector <std::string> fpv_conffile, fpv_classfile;  

class Instance *supervisor_instance;
std::vector <Instance *> worker_instance;

class NiceShaper *ns;
class Conv *conv;

extern char **environ;
static char **argv_buffer = NULL;
static size_t argv_size = 0;

bool devmode = false;

int main(int argc, char *argv[])
{
    std::vector <unsigned int> protected_marks;
    std::vector <std::string> runtime_line;
    std::string runtime_cmd = "";    
    bool runtime_param_daemon_mode = true;
    std::string runtime_param_stats_unit = "";
    std::string runtime_param_stats_remote = "";
    std::string runtime_param_stats_password = "";
    int runtime_param_stats_watch = 0;
    char *env_lang = getenv("LANG");

    srand (time(NULL));
    umask (umask(000) | 007);

    if (env_lang != NULL) {
        if (std::string(env_lang) == "pl_PL.UTF-8") { log.setLang(PL_UTF8); }
    }

    for (int n=0; n<argc; n++) runtime_line.push_back(std::string(argv[n]));

    if (runtime_line.size() <= 1) { log.dumpHelp(); exit(0); }

    sys = new Sys;
    ifaces = new IfacesMap;
    test = new Tests;
    conv = new Conv;
    talk = new Talk;
    
    // Get command line parameters
    runtime_cmd = runtime_line.at(1);
    if (read_cmdline_params (runtime_line, runtime_param_daemon_mode, runtime_param_stats_unit, runtime_param_stats_remote, runtime_param_stats_password, runtime_param_stats_watch) == -1) exit(-1);

    // Read configuration
    if (conv->convertToFpv (confdir, conffile, CONFTYPE, protected_marks, fpv_conffile) == -1) exit (-1);

    if (proceed_global_config() == -1) exit (-1);

    if ((runtime_cmd == "stats") || (runtime_cmd == "stop") || (runtime_cmd == "restart")) {
        if (controller (runtime_cmd, runtime_param_stats_unit, runtime_param_stats_remote, runtime_param_stats_password, runtime_param_stats_watch) == -1) exit (-1);
        if (runtime_cmd != "restart") exit (0);
    }

    if ((runtime_cmd == "start") || (runtime_cmd == "restart")) {
        supervisor (argc, argv, runtime_param_daemon_mode, protected_marks);
        sig_exit_supervisor (SIGTERM);
    }
    else if (runtime_cmd == "convert") {
        converter (runtime_line);
        exit (0);
    }
    else log.dumpHelp();
    
    return 0;
}

int controller(std::string runtime_cmd, std::string runtime_param_stats_unit, std::string runtime_param_stats_remote, std::string runtime_param_stats_password, int runtime_param_stats_watch)
{
    std::vector <std::string> stats_vector;
    std::string stats_request = "";
    std::string buf = "";
    std::ifstream ifd;
    int dpid;
    int connection_socket;
    struct sockaddr_in address;

    if (runtime_cmd == "stats") 
    {
        stats_request = "stats";
        if (runtime_param_stats_unit.size()) stats_request += " --unit " + unit_to_str(get_unit(runtime_param_stats_unit), 0);

        if (runtime_param_stats_remote.size()) {
            if (runtime_param_stats_password.empty()) { log.error(58); return -1; }
            if (conv->setStatsListenAddr (runtime_param_stats_remote) == -1) return -1;
            stats_request += " --password " + runtime_param_stats_password;
        }
        else {
            // Get listen address and password
            ifd.open(svinfofile.c_str());
            if (ifd.is_open()) {
                getline(ifd, buf);
                ifd.close();
                if (conv->setStatsListenAddr (awk (buf, ":", 1) + ":" + awk (buf, ":", 2)) == -1) return -1;
                if (runtime_param_stats_password.size()) stats_request += " --password " + runtime_param_stats_password;
                else stats_request += " --password " + awk (buf, ":", 3);
            }
            else {
                log.error (201, svinfofile);
                return -1;
            }
        }

        do {
            stats_vector.clear();

            if ((connection_socket = socket (AF_INET, SOCK_STREAM, 0)) < 0 ) {
                log.error (49, "");
                return -1;
            }

            bzero((char *) &address, sizeof(address));
            address.sin_port = htons(conv->getStatsListenPort());
            address.sin_addr.s_addr = inet_addr(conv->getStatsListenIp().c_str());
            address.sin_family = AF_INET;

            if (connect(connection_socket, (struct sockaddr*)&address, sizeof(struct sockaddr)) < 0) {
                log.error (49, "");
                close (connection_socket);
                return -1;        
            }

            if (talk->sendMsg (connection_socket, stats_request) < 0) {
            close (connection_socket);
            return -1;
            }

            if (talk->getMsgv (connection_socket, stats_vector) < 0) {
                close (connection_socket);
                return -1;
            }

            close (connection_socket);

            if (runtime_param_stats_watch) system ("clear");
            for ( unsigned int n=0; n<stats_vector.size(); n++ ) {
                std::cout << stats_vector[n] << std::endl;
            }
        } while (runtime_param_stats_watch && (usleep(runtime_param_stats_watch*1000000) != -1));

        exit (0); 
    }
    else if ((runtime_cmd == "stop") || (runtime_cmd == "restart"))
    {
        // Get supervisor pid
        ifd.open(pidfile.c_str());
        if (!ifd) { log.error(45); return -1;  }
        getline(ifd, buf);
        ifd.close();
        dpid = str_to_int(buf);
        if (dpid <= 0) { log.error (45); return -1; }

        kill (dpid, SIGTERM);
        log.setDoNotPutNewLineChar (true);
        log.info (6);
        while (!access (pidfile.c_str(), 0))
        {
            log.setDoNotPutNewLineChar (true);
            log.onTerminal (".");
            usleep (100000);
        }
        log.info (2);
        if (runtime_cmd == "restart") return 0;
    }
    else return -1;
    
    exit (0);
}

int supervisor(int argc, char *argv[], bool daemon_mode, std::vector <unsigned int> &protected_marks)
{
    bool i_am_supervisor;
    std::vector <std::string>::iterator fpvi;
    std::vector <std::string>::iterator si;
    std::ofstream ofd;
    FILE *fp;
    struct stat vardir_stat;
    bool vardir_exists = true;
    char cbuf[MAX_LONG_BUF_SIZE];
    std::string buf;
   
    // NiceShaper already running
    if (!access(pidfile.c_str(), 0)) { log.error(44); exit(0); }

    // Check for vardir
    if ((stat(vardir.c_str(), &vardir_stat) == -1) || (!S_ISDIR(vardir_stat.st_mode))) { log.warning(14, vardir); vardir_exists = false; }
    
    // Check for executables
    if (test->whichExecutable("iptables-save").empty()) { log.warning(16); g_fallback_to_ipt = true; }
    if (test->whichExecutable("iptables-restore").empty()) { log.warning(17); g_fallback_to_ipt = true; }
    if (test->whichExecutable("iptables").empty()) { log.error(703); exit(-1); }
    if (test->whichExecutable("tc").empty()) { log.error(704); exit(-1); }

    // Create empty pid file
    ofd.open(pidfile.c_str());
    if (!ofd) { log.error(48, pidfile); exit(0); }
    ofd.close();

    if (daemon_mode) {
        int res = fork();
        if (res == -1) {
            exit (0);
        }
        else if (res) {
            // Wait for SIGUSR1 after proper initialization. If not, pidfile will be removed
            signal (SIGUSR1, sig_exit_controller);
            while (!access( pidfile.c_str(), 0)) usleep(1000000);
            // Something went wrong
            exit (0);
        }
        setsid ();
        chdir ("/");
    }

    ofd.open(pidfile.c_str());
    ofd << int_to_str(getpid()) + '\n';
    ofd.close();

    if (log.getFatalError()) quit_supervisor_init(); 
    if (conv->RunningSections.empty()) { log.error( 15 ); quit_supervisor_init(); }

    if (!conv->getUsersReplaceClasses()) {
        if (conv->convertToFpv (confdir, classfile, CLASSTYPE, protected_marks, fpv_classfile) == -1) quit_supervisor_init();
    }
    else {
        if (conv->convertUsersFile (confdir, usersfile, conv->getUsersDownloadSection(), conv->getUsersUploadSection(), conv->getUsersIfaceInet(), conv->getUsersResolveHostName(), false, fpv_classfile) == -1) quit_supervisor_init();
    }
   
    /*fpvi = fpv_classfile.begin();
    while ( fpvi < fpv_classfile.end() ) {
        std::cout << "cl:" << *fpvi << std::endl;
        fpvi++;
    }*/
    
    if (prepare_env() == -1) quit_supervisor_init(); 

    // Create batch file for iptables-restore
    if (!g_fallback_to_ipt) {
        ofd.open(iptfile.c_str());
        if (ofd.is_open()) {
            fp = popen("iptables-save -t mangle", "r");
            if(!fp) { log.error(0); exit(-1); }
            while (fgets(cbuf, MAX_LONG_BUF_SIZE, fp)) {
                buf = trim_legacy(std::string(cbuf));
                if (buf.find_first_of("#") != std::string::npos) buf.erase(buf.find_first_of("#"));
                if (buf == "COMMIT") continue;  
                ofd << buf << std::endl;
            }
            pclose(fp);
            ofd.flush();
            ofd.close();
        }
        else {
            log.warning(15, iptfile);
            g_fallback_to_ipt = true;
        }
    }

    // Create workers
    si = conv->RunningSections.begin();
    while (si < conv->RunningSections.end()) {
        g_section_name = *si;
        worker_instance.push_back(new Instance(i_am_supervisor));
        if (i_am_supervisor) {
            supervisor_instance = NULL;
            g_section_name = "niceshaper";   
            if (talk->getInitStatus(worker_instance.back()->pipeRead()) == -1) return -1;
        } 
        else {            
            quotafile = vardir + "/" + g_section_name + ".quota";
            if (!vardir_exists) quotafile = "";
            supervisor_instance = worker_instance.back();
            for (int n=worker_instance.size()-2; n>=0; n--) delete worker_instance[n];
            worker_instance.clear(); 
            set_proc_title_init (argc, argv);
            set_proc_title (g_section_name);
            worker_loop (protected_marks);
            exit(0); // worket should not reach this point, it should make exit from worker_loop();
        }
        si++;
        if (si < conv->RunningSections.end()) g_section_id++;
    }

    return supervisor_loop ();
}

int supervisor_loop ()
{
    struct timeval stats_now, stats_past, tv, tv_stats_file_rewrite;
    struct sockaddr_in address; 
    bool htb_fallback_fully_initialized = false;
    bool sections_all_reloaded = false;
    bool stats_file_out_of_date = false;
    int supervisor_max_connections = 2;
    int connection_socket;
    int fd_max;
    int yes = 1;
    int result;
    fd_set rfds;
    std::string buf;
    std::ofstream ofd;

    signal (SIGTERM, sig_exit_supervisor);
    signal (SIGINT, sig_exit_supervisor);
    signal (SIGCHLD, sig_sigchld);
    
    if (conv->getStatsFilePath().size())
    {
        if (access (conv->getStatsFilePath().c_str(), W_OK)) creat (conv->getStatsFilePath().c_str(), O_WRONLY);
        buf = "chown " + conv->getStatsFileOwner() + ":" + conv->getStatsFileGroup() + " " + conv->getStatsFilePath()
            + " && chmod " + conv->getStatsFileMode() + " " + conv->getStatsFilePath() + " &";
        system (buf.c_str());
    }

    // Initialize iptables rules    
    if (!g_fallback_to_ipt) {
        ofd.open(iptfile.c_str(), std::ios::app);
        if (!ofd.is_open()) {
            log.error(27, iptfile);
            return -1;
        }
        ofd << "COMMIT" << std::endl;
        ofd.flush();
        ofd.close();

        if (sys->ipt("iptables-restore < " + iptfile) != 0) { log.error(705, iptfile); return -1; }
            
        unlink (iptfile.c_str());
    } 

    // Create socket for stats demands
    tv.tv_sec = 30;
    supervisor_socket = socket (AF_INET, SOCK_STREAM, 0);
    setsockopt (supervisor_socket, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(int));
    setsockopt (supervisor_socket, SOL_SOCKET, SO_RCVTIMEO, (struct timeval *)&tv, sizeof(struct timeval));
    bzero((char *) &address, sizeof(address));
    address.sin_port = htons(conv->getStatsListenPort());
    address.sin_addr.s_addr = inet_addr (conv->getStatsListenIp().c_str());
    address.sin_family = AF_INET; 
    if (bind (supervisor_socket, (struct sockaddr *) &address, sizeof(address)) == -1 ) {
        log.error (51, (conv->getStatsListenIp() + ":" + int_to_str(conv->getStatsListenPort())));
        return -1;
    }
    listen (supervisor_socket, 10);
    
    // Disclose connection address and password for from localhost connections
    ofd.open(svinfofile.c_str());
    if (!ofd.is_open()) {
        log.error(48, svinfofile);
        return -1;
    }
    ofd << conv->getStatsListenIp() << ":" << conv->getStatsListenPort() << ":" << conv->getStatsPassword() << '\n';
    ofd.close();

    // Send "proper initialized" messages
    log.info (1);
    kill (getppid(), SIGUSR1);

    log.setLogOnTerminal (false);
    
    freopen( "/dev/null", "r", stdin);
    if (!devmode) {
        freopen( "/dev/null", "w", stdout);
        freopen( "/dev/null", "w", stderr);
    }

    // Send start signal to all workers
    for (unsigned int n=0; n<worker_instance.size(); n++)        
    {
        talk->sendContinue(worker_instance[n]->pipeWrite());
        usleep(100000);
    }
    
    gettimeofday(&stats_past, NULL);
    stats_file_out_of_date = false;

    while (true)
    {
        FD_ZERO (&rfds);
        FD_SET (supervisor_socket, &rfds);
        fd_max = supervisor_socket + 1;
        tv_stats_file_rewrite.tv_sec = conv->getStatsFileRewrite();
        tv_stats_file_rewrite.tv_usec = 0;

        for (unsigned int n=0; n<worker_instance.size(); n++)
        {
            FD_SET (worker_instance[n]->pipeRead(), &rfds);
            if (worker_instance[n]->pipeRead() >= fd_max) fd_max=worker_instance[n]->pipeRead()+1;
        }

        if (conv->getStatsFilePath().size()) 
            result = select(fd_max, &rfds, NULL, NULL, &tv_stats_file_rewrite);
        else 
            result = select(fd_max, &rfds, NULL, NULL, NULL);
        
        if (result == -1) {
            if (errno == EINTR) continue; 
            return -1;
        }
        else if (result > 0) {
            // Get results from worker
            for (unsigned int n=0; n<worker_instance.size(); n++)        
            {
                if (FD_ISSET(worker_instance[n]->pipeRead(), &rfds)) {
                    if (worker_instance[n]->getResult() == -1) return -1;
                    talk->sendContinue(worker_instance[n]->pipeWrite());            
                    stats_file_out_of_date = true;
                    worker_instance.at(n)->incResultsCounter();
                }
            }

            // Get command from controller
            if (FD_ISSET(supervisor_socket, &rfds)) {
                connection_socket = accept (supervisor_socket, NULL, NULL);
                if (connection_socket < 0) { 
                    log.error(301);
                }
                else if (supervisor_connections < supervisor_max_connections) {
                    switch (fork()) {
                        case 0:
                            close (supervisor_socket);
                            handle_controller (connection_socket);
                            shutdown (connection_socket, SHUT_RDWR);
                            close (connection_socket);
                            exit(0);
                        default:
                            supervisor_connections++;
                            close (connection_socket);
                    }
                } 
                else {
                    shutdown (connection_socket, SHUT_RDWR);
                    close (connection_socket);
                }
            }
        
            // Initialize htb fallback with proper rate if all sections reloaded at least once
            if (!htb_fallback_fully_initialized) {
                sections_all_reloaded = true;
                for (unsigned int n=0; n<worker_instance.size(); n++) {
                    if (!worker_instance.at(n)->getResultsCounter()) sections_all_reloaded = false;
                }
                if (sections_all_reloaded) {
                    if (ifaces->endUpHtbFallbackOnControlled() == -1) return -1;
                    htb_fallback_fully_initialized = true;
                }
            }
        }

        // Dump stats to file if expected
        if (conv->getStatsFilePath().size() && stats_file_out_of_date) {
            gettimeofday (&stats_now, NULL);
            if ((stats_now.tv_sec-stats_past.tv_sec) >= conv->getStatsFileRewrite()) {
                dump_stats_to_file(); 
                gettimeofday (&stats_past, NULL);
                stats_file_out_of_date = false;
            }
        }
    }
}

void handle_controller (int connection_socket)
{
    UnitType stats_unit = conv->getStatsUnit();
    std::vector <std::string> runtime_params; // From controller
    std::vector <std::string> stats_vector;
    std::string stats_password = "";
    std::string buf;  

    if (talk->getMsg (connection_socket, buf) == -1) return;

    if (buf.empty()) return;

    for (int n=1; awk(buf, n).size(); n++) {
        runtime_params.push_back(awk(buf, n));
    }

    if (runtime_params.at(0) == "stats") {
        // Proceed runtime params
        for (unsigned int n=1; n<runtime_params.size(); n++) {
            if (runtime_params.at(n) == "--unit") {
                stats_unit = get_unit (runtime_params.at(++n));
            }
            else if (runtime_params.at(n) == "--password") {
                stats_password = runtime_params.at(++n);
            }
        }
        stats_vector.clear();

        if (stats_password != conv->getStatsPassword()) {
            stats_vector.push_back(log.getErrorMessage(57));
            talk->sendMsgv(connection_socket, stats_vector);
            return;
        }

        for (unsigned int n=0; n<worker_instance.size(); n++) {
            worker_instance[n]->formatStats(stats_vector, stats_unit);   
        }

        if (stats_vector.size()) {
            talk->sendMsgv(connection_socket, stats_vector);
        } else {
            talk->sendMsg(connection_socket, "");
        }
    }

    return;    
}

void dump_stats_to_file()
{
    std::vector <std::string> stats_vector;
    std::vector <std::string>::iterator svi;
    std::string buf;
    int fd;

    fd = open( conv->getStatsFilePath().c_str(), O_RDWR | O_TRUNC );
    for ( unsigned int n=0; n<worker_instance.size(); n++ ) worker_instance[n]->formatStats(stats_vector, conv->getStatsUnit());
    svi = stats_vector.begin();    
    while (svi < stats_vector.end()) {
        write(fd, (*svi + "\n").c_str(), (*svi).size()+1);
        svi++;
    }
    buf = "Powered by NiceShaper\n";
    write( fd, buf.c_str(), buf.size());
    buf = "http://niceshaper.jedwabny.net\n";
    write( fd, buf.c_str(), buf.size());
    close(fd);
}

void worker_loop (std::vector <unsigned int> &protected_marks)
{
    struct timeval now, past, quota_save_now, quota_save_past;
    unsigned int cycle;
    std::vector <std::string> ipt_rules;
    std::vector <std::string>::iterator qti;
    std::ofstream ofd;

    signal (SIGTERM, sig_exit_worker);
    signal (SIGINT, sig_exit_worker);
 
    log.setDoNotPutNewLineChar(true);
    log.info(3);
    ns = new NiceShaper();
    if (ns->init(fpv_conffile, fpv_classfile, protected_marks, ipt_rules) == -1) {
        log.error(30);
        talk->sendInitStatusError(supervisor_instance->pipeWrite());
        while (true) usleep(1000000); // waiting for SIGTERM, from parent instance
    }
    else {
        if (!g_fallback_to_ipt && ipt_rules.size()) {                    
            ofd.open(iptfile.c_str(), std::ios::app);
            if (ofd.is_open()) {
                qti = ipt_rules.begin();
                while (qti < ipt_rules.end()) {
                    ofd << trim_legacy(*qti) << std::endl;
                    qti++;
                }
                ofd.flush();
                ofd.close();
            }
            else {
                log.error(27, iptfile);
                talk->sendInitStatusError(supervisor_instance->pipeWrite());
                while (true) usleep(1000000); // waiting for SIGTERM, from parent instance
            }
        }

        talk->sendInitStatusOk(supervisor_instance->pipeWrite());
    }

    if (quotafile.size()) {
        restore_quota_counters ();
        rename (quotafile.c_str(), (quotafile+".bak").c_str());
        save_quota_counters();
    }

    log.setLogOnTerminal (false);

    freopen( "/dev/null", "r", stdin);
    if (!devmode) {    
        freopen( "/dev/null", "w", stdout);
        freopen( "/dev/null", "w", stderr);
    }

    // Wait for start signal from supervisor
    gettimeofday (&past, NULL);

    talk->waitForContinue(supervisor_instance->pipeRead());

    gettimeofday (&quota_save_past, NULL);

    while (true) 
    {
        gettimeofday (&now, NULL);
        cycle = (now.tv_sec-past.tv_sec)*1000000+(now.tv_usec-past.tv_usec);
        if ((ns->reload()*100000) > cycle) usleep (ns->reload()*100000-cycle);
        gettimeofday(&past, NULL);
        sys->rtnlOpen();
        if (ns->judge() == -1)
        {
            sys->rtnlClose();
            talk->sendFatalError(supervisor_instance->pipeWrite());
            while (true) usleep(1000000); // waiting for SIGTERM from parent instance
        }
        sys->rtnlClose();
        supervisor_instance->sendResult(ns);
        talk->waitForContinue(supervisor_instance->pipeRead());

        // Save quota counters
        gettimeofday (&quota_save_now, NULL);
        if ((quota_save_now.tv_sec-quota_save_past.tv_sec) >= 300 ) {
            save_quota_counters(); 
            gettimeofday (&quota_save_past, NULL);
        }
    }
}

void converter(std::vector <std::string> runtime_params)
{
    std::vector <std::string> fpv;  
    std::vector <std::string>::iterator fpvi, fpvi_begin, fpvi_end;

    std::string download_section = conv->getUsersDownloadSection();
    std::string upload_section = conv->getUsersUploadSection();
    std::string iface_inet = conv->getUsersIfaceInet();
    bool resolve_hostname = conv->getUsersResolveHostName();
    
    for ( unsigned int n=2; n<runtime_params.size(); n++ ) {
        if ( runtime_params.at(n) == "--download-section" ) { 
            if ( (n+1)>=runtime_params.size()) { log.error( 28, runtime_params.at(n)); exit(0); }
            download_section = runtime_params.at(++n);
        }   
        else if ( runtime_params.at(n) == "--upload-section" ) { 
            if ( (n+1)>=runtime_params.size()) { log.error( 28, runtime_params.at(n)); exit(0); }
            upload_section = runtime_params.at(++n);
        }
        else if ( runtime_params.at(n) == "--iface-inet" ) {
            if ( (n+1)>=runtime_params.size()) { log.error( 28, runtime_params.at(n)); exit(0); }
            iface_inet = runtime_params.at(++n);
        }
        else if ( runtime_params.at(n) == "--resolve-hostname" ) { 
            if ( (n+1)>=runtime_params.size()) { log.error( 28, runtime_params.at(n)); exit(0); }
            if ( runtime_params.at(++n) == "yes" ) resolve_hostname=true; 
            else if ( runtime_params.at(n) == "true" ) {
                resolve_hostname=true; 
                log.warning( 10, "--resolve-hostname true" );
            }
            else resolve_hostname=false;
        }   

    }
    if (log.getFatalError()) exit(-1);

    if (conv->convertUsersFile (confdir, usersfile, download_section, upload_section, iface_inet, resolve_hostname, true, fpv) == -1) exit (-1);

    fpvi=fpv.begin();             
    fpvi_end=fpv.end();

    while (fpvi != fpvi_end) {
        std::cout << *fpvi << std::endl;
        fpvi++;
    }

    exit(0);        
}

int read_cmdline_params (std::vector <std::string> runtime_params, bool &daemon_mode, std::string &stats_unit, std::string &stats_remote, std::string &stats_password, int &stats_watch)
{
    bool is_set_conffile = false;
    bool is_set_classfile = false;
    bool is_set_usersfile = false;
    std::string param = "", value = "";

    for (unsigned int n=2; n<runtime_params.size(); n++) {
        param = runtime_params.at(n);
        if (runtime_params.at(n) != "--no-daemon") {
            if ((n+1) >= runtime_params.size()) {
                log.error (28, runtime_params.at(n)); 
                return -1; 
            }
            value = runtime_params.at(++n);
        }

        if (param == "--no-daemon") { 
            daemon_mode = false; 
        }
        else if (param == "--confdir") {
            confdir = value;
            if (!is_set_conffile) conffile = confdir + "/config.conf";
            if (!is_set_classfile) classfile = confdir + "/class.conf";
            if (!is_set_usersfile) usersfile = confdir + "/users.conf";
        }
        else if (param == "--conffile") { 
            conffile = value;
            is_set_conffile = true;
        }
        else if (param == "--classfile") { 
            classfile = value;
            is_set_classfile = true;
        }                                                   
        else if (param == "--usersfile") { 
            usersfile = value;
            is_set_usersfile = true;
        }
        else if (param == "--unit") { 
            stats_unit = value;
        }
        else if (param == "--remote") { 
            stats_remote = value;
        }
        else if (param == "--password") { 
            stats_password = value;
        }
        else if (param == "--watch") { 
            stats_watch = str_to_int(value);
            if ((stats_watch < 1) || (stats_watch > 60)) { log.error (28, param); return -1; }
        }
        else if ((param != "--outfile") &&
                (param != "--download-section") &&
                (param != "--upload-section") &&
                (param != "--iface-inet") &&
                (param != "--resolve-hostname")) { log.error (28, param); return -1; }
    }

    return 0;
}

int proceed_global_config ()
{
    std::vector <std::string>::iterator fpvi, fpvi_begin, fpvi_end;
    std::string option, param, value, dev;

    if ( fpv_section_i( fpvi_begin, fpvi_end, fpv_conffile, "global" ) == -1 ) return -1;;

    /*fpvi = fpvi_begin;                                                
    while ( fpvi <= fpvi_end ) {                                                                        
        std::cout << "cf:" << *fpvi << std::endl;
        fpvi++;                                                                                    
    }*/

    fpvi=fpvi_begin;
    while (fpvi <= fpvi_end) {
        option = awk(*fpvi, 1);
        param = awk(*fpvi, 2);
        value = awk(*fpvi, 3);

        if ( option == "run" ) {
            if (param.size() > MAX_SECTION_NAME_SIZE) { log.error(54, *fpvi); return -1; }
            conv->addRunningSection(param);
        }
        else if (option == "mark-on-ifaces") 
        {
            param=trim_dev(param);
            if (!ifaces->isValidSysDev(param)) { log.error (16, *fpvi); return -1; }
            ifaces->setTcFilterType(param, FW);
        }
        else if ((awk(option, "-", 1) == "iface") && (ifaces->isValidSysDev(awk(option, "-", 2)))) {
            dev = awk(option, "-", 2);
            if (param == "speed") {
                if ((unit_convert(value, BITS) > MAX_RATE) || (unit_convert(value, BITS) < MIN_RATE)) { log.error(806, *fpvi); return -1; }
                ifaces->setSpeed(dev, unit_convert(value, BITS));
                // TODO: warning: comparison between signed and unsigned integer expressions [-Wsign-compare]
            }
            else if (param == "do-not-shape-method") {
                if (value == "safe") ifaces->setDNShapeMethodSafe(dev, true);
                else if (value == "full-throttle") ifaces->setDNShapeMethodSafe(dev, false);
                else { log.error ( 11, *fpvi ); return -1; }
            }
            else if (param == "fallback-rate") {
                if ((unit_convert(value, BITS) > MAX_RATE) || (unit_convert(value, BITS) < MIN_RATE)) { log.error(806, *fpvi); return -1; }
                ifaces->setFallbackRate(dev, unit_convert(value, BITS));
            }    
            else { log.error ( 11, *fpvi ); return -1; }
        }
        else if ( option == "lang" ) {  
            if ( param == "pl" ) log.setLang(PL_UTF8);  
            else if ( param == "en" ) log.setLang(EN);
            else { log.error ( 11, *fpvi ); return -1; }
        }
        else if ( option == "stats" ) {
            if ( param == "unit" ) 
            {
                conv->setStatsUnit(get_unit(value));
            }
            else if ( param == "classes" )
            {
                if ( value == "all" ) g_stats_classes = SC_ALL;
                else if ( value == "active" ) g_stats_classes = SC_ACTIVE;
                else if ( value == "working" ) g_stats_classes = SC_WORKING;
                else if ( value == "no" ) g_stats_classes = SC_FALSE;
                else if (( value == "false" ) || ( value == "none" )) {
                    g_stats_classes = SC_FALSE;
                    log.warning(9, *fpvi);
                }
                else { log.error(11, *fpvi); }
            }
            else if ( param == "sum" )
            {
                if ( value == "top" ) g_stats_sum_position = SS_TOP;
                else if ( value == "bottom" ) g_stats_sum_position = SS_BOTTOM;
                else if ( value == "no" ) g_stats_sum_position = SS_FALSE;
                else if (( value == "false" ) || ( value == "none" )) {
                    g_stats_sum_position = SS_FALSE;
                    log.warning ( 9, *fpvi );
                }
                else { log.error( 11, *fpvi ); }
            }
            else if ( param == "listen" ) {
                if (conv->setStatsListenAddr(value) == -1) return -1;
            }
            else if (param == "password") {
                conv->setStatsPassword(value);
            }
            else if ( param == "file" ) 
            {
                if (( value == "no" ) || value.empty()) conv->setStatsFilePath("");
                else if (( value == "none" ) || ( value == "false" )) {
                    conv->setStatsFilePath("");
                    log.warning( 9, *fpvi );
                }
                else conv->setStatsFilePath(value);
            }
            else if ( param == "owner" ) {
                conv->setStatsFileOwner(value);
            }
            else if ( param == "group" ) {
                conv->setStatsFileGroup(value);
            }
            else if ( param == "mode" ) {
                conv->setStatsFileMode(value);
            }
            else if (param == "rewrite") {
                conv->setStatsFileRewrite(str_to_int(value));
                if ((conv->getStatsFileRewrite() < 1) || (conv->getStatsFileRewrite() > 3600)) {
                    log.error(805, *fpvi);
                    return -1;
                }
            }
        } 
        else if ( option == "log" ) {
            if ( param == "terminal" ) {
                if ( value == "yes" ) {
                    log.setLogOnTerminal(true);
                }
                else if ( value == "no" ) {
                    log.setLogOnTerminal(false);
                }
                else if ( value == "true" ) {
                    log.warning( 10, *fpvi );
                    log.setLogOnTerminal(true);
                }
                else if (( value == "false" ) || ( value == "none" )) {
                    log.warning( 9, *fpvi );
                    log.setLogOnTerminal(false);
                }
                else {
                    log.error ( 11, *fpvi );
                    log.setLogOnTerminal(true);
                }
            }   
            else if ( param == "syslog" ) {
                if ( value == "yes" ) {
                    log.setLogToSyslog(true);
                }
                else if ( value == "no" ) {
                    log.setLogToSyslog(false);
                }
                else if ( value == "true" ) {
                    log.warning( 10, *fpvi );
                    log.setLogToSyslog(true);
                }
                else if (( value == "false" ) || ( value == "none" )) {
                    log.warning( 9, *fpvi );
                    log.setLogToSyslog(false);
                }
                else {
                    log.error ( 11, *fpvi );
                    log.setLogToSyslog(true);
                }
            }   
            else if ( param == "file" ) {
                if ( value == "no" ) {
                    log.setLogFile("");
                }
                else if (( value == "none" ) || ( value == "false" )) {
                    log.warning( 9, *fpvi);
                    log.setLogFile("");
                }
                else {
                    log.setLogFile(value);
                }
            }   
        }
        else if (option == "users") {
            if (param.empty() || value.empty()) { log.error(24, *fpvi); return -1; }
            if (param == "replace-classes") {
                if (value == "yes") conv->setUsersReplaceClasses(true);
                else if (value == "no") conv->setUsersReplaceClasses(false);
                else { log.error (11, *fpvi); return -1; }
            }
            else if (param == "download-section") {
                conv->setUsersDownloadSection(value);
            }
            else if (param == "upload-section") {
                conv->setUsersUploadSection(value);
            }
            else if (param == "iface-inet") {
                conv->setUsersIfaceInet(value);
            }
            else if (param == "resolve-hostname") {
                if (value == "yes") conv->setUsersResolveHostName(true);
                else if (value == "no") conv->setUsersResolveHostName(false);
                else { log.error (11, *fpvi); return -1; }
            }
        }
        else if ( option == "fallback" ) {
            if ( param == "iproute" ) g_fallback_to_tc = true;
            else if ( param == "iptables" ) g_fallback_to_ipt = true;
            else { log.error( 11, *fpvi ); return -1; }
        }
        fpvi++;
    }
    return 0;
}

int prepare_env ()
{
    std::vector <std::string>::iterator fpvi, fpvi_begin, fpvi_end;
    std::vector <std::string> devs_to_prepare;
    std::string option, param, value;
    std::string dev = "";
    std::string section = "";
    unsigned int section_htb_ceil;
    bool in_running_class = false;

    fpvi = fpv_classfile.begin();                                                
    while (fpvi < fpv_classfile.end()) { 
        option = awk(*fpvi, 1);
        param = awk(*fpvi, 2);
        value = awk(*fpvi, 3);
        if (in_running_class && (option == "type") && ((param == "wrapper") || ((param == "do-not-shape") && (ifaces->isDNShapeMethodSafe(dev))))) {
            ifaces->setHtbDNWrapperClass(dev, true);
        }
        else if ((awk(*fpvi, 1) == "class")) {
            if (!is_in_vector(conv->RunningSections, param)) { in_running_class=false; fpvi++; continue; }
            in_running_class = true;
            dev = trim_dev(value);
            if (!ifaces->isValidSysDev(dev)) { log.error(16, *fpvi); return -1; }
            if (!is_in_vector(devs_to_prepare, dev)) devs_to_prepare.push_back(dev);
            ifaces->setAsControlled(dev);
            if (!ifaces->isInSections(dev, param)) ifaces->addSection(dev, param); 
        }
        fpvi++;
    }

    /* reading running sections config */
    for (unsigned int n=0; n < conv->RunningSections.size(); n++)
    {
        section = conv->RunningSections.at(n);
        if (fpv_section_i(fpvi_begin, fpvi_end, fpv_conffile, section) == -1) {
            return -1;
        }

        fpvi = fpvi_begin;
        while ( fpvi <= fpvi_end )
        {
            option = awk(*fpvi, 1);
            param = awk(*fpvi, 2);
            value = awk(*fpvi, 3);
            if ((option == "section") && (param == "speed")) {
                section_htb_ceil = unit_convert(value, BITS);
                if ((section_htb_ceil > MAX_RATE) || (section_htb_ceil < MIN_RATE)) { log.error(806, *fpvi); return -1; }
                for (unsigned int m=0; m < devs_to_prepare.size(); m++) {
                //for (unsigned int m=0; m < Ifaces.size(); m++) {
                    //if (!ifaces->isControlled())
                    dev = devs_to_prepare.at(m);
                    if (ifaces->isInSections(dev, section)) {
                        if (ifaces->addToSectionsSpeedSum(dev, section_htb_ceil) == -1) return -1;
                    }
                }
                //if ifaces->ifaceSectionsHtbCeil > max error
                fpvi = fpvi_end;
            }
            fpvi++;
        }
    }

    if (!g_fallback_to_tc) {
        if (sys->rtnlOpen() == -1) return -1;
    }
    if (ifaces->initHtbOnControlled() == -1) return -1;
    if (!g_fallback_to_tc) sys->rtnlClose();
    
    // Clear iptables
    sys->ipt ("for n in `iptables -t mangle -L PREROUTING -nv --line-numbers | grep ns_ | awk '{print $1}' | sort -r`; do iptables -t mangle -D PREROUTING $n; done");
    sys->ipt ("for n in `iptables -t mangle -L POSTROUTING -nv --line-numbers | grep ns_ | awk '{print $1}' | sort -r`; do iptables -t mangle -D POSTROUTING $n; done");
    sys->ipt ("for n in `iptables -t mangle -L INPUT -nv --line-numbers | grep ns_ | awk '{print $1}' | sort -r`; do iptables -t mangle -D INPUT $n; done");
    sys->ipt ("for n in `iptables -t mangle -L OUTPUT -nv --line-numbers | grep ns_ | awk '{print $1}' | sort -r`; do iptables -t mangle -D OUTPUT $n; done");
    sys->ipt ("for n in `iptables -t mangle -L FORWARD -nv --line-numbers | grep ns_ | awk '{print $1}' | sort -r`; do iptables -t mangle -D FORWARD $n; done");
    sys->ipt ("for n in `iptables -t mangle -L -nv | grep 'Chain ns_' | awk '{print $2}'`; do iptables -t mangle -F $n ; iptables -t mangle -X $n; done");

    return 1;
}

int save_quota_counters()
{
    std::vector <std::string> quotas_table;
    std::vector <std::string>::iterator qti;
    std::ofstream ofd;

    quotas_table = ns->dumpQuotaCounters();
    if (quotas_table.size()) {        
        ofd.open(quotafile.c_str());
        if (ofd.is_open()) {
            qti = quotas_table.begin();
            while (qti < quotas_table.end()) {
                ofd << *qti + '\n';;
                qti++;
            }
            ofd.close();
        }
        else {
            log.error(27, quotafile);
        }
    }

    return 0;
}

int restore_quota_counters()
{
    std::vector <std::string> quota_counters_table;
    std::ifstream ifd;
    std::string buf;

    ifd.open (quotafile.c_str());
    if (ifd.is_open()) {
        while (getline(ifd ,buf)) {
            quota_counters_table.push_back (buf);
        }            
        ifd.close();
        if (quota_counters_table.size()) ns->setQuotaCounters(quota_counters_table);
    }

    return 0;
}

void sig_sigchld(int sig)
{
    signal (SIGCHLD, sig_sigchld);

    while (waitpid (-1, NULL, WNOHANG) > 0) {
        supervisor_connections--;
    }

    get_rid_of_unused (sig);
}

void sig_exit_controller(int sig)
{
    signal(SIGUSR1, sig_exit_supervisor);

    get_rid_of_unused (sig);

    exit (0);
}

void sig_exit_supervisor(int sig)
{
    signal(SIGTERM, sig_exit_supervisor);
    signal(SIGINT, sig_exit_supervisor);

    close (supervisor_socket);

    for ( int n=worker_instance.size()-1; n>=0; n-- ) {
        worker_instance[n]->killChild();
        delete worker_instance[n];
    }

    worker_instance.clear();
    delete ifaces;
    usleep(100000);
    log.info(2);
    unlink(pidfile.c_str());
    if (test->fileExists(svinfofile)) unlink(svinfofile.c_str());

    get_rid_of_unused (sig);

    exit(0);
}

void sig_exit_worker(int sig)
{
    signal(SIGTERM, sig_exit_worker);
    signal(SIGINT, sig_exit_worker);

    if (quotafile.size()) save_quota_counters ();

    delete ns;
    delete supervisor_instance;

    log.info(2);

    get_rid_of_unused (sig);

    exit(0);
}

void quit_supervisor_init() 
{
    log.error(30);
    if (test->fileExists(pidfile)) unlink(pidfile.c_str());
    if (test->fileExists(svinfofile)) unlink(svinfofile.c_str());
    exit(-1);
}

/***
  This code is taken from avahi setproctitle.c.
    
  avahi is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  avahi is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General
  Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with avahi; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.

  Adapt to NiceShaper - Mariusz Jedwabny, <mariusz@jedwabny.net> 
***/

void set_proc_title_init(int argc, char *argv[]) 
{
    unsigned i;
    char **new_environ, *endptr;

    /* This code is really really ugly. We make some memory layout
     * assumptions and reuse the environment array as memory to store
     * our process title in */
    for (i = 0; environ[i]; i++);

    endptr = i ? environ[i-1] + strlen(environ[i-1]) : argv[argc-1] + strlen(argv[argc-1]);

    argv_buffer = argv;
    argv_size = endptr - argv_buffer[0];

    /* Make a copy of environ */
    //new_environ = avahi_malloc(sizeof(char*) * (i + 1));
    new_environ = (char **)malloc(sizeof(char*) * (i + 1));
    //new_environ = new char*[i+1];
    //for (i = 0; environ[i]; i++) new_environ[i] = avahi_strdup(environ[i]);
    for (i = 0; environ[i] != NULL; i++) new_environ[i] = strdup (environ[i]);
    new_environ[i] = NULL;

    environ = new_environ;
}

void set_proc_title(std::string title) {
    size_t len;

    if (!argv_buffer) return;

    snprintf (argv_buffer[0], argv_size, std::string("niceshaper: " + title).c_str());

    len = strlen(argv_buffer[0]);

    memset(argv_buffer[0] + len, 0, argv_size - len);
    argv_buffer[1] = NULL;

    //PR_SET_NAME is since Linux 2.6.9
#ifdef PR_SET_NAME
    prctl(PR_SET_NAME, (unsigned long) std::string("ns:" + title).substr(0, MAX_SECTION_NAME_SIZE).c_str(), 0, 0, 0);
#endif
}

