/*
 * gst-filter-helper.h - Header for helper functions for GstFilter
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef __GST_FILTER_HELPER_H__
#define __GST_FILTER_HELPER_H__

#include <gst/filters/gst-filter.h>


G_BEGIN_DECLS


/**
 * GST_FILTER_GET_LOCK:
 * @obj: a #GstFilter
 *
 * Acquire a reference to the mutex of this filter.
 */
#define GST_FILTER_GET_LOCK(obj)      (GST_FILTER(obj)->lock)

/**
 * GST_FILTER_LOCK:
 * @obj: a #GstFilter to lock
 *
 * This macro will obtain a lock on the filter, making serialization possible.
 * It blocks until the lock can be obtained.
 * All virtual methods get called with the mutex locked so filters do not need
 * to bother with threading. However, if the filter needs to lock the mutex to
 * do something outside of the virtual methods (like in a g_object_set or in
 * a timer), then it should lock the filter's mutex by calling this macro.
 *
 * See also: GST_FILTER_UNLOCK()
 */
#define GST_FILTER_LOCK(obj)	       g_mutex_lock(GST_FILTER_GET_LOCK(obj))

/**
 * GST_FILTER_TRYLOCK:
 * @obj: a #GstFilter.
 *
 * This macro will try to obtain a lock on the filter, but will return with
 * FALSE if it can't get it immediately.
 */
#define GST_FILTER_TRYLOCK(obj)	       g_mutex_trylock(GST_FILTER_GET_LOCK(obj))

/**
 * GST_FILTER_UNLOCK:
 * @obj: a #GstFilter to unlock.
 *
 * This macro releases a lock on the filter.
 * All virtual methods get called with the mutex locked so filters do not need
 * to bother with threading. However, if the filter needs to lock the mutex to
 * do something outside of the virtual methods (like in a g_object_set or in
 * a timer), then it should call GST_FILTER_LOCK() then unlock it's mutex
 * by calling this macro.
 *
 * See also: GST_FILTER_LOCK()
 */
#define GST_FILTER_UNLOCK(obj)         g_mutex_unlock(GST_FILTER_GET_LOCK(obj))




/**
 * GST_DEFINE_FILTER:
 * @type: The type of the class to be defined
 * @filter_name: The name of the filter to define
 *
 * This will define a new #GstFilter derived class by calling G_DEFINE_TYPE()
 * and by declaring and setting the apply, revert and name fields of the
 * #GstFilter parent class. The defined apply/revert methods will be
 * gst_$(filter_name)_filter_apply and gst_$(filter_name)_filter_revert
 */
#define GST_DEFINE_FILTER(type, filter_name) \
  G_DEFINE_TYPE (type, gst_##filter_name##_filter, GST_TYPE_FILTER);    \
                                                                        \
  static GstPad *gst_##filter_name##_filter_apply (GstFilter *filter,   \
      GstBin *bin,                                                      \
      GstPad *pad);                                                     \
  static GstPad *gst_##filter_name##_filter_revert (GstFilter *filter,  \
      GstBin *bin,                                                      \
      GstPad *pad);                                                     \
                                                                        \
  static void                                                           \
  gst_##filter_name##_filter_class_init (type##Class *klass)            \
  {                                                                     \
    GstFilterClass *gstfilter_class = GST_FILTER_CLASS (klass);         \
                                                                        \
    gstfilter_class->apply = gst_##filter_name##_filter_apply;          \
    gstfilter_class->revert = gst_##filter_name##_filter_revert;        \
    gstfilter_class->name = #filter_name;                               \
  }


gboolean gst_filter_add_element (GstBin * bin,
    GstPad * pad, GstElement * element, GstPad * element_pad);
GstElement *gst_filter_add_element_by_name (GstBin * bin, GstPad * pad,
    const gchar * element_name, const gchar * pad_name,
    const gchar * out_pad_name, GstPad ** out_pad);
GstElement *gst_filter_add_element_by_name_default (GstBin * bin,
    GstPad * pad, const gchar * element_name, GstPad ** out_pad);
GstElement *gst_filter_add_element_by_description (GstBin * bin,
    GstPad * pad, const gchar * description, GstPad ** out_pad);
GstPad *gst_filter_apply_element_by_name (GstBin * bin, GstPad * pad,
    const gchar * element_name,
    const gchar * pad_name, const gchar * out_pad_name,
    GstElement ** element, GList ** elements);
GstPad *gst_filter_apply_element_by_name_default (GstBin * bin,
    GstPad * pad, const gchar * element_name,
    GstElement ** element, GList ** elements);
GstPad *gst_filter_apply_element_by_description (GstBin * bin,
    GstPad * pad, const gchar * description,
    GstElement ** element, GList ** elements);
GstPad *gst_filter_revert_element (GstBin * bin, GstPad * pad,
    const gchar * applied_pad_name,  GList ** elements);
GstPad *gst_filter_revert_element_default (GstBin * bin, GstPad * pad,
    GList ** elements);

G_END_DECLS
#endif /* __GST_FILTER__HELPER_H__ */
