/*
 * test-level.c - Example use of the GstLevelFilter class
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <gst/gst.h>
#include <gst/filters/gst-single-filter-manager.h>
#include <gst/filters/gst-level-filter.h>

#undef G_DISABLE_ASSERT

static GMainLoop *mainloop = NULL;


static gboolean
done (gpointer data)
{
  g_debug ("Dumping pipeline");
  GST_DEBUG_BIN_TO_DOT_FILE_WITH_TS (data, GST_DEBUG_GRAPH_SHOW_ALL,
      "test-level");
  g_main_loop_quit (mainloop);
  return FALSE;
}

static void
got_level (GstLevelFilter * filter, gdouble level)
{
  g_debug ("Got level : %f", level);
}


static GstBusSyncReply
_bus_callback (GstBus * bus, GstMessage * message, gpointer user_data)
{
  GstFilterManager *filters = user_data;

  if (!gst_filter_manager_handle_message (filters, message))
    return GST_BUS_PASS;

  gst_message_unref (message);
  return GST_BUS_DROP;
}


int
main (int argc, char *argv[])
{
  GstElement *pipeline = NULL;
  GstBus *bus = NULL;
  GstElement *src = NULL;
  GstElement *sink = NULL;
  GstPad *out_pad = NULL;
  GstPad *src_pad = NULL;
  GstPad *sink_pad = NULL;
  GstFilterManager *filters = NULL;
  GstFilter *filter = NULL;

  gst_init (&argc, &argv);

  filters = gst_single_filter_manager_new ();
  mainloop = g_main_loop_new (NULL, FALSE);
  pipeline = gst_pipeline_new (NULL);

  bus = gst_element_get_bus (pipeline);
  gst_bus_set_sync_handler (bus, _bus_callback, filters);
  gst_object_unref (bus);

  src = gst_element_factory_make ("autoaudiosrc", NULL);
  sink = gst_element_factory_make ("fakesink", NULL);
  g_assert (src != NULL);
  g_assert (sink != NULL);
  g_object_set (sink, "sync", FALSE, "async", FALSE, NULL);
  src_pad = gst_element_get_static_pad (src, "src");
  sink_pad = gst_element_get_static_pad (sink, "sink");
  g_assert (src_pad != NULL);
  g_assert (sink_pad != NULL);


  filter = GST_FILTER (gst_level_filter_new ());
  gst_filter_manager_append_filter (filters, filter);

  g_signal_connect (filter, "level", (GCallback) got_level, NULL);
  g_object_unref (filter);

  gst_bin_add (GST_BIN (pipeline), src);
  gst_bin_add (GST_BIN (pipeline), sink);

  out_pad = gst_filter_manager_apply (filters, GST_BIN (pipeline), sink_pad);
  g_assert (out_pad != NULL);
  gst_object_unref (sink_pad);
  gst_pad_link (src_pad, out_pad);

  g_timeout_add (5000, done, pipeline);

  gst_element_set_state (pipeline, GST_STATE_PLAYING);

  g_debug ("Running mainloop");
  g_main_loop_run (mainloop);
  g_debug ("Mainloop terminated");

  gst_element_set_state (pipeline, GST_STATE_NULL);

  gst_filter_manager_revert (filters, GST_BIN (pipeline), src_pad);

  gst_object_unref (out_pad);
  out_pad = gst_pad_get_peer (src_pad);

  gst_pad_unlink (src_pad, out_pad);
  gst_element_release_request_pad (src, src_pad);
  gst_object_unref (src_pad);
  gst_bin_remove (GST_BIN (pipeline), src);
  gst_filter_manager_revert (filters, GST_BIN (pipeline), out_pad);
  gst_object_unref (sink_pad);
  gst_object_unref (out_pad);
  gst_bin_remove (GST_BIN (pipeline), sink);

  g_object_unref (filters);
  gst_object_unref (pipeline);
  g_main_loop_unref (mainloop);

  return 0;
}
