/* Copyright (C) 2006-2009, 2010, 2011 P. F. Chimento
 * This file is part of GNOME Inform 7.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gtksourceview/gtksourceiter.h>
#include "extension.h"
#include "app.h"
#include "colorscheme.h"
#include "configfile.h"
#include "document.h"
#include "error.h"
#include "file.h"
#include "lang.h"

typedef struct _I7ExtensionPrivate I7ExtensionPrivate;
struct _I7ExtensionPrivate
{
	/* Built-in extension or not */
	gboolean readonly;
	/* View with elastic tabstops (not saved) */
	gboolean elastic;
};

#define I7_EXTENSION_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), I7_TYPE_EXTENSION, I7ExtensionPrivate))
#define I7_EXTENSION_USE_PRIVATE(o,n) I7ExtensionPrivate *n = I7_EXTENSION_PRIVATE(o)

G_DEFINE_TYPE(I7Extension, i7_extension, I7_TYPE_DOCUMENT);

/* SIGNAL HANDLERS */

static void
on_heading_depth_value_changed(GtkRange *range, I7Extension *extension)
{
	double value = gtk_range_get_value(range);
	i7_document_set_headings_filter_level(I7_DOCUMENT(extension), (gint)value);
}

/* Save window size */
static void
save_extwindow_size(GtkWindow *window)
{
	gint w, h;
	gtk_window_get_size(window, &w, &h);
	config_file_set_int(PREFS_EXT_WINDOW_WIDTH, w);
	config_file_set_int(PREFS_EXT_WINDOW_HEIGHT, h);
}

static gboolean
on_extensionwindow_delete_event(GtkWidget *window, GdkEvent *event)
{
	if(i7_document_verify_save(I7_DOCUMENT(window))) {
		save_extwindow_size(GTK_WINDOW(window));
		return FALSE;
	}
	return TRUE;
}

static void
on_notebook_switch_page(GtkNotebook *notebook, GtkNotebookPage *page, guint page_num, I7Extension *extension)
{
	if(page_num != I7_SOURCE_VIEW_TAB_CONTENTS)
		return;
	i7_document_reindex_headings(I7_DOCUMENT(extension));
}

static void
on_headings_row_activated(GtkTreeView *view, GtkTreePath *path, GtkTreeViewColumn *column, I7Extension *extension)
{
	GtkTreePath *real_path = i7_document_get_child_path(I7_DOCUMENT(extension), path);
	i7_document_show_heading(I7_DOCUMENT(extension), real_path);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(extension->sourceview->notebook), I7_SOURCE_VIEW_TAB_SOURCE);
}

static void
on_previous_action_notify_sensitive(GObject *action, GParamSpec *paramspec, I7Extension *extension)
{
	gboolean sensitive;
	g_object_get(action, "sensitive", &sensitive, NULL);
	if(sensitive)
		gtk_widget_show(extension->sourceview->previous);
	else
		gtk_widget_hide(extension->sourceview->previous);
}

static void
on_next_action_notify_sensitive(GObject *action, GParamSpec *paramspec, I7Extension *extension)
{
	gboolean sensitive;
	g_object_get(action, "sensitive", &sensitive, NULL);
	if(sensitive)
		gtk_widget_show(extension->sourceview->next);
	else
		gtk_widget_hide(extension->sourceview->next);
}

/* IMPLEMENTATIONS OF VIRTUAL FUNCTIONS */

static gchar *
i7_extension_extract_title(I7Document *document, gchar *text)
{
	I7App *app = i7_app_get();
	GMatchInfo *match = NULL;

	if(!g_regex_match(app->regices[I7_APP_REGEX_EXTENSION], text, 0, &match)) {
		g_match_info_free(match);
		return g_strdup("Untitled");
	}

	gchar *title = g_match_info_fetch_named(match, "title");
	g_match_info_free(match);
	if(!title)
		return g_strdup("Untitled");
	return title;
}

static void
i7_extension_set_contents_display(I7Document *document, I7ContentsDisplay display)
{
	i7_source_view_set_contents_display(I7_EXTENSION(document)->sourceview, display);
}

/* Save extension, in the previous location if it exists and is not a directory,
otherwise ask for a new location */
static gboolean
i7_extension_save(I7Document *document)
{
	if(I7_EXTENSION_PRIVATE(document)->readonly) {
		GtkWidget *dialog = gtk_message_dialog_new_with_markup(GTK_WINDOW(document), GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
			_("<big><b>You are editing a built-in Inform extension.</b></big>"));
		gtk_message_dialog_format_secondary_markup(GTK_MESSAGE_DIALOG(dialog),
			_("You are not allowed to overwrite the extensions built into "
			"Inform. Instead, select <i>Save As...</i> or <i>Save a Copy...</i>"
			" from the <i>File</i> menu to save a copy of the extension to a "
			"different file. You can then install the extension to the local "
			"extensions directory by selecting <i>Install Extension</i> from "
			"the <i>File</i> menu, and the compiler will use that extension "
			"instead of the built-in one."));
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		return FALSE;
	}

	gchar *filename = i7_document_get_path(document);
	if(filename && g_file_test(filename, G_FILE_TEST_EXISTS) && !g_file_test(filename, G_FILE_TEST_IS_DIR))
		i7_document_save_as(document, filename);
	else {
		gchar *newname = get_filename_from_save_dialog(document, filename);
		if(!newname)
			return FALSE;
		i7_document_set_path(document, newname);
		i7_document_save_as(document, newname);
		g_free(newname);
	}
	if(filename)
		g_free(filename);
	return TRUE;
}

/* Update the list of recently used files */
static void
update_recent_extension_file(I7Extension *extension, const gchar *filename, gboolean readonly)
{
	GError *err = NULL;
	GtkRecentManager *manager = gtk_recent_manager_get_default();
	gchar *file_uri;
	if((file_uri = g_filename_to_uri(filename, NULL, &err)) == NULL) {
		/* fail discreetly */
		WARN(_("Cannot convert project filename to URI"), err);
		g_error_free(err);
		err = NULL; /* clear error */
	} else {
		/* We use the groups "inform7_project", "inform7_extension", and
		 "inform7_builtin" to determine how to open a file from the recent manager */
		gchar *groups_readonly[] = { "inform7_builtin", NULL };
		gchar *groups_regular[] = { "inform7_extension", NULL };
		GtkRecentData recent_data = {
			NULL, NULL, "text/x-natural-inform", "GNOME Inform 7",
			"gnome-inform7", NULL, FALSE
		};
		recent_data.display_name = g_filename_display_basename(filename);
		/* Use the "begins here" line as the description,
		retrieved from the first line of the text */
		GtkTextIter start, end;
		GtkTextBuffer *buffer = GTK_TEXT_BUFFER(i7_document_get_buffer(I7_DOCUMENT(extension)));
		gtk_text_buffer_get_iter_at_line(buffer, &start, 0);
		gtk_text_buffer_get_iter_at_line(buffer, &end, 0);
		gtk_text_iter_forward_to_line_end(&end);
		recent_data.description = gtk_text_buffer_get_text(buffer, &start, &end, FALSE);
		recent_data.groups = readonly? groups_readonly : groups_regular;
		gtk_recent_manager_add_full(manager, file_uri, &recent_data);
		g_free(recent_data.display_name);
		g_free(recent_data.description);
	}
	g_free(file_uri);
}

/* Save extension in the given directory  */
static void
i7_extension_save_as(I7Document *document, gchar *filename)
{
	GError *err = NULL;

	i7_document_display_status_message(document, _("Saving project..."), FILE_OPERATIONS);

	i7_document_stop_file_monitor(document);

	/* Save the source */
	gchar *text = i7_document_get_source_text(document);
	/* Write text to file */
	if(!g_file_set_contents(filename, text, -1, &err)) {
		error_dialog(GTK_WINDOW(document), err, _("Error saving file '%s': "), filename);
		g_free(text);
		return;
	}
	g_free(text);

	update_recent_extension_file(I7_EXTENSION(document), filename, FALSE);

	/* Start file monitoring again */
	i7_document_monitor_file(document, filename);

	i7_document_set_modified(document, FALSE);

	i7_document_remove_status_message(document, FILE_OPERATIONS);
}

static GtkTextView *
i7_extension_get_default_view(I7Document *document)
{
	return GTK_TEXT_VIEW(I7_EXTENSION(document)->sourceview->source);
}

static void
i7_extension_scroll_to_selection(I7Document *document)
{
	GtkTextBuffer *buffer = GTK_TEXT_BUFFER(i7_document_get_buffer(document));
	GtkTextView *view = GTK_TEXT_VIEW(I7_EXTENSION(document)->sourceview->source);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(I7_EXTENSION(document)->sourceview->notebook), I7_SOURCE_VIEW_TAB_SOURCE);
	gtk_text_view_scroll_to_mark(view, gtk_text_buffer_get_insert(buffer), 0.25, FALSE, 0.0, 0.0);
	gtk_widget_grab_focus(GTK_WIDGET(view));
}

/* Only update the tabs in this extension window */
static void
i7_extension_update_tabs(I7Document *document)
{
	if(!I7_IS_EXTENSION(document))
		return;
	g_idle_add((GSourceFunc)update_tabs, GTK_SOURCE_VIEW(I7_EXTENSION(document)->sourceview->source));
}

static gboolean
update_font_tabs(GtkSourceView *view)
{
	update_font(GTK_WIDGET(view));
	update_tabs(view);
	return FALSE; /* one-shot idle function */
}

/* Update the fonts in this extension window, but not the
widgets that only need their font size updated */
static void
i7_extension_update_fonts(I7Document *document)
{
	if(!I7_IS_EXTENSION(document))
		return;
	g_idle_add((GSourceFunc)update_font_tabs, GTK_SOURCE_VIEW(I7_EXTENSION(document)->sourceview->source));
}

static void
i7_extension_update_font_sizes(I7Document *document)
{
	return; /* No font sizes to update */
}

static void
i7_extension_expand_headings_view(I7Document *document)
{
	gtk_tree_view_expand_all(GTK_TREE_VIEW(I7_EXTENSION(document)->sourceview->headings));
}

static gboolean
do_search(GtkTextView *view, const gchar *text, gboolean forward, const GtkTextIter *startpos, GtkTextIter *start, GtkTextIter *end)
{
	if(GTK_IS_SOURCE_VIEW(view)) {
		if(forward)
			return gtk_source_iter_forward_search(startpos, text, GTK_SOURCE_SEARCH_VISIBLE_ONLY | GTK_SOURCE_SEARCH_TEXT_ONLY | GTK_SOURCE_SEARCH_CASE_INSENSITIVE, start, end, NULL);
		return gtk_source_iter_backward_search(startpos, text, GTK_SOURCE_SEARCH_VISIBLE_ONLY | GTK_SOURCE_SEARCH_TEXT_ONLY | GTK_SOURCE_SEARCH_CASE_INSENSITIVE, start, end, NULL);
	}
	if(forward)
		return gtk_text_iter_forward_search(startpos, text, GTK_TEXT_SEARCH_VISIBLE_ONLY | GTK_TEXT_SEARCH_TEXT_ONLY, start, end, NULL);
	return gtk_text_iter_backward_search(startpos, text, GTK_TEXT_SEARCH_VISIBLE_ONLY | GTK_TEXT_SEARCH_TEXT_ONLY, start, end, NULL);
}

static gboolean
i7_extension_highlight_search(I7Document *document, const gchar *text, gboolean forward)
{
	if(*text == '\0') {
		/* If the text is blank, unhighlight everything and return TRUE */
		i7_document_unhighlight_quicksearch(document);
		return TRUE;
	}

	GtkWidget *focus = I7_EXTENSION(document)->sourceview->source;

	if(gtk_notebook_get_current_page(GTK_NOTEBOOK(I7_EXTENSION(document)->sourceview->notebook)) == I7_SOURCE_VIEW_TAB_CONTENTS) {
		/* Headings view is visible, switch back to source code view */
		gtk_notebook_set_current_page(GTK_NOTEBOOK(I7_EXTENSION(document)->sourceview->notebook), I7_SOURCE_VIEW_TAB_SOURCE);
		gtk_widget_grab_focus(document->findbar_entry);
	}

	i7_document_set_highlighted_view(document, focus);

	/* Source view and text view */
	GtkTextIter iter, start, end;
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(focus));

	/* Start the search at either the beginning or end of the selection
	 depending on the direction */
	GtkTextMark *startmark = forward? gtk_text_buffer_get_selection_bound(buffer) : gtk_text_buffer_get_insert(buffer);
	gtk_text_buffer_get_iter_at_mark(buffer, &iter, startmark);
	if(!do_search(GTK_TEXT_VIEW(focus), text, forward, &iter, &start, &end)) {
		if(forward)
			gtk_text_buffer_get_start_iter(buffer, &iter);
		else
			gtk_text_buffer_get_end_iter(buffer, &iter);
		if(!do_search(GTK_TEXT_VIEW(focus), text, forward, &iter, &start, &end))
			return FALSE;
	}
	gtk_text_buffer_select_range(buffer, &start, &end);
	gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(focus), gtk_text_buffer_get_insert(buffer), 0.25, FALSE, 0.0, 0.0);

	return TRUE;
}

static void
i7_extension_set_spellcheck(I7Document *document, gboolean spellcheck)
{
	i7_source_view_set_spellcheck(I7_EXTENSION(document)->sourceview, spellcheck);
}

static void
i7_extension_check_spelling(I7Document *document)
{
	i7_source_view_check_spelling(I7_EXTENSION(document)->sourceview);
}

static void
i7_extension_set_elastic_tabs(I7Document *document, gboolean elastic)
{
	I7_EXTENSION_USE_PRIVATE(document, priv);
	priv->elastic = elastic;
	i7_source_view_set_elastic_tabs(I7_EXTENSION(document)->sourceview, elastic);
}

/* TYPE SYSTEM */

static void
i7_extension_init(I7Extension *self)
{
	I7_EXTENSION_USE_PRIVATE(self, priv);
	GError *error = NULL;

	priv->readonly = FALSE;

	/* Build the menus and toolbars from the GtkUIManager file */
	gchar *filename = i7_app_get_datafile_path(i7_app_get(), "ui/extension.uimanager.xml");
	gtk_ui_manager_add_ui_from_file(I7_DOCUMENT(self)->ui_manager, filename, &error);
	g_free(filename);
	if(error)
		ERROR(_("Building menus failed"), error);
	GtkWidget *menu = gtk_ui_manager_get_widget(I7_DOCUMENT(self)->ui_manager, "/ExtensionMenubar");
	I7_DOCUMENT(self)->toolbar = gtk_ui_manager_get_widget(I7_DOCUMENT(self)->ui_manager, "/ExtensionToolbar");
	gtk_widget_set_no_show_all(I7_DOCUMENT(self)->toolbar, TRUE);
	i7_document_add_menus_and_findbar(I7_DOCUMENT(self));

	/* Build the rest of the interface */
	gtk_box_pack_start(GTK_BOX(I7_DOCUMENT(self)->box), menu, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(I7_DOCUMENT(self)->box), I7_DOCUMENT(self)->toolbar, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(I7_DOCUMENT(self)->box), I7_DOCUMENT(self)->findbar, FALSE, FALSE, 0);

	/* Create source view */
	self->sourceview = I7_SOURCE_VIEW(i7_source_view_new());
	gtk_widget_show(GTK_WIDGET(self->sourceview));
	gtk_box_pack_start(GTK_BOX(I7_DOCUMENT(self)->box), GTK_WIDGET(self->sourceview), TRUE, TRUE, 0);

	/* Set up the signals to do the menu hints in the statusbar */
	i7_document_attach_menu_hints(I7_DOCUMENT(self), GTK_MENU_BAR(menu));

	/* Build the Open Extensions menu */
	i7_app_update_extensions_menu(i7_app_get());

	/* Set the last saved window size */
	gtk_window_resize(GTK_WINDOW(self), config_file_get_int(PREFS_EXT_WINDOW_WIDTH), config_file_get_int(PREFS_EXT_WINDOW_HEIGHT));

	/* Set up the Natural Inform highlighting */
	GtkSourceBuffer *buffer = i7_document_get_buffer(I7_DOCUMENT(self));
	set_buffer_language(buffer, "inform7x");
	set_highlight_styles(buffer);

	/* Connect other signals */
	g_signal_connect(self->sourceview->heading_depth, "value-changed", G_CALLBACK(on_heading_depth_value_changed), self);
	g_signal_connect(self->sourceview->notebook, "switch-page", G_CALLBACK(on_notebook_switch_page), self);
	g_signal_connect(self->sourceview->headings, "row-activated", G_CALLBACK(on_headings_row_activated), self);

	/* Connect various models to various views */
	gtk_text_view_set_buffer(GTK_TEXT_VIEW(self->sourceview->source), GTK_TEXT_BUFFER(buffer));
	gtk_tree_view_set_model(GTK_TREE_VIEW(self->sourceview->headings), i7_document_get_headings(I7_DOCUMENT(self)));

	/* Connect the Previous Section and Next Section actions to the up and down buttons */
	gtk_action_connect_proxy(I7_DOCUMENT(self)->previous_section, self->sourceview->previous);
	gtk_action_connect_proxy(I7_DOCUMENT(self)->next_section, self->sourceview->next);

	/* We don't need to keep a reference to the buffer and model anymore */
	g_object_unref(buffer);
	g_object_unref(i7_document_get_headings(I7_DOCUMENT(self)));

	/* Set up the Previous Section and Next Section actions to synch with the buttons */
	g_signal_connect(I7_DOCUMENT(self)->previous_section, "notify::sensitive", G_CALLBACK(on_previous_action_notify_sensitive), self);
	g_signal_connect(I7_DOCUMENT(self)->next_section, "notify::sensitive", G_CALLBACK(on_next_action_notify_sensitive), self);
	/* For some reason this needs to be triggered even if the buttons are set to invisible in Glade */
	gtk_action_set_sensitive(I7_DOCUMENT(self)->previous_section, FALSE);
	gtk_action_set_sensitive(I7_DOCUMENT(self)->next_section, FALSE);

	/* Set font sizes, etc. */
	i7_document_update_fonts(I7_DOCUMENT(self));

	/* Set spell checking and elastic tabstops */
	gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(I7_DOCUMENT(self)->autocheck_spelling), config_file_get_bool(PREFS_SPELL_CHECK_DEFAULT));
	i7_document_set_spellcheck(I7_DOCUMENT(self), config_file_get_bool(PREFS_SPELL_CHECK_DEFAULT));
	priv->elastic = config_file_get_bool(PREFS_ELASTIC_TABS_DEFAULT);
	gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(I7_DOCUMENT(self)->enable_elastic_tabs), priv->elastic);

	/* Create a callback for the delete event */
	g_signal_connect(self, "delete-event", G_CALLBACK(on_extensionwindow_delete_event), NULL);
}

static void
i7_extension_finalize(GObject *self)
{
	G_OBJECT_CLASS(i7_extension_parent_class)->finalize(self);
}

static void
i7_extension_class_init(I7ExtensionClass *klass)
{
	I7DocumentClass *document_class = I7_DOCUMENT_CLASS(klass);
	document_class->extract_title = i7_extension_extract_title;
	document_class->set_contents_display = i7_extension_set_contents_display;
	document_class->get_default_view = i7_extension_get_default_view;
	document_class->save = i7_extension_save;
	document_class->save_as = i7_extension_save_as;
	document_class->scroll_to_selection = i7_extension_scroll_to_selection;
	document_class->update_tabs = i7_extension_update_tabs;
	document_class->update_fonts = i7_extension_update_fonts;
	document_class->update_font_sizes = i7_extension_update_font_sizes;
	document_class->expand_headings_view = i7_extension_expand_headings_view;
	document_class->highlight_search = i7_extension_highlight_search;
	document_class->set_spellcheck = i7_extension_set_spellcheck;
	document_class->check_spelling = i7_extension_check_spelling;
	document_class->set_elastic_tabs = i7_extension_set_elastic_tabs;

	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	object_class->finalize = i7_extension_finalize;

	g_type_class_add_private(klass, sizeof(I7ExtensionPrivate));
}

/* PUBLIC FUNCTIONS */

I7Extension *
i7_extension_new(I7App *app, const gchar *filename, const gchar *title, const gchar *author)
{
	I7Extension *extension = I7_EXTENSION(g_object_new(I7_TYPE_EXTENSION, NULL));

	i7_document_set_path(I7_DOCUMENT(extension), filename);

	gchar *text = g_strconcat(title, " by ", author, " begins here.\n\n", title, " ends here.\n", NULL);
	i7_document_set_source_text(I7_DOCUMENT(extension), text);
	i7_document_set_modified(I7_DOCUMENT(extension), TRUE);

	/* Add document to global list */
	i7_app_register_document(app, I7_DOCUMENT(extension));

	/* Bring window to front */
	gtk_widget_show(GTK_WIDGET(extension));
	gtk_window_present(GTK_WINDOW(extension));
	return extension;
}

I7Extension *
i7_extension_new_from_file(I7App *app, const gchar *filename, gboolean readonly)
{
	gchar *fullpath = expand_initial_tilde(filename);
	I7Document *dupl = i7_app_get_already_open(app, filename);
	if(dupl && I7_IS_EXTENSION(dupl)) {
		gtk_window_present(GTK_WINDOW(dupl));
		g_free(fullpath);
		return NULL;
	}

	I7Extension *extension = I7_EXTENSION(g_object_new(I7_TYPE_EXTENSION, NULL));
	if(!i7_extension_open(extension, fullpath, readonly)) {
		g_free(fullpath);
		g_object_unref(extension);
		return NULL;
	}
	g_free(fullpath);

	/* Add document to global list */
	i7_app_register_document(app, I7_DOCUMENT(extension));

	/* Bring window to front */
	gtk_widget_show(GTK_WIDGET(extension));
	gtk_window_present(GTK_WINDOW(extension));
	return extension;
}

I7Extension *
i7_extension_new_from_uri(I7App *app, const gchar *uri, gboolean readonly)
{
	GError *error = NULL;
	I7Extension *extension = NULL;

	gchar *filename;
	if((filename = g_filename_from_uri(uri, NULL, &error)) == NULL) {
		WARN_S(_("Cannot get filename from URI"), uri, error);
		g_error_free(error);
		return NULL;
	}

	extension = i7_extension_new_from_file(app, filename, readonly);
	return extension;
}

/* Opens the extension from filename, and returns success. */
gboolean
i7_extension_open(I7Extension *extension, const gchar *filename, gboolean readonly)
{
	i7_document_set_path(I7_DOCUMENT(extension), filename);

	/* If it was a built-in extension, set it read-only */
	i7_extension_set_read_only(extension, readonly);

	/* Read the source */
	gchar *text = read_source_file(filename);
	if(!text)
		return FALSE;

	update_recent_extension_file(extension, filename, readonly);

	/* Watch for changes to the source file */
	i7_document_monitor_file(I7_DOCUMENT(extension), filename);

	/* Write the source to the source buffer, clearing the undo history */
	i7_document_set_source_text(I7_DOCUMENT(extension), text);
	g_free(text);

	GtkTextIter start;
	GtkTextBuffer *buffer = GTK_TEXT_BUFFER(i7_document_get_buffer(I7_DOCUMENT(extension)));
	gtk_text_buffer_get_start_iter(buffer, &start);
	gtk_text_buffer_place_cursor(buffer, &start);

	i7_document_set_modified(I7_DOCUMENT(extension), FALSE);

	return TRUE;
}

void
i7_extension_set_read_only(I7Extension *extension, gboolean readonly)
{
	I7_EXTENSION_PRIVATE(extension)->readonly = readonly;
}
