/*
 * which()
 *  -searches PATH for executable.
 *
 * whereami()
 *  -for the running program to find the directory of itself.
 *
 * spawnwait()
 *  -because system() ain't right.
 *
 * The creator of this file hereby gives his contribution to public domain.
 */

#ifdef _WIN32
# define CHAR_DIRSEP '\\'
# define CHAR_PATHSEP ';'
# define PROGLEN_ADD 6 /* strlen("\\")+strlen("\0")+strlen(".exe") */
#else
# define CHAR_DIRSEP '/'
# define CHAR_PATHSEP ':'
# define PROGLEN_ADD 2
#endif

#include <sys/stat.h> /* stat   */
#include <stdlib.h>   /* malloc */
#include <string.h>   /* strlen */
#include <errno.h>

char *which(const char *program){
	char *find;
	if(strchr(program, CHAR_DIRSEP)){
		/* We could returned program, but later,
		 * we would not know when not to free() it.
		 */
		find = strdup(program);
		if(!find) goto fail;
	}else{
		char *path = getenv("PATH");
		if(!path) goto fail;
		size_t proglen = strlen(program) + PROGLEN_ADD;
		size_t findcap=24;
		find = malloc(findcap);
		if(!find) goto fail;
		char *partstart, *partend;
		for(partend=path; ; partend++){
			partstart = partend;
			while(*partend && *partend != CHAR_PATHSEP) partend++;
			size_t req = partend - partstart + proglen;
			if(req > findcap){
				free(find);
				find = malloc(req);
				if(!find) goto fail;
				findcap = req;
			}
			memcpy(find, partstart, partend-partstart);
			find[partend-partstart] = CHAR_DIRSEP;
			strcpy(find + (size_t)(partend-partstart) + 1, program);
#ifdef _WIN32
			strcat(find + (size_t)(partend-partstart) + 2, ".exe");
#endif
			struct stat info;
			if(stat(find, &info) == 0           //exists
			&&(info.st_mode & S_IFMT) == S_IFREG//regular file
			&& info.st_mode & 0111){            //executable
				break;
			}
			if(*partend == '\0'){
				free(find);
				errno = ENOENT;
fail:
				return NULL;
			}
		}
	}
	return find;
}

char *whereami(const char *program){
	char *path = which(program);
	if(!path) return NULL;
	char *find, *base=path;
	while(base[0] == '.' && base[1] == CHAR_DIRSEP) base += 2;
	char *end=base;
	for(find=base; *find; find++) if(*find == CHAR_DIRSEP) end = find;
	end[1] = '\0';
	if(base != path){
		base = strdup(base);
		free(path);
	}
	return base;
}

/*         SYSTEM() IS WRONG
 * Because it invokes the shell.
 * 1. You must counterwork the shell by quoting and concatenating arguments.
 * 2. You can neither quote, nor concatenate without knowing the rules of
 *    the shell. Which shell? Bash is very different from the Windows shell.
 *    Guessing by the OS is naive.
 *
 * This realloc()-happy string processing is pointless when you know that the
 * shell reverses all of it to create the array of arguments you already had!
 *
 *         A BETTER SYSTEM()
 * Like system(), wait for the child process to finish, always return.
 * Like execve(), take program arguments as null terminated array of char pointers.
 * Like system(), do not specify the program name twice.
 * Unlike system(), do not go via the shell.
 *
 *         RETURN VALUE
 * On success, return the return value of the child process.
 * On failure, return -errno.
 */

//FIXME
#ifdef _WIN32
# define HAVE_SPAWNVE
#else
# define HAVE_POSIX_SPAWN
#endif

#ifdef HAVE_POSIX_SPAWN
/*
 * Linux has posix_spawn. Perfect.
 */
# include <spawn.h>
# include <sys/wait.h>

int spawnwait(char *args[], char *envp[]){
	int ret;
	pid_t childpid;
	ret = posix_spawn(&childpid, args[0], NULL, NULL, args, envp);
	if(ret){
		return -ret;
	}
	while(waitpid(childpid, &ret, 0) == -1){
		if (errno == EINTR) continue;
		return -errno;
	}
	return WEXITSTATUS(ret);
}

#elif defined HAVE_SPAWNVE
/*
 * Windows has the braindead _spawnve.
 * Although _spawnve avoids the shell, it does exactly what we hate!
 * And that particular behavior is undocumented...
 * Well, think of it as a leaky abstraction of CreateProcess:
 * http://msdn.microsoft.com/en-us/library/ms682425.aspx
 */
# include <process.h>

int spawnwait(char *args[], char *envp[]){
	char *prog = args[0];

	//begin counterwork
	char catbuf[0x400];
	unsigned cat=0;
	goto loopstart;
	for(; *args; args++){
		if(cat >= 0x400-3) return E2BIG;
		catbuf[cat++] = ' ';
		loopstart:
		catbuf[cat++] = '"';
		char *cp;
		for(cp = *args; *cp; cp++){
			if(cat >= 0x400-2) return E2BIG;
			catbuf[cat++] = *cp;
			if(*cp == '"'){
				if(cat >= 0x400-2) return E2BIG;
				catbuf[cat++] = *cp;
			}
		}
		catbuf[cat++] = '"'; //have accounted for this
	}
	catbuf[cat] = '\0'; //have accounted for this
	char *newargs[] = {catbuf, NULL};
	//end counterwork

	int ret = _spawnve(
		_P_WAIT,
		(const char*)prog,
		(const char* const*)newargs,
		(const char* const*)envp
	);
	if(ret == -1) ret = -errno;
	return ret;
}

#else
/* This implementation was adapted from a code sample of the
 * linux man page of system (3p), to meet the specification above.
 * The original can be found (as of June 18 2009) at
 * http://www.opengroup.org/onlinepubs/000095399/functions/system.html
 * and contains the following copyright notice:
 *  The Open Group Base Specifications Issue 6
 *  IEEE Std 1003.1, 2004 Edition
 *  Copyright © 2001-2004 The IEEE and The Open Group, All Rights reserved.
 * The role of the original code sample is described in the man page:
 * "The following code sample illustrates how system() might be
 * implemented on an implementation conforming to IEEE Std 1003.1-2001."
 */
# include <unistd.h>
# include <signal.h>
# include <errno.h>

int spawnwait(char *args[], char *envp[]){
	int ret;
	pid_t pid;
	struct sigaction sa, savintr, savequit;
	sigset_t saveblock;
	sa.sa_handler = SIG_IGN;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = 0;
	sigemptyset(&savintr.sa_mask);
	sigemptyset(&savequit.sa_mask);
	sigaction(SIGINT, &sa, &savintr);
	sigaction(SIGQUIT, &sa, &savequit);
	sigaddset(&sa.sa_mask, SIGCHLD);
	sigprocmask(SIG_BLOCK, &sa.sa_mask, &saveblock);
	if ((pid = vfork()) == 0) {
		sigaction(SIGINT, &savintr, (struct sigaction *)0);
		sigaction(SIGQUIT, &savequit, (struct sigaction *)0);
		sigprocmask(SIG_SETMASK, &saveblock, (sigset_t *)0);
		execve(args[0], args, envp);
		_exit(-errno); // errno comes from execve()
	} else if (pid == -1) {
		ret = -errno; // errno comes from vfork()
	} else {
		while (waitpid(pid, &ret, 0) == -1) {
			if (errno == EINTR) continue;
			ret = -errno; // errno comes from waitpid()
			break;
		}
	}
	sigaction(SIGINT, &savintr, (struct sigaction *)0);
	sigaction(SIGQUIT, &savequit, (struct sigaction *)0);
	sigprocmask(SIG_SETMASK, &saveblock, (sigset_t *)0);
	return (ret < 0) ? ret : WEXITSTATUS(ret);
}

#endif
