/*
 *      EasyPMP playlist conversion.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: playlist.c 339 2007-02-11 17:33:05Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>
#include <playlist.h>

#include <easypmp.h>

typedef struct {
	void *instance;
	easypmp_progress_t progress;
} callback_data_t;

static void callback_from_playlist(void *instance, int level, ucs2char_t* message)
{
	int phase = EASYPMPP_PLAYLIST_CONVERT;
	callback_data_t* cd = (callback_data_t*)instance;

	switch (level) {
	case PLCALLBACK_JSPL_MESSAGE:		phase |= EASYPMPSP_JSPL_OUTPUT;		break;
	case PLCALLBACK_JSPL_ERROR:			phase |= EASYPMPSP_JSPL_ERROR;		break;
	case PLCALLBACK_JSPL_ERROR_POS:		phase |= EASYPMPSP_JSPL_ERROR_POS;	break;
	case PLCALLBACK_JSPL_ERROR_LINE:	phase |= EASYPMPSP_JSPL_ERROR_LINE;	break;
	default:							phase |= EASYPMPSP_PROGRESS;		break;
	}

	cd->progress(cd->instance, phase, 0, 0, message);
}

int
easypmp_playlist(
	pmp_t* pmp,
	const easypmp_filelist_t* playlists,
	const easypmp_filelist_t* musics,
	const option_t* opt,
	pmp_music_record_t* records,
	uint32_t num_records,
	easypmp_progress_t progress,
	void *instance
	)
{
	int result = 0;
	int i, j, num_succeeded = 0;
	char *mbs = NULL;
	pmp_music_t* music = NULL;
	playlist_mediafile_t* mediafiles = NULL;
	pmp_playlist_t* pmppls = NULL;
	int num_pmppls = 0;
	result_t res = 0;
	callback_data_t cd;

	cd.instance = instance;
	cd.progress = progress;

	// Query the interface to music.
	music = pmp->music;

	// Convert easypmp_filelist_t array to playlist_mediafile_t array.
	mediafiles = calloc(musics->num_elements, sizeof(playlist_mediafile_t));
	if (musics->num_elements > 0 && !mediafiles) {
		result = EASYPMPE_INSUFFICIENT_MEMORY;
		goto error_exit;
	}
	for (i = 0;i < musics->num_elements;++i) {
		ucs2cpy(mediafiles[i].path, musics->elements[i].pathname);
		ucs2cpy(mediafiles[i].file, musics->elements[i].filename);
	}

	// Prepare playlist conversion for finding music files.
	playlist_normalize_prepare(mediafiles, musics->num_elements);

	// Start playlist conversion.
	if (progress(instance, EASYPMPP_PLAYLIST_CONVERT | EASYPMPSP_START, playlists->num_elements, 0, NULL) != 0) {
		result = EASYPMPE_CANCEL;
		goto error_exit;
	}

	// Loop over playlist files.
	for (i = 0;i < playlists->num_elements;++i) {
		playlists_t pls;
		ucs2char_t plf_path[MAX_PATH];
		ucs2char_t src[MAX_PATH];
		const easypmp_filename_t* plf = &playlists->elements[i];
		int k = 0;
		int n = 0;

		// Generate the source filename
		ucs2cpy(plf_path, plf->pathname);
		filepath_decode(plf_path);
		filepath_combinepath(src, MAX_PATH, plf_path, plf->filename);

		// Report the source filename.
		if (progress(instance, EASYPMPP_PLAYLIST_CONVERT | EASYPMPSP_PROGRESS, i, 0, src) != 0) {
			result = EASYPMPE_CANCEL;
			goto error_exit;
		}

		// Initialize playlist(s) instance.
		playlist_init(&pls);

		// Read the source playlist.
		if (playlist_read(
			&pls,
			src,
			opt->path_to_include,
			opt->verb & MODE_PLAYLIST_JSPL ? records : NULL,
			num_records,
			callback_from_playlist,
			&cd
			) != 0) {
			result = EASYPMPE_PLAYLIST_READ;
			goto error_exit;
		}

		for (k = 0;k < pls.num_playlists;++k) {
			ucs2char_t** contents = NULL;
			playlist_t* pl = &pls.playlists[k];
			ucs2char_t dst[MAX_PATH];
			pmp_playlist_t* pmppl = NULL;

			// Generate a destination filename.
			ucs2cpy(dst, filepath_skippath(pl->name));
			filepath_remove_extension(dst);

			// Skip an empty playlist.
			if (pl->num_entries == 0) {
				if (progress(instance, EASYPMPP_PLAYLIST_CONVERT | EASYPMPSP_SKIP_PLAYLIST, i, 0, src) != 0) {
					result = EASYPMPE_CANCEL;
					goto error_exit;
				}
				continue;
			}

			// Normalize the playlist.
			n = playlist_normalize(
				pl,
				pl->name,
				pmp->info.path_to_root,
				mediafiles,
				musics->num_elements
				);
			if (n < pl->num_entries) {
				if ((opt->verb & MODE_PLAYLIST_SKIP) && 0 < n) {
					// Report a warning.
					if (progress(instance, EASYPMPP_PLAYLIST_CONVERT | EASYPMPSP_WARN_PLAYLIST, i, 0, src) != 0) {
						result = EASYPMPE_CANCEL;
						goto error_exit;
					}
				} else {
					// Report a error (skipping).
					if (progress(instance, EASYPMPP_PLAYLIST_CONVERT | EASYPMPSP_SKIP_PLAYLIST, i, 0, src) != 0) {
						result = EASYPMPE_CANCEL;
						goto error_exit;
					}
				}

				// Report missing media files.
				for (j = 0;j < pl->num_entries;++j) {
					if (!pl->entries[j].valid) {
						if (progress(instance, EASYPMPP_PLAYLIST_CONVERT | EASYPMPSP_MISSING_MEDIA, i, 0, pl->entries[j].filename) != 0) {
							result = EASYPMPE_CANCEL;
							goto error_exit;
						}
					}
				}

				// Skip if "skip missing" flag is unset or the number of elements is zero.
				if (!(opt->verb & MODE_PLAYLIST_SKIP) || n == 0) {
					continue;
				}
			}

			pmppls = (pmp_playlist_t*)realloc(pmppls, sizeof(pmp_playlist_t) * (num_pmppls+1));
			pmppl = &pmppls[num_pmppls];

			// Allocate a memory block for the contents of the playlist.
			pmppl->entries = (ucs2char_t**)calloc(pl->num_entries, sizeof(ucs2char_t*));
			pmppl->num_entries = pl->num_entries;
			pmppl->name = ucs2dup(dst);

			if (!pmppl->entries) {
				result = EASYPMPE_INSUFFICIENT_MEMORY;
				goto error_exit;
			}

			// Store the playlist.
			for (j = 0, n = 0;j < pl->num_entries;++j) {
				if (pl->entries[j].valid && pl->entries[j].filename[0]) {
					pmppl->entries[n] = ucs2dup(pl->entries[j].filename);
					++n;
				}
			}

			++num_succeeded;
			++num_pmppls;
		}
	}

	// Set playlists.
	music->set_playlists(music, pmppls, num_pmppls);

	// Finish playlist conversion.
	if (progress(instance, EASYPMPP_PLAYLIST_CONVERT | EASYPMPSP_END, num_succeeded, 0, NULL) != 0) {
		result = EASYPMPE_CANCEL;
		goto error_exit;
	}

error_exit:
	for (i = 0;i < num_pmppls;++i) {
		pmplib_playlist_finish(&pmppls[i]);
	}
	free(pmppls);
	free(mediafiles);
	return result;
}
