/*
 *      EasyPMP [CUI] main.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: main.c 340 2007-02-11 17:45:15Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <locale.h>
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/
#ifdef	_MSC_VER
#include <direct.h>		/* getcwd() */
#endif/*_MSC_VER*/

#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <gmi.h>
#include <pmplib/pmp.h>
#include <easypmp.h>

#include "option.h"
#include "console.h"
#include "util.h"

#ifdef	_WIN32
#include <windows.h>
#endif

#define	APPLICATION_S	"EasyPMP [CUI]"
#define	VERSION_S		"0.14 alpha"
#define	COPYRIGHT_S		"Copyright (c) 2005-2007 PMPlib Project"

int database_dump(pmp_t* pmp, FILE *fpo, int level);
void device_show_information(pmp_t* pmp, FILE *fpe);
void device_enumerate(pmplib_t* pmplib);

int
easypmp_progress_num_str(
	FILE *fp,
	size_t n,
	const ucs2char_t* msg);

static int
easypmp_enumerate_progress(
	void *instance,
	const ucs2char_t* path,
	const ucs2char_t* file,
	size_t n)
{
	FILE *fpe = stdout;
	easypmp_progress_num_str(fpe, n, file);
	return 0;
}


static int
easypmp_progress(
	void *instance,
	int phase,
	int param_int,
	double param_float,
	ucs2char_t* param_str
	)
{
	FILE *fpo = stdout;
	FILE *fpe = stderr;

	switch (phase) {
	case EASYPMPP_START:
		break;
	case EASYPMPP_OPEN|EASYPMPSP_START:
		fprintf(fpo, "Opening the portable media device\n");
		break;
	case EASYPMPP_OPEN|EASYPMPSP_END:
		fprintf(fpo, "\n");
		break;
	case EASYPMPP_CLOSE|EASYPMPSP_START:
		fprintf(fpo, "Closing the device\n");
		break;
	case EASYPMPP_CLOSE|EASYPMPSP_END:
		fprintf(fpo, "\n");
		break;
	case EASYPMPP_END:
		break;

	case EASYPMPP_MUSIC_GMI|EASYPMPSP_START:
		fprintf(fpo, "Obtaining media information from %d files\n", param_int);
		break;
	case EASYPMPP_MUSIC_GMI|EASYPMPSP_PROGRESS:
		easypmp_progress_num_str(fpo, param_int+1, filepath_skippath(param_str));
		break;
	case EASYPMPP_MUSIC_GMI|EASYPMPSP_WARN_FAILURE:
		console_clearln(fpo);
		fprintf(fpe, "  WARNING: Could not obtain the media information\n");
		fprints(fpe, "    %s\n", param_str);
		break;
	case EASYPMPP_MUSIC_GMI|EASYPMPSP_WARN_EMPTY:
		console_clearln(fpo);
		fprintf(fpe, "  WARNING: Empty artist or album name found\n");
		fprints(fpe, "    %s\n", param_str);
		break;
	case EASYPMPP_MUSIC_GMI|EASYPMPSP_END:
		console_clearln(fpo);
		fprintf(fpo, "  %d files were imported\n", param_int);
		break;
	case EASYPMPP_MUSIC_UPDATE|EASYPMPSP_START:
		fprintf(fpo, "Updating database entries\n");
		break;
	case EASYPMPP_MUSIC_UPDATE|EASYPMPSP_END:
		fprintf(fpo, "\n");
		break;

	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_START:
		fprintf(fpo, "Converting playlists\n");
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_PROGRESS:
		easypmp_progress_num_str(fpo, param_int+1, filepath_skippath(param_str));
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_WARN_PLAYLIST:
		fprintf(fpe, "\n");
		fprintf(fpe, "    WARNING: following media files are missing:\n");
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_SKIP_PLAYLIST:
		fprintf(fpe, "\n");
		fprintf(fpe, "    ERROR: following media files are missing:\n");
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_MISSING_MEDIA:
		fprints(fpe, "      %s\n", param_str);
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_JSPL_ERROR:
		fprintf(fpe, "\n");
		fprints(fpe, "  JSPL(ERROR): %s\n", param_str);
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_JSPL_ERROR_POS:
		fprints(fpe, "    at: %s\n", param_str);
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_JSPL_ERROR_LINE:
		fprints(fpe, "    line: %s\n", param_str);
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_JSPL_OUTPUT:
		fprints(fpo, "  JSPL: %s\n", param_str);
		break;
	case EASYPMPP_PLAYLIST_CONVERT|EASYPMPSP_END:
		console_clearln(fpo);
		fprintf(fpo, "  %d files were successfully converted\n", param_int);
		fprintf(fpo, "\n");
		break;
	}
	fflush(fpe);

	return 0;
}

#ifndef		_WIN32

int set_encoding(const char *encoding)
{
	if (ucs2setenc(encoding) != 0) {
		fprintf(stderr, "ERROR: iconv does not support %s\n", encoding);
		return 1;
	}
}
#endif

int main(int argc, char *argv[])
{
	int ret = 0;
	result_t res = 0;
	int used_args = 0;
	char *mbs = NULL;
	pmp_t* pmp = NULL;
	pmplib_t* pmplib = NULL;
	easypmp_filelist_t musics, playlists;
	pmp_music_record_t* records = NULL;
	uint32_t num_records = 0;
	uint32_t openflag = 0;
	FILE *fpi = stdin, *fpo = stdout, *fpe = stderr;
	option_t opt;

	// Initialize various routines.
	setlocale(LC_ALL, "");
	ucs2init();
	console_init();

	// Initialize option values.
	option_init(&opt);

	// Show copyright information.
	fprintf(fpo, APPLICATION_S " " VERSION_S "  " COPYRIGHT_S "\n");
	fprintf(fpo, "\n");

	// Parse the command-line arguments.
	used_args = option_parse(&opt, argc, argv, fpe);
	if (used_args < 0) {
		ret = -1;
		goto exit_main;
	}

#ifndef	_WIN32
	if (opt.system_encoding) {
		set_encoding(opt.system_encoding);
	}
#endif/*_WIN32*/

	// Set words to be stripped from artist names.
	if (opt.mbs_strip_words) {
		ucs2char_t* ucs2str = mbsdupucs2(opt.mbs_strip_words);
		easypmp_set_strip_words(&opt, ucs2str);
		ucs2free(ucs2str);
		opt.media_info_source |= GMIF_STRIP_ARTIST;
	}

	// Obtain the path to root directory (path_to_root) from the command line if any.
    if (used_args < argc) {
		size_t length = strlen(argv[used_args]);
		if (length > 0) {
			mbstoucs2(opt.path_to_root, MAX_PATH, argv[used_args], strlen(argv[used_args])+1);
		}
		++used_args;
	}

	// Exit when version mode specified.
	if (opt.verb & MODE_VERSION) {
		ret = 0;
		goto exit_main;
	}

	// Show help message and exit.
	if (opt.verb & MODE_HELP) {
		option_usage(fpo, argv[0]);
		ret = 0;
		goto exit_main;
	}

	// Show help message for variables and exit.
	if (opt.verb & MODE_HELP_VARIABLE) {
		option_usage_variable(fpo);
		ret = 0;
		goto exit_main;
	}

	// Initialize PMPlib library.
	if (res = pmplib_init(&pmplib)) {
		fprintf(fpe, "ERROR: Failed to initialize PMPlib library (0x%X)\n", res);
		ret = 1;
		goto exit_main;
	}

	// Show the list of supported devices and exit.
	if (opt.verb & MODE_LIST_DEVICES) {
		device_enumerate(pmplib);
		ret = 0;
		goto exit_main;
	}

	// Convert the source directory for playlist conversion described in relative path.
	if (opt.path_to_playlist_source) {
		if (filepath_is_relative(opt.path_to_playlist_source)) {
			char pwd[MAX_PATH+1];
			ucs2char_t pwd_ucs2[MAX_PATH+1];
			ucs2char_t tmp[MAX_PATH+1];

			getcwd(pwd, MAX_PATH);
			mbstoucs2(pwd_ucs2, MAX_PATH, pwd, strlen(pwd)+1);

			filepath_relative_to_absolute(tmp, pwd_ucs2, opt.path_to_playlist_source);
			ucs2free(opt.path_to_playlist_source);
			opt.path_to_playlist_source = ucs2dup(tmp);
		}
	}

	// Generate the path_to_root (path to the root directory of a device).
	if (!opt.path_to_root[0]) {
		// If path_to_root is not set, use the current directory.
		char pwd[MAX_PATH+1];
		getcwd(pwd, MAX_PATH);
		mbstoucs2(opt.path_to_root, MAX_PATH, pwd, strlen(pwd)+1);
	}
	filepath_addslash(opt.path_to_root);

	// Generate the path_to_include
	if (!opt.path_to_include[0]) {
#ifdef	_WIN32
		// "jspl" folder under the location where the executable is installed.
		ucs2char_t ucs2cs_jspl[] = {'j','s','p','l',0};

		GetModuleFileNameW(GetModuleHandle(NULL), opt.path_to_include, MAX_PATH);
		filepath_remove_filespec(opt.path_to_include);
		filepath_addslash(opt.path_to_include);
		ucs2cat(opt.path_to_include, ucs2cs_jspl);
		filepath_addslash(opt.path_to_include);
#else
		// Obtain JavaScript Playlist directory from PMP_JSPLINCL_DIR
		mbstoucs2(opt.path_to_include, MAX_PATH, PMP_JSPL_DIR, strlen(PMP_JSPL_DIR)+1);
#endif
	}

	// Create a PMP instance.
	res = pmplib_create(pmplib, &pmp, opt.path_to_root, opt.model);
	if (!pmp) {
		fprintf(fpe, "ERROR: Failed to find a player (0x%X)\n", res);
		ret = 1;
		goto exit_main;
	}

	// Modify the default setting.
	if (opt.path_to_pmp_music) {
		ucs2cpy(pmp->info.path_to_music, opt.path_to_pmp_music);
	}
	if (opt.path_to_pmp_playlist) {
		ucs2cpy(pmp->info.path_to_playlist, opt.path_to_pmp_playlist);
	}

	// Generate open flag.
	if (opt.verb & MODE_DATABASE) {
		openflag |= PMPOF_MUSIC_DB_WRITE;
		if (opt.verb & MODE_DATABASE_UPDATE) {
			openflag |= PMPOF_MUSIC_DB_READ;
		}
	}
	if (opt.verb & MODE_DATABASE_REPR) {
		openflag |= PMPOF_MUSIC_DB_READ;
	}
	if (opt.verb & MODE_PLAYLIST) {
		openflag |= PMPOF_MUSIC_PL_WRITE;
	}

	// Open the portable media device.
	if (easypmp_progress(NULL, EASYPMPP_OPEN | EASYPMPSP_START, 0, 0, NULL) != 0) {
		ret = 1;
		goto exit_main;
	}
	res = pmp->open(pmp, openflag);
	if (res) {
		fprintf(fpe, "ERROR: Failed to open the player (0x%X)\n", res);
		ret = 1;
		goto exit_main;
	}
	if (easypmp_progress(NULL, EASYPMPP_OPEN | EASYPMPSP_END, 0, 0, NULL) != 0) {
		ret = 1;
		goto exit_main;
	}

	// Show player information.
	device_show_information(pmp, fpo);
	fprintf(fpo, "\n");

	// Enumerate music files for database and playlist modes.
	memset(&musics, 0, sizeof(musics));
	if ((opt.verb & MODE_DATABASE) || (opt.verb & MODE_PLAYLIST)) {
		fprintf(fpo, "Enumerating music files in ");
		fputucs2str_path(fpo, pmp->info.path_to_root);
		fputucs2str_path(fpo, pmp->info.path_to_music);
		fprintf(fpo, "\n");
		easypmp_enumerate_music(&musics, pmp, &opt, easypmp_enumerate_progress, NULL);
		console_clearln(fpo);
		fprintf(fpo, "  %d music files were found\n", musics.num_elements);
		fprintf(fpo, "\n");
	}

	// Enumerate playlist files for playlist mode.
	memset(&playlists, 0, sizeof(playlists));
	if (opt.verb & MODE_PLAYLIST) {
		fprintf(fpe, "Enumerating playlist files in ");
		if (opt.path_to_playlist_source) {
			fprints(fpo, "%s", opt.path_to_playlist_source);
		} else {
			fputucs2str_path(fpo, pmp->info.path_to_root);
			fputucs2str_path(fpo, pmp->info.path_to_playlist);
		}
		fprintf(fpo, "\n");
		easypmp_enumerate_playlist(&playlists, pmp, &opt, easypmp_enumerate_progress, NULL);
		console_clearln(fpo);
		fprintf(fpo, "  %d playlist files were found\n", playlists.num_elements);
		fprintf(fpo, "\n");
	}

	// Update database entries for database mode and JSPL playlists.
	if ((opt.verb & MODE_DATABASE) || (opt.verb & MODE_PLAYLIST_JSPL)) {
		easypmp_database(pmp, &musics, &opt, &records, &num_records, easypmp_progress, NULL);
	}

	// Convert playlists.
	if (opt.verb & MODE_PLAYLIST) {
		easypmp_playlist(pmp, &playlists, &musics, &opt, records, num_records, easypmp_progress, NULL);
	}

	// Dump the database if specified.
	if (opt.verb & MODE_DATABASE_REPR) {
		database_dump(pmp, fpo, opt.repr_level);
	}

	// Free the music records.
	if (records && num_records) {
		pmplib_records_finish(records, num_records);
	}

	// Close the portable media device.
	if (easypmp_progress(NULL, EASYPMPP_CLOSE | EASYPMPSP_START, 0, 0, NULL) != 0) {
		ret = 1;
		goto exit_main;
	}
	res = pmp->close(pmp);
	if (res) {
		fprintf(fpe, "ERROR: Failed to close the player (0x%X)\n", res);
		ret = 1;
		goto exit_main;
	}
	if (easypmp_progress(NULL, EASYPMPP_CLOSE | EASYPMPSP_END, 0, 0, NULL) != 0) {
		ret = 1;
		goto exit_main;
	}

	// Uninitialize.
	pmp->release(pmp);
	pmplib_finish(pmplib);
	option_finish(&opt);

	return ret;

exit_main:
	if (pmp) {
		pmp->release(pmp);
		pmp = NULL;
	}
	if (pmplib) {
		pmplib_finish(pmplib);
		pmplib = NULL;
	}
	option_finish(&opt);
	return ret;
}
