/*
 *      UCS-2 character set library.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ucs2char.h 328 2007-02-10 17:50:11Z nyaochi $ */

#ifndef	__UCS2CHAR_H__
#define	__UCS2CHAR_H__

#include <stdio.h>

#ifdef	UCS2_EXPORTS
#define	UCS2API	__declspec(dllexport)
#else
#define	UCS2API
#endif

#ifdef	__cplusplus
extern "C" {
#endif/*__cplusplus*/

/** 
 * \addtogroup ucs2 UCS-2 Character/String API
 * @{
 *
 *	The PMPlib UCS-2 Character/String API provides a manupulation utility
 *	for characters and strings encoded in UCS-2. The PMPlib supports unicode
 *	characters and strings in UCS-2 encoding since a number of media players
 *	employ the encoding for displaying song information. An application is
 *	required to deal with UCS-2 characters/strings for using the PMPlib API.
 *	The PMPlib UCS-2 Character/String API implements character encoding
 *	converter, UCS-2 character routine, ANSI C compatible string manupulator,
 *	miscellaneous string manupulator, stdio/stdlib compatible routine, and
 *	memory manager.
 *
 *	A UCS-2 character is represented by a unsigned 16-bit value
 *	(::ucs2char_t).
 *	The byte order of a UCS-2 character is dependent on the CPU architecture
 *	on which this code runs, e.g., little-endian on Intel IA-32/IA-64 and
 *	big-endian on IBM PowerPC.
 *
 *	An application must call ucs2init() before using this API.
 */

/**
 * Type definition of a UCS-2 character.
 */
typedef uint16_t ucs2char_t;

/**
 * Initialize the UCS-2 Character/String library.
 *
 *	This function initializes internal variables used in the UCS-2
 *	Character/String library. Call this function before using any other
 *	functions in this API.
 *
 *	By default, this function detects the character encoding of multi-byte
 *	characters on the current system based on the value of LANG variable
 *	(for POSIX) or via GetACP() function (for Win32). If this function
 *	could not detect the character encoding of the current system, this
 *	function would assume it as "UTF-8".
 *
 *	@retval	int			Zero if succeeded, otherwise non-zero value. Non-zero
 *						value indicates that the library could not initialize
 *						character conversion routines with the character
 *						encoding of the current locale.
 */
UCS2API int ucs2init();


/**
 * @defgroup ucs2_conv Character encoding converter
 * @{
 *
 *	Character encoding converter is to convert a string in one character
 *	encoding to another. The API subset supports mutual conversions between:
 *		UCS-2 and multi-byte character (i.e., \c char);
 *		UCS-2 and UTF-8.
 *	Character encoding conversion is performed by MultiByteToWideChar() and
 *	WideCharToMultiByte() function in Win32 API (for Windows environments)
 *	iconv() function in libc or libiconv (for POSIX environments).
 */

/**
 * Set the encoding for multi-byte characters (for iconv/libiconv).
 *
 *	This function changes the default encoding for multi-byte characters to the
 *	character encoding specified by the \p encoding argument. 
 *
 *	@param	encoding	The pointer to the string specifying the character
 *						encoding.
 *
 *	@note
 *		This function is effective only on environments with iconv (libiconv).
 */
UCS2API int ucs2setenc(const char *encoding);

/**
 * Get the encoding for multi-byte characters (for iconv/libiconv).
 *
 *	This function returns the default encoding for multi-byte characters used
 *	in the UCS-2 API.
 *	
 *	@retval	const char*	The pointer to the string of the character encoding.
 *
 *	@note
 *		This function is effective only on environments with iconv (libiconv).
 */
UCS2API const char *ucs2getenc();

/**
 * Set the code page for multi-byte characters (for Win32).
 *
 *	This function changes the default encoding for multi-byte characters to the
 *	code page specified by the \p cp argument. 
 *
 *	@param	cp			The code page.
 *
 *	@note
 *		This function is effective only on Win32 environments.
 */
UCS2API void ucs2setcp(int cp);

/**
 * Get the code page for multi-byte characters (for Win32).
 *
 *	This function returns the default code page for multi-byte characters. 
 *
 *	@note
 *		This function is effective only on Win32 environments.
 */
UCS2API int ucs2getcp();

/**
 * Convert a UCS-2 string to multi-byte characters.
 *
 *	@param	mbstr		The pointer to the buffer that receives multi-byte
 *						characters converted from the UCS-2 string. If
 *						\p mbs_size is zero, this argument is not be used.
 *	@param	mbs_size	The size, in bytes, of the buffer pointed to by the
 *						\p mbstr argument. If this value is zero, the function
 *						returns the number of bytes required for the buffer.
 *	@param	ucs2str		The pointer to the UCS-2 string to be converted.
 *	@param	ucs_size	The size, in number of UCS-2 characters, of the UCS-2
 *						string, \p ucs2str.
 *	@retval	size_t		The number of bytes written to \p mbstr buffer if
 *						the conversion is successful. If \p mbs_size is zero,
 *						the return value is the required size, in bytes, for a
 *						buffer to receive the converted string. This function
 *						returns zero if an error occurred.
 */
UCS2API size_t ucs2tombs(char *mbstr, size_t mbs_size, const ucs2char_t *ucs2str, size_t ucs_size);

/**
 * Convert multi-byte characters to a UCS-2 string.
 *
 *	@param	ucs2str		The pointer to the buffer that receives UCS-2 string
 *						converted from the multi-byte characters. If
 *						\p ucs_size is zero, this argument is not be used.
 *	@param	ucs_size	The size, in number of UCS-2 characters, of the buffer
 *						pointed to by the \p ucs2str argument. If this value is
 *						zero, the function returns the number of UCS-2
 *						characters required for the buffer.
 *	@param	mbstr		The pointer to the multi-byte characters to be
 *						converted.
 *	@param	mbs_size	The size, in bytes, of the multi-byte characters,
 *						\p mbstr.
 *	@retval	size_t		The number of UCS-2 characters written to \p ucs2str
 *						buffer if the conversion is successful. If \p ucs_size
 *						is zero, the return value is the required size, in
 *						number of UCS-2 characters, for a buffer to receive the
 *						converted string. This function returns zero if an error
 *						occurred.
 */
UCS2API size_t mbstoucs2(ucs2char_t *ucs2str, size_t ucs_size, const char *mbstr, size_t mbs_size);

/**
 * Convert multi-byte characters in a specific encoding to a UCS-2 string.
 *
 *	@param	ucs2str		The pointer to the buffer that receives UCS-2 string
 *						converted from the multi-byte characters. If
 *						\p ucs_size is zero, this argument is not be used.
 *	@param	ucs_size	The size, in number of UCS-2 characters, of the buffer
 *						pointed to by the \p ucs2str argument. If this value is
 *						zero, the function returns the number of UCS-2
 *						characters required for the buffer.
 *	@param	mbstr		The pointer to the multi-byte characters to be
 *						converted.
 *	@param	mbs_size	The size, in bytes, of the multi-byte characters,
 *						\p mbstr.
 *	@param	charset		The pointer to the string specifying the encoding of
 *						the multi-byte characters.
 *	@retval	size_t		The number of UCS-2 characters written to \p ucs2str
 *						buffer if the conversion is successful. If \p ucs_size
 *						is zero, the return value is the required size, in
 *						number of UCS-2 characters, for a buffer to receive the
 *						converted string. This function returns zero if an error
 *						occurred.
 *	@note
 *		\p charset is ignored on Win32 environments.
 */
UCS2API size_t mbstoucs2_charset(ucs2char_t *ucs2str, size_t ucs_size, const char *mbstr, size_t mbs_size, const char *charset);

/**
 * Convert a UCS-2 string to a UTF-8 string.
 *
 *	@param	mbstr		The pointer to the buffer that receives UTF-8 string
 *						converted from the UCS-2 string. If	\p mbs_size is
 *						zero, this argument is not be used.
 *	@param	mbs_size	The size, in bytes, of the buffer pointed to by the
 *						\p mbstr argument. If this value is zero, the function
 *						returns the number of bytes required for the buffer.
 *	@param	ucs2str		The pointer to the UCS-2 string to be converted.
 *	@param	ucs_size	The size, in number of UCS-2 characters, of the UCS-2
 *						string, \p ucs2str.
 *	@retval	size_t		The number of bytes written to \p mbstr buffer if
 *						the conversion is successful. If \p mbs_size is zero,
 *						the return value is the required size, in bytes, for a
 *						buffer to receive the converted string. This function
 *						returns zero if an error occurred.
 */
UCS2API size_t ucs2toutf8(char *mbstr, size_t mbs_size, const ucs2char_t *ucs2str, size_t ucs_size);

/**
 * Convert a UTF-8 string to a UCS-2 string.
 *
 *	@param	ucs2str		The pointer to the buffer that receives UCS-2 string
 *						converted from the UTF-8 string. If	\p ucs_size is
 *						zero, this argument is not be used.
 *	@param	ucs_size	The size, in number of UCS-2 characters, of the buffer
 *						pointed to by the \p ucs2str argument. If this value is
 *						zero, the function returns the number of UCS-2
 *						characters required for the buffer.
 *	@param	mbstr		The pointer to the UTF-8 string to be converted.
 *	@param	mbs_size	The size, in bytes, of the UTF-8 string, \p mbstr.
 *	@retval	size_t		The number of UCS-2 characters written to \p ucs2str
 *						buffer if the conversion is successful. If \p ucs_size
 *						is zero, the return value is the required size, in
 *						number of UCS-2 characters, for a buffer to receive the
 *						converted string. This function returns zero if an error
 *						occurred.
 */
UCS2API size_t utf8toucs2(ucs2char_t *ucs2str, size_t ucs_size, const char *mbstr, size_t mbs_size);

/**
 * Convert and duplicate a UCS-2 string to multi-byte characters.
 *
 *	@param	ucs2str		The pointer to a UCS-2 string.
 *	@retval	char*		The pointer to the duplicated string. Call ucs2free()
 *						to free the memory block allocated by this function.
 */
UCS2API char *ucs2dupmbs(const ucs2char_t *ucs2str);

/**
 * Convert and duplicate multi-byte characters to a UCS-2 string.
 *
 *	@param	mbstr		The pointer to multi-byte characters.
 *	@retval	char*		The pointer to the duplicated string. Call ucs2free()
 *						to free the memory block allocated by this function.
 */
UCS2API ucs2char_t* mbsdupucs2(const char *mbstr);

/**
 * Convert and duplicate multi-byte characters in a specific encoding
 * to a UCS-2 string.
 *
 *	@param	mbstr		The pointer to multi-byte characters.
 *	@param	charset		The pointer to the string specifying the encoding of
 *						the multi-byte characters.
 *	@retval	char*		The pointer to the duplicated string. Call ucs2free()
 *						to free the memory block allocated by this function.
 */
UCS2API ucs2char_t* mbsdupucs2_charset(const char *mbstr, const char *charset);

/**
 * Convert and duplicate a UCS-2 string to a UTF-8 string.
 *
 *	@param	ucs2str		The pointer to a UCS-2 string.
 *	@retval	char*		The pointer to the duplicated string. Call ucs2free()
 *						to free the memory block allocated by this function.
 */
UCS2API char *ucs2duputf8(const ucs2char_t *ucs2str);

/**
 * Convert and duplicate a UTF-8 string to a UCS-2 string.
 *
 *	@param	utf8str		The pointer to a UTF-8 string.
 *	@retval	char*		The pointer to the duplicated string. Call ucs2free()
 *						to free the memory block allocated by this function.
 */
UCS2API ucs2char_t* utf8dupucs2(const char *utf8str);

/** 
 * @} 
 */



/**
 * @defgroup ucs2_memory Memory manager
 * @{ 
 *
 *	Memory manager provides functions to allocate/expand/free memory blocks.
 *	Use these functions to manage memory blocks allocated by the PMPlib API.
 */

/**
 * Allocate a memory block.
 *
 *	@param	size		The size, in bytes, of the memory block.
 *	@retval	void*		The pointer to the allocated memory block.
 */
UCS2API void *ucs2malloc(size_t size);

/**
 * Allocate a memory block with values initialized as zero.
 *
 *	@param	size		The size, in bytes, of the memory block.
 *	@retval	void*		The pointer to the allocated memory block.
 */
UCS2API void *ucs2calloc(size_t size);

/**
 * Resize a memory block.
 *
 *	@param	ptr			The pointer to the memory block to be resized.
 *	@param	size		The size, in bytes, of the new memory block.
 *	@retval	void*		The pointer to the new memory block.
 */
UCS2API void *ucs2realloc(void *ptr, size_t size);

/**
 * Free a memory block.
 *
 *	@param	ptr			The pointer to the memory block to be free.
 */
UCS2API void ucs2free(void* ptr);

/** 
 * @} 
 */

/**
 * @defgroup ucs2_char UCS-2 character routines
 * @{ 
 *
 *	UCS-2 character routines provide the utility to classify characters and
 *	convert from upper and lower cases.
 */

/**
 * Test whether a USC-2 character is a surrogate-pair character in UTF-16.
 *	@param	c			The UCS-2 character to be tested.
 *	@retval	int			Non-zero value if the test is true, zero otherwise.
 */
UCS2API int ucs2issurrogate(ucs2char_t c);

/**
 * Test whether a USC-2 character is a whitespace character.
 *	@param	c			The UCS-2 character to be tested.
 *	@retval	int			Non-zero value if the test is true, zero otherwise.
 */
UCS2API int ucs2isspace(ucs2char_t c);

/**
 * Test whether a USC-2 character is a numeric character.
 *	@param	c			The UCS-2 character to be tested.
 *	@retval	int			Non-zero value if the test is true, zero otherwise.
 */
UCS2API int ucs2isdigit(ucs2char_t c);

/**
 * Convert a UCS-2 character to lower case.
 *	@param	c			The UCS-2 character to be coverted.
 *	@retval	ucs2char_t	The resultant UCS-2 character.
 */
UCS2API ucs2char_t ucs2lower(ucs2char_t c);

/**
 * Convert a UCS-2 character to upper case.
 *	@param	c			The UCS-2 character to be coverted.
 *	@retval	ucs2char_t	The resultant UCS-2 character.
 */
UCS2API ucs2char_t ucs2upper(ucs2char_t c);

/** 
 * @} 
 */





/**
 * @defgroup ucs2_string_ansi ANSI C compatible string routines
 * @{ 
 *
 *	ANSI C compatible string routines correspond to string manupulation
 *	routines in the ANSI C standard. The following ANSI C functions are not
 *	defined in this API subset:
 *		strcoll, strerror, strtok, strxfrm.
 */

/**
 * Concatenate two strings.
 *
 *	@param	dst			The pointer to the destination of a string.
 *	@param	src			The pointer to the source of a string.
 *	@retval	ucs2char_t*	The pointer to the destination string.
 */
UCS2API ucs2char_t* ucs2cat(ucs2char_t* dst, const ucs2char_t* src);

/**
 * Search for the first occurrence of a character in a string.
 *
 *	@param	str			The pointer to the string in which \p c is searched.
 *	@param	c			The target character.
 *	@retval	ucs2char_t*	The pointer to the character \p c in \p str, or \c NULL
 *						if \p c does not occur in \p str.
 */
UCS2API ucs2char_t* ucs2chr(const ucs2char_t* str, ucs2char_t c);

/**
 * Compare two strings.
 *
 *	@param	x			The pointer to a string.
 *	@param	y			The pointer to another string.
 *	@retval	int			A positive value if \p x is greater than \p y;
 *						a negative value if \p x is smaller than \p y;
 *						zero if \p x is identical to \p y.
 */
UCS2API int ucs2cmp(const ucs2char_t* x, const ucs2char_t* y);

/**
 * Copy a string.
 *
 *	@param	dst			The pointer to the destination of a string.
 *	@param	src			The pointer to the source of a string.
 *	@retval	ucs2char_t*	The pointer to the destination string.
 */
UCS2API ucs2char_t* ucs2cpy(ucs2char_t* dst, const ucs2char_t* src);

/**
 * Count the number of characters not appearing in a character set.
 *
 *	@param	str			The pointer to a string.
 *	@param	charset		The pointer to a character set.
 *	@retval	size_t		The number of characters from the beginning of \p str
 *						in which any character in \p charset appear for the
 *						first time, or the length of \p str if such a character
 *						does not exist.
 */
UCS2API size_t ucs2cspn(const ucs2char_t *str, const ucs2char_t *charset);

/**
 * Count the length of a string.
 *
 *	@param	str			The pointer to a string.
 *	@retval	size_t		The number of characters in \p str.
 */
UCS2API size_t ucs2len(const ucs2char_t* str);

/**
 * Concatenate two strings (no more than the maximum length).
 *
 *	@param	dst			The pointer to the destination of a string.
 *	@param	src			The pointer to the source of a string.
 *	@param	n			The number of characters to be concatenated.
 *	@retval	ucs2char_t*	The pointer to the destination string.
 */
UCS2API ucs2char_t* ucs2ncat(ucs2char_t *dst, const ucs2char_t *src, size_t n);

/**
 * Compare two strings (no longer than the maximum length).
 *
 *	@param	x			The pointer to a string.
 *	@param	y			The pointer to another string.
 *	@param	n			The number of characters to be compared.
 *	@retval	int			A positive value if \p x is greater than \p y;
 *						a negative value if \p x is smaller than \p y;
 *						zero if \p x is identical to \p y.
 */
UCS2API int ucs2ncmp(const ucs2char_t* x, const ucs2char_t* y, size_t n);

/**
 * Copy a string (no more than the maximum length).
 *
 *	@param	dst			The pointer to the destination of a string.
 *	@param	src			The pointer to the source of a string.
 *	@param	n			The number of characters to be copied.
 *	@retval	ucs2char_t*	The pointer to the destination string.
 */
UCS2API ucs2char_t* ucs2ncpy(ucs2char_t* dst, const ucs2char_t* src, size_t n);

/**
 * Find a character in a string that belongs to a character set.
 *
 *	@param	str			The pointer to the string where \p charset is searched.
 *	@param	charset		The pointer to a character set.
 *	@retval	ucs2char_t*	The pointer to the character in \p str that belongs
 *						to \p charset, or \c NULL if such a character does not
 *						exist.
 */
UCS2API ucs2char_t* ucs2pbrk(const ucs2char_t *str, const ucs2char_t *charset);

/**
 * Search for the last occurrence of a character in a string.
 *
 *	@param	str			The pointer to the string in which \p c is searched.
 *	@param	c			The target character.
 *	@retval	ucs2char_t*	The pointer to the character \p c in \p str, or \c NULL
 *						if \p c does not occur in \p str.
 */
UCS2API ucs2char_t* ucs2rchr(const ucs2char_t* str, ucs2char_t c);

/**
 * Find a character in a string that does not belong to a character set.
 *
 *	@param	str			The pointer to the string where \p charset is searched.
 *	@param	charset		The pointer to a character set.
 *	@retval	ucs2char_t*	The pointer to the character in \p str that does not
 *						belong to \p charset, or \c NULL if such a character
 *						does not exist.
 */
UCS2API size_t ucs2spn(const ucs2char_t *str, const ucs2char_t *charset);

/**
 * Find a substring in a string.
 *
 *	@param	str			The pointer to the string where \p substr is searched.
 *	@param	substr		The pointer to the substring.
 *	@retval	ucs2char_t*	The pointer to the character where \p substr begins
 *						in \p str for the first time, or \c NULL if \p str
 *						does not contain \p substr.
 */
UCS2API ucs2char_t* ucs2str(const ucs2char_t* str, const ucs2char_t* substr);

/** 
 * @}
 */





/**
 * @defgroup ucs2_string_non_ansi Miscellaneous string routines
 * @{ 
 *
 *	Non ANSI C compatible string routines.
 */

/**
 * Duplicate a string.
 *
 *	@param	str			The pointer to a string.
 *	@retval	ucs2char_t*	The pointer to the duplicated string. Call ucs2free()
 *						to free the memory block allocated by this function.
 */
UCS2API ucs2char_t* ucs2dup(const ucs2char_t* str);

/**
 * Duplicate a string no longer than the specified length.
 *
 *	@param	str			The pointer to a string.
 *	@param	length		The maximum length of the duplicated string.
 *	@retval	ucs2char_t*	The pointer to the duplicated string. Call ucs2free()
 *						to free the memory block allocated by this function.
 */
UCS2API ucs2char_t* ucs2ndup(const ucs2char_t* str, size_t length);

/**
 * Convert a string to lowercase.
 *
 *	Given \p str argument, this function converts uppercase letters in the
 *	string to lowercase and overwrites the resultant string on the same buffer
 *	pointed by the \p str argument.
 *
 *	@param	str			The pointer to a string. 
 *	@retval	ucs2char_t*	The pointer to the string, which is the same value as
 *						\p str.
 */
UCS2API ucs2char_t* ucs2lwr(ucs2char_t* str);

/**
 * Convert a string to uppercase.
 *
 *	Given \p str argument, this function converts lowercase letters in the
 *	string to uppercase and overwrites the resultant string on the same buffer
 *	pointed by the \p str argument.
 *
 *	@param	str			The pointer to a string. 
 *	@retval	ucs2char_t*	The pointer to the string, which is the same value as
 *						\p str.
 */
UCS2API ucs2char_t* ucs2upr(ucs2char_t* str);

/**
 * Compare two strings incasesensitively.
 *
 *	@param	x			The pointer to a string.
 *	@param	y			The pointer to another string.
 *	@retval	int			A positive value if \p x is greater than \p y;
 *						a negative value if \p x is smaller than \p y;
 *						zero if \p x is identical to \p y.
 */
UCS2API int ucs2icmp(const ucs2char_t* x, const ucs2char_t* y);

/**
 * Compare two strings incasesensitively (no longer than the maximum length).
 *
 *	@param	x			The pointer to a string.
 *	@param	y			The pointer to another string.
 *	@param	n			The number of characters to be compared.
 *	@retval	int			A positive value if \p x is greater than \p y;
 *						a negative value if \p x is smaller than \p y;
 *						zero if \p x is identical to \p y.
 */
UCS2API int ucs2incmp(const ucs2char_t* x, const ucs2char_t* y, size_t n);

/**
 * Strip whitespace characters at the head and tail of a string.
 *
 *	Given \p str argument, this function trims whilespace characters at the
 *	head and tail of the string and overwrites the resultant string on the
 *	same buffer pointed by the \p str argument.
 *
 *	@param	str			The pointer to a string.
 *	@retval	ucs2char_t*	The pointer to the string, which is the same value as
 *						\p str.
 */
UCS2API ucs2char_t* ucs2strip(ucs2char_t* str);

/** 
 * @} 
 */



/**
 * @defgroup ucs2_std ANSI C compatible stdio/stdlib routines
 * @{ 
 *
 *	ANSI C compatible stdio/stdlib routines provide utilities for streams and
 *	string-integer conversions.
 */

/**
 * Convert a UCS-2 string to integer value.
 *	@param	str			The pointer to a string.
 *	@retval	int			The corresponding integer value.
 */
UCS2API int ucs2toi(const ucs2char_t* str);

/**
 * Convert an integer value to UCS-2 string.
 *	@param	value		The integer value.
 *	@param	string		The pointer to the buffer to receive the string.
 *	@param	radix		Radix of the \p value.
 *	@retval	ucs2char_t*	The pointer to the string, which is the same value as
 *						\p str.
 */
UCS2API ucs2char_t* itoucs2(int value, ucs2char_t *string, int radix);

/**
 * Open a stream from a file.
 *	@param	filename	The pointer to the UCS-2 string for the file name.
 *	@param	mode		The pointer to the C string for the open mode.
 *	@retval	FILE*		The pointer to the opened stream if successful,
 *						NULL otherwise.
 */
UCS2API FILE *ucs2fopen(const ucs2char_t *filename, const char *mode);

/**
 * Write a UCS-2 character to a stream.
 *	@param	c			The character to be written.
 *	@param	fp			The pointer to the output stream.
 *	@retval	int			Zero if successful, non-zero otherwise.
 */
UCS2API int fputucs2c(ucs2char_t c, FILE *fp);



/** @} */

/** @} */

#ifdef	__cplusplus
}
#endif/*__cplusplus*/

#endif/*__UCS2CHAR_H__*/
