/*
 *      File/path utility implemented with POSIX API.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *      Copyright (c) 2006-2007 Martin Ellis <martin.ellis@kdemail.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: filepath_posix.c 331 2007-02-10 18:39:35Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>

#include <dirent.h>
#include <limits.h>
#include <string.h>
#include <sys/stat.h>

#include <pmplib/filepath.h>
#include "rel2abs.h"

static int _decode(char* path)
{
	while (*path) {
		if (*path == 0x5C) {
			*path = 0x2F;
		}
		path++;
	}
	return 0;
}

int find_file(const ucs2char_t* path, int recursive, filepath_findfile_callback callback, void *instance)
{
	DIR *dirp = NULL;
	struct dirent *entp = NULL;
	ucs2char_t	tmp[MAX_PATH];
	char mbs[MAX_PATH+1];

	ucs2cpy(tmp, path);
	filepath_removeslash(tmp);
	filepath_decode(tmp);
	ucs2tombs(mbs, MAX_PATH, tmp, ucs2len(tmp)+1);

	//printf("opendir: %s\n", mbs);
	dirp = opendir(mbs);
	if (dirp) {
		while (entp = readdir(dirp)) {
			ucs2char_t filename[MAX_PATH+1];
			if (strcmp(entp->d_name, ".") == 0 || strcmp(entp->d_name, "..") == 0) {
				continue;
			}
			//printf("readdir: %s\n", entp->d_name);
			mbstoucs2(filename, MAX_PATH, entp->d_name, strlen(entp->d_name)+1);

			ucs2cpy(tmp, path);
			ucs2cat(tmp, filename);
			if (!filepath_is_dir(tmp)){
				callback(instance, path, filename);
			}
		}
		closedir(dirp);
	}

	if (recursive) {
		ucs2cpy(tmp, path);
		filepath_removeslash(tmp);
		filepath_decode(tmp);
		ucs2tombs(mbs, MAX_PATH, tmp, ucs2len(tmp)+1);

		//printf("opendir2: %s\n", mbs);
		dirp = opendir(mbs);
		if (dirp) {
			while (entp = readdir(dirp)) {
				struct stat st;
				ucs2char_t filename[MAX_PATH+1];
				mbstoucs2(filename, MAX_PATH, entp->d_name, strlen(entp->d_name)+1);

				//printf("readdir2: %s\n", entp->d_name);
				if (strcmp(entp->d_name, ".") == 0 || strcmp(entp->d_name, "..") == 0) {
					continue;
				}

				ucs2cpy(tmp, path);
				ucs2cat(tmp, filename);

				if (filepath_is_dir(tmp)){
					filepath_addslash(tmp);
					find_file(tmp, recursive, callback, instance);
				}
			}
			closedir(dirp);
		}
	}

	return 0;
}

static ucs2char_t* search_extension(ucs2char_t* path)
{
	ucs2char_t* p = ucs2rchr(path, PATHCHAR);
	ucs2char_t* q = ucs2rchr(path, '.');
	if (q) {
		if (p < q) {
			return q;
		}
	}
	return NULL;
}

ucs2char_t* filepath_addslash(ucs2char_t* path)
{
	size_t length = ucs2len(path);
	if (length > 0) {
		if (path[length-1] != PATHCHAR) {
			/* Append PATHCHAR to path. */
			path[length++] = PATHCHAR;
			path[length] = 0;
		}
	}
	return (path + length);
}

ucs2char_t* filepath_removeslash(ucs2char_t* path)
{
	size_t length = ucs2len(path)-1;
	while (length >= 0 && path[length] == PATHCHAR) {
		path[length] = 0;
		length--;
	}
	return (path + length);
}

int filepath_hasext(const ucs2char_t* filename, const ucs2char_t* ext)
{
	const ucs2char_t* p = search_extension((ucs2char_t*)filename);
	if (!p) {
		return 0;
	}
	while (*p && *ext) {
		if (ucs2lower(*p) != ucs2lower(*ext)) {
			return 0;
		}
		p++;
		ext++;
	}
	/* ME: This seems redundant... won't "return (*p == *q);" do?
	       We only need to know whether we've reached the end of both
	       strings...
	 */
	return (ucs2lower(*p) == ucs2lower(*ext));
}


const ucs2char_t* filepath_combinepath(ucs2char_t* dst, size_t size, const ucs2char_t* path, const ucs2char_t* file)
{
	size_t length = ucs2len(path);
	ucs2cpy(dst, path);

	// Ensure the first part ends in the path separator
	filepath_addslash(dst);

	// Then strip copy the rest of the path, except any initial
	// path separator.
	if (*file == PATHCHAR) {
	  ucs2cat(dst, &(file[1]));
	} else {
	  ucs2cat(dst, file);
	}

	return dst;
}

const ucs2char_t* filepath_skiproot(const ucs2char_t* path, const ucs2char_t* root)
{
	if (ucs2ncmp(path, root, ucs2len(root)) == 0) {
		return path + ucs2len(root);
	} else {
		return path;
	}
}

const ucs2char_t* filepath_skippath(const ucs2char_t* path)
{
	ucs2char_t* p = ucs2rchr(path, PATHCHAR);
	return p ? p+1 : path;
}

const ucs2char_t* filepath_changeroot(const ucs2char_t* path, const ucs2char_t* root)
{
	const ucs2char_t* p = filepath_skiproot(path, root);
	if (p == NULL) p = path;
	if (path < p && p[-1] == PATHCHAR) {
		--p;
	}
	return p;
}

const ucs2char_t* filepath_skipadirectory(const ucs2char_t* path)
{
	ucs2char_t* p = ucs2chr(path, PATHCHAR);
	return p ? p+1 : NULL;
}

void filepath_strippath(ucs2char_t* path)
{
	ucs2char_t* p = path + ucs2len(path) - 1;
	if (path <= p) {
		if (*p == PATHCHAR) {
			p--;
		}
		while (path <= p) {
			if (*p == PATHCHAR) {
				p++;
				break;
			}
			p--;
		}
		if (path < p) {
			for (;;) {
				*path = *p;
				if (!*p) {
					break;
				}
				path++;
				p++;
			}
		}
	}
}

void filepath_add_extension(ucs2char_t* path, const ucs2char_t* ext)
{
	ucs2char_t* p = search_extension(path);
	if (!p) {
		ucs2cat(path, ext);
	}
}

void filepath_remove_extension(ucs2char_t* path)
{
	ucs2char_t* p = search_extension(path);
	if (p) {
		*p = 0;
	}
}

void filepath_remove_filespec(ucs2char_t* path)
{
	ucs2char_t* p = ucs2rchr(path, PATHCHAR);
	if (p) {
		*p = 0;
	}
}

int filepath_is_relative(const ucs2char_t* path)
{
	return (*path != PATHCHAR);
}

int filepath_file_exists(const ucs2char_t *filename)
{
	int ret = 0;
	struct stat st;
	char *mbs_path = ucs2dupmbs(filename);

	if (mbs_path) {
		_decode(mbs_path);
		//fprintf(stderr, "is_exist: %s\n", mbs_path);
		ret = open(mbs_path, 0);
		free(mbs_path);
		if (ret != -1) {
			close(ret);
		}
		return (ret != -1);
	}
	return 0;
}

int filepath_relative_to_absolute(ucs2char_t* absolute, const ucs2char_t* base, const ucs2char_t* relative)
{
	return (rel2abs(relative, base, absolute, MAX_PATH) != NULL);
}

int filepath_is_same_root(const ucs2char_t* path, const ucs2char_t* root)
{
	return (ucs2ncmp(path, root, ucs2len(root)) == 0);
}

int filepath_compare_lastupdate(const ucs2char_t* file1, const ucs2char_t* file2)
{
	int ret = 0;
	struct stat st1;
	struct stat st2;

	time_t ret1 = filepath_mtime(file1);
	time_t ret2 = filepath_mtime(file2);
	return (ret1 - ret2);
}

int filepath_copyfile(const ucs2char_t* src, const ucs2char_t* dst)
{
	int ret = 0;
	FILE *fpi = ucs2fopen(src, "rb");
	FILE *fpo = ucs2fopen(dst, "wb");

	if (!fpi || !fpo) {
		return -1;
	}

	for (;;) {
		char buffer[4096];
		int bytes_read = fread(buffer, sizeof(char), 4096, fpi);
		int bytes_write = fwrite(buffer, sizeof(char), bytes_read, fpo);
		if (bytes_write < bytes_read) {
			ret = -1;
			break;
		}
		if (feof(fpi)) {
			ret = 0;
			break;
		}
	}

	fclose(fpo);
	fclose(fpi);
	return ret;
}

int filepath_removefile(const ucs2char_t* file)
{
	int ret = 0;
	char *fn = ucs2dupmbs(file);
	ret = (remove(fn) == 0);
	ucs2free(fn);
	return ret;
}

int filepath_encode(ucs2char_t* path)
{
	while (*path) {
		if (*path == 0x002F) {
			*path = 0x005C;
		}
		path++;
	}
	return 0;
}

int filepath_decode(ucs2char_t* path)
{
	while (*path) {
		if (*path == 0x005C) {
			*path = 0x002F;
		}
		path++;
	}
	return 0;
}

time_t filepath_mtime(const ucs2char_t *filename)
{
	int ret = 0;
	struct stat st;
	char *mbs_path = ucs2dupmbs(filename);

	if (mbs_path) {
		_decode(mbs_path);
		ret = stat(mbs_path, &st);
		free(mbs_path);
		if (ret == 0) {
			return st.st_mtime;
		}
	}
	return 0;
}

uint32_t filepath_size(const ucs2char_t *filename)
{
	int ret = 0;
	struct stat st;
	char *mbs_path = ucs2dupmbs(filename);

	if (mbs_path) {
		_decode(mbs_path);
		ret = stat(mbs_path, &st);
		free(mbs_path);
		if (ret == 0) {
			return (uint32_t)st.st_size;
		}
	}
	return 0;
}

int filepath_is_dir(const ucs2char_t *filename)
{
	int ret = 0;
	struct stat st;
	char *mbs_path = ucs2dupmbs(filename);

	if (mbs_path) {
		_decode(mbs_path);
		ret = stat(mbs_path, &st);
		free(mbs_path);
		if (ret == 0) {
			return ((st.st_mode & S_IFMT) == S_IFDIR);
		}
	}
	return 0;
}
