/*
 *      JavaScript Playlist (JSPL) routine.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: jspl.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>
#include <playlist.h>

#include <jsapi.h>

#define	MAX_SOURCE_DEPTH	64

struct tag_jspl {
	JSRuntime	*runtime;
	JSContext	*context;
	JSObject	*global;

	const ucs2char_t *path_to_include;
	const ucs2char_t path_to_source[MAX_PATH][MAX_SOURCE_DEPTH];
	int			source_depth;

	int			num_records;
	JSObject	**records;
	JSObject	*tracks;

	void		*instance;
	playlist_callback_t	callback;
};
typedef struct tag_jspl jspl_t;

static int jspl_load_script(jspl_t* jspl, const ucs2char_t* filename);


static jschar* JS_ucstrdup(JSContext *cx, const ucs2char_t* src, size_t* ptr_length)
{
	size_t i, length = ucs2len(src);
	jschar* dst = JS_malloc(cx, length * sizeof(jschar));

	if (dst) {
		for (i = 0;i < length;++i) {
			dst[i] = src[i];
		}
		if (ptr_length) {
			*ptr_length = length;
		}
	}
	return dst;
}

static ucs2char_t* JSString_to_ucs2(JSString* src)
{
	size_t i, length = JS_GetStringLength(src);
	jschar* jssrc = JS_GetStringChars(src);
	ucs2char_t* dst = ucs2malloc((length+1) * sizeof(ucs2char_t));
	for (i = 0;i < length;++i) {
		dst[i] = jssrc[i];
	}
	dst[i] = 0;
	return dst;
}

static void error_handler(JSContext *cx, const char *message, JSErrorReport *report)
{
	ucs2char_t* ucs2 = NULL;
	jspl_t* jspl = (jspl_t*)JS_GetContextPrivate(cx);
	const char *msg = message ? message : "Unknown error";
	const char *filename = report->filename ? report->filename : "(Unknown filename)";
	char *pos = alloca(strlen(filename) + 64);	/* extra 64 bytes for line numbers. */
	
	ucs2 = mbsdupucs2(msg);
	jspl->callback(jspl->instance, PLCALLBACK_JSPL_ERROR, ucs2);
	ucs2free(ucs2);

	sprintf(pos, "lines %d in %s", report->lineno, filename);
	ucs2 = mbsdupucs2(pos);
	jspl->callback(jspl->instance, PLCALLBACK_JSPL_ERROR_POS, ucs2);
	ucs2free(ucs2);

	if (report->uclinebuf) {
		jspl->callback(jspl->instance, PLCALLBACK_JSPL_ERROR_LINE, (ucs2char_t*)report->uclinebuf);
	} else if (report->linebuf) {
		ucs2 = mbsdupucs2(report->linebuf);
		jspl->callback(jspl->instance, PLCALLBACK_JSPL_ERROR_LINE, ucs2);
		ucs2free(ucs2);
	}
}

static int function_exists(jspl_t* jspl, const char *funcname)
{
	jsval rval;
	char script[1024];

	sprintf(script, "(%s ? true : false);\n", funcname);

	if (JS_EvaluateScript(jspl->context, jspl->global, script, strlen(script), "easypmp", 0, &rval)) {
		return 1;
	}
	return 0;
}

static int object_is_array(JSContext* cx, JSObject* obj)
{
	JSClass *jsclass = NULL;

#ifdef	JS_THREADSAFE
	jsclass = JS_GetClass(cx, obj);
#else
	jsclass = JS_GetClass(obj);
#endif

	if (strcmp(jsclass->name, "Array") == 0) {
		return 1;
	} else {
		jsint length = 0;
		if (JS_GetArrayLength(cx, obj, &length)) {
			if (length > 0) {
				return 1;
			}
		}
	}
	return 0;
}

static JSBool
js_print(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
    uintN i, n;
	size_t length = 1;
	ucs2char_t* message = ucs2calloc(1 * sizeof(ucs2char_t));
	jspl_t* jspl = (jspl_t*)JS_GetContextPrivate(cx);
	static ucs2char_t space[] = {' ',0};

    for (i = 0, n = 0; i < argc; i++) {
		size_t length_to_add = 0;
		ucs2char_t* ucs2 = NULL;
	    JSString *str = JS_ValueToString(cx, argv[i]);
		if (!str) {
            return JS_FALSE;
		}

		ucs2 = JSString_to_ucs2(str);
		length_to_add = ucs2len(ucs2) + 1;
		message = ucs2realloc(message, (length + length_to_add) * sizeof(ucs2char_t));
		if (i) {
			ucs2cat(message, space);
		}
		ucs2cat(message, ucs2);
		length += length_to_add;
    }
	jspl->callback(jspl->instance, PLCALLBACK_JSPL_MESSAGE, message);
	ucs2free(message);
    return JS_TRUE;
}

static JSBool
js_include(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
    uintN i, n;
	jspl_t* jspl = (jspl_t*)JS_GetContextPrivate(cx);

    for (i = 0, n = 0; i < argc; i++) {
		ucs2char_t* ucs2 = NULL;
		ucs2char_t filename[MAX_PATH+1];
	    JSString *str = JS_ValueToString(cx, argv[i]);
		if (!str) {
            return JS_FALSE;
		}

		ucs2 = JSString_to_ucs2(str);

		filepath_combinepath(filename, MAX_PATH, jspl->path_to_include, ucs2);
		jspl_load_script(jspl, filename);
    }

    return JS_TRUE;
}

static JSFunctionSpec g_global_functions[] = {
    {"print",           js_print,			0},
	{"include",			js_include,			0},
    {0}
};

static int set_property_string(jspl_t* jspl, JSObject* obj, const char *name, const ucs2char_t* value)
{
	jsval val;

	/* Set JavaScript value. */
	if (value && *value) {
		jschar* jsstr = NULL;
		JSString *str = NULL;
		size_t length = 0;

		jsstr = JS_ucstrdup(jspl->context, value, &length);
		if (!jsstr) {
			return 1;
		}
		str = JS_NewUCString(jspl->context, jsstr, length);
		if (!str) {
			return 1;
		}
		val = STRING_TO_JSVAL(str);
	} else {
		val = JSVAL_NULL;
	}

	if (!JS_SetProperty(jspl->context, obj, name, &val)) {
		return 1;
	}

	return 0;
}

static int set_property_int(jspl_t* jspl, JSObject* obj, const char *name, int value)
{
	jsval val = INT_TO_JSVAL(value);
	if (!JS_SetProperty(jspl->context, obj, name, &val)) {
		return 1;
	}
	return 0;
}

static int jspl_init(jspl_t** ptr_jspl)
{
	jspl_t* jspl = calloc(1, sizeof(jspl_t));

	/* Initialize a JavaScript run time. */
	jspl->runtime = JS_NewRuntime(8L * 1024L * 1024L);
	if (!jspl->runtime) {
		return -1;
	}
	
	/* Establish a context */
	jspl->context = JS_NewContext(jspl->runtime, 8192);
	if (!jspl->context) {
		return -1;
	}

	/* Create a global object. */
	jspl->global = JS_NewObject(jspl->context, NULL, NULL, NULL);
	if (!jspl->context) {
		return -1;
	}

	/* Initialize built-in and global JavaScript objects. */
	if (!JS_InitStandardClasses(jspl->context, jspl->global)) {
		return -1;
	}

	/* Define global functions exported by this program. */
	if (!JS_DefineFunctions(jspl->context, jspl->global, g_global_functions)) {
		return -1;
	}

	/* Register an error handler. */
	JS_SetErrorReporter(jspl->context, error_handler);
	JS_SetContextPrivate(jspl->context, jspl);

	*ptr_jspl = jspl;
	return 0;
}

static int jspl_finish(jspl_t* jspl)
{
	if (jspl) {
		if (jspl->context) {
			JS_DestroyContext(jspl->context);
		}
		if (jspl->runtime) {
			JS_DestroyRuntime(jspl->runtime);
		}
		JS_ShutDown();
		free(jspl);
	}
	return 0;
}

static int jspl_set(jspl_t* jspl, const pmp_music_record_t* records, int num_records)
{
	int i;

	/* Allocate a JavaScript array object.
	 *	According to the notes for JS_NewArrayObject() function in the
	 *	reference document, we should create an empty array, store the
	 *	returned object in a GC root, populate its element, and then
	 *	drop the root object to avoid unrooted jsvals in vector from
	 *	being subject to garbage collection until the new object has
	 *	been populated
	 */
	jspl->tracks = JS_NewArrayObject(jspl->context, 0, NULL);

	/* */
	if (!JS_AddRoot(jspl->context, &jspl->tracks)) {
		return 1;
	}

	/* Allocate a records array. */
	free(jspl->records);
	jspl->records = (JSObject**)calloc(num_records, sizeof(JSObject*));

	for (i = 0;i < num_records;++i) {
		const pmp_music_record_t* record = &records[i];
		JSObject* obj = NULL;
		jsval jsval_obj;

		/* Create a JavaScript object. */
		obj = JS_NewObject(
			jspl->context,
			NULL,
			NULL,
			NULL
			);
		if (!obj) {
			return 1;
		}

		/* Attach properties to the object. */
		set_property_int(jspl, obj, "id", i);
		set_property_string(jspl, obj, "filename", record->filename);
		set_property_string(jspl, obj, "title", record->title);
		set_property_string(jspl, obj, "artist", record->artist);
		set_property_string(jspl, obj, "composer", record->composer);
		set_property_string(jspl, obj, "album", record->album);
		set_property_string(jspl, obj, "genre", record->genre);
		set_property_string(jspl, obj, "date", record->date);
		set_property_int(jspl, obj, "codec", (int)record->codec);
		set_property_int(jspl, obj, "track_number", (int)record->track_number);
		set_property_int(jspl, obj, "sample_rate", (int)record->sample_rate);
		set_property_int(jspl, obj, "bitrate", (int)record->bitrate);
		set_property_int(jspl, obj, "duration", (int)record->duration);
		set_property_int(jspl, obj, "update_timestamp", (int)record->ts_update);
		set_property_int(jspl, obj, "rating", (int)record->rating);
		set_property_int(jspl, obj, "play_count", (int)record->play_count);
		set_property_int(jspl, obj, "playback_timestamp", (int)record->ts_playback);
		set_property_int(jspl, obj, "import_timestamp", (int)record->ts_import);

		/* Register the object pointer to the record array. */
		jspl->records[i] = obj;

		jsval_obj = OBJECT_TO_JSVAL(obj);
		if (!JS_SetElement(jspl->context, jspl->tracks, i, &jsval_obj)) {
			return 1;
		}
	}

	jspl->num_records = num_records;
	if (!JS_RemoveRoot(jspl->context, &jspl->tracks)) {
		return 1;
	}

	return 0;
}

static int jspl_init_register_codec(jspl_t* jspl, const char *name, uint32_t codec, int *lines)
{
	int ret = 0;
	jsval retval;
	char buffer[1024];

	sprintf(buffer, "Codec.%s = %u;\n", name, codec);
	ret = JS_EvaluateScript(jspl->context, jspl->global, buffer, strlen(buffer), "easypmp.js", *lines, &retval);
	if (!ret) {
		return -1;
	}
	++(*lines);
	return 0;
}

static int jspl_init_codec_class(jspl_t* jspl)
{
	jsval retval;
	int lines = 1;
	static const char *script = "Codec = new Object();\n";

	int ret = JS_EvaluateScript(jspl->context, jspl->global, script, strlen(script), "easypmp.js", lines, &retval);
	if (!ret) {
		return -1;
	}
	++lines;

	/* Register codec IDs. */
	if (jspl_init_register_codec(jspl, "MP3", PMPCODEC_MPEGLAYER3, &lines) < 0) {
		return -1;
	}
	if (jspl_init_register_codec(jspl, "WMA", PMPCODEC_WMA, &lines) < 0) {
		return -1;
	}
	if (jspl_init_register_codec(jspl, "OggVorbis", PMPCODEC_VORBIS, &lines) < 0) {
		return -1;
	}
	if (jspl_init_register_codec(jspl, "WAV", PMPCODEC_WAV, &lines) < 0) {
		return -1;
	}

	return 0;
}

static int jspl_load_script(jspl_t* jspl, const ucs2char_t* filename)
{
#if 0
	FILE *fp = NULL;
	int is_utf8 = 0;
	char *buff = NULL;
	char *p = NULL;
	char *mbsfilename = NULL;
	ucs2char_t* script = NULL;
	uint32_t filesize = 0;
	JSBool ret;
	jsval retval;
	
	/* Open the JavaScript file. */
	fp = ucs2fopen(filename, "r");
	if (!fp) {
		return -1;
	}

	mbsfilename = ucs2dupmbs(filename);

	/* Obtain the filesize. */
	filesize = filepath_size(filename);

	/* Read the entire file at one time. */
	buff = (char*)malloc(filesize+1);
	if (!buff) {
		fclose(fp);
		return -1;
	}
	fread(buff, 1, filesize, fp);
	buff[filesize] = 0;
	fclose(fp);

	p = buff;

	/* Check if this script is written in UTF-8 or noe. */
	if (filesize > 3) {
		unsigned char *q = (unsigned char *)p;
		if (q[0] == 0xEF && q[1] == 0xBB && q[2] == 0xBF) {
			p += 3;
			is_utf8 = 1;
		}
	}

	/* Convert the script into UNICODE. */
	script = is_utf8 ? utf8dupucs2(p) : mbsdupucs2(p);
	if (!script) {
		return -1;
	}

	/* Evaluate the script. */
	ret = JS_EvaluateUCScript(jspl->context, jspl->global, script, ucs2len(script), mbsfilename, 1, &retval);
	if (!ret) {
		return -1;
	}

	ucs2free(script);
	free(buff);
	return 0;

#else
	char *mbsfilename = ucs2dupmbs(filename);
	JSScript* script = NULL;
	JSBool ret;
	jsval retval;

	script = JS_CompileFile(jspl->context, jspl->global, mbsfilename);
	if (!script) {
		return -1;
	}
	ret = JS_ExecuteScript(jspl->context, jspl->global, script, &retval);
	if (!ret) {
		return -1;
	}

	return 0;

#endif
}

static int generate_playlist(
	jspl_t* jspl,
	playlists_t* pls,
	JSObject* jsobj_array,
	const ucs2char_t* name
	)
{
	int ipl = playlist_add_playlist(pls, name);
	if (ipl >= 0) {
		playlist_t* pl = &pls->playlists[ipl];
		jsuint i, length;
		
		if (JS_HasArrayLength(jspl->context, jsobj_array, &length)) {
			for (i = 0;i < length;++i) {
				jsval jsval_element;
				if (JS_GetElement(jspl->context, jsobj_array, (jsint)i, &jsval_element)) {
					JSObject* jsobj_element = JSVAL_TO_OBJECT(jsval_element);
					jsval jsval_filename;
					if (JS_GetProperty(jspl->context, jsobj_element, "filename", &jsval_filename)) {
						ucs2char_t* filename = JSString_to_ucs2(JSVAL_TO_STRING(jsval_filename));
						playlist_append(pl, filename, i);				
					} else {
						return PLAYLIST_E_JSINVALIDTRACK;
					}
				} else {
					return PLAYLIST_E_JSINVALIDARRAY;
				}
			}
		} else {
			return PLAYLIST_E_JSINVALIDARRAY;
		}
	} else {
		return PLAYLIST_E_OUTOFMEMORY;
	}

	return 0;
}


int playlist_jspl_read(
	playlists_t* pls,
	const ucs2char_t *filename,
	const ucs2char_t *path_to_include,
	pmp_music_record_t* records,
	int num_records,
	playlist_callback_t callback,
	void *instance
	)
{
	int ret = 0;
	jspl_t* jspl = NULL;
	ucs2char_t name[MAX_PATH];
	JSString* jstr_name = NULL;
	size_t length = 0;
	jschar* js_name = NULL;
	jsval argv[2], retval;

	/* Intialize JavaScript engine. */
	if (jspl_init(&jspl) != 0) {
		ret = PLAYLIST_E_JSINITENGINE;
		goto error_exit;
	}

	/* Initialize Codec class. */
	if (jspl_init_codec_class(jspl) != 0) {
		ret = PLAYLIST_E_JSINITENGINE;
		goto error_exit;
	}

	/* Initialize other fields. */
	jspl->path_to_include = path_to_include;
	jspl->callback = callback;
	jspl->instance = instance;

	/* Create JavaScript objects represent media information of the records. */
	if (jspl_set(jspl, records, num_records) != 0) {
		ret = PLAYLIST_E_JSINITMEDIA;
		goto error_exit;
	}

	/* Load the script. */
	if (jspl_load_script(jspl, filename) != 0) {
		ret = PLAYLIST_E_JSEVALSCRIPT;
		goto error_exit;
	}

	/* Generate a playlist name. */
	ucs2cpy(name, filepath_skippath(filename));
	filepath_remove_extension(name);

	/* Convert the type of the playlist name to JavaScript string. */
	js_name = JS_ucstrdup(jspl->context, name, &length);
	if (!js_name) {
		ret = PLAYLIST_E_OUTOFMEMORY;
		goto error_exit;
	}
 	jstr_name = JS_NewUCString(jspl->context, js_name, length);
	if (!jstr_name) {
		ret = PLAYLIST_E_OUTOFMEMORY;
		goto error_exit;
	}

	/* Check if main() function exists. */
	/*if (!function_exists(jspl, "main")) {
		ret = PLAYLIST_E_JSMAINNOTFOUND;
		goto error_exit;
	}*/

	/* Call main() function. */
	argv[0] = OBJECT_TO_JSVAL(jspl->tracks);
	argv[1] = STRING_TO_JSVAL(jstr_name);
	if (!JS_CallFunctionName(jspl->context, jspl->global, "main", 2, argv, &retval)) {
		ret = PLAYLIST_E_JSCALLMAIN;
		goto error_exit;
	}

	/* Check if the returned value is a JavaScript object. */
	if (JSVAL_IS_OBJECT(retval) && !JSVAL_IS_PRIMITIVE(retval)) {
		/* Convert the return value to JSObject. */
		JSObject* jsobj = JSVAL_TO_OBJECT(retval);
		if (!jsobj) {
			ret = PLAYLIST_E_JSINVALIDCAST;
			goto error_exit;
		}

		if (object_is_array(jspl->context, jsobj)) {
			/* Single playlist when the returned value is JavaScript array. */
			ret = generate_playlist(jspl, pls, jsobj, name);
			if (ret != 0) {
				goto error_exit;
			}
		} else {
			/* Multiple playlists when the returned value is JavaScript object with attributes. */
			jsint n;
			JSIdArray* keys = JS_Enumerate(jspl->context, jsobj);
			if (!keys) {
				ret = PLAYLIST_E_JSINVALIDOBJECT;
				goto error_exit;
			}

			/* Loop over attributes to enumerate all playlist names. */
			for (n = 0;n < keys->length;++n) {
				/* Obtain the playlist name. */
				jsval jsval_name;
				if (JS_IdToValue(jspl->context, keys->vector[n], &jsval_name)) {
					jsval jsval_obj;
					JSString* jsstr_name = JSVAL_TO_STRING(jsval_name);
					
					/* Obtain the attribute value for the playlist name. */
					if (JS_LookupUCProperty(jspl->context, jsobj, JS_GetStringChars(jsstr_name), JS_GetStringLength(jsstr_name), &jsval_obj)) {
						if (JSVAL_IS_OBJECT(jsval_obj) && !JSVAL_IS_PRIMITIVE(jsval_obj)) {
							JSObject* jsobj = JSVAL_TO_OBJECT(jsval_obj);
							if (jsobj && object_is_array(jspl->context, jsobj)) {
								ucs2char_t* thisname = JSString_to_ucs2(jsstr_name);
								ret = generate_playlist(jspl, pls, jsobj, thisname);
								if (ret != 0) {
									goto error_exit;
								}
							}
						}
					}
				}
			}
		}
	} else {
		/* Does nothing when main() function returns a non-object. */
	}

error_exit:
	jspl_finish(jspl);
	return ret;
}
