/*
 *      Playlist library.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: playlist.c 339 2007-02-11 17:33:05Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef  HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/
#include <time.h>
#include <pmplib/ucs2char.h>
#include <playlist.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>

static const ucs2char_t ucs2cs_m3u[] = {'.','m','3','u',0};
static const ucs2char_t ucs2cs_m3u8[] = {'.','m','3','u','8',0};
static const ucs2char_t ucs2cs_pls[] = {'.','p','l','s',0};

#ifdef HAVE_JSAPI_H
static const ucs2char_t ucs2cs_jspl[] = {'.','j','s','p','l',0};
#endif/*HAVE_JSAPI_H*/

int playlist_m3u_read(playlists_t* pls, const ucs2char_t *filename, int is_utf8);
int playlist_pls_read(playlists_t* pls, const ucs2char_t *filename);

#ifdef HAVE_JSAPI_H
int playlist_jspl_read(
	playlists_t* pls,
	const ucs2char_t *filename,
	const ucs2char_t *path_to_include,
	pmp_music_record_t* records,
	int num_records,
	playlist_callback_t callback,
	void *instance
	);
#endif/*HAVE_JSAPI_H*/

void playlist_init(playlists_t* pls)
{
	memset(pls, 0, sizeof(*pls));
}

void playlist_finish(playlists_t* pls)
{
	int i;
	for (i = 0;i < pls->num_playlists;++i) {
		free(pls->playlists[i].entries);
	}
	free(pls->playlists);
}

int playlist_read(
	playlists_t* pls,
	const ucs2char_t* filename,
	const ucs2char_t* path_to_include,
	pmp_music_record_t* records,
	int num_records,
	playlist_callback_t callback,
	void *instance
	)
{
	if (filepath_hasext(filename, ucs2cs_m3u)) {
		return playlist_m3u_read(pls, filename, 0);
	} else if (filepath_hasext(filename, ucs2cs_m3u8)) {
		return playlist_m3u_read(pls, filename, 1);
	} else if (filepath_hasext(filename, ucs2cs_pls)) {
		return playlist_pls_read(pls, filename);
#ifdef HAVE_JSAPI_H
	} else if (records && filepath_hasext(filename, ucs2cs_jspl)) {
		return playlist_jspl_read(pls, filename, path_to_include, records, num_records, callback, instance);
#endif/*HAVE_JSAPI_H*/
	} else {
		return -1;
	}
}

int playlist_find_playlist(playlists_t* pls, const ucs2char_t* name)
{
	int i = 0;

	for (i = 0;i < pls->num_playlists;++i) {
		if (ucs2cmp(pls->playlists[i].name, name) == 0) {
			break;
		}
	}
	return i;
}

int playlist_add_playlist(playlists_t* pls, const ucs2char_t* name)
{
	int index = playlist_find_playlist(pls, name);
	if (index >= pls->num_playlists) {
		pls->playlists = realloc(pls->playlists, sizeof(playlist_t) * (index+1));
		if (!pls->playlists) {
			return -1;
		}
		ucs2cpy(pls->playlists[index].name, name);
		pls->playlists[index].entries = NULL;
		pls->playlists[index].num_entries = 0;
		++pls->num_playlists;
	}
	return index;
}

int playlist_append(playlist_t* pl, const ucs2char_t* filename, int userdata)
{
    pl->entries = (playlist_entry_t*)realloc(pl->entries, sizeof(playlist_entry_t) * (pl->num_entries+1));
	if (!pl->entries) {
		return -1;
	}
	ucs2cpy(pl->entries[pl->num_entries].filename, filename);
	pl->entries[pl->num_entries].valid = 1;
	pl->entries[pl->num_entries].order = userdata;
	pl->num_entries++;
	return 0;
}

int playlist_is_supported(const ucs2char_t* filename, int flag)
{
	int ret = 0;
	ret |= filepath_hasext(filename, ucs2cs_m3u);
	ret |= filepath_hasext(filename, ucs2cs_m3u8);
	ret |= filepath_hasext(filename, ucs2cs_pls);

#ifdef HAVE_JSAPI_H
	if (flag & PLAYLIST_JSPL) {
		ret |= filepath_hasext(filename, ucs2cs_jspl);
	}
#endif/*HAVE_JSAPI_H*/

	return ret;
}


#ifdef	_WIN32
#define COMP_STR(x, y)	lstrcmpiW(x, y)	/* FAT32 treats upper/lower letters as identical. */
#else
#define COMP_STR(x, y)	ucs2cmp(x, y)
#endif

static int comp_filename(const void *_x, const void *_y)
{
	const playlist_mediafile_t *x = (const playlist_mediafile_t *)_x;
	const playlist_mediafile_t *y = (const playlist_mediafile_t *)_y;
	return COMP_STR(x->file, y->file);
}

static int find_musicfile(
	playlist_mediafile_t* mediafiles,
	int num_mediafiles,
	const ucs2char_t* filename,
	int *begin,
	int *end
	)
{
	int low = 0, high = num_mediafiles-1;

	/* Binary search. */
	while (low <= high) {
		int middle = (low + high) / 2;
		int comp = COMP_STR(filename, mediafiles[middle].file);
		if (comp == 0) {
			/* Found */
			*begin = *end = middle;
			while (*begin >= 0 && COMP_STR(filename, mediafiles[*begin].file) == 0) {
				(*begin)--;
			}
			(*begin)++;
			while (*end < num_mediafiles && COMP_STR(filename, mediafiles[*end].file) == 0) {
				(*end)++;
			}
			return 1;
		} else if (comp < 0) {
			high = middle - 1;
		} else {
			low = middle + 1;
		}
	}
	return 0;
}

static int count_frequency(const ucs2char_t* str, const ucs2char_t* substr)
{
	const ucs2char_t *p = str;
	int freq = 0;
	while (p = ucs2str(p, substr)) {
		p++;
		freq++;
	}
	return freq;
}

static double calc_similarity(const ucs2char_t* x, const ucs2char_t* y)
{
	int match = 0;
	size_t nx = ucs2len(x);
	size_t ny = ucs2len(y);

	/* Calculate Jaccard coefficients of letter bi-grams. */
	while (*x) {
		int freq = 0;
		ucs2char_t bigram_x[3];
		bigram_x[0] = *x;
		bigram_x[1] = *(x+1);
		bigram_x[2] = 0;

		freq = count_frequency(y, bigram_x);
		if (freq > 0) {
			match++;
		}
		x++;
	}

	if (nx == 0 || ny == 0) {
		return 0;
	} else {
		return (double)match / (double)(nx + ny - match);
	}
}

static int search_missing_music(
	ucs2char_t* mediafile,
	playlist_mediafile_t* mediafiles,
	int num_mediafiles
	)
{
	int begin = 0, end = 0;
	const ucs2char_t* filename = NULL;
	ucs2char_t pathname[MAX_PATH];

	if (!mediafile) {
		return 1;
	}

	/* Obtain filename and pathname. */
	filename = filepath_skippath(mediafile);
	ucs2cpy(pathname, mediafile);
	filepath_remove_filespec(pathname);

	/* Search for the file name in music folders. */
	if (find_musicfile(mediafiles, num_mediafiles, filename, &begin, &end)) {
		/* Found music file(s) that have the filename. */
		playlist_mediafile_t* found_music = &mediafiles[begin];

		if (begin + 1 != end) {
			int i, j;
			double max_similarity = 0.0;

			/* Calculate similarities of paths using Jaccard coefficient. */
			for (i = begin;i < end;i++) {
				double similarity = calc_similarity(mediafiles[i].path, pathname);
				if (max_similarity < similarity) {
					j = i;
					max_similarity = similarity;
				}
			}
			found_music = &mediafiles[j];
		}

		ucs2cpy(mediafile, found_music->path);
		ucs2cat(mediafile, found_music->file);
		return 0;
	}

	return 1;
}

void playlist_normalize_prepare(
	playlist_mediafile_t* mediafiles,
	int num_mediafiles
	)
{
	qsort(mediafiles, num_mediafiles, sizeof(*mediafiles), comp_filename);
}

int playlist_normalize(
	playlist_t* pl,
	const ucs2char_t* path_to_playlist,
	const ucs2char_t* path_to_root,
	playlist_mediafile_t* mediafiles,
	int num_mediafiles
	)
{
	int error_occurred = 0, num_success = 0;
	int i;

	for (i = 0;i < pl->num_entries;i++) {
		int valid = 1, success = 0;
		ucs2char_t music_file[MAX_PATH];
		ucs2char_t *wcstr = ucs2dup(pl->entries[i].filename);

		if (wcstr) {
			ucs2cpy(music_file, wcstr);
			filepath_decode(music_file);

			if (filepath_is_relative(music_file)) {
				ucs2char_t tmp[MAX_PATH*2];

				/* Relative path: convert the path to the absolute path. */
				filepath_relative_to_absolute(tmp, path_to_playlist, music_file);
				ucs2cpy(music_file, tmp);
			} else if (music_file[0] == PATHCHAR) {
				ucs2char_t tmp[MAX_PATH*2];

				/* The path beginning from '\': convert the path to the complete form. */
				filepath_relative_to_absolute(tmp, path_to_root, music_file);
				ucs2cpy(music_file, tmp);
			}

			filepath_decode(music_file);

			/* Absolute path: check if the file exists in the H10 drive. */
			if (!filepath_is_same_root(music_file, path_to_root)) {
				valid = 0;	/* Set the error flag. */
			}

			/* Make sure that the music file exists. */
			if (!filepath_file_exists(music_file)) {
				valid = 0;	/* Set the error flag. */
			}

			/* Search for the missing music files in music folders if indicated. */
			if (!valid && mediafiles && num_mediafiles > 0) {
				if (search_missing_music(music_file, mediafiles, num_mediafiles) == 0) {
					valid = 1;	/* Search succeeded: cancel the error flag. */
				}
			}

			/* Strip the root. */
			if (valid) {
				ucs2char_t* ucs2 = ucs2dup(music_file);
				if (ucs2) {
					ucs2cpy(pl->entries[i].filename, ucs2);
					success = 1;
					ucs2free(ucs2);
				}
				pl->entries[i].valid = 1;
			} else {
				pl->entries[i].valid = 0;
			}

			if (success) {
				num_success++;
			}
		}

		ucs2free(wcstr);
	}
	return num_success;
}

static int comp_order(const void* _x, const void* _y)
{
	const playlist_entry_t* x = (const playlist_entry_t*)_x;
	const playlist_entry_t* y = (const playlist_entry_t*)_y;
	return ((x->order)>(y->order))-((x->order)<(y->order));
}

void playlist_shuffle(playlist_t* pl)
{
	int i;

	srand((unsigned int)time(NULL));
	for (i = 0;i < pl->num_entries;++i) {
		pl->entries[i].order = rand();
	}
	qsort(pl->entries, pl->num_entries, sizeof(playlist_entry_t), comp_order);
}
