/*
 *      Media database reader/writer for players with iriver plus 2.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ip2db.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>

#include "util.h"
#include "ip2db.h"

result_t ip2db_init(ip2db_t* db)
{
	int i;

	memset(db, 0, sizeof(*db));

	/* Empty dat array. */
	db->dat_size = 0;
	db->dat_array = NULL;

	/* We need at least pages IP2DBIDX_PAGE_HEADER to IP2DBIDX_PAGE_NODE_FORMAT. */
	db->idx_buffer = (uint8_t*)calloc(IP2DBIDX_PAGE_NODE_FORMAT, PAGESIZE);
	db->idx_pages = (idxpage_t*)calloc(IP2DBIDX_PAGE_NODE_FORMAT, sizeof(idxpage_t));
	for (i = IP2DBIDX_PAGE_HEADER;i <= IP2DBIDX_PAGE_NODE_FORMAT;++i) {
		g_idx_exports[i].init(PAGEBLOCK(db->idx_buffer, i));
		db->idx_pages[i-1].offset = 0x400 * (i-1);
		db->idx_pages[i-1].type = i;
	}

	return 0;
}

void ip2db_finish(ip2db_t* db)
{
	free(db->dat_array);
	free(db->idx_pages);
	free(db->idx_buffer);
	ip2db_init(db);
}

result_t ip2db_read(ip2db_t* db, const ucs2char_t* dat_filename, const ucs2char_t* idx_filename)
{
	result_t ret = 0;
	FILE *fp = NULL;

	// Read dat.
	fp = ucs2fopen(dat_filename, "rb");
	if (!fp) {
		return PMPERR_OPENFORREAD;
	}
	ret = ip2dbdat_read(db, fp);
	if (ret != 0) {
		fclose(fp);
		return ret;
	}
	fclose(fp);

	// Read idx.
	fp = ucs2fopen(idx_filename, "rb");
	if (!fp) {
		return PMPERR_OPENFORREAD;
	}
	ret = ip2dbidx_read(db, fp);
	if (ret != 0) {
		fclose(fp);
		return ret;
	}
	fclose(fp);

	return 0;
}

result_t ip2db_write(ip2db_t* db, const ucs2char_t* dat_filename, const ucs2char_t* idx_filename)
{
	result_t ret = 0;
	FILE *fp = NULL;

	/* Write dat. */
	// Remove the file first so that we can overwrite a hidden file.
	filepath_removefile(dat_filename);
	fp = ucs2fopen(dat_filename, "wb");
	if (!fp) {
		return PMPERR_OPENFORREAD;
	}
	ret = ip2dbdat_write(db, fp);
	if (ret != 0) {
		fclose(fp);
		return ret;
	}
	fclose(fp);

	/* Write idx. */
	filepath_removefile(idx_filename);
	fp = ucs2fopen(idx_filename, "wb");
	if (!fp) {
		return PMPERR_OPENFORREAD;
	}
	ret = ip2dbidx_write(db, fp);
	if (ret != 0) {
		fclose(fp);
		return ret;
	}
	fclose(fp);

	return 0;
}

int ip2db_is_supported_codec(uint32_t codec)
{
	return (
		(codec == PMPCODEC_MPEGLAYER3) ||
		(codec == PMPCODEC_WMA) ||
		(codec == PMPCODEC_VORBIS)
		) ? 1 : 0;
}

int ip2db_is_supported_ext(const ucs2char_t* filename)
	{
	static const ucs2char_t ucs2cs_mp3[] = {'.','m','p','3',0};
	static const ucs2char_t ucs2cs_wma[] = {'.','w','m','a',0};
	static const ucs2char_t ucs2cs_ogg[] = {'.','o','g','g',0};

	return (
		filepath_hasext(filename, ucs2cs_mp3) ||
		filepath_hasext(filename, ucs2cs_wma) ||
		filepath_hasext(filename, ucs2cs_ogg)
		) ? 1 : 0;
}


result_t ip2db_set(ip2db_t* db, const pmp_music_record_t* records, uint32_t num_records, const ucs2char_t* path_to_root)
{
	result_t ret = 0;
	uint32_t i, j, n = 0;
	ip2db_record_t* ip2db_records = NULL;
	static const ucs2char_t ucs2cs_unknown[] = {'u','n','k','n','o','w','n',0};

	// Count valid entries.
	for (i = 0;i < num_records;++i) {
		if (ip2db_is_supported_codec(records[i].codec)) {
			++n;
		}
	}
	
	// Allocate IP2DB records.
	ip2db_records = (ip2db_record_t*)calloc(n, sizeof(ip2db_record_t));
	if (!ip2db_records) {
		return PMPERR_INSUFFICIENTMEMORY;
	}

	// Convert records from pmp_music_record_t to ip2db_record_t.
	for (i = 0, j = 0;i < num_records;++i) {
		const pmp_music_record_t* src = &records[i];
		ip2db_record_t* dst = &ip2db_records[j];

		if (!ip2db_is_supported_codec(src->codec)) {
			continue;
		}

		ip2db_init_record(dst);

		switch (src->codec) {
		case PMPCODEC_MPEGLAYER3:	dst->format = 0; break;
		case PMPCODEC_VORBIS:		dst->format = 3; break;
		case PMPCODEC_WMA:			dst->format = 5; break;
		}

		dst->entry_number = j+1;
		dst->pathname = ucs2dup(filepath_changeroot(src->filename, path_to_root));
		filepath_remove_filespec(dst->pathname);
		filepath_encode(dst->pathname);
		dst->filename = ucs2dup(filepath_skippath(src->filename));
		dst->title = ucs2dup(src->title ? src->title : dst->filename);
		dst->artist = ucs2dup(src->artist ? src->artist : ucs2cs_unknown);
		dst->album = ucs2dup(src->album ? src->album : ucs2cs_unknown);
		dst->genre = ucs2dup(src->genre ? src->genre : ucs2cs_unknown);
		dst->rating = src->rating;
		dst->play_count = src->play_count;
		dst->recent_play = src->ts_playback;
		dst->track_number = src->track_number;
		if (src->date) {
			dst->year = ucs2toi(src->date);
		}
		dst->filesize = src->filesize;
		dst->duration = src->duration;
		dst->sample_rate = src->sample_rate;
		dst->bitrate = src->bitrate;
		dst->timestamp = src->ts_update;
		++j;
	}

	// Construct dat_t array.
	ret = ip2dbdat_construct(db, ip2db_records, n);
	if (ret) {
		return ret;
	}

	// Construct idx_t nodes.
	ret = ip2dbidx_construct(db, ip2db_records, n);
	if (ret) {
		return ret;
	}

	// Set unknown2 value in idx_header as (n+1).
	i = n + 1;
	idxheader_setget_unknown2(db->idx_buffer, &i, 1);

	// Free the allocated memory block.
	if (ip2db_records) {
		for (i = 0;i < n;++i) {
			ip2db_free_record(&ip2db_records[i]);
		}
		free(ip2db_records);
	}

	return ret;
}

result_t ip2db_get(ip2db_t* db, pmp_music_record_t* records, uint32_t* num_records, const ucs2char_t* path_to_root)
{
	uint32_t i = 0, j = 0, n = ip2db_get_num_record(db);

	if (!records) {
		*num_records = n;
		return 0;
	}

	if (*num_records < n) {
		*num_records = n;
		return PMPERR_INSUFFICIENTMEMORY;
	}

	for (i = 1, j = 0;i <= n;++i) {		// recid starts at 1.
		size_t length = 0;
		ucs2char_t tmp[128];
		ip2db_record_t src;
		pmp_music_record_t* dst = &records[j];

		pmplib_record_init(dst);

		ip2db_get_record(db, i, &src);

		filepath_decode(src.pathname);
		length  = ucs2len(path_to_root);
		length += ucs2len(src.pathname);
		length += ucs2len(src.filename);
		length += 3;

		dst->filename = (ucs2char_t*)ucs2malloc(sizeof(ucs2char_t) * length);
		filepath_combinepath(dst->filename, length, path_to_root, src.pathname);
		filepath_addslash(dst->filename);
		ucs2cat(dst->filename, src.filename);

		dst->title = ucs2dup(src.title);
		dst->artist = ucs2dup(src.artist);
		dst->album = ucs2dup(src.album);
		dst->genre = ucs2dup(src.genre);
		itoucs2(src.year, tmp, 10);
		dst->date = ucs2dup(tmp);

		switch (src.format) {
		case 0:	dst->codec = PMPCODEC_MPEGLAYER3;	break;
		case 3: dst->codec = PMPCODEC_VORBIS;		break;
		case 5: dst->codec = PMPCODEC_WMA;			break;
		}

		dst->track_number = src.track_number;
		dst->sample_rate = src.sample_rate;
		dst->bitrate = src.bitrate;
		dst->duration = src.duration;
		dst->filesize = src.filesize;
		dst->ts_update = src.timestamp;
		dst->rating = src.rating;
		dst->play_count = src.play_count;
		dst->ts_playback = src.recent_play;

		++j;
	}

	*num_records = n;

	return 0;
}

uint32_t ip2db_get_num_record(ip2db_t* db)
{
	uint32_t value;
	idxheader_setget_num_entries(db->idx_buffer, &value, 0);
	return value;
}

void ip2db_set_num_record(ip2db_t* db, uint32_t value)
{
	idxheader_setget_num_entries(db->idx_buffer, &value, 1);
}

uint32_t ip2db_get_num_pages(ip2db_t* db)
{
	uint32_t value;
	idxheader_setget_num_pages(db->idx_buffer, &value, 0);
	return value;
}

void ip2db_init_record(ip2db_record_t* record)
{
	memset(record, 0, sizeof(*record));
}

void ip2db_free_record(ip2db_record_t* record)
{
	ucs2free(record->pathname);
	ucs2free(record->filename);
	ucs2free(record->title);
	ucs2free(record->artist);
	ucs2free(record->album);
	ucs2free(record->genre);
	ip2db_init_record(record);
}

result_t ip2db_get_record(ip2db_t* db, uint32_t recid, ip2db_record_t* record)
{
	uint32_t leafid = 0;
	dat_t* dat = NULL;
	ip2db_idxleaf_data_t ld;
	
	/* Check if the leafid is valid. */
	if (recid < 1 || ip2db_get_num_record(db) < recid) {
		return PMPERR_INCONSISTENCY;
	}

	/* Fill the record from an element of dat array. */
	dat = &db->dat_array[recid-1];
	record->entry_number = dat->entry_number;
	record->rating = dat->rating;
	record->play_count = dat->play_count;
	record->recent_play = dat->recent_play;
	record->format = dat->format;
	record->track_number = dat->track_number;
	record->year = dat->year;
	record->filesize = dat->filesize;
	record->duration = dat->duration;
	record->sample_rate = dat->sample_rate;
	record->bitrate = dat->bitrate;
	record->timestamp = dat->timestamp;

	/* Fill the record from an idx leaf. */
	idx_leafdata_init(&ld);
	idxleaf_setget_data(
		PAGEBLOCK(db->idx_buffer, dat->idx_item_page),
		dat->idx_item_index,
		dat->idx_item_field_access,
		&ld,
		0
		);
	record->pathname = ld.pathname;
	record->filename = ld.filename;
	record->title = ld.title;
	record->artist = ld.artist;
	record->album = ld.album;
	record->genre = ld.genre;

	return 0;
}

result_t ip2db_dump(ip2db_t* db, FILE *fp)
{
	uint32_t i, num_pages, num_records;

	/* Dump the content of idx. */
	fprintf(fp, "// IP2DB idx\n");
	idxheader_setget_num_pages(db->idx_buffer, &num_pages, 0);
	for (i = 0;i < num_pages;++i) {
		uint8_t *p = &db->idx_buffer[db->idx_pages[i].offset];
		g_idx_exports[db->idx_pages[i].type].dump(p, db->idx_pages[i].offset, fp);
	}
	fprintf(fp, "\n");

	fprintf(fp, "// IP2DB dat\n");
	num_records = ip2db_get_num_record(db);
	for (i = 1;i <= num_records;++i) {
		fprintf(fp, "RECORD %d = {\n", i);
		dat_dump(&db->dat_array[i-1], fp);
		fprintf(fp, "}\n");
	}
	fprintf(fp, "\n");
	return 0;
}

result_t ip2db_repr(ip2db_t* db, FILE *fp)
{
	uint32_t i, type, n = ip2db_get_num_record(db);

	/* IP2DB header information. */
	fprintf(fp, "// IP2DB idx header\n");
	idxheader_repr(db->idx_buffer, 0, fp);
	fprintf(fp, "\n");

	/* List of records. */
	fprintf(fp, "// IP2DB record list\n");
	for (i = 1;i <= n;++i) {
		ip2db_record_t record;
		ip2db_init_record(&record);
		ip2db_get_record(db, i, &record);

		fprintf(fp, "RECORD %d = [\n", i);
		fprintf(fp, "  entry_number: %d\n", record.entry_number);
		fprints(fp, "  pathname: %s\n", record.pathname);
		fprints(fp, "  filename: %s\n", record.filename);
		fprints(fp, "  title: %s\n", record.title);
		fprints(fp, "  artist: %s\n", record.artist);
		fprints(fp, "  album: %s\n", record.album);
		fprints(fp, "  genre: %s\n", record.genre);
		fprintf(fp, "  rating: %d\n", record.rating);
		fprintf(fp, "  play_count: %d\n", record.play_count);
		fprintt(fp, "  recent_play: %s", record.recent_play);
		fprintf(fp, "  format: %d\n", record.format);
		fprintf(fp, "  track_number: %d\n", record.track_number);
		fprintf(fp, "  year: %d\n", record.year);
		fprintf(fp, "  filesize: %d\n", record.filesize);
		fprintf(fp, "  duration: %d\n", record.duration);
		fprintf(fp, "  sample_rate: %d\n", record.sample_rate);
		fprintf(fp, "  bitrate: %d\n", record.bitrate);
		fprintf(fp, "  timestamp: %d\n", record.timestamp);
		fprintf(fp, "];\n");

		ip2db_free_record(&record);
	}
	fprintf(fp, "\n");

	/* List of indices */
	fprintf(fp, "// IP2DB indices\n");
	for (type = IP2DBIDX_PAGE_NODE_ENTRYNUMBER;type <= IP2DBIDX_PAGE_NODE_FORMAT;++type) {
		idx_key_t idxkey;
		uint32_t recid, order = 0;
		result_t idxres;

		/* Output index type. */
		fprintf(fp, "INDEX ");
		switch (type) {
		case IP2DBIDX_PAGE_NODE_ENTRYNUMBER:		fprintf(fp, "(ENTRYNUMBER)"); break;
		case IP2DBIDX_PAGE_NODE_FILENAME:			fprintf(fp, "(FILENAME)"); break;
		case IP2DBIDX_PAGE_NODE_TITLE:				fprintf(fp, "(TITLE)"); break;
		case IP2DBIDX_PAGE_NODE_ARTIST:				fprintf(fp, "(ARTIST)"); break;
		case IP2DBIDX_PAGE_NODE_ALBUM:				fprintf(fp, "(ALBUM)"); break;
		case IP2DBIDX_PAGE_NODE_GENRE:				fprintf(fp, "(GENRE)"); break;
		case IP2DBIDX_PAGE_NODE_GENRE_ARTIST:		fprintf(fp, "(GENRE_ARTIST)"); break;
		case IP2DBIDX_PAGE_NODE_GENRE_ALBUM:		fprintf(fp, "(GENRE_ALBUM)"); break;
		case IP2DBIDX_PAGE_NODE_GENRE_ARTIST_ALBUM:	fprintf(fp, "(GENRE_ARTIST_ALBUM)"); break;
		case IP2DBIDX_PAGE_NODE_ARTIST_ALBUM:		fprintf(fp, "(ARTIST_ALBUM)"); break;
		case IP2DBIDX_PAGE_NODE_RATING:				fprintf(fp, "(RATING)"); break;
		case IP2DBIDX_PAGE_NODE_PLAYCOUNT:			fprintf(fp, "(PLAYCOUNT)"); break;
		case IP2DBIDX_PAGE_NODE_RECENTPLAY:			fprintf(fp, "(RECENTPLAY)"); break;
		case IP2DBIDX_PAGE_NODE_FORMAT:				fprintf(fp, "(FORMAT)"); break;
		}
		fprintf(fp, " = [\n");

		memset(&idxkey, 0, sizeof(idxkey));
		idxkey.type = type;

		idxres = idx_first(db->idx_buffer, idxkey.type, &idxkey, &recid);
		while (idxres == 0) {
			fprintf(fp, "  #%-4d ", order++);
			fprintf(fp, "  ");

			switch (type) {
			case IP2DBIDX_PAGE_NODE_ENTRYNUMBER:
				fprintf(fp, "%d: %d\n", idxkey.data.dword, recid);
				break;
			case IP2DBIDX_PAGE_NODE_FILENAME:
			case IP2DBIDX_PAGE_NODE_TITLE:
			case IP2DBIDX_PAGE_NODE_ARTIST:
			case IP2DBIDX_PAGE_NODE_ALBUM:
			case IP2DBIDX_PAGE_NODE_GENRE:
				fprintf(fp, "(");
				print_keystr(fp, idxkey.data.str, 8);
				fprintf(fp, ", %d): %d\n", idxkey.dup, recid);
				break;
			case IP2DBIDX_PAGE_NODE_GENRE_ARTIST:
			case IP2DBIDX_PAGE_NODE_GENRE_ALBUM:
			case IP2DBIDX_PAGE_NODE_ARTIST_ALBUM:
				fprintf(fp, "(");
				print_keystr(fp, idxkey.data.str, 16);
				fprintf(fp, ", %d): %d\n", idxkey.dup, recid);
				break;
			case IP2DBIDX_PAGE_NODE_GENRE_ARTIST_ALBUM:
				fprintf(fp, "(");
				print_keystr(fp, idxkey.data.str, 24);
				fprintf(fp, ", %d): %d\n", idxkey.dup, recid);
				break;
			case IP2DBIDX_PAGE_NODE_RATING:
			case IP2DBIDX_PAGE_NODE_PLAYCOUNT:
			case IP2DBIDX_PAGE_NODE_RECENTPLAY:
			case IP2DBIDX_PAGE_NODE_FORMAT:
				fprintf(fp, "(%d, %d): %d\n", idxkey.data.dword, idxkey.dup, recid);
				break;
			}

			/* Retrieve the next item. */
			idxres = idx_next(db->idx_buffer, idxkey.type, &idxkey, &recid);
		}
		fprintf(fp, "]\n");
	}
	fprintf(fp, "\n");

	return 0;
}
