/*
 *      Media database reader/writer for players with iriver plus 2.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ip2db.h 328 2007-02-10 17:50:11Z nyaochi $ */

#ifndef	__IP2DB_IP2DB_H__
#define	__IP2DB_IP2DB_H__

#include <pmplib/pmp.h>


#define	PAGESIZE				0x400
#define	NUM_LEAF_CHAIN_LISTS	5
#define	DAT_SIZE_PER_RECORD		0x13F
#define	PAGEBLOCK(base, page)	&(base)[PAGESIZE * ((page)-1)]




enum {
	IP2DBDAT_IDX_PATHNAME = 0,
	IP2DBDAT_IDX_FILENAME,
	IP2DBDAT_IDX_TITLE,
	IP2DBDAT_IDX_ARTIST,
	IP2DBDAT_IDX_ALBUM,
	IP2DBDAT_IDX_GENRE,
	IP2DBDAT_IDX_LAST,
};

typedef struct {
	uint16_t	offset;
	uint16_t	size;
} offset_size_t;

typedef struct {
	uint32_t		entry_number;
	ucs2char_t		filename[8];
	ucs2char_t		title[8];
	ucs2char_t		artist[8];
	ucs2char_t		album[8];
	ucs2char_t		genre[8];
	uint32_t		rating;
	uint32_t		play_count;
	uint32_t		recent_play;
	uint32_t		format;
	uint32_t		track_number;
	uint32_t		unknown5;
	uint32_t		year;
	uint32_t		filesize;
	uint32_t		duration;
	uint32_t		sample_rate;
	uint32_t		bitrate;
	uint32_t		timestamp;
	uint8_t			unknown1[156];
	offset_size_t	idx_item_field_access[IP2DBDAT_IDX_LAST];
	uint8_t			unknown2;				/* 0x0A	*/
	uint16_t		idx_item_size;
	uint8_t			idx_item_index;
	uint8_t			unknown3;				/* 0x00 */
	uint16_t		idx_item_page;
} dat_t;


enum {
	IP2DBIDX_PAGE_NONE = 0,
	IP2DBIDX_PAGE_HEADER,					/* @1 */
	IP2DBIDX_PAGE_DESCRIPTOR,				/* @2 */
	IP2DBIDX_PAGE_NODE_ENTRYNUMBER,			/* @3 */
	IP2DBIDX_PAGE_NODE_FILENAME,			/* @4 */
	IP2DBIDX_PAGE_NODE_TITLE,				/* @5 */
	IP2DBIDX_PAGE_NODE_ARTIST,				/* @6 */
	IP2DBIDX_PAGE_NODE_ALBUM,				/* @7 */
	IP2DBIDX_PAGE_NODE_GENRE,				/* @8 */
	IP2DBIDX_PAGE_NODE_GENRE_ARTIST,		/* @9 */
	IP2DBIDX_PAGE_NODE_GENRE_ALBUM,			/* @10 */
	IP2DBIDX_PAGE_NODE_GENRE_ARTIST_ALBUM,	/* @11 */
	IP2DBIDX_PAGE_NODE_ARTIST_ALBUM,		/* @12 */
	IP2DBIDX_PAGE_NODE_RATING,				/* @13 */
	IP2DBIDX_PAGE_NODE_PLAYCOUNT,			/* @14 */
	IP2DBIDX_PAGE_NODE_RECENTPLAY,			/* @15 */
	IP2DBIDX_PAGE_NODE_FORMAT,				/* @16 */
	IP2DBIDX_PAGE_LEAF,						/* @17 (usually) */
};



typedef struct {
	ucs2char_t*		pathname;
	ucs2char_t*		filename;
	ucs2char_t*		title;
	ucs2char_t*		artist;
	ucs2char_t*		album;
	ucs2char_t*		genre;
} ip2db_idxleaf_data_t;


typedef struct {
	int type;
	uint32_t offset;
} idxpage_t;

typedef struct {
	dat_t*		dat_array;
	uint32_t	dat_size;

	uint8_t*	idx_buffer;
	uint32_t	idx_size;
	idxpage_t*	idx_pages;
} ip2db_t;

typedef struct {
	int type;
	union {
		uint32_t	dword;
		ucs2char_t	str[24];
	} data;
	uint32_t dup;
} idx_key_t;

typedef struct {
	uint32_t		entry_number;
	ucs2char_t		*filename;
	ucs2char_t		*pathname;
	ucs2char_t		*title;
	ucs2char_t		*artist;
	ucs2char_t		*album;
	ucs2char_t		*genre;
	uint32_t		rating;
	uint32_t		play_count;
	uint32_t		recent_play;
	uint32_t		format;
	uint32_t		track_number;
	uint32_t		year;
	uint32_t		filesize;
	uint32_t		duration;
	uint32_t		sample_rate;
	uint32_t		bitrate;
	uint32_t		timestamp;
} ip2db_record_t;

typedef struct {
	ip2db_record_t* records;
	uint32_t index;
} sortitem_t;

typedef struct {
	void (*init)(uint8_t* block);
	void (*setget_num_childlen)(uint8_t* block, uint16_t* value, int is_storing);
	uint8_t (*get_height)(uint8_t* block);
	void (*record_to_key)(idx_key_t* dst, const ip2db_record_t* src);
	void (*setget_child)(uint8_t* block, int i, idx_key_t* key, uint32_t* child, int is_storing);
	int (*comp_idxkey)(const idx_key_t* x, const idx_key_t* y);
	void (*dump)(uint8_t* block, uint32_t offset, FILE *fp);
	int (*comp)(const sortitem_t *x, const sortitem_t *y);
} idx_export_t;



// dat.c
void dat_init(dat_t* dat);
size_t dat_serialize(dat_t* dat, uint8_t* buffer, int is_storing);
void dat_dump(dat_t* dat, FILE *fp);

// idx.c
void idx_repr(uint8_t* buffer, idxpage_t* pages, FILE *fpo);
int idx_first(uint8_t* buffer, uint32_t node, idx_key_t* idxkey, uint32_t* leafid);
int idx_next(uint8_t* buffer, uint32_t node, idx_key_t* idxkey, uint32_t* leafid);
result_t idx_get(uint8_t* buffer, idx_key_t* idxkey, uint32_t* leafid, int flag);

// idx_descriptor.c
void idxdescriptor_init(uint8_t* block);
void idxdescriptor_repr(uint8_t* block, uint32_t offset, FILE *fp);

// idx_exports.c
int ucs2cmp_fixed(const ucs2char_t* x, const ucs2char_t* y, size_t n);
extern idx_export_t g_idx_exports[IP2DBIDX_PAGE_LEAF+1];

// idx_header.c
void idxheader_setget_unknown1(uint8_t* block, uint8_t* array, int is_storing);
void idxheader_setget_num_entries(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_setget_num_pages(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_setget_unknown2(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_setget_unknown3(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_setget_unknown4(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_setget_unknown5(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_setget_unknown6(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_setget_leaftail(uint8_t* block, uint32_t* values, int is_storing);
void idxheader_setget_unknown7(uint8_t* block, uint8_t* array, int is_storing);
void idxheader_setget_unknown8(uint8_t* block, uint32_t* value, int is_storing);
void idxheader_init(uint8_t* block);
void idxheader_repr(uint8_t* block, uint32_t offset, FILE *fp);

// idx_leaf.c
void idxleaf_setget_unknown1(uint8_t* block, uint32_t* value, int is_storing);
void idxleaf_setget_leaf_prev(uint8_t* block, uint32_t* value, int is_storing);
void idxleaf_setget_leaf_next(uint8_t* block, uint32_t* value, int is_storing);
void idxleaf_setget_size_avail(uint8_t* block, uint16_t* value, int is_storing);
void idxleaf_setget_size_used(uint8_t* block, uint16_t* value, int is_storing);
void idxleaf_setget_unknown2(uint8_t* block, uint16_t* value, int is_storing);
void idxleaf_setget_unknown3(uint8_t* block, uint16_t* value, int is_storing);
void idxleaf_setget_num_data(uint8_t* block, uint16_t* value, int is_storing);
void idxleaf_setget_leaf_line(uint8_t* block, uint8_t* value, int is_storing);
void idxleaf_setget_unknown5(uint8_t* block, uint8_t* value, int is_storing);
void idxleaf_setget_unknown6(uint8_t* block, uint16_t* value, int is_storing);
void idxleaf_setget_data_offset_size(uint8_t* block, int i, offset_size_t* value, int is_storing);
void idxleaf_setget_data_direct(uint8_t* block, int i, ip2db_idxleaf_data_t* data, int is_storing);
void idxleaf_setget_data(
	uint8_t* block,
	int i,
	offset_size_t* field_access,
	ip2db_idxleaf_data_t* data,
	int is_storing
	);
void idxleaf_init(uint8_t* block);
void idxleaf_set_field_access(ip2db_idxleaf_data_t* data, offset_size_t* field_access);
void idxleaf_repr(uint8_t* block, uint32_t offset, FILE *fp);
void idx_leafdata_init(ip2db_idxleaf_data_t* data);
void idx_leafdata_free(ip2db_idxleaf_data_t* data);

// idx_node.c
void idxnode_setget_size(uint8_t* block, uint16_t* value, int is_storing);
void idxnode_setget_num_children(uint8_t* block, uint16_t* value, size_t keysize, int is_storing);
void idxnode_setget_height(uint8_t* block, uint8_t* value, int is_storing);
void idxnode_setget_dword(uint8_t* block, uint16_t i, uint32_t* key, uint32_t* dup, uint32_t* next, int is_storing);
void idxnode_setget_str(uint8_t* block, uint16_t i, ucs2char_t* key, size_t keylen, uint32_t* dup, uint32_t* next, int is_storing);
void idxnode_init(uint8_t* block);
void idxnode_repr_dword(uint8_t* block, int has_dup, uint32_t offset, FILE *fp);
void idxnode_repr_str(uint8_t* block, size_t keylen, uint32_t offset, FILE *fp);

// ip2db.c
result_t ip2db_init(ip2db_t* db);
void ip2db_finish(ip2db_t* db);
result_t ip2db_read(ip2db_t* db, const ucs2char_t* dat_filename, const ucs2char_t* idx_filename);
result_t ip2db_write(ip2db_t* db, const ucs2char_t* dat_filename, const ucs2char_t* idx_filename);
int ip2db_is_supported_codec(uint32_t codec);
int ip2db_is_supported_ext(const ucs2char_t* filename);
result_t ip2db_set(ip2db_t* db, const pmp_music_record_t* records, uint32_t num_records, const ucs2char_t* path_to_root);
result_t ip2db_get(ip2db_t* db, pmp_music_record_t* records, uint32_t* num_records, const ucs2char_t* path_to_root);
uint32_t ip2db_get_num_record(ip2db_t* db);
void ip2db_set_num_record(ip2db_t* db, uint32_t value);
uint32_t ip2db_get_num_pages(ip2db_t* db);
void ip2db_init_record(ip2db_record_t* record);
void ip2db_free_record(ip2db_record_t* record);
result_t ip2db_get_record(ip2db_t* db, uint32_t recid, ip2db_record_t* record);
result_t ip2db_dump(ip2db_t* db, FILE *fp);
result_t ip2db_repr(ip2db_t* db, FILE *fp);

// ip2db_dat.c
result_t ip2dbdat_construct(ip2db_t* db, ip2db_record_t* records, uint32_t num_records);
result_t ip2dbdat_read(ip2db_t* db, FILE *fp);
result_t ip2dbdat_write(ip2db_t* db, FILE *fp);

// ip2db_idx.c
result_t ip2dbidx_read(ip2db_t* db, FILE *fp);
result_t ip2dbidx_write(ip2db_t* db, FILE *fp);
uint32_t ip2dbidx_add_page(ip2db_t* db, int type);
result_t ip2dbidx_construct(ip2db_t* db, ip2db_record_t* records, uint32_t num_records);

// playlist.c
int ip2db_playlist_write(
	ip2db_t* db,
	const ucs2char_t *name,
	ucs2char_t* const mediafiles[],
	int num_mediafiles,
	const ucs2char_t *path_to_root,
	const ucs2char_t *path_to_playlist,
	const ucs2char_t *ext
	);

#endif/*__IP2DB_IP2DB_H__*/
