/*
 *      High level interface for dat (e.g., U10.dat).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ip2db_dat.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/pmp.h>

#include "util.h"
#include "ip2db.h"


static void fill_dat(dat_t* dat, ip2db_record_t* record)
{
	dat_init(dat);
	dat->entry_number = record->entry_number;
	dat->rating = record->rating;
	dat->play_count = record->play_count;
	dat->recent_play = record->recent_play;
	dat->format = record->format;
	dat->track_number = record->track_number;
	dat->year = record->year;
	dat->filesize = record->filesize;
	dat->duration = record->duration;
	dat->sample_rate = record->sample_rate;
	dat->bitrate = record->bitrate;
	dat->timestamp = record->timestamp;
	ucs2ncpy(dat->filename, record->filename, 8);
	ucs2ncpy(dat->title, record->title, 8);
	ucs2ncpy(dat->artist, record->artist, 8);
	ucs2ncpy(dat->album, record->album, 8);
	ucs2ncpy(dat->genre, record->genre, 8);
}

result_t ip2dbdat_construct(ip2db_t* db, ip2db_record_t* records, uint32_t num_records)
{
	uint32_t i;

	// (Re)allocate array for records.
	free(db->dat_array);
	db->dat_array = (dat_t*)malloc(sizeof(dat_t) * num_records);
	db->dat_size = num_records;
	if (!db->dat_array) {
		db->dat_size = 0;
		return PMPERR_INSUFFICIENTMEMORY;
	}

	// Just set records.
	for (i = 0;i < num_records;++i) {
		// Fill record fields (idx related fields are not set).
		fill_dat(&db->dat_array[i], &records[i]);
	}

	// Update the number of records.
	ip2db_set_num_record(db, num_records);

	return 0;
}

result_t ip2dbdat_read(ip2db_t* db, FILE *fp)
{
	int i;
	long filesize;
	uint8_t *buffer = NULL, *p = NULL;

	/* Read dat file. */
	fread_all(fp, &buffer, &filesize);

	/* Allocate buffers for dat. */
	db->dat_size = filesize / DAT_SIZE_PER_RECORD;
	db->dat_array = (dat_t*)malloc(sizeof(dat_t) * db->dat_size);

	/* Parse dat. */
	p = buffer;
	for (i = 0;i < (int)db->dat_size;++i) {
		size_t size = dat_serialize(&db->dat_array[i], p, 0);
		if (size == 0) {
			break;
		}
		p += size;
	}

	free(buffer);

	return 0;
}

result_t ip2dbdat_write(ip2db_t* db, FILE *fp)
{
	if (0 < db->dat_size) {
		int i;
		uint8_t *p = NULL;
		uint8_t *dat_buffer = malloc(DAT_SIZE_PER_RECORD * db->dat_size);
		if (!dat_buffer) {
			return PMPERR_INSUFFICIENTMEMORY;
		}

		p = dat_buffer;
		for (i = 0;i < (int)db->dat_size;++i) {
			size_t size = dat_serialize(&db->dat_array[i], p, 1);
			if (size == 0) {
				break;
			}
			p += size;
		}
		fwrite(dat_buffer, DAT_SIZE_PER_RECORD, db->dat_size, fp);

		free(dat_buffer);
	}
	return 0;
}
