/*
 *      PMP library implementation for the players with iriver plus 2.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: pmp_iriverplus2.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>

#include "ip2db.h"

#ifdef	PMP_IRIVERPLUS2_EXPORTS
#define	PMPIRIVERPLUS2API	__declspec(dllexport)
#else
#define	PMPIRIVERPLUS2API
#endif


typedef struct {
	const char *id;
	const char *manufacturer;
	const char *name;
	const char *mode;
	const char *min_version;
	const char *max_version;
	const char *sys_filename;
	const char *dat_filename;
	const char *idx_filename;
	const char *extensions;
	uint32_t   codecs[8];
	const char *path_to_system;
	const char *path_to_music;
	const char *path_to_playlist;
	const char *playlist_ext;
} ip2model_descriptor_t;

static const ip2model_descriptor_t g_model_descriptions[] = {
	{
		"iriver_h10jr_ums_1.00-1.61", "iriver", "H10Jr UMS", "UM",
		"1.00", "1.61",
		"System\\H10_Jr.SYS", "System\\H10_Jr.DAT", "System\\H10_Jr.IDX",
		".mp3\0.ogg\0.wma\0.wav\0",
		{PMPCODEC_MPEGLAYER3, PMPCODEC_VORBIS, PMPCODEC_WMA, PMPCODEC_WAV, 0, 0, 0, 0},
		"System", "Music", "Playlists",
		".plp",
	},
	{
		"iriver_u10_ums_1.00-1.65", "iriver", "U10 UMS", "UM",
		"1.00", "1.65",
		"System\\U10.SYS", "System\\U10.dat", "System\\U10.idx",
		".mp3\0.ogg\0.wma\0.wav\0",
		{PMPCODEC_MPEGLAYER3, PMPCODEC_VORBIS, PMPCODEC_WMA, PMPCODEC_WAV, 0, 0, 0, 0},
		"System", "Music", "Playlists",
		".plp",
	},
	{
		NULL, NULL, NULL, NULL,
		NULL, NULL,
		NULL, NULL, NULL,
		NULL,
		{0, 0, 0, 0, 0, 0, 0, 0},
		NULL, NULL, NULL,
		NULL,
	},
};

typedef struct {
	const ip2model_descriptor_t* decl;
} pmp_internal_t;

typedef struct {
	ip2db_t ip2db;
	pmp_music_record_t* records;
	int num_records;
	pmp_playlist_t* playlists;
	int num_playlists;
} pmp_music_internal_t;

static uint32_t pmp_add_ref(pmp_t* pmp);
static uint32_t pmp_release(pmp_t* pmp);
static result_t pmp_open(pmp_t* pmp, uint32_t flag);
static result_t pmp_close(pmp_t* pmp);
static result_t pmp_create_instance_music(pmp_t* pmp, pmp_music_t** ptr_pmpdb);

static uint32_t pmpmusic_release(pmp_music_t* music);
static uint32_t pmpmusic_open(pmp_music_t* music);
static uint32_t pmpmusic_close(pmp_music_t* music);
static result_t pmpmusic_set_records(pmp_music_t* music, const pmp_music_record_t* records, uint32_t num_records);
static result_t pmpmusic_get_records(pmp_music_t* music, pmp_music_record_t* records, uint32_t* num_records);
static result_t pmpmusic_dump(pmp_music_t* music, FILE *fp, int level);
static result_t pmpmusic_set_playlists(pmp_music_t* music, const pmp_playlist_t* playlists, uint32_t num_playlists);


#define	COMP(a, b)	((a)>(b))-((a)<(b))

static char* strip(char *str)
{
	char *p = str + strlen(str) - 1;
	while (*str && isspace(*str)) {
		str++;
	}
	while (str <= p && isspace(*p)) {
		*p-- = 0;
	}
	return str;
}

static const char *strcpy_if_empty(char *dst, const char *src)
{
	return *dst ? dst : strcpy(dst, src);
}

static void set_device_info(
	const char *id,
	const ucs2char_t* path_to_device,
	const ip2model_descriptor_t* md,
	pmp_device_information_t* info
	)
{
	uint32_t n;
	const char *p = NULL;
	ucs2char_t* ucs2 = NULL;
	pmp_device_description_t* decl = (pmp_device_description_t*)&info->decl;

	strcpy_if_empty(decl->id, id);
	strcpy_if_empty(decl->manufacturer, md->manufacturer);
	strcpy_if_empty(decl->name, md->name);
	strcpy_if_empty(decl->mode, md->mode);
	//strcpy_if_empty(decl->language, md->language);
	//strcpy_if_empty(decl->version, md->version);
	strcpy_if_empty(decl->min_version, md->min_version);
	strcpy_if_empty(decl->max_version, md->max_version);

	ucs2cpy(info->path_to_root, path_to_device);

	ucs2 = mbsdupucs2(md->path_to_system);
	ucs2cpy(info->path_to_system, ucs2);
	ucs2free(ucs2);

	ucs2 = mbsdupucs2(md->path_to_music);
	ucs2cpy(info->path_to_music, ucs2);
	ucs2free(ucs2);

	ucs2 = mbsdupucs2(md->path_to_playlist);
	ucs2cpy(info->path_to_playlist, ucs2);
	ucs2free(ucs2);

	info->music_flag = PMPMF_SUPPORT | PMPMF_RECURSIVE;
	info->playlist_flag = PMPPF_SUPPORT;

	// Audio codecs.
	for (n = 0;md->codecs[n];++n) ;
	info->num_audio_codecs = n;
	info->audio_codecs = (uint32_t*)ucs2malloc(sizeof(uint32_t) * info->num_audio_codecs);
	for (n = 0;n < info->num_audio_codecs;++n) {
		info->audio_codecs[n] = md->codecs[n];
	}

	// Obtain the number of extensions separated by '\0' characters.
	for (n = 0, p = md->extensions;*p;p += (strlen(p)+1)) {
		n++;
	}
	info->num_audio_extensions = n;
	info->audio_extensions = (ucs2char_t**)ucs2malloc(sizeof(ucs2char_t*) * info->num_audio_extensions);
	for (n = 0, p = md->extensions;*p;p += (strlen(p)+1)) {
		info->audio_extensions[n++] = mbsdupucs2(p);
	}
}

static void free_device_info(pmp_device_information_t* info)
{
	uint32_t i;
	for (i = 0;i < info->num_audio_extensions;++i) {
		ucs2free(info->audio_extensions[i]);
	}
	ucs2free(info->audio_codecs);
	ucs2free(info->audio_extensions);
	memset(info, 0, sizeof(*info));
}

static void set_filenames(ucs2char_t *sys, ucs2char_t *dat, ucs2char_t *idx, pmp_t *pmp)
{
	ucs2char_t* ucs2 = NULL;
	pmp_internal_t* pmpi = (pmp_internal_t*)pmp->instance;

	ucs2cpy(sys, pmp->info.path_to_root);
	filepath_addslash(sys);
	ucs2 = mbsdupucs2(pmpi->decl->sys_filename);
	ucs2cat(sys, ucs2);
	ucs2free(ucs2);

	ucs2cpy(dat, pmp->info.path_to_root);
	filepath_addslash(dat);
	ucs2 = mbsdupucs2(pmpi->decl->dat_filename);
	ucs2cat(dat, ucs2);
	ucs2free(ucs2);

	ucs2cpy(idx, pmp->info.path_to_root);
	filepath_addslash(idx);
	ucs2 = mbsdupucs2(pmpi->decl->idx_filename);
	ucs2cat(idx, ucs2);
	ucs2free(ucs2);
}


static int compare_version(const char *x, const char *y)
{
	char *p = NULL, *q = NULL;

	for (;;) {
		long a = strtol(x, &p, 10);
		long b = strtol(y, &q, 10);
		int value = COMP(a, b);
		if (value != 0) {
			return value;
		}
		if (!*p || !*q || *p != *q) {
			return COMP(*p, *q);
		}
		x = p+1;
		y = q+1;
	}
}

static int detect_model(
	const ucs2char_t* path_to_device,
	const ip2model_descriptor_t* md,
	pmp_device_information_t* ptr_info
	)
{
	ucs2char_t* ucs2 = NULL;
	ucs2char_t filename[MAX_PATH];
	pmp_device_description_t decl;

	memset(&decl, 0, sizeof(decl));

	// filename = "${path_to_device}/${md->sys_filename}"
	ucs2cpy(filename, path_to_device);
	filepath_addslash(filename);
	ucs2 = mbsdupucs2(md->sys_filename);
	ucs2cat(filename, ucs2);
	ucs2free(ucs2);

	if (filepath_file_exists(filename)) {
		int match = 1;
		char line[128];
		FILE *fp = ucs2fopen(filename, "r");
		if (!fp) {
			return 0;
		}

		while (fgets(line, sizeof(line)-1, fp)) {
			char *p = strip(line);
			if (p[0] == '[' && line[strlen(p)-1] == ']') {
				p[strlen(p)-1] = 0;
				strcpy(decl.name, p+1);
			} else if (strncmp(p, "version = ", 10) == 0) {
				strcpy(decl.version, p+10);
			} else if (strncmp(p, "language = ", 11) == 0) {
				strcpy(decl.language, p+11);
			} else if (strncmp(p, "mode = ", 7) == 0) {
				strcpy(decl.mode, p+7);
			}
		}
		fclose(fp);

		match &= (strcmp(decl.mode, md->mode) == 0);
		match &= (compare_version(md->min_version, decl.version) <= 0);
		match &= (compare_version(decl.version, md->max_version) <= 0);

		if (match) {
			memcpy((pmp_device_description_t*)&ptr_info->decl, &decl, sizeof(decl));
			return 1;
		}
	}
	return 0;
}



PMPIRIVERPLUS2API result_t pmp_enumerate_devid(pmplib_enumerate_devid_callback_t callback, void *instance)
{
	const ip2model_descriptor_t* md = g_model_descriptions;
	for (;md->id;++md) {
		callback(instance, md->id);
	}
	return 0;
}

PMPIRIVERPLUS2API result_t pmp_create(pmp_t** ptr_pmp, const ucs2char_t* path_to_device, const char *id)
{
	result_t ret = 0;
	pmp_t* pmp = NULL;
	pmp_internal_t* pmpi = NULL;
	const ip2model_descriptor_t* md = NULL;
	pmp_device_information_t info;

	// Initialize device information.
	memset(&info, 0, sizeof(info));

	// Return a NULL pointer by default.
	*ptr_pmp = 0;

	// Find a suitable model for the device.
	md = g_model_descriptions;
	for (;md->id;++md) {
		if (id && *id) {
			// Match the device identifier.
			if (strcmp(md->id, id) == 0) {
				// This will fill some members in decl.
				detect_model(path_to_device, md, &info);
				set_device_info(id, path_to_device, md, &info);
				break;
			}
		} else {
			// Detect the model automatically.
			if (detect_model(path_to_device, md, &info)) {
				set_device_info(md->id, path_to_device, md, &info);
				break;
			}
		}
	}
	if (!md->id) {
		return PMPERR_DEVICENOTFOUND;
	}

	// Allocate PMP class instance.
	pmp = (pmp_t*)calloc(1, sizeof(pmp_t));
	if (!pmp) {
		return PMPERR_INSUFFICIENTMEMORY;
	}

	pmp->add_ref = pmp_add_ref;
	pmp->release = pmp_release;
	pmp->open = pmp_open;
	pmp->close = pmp_close;
	pmp->add_ref(pmp);

	// Allocate the internal variables.
	pmpi = (pmp_internal_t*)calloc(1, sizeof(pmp_internal_t));
	if (!pmpi) {
		free(pmp);
		return PMPERR_INSUFFICIENTMEMORY;
	}
	pmpi->decl = md;

	// Initialize the internal variables.
	pmp->instance = pmpi;
	memcpy((pmp_device_information_t*)&pmp->info, &info, sizeof(info));

	// Create music instance.
	ret = pmp_create_instance_music(pmp, &pmp->music);
	if (ret != 0) {
		pmp_release(pmp);
		return ret;
	}

	// Prepare
	*ptr_pmp = pmp;
	return 0;
}

static uint32_t pmp_add_ref(pmp_t* pmp)
{
	return pmplib_interlocked_increment(&pmp->ref_count);
}

static uint32_t pmp_release(pmp_t* pmp)
{
	uint32_t count = pmplib_interlocked_decrement(&pmp->ref_count);
	if (count == 0) {
		pmpmusic_release(pmp->music);
		free_device_info(&pmp->info);
		free(pmp->instance);
		free(pmp);
	}
	return count;
}

static result_t pmp_open(pmp_t* pmp, uint32_t flag)
{
	result_t ret = 0;

	// Set the open flag.
	pmp->flag = flag;
	
	// Open the music database.
	ret = pmpmusic_open(pmp->music);
	if (ret) {
		return ret;
	}

	return 0;
}

static result_t pmp_close(pmp_t* pmp)
{
	result_t ret = 0;

	// Close the music database.
	ret = pmpmusic_close(pmp->music);
	if (ret) {
		return ret;
	}

	return 0;
}

static result_t pmp_create_instance_music(pmp_t* pmp, pmp_music_t** ptr_music)
{
	pmp_music_t* music = NULL;
	pmp_internal_t* pmpi = (pmp_internal_t*)pmp->instance;
	pmp_music_internal_t* pmpmi = NULL;

	*ptr_music = 0;

	music = calloc(1, sizeof(pmp_music_t));
	if (!music) {
		return PMPERR_INSUFFICIENTMEMORY;
	}

	pmpmi = calloc(1, sizeof(pmp_music_internal_t));
	if (!pmpmi) {
		free(music);
		return PMPERR_INSUFFICIENTMEMORY;
	}
	music->set_records = pmpmusic_set_records;
	music->get_records = pmpmusic_get_records;
	music->dump = pmpmusic_dump;
	music->set_playlists = pmpmusic_set_playlists;
	music->pmp = pmp;
	music->instance = pmpmi;

	*ptr_music = music;
	return 0;
}






static uint32_t pmpmusic_release(pmp_music_t* music)
{
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;
	pmplib_records_finish(pmpmi->records, pmpmi->num_records);
	free(pmpmi->playlists);
	free(pmpmi);
	free(music);
	return 0;
}

static uint32_t pmpmusic_open(pmp_music_t* music)
{
	result_t ret = 0;
	pmp_t* pmp = music->pmp;
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;

	// Initialize IP2DB.
	ip2db_init(&pmpmi->ip2db);

	// Free the existing records.
	pmplib_records_finish(pmpmi->records, pmpmi->num_records);
	pmpmi->records = 0;
	pmpmi->num_records = 0;

	// Open the music database if necessary.
	if (pmp->flag & PMPOF_MUSIC_DB_READ) {
		ucs2char_t sys[MAX_PATH], dat[MAX_PATH], idx[MAX_PATH];
		pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)pmp->music->instance;

		// Read the music database.
		set_filenames(sys, dat, idx, pmp);
		ret = ip2db_read(&pmpmi->ip2db, dat, idx);
		if (ret) {
			goto exit_this;
		}

		// Obtain the number of records.
		ret = ip2db_get(&pmpmi->ip2db, NULL, &pmpmi->num_records, pmp->info.path_to_root);
		if (ret) {
			goto exit_this;
		}

		// Allocate record array.
		pmpmi->records = (pmp_music_record_t*)malloc(sizeof(pmp_music_record_t) * pmpmi->num_records);
		if (!pmpmi->records) {
			ret = PMPERR_INSUFFICIENTMEMORY;
			goto exit_this;
		}

		ret = ip2db_get(&pmpmi->ip2db, pmpmi->records, &pmpmi->num_records, pmp->info.path_to_root);
		if (ret) {
			goto exit_this;
		}
	}

exit_this:
	// Finish the IP2DB.
	ip2db_finish(&pmpmi->ip2db);
	return ret;
}

static uint32_t pmpmusic_close(pmp_music_t* music)
{
	result_t ret = 0;
	pmp_t* pmp = music->pmp;
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;
	static const ucs2char_t plp_ext[] = {'.','p','l','p',0};

	// Initialize IP2DB.
	ip2db_init(&pmpmi->ip2db);

	if (pmp->flag & PMPOF_MUSIC_DB_WRITE) {
		ucs2char_t sys[MAX_PATH], dat[MAX_PATH], idx[MAX_PATH];

		ip2db_set(&pmpmi->ip2db, pmpmi->records, pmpmi->num_records, pmp->info.path_to_root);

		set_filenames(sys, dat, idx, pmp);
		ret = ip2db_write(&pmpmi->ip2db, dat, idx);
		if (ret) {
			goto exit_this;
		}
	}

	if (pmp->flag & PMPOF_MUSIC_PL_WRITE) {
		int i;

		for (i = 0;i < pmpmi->num_playlists;++i) {
			const pmp_playlist_t* pl = &pmpmi->playlists[i];
			if (ip2db_playlist_write(
				&pmpmi->ip2db,
				pl->name,
				pl->entries,
				pl->num_entries,
				pmp->info.path_to_root,
				pmp->info.path_to_playlist,
				plp_ext
				) != 0) {
				ret = PMPERR_WRITE;
				goto exit_this;
			}
		}
	}

exit_this:
	// Finish the IP2DB.
	ip2db_finish(&pmpmi->ip2db);
	return ret;
}

static result_t pmpmusic_set_records(pmp_music_t* music, const pmp_music_record_t* records, uint32_t num_records)
{
	pmp_t* pmp = music->pmp;
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;

	/* Free records attached to pmpmi. */
	pmplib_records_finish(pmpmi->records, pmpmi->num_records);

	/* Allocate new records. */
	pmpmi->records = (pmp_music_record_t*)ucs2malloc(sizeof(pmp_music_record_t) * num_records);
	pmpmi->num_records = num_records;
	pmplib_records_clone(pmpmi->records, records, num_records);

	return 0;
 }

static result_t pmpmusic_get_records(pmp_music_t* music, pmp_music_record_t* records, uint32_t* num_records)
{
	pmp_t* pmp = music->pmp;
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;

	if (!records) {
		*num_records = pmpmi->num_records;
		return 0;
	} else if (*num_records == pmpmi->num_records) {
		pmplib_records_clone(records, pmpmi->records, pmpmi->num_records);
		return 0;
	} else {
		return PMPERR_INSUFFICIENTMEMORY;
	}
}

static result_t pmpmusic_dump(pmp_music_t* music, FILE *fp, int level)
{
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;
	if (level > 0) {
		return ip2db_repr(&pmpmi->ip2db, fp);
	} else {
		return ip2db_dump(&pmpmi->ip2db, fp);
	}
}

static result_t pmpmusic_set_playlists(pmp_music_t* music, const pmp_playlist_t* playlists, uint32_t num_playlists)
{
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;

	// Free playlists attached to pmpmi.
	pmplib_playlists_finish(pmpmi->playlists, pmpmi->num_playlists);

	// Allocate a new playlists.
	pmpmi->playlists = (pmp_playlist_t*)calloc(num_playlists, sizeof(pmp_playlist_t));
	pmpmi->num_playlists = num_playlists;

	// Copy the playlist array.
	pmplib_playlists_clone(pmpmi->playlists, playlists, num_playlists);

	return 0;
}
