/*
 *      Reader/writer for iRivNavi.iDB.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: irivnavi.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/pmp.h>

#include "serialize.h"
#include "irivnavi.h"

void record_init(record_t* record)
{
	memset(record, 0, sizeof(*record));
}

void record_finish(record_t* record)
{
	ucs2free(record->filename);
	ucs2free(record->title);
	ucs2free(record->artist);
	ucs2free(record->album);
	ucs2free(record->genre);
	record_init(record);
}

size_t record_size(const record_t* record)
{
	size_t size = 0;
	size += strlen(record->filename)+1;
	size += sizeof(uint16_t);
	size += strlen(record->title)+1;
	size += sizeof(uint16_t);
	size += strlen(record->artist)+1;
	size += sizeof(uint16_t);
	size += strlen(record->album)+1;
	size += sizeof(uint16_t);
	size += strlen(record->genre)+1;
	size += sizeof(uint16_t);
#ifdef	IRIVNAVI_USES_TIMESTAMP
	size += sizeof(uint32_t);	// timestamp.
#endif
	return size;
}

size_t record_serialize(record_t* record, uint8_t* buffer, int is_storing)
{
	uint8_t *p = buffer;
	uint16_t filename_size = 0;
	uint16_t title_size = 0;
	uint16_t artist_size = 0;
	uint16_t album_size = 0;
	uint16_t genre_size = 0;

	if (is_storing) {
		filename_size = (uint16_t)strlen(record->filename)+1;
		title_size = (uint16_t)strlen(record->title)+1;
		artist_size = (uint16_t)strlen(record->artist)+1;
		album_size = (uint16_t)strlen(record->album)+1;
		genre_size = (uint16_t)strlen(record->genre)+1;
	}

	p += serialize_uint16le(p, &filename_size, is_storing);
	p += serialize_uint16le(p, &title_size, is_storing);
	p += serialize_uint16le(p, &artist_size, is_storing);
	p += serialize_uint16le(p, &album_size, is_storing);
	p += serialize_uint16le(p, &genre_size, is_storing);

	if (!is_storing) {
		record_finish(record);
		record->filename = ucs2malloc(filename_size);
		record->title = ucs2malloc(title_size);
		record->artist = ucs2malloc(artist_size);
		record->album = ucs2malloc(album_size);
		record->genre = ucs2malloc(genre_size);
	}

	p += serialize_uint8_array(p, (uint8_t*)record->filename, filename_size, is_storing);
	p += serialize_uint8_array(p, (uint8_t*)record->title, title_size, is_storing);
	p += serialize_uint8_array(p, (uint8_t*)record->artist, artist_size, is_storing);
	p += serialize_uint8_array(p, (uint8_t*)record->album, album_size, is_storing);
	p += serialize_uint8_array(p, (uint8_t*)record->genre, genre_size, is_storing);
#ifdef	IRIVNAVI_USES_TIMESTAMP
	p += serialize_uint32le(p, &record->timestamp, is_storing);
#endif

	return (size_t)(p - buffer);
}

void record_dump(record_t* record, FILE *fp)
{
	fprintf(fp, "  filename: %s\n", record->filename);
	fprintf(fp, "  title: %s\n", record->title);
	fprintf(fp, "  artist: %s\n", record->artist);
	fprintf(fp, "  album: %s\n", record->album);
	fprintf(fp, "  genre: %s\n", record->genre);
#ifdef	IRIVNAVI_USES_TIMESTAMP
	fprintf(fp, "  timestamp: %d\n", record->timestamp);
#endif
}

void irivnavi_init(irivnavi_t* db)
{
	memset(db, 0, sizeof(*db));
	strcpy((char *)db->header0, "iRivDB Ver 0.12");
	strcpy((char *)db->header1, "iRiver iHP-100 DB File");
	strcpy((char *)db->footer, "Designed by iRiver");
}

void irivnavi_finish(irivnavi_t* db)
{
	uint32_t i;

	for (i = 0;i < db->num_records;++i) {
		record_finish(&db->records[i]);
	}
	free(db->records);
	free(db->offsets);
	irivnavi_init(db);
}

int irivnavi_init_records(irivnavi_t* db, uint32_t num_records)
{
	uint32_t i;

	free(db->offsets);
	db->offsets = malloc(sizeof(uint32_t) * num_records);
	if (!db->offsets) {
		return 1;
	}
	free(db->records);
	db->records = malloc(sizeof(record_t) * num_records);
	if (!db->records) {
		return 1;
	}
	for (i = 0;i < num_records;++i) {
		record_init(&db->records[i]);
	}
	db->num_records = num_records;

	return 0;
}

void irivnavi_update(irivnavi_t* db)
{
	uint32_t i;
	size_t offset = 0;

	// Just to calculate record offsets and database size.
	for (i = 0;i < db->num_records;++i) {
		db->offsets[i] = (uint32_t)offset;
		offset += record_size(&db->records[i]);
	}
	db->size = (uint32_t)(0xA0C0 + offset);
}

size_t irivnavi_serialize(irivnavi_t* db, uint8_t* buffer, int is_storing)
{
	uint32_t i;
	uint8_t *p = NULL;

	// Header and number of records.
	p = buffer;
	p += serialize_uint8_array(p, db->header0, sizeof(db->header0), is_storing);
	p += serialize_uint8_array(p, db->header1, sizeof(db->header1), is_storing);
	p += serialize_uint32le(p, &db->num_records, is_storing);

	// Allocate memory block for offsets and records when reading.
	if (!is_storing) {
		int ret = irivnavi_init_records(db, db->num_records);
		if (ret != 0) {
			irivnavi_finish(db);
			return 0;
		}
	}
	
	// Offsets.
	p = &buffer[0x0080];
	for (i = 0;i < db->num_records;++i) {
		p += serialize_uint32le(p, &db->offsets[i], is_storing);
	}

	// Footer.
	p = &buffer[0xA080];
	p += serialize_uint8_array(p, db->footer, sizeof(db->footer), is_storing);

	// Actual records.
	p = &buffer[0xA0C0];
	for (i = 0;i < db->num_records;++i) {
		p += record_serialize(&db->records[i], &buffer[0x0000A0C0 + db->offsets[i]], is_storing);
	}

	return (p-buffer);
}

void irivnavi_repr(irivnavi_t* db, FILE *fp, int level)
{
	uint32_t i;

	fprintf(fp, "// Header\n");
	fprintf(fp, "header0: %s\n", db->header0);
	fprintf(fp, "header1: %s\n", db->header1);
	fprintf(fp, "num_records: %d\n", db->num_records);
	fprintf(fp, "footer: %s\n", db->footer);
	fprintf(fp, "\n");

	if (level > 0) {
		fprintf(fp, "// Offsets\n");
		for (i = 0;i < db->num_records;++i) {
			fprintf(fp, "offsets[%d]: %d\n", i, db->offsets[i]);
		}
		fprintf(fp, "\n");
	}

	fprintf(fp, "// Records\n");
	for (i = 0;i < db->num_records;++i) {
		fprintf(fp, "RECORD %d = [\n", i);
		record_dump(&db->records[i], fp);
		fprintf(fp, "]\n");
	}
	fprintf(fp, "\n");
}
