/*
 *      Low level interface for dat (e.g., H10DB.dat, DB5000.DAT, ...).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: dat.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#include "serialize.h"
#include "util.h"
#include "pp1db.h"

int dat_init(dat_t* record, hdr_t* hdr)
{
	uint32_t i;

	// Zero clear.
	memset(record, 0, sizeof(*record));

	// Allocate a field array.
	record->fields = calloc(hdr->num_dat_fields, sizeof(dat_field_t));
	if (!record->fields) {
		return -1;
	}

	// Initialize field types as well.
	for (i = 0;i < hdr->num_dat_fields;++i) {
		record->fields[i].type = hdr->fd[i].field_type;
	}

	return 0;
}

void dat_finish(dat_t* record, hdr_t* hdr)
{
	uint32_t i;

	// Free the contents of fields.
	for (i = 0;i < hdr->num_dat_fields;++i) {
		if (record->fields[i].type == PP1DB_FIELDTYPE_STRING) {
			ucs2free(record->fields[i].value.str);
		}
	}
	// Free the field array.
	free(record->fields);
	memset(record, 0, sizeof(*record));
}

void dat_free_array(dat_t* records, hdr_t* hdr)
{
	uint32_t i;
	for (i = 0;i < hdr->num_dat_entries;++i) {
		dat_finish(&records[i], hdr);
	}
	free(records);
}

size_t dat_serialize(uint8_t* block, dat_t* record, uint32_t num_fields, uint16_t* offsets, int is_storing)
{
	uint32_t i = 0, ret = 0;

	serialize_uint32le(&block[0], &record->status, is_storing);
	serialize_uint32le(&block[4], &record->unknown1, is_storing);

	if (is_storing) {
		offsets[0] = 8;
		for (i = 0;i < num_fields;++i) {
			uint8_t *p = &block[offsets[i]];
			size_t field_length = 0, field_size = 0;

			switch (record->fields[i].type) {
			case PP1DB_FIELDTYPE_STRING:
				field_length = ucs2len(record->fields[i].value.str)+1;
				field_size = serialize_ucs2le_string_fixed(p, record->fields[i].value.str, field_length, is_storing) * sizeof(ucs2char_t);
				break;
			case PP1DB_FIELDTYPE_INTEGER:
				field_size = serialize_uint32le(p, &record->fields[i].value.dword, is_storing);
				break;
			}
			offsets[i+1] = offsets[i] + (uint16_t)field_size;
		}
	} else {
		for (i = 0;i < num_fields;++i) {
			uint8_t *p = &block[offsets[i]];
			uint16_t field_length = offsets[i+1] - offsets[i];

			switch (record->fields[i].type) {
			case PP1DB_FIELDTYPE_STRING:
				ucs2free(record->fields[i].value.str);
				record->fields[i].value.str = ucs2calloc(sizeof(ucs2char_t) * (field_length+1));
				if (!record->fields[i].value.str) {
					return 0;
				}
				serialize_ucs2le_string_fixed(p, record->fields[i].value.str, field_length, is_storing);
				break;
			case PP1DB_FIELDTYPE_INTEGER:
				field_length /= sizeof(uint32_t);
				serialize_uint32le(p, &record->fields[i].value.dword, is_storing);
				break;
			}
		}
	}
	return offsets[num_fields];
}

void dat_round(dat_t* record, hdr_t* hdr)
{
	uint32_t i;
	for (i = 0;i < hdr->num_dat_fields;++i) {
		if (record->fields[i].type == PP1DB_FIELDTYPE_STRING) {
			ucs2char_t* value = record->fields[i].value.str;
			uint32_t length = hdr->fd[i].max_length-1;
			ucs2strip(value);
			if (length < ucs2len(value)) {
				value[length] = 0;
				ucs2strip(value);
			}
		}
	}
}
