/*
 *      Media database reader/writer for PortalPlayer platform.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: pp1db.h 328 2007-02-10 17:50:11Z nyaochi $ */

#ifndef	__PP1DB_H__
#define	__PP1DB_H__

#include <pmplib/pmp.h>

#define	PP1DB_PATHLENGTH		0x100		/**< Maximum length of path/file name. */

#define	FIELDOFFSETS(ptr, i, n)	((ptr) + ((n)+1) * (i))

enum {
	PP1DB_FIELDTYPE_NONE = 0,	
	PP1DB_FIELDTYPE_STRING,
	PP1DB_FIELDTYPE_INTEGER,
};

typedef struct {
	uint32_t	type;
	union {
		uint32_t	dword;
		ucs2char_t*	str;
	} value;
} dat_field_t;

typedef struct {
	uint32_t	status;			/**< 0 if this element is active, otherwise 1. */
	uint32_t	unknown1;		/**< 0 (Reserved) */
	dat_field_t	*fields;
} dat_t;

typedef struct {
	uint32_t status;			/**< 0 if the element is active, otherwise 1. */
	uint32_t index;				/**< index number pointing to the entry in dat. */
	uint32_t check_value;		/**< check value for the relevant field in dat entry. */
} idx_t;

typedef struct {
	uint32_t	id;				/**< Field identifier. */
	uint32_t	field_type;		/**< 1: UCS2-LE String; 2: uint32_t. */
	uint32_t	max_length;		/**< Maximum length of the field */
	uint32_t	unknown4;
	uint32_t	unknown5;
	uint32_t	has_index;		/**< 1 if this field needs an index (*.idx), otherwise 0. */
	uint32_t	unknown6;
	uint32_t	unknown7;
	ucs2char_t	index_pathname[PP1DB_PATHLENGTH];	/**< Pathname of the index file */
} field_descriptor_t;

typedef void (*dat_repr_t)(const dat_t* record, FILE *fp);
typedef int (*dat_set_t)(dat_t* dst, const pmp_music_record_t* src, const ucs2char_t* path_to_root);
typedef int (*dat_get_t)(pmp_music_record_t* dst, const dat_t* src, const ucs2char_t* path_to_root);

typedef struct {
	size_t		size;
	uint32_t	max_fields;
	uint32_t	max_entries;
	uint32_t	padding_size;
	uint32_t	has_unknown5;

	dat_repr_t	proc_dat_repr;
	dat_set_t	proc_dat_set;
	dat_get_t	proc_dat_get;
} hdr_parameter_t;

typedef struct {
	uint32_t	unknown1;							/**< 0 AFAIK */
	uint32_t	unknown2;							/**< 0 AFAIK */
	ucs2char_t	pathname_dat[PP1DB_PATHLENGTH];		/**< Pathname of H10DB.dat file */
	uint32_t	unknown3;							/**< 1 AFAIK */
	ucs2char_t	pathname_hdr[PP1DB_PATHLENGTH];		/**< Pathname of H10DB.hdr file */
	uint32_t	unknown4;							/**< Version of the database specification? (0x0000042C) */
	uint32_t	num_dat_entries;					/**< Number of entries in H10DB.DAT */
	uint32_t	num_dat_inactive_entries;			/**< Number of inactive entries in H10DB.DAT */
	uint32_t	num_dat_fields;						/**< H10DB_NUM_DAT_FIELDS (22) AFAIK */
	field_descriptor_t	*fd;	/**< Field descriptors. */
													/**< 1032 bytes padding. */
	uint32_t	*max_dat_field_size;
	uint32_t	dat_size;							/**< Size of H10DB.dat file */
	uint32_t	unknown5;							/**< 1 AFAIK (This field exists only in 20GB MTP 2.50) */
	uint16_t	*dat_field_offset;					/* [H10DB_MAX_ENTRY] */
	uint32_t	*dat_record_offset;					/* [H10DB_MAX_ENTRY+1] */

	/* These fields are not serizlied in H10DB.hdr */
	hdr_parameter_t		param;
} hdr_t;

typedef struct {
	hdr_t*	hdr;
	dat_t*	dat;
	idx_t*	*idx;
} pp1db_t;

typedef struct {
	char	id[128];
	char	name[128];
	char	mode[128];
	char	language[128];
	char	version[128];
} pp1model_t;

typedef int (*hdr_init_t)(hdr_t* hdr);
typedef int (*parse_model_t)(const ucs2char_t* firmware, pp1model_t* model);

// hdr.c
void hdr_finish(hdr_t* hdr);
size_t hdr_serialize(uint8_t* block, hdr_t* hdr, int is_storing);
void hdr_repr(const hdr_t* hdr, FILE *fp);

// field_descriptor.c
void field_descriptor_init(field_descriptor_t* fd);
void field_descriptor_finish(field_descriptor_t* fd);
size_t field_descriptor_serialize(uint8_t* block, field_descriptor_t* fd, int is_storing);
void field_descriptor_repr(const field_descriptor_t* fd, FILE *fp);
uint32_t field_descriptor_get_max_record_size(const hdr_t* hdr);

// dat.c
int dat_init(dat_t* record, hdr_t* hdr);
void dat_finish(dat_t* record, hdr_t* hdr);
void dat_free_array(dat_t* records, hdr_t* hdr);
size_t dat_serialize(uint8_t* block, dat_t* record, uint32_t num_fields, uint16_t* offsets, int is_storing);
void dat_round(dat_t* record, hdr_t* hdr);

// idx.c
void idx_init(idx_t* elem);
size_t idx_serialize(uint8_t* block, idx_t* elem, int is_storing);
void idx_dump(const idx_t* elem, FILE *fp);
void idx_repr(const idx_t* elem, FILE *fp);

// playlist.c
int pp1db_playlist_write(
	pp1db_t* db,
	const ucs2char_t *name,
	ucs2char_t* const mediafiles[],
	int num_mediafiles,
	const ucs2char_t *path_to_root,
	const ucs2char_t *path_to_playlist,
	const ucs2char_t *ext
	);

// pp1db.c
void pp1db_init(pp1db_t* db, hdr_init_t hdrinit);
void pp1db_finish(pp1db_t* db);
result_t pp1db_read(pp1db_t* db, const ucs2char_t* hdr_filename);
result_t pp1db_write(pp1db_t* db, const ucs2char_t* hdr_filename);
result_t pp1db_repr(pp1db_t* db, FILE *fp, int level);
int pp1db_is_supported_ext(const ucs2char_t* filename);
int pp1db_is_supported_codec(uint32_t codec);
result_t pp1db_set(pp1db_t* db, const pmp_music_record_t* records, uint32_t num_records, ucs2char_t* path_to_root);
result_t pp1db_get(pp1db_t* db, pmp_music_record_t* records, uint32_t* num_records, ucs2char_t* path_to_root);


#endif/*__PP1DB_H__*/
