/*
 * gst-filter-manager.h - Header for GstFilterManager
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef __GST_FILTER_MANAGER_H__
#define __GST_FILTER_MANAGER_H__

#include <gst/gst.h>
#include <gst/filters/gst-filter.h>


G_BEGIN_DECLS
#define GST_TYPE_FILTER_MANAGER                 \
  (gst_filter_manager_get_type())
#define GST_FILTER_MANAGER(obj)                 \
  (G_TYPE_CHECK_INSTANCE_CAST ((obj),           \
      GST_TYPE_FILTER_MANAGER,                  \
      GstFilterManager))
#define GST_IS_FILTER_MANAGER(obj)              \
  (G_TYPE_CHECK_INSTANCE_TYPE ((obj),           \
      GST_TYPE_FILTER_MANAGER))
#define GST_FILTER_MANAGER_GET_IFACE(inst)      \
  (G_TYPE_INSTANCE_GET_INTERFACE ((inst),       \
      GST_TYPE_FILTER_MANAGER,                  \
      GstFilterManagerInterface))

/**
 * GstFilterId:
 *
 * An opaque structure to represent a single filter application in the filter
 * manager.
 * A filter can be applied multiple times in a filter manager, so this unique
 * #GstFilterId represents the filter in a specific position in the filter
 * manager.
 * It becomes invalid as soon as the filter gets removed from the manager.
 */
typedef struct _GstFilterId GstFilterId;

/**
 * GstFilterManager:
 *
 * Opaque structure representing an object implementing the
 * #GstFilterManagerInterface
 */
typedef struct _GstFilterManager GstFilterManager;
typedef struct _GstFilterManagerInterface GstFilterManagerInterface;

/**
 * GstFilterManagerInterface:
 * @parent: parent interface type.
 * @list_filters: List the #GstFilterId in the filter manager
 * @insert_filter_before: Insert the #GstFilter before the #GstFilterId
 * @insert_filter_after:  Insert the #GstFilter after the #GstFilterId
 * @replace_filter: Remove the #GstFilterId and replace it with the #GstFilter
 * @insert_filter: Insert the #GstFilter at the specified position
 * @remove_filter: Remove the #fsuFilterId from the filter manager
 * @get_filter_by_id: Return the #GstFilter associated with the #GstFilterId
 * @apply: Apply the filter manager on the #GstPad on the #GstBin
 * @revert: Revert the filter manager from the #GstPad on the #GstBin
 * @handle_message: Handle the #GstMessage by all the filters
 *
 * This is the interface for the #GstFilterManager that needs to be implemented
 */
struct _GstFilterManagerInterface
{
  GTypeInterface parent;

  GList *(*list_filters) (GstFilterManager * self);
  GstFilterId *(*insert_filter_before) (GstFilterManager * self,
      GstFilter * filter, GstFilterId * before);
  GstFilterId *(*insert_filter_after) (GstFilterManager * self,
      GstFilter * filter, GstFilterId * after);
  GstFilterId *(*replace_filter) (GstFilterManager * self,
      GstFilter * filter, GstFilterId * replace);
  GstFilterId *(*insert_filter) (GstFilterManager * self,
      GstFilter * filter, gint position);
    gboolean (*remove_filter) (GstFilterManager * self, GstFilterId * id);
  GstFilter *(*get_filter_by_id) (GstFilterManager * self, GstFilterId * id);
  GstPad *(*apply) (GstFilterManager * self, GstBin * bin, GstPad * pad);
  GstPad *(*revert) (GstFilterManager * self, GstBin * bin, GstPad * pad);
    gboolean (*handle_message) (GstFilterManager * self, GstMessage * message);
};


GType
gst_filter_manager_get_type (void)
    G_GNUC_CONST;

GList *gst_filter_manager_list_filters (GstFilterManager * self);
GstFilterId *gst_filter_manager_prepend_filter (GstFilterManager * self,
    GstFilter * filter);
GstFilterId *gst_filter_manager_append_filter (GstFilterManager * self,
    GstFilter * filter);
GstFilterId *gst_filter_manager_insert_filter_before (GstFilterManager *
    self, GstFilter * filter, GstFilterId * before);
GstFilterId *gst_filter_manager_insert_filter_after (GstFilterManager *
    self, GstFilter * filter, GstFilterId * after);
GstFilterId *gst_filter_manager_replace_filter (GstFilterManager * self,
    GstFilter * filter, GstFilterId * replace);
GstFilterId *gst_filter_manager_insert_filter (GstFilterManager * self,
    GstFilter * filter, gint position);
gboolean gst_filter_manager_remove_filter (GstFilterManager * self,
    GstFilterId * id);
GstFilter *gst_filter_manager_get_filter_by_id (GstFilterManager * self,
    GstFilterId * id);

GstPad *gst_filter_manager_apply (GstFilterManager * self,
    GstBin * bin, GstPad * pad);
GstPad *gst_filter_manager_revert (GstFilterManager * self,
    GstBin * bin, GstPad * pad);

gboolean gst_filter_manager_handle_message (GstFilterManager * self,
    GstMessage * message);

G_END_DECLS
#endif /* __GST_FILTER_MANAGER_H__ */
