/*
 * gst-filter.c - Source for GstFilter
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * SECTION:gst-filter
 * @short_description: Filters to apply on a pipeline
 *
 * This base class represents filters that can be applied on a pipeline or that
 * can be inserted in a #GstFilterManager. Each filter does a specific task
 * and provides an easy to use API or properties/signals to control your
 * pipeline.
 * It also provides some helper functions to more easily add/revert elements
 * or bins to the pipeline.
 *
 * See also: #GstFilterManager
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#include <gst/filters/gst-filter-helper.h>
#include "gstfilters-marshal.h"

G_DEFINE_TYPE (GstFilter, gst_filter, G_TYPE_OBJECT);

static void gst_filter_dispose (GObject * object);
static void gst_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec);

/* signals  */
enum
{
  SIGNAL_APPLIED,
  SIGNAL_APPLY_FAILED,
  SIGNAL_REVERTED,
  SIGNAL_REVERT_FAILED,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };



/* properties */
enum
{
  PROP_NAME = 1,
  LAST_PROPERTY
};

struct _GstFilterPrivate
{
  GHashTable *pads;
};

static void
gst_filter_class_init (GstFilterClass * klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (GstFilterPrivate));

  gobject_class->get_property = gst_filter_get_property;
  gobject_class->dispose = gst_filter_dispose;

  g_object_class_install_property (gobject_class, PROP_NAME,
      g_param_spec_string ("name", "The name of the filter",
          "The name of the filter",
          NULL, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  /**
   * GstFilter::applied:
   * @filter: The #GstFilter
   * @bin: The #GstBin the filter got applied on
   * @pad: The #GstPad the filter got applied on
   * @out_pad: The resulting output pad
   *
   * This signal is sent when the filter gets applied on a pad.
   */
  signals[SIGNAL_APPLIED] = g_signal_new ("applied",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
      0,
      NULL, NULL,
      _gst_filters_marshal_VOID__OBJECT_OBJECT_OBJECT,
      G_TYPE_NONE, 3, GST_TYPE_BIN, GST_TYPE_PAD, GST_TYPE_PAD);

  /**
   * GstFilter::apply-failed:
   * @filter: The #GstFilter
   * @bin: The #GstBin the filter got applied on
   * @pad: The #GstPad the filter got applied on
   *
   * This signal is sent when the filter fails to apply on a pad.
   */
  signals[SIGNAL_APPLY_FAILED] = g_signal_new ("apply-failed",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
      0,
      NULL, NULL,
      _gst_filters_marshal_VOID__OBJECT_OBJECT,
      G_TYPE_NONE, 2, GST_TYPE_BIN, GST_TYPE_PAD);

  /**
   * GstFilter::reverted:
   * @filter: The #GstFilter
   * @bin: The #GstBin the filter got reverted from
   * @pad: The #GstPad the filter got reverted from
   * @out_pad: The resulting output pad
   *
   * This signal is sent when the filter gets reverted from a pad.
   */
  signals[SIGNAL_REVERTED] = g_signal_new ("reverted",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
      0,
      NULL, NULL,
      _gst_filters_marshal_VOID__OBJECT_OBJECT_OBJECT,
      G_TYPE_NONE, 3, GST_TYPE_BIN, GST_TYPE_PAD, GST_TYPE_PAD);

  /**
   * GstFilter::revert-failed:
   * @filter: The #GstFilter
   * @bin: The #GstBin the filter got reverted from
   * @pad: The #GstPad the filter got reverted from
   *
   * This signal is sent when the filter fails to revert from a pad.
   */
  signals[SIGNAL_REVERT_FAILED] = g_signal_new ("revert-failed",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
      0,
      NULL, NULL,
      _gst_filters_marshal_VOID__OBJECT_OBJECT,
      G_TYPE_NONE, 2, GST_TYPE_BIN, GST_TYPE_PAD);
}

static void
gst_filter_init (GstFilter * self)
{
  GstFilterPrivate *priv = G_TYPE_INSTANCE_GET_PRIVATE (self, GST_TYPE_FILTER,
      GstFilterPrivate);

  self->priv = priv;
  priv->pads = g_hash_table_new_full (NULL, NULL,
      gst_object_unref, gst_object_unref);
  self->lock = g_mutex_new ();
}


static void
gst_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec)
{
  GstFilter *self = GST_FILTER (object);
  GstFilterClass *klass = GST_FILTER_GET_CLASS (self);

  switch (property_id) {
    case PROP_NAME:
      g_value_set_string (value, klass->name);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
gst_filter_dispose (GObject * object)
{
  GstFilter *self = GST_FILTER (object);
  GstFilterPrivate *priv = self->priv;

  if (priv->pads)
    g_hash_table_destroy (priv->pads);
  priv->pads = NULL;

  if (self->lock)
    g_mutex_free (self->lock);
  self->lock = NULL;

  G_OBJECT_CLASS (gst_filter_parent_class)->dispose (object);
}


/**
 * gst_filter_apply:
 * @self: The #GstFilter
 * @bin: The #GstBin to apply the filter to
 * @pad: The #GstPad to apply the filter to
 *
 * This will apply the filter to a bin on a specific pad.
 * This will make the filter add elements to the @bin and link them with @pad.
 * It will return an output pad at the end of the inserted elements.
 * The filter can be applied on either a source pad or a sink pad, it
 * will still work the same.
 *
 * Returns: (transfer full): The new applied #GstPad to link with the rest
 * of the pipeline or #NULL in case of an error.
 * See also: gst_filter_revert()
 */
GstPad *
gst_filter_apply (GstFilter * self, GstBin * bin, GstPad * pad)
{
  GstFilterClass *klass = GST_FILTER_GET_CLASS (self);
  GstFilterPrivate *priv = self->priv;
  GstPad *out_pad = NULL;

  g_assert (klass->apply);

  GST_FILTER_LOCK (self);
  out_pad = klass->apply (self, bin, pad);
  GST_FILTER_UNLOCK (self);

  if (out_pad) {
    gst_object_ref (out_pad);
    GST_FILTER_LOCK (self);
    g_hash_table_insert (priv->pads, out_pad, gst_pad_get_peer (pad));
    GST_FILTER_UNLOCK (self);
    g_signal_emit (self, signals[SIGNAL_APPLIED], 0, bin, pad, out_pad);
  } else {
    g_signal_emit (self, signals[SIGNAL_APPLY_FAILED], 0, bin, pad);
  }

  return out_pad;
}

/**
 * gst_filter_revert:
 * @self: The #GstFilter
 * @bin: The #GstBin to revert the filter from
 * @pad: The #GstPad to revert the filter from
 *
 * This will revert the filter from a bin on the specified pad.
 * This should remove any elements the filter previously added to the bin and
 * return the original pad it received in the gst_filter_apply()
 * The pad to revert should be the output pad returned from gst_filter_apply().
 * Note that the returned pad may not be the same as the input pad of
 * gst_filter_apply() because if the pad before the filter's application
 * point was unlinked and relinked somewhere else, that might change (as in the
 * case of adding/removing a filter from a #GstFilterManager in which this
 * filter resides)
 *
 * Returns: (transfer full): The original #GstPad from which the filter
 * manager was applied
 * See also: gst_filter_manager_apply()
 */
GstPad *
gst_filter_revert (GstFilter * self, GstBin * bin, GstPad * pad)
{
  GstFilterClass *klass = GST_FILTER_GET_CLASS (self);
  GstFilterPrivate *priv = self->priv;
  GstPad *expected = NULL;
  GstPad *in_pad = NULL;
  GstPad *out_pad = NULL;

  g_assert (klass->revert);

  GST_FILTER_LOCK (self);
  in_pad = g_hash_table_lookup (priv->pads, pad);

  if (!in_pad) {
    GST_FILTER_UNLOCK (self);
    g_signal_emit (self, signals[SIGNAL_REVERT_FAILED], 0, bin, pad);
    return NULL;
  }
  expected = gst_pad_get_peer (in_pad);
  g_hash_table_remove (priv->pads, pad);

  out_pad = klass->revert (self, bin, pad);
  GST_FILTER_UNLOCK (self);

  if (!out_pad)
    g_signal_emit (self, signals[SIGNAL_REVERT_FAILED], 0, bin, pad);
  else
    g_signal_emit (self, signals[SIGNAL_REVERTED], 0, bin, pad, out_pad);

  if (out_pad != expected) {
    g_warning ("Reverted pad on filter %s (%p) not as expected",
        klass->name, self);
    if (out_pad)
      gst_object_unref (out_pad);
    out_pad = expected;
  } else {
    gst_object_unref (expected);
  }

  return out_pad;
}


/**
 * gst_filter_follow:
 * @self: The #GstFilter
 * @pad: The #GstPad to follow the filter from
 *
 * This will not do anything but it will give you the expected output result
 * from gst_filter_revert().
 * It basically will follow the filter from it's output pad all the way to the
 * other side of the elements that it added and will give you the input pad that
 * it should return if gst_filter_revert() was called but without modifying
 * the pipeline.
 *
 * Returns: (transfer full): The original #GstPad from which the
 * filter manager was applied
 * See also: gst_filter_manager_revert()
 */
GstPad *
gst_filter_follow (GstFilter * self, GstPad * pad)
{
  GstFilterPrivate *priv = self->priv;
  GstPad *in_pad = NULL;
  GstPad *out_pad = NULL;

  GST_FILTER_LOCK (self);
  in_pad = g_hash_table_lookup (priv->pads, pad);
  if (in_pad)
    out_pad = gst_pad_get_peer (in_pad);
  GST_FILTER_UNLOCK (self);

  return out_pad;
}

/**
 * gst_filter_handle_message:
 * @self: The #GstFilter
 * @message: The message to handle
 *
 * Try to handle a message originally received on the #GstBus to the filter.
 *
 * Returns: #TRUE if the message has been handled and should be dropped,
 * #FALSE otherwise.
 */
gboolean
gst_filter_handle_message (GstFilter * self, GstMessage * message)
{
  GstFilterClass *klass = GST_FILTER_GET_CLASS (self);
  gboolean ret = FALSE;

  if (klass->handle_message) {
    GST_FILTER_LOCK (self);
    ret = klass->handle_message (self, message);
    GST_FILTER_UNLOCK (self);
  }

  return ret;
}
