/*
 * gst-framerate-filter.c - Source for GstFramerateFilter
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * SECTION:gst-framerate-filter
 * @short_description: A video framerate filter
 *
 * This filter will force the output video framerate to be constant.
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#include <gst/filters/gst-framerate-filter.h>
#include <gst/filters/gst-filter-helper.h>

G_DEFINE_TYPE (GstFramerateFilter, gst_framerate_filter, GST_TYPE_FILTER);

static void gst_framerate_filter_constructed (GObject * object);
static void gst_framerate_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec);
static void gst_framerate_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec);
static void gst_framerate_filter_dispose (GObject * object);
static GstPad *gst_framerate_filter_apply (GstFilter * filter,
    GstBin * bin, GstPad * pad);
static GstPad *gst_framerate_filter_revert (GstFilter * filter,
    GstBin * bin, GstPad * pad);

#define DEFAULT_FRAMERATE 30

/* properties */
enum
{
  PROP_FPS = 1,
  LAST_PROPERTY
};


struct _GstFramerateFilterPrivate
{
  GList *elements;
  GstCaps *caps;
  guint fps;
};

static void
gst_framerate_filter_class_init (GstFramerateFilterClass * klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GstFilterClass *gstfilter_class = GST_FILTER_CLASS (klass);

  g_type_class_add_private (klass, sizeof (GstFramerateFilterPrivate));

  gobject_class->constructed = gst_framerate_filter_constructed;
  gobject_class->get_property = gst_framerate_filter_get_property;
  gobject_class->set_property = gst_framerate_filter_set_property;
  gobject_class->dispose = gst_framerate_filter_dispose;

  gstfilter_class->apply = gst_framerate_filter_apply;
  gstfilter_class->revert = gst_framerate_filter_revert;
  gstfilter_class->name = "framerate";

  /**
   * GstFramerateFilter:fps:
   *
   * The maximum framerate allowed
   */
  g_object_class_install_property (gobject_class, PROP_FPS,
      g_param_spec_uint ("fps", "Frames per second",
          "The framerate per second to set",
          0, G_MAXINT,
          DEFAULT_FRAMERATE, G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

}

static void
gst_framerate_filter_init (GstFramerateFilter * self)
{
  GstFramerateFilterPrivate *priv =
      G_TYPE_INSTANCE_GET_PRIVATE (self, GST_TYPE_FRAMERATE_FILTER,
      GstFramerateFilterPrivate);

  self->priv = priv;
  priv->fps = DEFAULT_FRAMERATE;
}


static void
gst_framerate_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec)
{
  GstFramerateFilter *self = GST_FRAMERATE_FILTER (object);
  GstFramerateFilterPrivate *priv = self->priv;


  switch (property_id) {
    case PROP_FPS:
      g_value_set_uint (value, priv->fps);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
gst_framerate_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec)
{
  GstFramerateFilter *self = GST_FRAMERATE_FILTER (object);
  GstFramerateFilterPrivate *priv = self->priv;

  switch (property_id) {
    case PROP_FPS:
    {
      GList *i;

      priv->fps = g_value_get_uint (value);

      if (priv->caps)
        gst_caps_unref (priv->caps);

      priv->caps = gst_caps_new_full (gst_structure_new ("video/x-raw-yuv",
              "framerate", GST_TYPE_FRACTION, priv->fps, 1,
              NULL),
          gst_structure_new ("video/x-raw-rgb",
              "framerate", GST_TYPE_FRACTION, priv->fps, 1,
              NULL),
          gst_structure_new ("video/x-raw-gray",
              "framerate", GST_TYPE_FRACTION, priv->fps, 1, NULL), NULL);

      GST_FILTER_LOCK (self);
      for (i = priv->elements; i; i = i->next) {
        GstElement *element = i->data;
        g_object_set (element, "caps", priv->caps, NULL);
      }
      GST_FILTER_UNLOCK (self);
    }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
gst_framerate_filter_dispose (GObject * object)
{
  GstFramerateFilter *self = GST_FRAMERATE_FILTER (object);
  GstFramerateFilterPrivate *priv = self->priv;
  GList *i;

  if (priv->caps)
    gst_caps_unref (priv->caps);

  for (i = priv->elements; i; i = i->next)
    gst_object_unref (i->data);
  g_list_free (priv->elements);
  priv->elements = NULL;

  G_OBJECT_CLASS (gst_framerate_filter_parent_class)->dispose (object);
}


static void
gst_framerate_filter_constructed (GObject * object)
{
  GstFramerateFilter *self = GST_FRAMERATE_FILTER (object);
  GstFramerateFilterPrivate *priv = self->priv;

  if (G_OBJECT_CLASS (gst_framerate_filter_parent_class)->constructed)
    G_OBJECT_CLASS (gst_framerate_filter_parent_class)->constructed (object);

  priv->caps = gst_caps_new_full (gst_structure_new ("video/x-raw-yuv",
          "framerate", GST_TYPE_FRACTION, priv->fps, 1,
          NULL),
      gst_structure_new ("video/x-raw-rgb",
          "framerate", GST_TYPE_FRACTION, priv->fps, 1,
          NULL),
      gst_structure_new ("video/x-raw-gray",
          "framerate", GST_TYPE_FRACTION, priv->fps, 1, NULL), NULL);
}

/**
 * gst_framerate_filter_new:
 * @fps: The maximum FPS allowed
 *
 * Creates a new framerate filter.
 * This filter will make sure that the stream does not output more frames than
 * the specified FPS. It will not generate duplicate frames, so this filter is
 * mainly to be used in a streaming pipeline.
 * It will basically add 'videomaxrate ! capsfilter' to the pipeline.
 *
 * Returns: (transfer full): A new #GstFramerateFilter
 */
GstFramerateFilter *
gst_framerate_filter_new (guint fps)
{
  return g_object_new (GST_TYPE_FRAMERATE_FILTER, "fps", fps, NULL);
}


static GstPad *
gst_framerate_filter_apply (GstFilter * filter, GstBin * bin, GstPad * pad)
{

  GstFramerateFilter *self = GST_FRAMERATE_FILTER (filter);
  GstFramerateFilterPrivate *priv = self->priv;
  GstElement *capsfilter = NULL;
  GstElement *filter_bin = NULL;
  GstPad *out_pad = NULL;

  filter_bin = gst_filter_add_element_by_description (bin, pad,
      "videorate ! capsfilter name=capsfilter", &out_pad);

  if (filter_bin) {
    capsfilter = gst_bin_get_by_name (GST_BIN (filter_bin), "capsfilter");

    if (capsfilter) {
      priv->elements = g_list_prepend (priv->elements, capsfilter);

      g_object_set (capsfilter, "caps", priv->caps, NULL);
    }

    gst_object_unref (filter_bin);
  }

  return out_pad;
}

static GstPad *
gst_framerate_filter_revert (GstFilter * filter, GstBin * bin, GstPad * pad)
{
  GstFramerateFilter *self = GST_FRAMERATE_FILTER (filter);
  GstFramerateFilterPrivate *priv = self->priv;
  GstElement *filter_bin = GST_ELEMENT (gst_pad_get_parent (pad));
  GstElement *capsfilter = NULL;
  GstPad *ret = NULL;

  capsfilter = gst_bin_get_by_name (GST_BIN (filter_bin), "capsfilter");
  if (g_list_find (priv->elements, capsfilter)) {
    priv->elements = g_list_remove (priv->elements, capsfilter);
    gst_object_unref (capsfilter);
  }
  if (capsfilter)
    gst_object_unref (capsfilter);

  ret = gst_filter_revert_element_default (bin, pad, NULL);
  gst_object_unref (filter_bin);

  return ret;
}
