/*
 * gst-gnome_effect-filter.c - Source for GstGnomeEffectFilter
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
 * SECTION:gst-gnome-effect-filter
 * @short_description: A Gnome Video Effect filter
 *
 * This filter will parse a Gnome Video Effect file and will generate the proper
 * pipeline for that effect, as well as give access to some of the effect's
 * configuration.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gst/filters/gst-gnome-effect-filter.h>
#include <gst/filters/gst-filter-helper.h>


G_DEFINE_TYPE (GstGnomeEffectFilter, gst_gnome_effect_filter, GST_TYPE_FILTER);

static void gst_gnome_effect_filter_constructed (GObject * object);
static void gst_gnome_effect_filter_dispose (GObject * object);
static void gst_gnome_effect_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec);
static void gst_gnome_effect_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec);
static GstPad *gst_gnome_effect_filter_apply (GstFilter * filter,
    GstBin * bin, GstPad * pad);
static GstPad *gst_gnome_effect_filter_revert (GstFilter * filter,
    GstBin * bin, GstPad * pad);


/* properties */
enum
{
  PROP_FILENAME = 1,
  PROP_NAME,
  PROP_PIPELINE_DESCRIPTION,
  PROP_CATEGORIES,
  LAST_PROPERTY
};



struct _GstGnomeEffectFilterPrivate
{
  gchar *filename;
  gchar *name;
  gchar *pipeline_description;
  GList *categories;
  GError *error;
};

static void
gst_gnome_effect_filter_class_init (GstGnomeEffectFilterClass * klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GstFilterClass *gstfilter_class = GST_FILTER_CLASS (klass);

  g_type_class_add_private (klass, sizeof (GstGnomeEffectFilterPrivate));

  gobject_class->constructed = gst_gnome_effect_filter_constructed;
  gobject_class->get_property = gst_gnome_effect_filter_get_property;
  gobject_class->set_property = gst_gnome_effect_filter_set_property;
  gobject_class->dispose = gst_gnome_effect_filter_dispose;

  gstfilter_class->apply = gst_gnome_effect_filter_apply;
  gstfilter_class->revert = gst_gnome_effect_filter_revert;
  gstfilter_class->name = "gnome-effect";

  /**
   * GstGnomeEffectFilter:filename:
   *
   * The filename to the Gnome Video Effect .effect file to parse
   */
  g_object_class_install_property (gobject_class, PROP_FILENAME,
      g_param_spec_string ("filename", "The filename of the Gnome effect",
          "The filename of the Gnome effect to read and build",
          NULL,
          G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  /**
   * GstGnomeEffectFilter:name:
   *
   * The user-friendly name to the Gnome Video Effect
   */
  g_object_class_install_property (gobject_class, PROP_NAME,
      g_param_spec_string ("name", "The name of the effect",
          "The name of the Gnome effect taken from the file",
          NULL, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  /**
   * GstGnomeEffectFilter:pipeline-description:
   *
   * The pipeline description of the Gnome Video Effect as taken from
   * the .effect file
   */
  g_object_class_install_property (gobject_class, PROP_PIPELINE_DESCRIPTION,
      g_param_spec_string ("pipeline-description",
          "The pipeline description of the effect",
          "The pipeline description of the Gnome effect taken from the file",
          NULL, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  /**
   * GstGnomeEffectFilter:categories:
   *
   * The list of categories as specified in the .effect file.
   * This returns a #GList of strings, each representing a category for this
   * Gnome Video Effect.
   */
  g_object_class_install_property (gobject_class, PROP_CATEGORIES,
      g_param_spec_boxed ("categories",
          "The categories of the effect",
          "The categories of the Gnome effect taken from the file",
          GST_TYPE_GNOME_EFFECT_CATEGORIES_LIST,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

}

static void
gst_gnome_effect_filter_init (GstGnomeEffectFilter * self)
{
  GstGnomeEffectFilterPrivate *priv =
      G_TYPE_INSTANCE_GET_PRIVATE (self, GST_TYPE_GNOME_EFFECT_FILTER,
      GstGnomeEffectFilterPrivate);

  self->priv = priv;
}

static void
gst_gnome_effect_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec)
{
  GstGnomeEffectFilter *self = GST_GNOME_EFFECT_FILTER (object);
  GstGnomeEffectFilterPrivate *priv = self->priv;

  switch (property_id) {
    case PROP_FILENAME:
      g_value_set_string (value, priv->filename);
      break;
    case PROP_NAME:
      g_value_set_string (value, priv->name);
      break;
    case PROP_PIPELINE_DESCRIPTION:
      g_value_set_string (value, priv->pipeline_description);
      break;
    case PROP_CATEGORIES:
      g_value_set_boxed (value, priv->categories);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
gst_gnome_effect_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec)
{
  GstGnomeEffectFilter *self = GST_GNOME_EFFECT_FILTER (object);
  GstGnomeEffectFilterPrivate *priv = self->priv;

  switch (property_id) {
    case PROP_FILENAME:
      priv->filename = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
gst_gnome_effect_filter_constructed (GObject * object)
{
  GstGnomeEffectFilter *self = GST_GNOME_EFFECT_FILTER (object);
  GstGnomeEffectFilterPrivate *priv = self->priv;
  GKeyFile *key_file = g_key_file_new ();
  gchar **categories = NULL;

  if (!g_key_file_load_from_file (key_file, priv->filename,
          G_KEY_FILE_NONE, &priv->error))
    goto done;

  priv->name = g_key_file_get_locale_string (key_file, "Effect",
      "Name", NULL, &priv->error);
  if (priv->error)
    goto done;

  priv->pipeline_description = g_key_file_get_string (key_file, "Effect",
      "PipelineDescription", &priv->error);
  if (priv->error)
    goto done;

  categories = g_key_file_get_string_list (key_file, "Effect",
      "Category", NULL, NULL);
  if (categories) {
    gchar **ptr = categories;
    while (*ptr) {
      priv->categories = g_list_append (priv->categories, g_strdup (*ptr));
      ptr++;
    }
    g_strfreev (categories);
  }

done:
  g_key_file_free (key_file);
  return;
}

static void
gst_gnome_effect_filter_dispose (GObject * object)
{
  GstGnomeEffectFilter *self = GST_GNOME_EFFECT_FILTER (object);
  GstGnomeEffectFilterPrivate *priv = self->priv;

  g_free (priv->filename);
  priv->filename = NULL;
  g_free (priv->name);
  priv->name = NULL;
  g_free (priv->pipeline_description);
  priv->pipeline_description = NULL;
  g_list_foreach (priv->categories, (GFunc) g_free, NULL);
  g_list_free (priv->categories);
  priv->categories = NULL;


  G_OBJECT_CLASS (gst_gnome_effect_filter_parent_class)->dispose (object);
}


/**
 * gst_gnome_effect_filter_new:
 * @filename: The filename to the .effect file representing the
 * Gnome Video Effect
 * @error: A pointer to a #GError to be set if the .effect file is invalid
 *
 * Creates a new Gnome Video Effect filter
 * This filter will add the video effect pipeline description with an
 * ffmpegcolorspace to the pipeline when applied.
 *
 * Returns: (transfer full): A new #GstGnomeEffectFilter or #NULL if an
 * error occured
 */
GstGnomeEffectFilter *
gst_gnome_effect_filter_new (const gchar * filename, GError ** error)
{
  GstGnomeEffectFilter *self = NULL;

  g_return_val_if_fail (filename, NULL);
  g_return_val_if_fail (g_file_test (filename,
          G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR), NULL);

  self = g_object_new (GST_TYPE_GNOME_EFFECT_FILTER,
      "filename", filename, NULL);

  if (self->priv->error) {
    g_propagate_error (error, self->priv->error);
    g_object_unref (self);
    return NULL;
  }

  return self;
}


static GstPad *
gst_gnome_effect_filter_apply (GstFilter * filter, GstBin * bin, GstPad * pad)
{
  GstGnomeEffectFilter *self = GST_GNOME_EFFECT_FILTER (filter);
  GstGnomeEffectFilterPrivate *priv = self->priv;
  GstPad *out_pad = NULL;
  gchar *description = NULL;

  description = g_strdup_printf ("ffmpegcolorspace ! %s",
      priv->pipeline_description);

  out_pad = gst_filter_apply_element_by_description (bin, pad,
      description, NULL, NULL);

  g_free (description);


  return out_pad;
}

static GstPad *
gst_gnome_effect_filter_revert (GstFilter * filter, GstBin * bin, GstPad * pad)
{
  return gst_filter_revert_element_default (bin, pad, NULL);
}

/**
 * gst_gnome_effect_categories_list_destroy:
 * @categories_list: The #GList of categories to free
 *
 * Frees and destroys the #GList of categories returned by the property
 * #GstGnomeEffectFilter:categories
 */
void
gst_gnome_effect_categories_list_destroy (GList * categories_list)
{
  g_list_foreach (categories_list, (GFunc) g_free, NULL);
  g_list_free (categories_list);
}

/**
 * gst_gnome_effect_categories_list_copy:
 * @categories_list: (element-type utf8): The #GList of categories to copy
 *
 * Creates a #GList copy of the categories list
 *
 * Returns: (transfer full) (element-type utf8): The copy of the categories list
 */
GList *
gst_gnome_effect_categories_list_copy (const GList * categories_list)
{
  GList *copy = NULL;
  const GList *lp;

  for (lp = categories_list; lp; lp = g_list_next (lp)) {
    /* prepend then reverse the list for efficiency */
    copy = g_list_prepend (copy, g_strdup (lp->data));
  }
  copy = g_list_reverse (copy);
  return copy;
}

GType
gst_gnome_effect_categories_list_get_type (void)
{
  static GType categories_list_type = 0;
  if (categories_list_type == 0) {
    categories_list_type =
        g_boxed_type_register_static ("GstGnomeEffectCategoriesGList",
        (GBoxedCopyFunc) gst_gnome_effect_categories_list_copy,
        (GBoxedFreeFunc) gst_gnome_effect_categories_list_destroy);
  }

  return categories_list_type;
}


/**
 * gst_gnome_effect_list_effects:
 * @directory: The directory to search for effects
 *
 * Searches in a directory for all .effect files and returns a #GList of strings
 * with the full path to the .effect files in it.
 *
 * Returns: (transfer full) (element-type utf8): A #GList of the filenames of the
 * .effect files in the @directory
 */
GList *
gst_gnome_effect_list_effects (const gchar * directory)
{
  GDir *dir = NULL;
  GList *effects = NULL;

  g_return_val_if_fail (directory, NULL);
  g_return_val_if_fail (g_file_test (directory,
          G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR), NULL);

  dir = g_dir_open (directory, 0, NULL);

  if (dir) {
    const gchar *name = g_dir_read_name (dir);

    while (name) {
      if (g_str_has_suffix (name, ".effect"))
        effects = g_list_append (effects,
            g_strdup_printf ("%s/%s", directory, name));
      name = g_dir_read_name (dir);
    }
    g_dir_close (dir);
  }
  return effects;
}


/**
 * gst_gnome_effect_effects_list_destroy:
 * @effects_list: (element-type utf8): The #GList of effects to free
 *
 * Frees and destroys the #GList of effects returned by
 * gst_gnome_effect_list_effects()
 */
void
gst_gnome_effect_effects_list_destroy (GList * effects_list)
{
  g_list_foreach (effects_list, (GFunc) g_free, NULL);
  g_list_free (effects_list);
}
