/*
 * gst-preview-filter.c - Source for GstPreviewFilter
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
 * SECTION:gst-preview-filter
 * @short_description: A video preview filter
 *
 * This filter will create a video output window to preview the current data
 * flowing in the pipeline.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#include <gst/filters/gst-preview-filter.h>
#include <gst/filters/gst-filter-helper.h>
#include <gst/filters/gst-single-filter-manager.h>
#include <gst/interfaces/xoverlay.h>


G_DEFINE_TYPE (GstPreviewFilter, gst_preview_filter, GST_TYPE_FILTER);

static void gst_preview_filter_dispose (GObject * object);
static void gst_preview_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec);
static void gst_preview_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec);
static GstPad *gst_preview_filter_apply (GstFilter * filter,
    GstBin * bin, GstPad * pad);
static GstPad *gst_preview_filter_revert (GstFilter * filter,
    GstBin * bin, GstPad * pad);
static gboolean gst_preview_filter_handle_message (GstFilter * self,
    GstMessage * message);


/* properties */
enum
{
  PROP_WINDOW_HANDLE = 1,
  PROP_FILTER_MANAGER,
  LAST_PROPERTY
};



struct _GstPreviewFilterPrivate
{
  guintptr window_handle;
  GstElement *sink;
  GstPad *sink_pad;
  GstFilterManager *manager;
};

static void
gst_preview_filter_class_init (GstPreviewFilterClass * klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GstFilterClass *gstfilter_class = GST_FILTER_CLASS (klass);

  g_type_class_add_private (klass, sizeof (GstPreviewFilterPrivate));

  gobject_class->get_property = gst_preview_filter_get_property;
  gobject_class->set_property = gst_preview_filter_set_property;
  gobject_class->dispose = gst_preview_filter_dispose;

  gstfilter_class->apply = gst_preview_filter_apply;
  gstfilter_class->revert = gst_preview_filter_revert;
  gstfilter_class->handle_message = gst_preview_filter_handle_message;
  gstfilter_class->name = "preview";

  /**
   * GstPreviewFilter:window-handle:
   *
   * The X window id to use for embedding the preview window
   */
  g_object_class_install_property (gobject_class, PROP_WINDOW_HANDLE,
      g_param_spec_uint64 ("window-handle",
          "The window handle for the preview",
          "The window handle (HWND or Xid) to use for embedding the "
          "preview window",
          0, G_MAXUINT64, 0,
          G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  /**
   * GstPreviewFilter:filter-manager:
   *
   * The filter manager to apply on the preview window.
   * This preview filter has a filter manager of its own which allows you to
   * add custom filters specific to a preview window. You can access it through
   * this property and modify the filter manager as you see fit.
   */
  g_object_class_install_property (gobject_class, PROP_FILTER_MANAGER,
      g_param_spec_object ("filter-manager", "Filter manager",
          "The filter manager to apply on the preview window",
          GST_TYPE_FILTER_MANAGER, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

}

static void
gst_preview_filter_init (GstPreviewFilter * self)
{
  GstPreviewFilterPrivate *priv =
      G_TYPE_INSTANCE_GET_PRIVATE (self, GST_TYPE_PREVIEW_FILTER,
      GstPreviewFilterPrivate);

  self->priv = priv;
  priv->manager = gst_single_filter_manager_new ();
}

static void
gst_preview_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec)
{
  GstPreviewFilter *self = GST_PREVIEW_FILTER (object);
  GstPreviewFilterPrivate *priv = self->priv;


  switch (property_id) {
    case PROP_WINDOW_HANDLE:
      g_value_set_uint64 (value, priv->window_handle);
      break;
    case PROP_FILTER_MANAGER:
      g_value_set_object (value, priv->manager);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
gst_preview_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec)
{
  GstPreviewFilter *self = GST_PREVIEW_FILTER (object);
  GstPreviewFilterPrivate *priv = self->priv;

  switch (property_id) {
    case PROP_WINDOW_HANDLE:
      priv->window_handle = (guintptr) g_value_get_uint64 (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
}

static void
gst_preview_filter_dispose (GObject * object)
{
  GstPreviewFilter *self = GST_PREVIEW_FILTER (object);
  GstPreviewFilterPrivate *priv = self->priv;

  if (priv->sink_pad)
    gst_object_unref (priv->sink_pad);
  priv->sink_pad = NULL;
  if (priv->sink)
    gst_object_unref (priv->sink);
  priv->sink = NULL;

  if (priv->manager)
    g_object_unref (priv->manager);
  priv->manager = NULL;

  G_OBJECT_CLASS (gst_preview_filter_parent_class)->dispose (object);
}


/**
 * gst_preview_filter_new:
 * @window_handle: The handle of the window in which to embed the video output
 *
 * Creates a new video preview filter.
 * This filter allows you to preview your video stream into a preview window by
 * creating a tee and linking it with an videosink to which it sets the
 * specified @window_handle value
 *
 * Returns: A new #GstPreviewFilter
 */
GstPreviewFilter *
gst_preview_filter_new (guintptr window_handle)
{
  return g_object_new (GST_TYPE_PREVIEW_FILTER,
      "window-handle", (guint64) window_handle, NULL);
}

static void
sink_element_added (GstBin * bin, GstElement * sink, gpointer user_data)
{
  g_object_set (sink, "async", FALSE, "sync", FALSE, NULL);
}


static GstPad *
gst_preview_filter_apply (GstFilter * filter, GstBin * bin, GstPad * pad)
{
  GstPreviewFilter *self = GST_PREVIEW_FILTER (filter);
  GstPreviewFilterPrivate *priv = self->priv;
  GstElement *tee = NULL;
  GstElement *clrspace = NULL;
  GstElement *sink = NULL;
  GstPad *out_pad = NULL;
  GstPad *tee_pad = NULL;
  GstPad *preview_pad = NULL;
  GstPad *filter_pad = NULL;
  GstPad *clrspace_srcpad = NULL;
  GstPad *clrspace_sinkpad = NULL;
  GstPad *sink_pad = NULL;

  if (priv->sink)
    return NULL;

  tee = gst_element_factory_make ("tee", NULL);
  sink = gst_element_factory_make ("autovideosink", NULL);
  clrspace = gst_element_factory_make ("ffmpegcolorspace", NULL);

  if (!tee || !sink || !clrspace) {
    goto error;
  }

  if (GST_PAD_IS_SRC (pad)) {
    tee_pad = gst_element_get_static_pad (tee, "sink");
    out_pad = gst_element_get_request_pad (tee, "src%d");
    if (out_pad)
      g_object_set (tee, "alloc-pad", out_pad, NULL);
  } else {
    out_pad = gst_element_get_static_pad (tee, "sink");
    tee_pad = gst_element_get_request_pad (tee, "src%d");
    if (tee_pad)
      g_object_set (tee, "alloc-pad", tee_pad, NULL);
  }
  preview_pad = gst_element_get_request_pad (tee, "src%d");
  sink_pad = gst_element_get_static_pad (sink, "sink");
  clrspace_srcpad = gst_element_get_static_pad (clrspace, "src");
  clrspace_sinkpad = gst_element_get_static_pad (clrspace, "sink");

  g_signal_connect_object (sink, "element-added",
      G_CALLBACK (sink_element_added), self, 0);

  if (!tee_pad || !out_pad || !preview_pad || !sink_pad ||
      !clrspace_srcpad || !clrspace_sinkpad)
    goto error;

  if (!gst_filter_add_element (bin, pad, tee, tee_pad))
    goto error;

  filter_pad = gst_filter_manager_apply (priv->manager, bin, preview_pad);

  if (!filter_pad)
    filter_pad = gst_object_ref (preview_pad);
  gst_object_unref (preview_pad);

  if (!gst_filter_add_element (bin, filter_pad, clrspace, clrspace_sinkpad)) {
    gst_bin_remove (bin, tee);
    tee = NULL;
    goto error;
  }

  if (!gst_filter_add_element (bin, clrspace_srcpad, sink, sink_pad)) {
    gst_bin_remove (bin, tee);
    tee = NULL;
    gst_bin_remove (bin, clrspace);
    clrspace = NULL;
    goto error;
  }

  gst_object_unref (tee_pad);
  gst_object_unref (filter_pad);
  priv->sink = gst_object_ref (sink);
  priv->sink_pad = sink_pad;

  return out_pad;

error:
  if (GST_PAD_IS_SRC (pad) && out_pad)
    gst_element_release_request_pad (tee, out_pad);
  if (GST_PAD_IS_SINK (pad) && tee_pad)
    gst_element_release_request_pad (tee, tee_pad);
  if (tee_pad)
    gst_object_unref (tee_pad);
  if (out_pad)
    gst_object_unref (out_pad);
  if (sink_pad)
    gst_object_unref (sink_pad);
  if (clrspace_sinkpad)
    gst_object_unref (clrspace_sinkpad);
  if (clrspace_srcpad)
    gst_object_unref (clrspace_srcpad);
  if (preview_pad) {
    gst_element_release_request_pad (tee, preview_pad);
    gst_object_unref (preview_pad);
  }

  if (tee)
    gst_object_unref (tee);
  if (sink)
    gst_object_unref (sink);
  if (clrspace)
    gst_object_unref (clrspace);

  return NULL;
}

static GstPad *
gst_preview_filter_revert (GstFilter * filter, GstBin * bin, GstPad * pad)
{
  GstPreviewFilter *self = GST_PREVIEW_FILTER (filter);
  GstPreviewFilterPrivate *priv = self->priv;
  GstElement *tee = GST_ELEMENT (gst_pad_get_parent (pad));
  GstElement *clrspace = NULL;
  GstPad *tee_pad = NULL;
  GstPad *clrspace_pad = NULL;
  GstPad *filter_pad = NULL;
  GstPad *out_pad = NULL;

  if (GST_PAD_IS_SRC (pad)) {
    GstPad *other_pad = NULL;
    other_pad = gst_element_get_static_pad (tee, "sink");
    out_pad = gst_pad_get_peer (other_pad);
    gst_object_unref (other_pad);
    tee_pad = gst_object_ref (pad);
  } else {
    g_object_get (tee, "alloc-pad", &tee_pad, NULL);
    out_pad = gst_pad_get_peer (tee_pad);
  }

  gst_element_release_request_pad (tee, tee_pad);
  gst_object_unref (tee_pad);
  clrspace_pad = gst_pad_get_peer (priv->sink_pad);
  clrspace = GST_ELEMENT (gst_pad_get_parent (clrspace_pad));
  gst_object_unref (clrspace_pad);
  clrspace_pad = gst_element_get_static_pad (clrspace, "sink");
  filter_pad = gst_pad_get_peer (clrspace_pad);
  tee_pad = gst_filter_manager_revert (priv->manager, bin, filter_pad);
  if (!tee_pad)
    tee_pad = gst_object_ref (filter_pad);
  gst_object_unref (filter_pad);
  gst_element_release_request_pad (tee, tee_pad);
  gst_object_unref (tee_pad);
  gst_bin_remove (bin, tee);
  gst_element_set_state (tee, GST_STATE_NULL);
  gst_object_unref (tee);

  gst_object_unref (clrspace_pad);
  gst_bin_remove (bin, clrspace);
  gst_object_unref (priv->sink_pad);
  priv->sink_pad = NULL;
  gst_bin_remove (bin, priv->sink);
  gst_element_set_state (priv->sink, GST_STATE_NULL);
  gst_object_unref (priv->sink);
  priv->sink = NULL;

  return out_pad;
}


static gboolean
gst_preview_filter_handle_message (GstFilter * filter, GstMessage * message)
{
  GstPreviewFilter *self = GST_PREVIEW_FILTER (filter);
  GstPreviewFilterPrivate *priv = self->priv;
  const GstStructure *s = gst_message_get_structure (message);

  if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_ELEMENT &&
      (GST_MESSAGE_SRC (message) == GST_OBJECT (priv->sink) ||
          gst_object_has_ancestor (GST_MESSAGE_SRC (message),
              GST_OBJECT (priv->sink))) &&
      gst_structure_has_name (s, "prepare-xwindow-id")) {
    gst_x_overlay_set_window_handle (GST_X_OVERLAY (GST_MESSAGE_SRC (message)),
        priv->window_handle);
    return TRUE;
  }
  return FALSE;
}
