/*
 * Edscott Wilson Garcia Copyright 2012
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include "rodent.h"
#include "fuse-group_options.h"
#include "fuse-common.h"

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "ftp"
//Not applicable:
//#define SUBMODULE_NAME "fuse"
#define MODULE_LABEL _("FTP servers")
#define MODULE_ICON_ID "xffm/emblem_atom/compositeC/emblem_ftp"
//#define MODULE_ICON_ID "rodent-ftp"
#define MODULE_ENTRY_TIP _("A graphical FTP client")
#define MODULE_PLUGIN_INFO _("FTP Client")
#define PARENT_MODULE_NAME "fuse"
#define MODULE_PREFERENCES_KEY "RODENT-FTP"

#define FTP_OPTIONS _("FTP Options")
#include "module-skeleton.h"
// Skeleton definitions

G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY(MODULE_PREFERENCES_KEY)
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(FALSE)
G_MODULE_EXPORT RFM_PLUGIN_INFO(MODULE_PLUGIN_INFO)
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)
G_MODULE_EXPORT RFM_MODULE_RELOAD(TRUE)
G_MODULE_EXPORT RFM_MODULE_SKIPWAIT(TRUE)
G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

// superceded
// G_MODULE_EXPORT RFM_ITEM_ICON_ID
// G_MODULE_EXPORT RFM_G_MODULE_CHECK_INIT
// g_module_check_init is now located at fuse-common.c

#define FTP_INFO1 _("FTP Configurations")
#define FTP_INFO2 _("When FTP connections are passive the client connects to the server, instead of the other way round, so firewalls do not block the connection; old FTP servers may not support Passive FTP though.")
#define FTP_PASSIVE  _("Passive file transfers")
#define FTP_USE_PROXY  _("FTP Proxy")
#define FTP_PROXY_HOST  _("FTP proxy host name")
#define FTP_PROXY_PORT  _("FTP proxy port")
#define FTP_PROXY_USER  _("Proxy Username:")
#define FTP_AUTHORIZATION  _("FTP Configurations")
#define FTP_TIP _("FTP servers")

#include "fuse-options.i"    
#include "ftp-options.i"

//#include "ftp-submodule.i"

#include "ftp-dialog.i"


// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT
xfdir_t *
module_xfdir_get (void *p) {
    void *argv[] = {
	p,
	(void *)"curlftpfs",
	(void *)"ftp://",
	(void *)"FUSE_MOUNT_POINT",
	(void *)MODULE_NAME,
	(void *)FTP_OPTIONS,
	NULL
    };
    return FUSE_xfdir(argv);
}

// this is a repeat...
G_MODULE_EXPORT
const gchar *
item_icon_id (void *p){
    void *argv[] = {
	p,
	(void *)MODULE_LABEL,
	(void *)module_icon_id(),
	(void *)FTP_OPTIONS,
	NULL
    };
    return FUSE_icon(argv);
}

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
double_click(void * p, void *q){
    record_entry_t *en = q;
    if (!en) return NULL;
    const gchar *url = en->pseudo_path;
    return FUSE_click((void *)confirm_ftp_host_dialog, url, en, MODULE_NAME);
}



void *
mount_url (widgets_t *widgets_p, const gchar *url){
    gchar **options_ftp = NULL;
    gchar **options_curlftpfs = NULL;
    gchar **options_fuse = NULL;
    gchar **options_module = NULL;
    gchar *mount_point = group_options_get_key_value (url, "FUSE_MOUNT_POINT");
    if (!fuse_mkdir(mount_point)){
	g_free(mount_point);
	return NULL;
    }

    gchar *computer = group_options_get_key_value (url, "FUSE_COMPUTER");
    gchar *login = group_options_get_key_value (url, "FUSE_LOGIN");

    gchar *proxy_host = group_options_get_key_value (url, "FTP_PROXY_HOST");
    gchar *proxy_port = group_options_get_key_value (url, "FTP_PROXY_PORT");
    gchar *proxy_user = group_options_get_key_value (url, "FTP_PROXY_USER");


    gboolean passive = group_options_get_key_boolean (url, "FTP_PASSIVE");
    gboolean use_proxy = group_options_get_key_boolean (url, "FTP_USE_PROXY");
    gboolean monitor = group_options_get_key_boolean (url, "FUSE_MONITOR");
    gchar *argv[MAX_COMMAND_ARGS];
    const gchar *remote = url;
    if (strncmp(remote, "ftp://", strlen("ftp://"))==0) {
	remote += strlen("ftp://");
    }
    gchar *fsname=NULL;
    
    gint i=0;
    argv[i++] = "curlftpfs";
    argv[i++] = computer;
    argv[i++] = mount_point;

    if (passive) {
 	argv[i++] = "-o";
	argv[i++] = "disable_epsv";
    }

    if (monitor){
	rfm_set_monitor_type(mount_point);
 	argv[i++] = "-o";
	fsname =  g_strdup_printf("fsname=monitor-%s",remote);
	argv[i++] = fsname;
    }


    if (login) {
	if (!strchr(login, ':')){
	    gchar *account = g_strdup_printf("<i>%s@%s</i>", 
		    login, computer);
	    gchar *text = g_strdup_printf(_("Enter your password for account\n%s"), 
		account);
	    g_free(account);
	    gchar *login_password = rfm_get_response (widgets_p, text, NULL, TRUE);
	    g_free(text);
	    if (!login_password || !strlen(login_password)){
		rfm_confirm(widgets_p, GTK_MESSAGE_ERROR,
		    _("For security reasons, you are not allowed to set an empty password."),
		    NULL, NULL);
		goto done;
	    }
	    gchar *g = g_strdup_printf("%s:%s", login, login_password);
	    g_free(login_password);
	    g_free(login);
	    login = g;
	} 
	gchar *g = g_strdup_printf("user=%s", login);
	g_free(login);
	login = g;
 	argv[i++] = "-o";
 	argv[i++] = login;
    }

    if (use_proxy) {
 	argv[i++] = "-o";
	argv[i++] = "httpproxy";
	if (proxy_host) {
	    argv[i++] = "-o";
	    gchar *g;
	    if (proxy_port){
		g = g_strdup_printf("proxy=%s:%s",proxy_host, proxy_port);
	    } else {
		g = g_strdup_printf("proxy=%s",proxy_host);
	    }
	    g_free(proxy_host);
	    proxy_host = g; 
	    argv[i++] = proxy_host;
	}
	if (proxy_user) {
	    if (!strchr(proxy_user, ':')) {
		gchar *proxy_password = 
		    rfm_get_response (widgets_p, _("Please enter proxy password"), NULL, TRUE);
		if (proxy_password && strlen(proxy_password)) {
		    gchar *g = g_strdup_printf("%s:%s", proxy_user, proxy_password);
		    g_free(proxy_user);
		    g_free(proxy_password);
		    proxy_user = g;
		}
	    } 
	    gchar *g = g_strdup_printf("proxy_user=%s",proxy_user);
	    g_free(proxy_user);
	    proxy_user = g;
	    argv[i++] = "-o";
	    argv[i++] = proxy_user;
	}
    }


    // The rest of options...

    // fuse options is FLAG_1
    gint flag_id=1;
    options_fuse = 
	group_options_get_key_options(url, flag_id, 
	    fuse_options);
    // module options is FLAG_2
    flag_id=2;
    options_module =
	group_options_get_key_options(url, flag_id, 
	    module_options);
    // ftp options is FLAG_4
    flag_id=4;
    options_ftp =
	group_options_get_key_options(url, flag_id, 
	    ftp_options);
    // curlftpfs options is FLAG_5
    flag_id=5;
    options_curlftpfs =
	group_options_get_key_options(url, flag_id, 
	    curlftpfs_options);
    gchar **o;
    for (o=options_ftp; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	argv[i++] = *o;
    }
    for (o=options_curlftpfs; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	argv[i++] = *o;
    }
    for (o=options_fuse; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	argv[i++] = *o;
    }
    for (o=options_module; o && *o && i+1 < MAX_COMMAND_ARGS; o++) {
	argv[i++] = *o;
    }


    argv[i++] = NULL;

    rfm_show_text(widgets_p);
    rfm_thread_run_argv (widgets_p, argv, FALSE);
done:
    g_free(fsname);
    g_free(login);
    g_free(computer);
    g_free(mount_point);

    g_strfreev(options_ftp);
    g_strfreev(options_curlftpfs);
    g_strfreev(options_fuse);
    g_strfreev(options_module);
    return GINT_TO_POINTER(TRUE);
}


static void
mount_host (GtkMenuItem * menuitem, gpointer user_data){
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en) return;
    widgets_t *widgets_p = rfm_get_widget("widgets_p");;
    mount_url(widgets_p,en->pseudo_path);
}

static void 
do_properties (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    const gchar *url = en->pseudo_path;
    FUSE_click((void *)confirm_ftp_host_dialog, url, NULL, MODULE_NAME);

    return ;
}


// gboolean
// This function is to generate a module specific popup menu, either on
// icon click or on void space click . 
// If this function will generate a popup menu, return value should be TRUE,
// otherwise return FALSE so Rodent will generate the default popup menu.
// Popup menu for modules should be set as view data on the paper object
// identified by "private_popup_menu".
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
     void *argv[] = {
	(void *)FTP_OPTIONS,
	(void *)do_properties,
	(void *)mount_host,
	NULL
    };
   return FUSE_popup(argv); 
}



//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT
void *
item_entry_tip(void *p){
    if (!p) return NULL;
    record_entry_t *en = p;
    if (!en->path) return NULL;
    if (strcmp(en->path, FTP_OPTIONS)==0){
	return g_strdup(FTP_OPTIONS);
    }
    if (rfm_g_file_test(en->path, G_FILE_TEST_IS_DIR)){
	gchar *text =  g_strdup_printf("\n%s\n\n%s\n",
		en->path,
		_("The mount point used for the media device connection."));
	return text;
    }
    if (en->module  && strcmp(en->module, MODULE_NAME)){
	const gchar *text = rfm_void(PLUGIN_DIR, en->module, "module_entry_tip");
	return g_strdup(text);
    } 
    return g_strdup("fixme: ftp-submodule.c");
}





