/*
 * Edscott Wilson Garcia Copyright 2012
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include "rodent.h"
#include "rfm_modules.h"
#include "fuse-group_options.h"
#include "fuse-common.h"

#define POPUP_ID "fuse_menu"
#define POPUP_MENU_ID "fuse_menu_menu"


#define FLAG_KEY(x)  g_strdup_printf("FLAG_%d", x)
#define FUSE_MENU_STUFF \
	{MENUITEM_TYPE,POPUP_MENU_ID,"fuse_new_tab",\
	    {0x2001, N_("Open in New Tab"),"xffm/stock_directory", \
		(gpointer) new_tab_open, NULL}},\
	{MENUITEM_TYPE,POPUP_MENU_ID,"fuse_new_window",\
	    {0x2002, N_("Open in New Window"),"rodent", \
		(gpointer) new_window_open, NULL}},\
	{MENUITEM_TYPE,POPUP_MENU_ID,"fuse_properties",\
	    {0x2003, N_("Properties"),"xffm/stock_properties", \
		(gpointer) variable_call, NULL}},\
	{MENUITEM_TYPE,POPUP_MENU_ID,"fuse_mount",\
	    {0x2004, N_("Mount"),"xffm/emblem_greenball", \
		(gpointer) variable_call, NULL}},\
	{MENUITEM_TYPE,POPUP_MENU_ID,"fuse_unmount",\
	    {0x2005, N_("Unmount"),"xffm/emblem_redball", \
		(gpointer) variable_call, NULL}},\
	{MENUITEM_TYPE,POPUP_MENU_ID,"fuse_broken_mount",\
	    {0x2006, N_("There was a network error."),"xffm/emblem_unreadable", \
		(gpointer) NULL, NULL}},\
        {MENUITEM_TYPE,POPUP_MENU_ID,"fuse_delete",\
	    {0x2007, N_("Delete"),"xffm/stock_delete", \
		(gpointer) remove_host, NULL}},\
	{NULL_TYPE} 


static gboolean
confirm_host (fuse_data_t *(*dialog_f)(const gchar *url), const gchar *url, const gchar *module_name);

static 
void *
submodule_up_item(void *p){
    xfdir_t *xfdir_p = p;
    xfdir_p->gl[0].en=rfm_mk_entry(0);
    xfdir_p->gl[0].en->st = NULL;
//    xfdir_p->gl[0].en->parent_module = MODULE_NAME;
    xfdir_p->gl[0].en->module = "fuse";
    gchar *g = rfm_void(PLUGIN_DIR, "fuse", "module_label");
    if (!g) g=g_strdup_printf("FIXME: no module label for \"%s\"", "fuse");
    xfdir_p->gl[0].en->path=g;
    xfdir_p->gl[0].pathv = g_strdup(g);
    SET_ROOT_TYPE(xfdir_p->gl[0].en->type);
    // Up icon qualifier (for special icon treatment)
    SET_UP_TYPE (xfdir_p->gl[0].en->type);
    // Dummy type qualifier (for paper placement at 0,0 treatment)
    SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
    
    return NULL;
}

static 
void
allocate_xfdir_p(xfdir_t *xfdir_p, gint items){
    xfdir_p->pathc=items;
    xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
    if (!xfdir_p->gl) g_error("malloc");
    memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
    return;
}

typedef struct wtf_t{
    widgets_t *widgets_p;
    gchar *application;
} wtf_t;

static 
void * wtf(void *data){
    wtf_t *wtf_p = data;
    gchar *text=g_strdup_printf (_("The \"%s\" utility is not installed.\nPlease install it."), wtf_p->application); 
// This rfm_confirm() has deadlock potential
//    rfm_confirm(NULL, GTK_MESSAGE_WARNING, text, NULL, _("Accept")); 
    rfm_threaded_show_text(wtf_p->widgets_p);
    rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm/emblem_applications",NULL);
    rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm_tag/stderr",g_strconcat(text, "\n",NULL));
#ifdef THIS_IS_BSD
    if (strstr(wtf_p->application, "sshfs")){
	rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm/stock_help",NULL);
	rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm_tag/green",
	    g_strconcat(_("For port sys-utils/fusefs-sshfs, you need BSD sources to build kernel module."),
	    "\n",NULL));
	rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm/stock_help",NULL);
	rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm_tag/green",
	    g_strconcat(_("Remember that user must be in group \"operator\" (see /dev/fuse0)."),
	    "\n",NULL));
	rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm/stock_help",NULL);
	rfm_threaded_diagnostics(wtf_p->widgets_p, "xffm_tag/green",
	    g_strconcat(_("Remember that /etc/sysctl.conf must have the line: \"vfs.usermount=1\"."),
	    "\n",NULL));
    }
#endif
    g_free(text); 
    g_free(wtf_p->application);
    g_free(data);
    return NULL;
}

// This we do in a thread to avoid confusion with a dead hearbeat.
// (Load must satisfy heartbeat condition at all times!)
void *
fuse_check_program(const gchar *program_to_check){
    if (!program_to_check) {
	DBG("check_program(): program_to_check==NULL!\n");
    }
    gchar *p = g_find_program_in_path(program_to_check);
    if (!p) {
	gchar *sbin = g_strdup_printf("/sbin/%s", program_to_check);
	if (rfm_g_file_test(sbin, G_FILE_TEST_IS_EXECUTABLE)) {
	    p = sbin;
	} else {
	    g_free(sbin);
	}
    }
    if (!p) {
	gchar *sbin = g_strdup_printf("/usr/sbin/%s", program_to_check);
	if (rfm_g_file_test(sbin, G_FILE_TEST_IS_EXECUTABLE)) {
	    p = sbin;
	} else {
	    g_free(sbin);
	}
    }
    if (!p) {
	gchar *sbin = g_strdup_printf("/usr/local/sbin/%s", program_to_check);
	if (rfm_g_file_test(sbin, G_FILE_TEST_IS_EXECUTABLE)) {
	    p = sbin;
	} else {
	    g_free(sbin);
	}
    }
    if (!p){
	widgets_t * widgets_p = rfm_get_widget("widgets_p");
	wtf_t *wtf_p = (wtf_t *)malloc(sizeof(wtf_t));
	memset(wtf_p, 0, sizeof(wtf_t));
	wtf_p->widgets_p = widgets_p;
	wtf_p->application = g_strdup(program_to_check);
	view_t *view_p = widgets_p->view_p;
	rfm_view_thread_create(view_p, wtf, wtf_p, "fuse_check_program() wtf thread\n");
	return NULL;
    }
    g_free(p);
    return GINT_TO_POINTER(TRUE);
}

static 
void *
options_dummy(void *p, const gchar *path, const gchar *module_name){
    xfdir_t *xfdir_p = p;
    xfdir_p->gl[1].en=rfm_mk_entry(0);
    xfdir_p->gl[1].en->st = NULL;
 //   xfdir_p->gl[1].en->parent_module = MODULE_NAME;
    xfdir_p->gl[1].en->module = module_name;
    xfdir_p->gl[1].en->path=g_strdup(path);
    xfdir_p->gl[1].pathv = g_strdup(path);
    SET_DUMMY_TYPE (xfdir_p->gl[1].en->type);
    return p;
}

static 
gchar **
loadcount_items (GKeyFile *key_file, const gchar *item_type, gint *items){
    gchar *file = g_build_filename(FUSE_KEY_FILE, NULL);
    gchar **groups = NULL;
    if (g_key_file_load_from_file (key_file, file, 0, NULL)){
	groups = g_key_file_get_groups (key_file, NULL);
	gchar **g_p=groups;
	for (;g_p && *g_p; g_p++) {
	    if (strncmp(*g_p, item_type, strlen(item_type))!=0) continue;
	    (*items)++;
	}
    }
    g_free(file);
    return groups;
}

static 
gboolean
mount_test(record_entry_t *en) {
    gint mounted = FSTAB_entry_is_mounted (en);
    widgets_t * widgets_p = rfm_get_widget("widgets_p");
    gchar *text = NULL;
    if (mounted==0) {
	text = g_strdup_printf("%s\n%s\n\n%s\n%s\n\n%s",
		_("Path:"), en->pseudo_path,
		_("Mount Point:"), en->path, 
		_("The volume is not mounted."));
	rfm_confirm(widgets_p, GTK_MESSAGE_ERROR, text, NULL, NULL);
    } else if (mounted<0) {
	text = g_strdup_printf("%s\n%s\n\n%s\n%s\n\n%s",
		_("Path:"), en->pseudo_path,
		_("Mount Point:"), en->path, 
		_("There was a network error."));
	rfm_confirm(widgets_p, GTK_MESSAGE_ERROR, text, NULL, NULL);
    }
    g_free(text);
    return mounted;
}


static 
void 
fuse_error(GtkWidget *dialog, const gchar *text){
    widgets_t * widgets_p = rfm_get_widget("widgets_p");
    rfm_confirm(widgets_p, GTK_MESSAGE_ERROR, text, NULL, _("Accept"));
    return;
}

static
gchar *
fuse_get_item(GtkWidget *dialog, const gchar *id){
    if (!id || ! dialog) return NULL;
    GtkWidget *entry = g_object_get_data(G_OBJECT(dialog), id);
    if (!entry) return NULL;
    const gchar *c =  gtk_entry_get_text (GTK_ENTRY(entry));
    if (!c || !strlen(c)) return NULL;
    return g_strdup(c);
}

static
void
fuse_set_key_item_f (gpointer key, gpointer value, gpointer data){
    key_options_t *key_options_p = data;
    if (GPOINTER_TO_INT(value) == -1) {
	g_key_file_set_boolean  (key_options_p->key_file, key_options_p->group, key, TRUE);
    } else if (value) {
	NOOP(stderr, "set key value, %s -- > %s\n", (gchar *)key, (gchar *)value);
	g_key_file_set_value (key_options_p->key_file,  key_options_p->group, key, value);
    }
    return;
}

static
GtkWidget *
fuse_make_entry_box(GtkWidget *dialog, 
	const gchar *text, 
	const gchar *id, 
	const gchar *colon, 
	gboolean visibility, 
	gpointer callback){
    GtkWidget *hbox = rfm_hbox_new(FALSE, 0);
    gchar *full_text = g_strconcat(text, colon, NULL);
    GtkWidget *label = gtk_label_new(full_text);
    g_free(full_text);
    GtkWidget *entry = gtk_entry_new();
    gtk_entry_set_visibility (GTK_ENTRY(entry), visibility);
    g_object_set_data(G_OBJECT(dialog), id, entry);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
    gtk_widget_show(hbox);
    gtk_widget_show(label);
    gtk_widget_show(entry);
    if (callback) {
	g_signal_connect (G_OBJECT (entry), "key-release-event", G_CALLBACK (callback), dialog);
    }
    return hbox;
}
    
static
void
fuse_set_check_button_state(GtkWidget *dialog,
	GKeyFile *key_file, 
	const gchar *group,
	const gchar *key,
	const gchar *id,
	gboolean default_state){
    GtkWidget *check = g_object_get_data(G_OBJECT(dialog), id);
    if (!check) return;
    if (strstr(id, "FTP_PASSIVE")){
	NOOP(stderr, "FTP_passive = %d\n",  
		g_key_file_get_boolean(key_file, group, key, NULL));
    }
    if (key_file){
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), 
		    g_key_file_get_boolean(key_file, group, key, NULL));
    } else {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), default_state);
    }
}

static
GtkWidget *
fuse_make_check_box(GtkWidget *dialog, 
	const gchar *text,
	const gchar *id,
	gpointer callback)
{
    GtkWidget *check;
    check = gtk_check_button_new_with_label (text);
    g_object_set_data(G_OBJECT(dialog), id, check);
    gtk_widget_show(check);
    if (callback){
	g_signal_connect (check, "toggled", G_CALLBACK (callback), dialog);
    }
    return check;
}
 
static
void
fuse_set_entry(GtkWidget *entry, 
	GKeyFile *key_file, 
	const gchar *group,
	const gchar *key, 
	const gchar *default_value){
    gchar *pre_set=NULL;
    if (key_file && group) {
	pre_set = g_key_file_get_value(key_file, group, key, NULL);
    }

    gtk_entry_set_text(GTK_ENTRY(entry),(pre_set)?pre_set:default_value); 
    g_free(pre_set); 
}

static
const gchar *
fuse_icon_id(void *p, const gchar *label,
	const gchar *module_icon_id, 
	const gchar *authorization)
{
    record_entry_t *en = (record_entry_t *)p;
    if (!en || !en->path) return "xffm/emblem_broken"; /* Error.*/ 
    // FUSE module: 
    if (en->module && strcmp(en->module, "fuse")==0) {
	// This never happens... Icon is resolved by module's item_icon_id
	return "xffm/emblem_broken";
    }

    // Authorization icons:
    if (strcmp(en->path, authorization)==0) {
	if (strcmp(en->module, "obex")==0) {
	    // Bluetooth filesystems
	    return "xffm/emblem_bluetooth/compositeSW/stock_add";
	} else if (strcmp(en->module, "ecryptfs")==0) {
	    // Encrypted filesystems
	    return "xffm/emblem_keyhole/compositeSW/stock_add";
	} else {
	    // Network filesystems
	    return "xffm/emblem_network/compositeSW/stock_add";
	}
    }

    // Module icons:
    if (!IS_UP_TYPE(en->type)) {
	if (en->path && strcmp(en->path, label)==0) {
	    return module_icon_id;
	}
    } else if (strcmp(en->module, "cifs")==0) {
	static gchar *cifs_up_icon = NULL;
	if (!cifs_up_icon) cifs_up_icon =
	    g_strconcat(module_icon_id, "/compositeSW/stock_go-up", NULL);
	// Up type icons: currently just for cifs module
	return cifs_up_icon;
    }

    // Plain item icons
    if (en->module) {
	void *mounted = rfm_natural(PLUGIN_DIR, "fstab", en->path, "is_mounted_with_wait");
	if (mounted) {

	    if (strcmp(en->module, "obex")==0){
		return "xffm/emblem_pda/compositeNW/emblem_greenball";
	    }
	    else if (strcmp(en->module, "sftp")==0){
		return "xffm/emblem_network/compositeNW/emblem_greenball";
	    }
	    else if (strcmp(en->module, "ftp")==0){
		return "xffm/emblem_network/compositeNW/emblem_greenball";
	    }
	    else if (strcmp(en->module, "cifs")==0){
		return "xffm/emblem_network/compositeNW/emblem_greenball";
	    }
	    else if (strcmp(en->module, "nfs")==0){
		return "xffm/emblem_network/compositeNW/emblem_greenball";
	    }
	    else if (strcmp(en->module, "ecryptfs")==0){
		return "xffm/emblem_network/compositeNW/emblem_greenball";
	    }
	} else {
	    if (strcmp(en->module, "obex")==0){
		return "xffm/emblem_pda/compositeNW/emblem_redball";
	    }
	    else if (strcmp(en->module, "sftp")==0){
		return "xffm/emblem_network/compositeNW/emblem_redball";
	    }
	    else if (strcmp(en->module, "ftp")==0){
		return "xffm/emblem_network/compositeNW/emblem_redball";
	    }
	    else if (strcmp(en->module, "cifs")==0){
		return "xffm/emblem_network/compositeNW/emblem_redball";
	    }
	    else if (strcmp(en->module, "nfs")==0){
		return "xffm/emblem_network/compositeNW/emblem_redball";
	    }
	    else if (strcmp(en->module, "ecryptfs")==0){
		return "xffm/emblem_network/compositeNW/emblem_redball";
	    }
	}



	return "xffm/emblem_shared/compositeSW/emblem_atom";
    }
    return "xffm/emblem_broken"; /* Error.*/ 

}

static
xfdir_t *
fuse_xfdir_get(void *p, 
	const gchar *program, 
	const gchar *scheme, 
	const gchar *mount_point,
	const gchar *module_name, 
	const gchar *authorization){
    xfdir_t *xfdir_p = p;
    NOOP(stderr, " fuse_xfdir_get...\n");
    gint items = 1;
    if (GPOINTER_TO_INT(fuse_check_program(program))) items++; 
    if (items == 1){
	    allocate_xfdir_p(xfdir_p, items);
	    submodule_up_item(xfdir_p);
	    return xfdir_p;
    }

    GKeyFile *key_file = g_key_file_new ();
    gchar **groups = loadcount_items (key_file, scheme, &items);
    allocate_xfdir_p(xfdir_p, items);
    submodule_up_item(xfdir_p);
    options_dummy(xfdir_p, authorization, module_name);
    if (items > 2) {
	gchar **g_p=groups;
	gint i=2;
	for (;g_p && *g_p; g_p++){
	    if (strncmp(*g_p, scheme, strlen(scheme))!=0) continue;
	    xfdir_p->gl[i].en=rfm_mk_entry(0);
	    xfdir_p->gl[i].en->st = (struct stat *) malloc(sizeof(struct stat));
	    memset(xfdir_p->gl[i].en->st, 0, sizeof(struct stat));
	    xfdir_p->gl[i].en->st->st_ino = 1; 

	    SET_SDIR(xfdir_p->gl[i].en->type);
	    xfdir_p->gl[i].en->path=
		g_key_file_get_value(key_file, *g_p, 
			mount_point, NULL);   
	    if (!xfdir_p->gl[i].en->path) {
		xfdir_p->gl[i].en->tag = rfm_default_url_mount_point(*g_p);
	    } 
	    xfdir_p->gl[i].en->module = module_name;
	    xfdir_p->gl[i].en->pseudo_path=g_strdup(*g_p);
	    xfdir_p->gl[i].pathv = g_strdup(*g_p);
	    i++;
	}
    }
    g_strfreev(groups);
    g_key_file_free(key_file);

    return xfdir_p;

}

void *
fuse_click(fuse_data_t * (*dialog_f)(const gchar *url), const gchar *url, record_entry_t *en, const gchar *module_name){
    widgets_t *widgets_p = rfm_get_widget("widgets_p");

    // If entry is up type, go for default handler
    if (en && IS_UP_TYPE(en->type)) return NULL;
    // If entry is NULL, we just want to pop up the properties dialog.
    // Or, if the url does not match the entry url (as in the "add item" icon)
    if (!en || !url) {
	confirm_host(dialog_f, url, module_name);
	return GINT_TO_POINTER(1);
    }

    // If entry is not NULL and item is mounted, we want to go to mount point.
    if (mount_test(en)) {
//	record_entry_t *t_en = rfm_copy_entry(en);
	record_entry_t *t_en = rfm_stat_entry(en->path, 0);
	t_en->module=NULL;
	rodent_push_view_go_history ();
	if(!rodent_refresh (widgets_p, t_en)) {
	    // cannot load
	    DBG("could not load %s\n", t_en->path);
	    rfm_destroy_entry(t_en);
	}
	return  GINT_TO_POINTER(1);	  
    }
    // Item is not mounted. Pop up the properties dialog.
    confirm_host(dialog_f, url, module_name);    
    return GINT_TO_POINTER(1);
}

/////////////////  popup callbacks....
static 
void
new_tab_open (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en) return;
    widgets_t * widgets_p = rfm_get_widget("widgets_p");


    if (!mount_test(en)) return;
    //
    view_t *view_p = widgets_p->view_p;
 
    if(view_p->tab_constructor) {
	(*(view_p->tab_constructor))(widgets_p, en->path);
    }
}


static 
void
new_window_open (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en) return;
    widgets_t * widgets_p = rfm_get_widget("widgets_p");


    if (!mount_test(en)) return;
    //
    gchar *new_path=NULL;
    new_path=g_strdup(en->path);
    rodent_new_gridview(widgets_p, new_path);
    g_free(new_path); 
    return;

}


static
void
unmount_host (GtkMenuItem * menuitem, gpointer user_data) {
//sshfs [user@]host:[dir] mountpoint
//fusermount -u mountpoint
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en) return;
    widgets_t * widgets_p = rfm_get_widget("widgets_p");

 
    gchar *argv[4];
    gint i=0;
#ifdef THIS_IS_LINUX
    argv[i++] = "fusermount";
    argv[i++] = "-u";
#else
# ifdef THIS_IS_BSD
    argv[i++] = "umount";
# else
# error "This should not happen"
# endif
#endif
    argv[i++] = en->path;
    argv[i++] = NULL;

    // FIXME: then remove mountpoint if empty (simple rmdir).

    rfm_show_text(widgets_p);
    rfm_thread_run_argv (widgets_p, argv, FALSE);  
  /*  argv[0] = "rmdir";
    argv[1] = en->tag;
    argv[2] = NULL;
    rfm_thread_run_argv (widgets_p, argv, FALSE);  */

}


static 
void
remove_host (GtkMenuItem * menuitem, gpointer user_data) {
    record_entry_t *en = g_object_get_data(G_OBJECT(menuitem), "entry");
    if (!en) return;
    widgets_t * widgets_p = rfm_get_widget("widgets_p");


    // Unmount first if mounted.
    gint mounted = FSTAB_entry_is_mounted (en);
    if (mounted > 0) {
	unmount_host (menuitem, user_data);
    } else if (mounted < 0) {
	gchar *text = g_strdup_printf("%s\n%s\n\n%s\n%s\n\n%s",
		_("Path:"), en->pseudo_path,
		_("Mount Point:"), en->path, 
		_("There was a network error."));
	rfm_confirm(widgets_p, GTK_MESSAGE_ERROR, text, NULL, NULL);
	g_free(text);
    }
    // Remove group
    group_options_remove_group(en->pseudo_path);
    // Refresh
    view_t *view_p = widgets_p->view_p;
    record_entry_t *t_en = rfm_copy_entry(view_p->en);
    if(!rodent_refresh (widgets_p, t_en)) {
	rfm_destroy_entry(t_en);
    }    
}

static void *
callback_f(void *p){
    void **arg = p;
    void ((*callback)(GtkMenuItem *, gpointer)) = arg[0];
    GtkMenuItem *menuitem = arg[1];
    void *data = arg[2];
    g_free(arg);
    (*callback)(menuitem, data);
    return NULL;
}





////////////////  end of popup callbacks....
static void
variable_call(GtkMenuItem * menuitem, gpointer data){
    void **arg = (void **)malloc(3*sizeof(void *));
    if (!arg) g_error("malloc: %s\n", strerror(errno));

    arg[0] = g_object_get_data(G_OBJECT(menuitem), "callback");
    arg[1] = menuitem;
    arg[2] = data;
    widgets_t *widgets_p = rfm_get_widget("widgets_p");
    view_t *view_p = widgets_p->view_p;
    rfm_view_thread_create(view_p, callback_f, arg, "fuse callback_f");
    return;
}

static void *
thread_popup(void *data){
    if (rfm_get_gtk_thread() == g_thread_self()){
	g_warning("thread_mk_popup_menu: only to be called from non main thread\n");
        return NULL;
    }

    GtkWidget *popup_widget=NULL;
    GMutex *popup_mutex = rfm_get_widget("fuse_popup_mutex");
    g_mutex_lock(popup_mutex);
    popup_widget = rfm_get_widget(POPUP_MENU_ID);
    if (!popup_widget) {
	popup_widget=rodent_thread_add_submenu (NULL,  "rodent-fuse", POPUP_ID,  NULL);  
	TRACE( "fuse_thread_popup creation...\n");
	RodentMenuDefinition menu_definitions_p[] = {FUSE_MENU_STUFF};
	rodent_thread_multimenu_make(NULL, menu_definitions_p); 
	widgets_t *widgets_p = rfm_get_widget ("widgets_p");
	view_t *view_p = widgets_p->view_p;
	xfdir_register_popup(view_p, popup_widget);
    } else {
	TRACE( "fuse_thread_popup retrieval...\n");
    }
    g_mutex_unlock(popup_mutex);
    return popup_widget;
}

static
void *
fuse_private_popup(const gchar *label_text,
	void (*properties_f)(GtkMenuItem *, gpointer),
	void (*mount_f)(GtkMenuItem *, gpointer),
	void (*umount_f)(GtkMenuItem *, gpointer)
	)
{
    TRACE( "fuse_private_popup...\n");
    // if umount function is not defined, then use fusermount.
    if (umount_f == NULL) umount_f = unmount_host;
    widgets_t *widgets_p = rfm_get_widget("widgets_p");
    view_t *view_p = widgets_p->view_p;
    if (!(g_slist_length(view_p->selection_list)==1)) return NULL;

    record_entry_t *en=(record_entry_t *)view_p->selection_list->data;
    if (en==NULL || !en->st){
	// No population item is found under the pointer
	// or item not defined correctly (st==NULL)
	// Do the default popup menu.
	return NULL;
    }

    GtkWidget *popup_widget = rfm_get_widget(POPUP_MENU_ID);
    if (!popup_widget) g_error("popup_widget is initialized on module load...\n");
    // Set variable functions: properties, mount and unmount.
    GtkWidget *item;
    item = rfm_get_widget("fuse_properties");
    g_object_set_data(G_OBJECT(item), "callback", properties_f);
    item = rfm_get_widget("fuse_mount");
    g_object_set_data(G_OBJECT(item), "callback", mount_f);
    item = rfm_get_widget("fuse_unmount");
    g_object_set_data(G_OBJECT(item), "callback", umount_f);

    // Set record entry information.
    gchar *items[]={"fuse_new_tab", "fuse_new_window", "fuse_properties", 
	"fuse_mount", "fuse_unmount", "fuse_delete", NULL};
    gchar **y = items;
    for (; y && *y; y++){
	GtkWidget *w = rfm_get_widget(*y);
	g_object_set_data(G_OBJECT(w),"widgets_p",widgets_p);
	g_object_set_data(G_OBJECT(w), "entry", en);
    }
    gint mounted = FSTAB_entry_is_mounted (en);

    if (mounted > 0) {
	HIDE_IT("fuse_broken_mount");
	HIDE_IT("fuse_mount");
	SHOW_IT("fuse_unmount");
    } else if (mounted == 0){
	HIDE_IT("fuse_broken_mount");
	HIDE_IT("fuse_unmount");
	SHOW_IT("fuse_mount");
    } else {
	HIDE_IT("fuse_unmount");
	HIDE_IT("fuse_mount");
	SHOW_IT("fuse_broken_mount");
    }


    gtk_menu_popup(GTK_MENU(popup_widget), NULL, NULL, NULL, NULL, 3, gtk_get_current_event_time());

    return GINT_TO_POINTER(1);

}


typedef struct fuse_login_t{
    guint64 flag;
    gchar *flag_key;
    gchar *computer;
    gchar *remote_path;
    gchar *login;
    gchar *empty_passphrase;
    gchar *secure_shell_key;
    gchar *url;
    gchar *mount_point;
    gchar *monitor;
    gchar *broadband;
}fuse_login_t;

void free_login_p(fuse_login_t *p){
    if (!p) return;
    g_free(p->flag_key);
    g_free(p->computer);
    g_free(p->remote_path);
    g_free(p->login);
    g_free(p->empty_passphrase);
    g_free(p->secure_shell_key);
    g_free(p->url);
    g_free(p->mount_point);
    g_free(p->monitor);
    g_free(p->broadband);
    g_free(p);
}

// This function is called by "accept", which is contained in GDK mutex.
// The extra MUTEX is redundant (recursive).
void *
fuse_get_login_info (fuse_data_t *fuse_data_p){
    if (!fuse_data_p->dialog) g_error("fuse_get_login_info(): dialog may not be NULL\n");
    const gchar *url_template = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "url_template");
    if (!url_template) g_error("url_template not defined\n");

    fuse_login_t *fuse_login_p = 
	(fuse_login_t *)malloc(sizeof(fuse_login_t));
    if (!fuse_login_p){
	g_error("unable to malloc fuse_login_p\n");
    }
    memset(fuse_login_p, 0, sizeof(fuse_login_t));

    const gchar *group = fuse_data_p->url;
    // Guint64 flag, FLAG_0 
    fuse_login_p->flag_key = FLAG_KEY(0);
    guint64 ui = 0x01;

    // Get specified host name, FUSE_COMPUTER, if it is in the dialog, of course.
    if (g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_COMPUTER")){
	fuse_login_p->computer = fuse_get_item(fuse_data_p->dialog, "FUSE_COMPUTER");
	if (!fuse_login_p->computer){
	    fuse_error(fuse_data_p->dialog,_("Invalid hostname"));
	    free_login_p(fuse_login_p);
	    fuse_login_p->url = g_strdup_printf("%s://%s%s", url_template, fuse_login_p->computer, 
		(fuse_login_p->remote_path)? fuse_login_p->remote_path: "");
	    return NULL;
	} else {
	    fuse_login_p->flag |= (ui << 0);
	} 
    } else if (fuse_data_p->computer == NULL){
	    fuse_error(fuse_data_p->dialog,_("Invalid hostname"));
	    free_login_p(fuse_login_p);
	    if (!fuse_login_p->url) {
		DBG("fuse_get_login_info(): fuse_login_p->url is not defined\n");
	    }
	    return NULL;
    } else {
	fuse_login_p->url = g_strdup(fuse_data_p->url);
	fuse_login_p->computer = g_strdup(fuse_data_p->computer);
    }


    // Get specified rpath, FUSE_REMOTE_PATH
    if (g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_REMOTE_PATH")){
	fuse_login_p->remote_path = fuse_get_item(fuse_data_p->dialog, "FUSE_REMOTE_PATH");
	if (!fuse_login_p->remote_path){
	    fuse_error(fuse_data_p->dialog,_("invalid path"));
	    free_login_p(fuse_login_p);
	    return NULL;
	} else {
	    fuse_login_p->flag |= (ui << 1);
	} 
    }



    GtkWidget *check = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_ALLOW_EMPTY_PASSPHRASE");
    if (check){
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))) {
	    fuse_login_p->empty_passphrase = g_strdup("yes");
	} else {
	    fuse_login_p->empty_passphrase = g_strdup("");
	}
    }
    if (fuse_login_p->empty_passphrase && strlen(fuse_login_p->empty_passphrase)) {
	fuse_login_p->flag |= (ui << 2);
    }

    check = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_SECURE_SHELL_KEY");
    if (check){
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))) {
	    fuse_login_p->secure_shell_key = g_strdup("yes");
	} else {
	    fuse_login_p->secure_shell_key = g_strdup("");
	}
    }
    if (fuse_login_p->secure_shell_key && strlen(fuse_login_p->secure_shell_key)) {
	fuse_login_p->flag |= (ui << 3);
    }

    if (fuse_login_p->empty_passphrase && strlen(fuse_login_p->empty_passphrase)) {
	fuse_login_p->flag |= (ui << 2);
    }
    if (fuse_login_p->secure_shell_key && strlen(fuse_login_p->secure_shell_key)){
	fuse_login_p->flag |= (ui << 3);
    } 
    { 
	// Get login (if any)
	fuse_login_p->login =
	    fuse_get_item(fuse_data_p->dialog, "FUSE_LOGIN");
	if (fuse_login_p->login){
	    fuse_login_p->flag |= (ui << 4);
	}
    }

    // Get specified mount point
    //GtkWidget *entry;
    TRACE( "Get specified mount point...\n");
    //if (rfm_void(PLUGIN_DIR, "fstab", "module_active"))
    {
    //if (!gtk_widget_get_sensitive(entry)){
	// Get record_entry
	record_entry_t *en = rfm_mk_entry(0);
	gchar *preset_mnt_point = fuse_get_item(fuse_data_p->dialog, "FUSE_MOUNT_POINT");
	if (preset_mnt_point && g_file_test(preset_mnt_point, G_FILE_TEST_IS_DIR)) {
	    en->tag = preset_mnt_point;
	} else {
	    g_free(preset_mnt_point);
	    // en->tag will set default mount point in dialog
	    // Do this for ecryptfs:
	    if (g_object_get_data(G_OBJECT(fuse_data_p->dialog), "ECRYPTFS_SIG")
		    && fuse_login_p->remote_path) {
		en->tag = g_strdup(fuse_login_p->remote_path);
	    }
	}
	if (!fuse_login_p->remote_path) {
	    en->path = g_strdup(fuse_login_p->computer);
	} else {
	    en->path = g_strdup_printf("%s%s%s", fuse_login_p->computer, 
		(fuse_login_p->remote_path[0]=='/')? "":"/", 
		fuse_login_p->remote_path);
	}

	GtkWidget *entry = g_object_get_data(G_OBJECT(fuse_data_p->dialog),
	    "FUSE_MOUNT_POINT");
	if (gtk_widget_get_sensitive(entry)){
	    en->pseudo_path = g_strdup(en->path);
	} else {
	    en->pseudo_path = g_strdup_printf("%s", fuse_login_p->computer);
	}
	gchar *d;
	if (gtk_widget_get_sensitive(entry)) for (d=en->path; d && *d; d++){
	    if (*d=='/' || *d=='@') *d='-'; //if (*d=='/') *d='\\';if (*d=='@') *d='\\';
	}
	
	fuse_login_p->mount_point = 
	    (gchar *)rfm_natural(RFM_MODULE_DIR, "callbacks", en, "callback_mnt_point");
	rfm_destroy_entry(en);
	NOOP(stderr, "fuse-common.c: mount point=%s\n", fuse_login_p->mount_point);
	if (!fuse_login_p->mount_point || !strlen(fuse_login_p->mount_point) || !g_path_is_absolute(fuse_login_p->mount_point)){
	    gchar *text = g_strconcat(_("Could not detect any mount point."),
		    "  (callbacks: callback_mnt_point)", NULL);

	    fuse_error(fuse_data_p->dialog,text);
	    g_free(text);
	    free_login_p(fuse_login_p);
	    return NULL;
	}
    } 
#if 0
    // Deprecated. Mount point dialog is now in callbacks module, which
    // will always be present.
    else {
	entry = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_MOUNT_POINT");		const gchar *c =  gtk_entry_get_text (GTK_ENTRY(entry));
	if (!c || !strlen(c) || !g_path_is_absolute(c)){
	    NOOP("2.invalid mountpoint\n");
	    fuse_error(fuse_data_p->dialog,_("Could not detect any mount point."));
	    free_login_p(fuse_login_p);
	    return NULL;
	}
	fuse_login_p->mount_point = g_strdup(c);
	if(g_mkdir_with_parents (fuse_login_p->mount_point, 0700) < 0) {
	    gchar *warn =  g_strdup_printf("mkdir(%s): %s\n", 
		    fuse_login_p->mount_point, strerror (errno));
	    fuse_error(fuse_data_p->dialog, warn);
	    g_free(warn);
	    free_login_p(fuse_login_p);
	    return NULL;
        }
    }
#endif

    fuse_login_p->flag |= (ui << 5);

    // These are tree state check buttons. 
    //     Null, does not exist
    //     strlen > 0, active
    //     strlen ==o, inactive.
    check = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_MONITOR");
    if (check){
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))) {
	    fuse_login_p->monitor = g_strdup("yes");
	} else {
	    fuse_login_p->monitor = g_strdup("");
	}
    }

    if (fuse_login_p->monitor && strlen(fuse_login_p->monitor)) {
	fuse_login_p->flag |= (ui << 6);
    }

    check = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_BROADBAND");
    if (check){
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))) {
	    fuse_login_p->broadband = g_strdup("yes");
	} else {
	    fuse_login_p->broadband = g_strdup("");
	}
    }
    if (fuse_login_p->broadband && strlen(fuse_login_p->broadband)) {
	fuse_login_p->flag |= (ui << 7);
    }



    // all applicable items are now set.
    
    // We now have enough to proceed...
    key_options_t *key_options_p = 
	(key_options_t *)malloc(sizeof(key_options_t));
    if (!key_options_p) g_error("cannot malloc key_options_p\n");
    memset(key_options_p, 0, sizeof(key_options_t));
    g_object_set_data(G_OBJECT(fuse_data_p->dialog), "key_options_p", key_options_p);

    key_options_p->key_file = g_key_file_new ();


    GHashTable *hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    g_object_set_data(G_OBJECT(fuse_data_p->dialog), "hash", hash);
    g_hash_table_ref( hash);
  
    if (g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_URL")){
	fuse_login_p->url = fuse_get_item(fuse_data_p->dialog, "FUSE_URL");
	if (!fuse_login_p->url){
	    g_error("invalid URL");
	}
    }
    key_options_p->group = g_strdup(fuse_login_p->url);
    NOOP( "group %s\n", key_options_p->group);


    // Load previous values, including comments and translations...
    gchar *file = g_build_filename(FUSE_KEY_FILE, NULL);
    if (!g_key_file_load_from_file (key_options_p->key_file, file,
		G_KEY_FILE_KEEP_COMMENTS|G_KEY_FILE_KEEP_TRANSLATIONS, NULL)){
	NOOP("New file: %s\n", file);
    }
    g_free(file);

    if (g_object_get_data(G_OBJECT(fuse_data_p->dialog), "FUSE_COMPUTER")){
        g_hash_table_insert(hash, g_strdup("FUSE_COMPUTER"), g_strdup(fuse_login_p->computer));
    }
    if(fuse_login_p->remote_path) {
	g_hash_table_insert(hash, g_strdup("FUSE_REMOTE_PATH"), g_strdup(fuse_login_p->remote_path));
    }
    if(fuse_login_p->login) {
	g_hash_table_insert(hash, g_strdup("FUSE_LOGIN"), g_strdup(fuse_login_p->login));
    }
    NOOP(stderr, "inserting to hash: %s\n", fuse_login_p->mount_point);
    g_hash_table_insert(hash, g_strdup("FUSE_MOUNT_POINT"), g_strdup(fuse_login_p->mount_point));

    // Remove previous group values (if any).
    // This includes previous url:
    if (group) g_key_file_remove_group(key_options_p->key_file, group, NULL);

    // And modified url:
    if (key_options_p->group) g_key_file_remove_group(key_options_p->key_file, key_options_p->group, NULL);

    // Add group with new values.
    if (fuse_login_p->broadband && strlen(fuse_login_p->broadband)){
	g_key_file_set_boolean(key_options_p->key_file, key_options_p->group,
		"FUSE_BROADBAND", TRUE); 
	g_key_file_set_boolean(key_options_p->key_file, key_options_p->group,
		"FUSE_MONITOR", FALSE); 
    } else if (fuse_login_p->monitor && strlen(fuse_login_p->monitor)){
	g_key_file_set_boolean(key_options_p->key_file, key_options_p->group,
		"FUSE_MONITOR", TRUE); 
    }  

    if (fuse_login_p->secure_shell_key && strlen(fuse_login_p->secure_shell_key)){
	g_key_file_set_boolean(key_options_p->key_file, key_options_p->group,
		"FUSE_SECURE_SHELL_KEY", TRUE); 
    }
    if (fuse_login_p->empty_passphrase && strlen(fuse_login_p->empty_passphrase)){
	g_key_file_set_boolean(key_options_p->key_file, key_options_p->group,
		"FUSE_ALLOW_EMPTY_PASSPHRASE", TRUE); 
    }

#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION<26
	g_key_file_set_integer (key_options_p->key_file, key_options_p->group, 
	    fuse_login_p->flag_key, (gint) fuse_login_p->flag);

#else
    g_key_file_set_uint64 (key_options_p->key_file, key_options_p->group, 
	    fuse_login_p->flag_key, fuse_login_p->flag);
#endif
    
    // replace current hash table values in key_file:
    if (hash){
	g_hash_table_foreach(hash, fuse_set_key_item_f, key_options_p);
	g_hash_table_destroy(hash);
    }
    // FLAG_0 is done.
    free_login_p(fuse_login_p);

    return GINT_TO_POINTER(1);
}

void *
fuse_set_options(fuse_data_t *fuse_data_p){
    if (!fuse_data_p) g_error("fuse_data_p cannot be null\n");
    
    guint64 flag = 0;
    gchar *flag_key =  FLAG_KEY(fuse_data_p->flag_id);
    group_options_t *options_p = fuse_data_p->options_p;
    gchar ***options_keys_p = fuse_data_p->options_keys_p;

    if (*options_keys_p == NULL) *options_keys_p = group_option_keys(options_p);
     
    key_options_t *key_options_p = 
	g_object_get_data(G_OBJECT(fuse_data_p->dialog), "key_options_p");
    if (!key_options_p) g_error("key_options_p cannot be null\n");

    GHashTable *hash = group_options_get_option_hash(fuse_data_p->dialog, *options_keys_p, &flag);
#if GLIB_MAJOR_VERSION==2 && GLIB_MINOR_VERSION<26
    g_key_file_set_integer (key_options_p->key_file, key_options_p->group, flag_key, (gint) flag);

#else
    g_key_file_set_uint64 (key_options_p->key_file, key_options_p->group, flag_key, flag);
#endif
    g_free(flag_key);
    if (hash){
	g_hash_table_foreach(hash, fuse_set_key_item_f, key_options_p);
	g_hash_table_destroy(hash);
    }

    return NULL;
}

gchar *
fuse_save_keyfile(fuse_data_t *fuse_data_p){
    key_options_t *key_options_p = 
	g_object_get_data(G_OBJECT(fuse_data_p->dialog), "key_options_p");
    if (!key_options_p) g_error("key_options_p cannot be null\n");
    NOOP(stderr, "writing keyfile for group: %s\n", key_options_p->group);
    gchar *url = g_strdup(key_options_p->group);
    group_options_write_keyfile(key_options_p->key_file);
    g_free(key_options_p->group);
    g_key_file_free(key_options_p->key_file);
    g_free(key_options_p);

    GHashTable *hash = 
	g_object_get_data(G_OBJECT(fuse_data_p->dialog), "hash");
    if (hash) g_hash_table_unref(hash);
    else{
	g_error("hash should not be null\n");
    }
    return url;
}


static 
void 
toggle_broad(GtkToggleButton *togglebutton , gpointer data){
    GtkWidget *dialog = data;
    GtkToggleButton *b = g_object_get_data(G_OBJECT(dialog), "FUSE_MONITOR");
    if (!b) return;
    if (gtk_toggle_button_get_active(togglebutton)){
	gtk_widget_set_sensitive(GTK_WIDGET(b), FALSE);
	gtk_toggle_button_set_active(b, FALSE);
    } else {
	gtk_widget_set_sensitive(GTK_WIDGET(b), TRUE);
    }
}

static 
void 
toggle_ssh(GtkToggleButton *togglebutton , gpointer data){
    GtkWidget *dialog = data;
    GtkWidget *b = g_object_get_data(G_OBJECT(dialog), "FUSE_ALLOW_EMPTY_PASSPHRASE");
    if (!b){
	return;
    }
    if (gtk_toggle_button_get_active(togglebutton)){
	gtk_widget_set_sensitive(b, TRUE);
    } else {
	gtk_widget_set_sensitive(b, FALSE );
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b), FALSE);
    }
}


static 
gint
make_absolute (GtkWidget * in_entry, GdkEventKey * event, gpointer data){
    const gchar *c =  gtk_entry_get_text (GTK_ENTRY(in_entry));
    if (c && g_path_is_absolute(c)) return FALSE;
    gchar *a = g_strconcat("/", c, NULL);
    gtk_entry_set_text(GTK_ENTRY(in_entry), a);

    g_free(a);
    gtk_editable_set_position (GTK_EDITABLE(in_entry), strlen(a));
    return FALSE;
}

static 
gint
on_key_press (GtkWidget * in_entry, GdkEventKey * event, gpointer data){
    GtkWidget *dialog = data;
    const gchar *url_template = g_object_get_data(G_OBJECT(data), "url_template");
    GtkWidget *entry;
    entry = g_object_get_data(G_OBJECT(dialog), "FUSE_COMPUTER");
    if (!entry) entry = g_object_get_data(G_OBJECT(dialog), "OBEX_NAME");
    gchar *host;
    if (entry) {
	const gchar *c =  (entry)?gtk_entry_get_text (GTK_ENTRY(entry)):NULL;
	if (!c || !strlen(c)){
	    host = g_strdup("");
	} else {
	    host = g_strdup(c);
	}
    } else {
	DBG("No host name defined in function fuse_common.c:on_key_press()\n");
	host = g_strdup("");
    }


    gchar *rpath=NULL;
    entry = g_object_get_data(G_OBJECT(dialog), "FUSE_REMOTE_PATH");
    if (entry) {
	const gchar *c =  gtk_entry_get_text (GTK_ENTRY(entry));
	if (!c || !strlen(c)){
	    rpath = g_strdup("/");
	} else {
	    if (c[0] != '/') rpath = g_strconcat("/", c, NULL);
	    else rpath = g_strdup(c);
	}
    }

    gchar *login=NULL;
    entry = g_object_get_data(G_OBJECT(dialog), "FUSE_LOGIN");
    if (entry) {
	const gchar *c =  gtk_entry_get_text (GTK_ENTRY(entry));
	if (c && strlen(c)){
	    login = g_strdup(c);
	}
    }

    gchar *url;
    if (login) { 
	url = g_strdup_printf("%s://%s@%s%s", url_template, login, host, 
		(rpath)?rpath:"");
    } else {
	url = g_strdup_printf("%s://%s%s", url_template, host, 
		(rpath)?rpath:"");
    }
    entry = g_object_get_data(G_OBJECT(dialog), "FUSE_URL");

    gtk_entry_set_text(GTK_ENTRY(entry), url);
    g_free(host);
    g_free(rpath);
    g_free(url);
    return FALSE;
}

GKeyFile *
fuse_load_keyfile(void){
    GKeyFile *key_file=NULL;
    key_file = g_key_file_new ();
    gchar *file = g_build_filename(FUSE_KEY_FILE, NULL);
    if (!g_key_file_load_from_file (key_file, file, 0, NULL)){
        g_key_file_free(key_file);
        key_file=NULL;
    }
    g_free(file);
    return key_file;
}

static void *
fuse_init_dialog_f(gpointer data){
    fuse_data_t *fuse_data_p = data;
    rfm_global_t *rfm_global_p = rfm_global();
    GtkWidget *dialog;


    dialog = gtk_dialog_new ();
    if (fuse_data_p->widgets_p) {
	g_object_set_data(G_OBJECT(dialog), 
		"widgets_p", fuse_data_p->widgets_p);
	view_t *view_p=fuse_data_p->widgets_p->view_p;
        if(view_p && view_p->flags.type == DESKVIEW_TYPE) {
	    gtk_window_set_keep_above (GTK_WINDOW(dialog), TRUE);
	    gtk_window_stick (GTK_WINDOW(dialog));
	} else {   
            gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
            gtk_window_set_transient_for (GTK_WINDOW (dialog), 
		    GTK_WINDOW (rfm_global_p->window));
        } 
    } else {
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    }


    GtkWidget *hbox = rfm_hbox_new (TRUE, 2);
    gtk_box_pack_start (GTK_BOX (gtk_dialog_get_content_area(GTK_DIALOG (dialog))), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);

    GdkPixbuf *pixbuf;
    pixbuf=rfm_get_pixbuf("xffm/stock_dialog-question", 24);
    GtkWidget *image=gtk_image_new_from_pixbuf(pixbuf);
    g_object_unref(pixbuf);
    gtk_widget_show(image);
    gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);

    gchar *text = g_strconcat(_("Options:"), " ", 
	    fuse_data_p->info1, "\n\n", 
	    fuse_data_p->info2, NULL);
    GtkWidget *label = gtk_label_new("");
    gtk_label_set_markup(GTK_LABEL(label), text);
    g_free(text);
    gtk_widget_show(label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

    //GtkWidget *tbox = gtk_dialog_get_content_area (GTK_DIALOG(dialog));
    GtkWidget *tbox = rfm_vbox_new(FALSE, 0);
    gtk_box_pack_start (GTK_BOX (gtk_dialog_get_content_area (GTK_DIALOG(dialog))), tbox, FALSE, FALSE, 0);
    GtkWidget *notebook = gtk_notebook_new ();
    g_object_set_data(G_OBJECT(dialog), "notebook", notebook);
    //   gtk_notebook_set_tab_pos ((gtknotebook *)notebook, gtk_pos_bottom);
    gtk_notebook_popup_enable (GTK_NOTEBOOK(notebook));
    gtk_notebook_set_scrollable (GTK_NOTEBOOK(notebook), TRUE);
    g_object_set (notebook,
                  "enable-popup", TRUE, 
                  "can-focus", FALSE,
                  "scrollable", TRUE, 
                  "show-border", FALSE,
                  "show-tabs", 
                  TRUE, "tab-pos",
                  GTK_POS_TOP, NULL);  


    gtk_box_pack_start (GTK_BOX (tbox), notebook, TRUE, TRUE, 0);
    GtkWidget *vbox = rfm_vbox_new (FALSE, 0);
    g_object_set_data(G_OBJECT(dialog), "vbox", vbox);
    gtk_widget_show(vbox);
    
    GtkWidget *tab_label = gtk_label_new(_("Login"));
    GtkWidget *menu_label = gtk_label_new(_("Login"));

    gtk_notebook_insert_page_menu (GTK_NOTEBOOK(notebook), vbox, tab_label, menu_label, 0);
    gtk_notebook_set_tab_reorderable (GTK_NOTEBOOK(notebook), vbox, TRUE);

    //get action area deprecated... whatever.
    //GtkWidget *action_area = gtk_dialog_get_action_area(GTK_DIALOG(dialog));
    GtkWidget *action_area = gtk_dialog_get_content_area(GTK_DIALOG(dialog));

    GtkWidget *button = rfm_dialog_button ("xffm/stock_no", _("Cancel"));
    gtk_box_pack_start (GTK_BOX (action_area), button, FALSE, FALSE, 0);
    //gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_NO);
    g_object_set_data (G_OBJECT (dialog), "action_FALSE_button", button);


    button = rfm_dialog_button ("xffm/stock_apply", _("Ok"));
    g_object_set_data (G_OBJECT (dialog), "action_TRUE_button", button);
    gtk_box_pack_start (GTK_BOX (action_area), button, FALSE, FALSE, 0);
    //gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_APPLY);

    button = rfm_dialog_button ("xffm/stock_yes", _("Mount"));
    g_object_set_data (G_OBJECT (dialog), "action_MOUNT_button", button);
    gtk_box_pack_start (GTK_BOX (action_area), button, FALSE, FALSE, 0);
    //gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_YES);

    gtk_window_set_resizable (GTK_WINDOW(dialog), TRUE);

    GKeyFile *key_file = (fuse_data_p->url)? fuse_load_keyfile(): NULL;

    g_object_set_data(G_OBJECT(dialog), "url", (void *)(fuse_data_p->url));
    g_object_set_data(G_OBJECT(dialog), "key_file", key_file);
    return dialog;

}

GtkWidget *
fuse_init_dialog(fuse_data_t *fuse_data_p){
    GtkWidget *w = rfm_context_function(fuse_init_dialog_f, fuse_data_p);
    return w;
}

GtkWidget *
fuse_add_entry(fuse_data_t *fuse_data_p){
    
    GtkWidget *vbox = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "vbox");
    const gchar *url = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "url");
    GKeyFile *key_file = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "key_file");
    
    const gchar *separator=" ";
    if (fuse_data_p->item_string && !strchr(fuse_data_p->item_string, ':')){
	separator = ": ";
    }

    GtkWidget *hbox = fuse_make_entry_box(fuse_data_p->dialog, fuse_data_p->item_string, fuse_data_p->item_id, separator, TRUE, on_key_press);
    if (strcmp(fuse_data_p->item_id, "FUSE_MOUNT_POINT")==0){
	g_object_set_data(G_OBJECT(fuse_data_p->dialog), "FUSE_MOUNT_POINT_BOX", hbox);
    }
	
	
    GtkWidget *entry = g_object_get_data(G_OBJECT(fuse_data_p->dialog), fuse_data_p->item_id);
    if (strcmp(fuse_data_p->item_id, "FUSE_REMOTE_PATH")==0) {
	g_signal_connect(G_OBJECT(entry), "key-release-event", G_CALLBACK(make_absolute), fuse_data_p->dialog);
    }
    gchar *default_value=NULL;
    gchar *user=NULL;
    gchar *computer=NULL;
    gchar *remote_path=NULL;
    gchar *u = NULL;
    if (url){
	u = g_strdup(url);
	gchar *p = strstr(u, "://");
	if (p) {
	    p = p+strlen("://");
	    if (strchr(p,'/')){
		*strchr(p,'/') = 0;
		if (strchr(p,'@')) {
		    user = g_strdup(p);
		    *strchr(user,'@') = 0;
		    p = strchr(p,'@') + 1;
		}
		computer = g_strdup(p);
		remote_path = g_strdup_printf("/%s",p + strlen(p) + 1);
	    }
	}
	g_free(u);
    }

    if (strcmp(fuse_data_p->item_id, "FUSE_COMPUTER")==0) {
	default_value = g_strdup(computer);
    } else if (strcmp(fuse_data_p->item_id, "FUSE_REMOTE_PATH")==0) {
	default_value = g_strdup(remote_path);
    }
    if (strcmp(fuse_data_p->item_id, "FUSE_LOGIN")==0) {
	if (user) {
	    default_value = g_strdup(user);
	} else {
	    if (getenv("USER")) 
		default_value = g_strdup(getenv("USER"));
	    else if (getenv("LOGNAME")) 
		default_value = g_strdup(getenv("LOGNAME"));
	    else default_value = g_strdup(getenv("GUEST"));
	}
    }
    g_free(user);
    g_free(computer);
    g_free(remote_path);
    if (url && strcmp(fuse_data_p->item_id, "FUSE_MOUNT_POINT")==0) {
	default_value = rfm_default_url_mount_point(url);
    }
    
    if (!default_value) default_value=g_strdup("");
    fuse_set_entry(entry, key_file, url, fuse_data_p->item_id, default_value);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    g_free(default_value);
    return hbox;
}


gchar *
fuse_get_combo_text(fuse_data_t *fuse_data_p){
    GtkWidget *combo = g_object_get_data(G_OBJECT(fuse_data_p->dialog), fuse_data_p->item_id);
    if (!combo) return NULL;
#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    gchar *string = gtk_combo_box_get_active_text (GTK_COMBO_BOX(combo));
#else
    gchar *string = gtk_combo_box_text_get_active_text (GTK_COMBO_BOX_TEXT(combo));
#endif   
   return string;
}

GtkWidget *
fuse_add_combo(fuse_data_t *fuse_data_p){
    GtkWidget *hbox = rfm_hbox_new(FALSE, 0);    
    GtkWidget *vbox = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "vbox");
    const gchar *url = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "url");
    GKeyFile *key_file = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "key_file");
   
    gchar *full_text;
    GtkWidget *label = gtk_label_new("");
    if (strchr(fuse_data_p->item_string, ':')){
	full_text = g_strconcat(fuse_data_p->item_string, " ", NULL);
    } else {
	full_text = g_strconcat(fuse_data_p->item_string, ": ", NULL);
    }
    gtk_label_set_markup(GTK_LABEL(label), full_text);
    g_free(full_text);

#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    // this is deprecated...
    GtkWidget *combobox = gtk_combo_box_new_text ();
#else
    GtkWidget *combobox = gtk_combo_box_text_new ();
#endif

    g_object_set_data(G_OBJECT(fuse_data_p->dialog), fuse_data_p->item_id, combobox);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), combobox, TRUE, TRUE, 0);
    
// Set default combo value.
    gchar *pre_set=NULL;
    if (key_file && url) {
	pre_set = g_key_file_get_value(key_file, url, fuse_data_p->item_id, NULL);
    }

#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    if (pre_set){
        gtk_combo_box_prepend_text(GTK_COMBO_BOX(combobox), pre_set);
    } else if (fuse_data_p->default_value){
        gtk_combo_box_prepend_text(GTK_COMBO_BOX(combobox), fuse_data_p->default_value);
    } else {
	gtk_widget_set_sensitive(hbox, FALSE);
    }
#else
    if (pre_set){
	gtk_combo_box_text_prepend_text(GTK_COMBO_BOX_TEXT(combobox),pre_set);
	gtk_combo_box_set_active (GTK_COMBO_BOX(combobox), 0);
    } else if (fuse_data_p->default_value){
	gtk_combo_box_text_prepend_text(GTK_COMBO_BOX_TEXT(combobox),fuse_data_p->default_value); 
    } else {
	gtk_widget_set_sensitive(combobox, FALSE);
    }

#endif
    g_free(pre_set); 

    gtk_widget_show(hbox);
    gtk_widget_show(label);
    gtk_widget_show(combobox);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    return hbox;
}

GtkWidget *
fuse_add_check(fuse_data_t *fuse_data_p){
    GtkWidget *vbox = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "vbox");
    GtkWidget *hbox = rfm_hbox_new(FALSE, 0);


    GtkWidget *check; 
    if (strcmp(fuse_data_p->item_id, "FUSE_BROADBAND")==0) {
	check = fuse_make_check_box(fuse_data_p->dialog, 
		fuse_data_p->item_string, 
		fuse_data_p->item_id, 
		toggle_broad); 
    } else if (strcmp(fuse_data_p->item_id, "FUSE_SECURE_SHELL_KEY")==0) {
	check = fuse_make_check_box(fuse_data_p->dialog, 
		fuse_data_p->item_string, 
		fuse_data_p->item_id, 
		toggle_ssh); 
    }else {
	check = fuse_make_check_box(fuse_data_p->dialog, 
		fuse_data_p->item_string, 
		fuse_data_p->item_id, 
		NULL); 
    }
    gtk_box_pack_start (GTK_BOX (hbox), check, FALSE, FALSE, 0);
    GtkWidget *label = gtk_label_new("");
    if (fuse_data_p->extra_text) {
	gchar *c = g_strdup_printf("<i>(%s)</i>", fuse_data_p->extra_text);
	gtk_label_set_markup(GTK_LABEL(label), c);
	g_free(c);
    } 
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_widget_show(label);
    gtk_widget_show(hbox);
    GKeyFile *key_file = g_object_get_data(G_OBJECT(fuse_data_p->dialog), 
	    "key_file");
    const gchar *url = g_object_get_data(G_OBJECT(fuse_data_p->dialog), 
	    "url");
    fuse_set_check_button_state(fuse_data_p->dialog, key_file, url, 
	    fuse_data_p->item_id, fuse_data_p->item_id,
	    fuse_data_p->default_state);

    if (strcmp(fuse_data_p->item_id, "FUSE_ALLOW_EMPTY_PASSPHRASE")==0){
	GtkWidget *t = g_object_get_data(G_OBJECT(fuse_data_p->dialog),
	    "FUSE_SECURE_SHELL_KEY");
	if (t && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(t))){
	    gtk_widget_set_sensitive(check, TRUE);
	} else if (t) {
	    gtk_widget_set_sensitive(check, FALSE);
	}

    }
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    return hbox;
}

void *
fuse_add_option_page(fuse_data_t *fuse_data_p) {
    GKeyFile *key_file = g_object_get_data(G_OBJECT(fuse_data_p->dialog), 
	    "key_file");
    const gchar *url = g_object_get_data(G_OBJECT(fuse_data_p->dialog),
	    "url");

    GtkWidget *vbox = group_options_box(fuse_data_p->dialog, 
	    fuse_data_p->options_p, key_file, url, fuse_data_p->flag_id);
    gtk_widget_show(vbox);
    GtkWidget *tab_label = gtk_label_new(fuse_data_p->label);
    GtkWidget *menu_label = gtk_label_new(fuse_data_p->label);
    GtkWidget *notebook = g_object_get_data(G_OBJECT(fuse_data_p->dialog), 
	    "notebook");
    gtk_notebook_append_page_menu (GTK_NOTEBOOK(notebook), vbox, tab_label, 
	    menu_label);  
    gtk_notebook_set_tab_reorderable (GTK_NOTEBOOK(notebook), vbox, TRUE);
    return NULL;
}

void *
fuse_reset_url_field(fuse_data_t *fuse_data_p){
    if (!fuse_data_p) return NULL;
    GKeyFile *key_file = g_object_get_data(G_OBJECT(fuse_data_p->dialog), 
	    "key_file");
    
    if (key_file) {
	on_key_press(NULL, NULL, fuse_data_p->dialog);
	g_key_file_free(key_file);
	g_object_set_data(G_OBJECT(fuse_data_p->dialog), "key_file", NULL);
    }
    return NULL;
}


fuse_data_t *
fuse_data_new(fuse_data_t *fuse_data_p){
    if (fuse_data_p == NULL) {
	fuse_data_p = (fuse_data_t *)malloc(sizeof(fuse_data_t));
	if (!fuse_data_p){
	    g_error("fuse_item_data_new(): cannot malloc item_data_p\n");
	}
    }
    memset(fuse_data_p, 0, sizeof(fuse_data_t));
    return fuse_data_p;
}


xfdir_t *
fuse_xfdir (void **argv){
    gint argc = 0;
    while (argv[argc]) argc++;
    if (argc < 6) g_error("fuse_xfdir(): insufficient arguments\n");
    return fuse_xfdir_get(argv[0], argv[1], argv[2], argv[3], argv[4], argv[5]);
}

const gchar *
fuse_icon (void **argv){
    gint argc = 0;
    while (argv[argc]) argc++;
    if (argc < 4) g_error("fuse_icon(): insufficient arguments\n");
    return fuse_icon_id(argv[0], argv[1], argv[2], argv[3]);
}

void *
fuse_popup(void **argv){
    gint argc = 0;
    while (argv[argc]) argc++;
    TRACE( "fuse_popup...\n");
    if (argc < 3){
	TRACE( "fuse_popup(): insufficient arguments\n");
	return NULL;
    }
	
    return fuse_private_popup(argv[0], argv[1], argv[2], argv[3]);
}

gboolean 
fuse_mkdir(const gchar *mount_point) {
	TRACE( "fuse_mkdir %s\n", mount_point);
    if (g_file_test(mount_point, G_FILE_TEST_IS_DIR)) return TRUE;
    widgets_t * widgets_p = rfm_get_widget("widgets_p");
    if (g_file_test(mount_point, G_FILE_TEST_EXISTS)) {
	TRACE( "fuse_mkdir 1\n");
	rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-error",NULL);
	gchar *text = g_strdup_printf(_("The file with pathname \"%s\" is not a directory."), mount_point);
	rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-error",text);
	g_free(text);
	return FALSE;
    }
    if (g_mkdir_with_parents(mount_point, 0700) < 0) {
	// XXX try sudo here (with dialog warning)...
	TRACE( "fuse_mkdir 21\n");
	return FALSE;
    }
    return TRUE;
}

void *
fuse_test_ini_file(void *p){
    widgets_t * widgets_p = rfm_get_widget("widgets_p");

    static gint count=1;

    struct stat st;
    gchar *file = g_build_filename(FUSE_KEY_FILE, NULL);
    if (stat(file, &st) < 0 ) return NULL;
    g_free(file);
    gint mtime = GPOINTER_TO_INT(
	    g_object_get_data(G_OBJECT(widgets_p->paper), "fuse_mtime")
	    );

    if (mtime == st.st_mtime) return NULL;
    g_object_set_data(G_OBJECT(widgets_p->paper), "fuse_mtime",
	    GINT_TO_POINTER(st.st_mtime));
	
    return GINT_TO_POINTER (++count);
}


typedef struct fuse_dialog_t {
    view_t *view_p;
    GMutex *mutex;
    GCond *signal;
} fuse_dialog_t;

static void *
hold_monitor(void *data){
    fuse_dialog_t *fuse_dialog_p = data;
    view_t *view_p = fuse_dialog_p->view_p;
    g_mutex_lock (fuse_dialog_p->mutex);
    rfm_rw_lock_reader_lock (&(view_p->mutexes.monitor_lock));
    g_cond_wait(fuse_dialog_p->signal, fuse_dialog_p->mutex);
    rfm_rw_lock_reader_unlock (&(view_p->mutexes.monitor_lock));
    g_mutex_unlock (fuse_dialog_p->mutex);
    rfm_mutex_free(fuse_dialog_p->mutex);
    rfm_cond_free(fuse_dialog_p->signal);
    g_free(fuse_dialog_p);
    return NULL;
}

GCond *
fuse_hold_monitor(void){
    widgets_t * widgets_p = rfm_get_widget("widgets_p");
    fuse_dialog_t *fuse_dialog_p = (fuse_dialog_t *)malloc(sizeof(fuse_dialog_t));
    if (!fuse_dialog_p) g_error("malloc: %s\n", strerror(errno));
    rfm_mutex_init(fuse_dialog_p->mutex);
    rfm_cond_init(fuse_dialog_p->signal);
    fuse_dialog_p->view_p = widgets_p->view_p;
    rfm_view_thread_create(fuse_dialog_p->view_p, hold_monitor, fuse_dialog_p, "hold_monitor");
    return fuse_dialog_p->signal;
}


static void
button_ok (GtkButton * button, gpointer data) {
    gint *response = data;
    *response = GTK_RESPONSE_APPLY;
    gtk_main_quit();
}

static void
button_mount (GtkEntry * entry, gpointer data) {
    gint *response = data;
    *response = GTK_RESPONSE_YES;
    gtk_main_quit();
}

static void
button_cancel (GtkButton * button, gpointer data) {
    gint *response = data;
    *response = GTK_RESPONSE_CANCEL;
    gtk_main_quit();
}


static gboolean 
response_delete(GtkWidget *dialog, GdkEvent *event, gpointer data){
    gint *response = data;
    *response = GTK_RESPONSE_CANCEL;
    gtk_main_quit();
    return TRUE;
}

static void *
confirm_host_f (gpointer data){
    void **arg = data;
    fuse_data_t *(*dialog_f)(const gchar *url) = arg[0];
    const gchar *url = arg[1];
    const gchar *module_name = arg[2];
    g_free(arg);
    widgets_t * widgets_p = rfm_get_widget("widgets_p");
    GCond *signal = fuse_hold_monitor();
    fuse_data_t *fuse_data_p = (*dialog_f) (url);
    if(!fuse_data_p || !fuse_data_p->dialog){
        return GINT_TO_POINTER(FALSE);
    }
    gint response = GTK_RESPONSE_CANCEL;
    GtkNotebook *notebook = 
	g_object_get_data(G_OBJECT(fuse_data_p->dialog), "notebook");
    gtk_notebook_set_current_page(notebook, 0);

    GtkWidget *button;

    button = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "action_TRUE_button");
    g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (button_ok), &response);
    button = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "action_FALSE_button");
    g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (button_cancel), &response);
    button = g_object_get_data(G_OBJECT(fuse_data_p->dialog), "action_MOUNT_button");
    g_signal_connect (G_OBJECT (button), "clicked", G_CALLBACK (button_mount), &response);

    g_signal_connect (G_OBJECT (fuse_data_p->dialog), "delete-event", G_CALLBACK (response_delete), &response);
retry:;
    
    gtk_widget_show_all (fuse_data_p->dialog);
    gtk_main();
    gtk_widget_hide (fuse_data_p->dialog);

    gboolean retval;
    if(response == GTK_RESPONSE_YES || response == GTK_RESPONSE_APPLY){
        retval = TRUE;
	//gchar *new_url = accept(fuse_data_p, url);
	gchar *new_url = rfm_rational(PLUGIN_DIR, module_name, fuse_data_p, (void*) url, "accept");
	if (!new_url) {
	    goto retry;
	}
	if(response == GTK_RESPONSE_YES) {
	    TRACE( "%s: mount_url\n", module_name);
	    retval = GPOINTER_TO_INT(rfm_rational(PLUGIN_DIR, module_name, widgets_p, new_url, "mount_url"));
	}
	view_t *view_p = widgets_p->view_p;
	record_entry_t *t_en = rfm_copy_entry(view_p->en);
	if(!rodent_refresh (widgets_p, t_en)) {
	    rfm_destroy_entry(t_en);
	}
	g_free(new_url);
    } else {
        retval = FALSE;
    }
	gtk_widget_destroy (fuse_data_p->dialog);
    g_cond_signal(signal);
    return GINT_TO_POINTER(retval);
}

static gboolean
confirm_host (fuse_data_t *(*dialog_f)(const gchar *url), const gchar *url, const gchar *module_name){
    void **arg = (void **)malloc(3*sizeof(void *));
    if (!arg) g_error("malloc: %s\n", strerror(errno));
    arg [0] = dialog_f;
    arg [1] = (void *)url;
    arg [2] = (void *)module_name;
    return GPOINTER_TO_INT(rfm_context_function(confirm_host_f, arg));

}

void *
g_module_check_init(GModule *module){
    NOOP("domain=(no gettext support)");
    GMutex *mutex = rfm_get_widget("fuse_popup_mutex");
    if (!mutex){
	rfm_mutex_init(mutex);
	rfm_set_widget(mutex, "fuse_popup_mutex");
    }
    rfm_view_thread_create(NULL, thread_popup, NULL, "fuse-common:thread_popup");
    return NULL;
}  

