/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#ifdef THIS_IS_BSD
#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/param.h>
#include <sys/mount.h>
#endif

#include "rodent.h"


#if defined(HAVE_GETMNTENT) 
# define LINUX_CONFIG
#else
# if defined(HAVE_GETFSENT) && defined(HAVE_STATFS) && defined(HAVE_GETMNTINFO)
#  define BSD_CONFIG
# endif
#endif



/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "fstab"
#define MODULE_LABEL _("Mount Point")
//#define MODULE_ICON_ID "xffm/emblem_atom/compositeC/stock_harddisk"
#define MODULE_ENTRY_TIP _("Mount local disks and devices")

#include "module-skeleton.h"
// Skeleton definitions
//G_MODULE_EXPORT RFM_G_MODULE_CHECK_INIT

G_MODULE_EXPORT RFM_BLOCKING_LOAD(FALSE)
G_MODULE_EXPORT RFM_MODULE_NAME
//Not applicable:
//G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
//G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

//Not applicable:
//G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY("RODENT-FSTAB");

G_MODULE_EXPORT RFM_IS_ROOT_MODULE(TRUE)

//Superceded:
//G_MODULE_EXPORT RFM_PLUGIN_INFO(_(""))

G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
// disable for tests only.. G_MODULE_EXPORT RFM_MODULE_MONITOR(FALSE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)

//Superceded:
G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

void *is_iso_image(void *p);
void *entry_is_mounted (void *p);
void * is_in_fstab (void *p);
static void set_mounts_info (record_entry_t * en);
#include "fstab-module.i"
#ifdef LINUX_CONFIG
# include "fstab-linux.i"
#else
# ifdef BSD_CONFIG
#  include "fstab-bsd.i"
# endif
#endif

 
G_MODULE_EXPORT
void *
module_icon_id(void){
    return "rodent-fstab";
}


G_MODULE_EXPORT void *
g_module_check_init (GModule * module) {
    BIND_TEXT_DOMAIN;
    stat_hash = g_hash_table_new_full (g_direct_hash, g_direct_equal,NULL, g_free);
    count_hash = g_hash_table_new_full (g_direct_hash, g_direct_equal,NULL, NULL);
    data_hash = g_hash_table_new_full (g_direct_hash, g_direct_equal,NULL, g_free);
    return NULL;
}

G_MODULE_EXPORT void 
g_module_unload(GModule * module){
    g_hash_table_destroy(stat_hash);
    g_hash_table_destroy(count_hash);
    g_hash_table_destroy(data_hash);
}

//
//////////////  Generalized Root Module functions ///////////////////
//

// Plugin functions. Plugin functions may be specified as one of the
// following types.
//
// void:     These functions all return a void pointer
//   	     and take no argument
// natural:  These functions all return a void pointer
// 	     and take a single void pointer argument.
// 	     The space of natural functions is isomorphic
// 	     to the set of natural numbers.
// rational: These functions all return a void pointer
// 	     and take a two void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of rational numbers.
// complex:  These functions all return a void pointer
// 	     and take a three void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of complex numbers with integer
// 	     imaginary component.
	

//////////////////////////////////////////////////////////////////////
//                          void functions                          //
//////////////////////////////////////////////////////////////////////

//  gchar *  
// This function returns a newly allocated string with the general information
// of the module. Rodent uses this to construct the popup tip. Returned value
// should be freed when no longer used.
G_MODULE_EXPORT 
void *
plugin_info (void) {
    return g_strdup_printf("%s: FSTAB\n * %s\n * %s\n * %s\n",
		_("Modules"),
		_("Mount local disks and devices"),
		_("Edit a partition's mount point and options."),
		_("Show mounted volumes on the desktop"));
}


//////////////////////////////////////////////////////////////////////
//                        natural functions                         //
//////////////////////////////////////////////////////////////////////


// gboolean  
// This function processes the double click action on the
// icon's label. If it returns FALSE, then the default
// Rodent action for label click is performed. The default
// Rodent action checks for file existance to do a rename
// entry (or symlink or duplicate).
// Parameter p is the item's entry pointer. 
G_MODULE_EXPORT 
void *label_click (void *p) {
    // Do nothing.
    // record_entry_t *en=p;
    return  GINT_TO_POINTER(1);
}



// const gchar * 
// This function returns a const pointer to the entry's icon
// identifier. 
// Parameter p is the item's entry pointer. 
// Identifier may be returned as a mimetype or an absolute path.
G_MODULE_EXPORT 
void *
item_icon_id (void *p) {
    NOOP(stderr, "fstab item_icon_id call...\n");
    if (!p) {
	NOOP(stderr, "fstab item_icon_id call...1\n");

	return "xffm/stock_dialog-warning";
    }
    record_entry_t *en = p;
    if (IS_DUMMY_TYPE(en->type) || !g_path_is_absolute(en->path)){
	NOOP(stderr, "fstab item_icon_id call...12\n");
	return module_icon_id();
//	return  ("xffm/emblem_atom/compositeC/stock_harddisk");
//	return  ("xffm/stock_harddisk/compositeSW/stock_jump-to");
    }
    gint mounted = GPOINTER_TO_INT(entry_is_mounted(en));
    if (IS_PARTITION_TYPE(en->type)){
	NOOP(stderr, "fstab item_icon_id call...13\n");
	const gchar *id = NULL;
	if (mounted > 0) {
	    id = "xffm/stock_harddisk/compositeNW/emblem_greenball";
	} else if (mounted < 0){
	    id = "xffm/stock_harddisk/compositeNW/emblem_unreadable";
	} else {
	    if (is_in_fstab(en->path)){
		id = "xffm/stock_harddisk/compositeNW/emblem_grayball";
	    }
	    id = "xffm/stock_harddisk/compositeNW/emblem_redball";
	}
	NOOP(stderr, "resolution: %s\n", id);
	return (void *)id;

    } else {
	NOOP(stderr, "fstab item_icon_id call...14\n");
	if (mounted) {
	    return "xffm/stock_directory/compositeNW/emblem_greenball";
	} else if (mounted < 0){
	    return "xffm/stock_directory/compositeNW/emblem_unreadable";
	} else {
	    if (is_in_fstab(en->path)){
		return "xffm/stock_directory/compositeNW/emblem_grayball";
	    } else {
		return "xffm/stock_directory/compositeNW/emblem_redball";
	    }
	}
    }

    return  ("xffm/emblem_chardevice");
}

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// selectable icon or not.
/*G_MODULE_EXPORT 
void *
is_selectable (void *p) {
    record_entry_t *en = (record_entry_t *) p;
    if (!en || !en->path) return NULL;
    if(rfm_g_file_test (en->path, G_FILE_TEST_IS_DIR))
        return (void *)"Yes";
    if(IS_PARTITION_TYPE (en->type))
        return (void *)"Yes";
    return NULL;
}*/

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// valid drop site. Drop may be processed by Rodent or by the
// module if process_drop() is defined.
G_MODULE_EXPORT 
void *
valid_drop_site (void *p) {
    record_entry_t *en = (record_entry_t *) p;
    if(!en || !en->path) return NULL;
    if(IS_SDIR (en->type)) return GINT_TO_POINTER(1);
    return NULL;
}

//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT 
void *
item_entry_tip (void *p) {
    return fstab_entry_tip (p);
}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested record entry
// is a partition type icon.
// Parameter p is the icon's record entry
G_MODULE_EXPORT 
void *
is_partition_type (void *p) {
    record_entry_t *en = p;
    if(!en) return NULL;
    if(IS_PARTITION_TYPE (en->type)){
        return GINT_TO_POINTER (1);
    }
    return NULL;
}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested item is
// mounted or not.
// Parameter p is the item's absolute path.
G_MODULE_EXPORT 
void *
is_mounted (void *p) {
    NOOP("is_mounted()\n");
    return private_is_mounted(p);
}
G_MODULE_EXPORT 
void *
is_mounted_with_wait (void *p) {
    gchar *path = p;
    return is_mounted_with_timeout(path);
}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested item is
// mounted or not.
// Parameter p is the item's entry.
G_MODULE_EXPORT 
void *
entry_is_mounted (void *p) {
    record_entry_t *en = p;
    if (!en) return NULL;
//	return private_is_mounted(en->path);
    if (IS_LOCAL_TYPE(en->type)||IS_PARTITION_TYPE(en->type)
	    || strcmp(en->path, "/")==0) {
        NOOP("entry_is_mounted()\n");
	return private_is_mounted(en->path);
    }
    void *retval = is_mounted_with_timeout(en->path);
    NOOP("entry is mounted = %d\n", GPOINTER_TO_INT(retval));
    return retval;
}

// gchar *
// This is an fstab specific function. 
// The return value is the mount directory for 
// a fstab define fsname. This is a newly allocated
// string. Return value should be freed when no longer
// used.
G_MODULE_EXPORT 
gchar *
get_mnt_dir (gchar * mnt_fsname) {
    return fstab_get_mnt_dir(mnt_fsname);
}

// gchar *
// This is an fstab specific function. 
// The return value is the fsname directory for 
// a fstab define mount directory. This is a newly allocated
// string. Return value should be freed when no longer
// used.
G_MODULE_EXPORT 
gchar *
get_mnt_fsname (gchar * mnt_dir) {
    return fstab_get_mnt_fsname(mnt_dir);

}

// gboolean
// This is an fstab specific function. 
// This function will determine if the requested item is
// in the fstab file or not.
// Parameter p is the item's absolute path.
G_MODULE_EXPORT 
void *
is_in_fstab (void *p) {
    return fstab_is_in_fstab(p);
}


// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT 
void *
module_xfdir_get (void *p) {
    return private_get_xfdir ((xfdir_t *) p);
}

//////////////////////////////////////////////////////////////////////
//                        rational functions                        //
//////////////////////////////////////////////////////////////////////

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
double_click (void *p, void *q) {
    return fstab_double_click(p, q);
}


// gchar *
// This function returns a newly allocated string which shall
// be used by Rodent to complement the initial text which
// appears in the status area (lp command area).
// Returned value should be freed when no longer used.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
sizetag(void *p, void *q){
    //record_entry_t *en=q;
    return g_strdup(_("Manage disks, partitions and file systems"));
}

// gboolean
// This function does a alternate properties dialog or action to
// be called from Rodent's "Properties" icon area popup menu item.
// Parameter p is the view's widgets_p pointer.
// Parameter q is a GList to the record entries of the elements
// for which the properties dialog is to be called with.
G_MODULE_EXPORT 
void *
do_properties (void *p, void *q) {
    return fstab_do_properties(p, q);
}

// gboolean
// This function is used to hide popup menu elements of 
// Rodent's standard icon area popup menu.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
hide_local_menu_items (void *p, void *q) {
    return fstab_hide_local_menu_items(p, q);
}

// gboolean
// This is an fstab specific function.
// This function will attempt to mount the item referenced
// by the record entry pointer parameter.
//
// If the item is listed in the fstab file, then the fstab
// mount point and configuration shall be used.
//
// If the item is not listed in the fstab file, then a dialog
// will ask for the mount point (defaults to ~/mount/[partition]).
//
// If the effective process user does not have privileges to mount
// the device, the the command will be attempted by means of
// "sudo -A".
//
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
fstab_mount (void *p, void *q) {
    //:if expand("%") == ""|browse confirm w|else|confirm w|endif
    //widgets_t *widgets_p = (widgets_t *) p;
    
    void *retval = private_fstab_mount(p, q);
    return retval;
}
#if 10
// Dialog to get mount point.
G_MODULE_EXPORT
gchar *
mnt_point(widgets_t *widgets_p, record_entry_t *en) {
    return get_mnt_point(widgets_p, en);
}
#endif

#ifdef LINUX_CONFIG
G_MODULE_EXPORT
void *
is_net_type (const gchar * mnt_point) {
    FILE *fstab_fd;
    if((fstab_fd = setmntent ("/etc/mtab", "r")) == NULL) {
        DBG ("Unable to open %s\n", "/etc/mtab");
        return FALSE;
    }
    struct mntent *mnt_struct;
    gint net_type = 0;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(!strstr (mnt_struct->mnt_type, MNTTYPE_NFS) && !strstr (mnt_struct->mnt_type, MNTTYPE_SMBFS))
            continue;
        if(!rfm_g_file_test (mnt_struct->mnt_dir, G_FILE_TEST_IS_DIR))
            continue;
        if(strcmp (mnt_point, mnt_struct->mnt_dir) == 0) {
            if(strstr (mnt_struct->mnt_type, MNTTYPE_NFS)) net_type = 1;
	    else net_type = 2;
            break;
        }
    }
    (void)endmntent (fstab_fd);
    return GINT_TO_POINTER(net_type);
}

G_MODULE_EXPORT
void *
mnt_type (const gchar * mnt_point) {
    FILE *fstab_fd;
    gchar *mnt_type = NULL;
    if((fstab_fd = setmntent ("/etc/mtab", "r")) == NULL) {
        DBG ("Unable to open %s\n", "/etc/mtab");
        return NULL;
    }
    struct mntent *mnt_struct;
    struct mntent mntbuf;
    gchar buf[2048]; 
    while ((mnt_struct = getmntent_r (fstab_fd, &mntbuf, buf, 2048)) != NULL) {
        if(strcmp (mnt_point, mnt_struct->mnt_dir) == 0) {
	    mnt_type = g_strdup(mnt_struct->mnt_type);
            break;
        }
    }
    (void)endmntent (fstab_fd);
    return (mnt_type);
}
#endif

G_MODULE_EXPORT
void *is_iso_image(void *p){
    if (!p) return NULL;
    record_entry_t *en = p;
    if (en->mimetype){
	if (strstr(en->mimetype, "application/x-iso9660-image")) {
	    return GINT_TO_POINTER(1);
	}
	if (strstr(en->mimetype, "application/x-cd-image")) {
	    return GINT_TO_POINTER(1);
	}
    }
    if (en->mimemagic){
	if (strstr(en->mimemagic, "application/x-iso9660-image")) {
	    return GINT_TO_POINTER(1);
	}
	if (strstr(en->mimemagic, "application/x-cd-image")) {
	    return GINT_TO_POINTER(1);
	}
    }    
    return NULL;
}


static off_t
st_sum (struct stat *st) {
    off_t sum;
    if(!st) sum = 0;
    else {
        sum = st->st_mtime + st->st_size + st->st_mode + st->st_nlink + st->st_uid + st->st_gid;
    }
    return sum;
}

// gint
// This function returns the count of elements in the module's
// view. This value is compared with current value to determine
// whether a reload is necessary.
// Parameter p is the view's widgets_p.
// Parameter p is the view's view_p record entry.
static gint
count_items (void) {
    gint items = count_elements () + count_partitions ();
    // Add link to fuse and up item
    items ++;
    if (rfm_void(PLUGIN_DIR, "fuse", "module_active")) items++;
    
    NOOP( "fstab items=%d\n", items);
    return items;
}

// gchar *
// This function returns a newly allocated string which will be 
// compared by the monitor function to determine whether a reload 
// is necessary.
// Parameter p is the view's view_p record entry.
// Return value should be freed when no longer used
static gchar *
fstab_data (void) {
    return fstab_df ();
}

void *
reload(void *view_p){
    gboolean reload_condition = FALSE;
// Test 1: stat condition
    if (rfm_get_gtk_thread() == g_thread_self()){
	g_warning("fstab module: reload() is a thread function only\n");
        return FALSE;
    }
    if (fstab_module_stat(NULL, NULL)){
	struct stat *new_stat;
	new_stat = (struct stat *)malloc(sizeof(struct stat));
	if (!new_stat) g_error("malloc: %s\n", strerror(errno));
	memset (new_stat, 0, sizeof(struct stat));
	fstab_module_stat(NULL, new_stat);
	GMutex *stat_hash_mutex = get_stat_hash_mutex();
	g_mutex_lock(stat_hash_mutex);
	struct stat *st = g_hash_table_lookup(stat_hash, view_p);
	if (!st) {
	    g_hash_table_insert(stat_hash, view_p, new_stat);
	} else {
	    if (st_sum(st) != st_sum(new_stat)){
		reload_condition = TRUE;
		g_hash_table_replace(stat_hash, view_p, new_stat);
	    }
	}
	g_mutex_unlock(stat_hash_mutex);
	if (reload_condition) return GINT_TO_POINTER(TRUE);
    }
// Test 2: count condition
    gint items = count_items();
    GMutex *count_hash_mutex = get_count_hash_mutex();
    g_mutex_lock(count_hash_mutex);
    void *count_p = g_hash_table_lookup(count_hash, view_p);
    if (!count_p) {
	g_hash_table_insert(count_hash, view_p, GINT_TO_POINTER(items));
    } else {
	if (items != GPOINTER_TO_INT(count_p)){
	    reload_condition = TRUE;
	    g_hash_table_replace(count_hash, view_p, GINT_TO_POINTER(items));
	}
    }
    g_mutex_unlock(count_hash_mutex);
    if (reload_condition) return GINT_TO_POINTER(TRUE);
// Test 3: data condition
    GMutex *data_hash_mutex = get_data_hash_mutex();
    g_mutex_lock(data_hash_mutex);
    gchar *data = fstab_data();
    gchar *data_p = g_hash_table_lookup(data_hash, view_p);
    if (!data_p) {
	g_hash_table_insert(data_hash, view_p, data);
    } else {
	if (strcmp(data, data_p)){
	    reload_condition = TRUE;
	    g_hash_table_replace(data_hash, view_p, data);
	} else g_free(data);
    }
    g_mutex_unlock(data_hash_mutex);
    return GINT_TO_POINTER(reload_condition);
}

