#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
//
//
#define POPUP_ID "ps_menu"
#define POPUP_MENU_ID "ps_menu_menu"

static GMutex *
get_reload_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}

static gchar **
compact_split(gchar **split){
    gint count=0;
    while (split && split[count]) count++;
    gchar **c_split =(gchar **)malloc((count+1)*sizeof(gchar *));
    if (!c_split) g_error("malloc: %s\n", strerror(errno));

    gchar **src=split;
    gchar **tgt=c_split;
    for (;src && *src; src++){
	if (strlen(*src)){
	    *tgt=*src;
	    NOOP ("----> %s\n", *tgt);
	    tgt++;
	}
    }
    *tgt=NULL;
    return c_split;
}

static 
void
ps_nice(GtkMenuItem *m, gpointer data){
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;
    if (!view_p->selection_list) return;

    record_entry_t *en=(record_entry_t *)view_p->selection_list->data;

    if (!en || !en->st) return;
    gchar pid[64];
    gchar ni[64];
    sprintf(pid,"%d",(int)en->st->st_uid);
    sprintf(ni,"+%d",(int)en->st->st_ctime+1);
    gchar *argv[]={"renice", ni, "-p", pid, NULL};
    rfm_show_text(widgets_p);
    rfm_thread_run_argv(widgets_p, argv, FALSE);
    return;
}

static 
void
ps_signal(GtkMenuItem *m, gpointer data){
    pid_t pid =  GPOINTER_TO_INT(rfm_get_widget("ps_uid"));
    if (!pid) return;
    // Is the effective runtime uid the same as that for the process to signal?
    
    // 1. Are we executing the command in a shell chain?
    gchar *pcommand = g_strdup_printf("ps ax -o ppid,pid");
    FILE *p = popen(pcommand, "r");
    if (!p){
        g_warning("pipe creation failed for %s\n", pcommand);
        g_free(pcommand);
        return;
    } 
    g_free(pcommand);
    gint cpid = -1;
    gchar *spid = g_strdup_printf("%0d", pid);
    gchar buffer[64];
    memset(buffer, 0, 64);
    while (fgets(buffer, 63, p) && !feof(p)){
        if (strncmp(buffer, spid, strlen(spid))==0){
            DBG("gotcha: %s", buffer);
            gchar **gg=g_strsplit(buffer, " ", -1);
            errno = 0;
            long l = strtol(gg[1], NULL, 10);
            if (errno) {
                g_warning("cannot parse to long: %s\n", gg[1]);
                pclose(p);
                g_free(spid);
                g_strfreev(gg);
                return;
            }
            cpid = l;
            g_strfreev(gg);
            break;
        }
    }
    pclose(p);
    // If cpid turns out > 0, then we are in a chained command and pid must change
    if (cpid > 0) {
        pid = cpid;
        g_free(spid);
        spid = g_strdup_printf("%0d", pid);
    }

    // 2. Does pid (either direct or from chained command) belong to us?
    pcommand = g_strdup_printf("ps -p %d -o uid", (int)pid);
    gboolean sudoize = FALSE;
    uid_t uid = geteuid();
    p = popen(pcommand, "r");
    long luid = geteuid();
    if (!p){
        g_warning("pipe creation failed for %s\n", pcommand);
    } else {
        gchar buffer[64];
        memset(buffer, 0, 64);
        while (fgets(buffer, 63, p) && !feof(p)){
	    if (strstr(buffer, "UID")) continue;
            errno=0;
            luid = strtol(buffer, NULL, 10);
            if (!errno){
		DBG("line: %s gotcha: %ld\n", buffer, luid);
                if (luid != uid) sudoize = TRUE;
                break;
            }
        }
        pclose(p);
    }
    g_free(pcommand);
    gchar *sudo = g_find_program_in_path("sudo");
    DBG("signal to pid: %s (owned by %d, sudo=%d)\n", spid, (int)luid, sudoize);
    gint sig=GPOINTER_TO_INT(data);
    if (sudoize && sudo) {
        DBG("ps_signal by means of sudo...\n");
        widgets_t *widgets_p = rfm_get_widget ("widgets_p");
        gchar *signal_number = g_strdup_printf("-%0d", sig);
        gchar *argv[]={sudo, "-A", "kill", signal_number, spid, NULL};
        rfm_thread_run_argv (widgets_p, argv, FALSE);
        g_free(signal_number);
    } else {
        if (!sudo) g_warning("sudo command not found to signal non-owned process\n");
        DBG("normal ps_signal to %d...\n", (int)pid);
        kill(pid, sig);
    }
    rfm_rational(RFM_MODULE_DIR, "callbacks", GINT_TO_POINTER(REFRESH_ACTIVATE), NULL, "callback");
    g_free(sudo);
    g_free(spid);
    return;
}

#if 0
// If we use this to fire up the main menu, then only top menu elements
// work with mouse. Submenu elements only work with keyboard.
static void *
main_menu_f(void *data){
    GdkEventButton  event;
    event.state = GDK_CONTROL_MASK;
    event.time = gtk_get_current_event_time();
    rodent_pop_menu ("main_popup_menu", &event);
    return NULL;
}

static 
void
ps_main_menu(GtkMenuItem *m, gpointer data){
    widgets_t * widgets_p = rfm_get_widget("widgets_p");
    gchar *text = g_strdup_printf("\nCTRL-CLICK: %s\n", _("Opens the main menu."));
    rfm_confirm (widgets_p, GTK_MESSAGE_INFO, text, NULL, NULL);
}
#endif

static 
void
ps_tree(GtkMenuItem *m, gpointer data){
    gchar *ps=NULL;
    widgets_t *widgets_p=rfm_get_widget("widgets_p");
    gint ps_module_flags = GPOINTER_TO_INT(rfm_get_widget("ps_module_flags"));
#ifdef  HAVE_BSD_XF
    if (ps_module_flags & ALL_PROCS) ps=g_strdup_printf("ps axf");
    else ps=g_strdup_printf("ps xf");
#else
#ifdef HAVE_UNIX_JH
    if (ps_module_flags & ALL_PROCS) ps=g_strdup_printf("ps -jHe");
    else ps=g_strdup_printf("ps -jHU %u",getuid()); 

#endif
#endif
    if (!ps){
	DBG("neither ps xf nor ps -jHU is available\n");
	return;
    }

    rfm_show_text(widgets_p);
    rfm_diagnostics(widgets_p,"xffm/stock_yes",ps,"\n",NULL);
    RFM_THREAD_RUN2ARGV (widgets_p, ps, FALSE);
    g_free(ps);
    return;
}
static 
void
ps_close(GtkMenuItem *m, gpointer data){
    rfm_rational(RFM_MODULE_DIR, "callbacks", GINT_TO_POINTER(QUIT_ACTIVATE), m, "callback");

}

static 
void
ps_set(GtkMenuItem *m, gpointer data){
    widgets_t *widgets_p=rfm_get_widget("widgets_p");

    view_t *view_p=widgets_p->view_p;
    gint flag=GPOINTER_TO_INT(data);
    gint ps_module_flags = GPOINTER_TO_INT(rfm_get_widget( "ps_module_flags"));
    ps_module_flags ^= flag;
    ps_module_flags |= DUMMY_FLAG;

    rfm_set_widget(GINT_TO_POINTER(ps_module_flags), "ps_module_flags");
        
    // reset to module root
    //
    GMutex *reload_mutex = get_reload_mutex();
    g_mutex_lock(reload_mutex);
    g_free(view_p->en->st);
    view_p->en->st=NULL;
    //view_p->xfdir_p->invalidate_icons=TRUE;
    g_mutex_unlock(reload_mutex);
    //
    // refresh (leave to monitor...)
    // 
    rfm_rational(RFM_MODULE_DIR, "callbacks", GINT_TO_POINTER(REFRESH_ACTIVATE), NULL, "callback");

    // deprecated: rodent_refresh_activate (m, NULL);

    return;
}

static
void *
private_date_column_string(void * p){
    gchar *tag;
    record_entry_t *en=p;
    if (!en || !en->st) return ("");

    tag=g_strdup_printf("%02d:%02d:%02d",
	    (int)(en->st->st_mtime/3600),
	    (int)((en->st->st_mtime%3600)/60),
	    (int)((en->st->st_mtime%3600)%60)
	    );
    return tag;
}

static
void *
private_size_column_string(void * p){
    gchar *tag=NULL;
    record_entry_t *en=p;
    if (!en || !en->st) return ("");

    if (!p) return ("");
    tag=g_strdup_printf("%d",(int)en->st->st_size);
    return tag;
}

static void *
ps_entry_tip(void *p)
{
    gchar *tip;
    record_entry_t *en=p;
    if (!en || !en->path || !strlen(en->path) || !en->st) return NULL;

    const gchar *status="";
    switch (en->st->st_rdev) {
		case SLEEPING:
		    status = _("Sleeping"); break;
		case DEEPSLEEP:
		    status = _("Sleep"); break;
		case RUNNING:
		    status = _("Running"); break;
		case STOPPED:
		    status = _("Stopped"); break;
		case ZOMBIE:
		    status = _("Zombie"); break;
    }
     
    tip=g_strdup_printf("%s: %s\n\n\
%s = %d\n\
%s = %d\n\
%s = %d\n\
%s = %s\n\
%s = %s\n\
%s = %d \n\
%s = %d%%\n\n\
%s = %s\n\n\
%s",
        _("Command"), (strchr(en->path,':'))?strchr(en->path,':')+1:en->path,

	_("Children"),(gint) en->st->st_nlink,
	_("The PID of the program"), en->st->st_uid,
	_("Parent ID"), en->st->st_gid,

	_("Resident Memory"),(gchar *)private_size_column_string(p),
	_("CPU Time"),(gchar *)private_date_column_string(p),

	_("Niceness"),(gint)(en->st->st_ctime),  
	_("CPU Usage"),en->st->st_mode,
	_("Status"), status,
	_("The priority of a process is given by its nice value. A lower nice value corresponds to a higher priority.")
	);

    return (void *)tip;
}

static void *
ps_item_icon_id (void *p) {

    int f=0;
    record_entry_t *en=(record_entry_t *)p;

    if(!en || IS_UP_TYPE (en->type)) {
        return "xffm/stock_go-up";
    }
    static gchar *icon=NULL;
    if (en->path && strcmp(en->path,_("System Processes"))==0) {
	if (icon == NULL){
	    icon = 
		g_strdup_printf("%s/pixmaps/rodent-ps.svg", PACKAGE_DATA_DIR);
	}
	return icon;
    }

    if (!en->st) return "xffm/emblem_chardevice";

    if (en->st->st_gid ==1)f = 0x01; // PPID is init daemon. Follows no one.
    else 		   f = 0x02; // Parent is not init 
    if (en->st->st_nlink) f |= 0x04; // has followers.
    if (en->st->st_blocks) f |= 0x08; // is following.
    NOOP ("ps_item_icon_id: %s --> 0x%x\n",(en->path)? en->path:"null", f);
    NOOP(stderr, "ps_item_icon_id: %s...0x%x\n", en->path, f);

    switch (f) {
	case 0x09: // showing all processes, has no followers

	case 0x0a: // non-init process with only parent
	case 0x01: // init process without children
	case 0x02: // non-init process without children or known parents
	    switch (en->st->st_rdev) {
		case SLEEPING:
		case DEEPSLEEP:
		    if (en->st->st_ctime > 0) {
			return "xffm/stock_execute/compositeNE/emblem_grayball";
		    }
		    return "xffm/stock_execute";
		case RUNNING:
		    if (en->st->st_ctime > 0) {
			return "xffm/stock_execute/compositeNE/emblem_blueball";
		    }
		    return "xffm/stock_execute/compositeNE/emblem_greenball";
		case STOPPED:
		    return "xffm/stock_execute/compositeNE/emblem_redball";
		case ZOMBIE:
		    return "xffm/stock_execute/compositeNE/emblem_core";
	    }
	    return "xffm/stock_execute";

	case 0x0d: // showing all processes, has followers

	case 0x0e: // non-init process with parent and children
	case 0x06: // non-init process with only children
	case 0x05: // init process with children   
	    switch (en->st->st_rdev) {
		case SLEEPING:
		case DEEPSLEEP:
		    if (en->st->st_ctime > 0) {
			return "xffm/stock_directory/compositeC/stock_execute/emblem_grayball";
		    }
		    return "xffm/stock_directory/compositeC/stock_execute";
		case RUNNING:
		    if (en->st->st_ctime > 0) {
			return "xffm/stock_directory/compositeC/stock_execute/emblem_blueball";
		    }
		    return "xffm/stock_directory/compositeC/stock_execute/emblem_greenball";
		case STOPPED:
		    return "xffm/stock_directory/compositeC/stock_execute/emblem_redball";
		case ZOMBIE:
		    return "xffm/stock_directory/compositeC/stock_execute/emblem_core";
	    }
	    return "xffm/stock_directory/compositeC/stock_execute";
    }

    return "xffm/emblem_disk";
}


static
int parse_int(gchar *string){
    gint r=atoi(string);
    return r;
}

static
long parse_long(gchar *string){
    gchar *f=string; 
    if (!f) {
        DBG("incorrect parsing...\n");
	return 0;
    }
    return atol(f);
}

static gchar *short_name(gchar *full_label){
    gchar *modified_label = g_strdup(strchr(full_label,':')+1);
    if (strchr(modified_label, ' ')) *strchr(modified_label, ' ') = 0;
    if (strchr(modified_label, G_DIR_SEPARATOR)){
	gchar *g = g_strdup(strrchr(modified_label, G_DIR_SEPARATOR)+1);
	g_free(modified_label);
	modified_label = g;
    }
    return modified_label;
}

static gint
ps_xfdir_get (void *pp) {
    GSList *ps_list=NULL;
    xfdir_t *xfdir_p = pp;
    FILE *pipe;
    GSList *tmp;
    gint i;

    
    NOOP(stderr, "module_xfdir_get\n");
    record_entry_t *en;
    if (!xfdir_p->en) {
	NOOP(stderr, "entry is null!\n");
	return 0;
    }
    
    record_entry_t *p_en=rfm_copy_entry(xfdir_p->en);
    gint parent_pid=0;
    if (p_en && p_en->st) {
	parent_pid=p_en->st->st_uid;
	// this is very important, since xfdir_p->st gets
	// put into view_p->en->st, which is used for reload
	memcpy(xfdir_p->en->st, p_en->st, sizeof(struct stat));
    }
    NOOP(stderr, "ps-module: parent is %d\n",parent_pid);

    gint ps_module_flags = GPOINTER_TO_INT(rfm_get_widget("ps_module_flags"));    



    gchar *ps=NULL;
#ifdef HAVE_UNIX_O_RSS
    const gchar *fmt="state,pid,ppid,pgid,pcpu,nice,rss,time,args";
#else 
    const gchar *fmt="state,pid,ppid,pgid,pcpu,nice,vsz,time,args";
#endif


    
#ifdef HAVE_UNIX_O
 #ifdef HAVE_UNIX_W
    if (ps_module_flags & ALL_PROCS) 
	ps=g_strdup_printf("ps -wwe -o %s",fmt);
    else
	ps=g_strdup_printf("ps -wwU %u -o %s",(unsigned)getuid(),fmt);
 #else
    if (ps_module_flags & ALL_PROCS) 
	ps=g_strdup_printf("ps -e -o %s",fmt);
    else
	ps=g_strdup_printf("ps -U %u -o %s",(unsigned)getuid(),fmt);
 #endif
#else
 #ifdef HAVE_BSD_O
  #ifdef HAVE_BSD_W
    if (ps_module_flags & ALL_PROCS) 
	ps=g_strdup_printf("ps awwo %s",fmt);
    else
	ps=g_strdup_printf("ps wwo %s",fmt);
  #else
    if (ps_module_flags & ALL_PROCS) 
        ps=g_strdup_printf("ps ao %s",fmt);
    else
        ps=g_strdup_printf("ps o %s",fmt);
  #endif
 #endif   
#endif
     
    NOOP(stderr, "ps=%s\n",ps);
    GHashTable *visible_processes=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    GHashTable *parent_hash=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    GHashTable *entry_hash=g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);

    xfdir_p->pathc=0;
    pipe = popen(ps, "r");
    g_free(ps);
    if(pipe) {
	gchar *f;
	gchar line[4096];
	memset(line,0,4096);
	while (fgets(line, 4095, pipe)){
	    if (!strchr(line,'\n')){
		DBG("memory overrun averted (this never happens)\n");
		break;
	    }
	    if (!strstr(line,"PPID")) {
		en=rfm_mk_entry(0);
		en->type=0; /* remove local-type attributes */
		en->st = (struct stat *)malloc(sizeof(struct stat));
		if (!en->st) g_error("malloc: %s\n", strerror(errno));
		memset(en->st,0,sizeof(struct stat));
		NOOP("line=%s",line);
		// split
		//gchar **src_split=rfm_split(line,' ');
		gchar **src_split = g_strsplit(line, " ", -1);
		if (!src_split) {
		    DBG("ps-module ps_xfdir_get(): split barfed\n");
		}
		gint is=0;
	    
		gchar **split=compact_split(src_split);
		/* state: */
		switch (*split[is]) {
		    case 'Z': //zombie
			en->st->st_rdev=ZOMBIE;
			break;
		    case 'T': //stopped
			en->st->st_rdev=STOPPED;
			break;
		    case 'R': //running
			en->st->st_rdev=RUNNING;
			break;
		    case 'S': //Interruptible sleep (waiting for an event to complete)
			en->st->st_rdev=SLEEPING;
			break;
		    case 'D': //Uninterruptible sleep (usually IO)
			en->st->st_rdev=DEEPSLEEP;
			break;		    
		}
		parse_int(split[is++]); //PID
		/* PID: */
		en->st->st_uid=(uid_t)parse_int(split[is++]); //PID

		/* PPID: */
		en->st->st_gid=(gid_t)parse_int(split[is++]); //PPID

		// Feed hash tables
		g_hash_table_replace(visible_processes, 
			g_strdup_printf("%d", (gint) en->st->st_uid),
			GINT_TO_POINTER(1)); 
		g_hash_table_replace(parent_hash, 
			g_strdup_printf("%d", (gint) en->st->st_gid),
			GINT_TO_POINTER(1)); 
		g_hash_table_replace(entry_hash, 
			g_strdup_printf("%d", (gint) en->st->st_uid),
			en); 



		/* PGID: */

		en->st->st_ino=(ino_t)parse_int(split[is++]); //PGID
		/* PCPU */

		en->st->st_mode=(mode_t)parse_int(split[is++]);

		/* NICE */

		en->st->st_ctime=(time_t)parse_int(split[is++]);
		NOOP("time=%d\n",(int)en->st->st_ctime);
		/* RSS/VSZ */
		
		en->st->st_size=(off_t)parse_long(split[is++]);
		/* TIME */
		{
		    gint h=0,m=0,s=0;
		    gchar *xp,*c=split[is++];
		    xp=strchr(c,':');
		    if (!xp) goto ko;
		    *xp=0;
		    h=atoi(c);
		    c=xp+1;
		    xp=strchr(c,':');
		    if (!xp) {
			m=h;
			h=0;
			s=atoi(c);
			goto ko;
		    } else {
			*xp=0;
			m=atoi(c);
			c=xp+1;
			s=atoi(c);
		    }
ko:
		    en->st->st_mtime=(time_t)(h*3600+m*60+s);
		}
		/* ARGS */
		f=g_strdup(split[is++]); 
		while (split[is]) {
		    gchar *ff=g_strconcat(f," ",split[is++],NULL);
		    g_free(f);
		    f=ff;
		}
		if (strchr(f, '\n')) *strchr(f, '\n')=0;
		//en->path=g_strdup(f);
		en->path=g_strdup_printf("%d:%s",en->st->st_uid,f);
		g_free(f);

		NOOP("path=%s\n",en->path);

		//if (strchr(f,' ')) strtok(f," ");
		en->module=MODULE_NAME;
		ps_list=g_slist_append(ps_list,en);
		//g_free(split[0]); g_free(src_split);
		g_free(split);
		g_strfreev(src_split);
		//NOOP("pid=%d ppid=%d gid=%d\n",en->st->st_uid, en->st->st_gid, (gint)en->st->st_ino);
	    }
	}		
	pclose(pipe);
    } else {
	DBG("rodent-ps: unable to pipe\n");
    }
	
    /* tag parent and child processes...*/

    if (!(ps_module_flags & PROC_CHILDREN)) {
	for (tmp=ps_list; tmp; tmp=tmp->next){
	    en=(record_entry_t *)tmp->data;
	    gchar *key = g_strdup_printf("%d", en->st->st_uid);
	    // is en a parent?
	    if (g_hash_table_lookup(parent_hash, key)) {
		    SET_SDIR(en->type);
		    en->st->st_nlink++; // yes
	    }
	    g_free(key);
	    // is en a child?
	    gchar *p_key = g_strdup_printf("%d", en->st->st_gid);
	    if (g_hash_table_lookup(visible_processes, p_key)) {
		en->st->st_blocks++;
	    }
	    g_free(p_key);
	}
    }

    // select appropriate elements
    GSList *list=NULL;
    for (tmp=ps_list; tmp; tmp=tmp->next){
	en=(record_entry_t *)tmp->data;
	if (parent_pid){
	    // Only children of parent_pid considered.
	    if (parent_pid == en->st->st_gid){
		NOOP("PS: 1.pid=%d adding %d = %s\n", parent_pid, en->st->st_uid, (const char *)en->path);
		list=g_slist_prepend(list, tmp->data);
	    }
	} else  if (en->st->st_blocks==0){
	    // Add if process if it is an orphan.
	    NOOP("PS: 2. adding %d = %s\n",en->st->st_uid, (const char *)en->path);
	    list=g_slist_prepend(list, tmp->data);
	}
    }

    xfdir_p->pathc=g_slist_length(list);	
    // up item: 
    xfdir_p->pathc++;

    xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
    if (!xfdir_p->gl) g_error("malloc: %s\n", strerror(errno));
    memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));

    // Up icon determination.
    if (parent_pid && xfdir_p->pathc > 1) {
	// Parent specified on function call.
	gchar *p_key = g_strdup_printf("%d", parent_pid);
	// Is the parent mapable?
	if (g_hash_table_lookup(visible_processes, p_key)) {
	    // Get the grandparent.
	    record_entry_t *p_en = g_hash_table_lookup(entry_hash, p_key);
	    gchar *g_key = g_strdup_printf("%d", p_en->st->st_gid);
	    if (g_hash_table_lookup(visible_processes, g_key)) {
		// Up is the grandparent.
		// Parent has a parent too.
		record_entry_t *g_en = g_hash_table_lookup(entry_hash, g_key);
		xfdir_p->gl[0].en=rfm_copy_entry(g_en);
		SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
		SET_UP_TYPE(xfdir_p->gl[0].en->type);
		NOOP(stderr, "PS: up set to %d\n",xfdir_p->gl[0].en->st->st_uid);
		xfdir_p->gl[0].pathv = g_strdup (g_en->path);
	    } else {
		// No grandparent.
		// Up is the module root.
		// The parent an orphan. 
		xfdir_p->gl[0].en=rfm_mk_entry(0);
		xfdir_p->gl[0].en->st = NULL;	
		xfdir_p->gl[0].en->path=g_strdup(MODULE_LABEL);
		xfdir_p->gl[0].en->tag = g_strdup(MODULE_LABEL);
		xfdir_p->gl[0].en->module = MODULE_NAME;
		SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
		SET_UP_TYPE(xfdir_p->gl[0].en->type);
		NOOP(stderr, "PS: parent is an orphan. Up is set to module root.\n");
		xfdir_p->gl[0].pathv = g_strdup(MODULE_LABEL);	  
	    }
	    g_free(g_key);
	}
	g_free(p_key);
    } else if (!parent_pid) {
	// No parent specified on function call.
	// Up is rodent root level.
	xfdir_p->gl[0].en = NULL;
	NOOP(stderr, "PS: up set to root level\n");
	xfdir_p->gl[0].pathv = g_strdup (g_get_host_name ());
    } else {
	// Parent pid is invalid
	// (nothing is loaded).
	xfdir_p->gl[0].en=rfm_mk_entry(0);
	xfdir_p->gl[0].en->st = NULL;	
	xfdir_p->gl[0].en->path=g_strdup(MODULE_LABEL);
	xfdir_p->gl[0].en->tag = g_strdup(MODULE_LABEL);
	xfdir_p->gl[0].en->module = MODULE_NAME;
	SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
	SET_UP_TYPE(xfdir_p->gl[0].en->type);
	NOOP(stderr, "PS: parent is an orphan. Up is set to module root.\n");
	xfdir_p->gl[0].pathv = g_strdup(MODULE_LABEL);	  

    }
    
    // add the rest of the items to the xfdir structure.
    for (i=1,tmp=list; tmp && tmp->data; tmp=tmp->next,i++){
	en=(record_entry_t *)tmp->data;
	xfdir_p->gl[i].en=rfm_copy_entry(en);
	//xfdir_p->gl[i].pathv = g_strdup(xfdir_p->gl[i].en->path);
	xfdir_p->gl[i].en->tag = short_name(xfdir_p->gl[i].en->path);
	xfdir_p->gl[i].pathv = short_name(xfdir_p->gl[i].en->path);
        xfdir_p->gl[i].en->module = MODULE_NAME;
	NOOP("%d = %s\n",xfdir_p->gl[i].en->st->st_uid, (const char *)xfdir_p->gl[i].en->path);
	//NOOP("pid=%d ppid=%d gid=%d\n",en->st->st_uid, en->st->st_gid, (gint)en->st->st_ino);
    }

    // Cleanup

    g_hash_table_destroy(visible_processes);
    g_hash_table_destroy(parent_hash);
    g_hash_table_destroy(entry_hash);
    
   
    g_slist_free(list);

    for (tmp=ps_list; tmp; tmp=tmp->next){
	en=(record_entry_t *)tmp->data;
	rfm_destroy_entry(en);
    }
    g_slist_free(ps_list);


/*    for (i=0; i<xfdir_p->pathc; i++){
	if (xfdir_p->gl[i].en && xfdir_p->gl[i].en->st) {
	    NOOP("*0x%xd = %s\n",GPOINTER_TO_INT(xfdir_p->gl[i].en->st), (const char *)xfdir_p->gl[i].en->path);
	    NOOP("pid=%d ppid=%d gid=%d\n",xfdir_p->gl[i].en->st->st_uid, 
		    xfdir_p->gl[i].en->st->st_gid, (gint)xfdir_p->gl[i].en->st->st_ino);
		NOOP("pid=%d RSS=%d\n", 
			(gint) xfdir_p->gl[i].en->st->st_uid,
			(gint) xfdir_p->gl[i].en->st->st_size);
	}
    }*/
    rfm_destroy_entry(p_en);
    return 1;
}
#if 0
	{MENUITEM_TYPE,"ps_menu_menu","main menu ps",\
	    {0x10001, N_("Opens the main menu."),"xffm/emblem_applications",\
	    (gpointer) ps_main_menu, NULL,0,0}},\

#endif
#define PS_MENU_STUFF \
	{MENUITEM_TYPE,"ps_menu_menu","Display process tree",\
	    {0x1001, N_("Display process tree"),"xffm/emblem_terminal",\
	    (gpointer) ps_tree, NULL,0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Show all processes",\
	    {0x1002, N_("Show all processes"),"xffm/emblem_greenball",\
	    (gpointer) ps_set, GINT_TO_POINTER(ALL_PROCS),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Show user own process",\
	    {0x1003, N_("Show user own process"),"xffm/emblem_redball",\
	    (gpointer) ps_set, GINT_TO_POINTER(ALL_PROCS),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Monitor child processes",\
	    {0x1004, N_("Monitor child processes"),"xffm/emblem_greenball",\
	    (gpointer) ps_set, GINT_TO_POINTER(PROC_CHILDREN),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Show parent/child relationship between processes",\
	    {0x1005, N_("Show parent/child relationship between processes"),"xffm/emblem_redball",\
	    (gpointer) ps_set, GINT_TO_POINTER(PROC_CHILDREN),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Close ps",\
	    {0x1005, N_("Close"),"xffm/stock_close",\
	    (gpointer) ps_close, NULL,0,0}},\
\
	{MENUITEM_TYPE,"ps_menu_menu","Renice Process",\
	    {0x1011, N_("Renice Process"),"xffm/emblem_wait",\
	    (gpointer) ps_nice, NULL,0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Suspend (STOP)",\
	    {0x1006, N_("Suspend (STOP)"),"xffm/stock_execute/compositeNE/emblem_redball",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGSTOP),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Continue (CONT)",\
	    {0x1007, N_("Continue (CONT)"),"xffm/stock_execute/compositeNE/emblem_greenball",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGCONT),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Interrupt (INT)",\
	    {0x1008, N_("Interrupt (INT)"),"xffm/emblem_cancel",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGINT),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Hangup (HUP)",\
	    {0x1009, N_("Hangup (HUP)"),"xffm/stock_refresh",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGHUP),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","User 1 (USR1)",\
	    {0x100a, N_("User 1 (USR1)"),"xffm/emblem_blueball",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGUSR1),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","User 2 (USR2)",\
	    {0x100b, N_("User 2 (USR2)"),"xffm/emblem_blueball",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGUSR2),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Terminate (TERM)",\
	    {0x100c, N_("Terminate (TERM)"),"xffm/emblem_cancel",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGKILL),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Kill (KILL)",\
	    {0x100d, N_("Kill (KILL)"),"xffm/emblem_redball",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGKILL),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Segmentation fault",\
	    {0x100e, N_("Segmentation fault"),"xffm/emblem_core",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGSEGV),0,0}},\
	{NULL_TYPE} 

#if 0
	{MENUITEM_TYPE,"ps_menu_menu","Terminate-TERM",\
	    {0x100a, N_("Terminate (TERM)"),"xffm/emblem_cancel",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGUSR1),0,0}},\
	{MENUITEM_TYPE,"ps_menu_menu","Kill-KILL",\
	    {0x100b, N_("Kill (KILL)"),"xffm/emblem_redball",\
	    (gpointer) ps_signal, GINT_TO_POINTER(SIGUSR2),0,0}},\

#endif
static void *
thread_popup(void *data){
    if (!rfm_global()) return NULL;
    if (rfm_get_gtk_thread() == g_thread_self()){
	g_warning("thread_mk_popup_menu: only to be called from non main thread\n");
        return NULL;
    }
    GtkWidget *popup_widget=rodent_thread_add_submenu (NULL,  "rodent-ps", POPUP_ID,  NULL);  

    RodentMenuDefinition menu_definitions_p[] = {PS_MENU_STUFF};
    rodent_thread_multimenu_make(NULL, menu_definitions_p); 
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;
    xfdir_register_popup(view_p, popup_widget);
    rfm_set_widget(GINT_TO_POINTER(DUMMY_FLAG), "ps_module_flags");

    return popup_widget;
}

static void *
ps_popup(void *p, void *q){
    GtkWidget *popup_widget=rfm_get_widget(POPUP_MENU_ID);

     if (!popup_widget) {
	TRACE( "!popup_widget:ps\n");
	return GINT_TO_POINTER(1);
    }

    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;

    record_entry_t *en = q;

    // If entry not specified, use the entry specified by
    // a single selection list.
    if (!en && g_slist_length(view_p->selection_list)==1){
	en = view_p->selection_list->data;
    }
 
    while (!rfm_rw_lock_reader_trylock (&(view_p->mutexes.monitor_lock))){
	gtk_main_iteration();
    }

    gchar *menu_text = NULL; 


	HIDE_IT("main menu ps");
	HIDE_IT("Display process tree");
	HIDE_IT("Show all processes");
	HIDE_IT("Monitor child processes");
	HIDE_IT("Show user own process");
	HIDE_IT("Show parent/child relationship between processes");
	HIDE_IT("Close ps");
	HIDE_IT("Renice Process");
	HIDE_IT("Suspend (STOP)");
	HIDE_IT("Continue (CONT)");
	HIDE_IT("Interrupt (INT)");
	HIDE_IT("Hangup (HUP)");
	HIDE_IT("User 1 (USR1)");
	HIDE_IT("User 2 (USR2)");
	HIDE_IT("Terminate (TERM)");
	HIDE_IT("Kill (KILL)");
	HIDE_IT("Segmentation fault");
	HIDE_IT("Display process tree");
	HIDE_IT("User 1 (USR1)-TERM");
	HIDE_IT("User 2 (USR2)-KILL");

    GtkLabel *label_widget = g_object_get_data(G_OBJECT(popup_widget), "label");
    if (en) {
	if (!en->st) g_error("ps_popup(): !en->st\n");
	// Set label to PID information
	if (label_widget){
	    menu_text = g_strdup_printf("<b>%s\n<i>%s = %d</i></b>", 
			(strchr(en->path,':'))?strchr(en->path,':')+1:en->path,
			_("PID"), en->st->st_uid);
	}
	rfm_set_widget(GINT_TO_POINTER(en->st->st_uid), "ps_uid");
	SHOW_IT("Renice Process");
	SHOW_IT("Suspend (STOP)");
	SHOW_IT("Continue (CONT)");
	if (q) {
	    gtk_widget_set_sensitive(rfm_get_widget("Continue (CONT)"), TRUE);
	    gtk_widget_set_sensitive(rfm_get_widget("Suspend (STOP)"), TRUE);
	} else {
	    if (en->st->st_rdev == STOPPED){
		// deactivate the stop item
		gtk_widget_set_sensitive(rfm_get_widget("Suspend (STOP)"), FALSE);
		gtk_widget_set_sensitive(rfm_get_widget("Continue (CONT)"), TRUE);
	    }
	    if (en->st->st_rdev != STOPPED){
		// deactivate the cont item
		gtk_widget_set_sensitive(rfm_get_widget("Suspend (STOP)"), TRUE);
		gtk_widget_set_sensitive(rfm_get_widget("Continue (CONT)"), FALSE);
	    }
	}
	SHOW_IT("Interrupt (INT)");
	SHOW_IT("Hangup (HUP)");
	SHOW_IT("User 1 (USR1)");
	SHOW_IT("User 2 (USR2)");
	SHOW_IT("Terminate (TERM)");
	SHOW_IT("Kill (KILL)");
	SHOW_IT("Segmentation fault");
    } else {
	    rfm_set_widget(NULL, "ps_uid");
	// Set label to plugin name
	if (label_widget){
	    menu_text = g_strdup_printf("<b>%s</b>",_(MODULE_LABEL));
	}
#ifdef HAVE_BSD_XF
	SHOW_IT("Display process tree");
#endif
#ifdef HAVE_UNIX_JH
	SHOW_IT("Display process tree");
#endif
	SHOW_IT("main menu ps");
	SHOW_IT("Close ps");


	gint flags[]={ALL_PROCS,PROC_CHILDREN};
	gchar *hideS[]={"Show all processes","Monitor child processes",NULL};
	gchar *showS[]={"Show user own process","Show parent/child relationship between processes", NULL};
	gint i;
	gint ps_module_flags = GPOINTER_TO_INT(rfm_get_widget( "ps_module_flags"));
	for (i=0; hideS[i]; i++){
	    if (ps_module_flags & flags[i]){
		SHOW_IT(showS[i]);
		HIDE_IT(hideS[i]);
	    } else {
		HIDE_IT(showS[i]);
		SHOW_IT(hideS[i]);
	    }
	}
    }
    if (label_widget){
	gtk_label_set_markup(label_widget, menu_text);	
	g_free(menu_text);
    }

    if (popup_widget) {
	TRACE( "popping 0x%x\n", GPOINTER_TO_INT(popup_widget));
	//gtk_menu_popup(GTK_MENU(popup_widget), NULL, NULL, NULL, NULL, 3, view_p->mouse_event.eventtime);
	gtk_menu_popup(GTK_MENU(popup_widget), NULL, NULL, NULL, NULL, 3, gtk_get_current_event_time());
    }

    rfm_rw_lock_reader_unlock (&(view_p->mutexes.monitor_lock));
    return GINT_TO_POINTER(1);
}

static void *
ps_module_argv(void *p, void *q){
    record_entry_t *en=p;
    if (!en) {
	DBG("ps-module: en == NULL\n");
	return NULL;
    }
    gchar **argv=q;
    if (!argv) {
	DBG("ps-module: argv == NULL\n");
	return NULL;
    }

    if (argv[2]!=NULL) {
	gint pid=atoi(argv[2]);
	if (!en->st) {
	    en->st=(struct stat *)malloc(sizeof(struct stat));
	    memset(en->st, 0, sizeof(struct stat));
	}
	en->st->st_uid=pid;
	// Set the window title
	g_free(en->path);
	en->path = g_strdup("rodent");
	// This is different from what ps reports, since ps follows symlinks:
        // rfm_global_t *rfm_global_p = rfm_global();
	// en->path = g_strdup(rfm_global_p->argv[0]);
    }
    return GINT_TO_POINTER(1);
}
#if 0
// FIXME if to keep
static void *
ps_module_icon_size (void *p, void *q) {
    view_t *view_p=p;
    gint size=GPOINTER_TO_INT(q);
    gint default_size = rfm_get_default_size();
    
    if (p && q){
	g_object_set_data(G_OBJECT(view_p->widgets.paper), 
		"ps_module-iconsize", GINT_TO_POINTER(size));
	NOOP("** setting ps_module-iconsize to %d\n", size);
	return q;
    }
    if (p){
	size = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(view_p->widgets.paper), 
		    "ps_module-iconsize"));
	if (size) {
	    NOOP("** got ps_module-iconsize %d\n", size);
	    return GINT_TO_POINTER(size);
	}
	NOOP("**no ps_module-iconsize has been set\n");
	g_object_set_data(G_OBJECT(view_p->widgets.paper), 
		"ps_module-iconsize", GINT_TO_POINTER(default_size));
    NOOP("** default ps_module-iconsize %d\n", -1);
    }
    return GINT_TO_POINTER (default_size);
}
#endif

static
void *
ps_double_click(void * p, void *q){
    record_entry_t *en=q;
    if (en && en->st && en->st->st_nlink == 0){
	// do not do anything with double click
	// and tell Rodent that double click is taken
	// care of so Rodent will not default the double
	// click action.
	return GINT_TO_POINTER(1);
    }
    // go for the Rodent default double click action (navigate)
    return NULL;
}


