/* wmudmount
 * Copyright © 2010-2014  Brad Jorsch <anomie@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#ifdef HAVE_GNOME_KEYRING
#include <gnome-keyring-memory.h>
#endif

#include "die.h"
#include "secureentry.h"

G_DEFINE_TYPE(SecureEntryBuffer, secure_entry_buffer, GTK_TYPE_ENTRY_BUFFER);

static const gchar *secure_entry_buffer_get_text(GtkEntryBuffer *buf, gsize *len){
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    if(len) *len = b->blen;
    return b->data;
}

static guint secure_entry_buffer_get_length(GtkEntryBuffer *buf){
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    return b->clen;
}

static guint secure_entry_buffer_insert_text(GtkEntryBuffer *buf, guint pos, const gchar *chars, guint n_chars){
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);

    gsize len = g_utf8_offset_to_pointer(chars, n_chars) - chars;
    while(len + b->blen + 1 > b->datalen){
        if(b->datalen == 0) b->datalen = 16;
        b->datalen *= 2;
#ifdef HAVE_GNOME_KEYRING
        if(b->allow_insecure_memory){
            b->data = gnome_keyring_memory_realloc(b->data,b->datalen);
        } else {
            b->data = gnome_keyring_memory_try_realloc(b->data,b->datalen);
            if(!b->data) die("Could not allocate %ld bytes of secure memory", b->datalen);
        }
#else
        if(b->allow_insecure_memory){
            b->data = realloc(b->data,b->datalen);
        } else {
            b->data = NULL;
        }
        if(!b->data) die("Could not allocate %ld bytes of (insecure) memory", b->datalen);
#endif
    }

    gsize at = g_utf8_offset_to_pointer(b->data, pos) - b->data;
    g_memmove(b->data+at+len, b->data+at, b->blen-at);
    memcpy(b->data+at, chars, len);

    b->blen += len;
    b->clen += n_chars;
    b->data[b->blen] = '\0';

    gtk_entry_buffer_emit_inserted_text(buf, pos, chars, n_chars);
    return n_chars;
}

static guint secure_entry_buffer_delete_text(GtkEntryBuffer *buf, guint pos, guint n_chars){
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);

    if(pos > b->clen) pos=b->clen;
    if(pos+n_chars > b->clen) n_chars=b->clen - pos;
    if(n_chars > 0){
        gsize start = g_utf8_offset_to_pointer(b->data, pos) - b->data;
        gsize end = g_utf8_offset_to_pointer(b->data, pos+n_chars) - b->data;
        g_memmove(b->data+start, b->data+end, b->blen+1-end);
        b->clen -= n_chars;
        b->blen -= end - start;
        gtk_entry_buffer_emit_deleted_text(buf, pos, n_chars);
    }

    return n_chars;
}

static void secure_entry_buffer_init(SecureEntryBuffer *b){
    b->data = NULL;
    b->allow_insecure_memory = FALSE;
    b->datalen = 0;
    b->blen = 0;
    b->clen = 0;
}

static void secure_entry_buffer_finalize(GObject *o){
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(o);
#ifdef HAVE_GNOME_KEYRING
    if(b->data) gnome_keyring_memory_free(b->data);
#else
    if(b->data) free(b->data);
#endif
    b->data = NULL;
    G_OBJECT_CLASS(secure_entry_buffer_parent_class)->finalize(o);
}

static void secure_entry_buffer_class_init(SecureEntryBufferClass *c){
    GObjectClass *go = G_OBJECT_CLASS(c);
    go->finalize = secure_entry_buffer_finalize;

    GtkEntryBufferClass *bc = GTK_ENTRY_BUFFER_CLASS(c);
    bc->get_text = secure_entry_buffer_get_text;
    bc->get_length = secure_entry_buffer_get_length;
    bc->insert_text = secure_entry_buffer_insert_text;
    bc->delete_text = secure_entry_buffer_delete_text;
}

GtkEntryBuffer *secure_entry_buffer_new(void){
    return g_object_new(TYPE_SECURE_ENTRY_BUFFER, NULL);
}

gboolean secure_entry_buffer_get_allow_insecure_memory(GtkEntryBuffer *buf){
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    return b->allow_insecure_memory;
}

void secure_entry_buffer_set_allow_insecure_memory(GtkEntryBuffer *buf, gboolean allow){
    SecureEntryBuffer *b = SECURE_ENTRY_BUFFER(buf);
    b->allow_insecure_memory = allow;
}
