/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm.h"
#include "rfm_modules.h"
#include "primary-run.i"


static const gchar *terminals_v[] = {
	"roxterm", 
	"sakura",
	"gnome-terminal", 
	"Eterm", 
	"konsole", 
	"Terminal", 
	"aterm", 
	"xterm", 
	"kterm", 
	"wterm", 
	"multi-aterm", 
	"evilvte",
	"mlterm",
	"xvt",
	"rxvt",
	"urxvt",
	"mrxvt",
	"tilda",
	NULL
};

static const gchar *editors_v[] = {
	"gvim -f",  
	"mousepad", 
	"gedit", 
	"kate", 
	"xemacs", 
	"nano",
	"vi",
	NULL
};

const gchar **rfm_get_terminals(void) {return terminals_v;}
const gchar **rfm_get_editors(void) {return editors_v;}

static GSList *children_list=NULL;

static pthread_mutex_t children_list_mutex = PTHREAD_MUTEX_INITIALIZER;

void rfm_remove_child(pid_t child){
    if (!children_list) return;
    pthread_mutex_lock(&(children_list_mutex));
    children_list = g_slist_remove(children_list, GINT_TO_POINTER(child));
    pthread_mutex_unlock(&(children_list_mutex));
    return;
}

void rfm_killall_children(void){
    TRACE("rfm_killall_children(): signalling controller children...\n");
    pthread_mutex_lock(&(children_list_mutex));

    GSList *list = children_list;
    for (;list && list->data; list = list->next){
	pid_t child = GPOINTER_TO_INT(list->data);
	TRACE( "ZZZZZZZ--- killing %d\n", child);
	kill(child, SIGTERM);

    }
    g_slist_free(children_list);
    children_list = NULL;
    pthread_mutex_unlock(&(children_list_mutex));

}

void rfm_add_child(pid_t child){ 
    pthread_mutex_lock(&(children_list_mutex));
    NOOP(stderr, "adding %d to children_list\n", child);
    children_list = g_slist_prepend(children_list, GINT_TO_POINTER(child));
    pthread_mutex_unlock(&(children_list_mutex));

    return;
}

gchar * 
rfm_get_text_editor_envar(const gchar *value){
    if(!value) return NULL;
    
    gchar *editor=g_path_get_basename(value);
    // if nano or vi, then use terminal emulator
    if (editor && 
	    (strncmp(editor, "vi",strlen("vi"))==0 
	     || 
	     strncmp(editor, "nano",strlen("nano"))==0)){
	const gchar *t=getenv("TERMINAL_CMD");
	gchar *term = g_find_program_in_path(t);
	if (term) g_free(term);
	else {
	    t=NULL;
	    gint i;
	    for (i=0; terminals_v[i]; i++){
		// sakura is broken... 
		if (strstr(terminals_v[i], "sakura")) continue;
		term = g_find_program_in_path(terminals_v[i]);
		if (term){
		    t=terminals_v[i];
		    g_free(term);
		    break;
		}
	    }
	}
	if (t && strlen(t)) {
	    gchar *b=g_strdup_printf("%s %s %s",
		    t, rfm_term_exec_option(t), editor);
	    g_free(editor);
	    editor = b;
	}
    } else {
	g_free(editor);
	editor = g_strdup(value);
    }
    return (editor);
}

///////
static pthread_mutex_t *
get_command_history_mutex(void){
    static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
    return &mutex;
}
//////////////// Module wraparounds ////////////////////////////
//
// Use rfm_thread_run_argv or rfm_thread_run?
// rfm_thread_run_argv will execute directly, not via a shell, and will
// not be saved in the lpterm history

// rfm_thread_run_argv:
// This modality will execute the command without shell


// rfm_thread_run:
// This modality will execute the command via "sh -c" and allow pipes and 
// redirection. and will be saved in the lpterm history file
    
pid_t
rfm_thread_run_argv (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm
	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), NULL,  NULL, NULL, NULL, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(RFM_MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));
}

pid_t
rfm_thread_run_argv_full (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm,
	gint *stdin_fd,
	void (*stdout_f) (void *stdout_data,
                      void *stream,
                      int childFD),
	void (*stderr_f) (void *stdout_data,
                      void *stream,
                      int childFD),
	void (*tubo_done_f) (void *data)
	){
    if (!argv || !argv[0]) return 0;
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), stdin_fd,  stdout_f, stderr_f, tubo_done_f};
    return GPOINTER_TO_INT(rfm_vector_run(RFM_MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

 }

pid_t
rfm_thread_run_argv_with_stdin (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm, 
	gint *stdin_fd
    	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), stdin_fd,  NULL, NULL, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(RFM_MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

 
}

pid_t
rfm_thread_run_argv_with_stdout (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm, 
	void (*stdout_f) (void *stdout_data,
                      void *stream,
                      int childFD)
	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), NULL,  stdout_f, NULL, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(RFM_MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

    
}

pid_t
rfm_thread_run_argv_with_stderr (
	widgets_t * widgets_p, 
	gchar ** argv, 
	gboolean interm, 
	void (*stderr_f) (void *stderr_data,
                      void *stream,
                      int childFD)
	){
    const void *vector[]={widgets_p, argv, GINT_TO_POINTER(interm), NULL,  NULL, stderr_f, NULL};
    return GPOINTER_TO_INT(rfm_vector_run(RFM_MODULE_DIR, "run", GINT_TO_POINTER(7),
		    vector, "m_thread_run_argv"));

}
////////////////////////////////////////////////////////////////////////////

void
rfm_recover_flags (gchar * in_cmd, gboolean * interm, gboolean * hold) {
    DBHashTable *runflags;
    GString *gs;
    int *flags;
    gchar *g = g_build_filename ( RUN_FLAG_FILE, NULL);
    TRACE("opening %s...\n",g); 
    if((runflags = dbh_new (g, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE)) == NULL) {
        TRACE ("Cannot open %s\n", g);
        *interm = 0;
        *hold = 0;
        return;
    }
    TRACE("opened %s.\n",g); 
    dbh_set_parallel_lock_timeout(runflags, 3);
    gs = g_string_new (in_cmd);
    sprintf ((char *)DBH_KEY (runflags), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    flags = (int *)runflags->data;
    dbh_load (runflags);
    *interm = flags[0];
    *hold = flags[1];
    dbh_close (runflags);

    NOOP ("flags recovered from dbh file for %s, interm=%d hold=%d\n", in_cmd, *interm, *hold);
}


// this function is run in a gthread at startup. 
// mutex is to protect from main thread running the
// save history function. Also protects history file
// from concurrent write access.
gpointer
rfm_load_sh_command_history (gpointer data) {
    // This thread is not barrier subject
    view_t *view_p = (view_t *) data;
    gchar *history = g_build_filename (LP_TERMINAL_HISTORY, NULL);
    pthread_mutex_t *command_history_mutex=get_command_history_mutex();
    pthread_mutex_lock(command_history_mutex);
    g_list_free (view_p->sh_command);
    GList *p;
    for(p = view_p->sh_command; p; p = p->next) {
        g_free (p->data);
    }
    view_p->sh_command = NULL;
    view_p->sh_command = g_list_append (view_p->sh_command, g_strdup (""));
    view_p->sh_command_counter = 0;

    FILE *sh_history = fopen (history, "r");
    if(sh_history) {
        // we should put a file lock here...
        NOOP ("rfm_load_sh_command_history(): readlock for %s\n", history);
        gchar line[2048];
        memset (line, 0, 2048);
        while(fgets (line, 2047, sh_history) && !feof (sh_history)) {
            NOOP ("HISTORY: line %s\n", line);
            if(strchr (line, '\n'))
                *strchr (line, '\n') = 0;
            if(strlen (line) == 0)
                continue;
            // skip invalid commands (except cd):
            if(!MIME_is_valid_command (line)) {
                if(strcmp (line, "cd") != 0 && strncmp (line, "cd ", strlen ("cd ")) != 0) {
                    NOOP ("HISTORY: invalid history command in %s: %s\n", history, line);
                    continue;
                }
            }
	    gchar *newline=compact_line(line);
	    GList *element=find_in_history_list(view_p->sh_command, newline);

	    if (element) { 
		// remove old element
		gchar *data=element->data;
		view_p->sh_command = g_list_remove(view_p->sh_command, data);
		g_free(data);
	    }
	    // put at top of the pile
	    view_p->sh_command =
		    g_list_insert_before (view_p->sh_command, g_list_last (view_p->sh_command), newline);
        }

        NOOP ("rfm_load_sh_command_history(): readunlock for %s\n", history);
        fclose (sh_history);
        view_p->sh_command_counter = g_list_length (view_p->sh_command) - 1;
    }
    g_free (history);
	
    pthread_mutex_unlock(command_history_mutex);
    return NULL;
}

void
rfm_save_sh_command_history (view_t * view_p, const gchar * command) {
    if (!view_p){
	return;
    }
    GList *p;
	    
    pthread_mutex_t *command_history_mutex=get_command_history_mutex();
    pthread_mutex_lock(command_history_mutex);
    p = g_list_previous (g_list_last (view_p->sh_command));
    if(!command || !strlen (command)){
	pthread_mutex_unlock(command_history_mutex);
        return;
    }
    gchar *command_p = g_strdup (command);
    g_strstrip (command_p);
    // if repeat of last command, skip it.
    if(!p || strcmp (command, (char *)p->data)) {
        view_p->sh_command = g_list_insert_before (view_p->sh_command, g_list_last (view_p->sh_command), command_p);
        //view_p->sh_command = g_list_append(view_p->sh_command, command_p);

        // don't save to file if invalid command 
        if(!MIME_is_valid_command (command_p)) {
            if(strcmp (command_p, "cd") != 0 && strncmp (command_p, "cd ", strlen ("cd ")) != 0) {
                DBG ("not saving %s\n", command_p);
                view_p->sh_command_counter = g_list_length (view_p->sh_command) - 1;
		pthread_mutex_unlock(command_history_mutex);
                return;
            }
        }
        // here we will rewrite the history, removing any item
        // which duplicate the command about to be saved.
        // This effectively moves the command to the bottom
        // of the list.
        // 

        gchar *history = g_build_filename (LP_TERMINAL_HISTORY, NULL);

        // read it first to synchronize with other rodent instances
        GList *disk_history = NULL;
        
        FILE *sh_history = fopen (history, "r");

        if(sh_history) {
 
            char line[2048];
            memset (line, 0, 2048);
            while(fgets (line, 2047, sh_history) && !feof (sh_history)) {
                if(strchr (line, '\n')) {
                    *strchr (line, '\n') = 0;
                }
                if(strcmp (line, command_p) != 0) {
                    disk_history = g_list_append (disk_history, g_strdup (line));
                }
            }

            fclose (sh_history);
        }
        disk_history = g_list_append (disk_history, g_strdup (command_p));

        sh_history = fopen (history, "w");
        if(sh_history) {
            // we should put a file lock here...
            GList *p;
            for(p = g_list_first (disk_history); p && p->data; p = p->next) {
                fprintf (sh_history, "%s\n", (char *)p->data);
                g_free (p->data);
            }
            fclose (sh_history);
        }
        g_list_free (disk_history);
        g_free (history);
    }
    view_p->sh_command_counter = g_list_length (view_p->sh_command) - 1;
    NOOP ("rfm_save_sh_command_history(); command_counter=%d\n",
	    view_p->sh_command_counter);
	
    pthread_mutex_unlock(command_history_mutex);
    return;
}

 
const gchar * 
rfm_term_exec_option(const gchar *terminal) {
    const gchar *exec_option = "-e";
    gchar *t = g_path_get_basename (terminal);
    if(strcmp (t, "gnome-terminal") == 0 || strcmp (t, "Terminal") == 0)
            exec_option = "-x";
    g_free(t);
    return exec_option;
}

const gchar *
rfm_what_term (void) {
    const gchar *term=getenv ("TERMINAL_CMD");
    gchar *t=NULL;
    if(term && strlen (term)) {
	if (strchr(term, ' ')){
	    gchar **g = g_strsplit(term, " ", -1);
	    t = g_find_program_in_path (g[0]);
	    g_strfreev(g);
	} else {
	    t = g_find_program_in_path (term);
	}
    }
    if(!t) {
	    const gchar **p=terminals_v;
	    for (;p && *p; p++){
		t = g_find_program_in_path (*p);
		if (t) {
		    term=*p;
		    break;  
		}  
	    }
    }
    if (t) {
	g_free(t);
	return term;
    }
    DBG ("TERMINAL_CMD=%s: %s\n", getenv ("TERMINAL_CMD"), strerror (ENOENT));

    return NULL;
}

