#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
   

static void
set_cairo_color(view_t *view_p, cairo_t *gdk_context, int color_id){
        // select background color
#if GTK_MAJOR_VERSION==3
	GdkRGBA *color=rfm_get_gdk_color (view_p, color_id);
	cairo_set_source_rgb(gdk_context,
                color->red, color->green, color->blue);
#else
	GdkColor *color=rfm_get_gdk_color (view_p, color_id);
	cairo_set_source_rgb(gdk_context, 
		(double)(color->red)/0xffff,
		(double)(color->green)/0xffff, 
		(double)(color->blue)/0xffff);
#endif
	g_free(color);

}

static gboolean
environment_condition(const gchar *environment_string, const gchar *current_value){
    const gchar *environment_value=getenv (environment_string);
    if (!environment_value && !current_value) return FALSE;
    else if (!environment_value && current_value) return TRUE;
    else if (environment_value && !current_value) return TRUE;
    else if (strcmp(environment_value, current_value))  return TRUE;
    return FALSE;
}
   
static gchar *
environment_dup(const gchar *environment_string){
    if (getenv(environment_string)) return g_strdup(getenv(environment_string));
    return NULL;
}

// XXX background image works for the iconview too.
//     Just create a different environment variable
//     to do configuration separate from deskview...
static gboolean
background_test(view_t *view_p) {
    gboolean background_condition=(view_p->flags.type == ICONVIEW_TYPE)? FALSE:
	environment_condition("RFM_DESKTOP_IMAGE",view_p->eyecandy.desktop_bg_file);
    gboolean deskcolor_condition=(view_p->flags.type == ICONVIEW_TYPE)? FALSE:
	environment_condition("RFM_DESKTOP_COLOR",view_p->eyecandy.desktop_color);
    gboolean gridcolor_condition=(view_p->flags.type == DESKVIEW_TYPE)? FALSE:
	environment_condition("RFM_ICONVIEW_COLOR",view_p->eyecandy.iconview_color);
	
    if (deskcolor_condition &&
	    getenv("RFM_DESKTOP_IMAGE") &&
	    strlen(getenv("RFM_DESKTOP_IMAGE"))) background_condition = TRUE;

    /*if (background_condition){
	TRACE( "background_condition = %s\n",getenv("RFM_DESKTOP_IMAGE"));
	if  (getenv("RFM_DESKTOP_IMAGE") && strlen(getenv("RFM_DESKTOP_IMAGE"))) {
	    g_free(view_p->eyecandy.desktop_bg_file);
	    view_p->eyecandy.desktop_bg_file = g_strdup(getenv("RFM_DESKTOP_IMAGE"));
	}
    }*/
    if (deskcolor_condition){
	g_free (view_p->eyecandy.desktop_color);
	view_p->eyecandy.desktop_color = environment_dup("RFM_DESKTOP_COLOR");
    }
    if (gridcolor_condition){
	g_free (view_p->eyecandy.iconview_color);
	view_p->eyecandy.iconview_color = environment_dup("RFM_ICONVIEW_COLOR");
    }
    return (gridcolor_condition | deskcolor_condition | background_condition);
}

void
rodent_clean_rectangle (view_t * view_p, int x, int y, int w, int h, cairo_t *gdk_context) {
    if (w == 0 || h == 0) return;
    NOOP(stderr, "rodent_clean_rectangle(%d, %d, %d, %d)\n", x, y, w, h);

    if (view_p->eyecandy.bg_surface) { // background surface is available
	cairo_set_source_surface (gdk_context, view_p->eyecandy.bg_surface, 0,0);
	cairo_rectangle(gdk_context, x, y, w, h);
	cairo_fill(gdk_context);
    } else 
    {
        set_cairo_color(view_p, gdk_context, BACKGROUND_COLOR);
	cairo_rectangle(gdk_context, x, y, w, h);
	cairo_fill(gdk_context);
    }
}
 
static void
rodent_clean_image (view_t * view_p, GdkRectangle *original_rect, int x, int y, int w, int h, cairo_t *gdk_context) {
    if (w == 0 || h == 0) return;
    NOOP(stderr, "rodent_clean_image(%d, %d, %d, %d)\n", x, y, w, h);

    
    if (view_p->eyecandy.bg_surface && original_rect) { 
	// background surface is available
	// Chop a piece off the surface... (gdk3)
#if 1
//#if GTK_MAJOR_VERSION==2
	GdkPixbuf *chopped = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, w, h );
	if (chopped) gdk_pixbuf_copy_area (view_p->bg_pixbuf,
		original_rect->x,original_rect->y, w, h,
		chopped, 0,0);
#else
	// This is slower...
	GdkPixbuf *chopped = gdk_pixbuf_get_from_surface (view_p->eyecandy.bg_surface,
		original_rect->x,original_rect->y, w, h); 
#endif 
	gdk_cairo_set_source_pixbuf (gdk_context, chopped, 0, 0);
	cairo_paint(gdk_context);
	if (chopped) g_object_unref(chopped);
    } else 
    {
        set_cairo_color(view_p, gdk_context, BACKGROUND_COLOR);

	cairo_rectangle(gdk_context, x, y, w, h);
	cairo_fill(gdk_context);
    }
}



void
rodent_clean_rect (view_t * view_p, GdkRectangle *rect) {
    cairo_t *gdk_context =  
	gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
    rodent_clean_rectangle (view_p, rect->x, rect->y, rect->width, rect->height,gdk_context);
    cairo_destroy(gdk_context);
}


#define RED 0
#define GREEN 1
#define CYAN 2
#define MAGENTA 3
#define GREY 4
#define BGCOLOR 5
#define RUBBERBAND_COLOR 6
#define RUBBERBAND_COLOR_DEFINITION 0.35, 0.65, 0.65

static void 
saturated_rectangle(view_t * view_p, gint x, gint y, gint width, gint height, cairo_t *gdk_context, gint color) {
	// saturated rectangle (thin margin)
	gboolean is_dark_bg=rfm_is_dark_background(view_p);
	
	switch (color){
	    case RUBBERBAND_COLOR:
		// No outline here... (use same color as fill color)
		cairo_set_source_rgb(gdk_context, RUBBERBAND_COLOR_DEFINITION);
		break;
	    case BGCOLOR:
		// No outline here... (use same color as fill color)
		if (is_dark_bg){
		    cairo_set_source_rgb(gdk_context, 0.05, 0.05, 0.05);
		} else {
		    cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.95);
		}
		break;
	    case RED:
		// No outline here... (use same color as fill color)
		if (is_dark_bg){
		    cairo_set_source_rgb(gdk_context, 0.950, 0.30, 0.30);
		} else {
		    cairo_set_source_rgb(gdk_context, 0.850, 0.0, 0.0);
		}
		break;
	    case GREEN:
		cairo_set_source_rgb(gdk_context, 0.0, 1.0, 0.0);
		break;
	    case CYAN:
		cairo_set_source_rgb(gdk_context, 0.0, 1.0, 1.0);
		break;
	    case MAGENTA:
		cairo_set_source_rgb(gdk_context, 1.0, 0.0, 1.0);
	    case GREY:
		cairo_set_source_rgb(gdk_context, 0.30, 0.30, 0.30);
		break;
	}
	cairo_rectangle(gdk_context, x, y, width, height);
	cairo_fill(gdk_context);
	
	rodent_clean_rectangle (view_p, x + 1, y + 1, width-2, height-2, gdk_context);
	if (color == RED) {
	    if (is_dark_bg){
		cairo_set_source_rgb(gdk_context, 0.75, 0.75, 0.95);
	    } else {
		cairo_set_source_rgb(gdk_context, 0.3, 0.3, 0.7);
	    }
	} else if (color == BGCOLOR) {
	    if (is_dark_bg){
		cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.95);
	    } else {
		cairo_set_source_rgb(gdk_context, 0.05, 0.05, 0.05);
	    }
	}  else if (color == RUBBERBAND_COLOR) {
	    // lighter blue background 
	    cairo_set_source_rgb(gdk_context, RUBBERBAND_COLOR_DEFINITION);
	} else {
	    // blue background 
	    cairo_set_source_rgb(gdk_context, 0.45, 0.65, 0.95);
	}
//	cairo_set_source_rgba(gdk_context, 0.0, 0.25, 0.85, 0.45);
	cairo_rectangle(gdk_context,  x + 1, y + 1, width-2, height-2);
	cairo_fill(gdk_context);
}

#define DETAILS_EXPOSE 0x01
#define COMPACT_EXPOSE 0x02
#define NORMAL_EXPOSE 0x04
#define SATURATED_LABEL_EXPOSE 0x08
#define SATURATED_ICON_EXPOSE 0x010
#define SATURATED_EXPOSE (SATURATED_LABEL_EXPOSE | SATURATED_ICON_EXPOSE)
#define SELECTED_EXPOSE 0x020
#define RUBBERBAND_EXPOSE 0x040
#define DARK_BACKGROUND_EXPOSE 0x080



static void
redraw_text_outline(view_t *view_p, cairo_t *gdk_context, const population_t *population_p, GdkRectangle label_rect){
    // redraw_text_outline
    // This only applies when we have a background image, really.
    if (view_p->flags.type != DESKVIEW_TYPE) return;
    if (!getenv ("RFM_DESKTOP_IMAGE")) return;
    if (!g_file_test(getenv ("RFM_DESKTOP_IMAGE"), G_FILE_TEST_EXISTS) ) return;

    
    set_cairo_color(view_p, gdk_context, BACKGROUND_COLOR);
    gint i, j;
    
    gint icon_size = ICON_SIZE(view_p);
    gint x = label_rect.x;
    gint y = label_rect.y;
    gint max_tiny_width = NAME_WIDTH(view_p);
    NOOP("outline for %s\n", population_p->en->path);
    for(i = x-1; i < x + 3; i++) {
	for(j = y-1; j < y + 3; j++) {
	    //if(i == x && j == y) continue;
	    cairo_move_to (gdk_context, i,j);
            if (PANGO_IS_LAYOUT (population_p->layout)) {
	        pango_cairo_show_layout (gdk_context, population_p->layout);
            }
	    if(population_p->layout2) {
		gint X=(icon_size >= SMALL_ICON_SIZE)?
			i:
			i + max_tiny_width;
		//i+population_p->logical_rect.width +2;
		gint Y=(icon_size >= SMALL_ICON_SIZE)?
			j + population_p->logical_rect.height:
			j;
		cairo_move_to (gdk_context, X, Y);
                if (PANGO_IS_LAYOUT (population_p->layout2)) {
		    pango_cairo_show_layout (gdk_context, population_p->layout2);
                }
	    }
	}
    }
}

static cairo_surface_t *
get_layout_surface(view_t *view_p, const population_t *population_p, 
	GdkRectangle item_rect, GdkRectangle label_rect, gint expose_type){ 
	
    if(!population_p->layout) return NULL;
    
    gint cellwidth = CELLWIDTH(view_p);
    gint cellheight = CELLHEIGHT(view_p);
    gint icon_size = ICON_SIZE(view_p);
     
    GdkRectangle original_rect;
    original_rect.x = item_rect.x;
    original_rect.y = label_rect.y;
    label_rect.x -= item_rect.x;
    label_rect.y = 0;


    item_rect.x = 0;
    item_rect.y = 0; 
    
    gint boxX = label_rect.x;
    gint boxW = label_rect.width;
    gint height = cellheight - icon_size - TEXTSPACING;
    gint boxH = label_rect.height;
    gint cleanX = 0;
    gint cleanW = cellwidth;
    if (expose_type & RUBBERBAND_EXPOSE) {
	boxH = height;
	boxW = cellwidth;
	boxX = 0;
    }
    if (expose_type & COMPACT_EXPOSE || expose_type & DETAILS_EXPOSE){
	height = cellheight - TEXTSPACING;
	boxH = label_rect.height;
	cleanX = label_rect.x;
	cleanW = cellwidth;
	boxW = cellwidth; 
	boxX = label_rect.x;
    }  
    original_rect.width = cleanW;
    original_rect.height = height;
    
 
    cairo_surface_t *buffer_surface = 
	cairo_image_surface_create(CAIRO_FORMAT_ARGB32, cellwidth, height);
    cairo_t *gdk_context = cairo_create(buffer_surface);



    

    gboolean yellow_text = FALSE;

// XXX: this borks details and compact views in deskview.
// 	I dunno why...
//      once this this mistery is solved, use a surface to expedite deskview expose.
   if (!(expose_type & COMPACT_EXPOSE) && !(expose_type & DETAILS_EXPOSE)){
	rodent_clean_image (view_p, &original_rect,
		cleanX, 
		0,
		cleanW,
		height, // boxH, // full_height
		gdk_context);
   }

    if ((expose_type & SELECTED_EXPOSE) && (expose_type & RUBBERBAND_EXPOSE)) {
	saturated_rectangle(view_p, 
		boxX,
		0,
		boxW, 
		boxH, // full_height
		gdk_context, RUBBERBAND_COLOR);
	yellow_text = TRUE;
    } else if (expose_type & SATURATED_ICON_EXPOSE){
	saturated_rectangle(view_p, 
		boxX, 0,
		boxW, 
		boxH, 
		gdk_context, RUBBERBAND_COLOR);
	yellow_text = TRUE;
    } /*else {
	rodent_clean_image (view_p, &original_rect,
		cleanX, 
		0,
		cleanW,
		height, // boxH, // full_height
		gdk_context);
    }*/

    // This applies to deskview with background image, otherwise does nothing:
    redraw_text_outline(view_p, gdk_context, population_p, label_rect);

    // redraw layouts;
    gint X = label_rect.x + 1;
    gint Y = label_rect.y + 1;

 
    if (yellow_text) {
	cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.05);
    } else {
        set_cairo_color(view_p, gdk_context, rodent_text_color(population_p));
    }
    cairo_move_to (gdk_context,  X, Y);
    pango_cairo_show_layout (gdk_context, population_p->layout);
#if 10
    if (!population_p->layout2) goto done;
    if (expose_type & COMPACT_EXPOSE) goto done;
    // Figure out layout2 position
    if(expose_type & DETAILS_EXPOSE){
	gint max_tiny_width = NAME_WIDTH(view_p);
	X = label_rect.x + max_tiny_width + 1;
    } else {
	X = label_rect.x + 1;
	Y = label_rect.y + population_p->logical_rect.height + 1;
    }

    cairo_move_to (gdk_context,  X, Y);
    pango_cairo_show_layout (gdk_context, population_p->layout2);

done:
#endif
    cairo_destroy(gdk_context);
    
    return buffer_surface;
}

static void
redraw_text (view_t * view_p, const population_t * population_p, cairo_t *gdk_context) {

    // Figure out what we are dealing with from the beginning...
    gint expose_type = 0;
    gint icon_size = ICON_SIZE(view_p);
    // Three basic types of view: details, compact and normal
    if (rfm_layout_is_details_size(view_p)){
	expose_type |= DETAILS_EXPOSE;
    } else if (icon_size == TINY_ICON_SIZE){
	expose_type |= COMPACT_EXPOSE;
    } else {
	expose_type |= NORMAL_EXPOSE;
    }
    if (population_p->flags & POPULATION_SATURATED) {
	expose_type |= SATURATED_ICON_EXPOSE;
    } else if (population_p->flags & LABEL_SATURATED) {
	expose_type |= SATURATED_LABEL_EXPOSE;
    } else if (population_p->flags & POPULATION_SELECTED) {
	expose_type |= SELECTED_EXPOSE;
    }
    if (view_p->mouse_event.rubberbanding) {
	expose_type |= RUBBERBAND_EXPOSE;
    }
    if (rfm_is_dark_background(view_p)) {
	expose_type |= DARK_BACKGROUND_EXPOSE;
    }


    if (!population_p->layout || !PANGO_IS_LAYOUT (population_p->layout))
    {
	// create layout (only for exposed items...)
	rfm_do_layout(view_p, (population_t *)population_p);
    }
    GdkRectangle item_rect;
    GdkRectangle label_rect;
    if (!rfm_get_population_rect(view_p, population_p, &item_rect)) return;
    if (!rfm_get_population_label_rect(view_p, population_p, &label_rect)) return;

    // Action starts here...
    if (!(expose_type & SATURATED_LABEL_EXPOSE)) {
	cairo_surface_t *buffer_surface = get_layout_surface(view_p, population_p, item_rect, label_rect, expose_type);
	if (!buffer_surface) return;
	cairo_set_source_surface (gdk_context, buffer_surface, item_rect.x, label_rect.y);
	cairo_paint(gdk_context);
	cairo_surface_destroy(buffer_surface);
    } else {
	GdkRectangle label_rect_full;
	if (!rfm_get_population_label_rect_full(view_p, population_p, &label_rect_full)) return;
	rodent_clean_rectangle (view_p, 
		item_rect.x, 
		label_rect.y,
		CELLWIDTH(view_p),
		CELLHEIGHT(view_p) - icon_size - TEXTSPACING,
		gdk_context);
	saturated_rectangle(view_p, 
		    label_rect_full.x, 
		    label_rect_full.y,
		    label_rect_full.width, 
		    label_rect_full.height, 
		gdk_context, RED);
	
	// Redefine text color.
	if (expose_type & DARK_BACKGROUND_EXPOSE) {
	    cairo_set_source_rgb(gdk_context, 0.05, 0.05, 0.05);
	} else {
	    cairo_set_source_rgb(gdk_context, 0.95, 0.95, 0.95);
	}
	cairo_move_to (gdk_context, label_rect_full.x + 1 , label_rect_full.y + 1);
	pango_cairo_show_layout (gdk_context, population_p->layout_full);
	if (!(expose_type & (DETAILS_EXPOSE | COMPACT_EXPOSE))) {
	    pango_cairo_show_layout (gdk_context, population_p->layout_full);
	}  
    }
    return;
}
    

static void
insert_layout (view_t * view_p, const population_t * population_p, cairo_t *gdk_context) {
    if (!view_p){
	DBG("insert_layout(): !view_p\n");
	return;
    }
    if (!population_p){
	//DBG("insert_layout(): !population_p\n");
	return;
    }
    redraw_text (view_p, population_p, gdk_context);

}

enum {
    EMBLEM_SIZE_TINY,
    EMBLEM_SIZE_SMALL,
    EMBLEM_SIZE_NORMAL,
    EMBLEM_SIZE_BIG,
    EXPOSE_SIZES
};

enum {
    EMBLEM_SHOW_HIDDEN,
    EMBLEM_IMAGE,
    EMBLEM_READONLY,
    EMBLEM_BOOKMARK,
    EMBLEM_STOCK_CUT,
    EMBLEM_STOCK_COPY,
    EMBLEM_SYMBOLIC_LINK,
    EMBLEM_ROOT,
    EMBLEM_BAK,
    EXPOSE_EMBLEMS
};

static gchar *emblem_id[]={
    "xffm/emblem_show-hidden",
    "xffm/emblem_image",
    "xffm/emblem_readonly",
    "xffm/emblem_bookmark",
    "xffm/stock_cut",
    "xffm/stock_copy",
    "xffm/emblem_symbolic-link",
    "xffm/emblem_root",
    "xffm/emblem_bak",
    NULL
};
    

static gboolean
insert_emblem(cairo_t *context, GdkRectangle *rect, GdkPixbuf *emblem, gchar *where, gdouble alpha){
    if (!emblem || !GDK_IS_PIXBUF(emblem) || !rect || !where) {
	NOOP(stderr, "kkkkk kkk %d=%d, %d, %s\n", 
		GPOINTER_TO_INT(emblem), GDK_IS_PIXBUF(emblem),GPOINTER_TO_INT(rect),where);
	return FALSE;
    }
    gint x,y;
    if (strcmp(where, "NE")==0) {
	x = rect->x +1 + rect->width - gdk_pixbuf_get_width(emblem);
	y = rect->y +1;
    } else if (strcmp(where, "NW")==0) {
	x = rect->x +1;
	y = rect->y +1;
    } else if (strcmp(where, "SE")==0) {
	x = rect->x +1 + rect->width - gdk_pixbuf_get_width(emblem);
	y = rect->y +1 + rect->height - gdk_pixbuf_get_height(emblem);
    } else if (strcmp(where, "SW")==0) {
	x = rect->x +1;
	y = rect->y +1 + rect->height - gdk_pixbuf_get_height(emblem);
    } else if (strcmp(where, "C")==0) {
	x = rect->x +1 + rect->width/2 - gdk_pixbuf_get_width(emblem)/2;
	y = rect->y +1 + rect->height/2 - gdk_pixbuf_get_height(emblem)/2;
    } else {
	DBG("insert_emblem(): unknown coordinates: %s\n", where);
	return FALSE;
    }

    NOOP(stderr, "paint at %d,%d\n", x,y);
    gdk_cairo_set_source_pixbuf(context, emblem, x, y );
    cairo_paint_with_alpha(context, alpha);
    return TRUE;
}

static cairo_surface_t *
get_pixbuf_surface(view_t *view_p, population_t *population_p, 
	GdkRectangle item_rect, GdkRectangle icon_rect){ 

    // Here's where the action starts...
    gint icon_size = ICON_SIZE(view_p);
        
    cairo_surface_t *buffer_surface = 
	cairo_image_surface_create(CAIRO_FORMAT_ARGB32, item_rect.width,
		icon_size+TEXTSPACING);
    cairo_t *gdk_context = cairo_create(buffer_surface);
    icon_rect.x -= item_rect.x;
    icon_rect.y -= item_rect.y;
    GdkRectangle original_rect;
    original_rect.x = item_rect.x;
    original_rect.y = item_rect.y;
    item_rect.x = 0;
    item_rect.y = 0;

    rodent_clean_image (view_p, &original_rect,
		item_rect.x, item_rect.y,
		item_rect.width,
		icon_size+TEXTSPACING, 
		gdk_context);
    


    gint cellwidth = CELLWIDTH(view_p);
    gint cellheight = CELLHEIGHT(view_p);
    gint x = (icon_size >= SMALL_ICON_SIZE)?
	    item_rect.x + ((cellwidth - icon_size) / 2):
	    item_rect.x;
 
    // saturated item background 
    // (the color is for the thin margin around the rectangle
    //  and will also determine the rectangle fill color)
    if(population_p->flags & POPULATION_SATURATED) {
		saturated_rectangle(view_p, 
			x,
			item_rect.y,
			icon_size + 2, 
			icon_size + 2, gdk_context, 
			RED);
     } else if (population_p->flags & POPULATION_SELECTED) {
	if(view_p->mouse_event.rubberbanding) {
	        // rubberband box
		saturated_rectangle(view_p, 
			item_rect.x,
			item_rect.y,
			cellwidth, 
			cellheight, 
			gdk_context, 
			RUBBERBAND_COLOR);
	} else {
		// simply select
		saturated_rectangle(view_p, 
			x,
			item_rect.y,
			icon_size + 2, 
			icon_size + 2, gdk_context, 
			MAGENTA);
	}
     } else {
	    rodent_clean_image (view_p, &original_rect,
		icon_rect.x, 
		icon_rect.y,
		icon_rect.width + 5,
		icon_rect.height + 5,
		gdk_context);
    }

    // icon pixbuf
    NOOP("---> inserting %s at %d,%d\n",
	    population_p->en->path, icon_rect.x+1, icon_rect.y+1);

    GdkPixbuf *src_pixbuf=population_p->pixbuf;

    gint x_offset=0;
    gint y_offset =0;

    x_offset = (rfm_layout_get_icon_size_id(view_p) >= SMALL_ICON_SIZE)?
	    (cellwidth - population_p->icon_size) / 2:0;
 //    if(population_p->en && !IS_SDIR(population_p->en->type) && 
//	    (population_p->flags & POPULATION_THUMBNAILED) ) {
    // Special stuff for thumbnails (but not for folders)
    if (population_p->thumbnail && GDK_IS_PIXBUF(population_p->thumbnail)){
	gint width = gdk_pixbuf_get_width(population_p->thumbnail);
	gint height = gdk_pixbuf_get_height(population_p->thumbnail);
	x_offset = (population_p->icon_size >= SMALL_ICON_SIZE)?
	    (cellwidth - width) / 2:
	    (population_p->icon_size - width) / 2;
	y_offset = (population_p->icon_size - height) / 2;
	src_pixbuf=population_p->thumbnail;
	icon_rect.x += x_offset;
	icon_rect.y += y_offset;
	// cute shadow
	if (!(population_p->flags & POPULATION_SATURATED) &&
	    !(population_p->flags & POPULATION_SELECTED)
	    ) {
	    cairo_set_source_rgba(gdk_context, 0.0, 0.0, 0.0, 0.35);
	    cairo_rectangle(gdk_context, 
		icon_rect.x+2,
		icon_rect.y+2,
		width + 3,
		height + 3);
	    cairo_fill(gdk_context);
	}
	// black frame
	cairo_set_source_rgb(gdk_context, 0.0, 0.0, 0.0);
	
	cairo_rectangle(gdk_context, 
		icon_rect.x,
		icon_rect.y,
		width + 2,
		height + 2);
	cairo_fill(gdk_context);

	// fill in with white color, for the benefit of transparent previews.
	if (population_p->flags & POPULATION_SELECTED) {
	    cairo_set_source_rgba(gdk_context, 1.0, 1.0, 1.0, 0.450);
	} else {
	    cairo_set_source_rgb(gdk_context, 1.0, 1.0, 1.0);
	}
	cairo_rectangle(gdk_context, 
		icon_rect.x+1,
		icon_rect.y+1,
		width,
		height);
	cairo_fill(gdk_context);
	
    } else {
	icon_rect.x += x_offset;
	icon_rect.y += y_offset;
    }

    if (GDK_IS_PIXBUF(src_pixbuf)){
		gdk_cairo_set_source_pixbuf(gdk_context, 
				src_pixbuf,
				icon_rect.x+1,
				icon_rect.y+1);
		if(population_p->flags & POPULATION_SELECTED) {
			cairo_paint_with_alpha(gdk_context, 0.450);
		} else {
			cairo_paint (gdk_context); 
		}
    }
    // Entry dependent pixbuf modifiers
    if (!population_p->en) {
        DBG("get_pixbuf_surface():done, !population_p->en\n");
        goto done;
    }

    // EMBLEMS/////////////////////////
    // expose emblems:
    // emblem_show-hidden(3), emblem_image(6),
    // emblem_readonly(3), emblem_bookmark(3),
    // stock_cut(3), stock_copy(3), emblem_symbolic-link(2)
    // emblem_root(2), emblem_default(2)

    static gsize initialized = 0;
    static GdkPixbuf *pixbuf_emblem[EXPOSE_EMBLEMS][EXPOSE_SIZES];
    if (g_once_init_enter(&initialized)){
	gint i,j;
	for (i=0; i< EXPOSE_SIZES; i++) {
	    gint isize = 48;
	    switch (i){
		case EMBLEM_SIZE_TINY: isize = 24; break;
		//case EMBLEM_SIZE_SMALL: isize = 48; break;
		case EMBLEM_SIZE_NORMAL: isize = 72; break;
		case EMBLEM_SIZE_BIG: isize = 96; break;
	    }
	    for (j=0; j<EXPOSE_EMBLEMS; j++){
		double default_factor = 0.3;
		switch (j) {
		    case EMBLEM_IMAGE: 
			default_factor = 0.15; break;
		    case EMBLEM_SYMBOLIC_LINK: 
		    case EMBLEM_ROOT: 
		    case EMBLEM_BAK: 
			default_factor = 0.5; break;
		}
		pixbuf_emblem[j][i] = rfm_get_pixbuf (emblem_id[j], default_factor * isize);
	    }
	}
	g_once_init_leave(&initialized, 1);
    }
    gint emblem_size = EMBLEM_SIZE_NORMAL;
    switch (population_p->icon_size){
	case 0:
	case 24:
	    emblem_size = EMBLEM_SIZE_TINY; break;
	case 48:
	    emblem_size = EMBLEM_SIZE_SMALL; break;
	case 72:
	    emblem_size = EMBLEM_SIZE_NORMAL; break;
	case 96:
	    emblem_size = EMBLEM_SIZE_BIG; break;
    }

    GdkPixbuf *emblem_pixbuf=NULL;
#if 0
    no longer used...
    // Emblems...
    // Module emblems
    if (POPULATION_MODULE(population_p)){
	// ask POPULATION_MODULE for emblem
	const gchar *emblem=(const gchar *)
	    rfm_natural(PLUGIN_DIR, POPULATION_MODULE(population_p), 
		    population_p->en, "get_emblem");
	if (emblem) {
	  emblem_pixbuf = rfm_get_pixbuf (emblem, icon_size/1.8);
	  if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "NE", 0.85))
	      NOOP(stderr, "ooooo--> %s\n", emblem);
	}
    }
#endif

    // Up type icon emblems...
    if (IS_UP_TYPE (population_p->en->type)){
	gboolean have_icons_plugin =
	    GPOINTER_TO_INT(rfm_void(RFM_MODULE_DIR, "icons", "module_active"));
	if (have_icons_plugin) {
	    if(view_p->flags.preferences & __SHOW_HIDDEN){
	      emblem_pixbuf = pixbuf_emblem[EMBLEM_SHOW_HIDDEN][emblem_size];
	      if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "NW", 0.85))
	      NOOP(stderr, "ooooo--> EMBLEM_SHOW_HIDDEN\n" );
	    }
	    if(view_p->flags.preferences & __SHOW_IMAGES){
	      emblem_pixbuf =  pixbuf_emblem[EMBLEM_IMAGE][emblem_size];
	      if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "NE", 0.85))
	      NOOP(stderr, "ooooo--> EMBLEM_IMAGE\n");
	    }
	}

    }


    // Readonly emblem
    gboolean no_write_emblem=FALSE;
    if (IS_LOCAL_TYPE(population_p->en->type)) {
	no_write_emblem = 
	    !IS_UP_TYPE(population_p->en->type) &&
	    !IS_SDIR(population_p->en->type) &&
	    IS_NOWRITE_TYPE(population_p->en->type);
    }
    if (no_write_emblem){
	emblem_pixbuf = pixbuf_emblem[EMBLEM_READONLY][emblem_size];
	if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "NE", 0.85))
	      NOOP(stderr, "ooooo--> EMBLEM_READONLY\n");
    }

    rfm_global_t *rfm_global_p = rfm_global();
    g_mutex_lock(rfm_global_p->status_mutex);
    gint status = rfm_global_p->status;
    g_mutex_unlock(rfm_global_p->status_mutex);
    if(status == STATUS_EXIT) return NULL;
    
    // Bookmark emblem
    if (!IS_UP_TYPE(population_p->en->type) &&
	    rodent_path_has_bookmark(population_p->en->path)){
	emblem_pixbuf = pixbuf_emblem[EMBLEM_BOOKMARK][emblem_size];
	if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "NE", 0.85))
	      NOOP(stderr, "ooooo--> EMBLEM_BOOKMARK\n");
    }

    // Pasteboard emblems...
    GdkPixbuf *pasteboard_pixbuf=NULL;
    // CUT emblem
    if (population_p->flags & POPULATION_CUT) {
	pasteboard_pixbuf =  pixbuf_emblem[EMBLEM_STOCK_CUT][emblem_size];
    }
    // COPIED emblem
    else if (population_p->flags & POPULATION_COPIED) {
	pasteboard_pixbuf =  pixbuf_emblem[EMBLEM_STOCK_COPY][emblem_size];
    }
    if (!insert_emblem(gdk_context, &icon_rect, pasteboard_pixbuf, "NW", 0.85))
	      NOOP(stderr, "ooooo--> pasteboard_pixbuf\n");
    
    // Symlink emblem...
    if (IS_SLNK(population_p->en->type) && 
	    !IS_UP_TYPE(population_p->en->type)) {
	emblem_pixbuf = pixbuf_emblem[EMBLEM_SYMBOLIC_LINK][emblem_size];
	if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "SW", 0.85))
	      NOOP(stderr, "ooooo--> EMBLEM_SYMBOLIC_LINK\n");
    }
    // Root emblem 
    if (!IS_UP_TYPE(population_p->en->type)
	    && strcmp(population_p->en->path, "/")==0)
    {
	emblem_pixbuf = pixbuf_emblem[EMBLEM_ROOT][emblem_size];
	if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "SE", 0.85))
	      NOOP(stderr, "ooooo--> EMBLEM_ROOT\n");
    }
    // Backup types...
    if ((view_p->flags.preferences & __SHOWS_BACKUP)
	    && rodent_is_gnu_backup_type(population_p->en->path))    
    {
	emblem_pixbuf = pixbuf_emblem[EMBLEM_BAK][emblem_size];
	if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "NW", 0.85))
	      NOOP(stderr, "ooooo--> EMBLEM_BAK\n");
    }
    // Blowfish emblem...
    if (population_p->en->mimetype &&
	    strstr(population_p->en->mimetype,"blowfish")) 
    {
	emblem_pixbuf = rfm_get_pixbuf ("xffm/emblem_blowfish", icon_size/1.8);
	if (!insert_emblem(gdk_context, &icon_rect, emblem_pixbuf, "NW", 0.85))
	      NOOP(stderr, "ooooo--> emblem_blowfish\n");
    }

done:
    
    cairo_destroy(gdk_context);
    
    return buffer_surface;
}

// readlock should be enabled before entering this function...
//
static void
insert_pixbuf (view_t * view_p, 
	population_t * population_p,
	cairo_t *gdk_context) {


    rfm_global_t *rfm_global_p = rfm_global();

    g_mutex_lock(rfm_global_p->status_mutex);
    gint status = rfm_global_p->status;
    g_mutex_unlock(rfm_global_p->status_mutex);
    if(status == STATUS_EXIT) return ;
    if (!view_p){
	DBG("insert_pixbuf(): !view_p\n");
	return;
    }
    if (!population_p){
	//DBG("insert_pixbuf(): !population_p\n");
	return;
    }
    /*if (!population_p->pixbuf) {
	DBG("insert_pixbuf(): !population_p->pixbuf\n");
	return;
    }*/

    if (!(population_p->flags & POPULATION_PIXBUF_CLEAN)){
        GdkPixbuf *pixbuf = NULL;
	 pixbuf = rfm_get_pixbuf (population_p->icon_id, population_p->icon_size);//refs
	 if (pixbuf) {
	    if (population_p->pixbuf && G_IS_OBJECT(population_p->pixbuf)) 
                g_object_unref(population_p->pixbuf);
	    population_p->pixbuf = pixbuf;
	 }
	population_p->flags |= POPULATION_PIXBUF_CLEAN;
        pixbuf=NULL;

     } 
	
    GdkRectangle item_rect;
    if (!rfm_get_population_rect(view_p, population_p, &item_rect)) {
	DBG("insert_pixbuf(): !rfm_get_population_rect(view_p, population_p, &item_rect)\n");
	return;
    }

    GdkRectangle icon_rect;
    if (!rfm_get_population_icon_rect(view_p, population_p, &icon_rect)){
	DBG("insert_pixbuf(): !rfm_get_population_icon_rect(view_p, population_p, &icon_rect)\n");
	return;
    }
	    
    // Here's where the action starts...
    // Let's do the icon rendering off screen to 
    // avoid the clean background flash...
    cairo_surface_t *buffer_surface = get_pixbuf_surface(view_p, population_p, item_rect, icon_rect);
    if (!buffer_surface) {
        DBG("!buffer_surface: population_p->en->path= %s\n", population_p->en->path);
        return;
    }
    cairo_set_source_surface (gdk_context, buffer_surface, item_rect.x, item_rect.y);
    cairo_paint(gdk_context);
    cairo_surface_destroy(buffer_surface);
    return;
}

gint rodent_set_draw_clip(view_t *view_p, cairo_t *context){
#if GTK_MAJOR_VERSION>2
    GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");
    if (!scrolled_window) return 0;
    gdouble d_position=gtk_adjustment_get_value (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window));
	
    gdouble page=gtk_adjustment_get_page_size (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window));        
    NOOP("scroll position=%lf\n", d_position);

    // gtk-3.12 fix: since expose is coming from vpane, make sure clip
    // does not exit drawing area...
    GtkPaned *vpane = g_object_get_data(G_OBJECT(view_p->widgets.paper), "vpane");
    if (!vpane){
	// deskview should not have entered this function.
	g_error("rfm_layout_set_vpane_allocation: vpane==NULL\n");
    }
    //gint position = gtk_paned_get_position (vpane);
    GdkRectangle rect;  
    gtk_widget_get_allocation (GTK_WIDGET(vpane), &rect);

    //if (view_p->widgets.scrolled_window) {
   /* GtkAdjustment *adjustment = 
	    gtk_scrolled_window_get_vadjustment(view_p->widgets.scrolled_window);
    gdouble v = floor(gtk_adjustment_get_value(adjustment));
    gint iv = v;*/

    // clip...
    rect.x=0;
    rect.y=d_position; // 0;
    rect.height = page; //position;
    gdk_cairo_rectangle (context, &rect);
    cairo_clip(context);
#endif
    return 1;
}

static
void
render_item_list(view_t *view_p, GSList *render_list){
    if (!render_list) return;
    GSList *slist;
  
    cairo_t *context =  
	    gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
    rodent_set_draw_clip(view_p, context); // gtk-3.12 fix
  
    rfm_global_t *rfm_global_p = rfm_global();
    for (slist=render_list; slist && slist->data; slist=slist->next){
        g_mutex_lock(rfm_global_p->status_mutex);
        gint status = rfm_global_p->status;
        g_mutex_unlock(rfm_global_p->status_mutex);
        if (status == STATUS_EXIT) return;
	population_t *population_p=slist->data;
	if (!rodent_valid_population_p(view_p, population_p)) continue;

	insert_pixbuf (view_p, population_p, context);
	insert_layout (view_p, population_p, context);
    }
    cairo_destroy(context);
}

