
/*
 * Copyright (C) 2002-2014 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent.h"
#include "rfm_modules.h"

#define CONTROL_MODE (event->state & GDK_CONTROL_MASK)
#define NO_CLICK 0
#define LABEL_CLICK 1
#define ICON_CLICK 2

#include "rodent_mouse.h"
#include "rodent_mouse.i"

void
rodent_saturate_icon (view_t *view_p, population_t * population_p) {
    saturate_icon (view_p, population_p);
}

void
rodent_unsaturate_icon (view_t *view_p) {
    unsaturate_icon (view_p);
}

void
rodent_unsaturate_label (view_t *view_p) {
    unsaturate_label (view_p);
}
/*******************************************************************/
/*           GTK   signal callbacks */
/******************************************************************/

gboolean static 
double_click(view_t *view_p, const population_t *population_p, GdkEventButton * event){
    NOOP(stderr, "double_click now...\n");
    // Check for invalid call:
    rfm_global_t *rfm_global_p = rfm_global();
    if (!view_p || !population_p || !event) {
	rfm_cursor_reset(rfm_global_p->window);
	return TRUE;
    }
    // Check for NO_CLICK
    if (view_p->mouse_event.single_click_mode_state == NO_CLICK) {
	rfm_cursor_reset(rfm_global_p->window);
	return TRUE; 
    }

    
    if((CONTROL_MODE || event->state & GDK_SHIFT_MASK) &&
	    view_p->mouse_event.single_click_mode_state != LABEL_CLICK) {
	// Update the number of selected items in status line:
	rfm_update_status_line (view_p);
	rfm_cursor_reset(rfm_global_p->window);	
    } else {
	// Actual double click will be processed
	rodent_hide_tip();
	// disactivate lpterm.
	if (view_p->widgets.status) {
	    g_object_set_data (G_OBJECT (view_p->widgets.status), "active", NULL);
	}

        if(view_p->mouse_event.single_click_mode_state == ICON_CLICK) {
	    NOOP( "rodent_mouse: ICON_CLICK.\n");
            rodent_unselect_all_pixbuf (view_p);
            rfm_select_pixbuf (view_p, population_p);
	    // get new reselect list.
	    reset_reselect_list(view_p);
	    // this is threaded, so leave saturated...
            //rodent_unsaturate_item (view_p);
	    rfm_expose_item (view_p, population_p);
	    rfm_natural(RFM_MODULE_DIR, "callbacks", GINT_TO_POINTER(ACTIVATE), "callback");
        } else { // LABEL_CLICK
	    if (POPULATION_MODULE(population_p) && 
		    rfm_natural(PLUGIN_DIR, POPULATION_MODULE(population_p), 
			population_p->en, "label_click")){
		NOOP("rodent_mouse: module label click function called.\n");
	    } else if(population_p->en && !IS_ROOT_TYPE (population_p->en->type)
               && !IS_DUMMY_TYPE (population_p->en->type)
               && population_p->en->path) {
                gint caso = RENAME_ACTIVATE;
                if(event->state & GDK_CONTROL_MASK)
                    caso = DUPLICATE_ACTIVATE;
                if(event->state & GDK_MOD1_MASK)
                    caso = SYMLINK_ACTIVATE;
                if(event->state & GDK_MOD5_MASK)
                    caso = SYMLINK_ACTIVATE;
                if(event->state & GDK_SHIFT_MASK && event->state & GDK_CONTROL_MASK)
                    caso = SYMLINK_ACTIVATE;
                //view_p->mouse_event.selected_p = population_p;
		gboolean modular = view_p->en == NULL || 
		    view_p->en->module || 
		    population_p->en->module;
                if(!modular && g_path_is_absolute(population_p->en->path))
		{
		    rfm_select_pixbuf (view_p, population_p);
		    rfm_natural(RFM_MODULE_DIR, "callbacks", GINT_TO_POINTER(caso), "callback");
                } 
	    }
	    rfm_cursor_reset(rfm_global_p->window);
        }
    }
    view_p->mouse_event.single_click_mode_state = NO_CLICK;
    return TRUE;

}

gboolean
rodent_signal_on_button_press (GtkWidget * widget, 
	GdkEventButton * event, 
	gpointer data) 
{
    NOOP(stderr, "SINGLE_CLICK_MODE = %d, RFM_DOUBLE_CLICK_NAVIGATION=\"%s\" strlen=%d\n",
	    SINGLE_CLICK_MODE, 
	    getenv("RFM_DOUBLE_CLICK_NAVIGATION"),
	    getenv("RFM_DOUBLE_CLICK_NAVIGATION")?
	    strlen(getenv("RFM_DOUBLE_CLICK_NAVIGATION")):0);
    NOOP ("rodent_mouse:  button_press 0x%x(%d) x=%lf y=%lf\n", 
	    GPOINTER_TO_INT(event), event->button, event->x, event->y);

    if (event->button == 2) return TRUE;
    view_t *view_p = (view_t *) data;
    rfm_set_widget(&(view_p->widgets),"widgets_p");
    /* cancel any pending tip : */
    if(view_p->widgets.rename) {
	rfm_natural(RFM_MODULE_DIR, "callbacks", GINT_TO_POINTER(DONE_WITH_RENAME), "callback");
        return TRUE;
    }
    if (view_p->mouse_event.rubberbanding) return TRUE;
    if (!rfm_population_try_read_lock (view_p, "rodent_signal_on_button_press")) return TRUE;

    if(event->button == 1) {
        view_p->mouse_event.mouseX = event->x;
        view_p->mouse_event.mouseY = event->y;
    } else {
        view_p->mouse_event.mouseX = -1;
        view_p->mouse_event.mouseY = -1;
    }
    /* ignore size events until button released... */
    view_p->mouse_event.eventtime = event->time;

    const population_t *population_p = rodent_find_in_population (view_p, event->x, event->y);
    gboolean label_click = FALSE;
    if(!population_p) {
        population_p = rodent_find_in_labels (view_p, event->x, event->y);
	if(population_p) {
	    NOOP("rodent_mouse:  label click, population=0x%x\n",
		GPOINTER_TO_INT(population_p));
	    label_click = TRUE;
	}
    } else {
	NOOP("rodent_mouse:  icon click, population=0x%x\n",
		GPOINTER_TO_INT(population_p));
    }
    rfm_global_t *rfm_global_p = rfm_global();
    if (population_p && population_p->en && population_p->en->path){
	rfm_cursor_wait(rfm_global_p->window); 
	if (g_path_is_absolute(population_p->en->path)){
	    gchar *base = g_path_get_basename(population_p->en->path);
	    rfm_status(&(view_p->widgets), population_p->icon_id, base, NULL);
	    g_free(base);
	} else {
	    rfm_status(&(view_p->widgets), population_p->icon_id, population_p->en->path, NULL);
	}
    }

    if(event->button == 3) {
	// Popup menu, with or without associated icon.
        NOOP("rodent_mouse: popup menu requested.\n");
	rfm_cursor_reset(rfm_global_p->window);
        if(!population_p) {
            view_p->mouse_event.selected_p = NULL;
	    rodent_unselect_all_pixbuf (view_p);
	}
        view_p->mouse_event.doing_drag_p = NULL;
        view_p->mouse_event.dragstate = FALSE;
        view_p->mouse_event.rubberbanding = FALSE;
        view_p->mouse_event.old_X = 
	    view_p->mouse_event.old_Y =
	    view_p->mouse_event.boxX =
	    view_p->mouse_event.boxY = -1;
        rodent_hide_tip ();
        // button_popup locks the monitor loop
        rfm_population_read_unlock (view_p, "rodent_signal_on_button_press");
	/*g_error("examine...window:0x%x paper:0x%x (0x%x)",
		GPOINTER_TO_INT(gtk_widget_get_window(rfm_global_p->window)), 
		GPOINTER_TO_INT(gtk_widget_get_window(view_p->widgets.paper)), 
		GPOINTER_TO_INT(view_p->widgets.paper)
		    );*/
        button_popup (event, view_p, population_p);
	NOOP("rodent_mouse:  button_press 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
        return TRUE;
    }
    // this will start the rubberband box selection:
    if(!population_p && event->button == 1) {
        view_p->mouse_event.selected_p = NULL;
        NOOP (stderr, "rodent_mouse:  starting rubberband selection at %f %f\n",
		event->x, event->y);
        if(!(event->state & GDK_CONTROL_MASK)) {
            unselect_all_pixbuf (view_p);
            unsaturate_icon (view_p);
            view_p->mouse_event.selected_p = NULL;
	    if (view_p->en) {
		g_free(view_p->en->tag);
		view_p->en->tag = g_strdup(_("No file selected"));
		NOOP (stderr, "rodent_mouse: rfm_update_status_line\n");
	    }
       }
        view_p->mouse_event.rubberbanding = TRUE;
        view_p->mouse_event.boxX = event->x;
        view_p->mouse_event.boxY = event->y;
        rfm_population_read_unlock (view_p, "rodent_signal_on_button_press");
	NOOP("rodent_mouse:  button_press 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
        return TRUE;
    }

    if(event->button == 2) {
        return TRUE;
    }

    view_p->mouse_event.dragstate = FALSE;

    if(event->state & GDK_SHIFT_MASK && !label_click) {
	// Add items to the selection list.
	if (population_p && view_p->population_pp){
	    NOOP("rodent_mouse:  adding 0x%x to selection list\n",
		    GPOINTER_TO_INT(population_p));
	    population_t **pp, **to=NULL, **from=NULL;
	    for (pp=view_p->population_pp; pp && *pp; pp++){
		population_t *p=*pp;
		gboolean from_c=
		    !from && 
		    (rfm_find_in_selection_list(view_p, p->en)
		     || p==population_p);
		if (from_c) from=pp;
		gboolean to_c=
		    from &&  
		    (rfm_find_in_selection_list(view_p, p->en)
		     || p==population_p);
		if (to_c && pp!=from) to=pp;
	    }
	    if (*from != population_p && *to  != population_p) to=NULL;
	    if (!to) to=from;
	    rodent_unselect_all_pixbuf (view_p);
	    for (pp=from; pp && *pp; pp++){
		rfm_select_pixbuf (view_p, *pp);
		rfm_expose_item (view_p, *pp);
		if (pp==to) break;
	    }
	    // get new reselect list.
	    reset_reselect_list(view_p);
	}

        rfm_population_read_unlock (view_p, "rodent_signal_on_button_press");
	NOOP("rodent_mouse:  button_press 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
        return TRUE;
    }
    if(!CONTROL_MODE) {
	NOOP("rodent_mouse: selecting single population_p 0x%x\n",
		GPOINTER_TO_INT(population_p));
        rodent_unselect_all_pixbuf (view_p);
        rfm_select_pixbuf (view_p, population_p);
	rfm_expose_item (view_p, population_p);
	// get new reselect list.
	reset_reselect_list(view_p);   
    }

    /* set up drag status */
    view_p->mouse_event.doing_drag_p = NULL;
    view_p->mouse_event.dragstate = FALSE;
    view_p->mouse_event.rubberbanding = FALSE;
    view_p->mouse_event.old_X = view_p->mouse_event.old_Y = view_p->mouse_event.boxX = view_p->mouse_event.boxY = -1;
    int x_spacing;
    x_spacing = (CELLWIDTH(view_p) - population_p->icon_size) / 2;
    if(x_spacing < 0) {
        x_spacing = 0;
    }
    if(event->button == 1) {
	NOOP ("rodent_mouse: setting up drag for selection list\n");
	setup_drag_state (view_p, event);
    }

    if(SINGLE_CLICK_MODE || event->type == GDK_2BUTTON_PRESS) {
        if(label_click) {
            rodent_unselect_all_pixbuf (view_p);
            unsaturate_icon (view_p);
            view_p->mouse_event.single_click_mode_state = LABEL_CLICK;
        } else {
            view_p->mouse_event.single_click_mode_state = ICON_CLICK;
        }

	if(SINGLE_CLICK_MODE) {
	    rfm_population_read_unlock (view_p, "rodent_signal_on_button_press");
	    /* double click status gets processed on button release... */
	    NOOP("rodent_mouse:  button_press 0x%x(%d) done\n", 
		GPOINTER_TO_INT(event), event->button);
	    return TRUE;
	}
    }

    /* now we process doubleclick, if not set to SINGLE_CLICK_MODE */
    if(event->type == GDK_2BUTTON_PRESS) {
	double_click(view_p, population_p, event);
    }
    rfm_population_read_unlock (view_p, "rodent_signal_on_button_press");
    NOOP("rodent_mouse:  button_press 0x%x(%d) done\n", 
	GPOINTER_TO_INT(event), event->button);
    return TRUE;
}

gboolean
rodent_signal_on_button_release (GtkWidget * widget, GdkEventButton * event, gpointer data) {
    NOOP ("rodent_mouse:  button_release 0x%x(%d) x=%lf y=%lf\n", 
	    GPOINTER_TO_INT(event), event->button, event->x, event->y);

    view_t *view_p = (view_t *) data;

    if (view_p->mouse_event.rubberbanding && event->button != 1) return TRUE;
    const population_t *population_p;
    if(event->x < 0){
        event->x = 0;
    }
    if(event->y < 0){
        event->y = 0;
    }

    view_p->mouse_event.mouseX = -1;
    view_p->mouse_event.mouseY = -1;

    if (view_p->mouse_event.doing_drag_p) return TRUE;
    rfm_global_t *rfm_global_p = rfm_global();
    if(!rfm_population_try_read_lock (view_p, "rodent_signal_on_button_release")) {
	NOOP("rodent_mouse:  button_release 0x%x(%d) done\n", 
	    GPOINTER_TO_INT(event), event->button);
	rfm_cursor_reset(rfm_global_p->window);
        return TRUE;
    }

    NOOP("rodent_mouse: population_sem: rodent_signal_on_button_release() obtained...\n");
    population_p = rodent_find_in_population (view_p, (gdouble) event->x, (gdouble) event->y);
    if(!population_p) {
        population_p = rodent_find_in_labels (view_p, event->x, event->y);
    }

    if(population_p && CONTROL_MODE) {
        if(!(population_p->flags  & POPULATION_SELECTED)) {
	    NOOP("rodent_mouse: selecting single population_p 0x%x\n",
		GPOINTER_TO_INT(population_p));
	    rfm_select_pixbuf (view_p, population_p);
	    rfm_expose_item (view_p, population_p);
        } else {
	    NOOP("rodent_mouse: unselecting single population_p 0x%x\n",
		GPOINTER_TO_INT(population_p));
            rfm_unselect_pixbuf (view_p, population_p);
	    rfm_expose_item(view_p, population_p);
        }
	rfm_cursor_reset(rfm_global_p->window);
    }


    if(SINGLE_CLICK_MODE && (event->button == 1)) {
	double_click(view_p, population_p, event);
    } else if(event->button == 1) {
	rfm_cursor_reset(rfm_global_p->window);	
    }

    view_p->mouse_event.doing_drag_p = NULL;
    if (view_p->mouse_event.rubberbanding){
	view_p->mouse_event.rubberbanding = FALSE;
	// rubber band function will do the final expose.
	rubber_band (view_p, event->x, event->y, TRUE);

    }
    view_p->mouse_event.old_X = 
	view_p->mouse_event.old_Y = 
	view_p->mouse_event.boxX = 
	view_p->mouse_event.boxY = -1;
    
//////
    rfm_population_read_unlock (view_p, "rodent_signal_on_button_release");

    NOOP("rodent_mouse:  button_release 0x%x(%d) done\n", 
	GPOINTER_TO_INT(event), event->button);
    return TRUE;
}

gboolean
rodent_signal_on_motion (GtkWidget * widget, GdkEventMotion * event, gpointer data) {

    NOOP("rodent_mouse:  on_motion\n");
    view_t *view_p = (view_t *) data;
    //widgets_t *widgets_p = &(view_p->widgets);

    if(event->x < 0) event->x = 0;
    if(event->y < 0)  event->y = 0;
    view_p->mouse_event.current_mouseX = event->x;
    view_p->mouse_event.current_mouseY = event->y;
    
    // Rubberband box selecting
    rfm_global_t *rfm_global_p = rfm_global();
    if(view_p->mouse_event.rubberbanding) {
	XGrabServer(rfm_global_p->Xdisplay);
        rubber_band (view_p, event->x, event->y, FALSE);
        gint X = event->x - view_p->mouse_event.boxX;
        gint Y = event->y - view_p->mouse_event.boxY;
        if(X * X + Y * Y > 2){
            view_p->mouse_event.single_click_mode_state = NO_CLICK;
	}
	XUngrabServer(rfm_global_p->Xdisplay);
        return TRUE;
    }  

    // get semaphore or return:
    if(!rfm_population_try_read_lock (view_p, "rodent_signal_on_motion")) {
        return TRUE;
    }
    NOOP("rodent_mouse:  on_motion, population read-lock obtained...\n");


    // If icon has been dragged more than a set distance, mark population
    // as possible drag item.
    gint distanceX = SMALL_ICON_SIZE / 4;
    gint distanceY = SMALL_ICON_SIZE / 4;
    gint motionX = abs(view_p->mouse_event.current_mouseX - view_p->mouse_event.mouseX);
    gint motionY = abs(view_p->mouse_event.current_mouseY - view_p->mouse_event.mouseY);
    gboolean in_window = view_p->mouse_event.current_mouseX >= 0;
    if(in_window && (motionX > distanceX || motionY > distanceY)){
        view_p->mouse_event.doing_drag_p =
	    rodent_find_in_population (view_p, 
		    view_p->mouse_event.mouseX, 
		    view_p->mouse_event.mouseY);

    }

    // If drag item is set, and drag distance is greater than or equal to 4
    // then enter drag state.
    if(view_p->mouse_event.doing_drag_p) {
	if ((motionX * motionX) + (motionY * motionY) >= 4.0){
	    NOOP("rodent_mouse:  on motion, turning drag on... 0x%x\n",
		GPOINTER_TO_INT(view_p->mouse_event.doing_drag_p));
	    enter_drag_state (view_p);
	}
    }

    population_t *population_p = (population_t *)
	rodent_find_in_population (view_p, event->x, event->y);
    
    g_mutex_lock(view_p->mutexes.status_mutex);
    gint status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);

    if(population_p) {
    NOOP(stderr, "rodent_signal_on_motion: tip_function TRUE ... \n");
	unsaturate_label (view_p);
	saturate_icon (view_p, population_p);

	if(status != STATUS_EXIT && 
		getenv ("RFM_ENABLE_TIPS") && strlen (getenv ("RFM_ENABLE_TIPS"))) {
	    rodent_activate_tip(view_p, population_p, TRUE);
	} 
	rfm_population_read_unlock (view_p, "rodent_signal_on_motion");
	return TRUE;
    } 
    unsaturate_icon (view_p);    

    
    population_p = 
	    (population_t *)rodent_find_in_labels (view_p, event->x, event->y);
    if (!population_p) {
    NOOP(stderr, "rodent_signal_on_motion: tip_function FALSE ... \n");
	unsaturate_label (view_p);    
	rodent_hide_tip();
#if 10
	// XXX  this loop might be taxing to main thread...
	gint i;
	for (i=0; view_p->population_pp && view_p->population_pp[i]; i++) {
	    if (view_p->population_pp[i]) {
		view_p->population_pp[i]->flags &= (POPULATION_TIP_BUSY ^ 0xffffffff);
	    }
	}
#endif
	rfm_population_read_unlock (view_p, "rodent_signal_on_motion");
	return TRUE;
    } 

    saturate_label (view_p, population_p);
    //rodent_label_event (view_p, population_p);
    rfm_population_read_unlock (view_p, "rodent_signal_on_motion");
    return TRUE;
}


/**  drag events **********************************************/
// This signal is received by the receiving end of the drag/drop action
void
rodent_signal_drag_data (GtkWidget * widget,
                  GdkDragContext * context,
                  gint x, gint y, 
		  GtkSelectionData * selection_data, 
		  guint info, 
		  guint time, 
		  gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_data\n");

    const population_t *population_p;
    view_t *view_p = (view_t *) data;

    record_entry_t *target_en;
    NOOP ("rodent_mouse: drag_data: drag_data...\n");
    /*view_p = (view_t *)g_object_get_data(G_OBJECT(widget),"view_p"); */
    if(!view_p) {
        DBG ("rodent_signal_drag_data() view_p==NULL\n");
        gtk_drag_finish (context, FALSE, FALSE, time);
        return;
    }
    target_en = view_p->en;

    if(!target_en || !target_en->path) {
        NOOP ("rodent_mouse: drag_data: !target_en || !target_en->path\n");
        NOOP ("rodent_mouse: --DND>>rodent_signal_drag_data !target_en || !target_en->path\n");
        gtk_drag_finish (context, FALSE, FALSE, time);
        return;
    }

    // drop will proceed...
    if (!rfm_population_try_read_lock (view_p, "rodent_signal_drag_data")){
        gtk_drag_finish (context, FALSE, FALSE, time);
    }
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_cursor_wait (rfm_global_p->window);

    NOOP ("rodent_mouse: population_sem: rodent_signal_drag_data() obtained...\n");
    population_p = rodent_find_in_population (view_p, x, y);
    if(!population_p) {
        population_p = rodent_find_in_labels (view_p, x, y);
    }

    if(population_p && population_p->en && population_p->en->path) {
        if(IS_SDIR(population_p->en->type)){
            target_en = population_p->en;
	} else if (population_p->en->mimetype &&
		strcmp(population_p->en->mimetype, "application/x-desktop")==0){
            target_en = population_p->en;
	}
    }

    gchar *source_path = NULL;
    read_drag_info(&source_path, NULL);
    
    NOOP ("rodent_mouse: DND>>rodent_signal_drag_data: target entry is %s, source is %s\n", (target_en)?target_en->path:NULL, source_path);
    // Here we check if source and target are the same, in which case dnd should 
    // be ignored.
    if(!target_en || !source_path || !target_en->path ||
	    strcmp (target_en->path, source_path) == 0){
	NOOP("rodent_mouse: ignoring drop command! source_path=%s target=%s\n", source_path ,(target_en)?target_en->path:NULL);
        rfm_cursor_reset (rfm_global_p->window);
        drag_view_p = NULL;
        gtk_drag_finish (context, FALSE, FALSE, time);
        rfm_population_read_unlock (view_p, "rodent_signal_drag_data");
        return;
    }
    g_free(source_path);

    NOOP ("rodent_mouse: drag_data...gui_drag_data\n");
    if(gui_drag_data (&(view_p->widgets), target_en, context, x, y, selection_data, info, time)) {
        NOOP ("rodent_mouse: drag_data...reload 0x%lx->0x%lx\n", (unsigned long)view_p, (unsigned long)view_p->en);
    }

    NOOP ("rodent_mouse: drag_data...all done\n");
    rfm_cursor_reset (rfm_global_p->window);
    rfm_population_read_unlock (view_p, "rodent_signal_drag_data");
    NOOP ("population_sem: rodent_signal_drag_data() released!\n");
    return;
}
//#define NOOP NOOP

void
rodent_signal_drag_leave (GtkWidget * widget, GdkDragContext * drag_context, guint time, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_leave\n");

}

void
rodent_signal_drag_delete (GtkWidget * widget, GdkDragContext * context, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_delete\n");
}


gboolean
rodent_signal_drag_motion (GtkWidget * widget, 
	GdkDragContext * dc, gint x, gint y, guint t, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_motion\n");

    gboolean target_ok = FALSE;
    view_t *view_p = (view_t *) data;
    if (!rfm_population_try_read_lock (view_p, "rodent_signal_drag_motion")) return TRUE;
        
    
    NOOP ("rodent_mouse: population_sem: rodent_signal_drag_motion() obtained...\n");

    population_t *population_p = (population_t *) rodent_find_in_population (view_p, x, y);

    NOOP ("rodent_mouse: on_drag_motion...x=%d, y= %d, population_p=0x%lx\n", x, y, (unsigned long)population_p);
    rodent_hide_tip ();

    gboolean local_target = TRUE;
    gboolean local_source = TRUE;
    gint type=0;
    read_drag_info(NULL, &type);
    if (!IS_LOCAL_TYPE(type))local_source = FALSE;
    if (view_p->en && !IS_LOCAL_TYPE(view_p->en->type))local_target = FALSE;
    if(population_p) {
        /* if not valid drop target, return */

        if(POPULATION_MODULE(population_p)) {
            if(rfm_natural (PLUGIN_DIR, POPULATION_MODULE(population_p),
			population_p->en, "valid_drop_site"))
                target_ok = TRUE;
        } else {                /* local */
	    if (population_p->en && 
		population_p->en->path) {
		if (IS_SDIR(population_p->en->type)) {
		    target_ok = TRUE;
		    if (!IS_LOCAL_TYPE(population_p->en->type))local_target = FALSE;
		}

		if (population_p->en->mimetype && 
			strcmp(population_p->en->mimetype,
			    "application/x-desktop")==0) {
		    target_ok = TRUE;
		}
	    }
	}
    }
    
    if(view_p->mouse_event.saturated_p != population_p) {
	unsaturate_icon (view_p);
    }
    if (target_ok) {
	saturate_icon (view_p, population_p);
    }

    if(view_p->mouse_event.doing_drag_p) {
        NOOP ("rodent_mouse: widget ok\n");
    }
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_motion source=%s target=%s\n",
	    (local_source)?"local":"remote",
	    (local_target)?"local":"remote");
    

    if(getenv ("RFM_DRAG_DOES_MOVE") && strlen (getenv ("RFM_DRAG_DOES_MOVE")))
        view_p->mouse_event.drag_action = GDK_ACTION_MOVE;
    else
        view_p->mouse_event.drag_action = GDK_ACTION_COPY;

    // Override remote dnd with copy
    // when target or source is remote.
    if (!local_target || !local_source) {
        view_p->mouse_event.drag_action = GDK_ACTION_COPY;
    } 

#if GTK_MAJOR_VERSION==2
    gint actions = dc->actions;
#else
    gint actions = gdk_drag_context_get_actions(dc);
#endif
    if(actions == GDK_ACTION_MOVE)
        gdk_drag_status (dc, GDK_ACTION_MOVE, t);
    else if(actions == GDK_ACTION_COPY)
        gdk_drag_status (dc, GDK_ACTION_COPY, t);
    else if(actions == GDK_ACTION_LINK)
        gdk_drag_status (dc, GDK_ACTION_LINK, t);
    else if(actions & view_p->mouse_event.drag_action)
        gdk_drag_status (dc, view_p->mouse_event.drag_action, t);
    else
        gdk_drag_status (dc, 0, t);
    rfm_population_read_unlock (view_p, "rodent_signal_drag_motion");
    NOOP ("rodent_mouse: population_sem: rodent_signal_drag_motion() released!\n");
    return (TRUE);
}

// This signal is received by the sending end of the drag/drop event
void
rodent_signal_drag_data_get (GtkWidget * widget,
                      GdkDragContext * context, GtkSelectionData * selection_data, guint info, guint time, gpointer data) {
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_data_get\n");
    view_t *view_p = (view_t *) data;
    rodent_hide_tip ();
    gui_drag_data_get (&(view_p->widgets), view_p->selection_list, context, selection_data, info, time);
    NOOP ("rodent_mouse: drag_data_get: all done\n");
}

void
rodent_signal_drag_begin (GtkWidget * widget, GdkDragContext * drag_context, gpointer data) {
    view_t *view_p = (view_t *) data;
    drag_view_p = view_p;
    rodent_hide_tip ();
    if (!view_p->en || !view_p->en->path) return; 
    write_drag_info(view_p->en->path, view_p->en->type);
    view_p->mouse_event.drag_event.context = drag_context;
}

void
rodent_signal_drag_end (GtkWidget * widget, GdkDragContext * context, gpointer data) {
    view_t *view_p = (view_t *) data;
    widgets_t *widgets_p = &(view_p->widgets);
    //rfm_diagnostics(widgets_p, "xffm_tag/red","rodent_mouse: DND>> rodent_signal_drag_end\n", NULL); 
    NOOP ("rodent_mouse: DND>> rodent_signal_drag_end\n" );
    view_p->mouse_event.doing_drag_p = NULL;
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_cursor_reset(rfm_global_p->window);
    
    // Immediate test for a reload condition by the thread monitor.
    if (!xfdir_monitor_control_greenlight(widgets_p)){
	rodent_trigger_reload(view_p);
    }



    if(dnd_data) {
        g_free (dnd_data);
        dnd_data = NULL;
    }
    drag_view_p = NULL;
    NOOP ("rodent_mouse: drag_end... alldone\n");
    // Remove old MIT-shm  dnd info.
    shm_unlink (DND_SHM_NAME);
}

/* end of drag signals ********************************/

void
rodent_create_target_list (view_t * view_p) {
    if(view_p->mouse_event.target_list)
        return;
    view_p->mouse_event.target_list = gtk_target_list_new (target_table, NUM_TARGETS);
/*    this does not seem to be necessary
 *    gtk_drag_source_set ((GtkWidget *) view_p->widgets.paper,
                         GDK_BUTTON1_MASK | GDK_BUTTON2_MASK, target_table,
                         NUM_TARGETS, GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);*/
    gtk_drag_dest_set ((GtkWidget *) view_p->widgets.paper,
                       GTK_DEST_DEFAULT_DROP, target_table, NUM_TARGETS, GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK);
    return;
}


void
rodent_unselect_all_pixbuf (view_t * view_p) {
    NOOP(stderr, "rodent_unselect_all_pixbuf\n");
    unselect_all_pixbuf(view_p);
}

///////////////////////
/*
test(view_t * view_p, population_t * population_p){
    GdkRectangle rect;
    rfm_get_population_label_rect(view_p, population_p, &rect);
    cairo_t *gdk_context =  
		gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
	    
    cairo_set_source_rgb(gdk_context, 0.05, 0.05, 0.05);
    cairo_move_to (gdk_context, rect.x, rect.y);
	if(population_p->layout_full){
	    pango_cairo_show_layout (gdk_context, population_p->layout_full);
	} else {
	    pango_cairo_show_layout (gdk_context, population_p->layout);
	}	    

    cairo_destroy(gdk_context);
}
*/



