#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/* This file is included by rodent_mouse.c */
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

/*************************************************************************/
/******************   dnd functions *************************************/
/************************************************************************/

#define MAXURILEN 4096          /* Longest URI to allow */

#define DRAG_TYPE_UNDEFINED	0
#define DRAG_TYPE_LOCAL		0x01
#define DRAG_TYPE_NET		0x02
#define DRAG_TYPE_INCONSISTENT	0x04


static GtkTargetEntry target_table[] = {
    {"text/uri-list", 0, TARGET_URI_LIST},
    {"text/x-moz-url", 0, TARGET_MOZ_URL},
    {"text/plain", 0, TARGET_PLAIN},
    {"UTF8_STRING", 0, TARGET_UTF8},
    {"STRING", 0, TARGET_STRING}
};

#define NUM_TARGETS (sizeof(target_table)/sizeof(GtkTargetEntry))
static gchar *dnd_data = NULL;
static view_t *drag_view_p = NULL;

extern RfmRWLock drag_info_lock;

#define DND_SHM_NAME "/rfm-dnd"

/*************   core *****************/
#if 0
void
on_drag_data_delete (GtkWidget * widget, GdkDragContext * context, gpointer data) {
    NOOP("rodent_mouse: on_drag_data_delete!\n\n");

    return;
}
#endif


static void
read_drag_info(gchar **path_p, gint *type_p) {
    rfm_rw_lock_reader_lock(&drag_info_lock);
    // get shared dnd-info pointer
    gint fd = shm_open (DND_SHM_NAME, O_RDONLY, S_IRUSR | S_IWUSR);
    if(fd < 0){
	NOOP("rodent_mouse: unable to get shm-dnd-info. Assuming local...\n");
    } else {
	// Figure out the size.
	void *p = mmap (NULL, sizeof(gint), 
		  PROT_READ, MAP_SHARED, fd, 0);
        gint length = *((gint *)p);
	if(msync (p, sizeof(gint), MS_SYNC) < 0){
	    DBG ("msync(%s): %s\n", DND_SHM_NAME, strerror (errno));
	}
	munmap (p, sizeof(gint));
	
	p = mmap (NULL, length, 
	    PROT_READ, MAP_SHARED, fd, 0);
	close(fd);
	if (type_p) *type_p = *((gint *)(p + sizeof(gint)));
	if (path_p) *path_p = g_strdup((gchar *)(p + (2*sizeof(gint))));

    }
    rfm_rw_lock_reader_unlock(&drag_info_lock);

    return;
}
static void write_drag_info(const gchar *path, const gint type){
    gint size = sizeof(gint)*2 + strlen(path) +1;
    NOOP("rodent_mouse: DND>> rodent_signal_drag_begin: size=%d (type:0x%x) %s\n", 
	    size, type, path);
    rfm_rw_lock_writer_lock(&drag_info_lock);


    // Remove old MIT-shm  dnd info (if any)
    shm_unlink (DND_SHM_NAME);
    
    gint fd = shm_open (DND_SHM_NAME, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    if(fd < 0){
        g_error ("rodent_signal_drag_begin(): shm_open(%s): %s", DND_SHM_NAME, strerror (errno));
    }

    // Truncate to necessary memory size to allocate.
    if(ftruncate (fd, size) < 0) {
        g_error ("rodent_signal_drag_begin(): ftruncate(%s): %s", DND_SHM_NAME, strerror (errno));
    }

    // Get a shared memory pointer.
   void *p = mmap (NULL, sizeof(size), 
	    PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    // Initialize to zero
    memset(p, 0, sizeof(size));

    // Save record size.
    memcpy(p, &size, sizeof(gint));
    // Save source type.
    memcpy(p+sizeof(gint), &type, sizeof(gint));
    // Save source path (null byte is set in initialization)
    memcpy(p+(2*sizeof(gint)), path, strlen(path));
    // Put in shared memory.
    if(msync (p, sizeof(size), MS_SYNC) < 0){
        DBG ("rodent_signal_drag_begin(): msync(%s): %s\n", DND_SHM_NAME, strerror (errno));
    }
    // Release map so other processes may shm_unlink.
    munmap (p, sizeof(size));
    // release writelock on shm_name file descriptor

    // Close file descriptor
    close(fd);
    rfm_rw_lock_writer_unlock(&drag_info_lock);

}
/*
 * DND sender: prepare data for the remote receiver.
 * event: drag_data_get
 */
static void
gui_drag_data_get (widgets_t * widgets_p,
                   GSList * drag_entry_list, 
		   GdkDragContext * context, 
		   GtkSelectionData * selection_data, 
		   guint info, 
		   guint time) {
    char *files;
    GSList *tmp;
    record_entry_t *en;
    int selection_len;
    //int drag_type;
    gchar *format = NULL;
    gchar *me,
     *she;

    if(!drag_entry_list || !g_slist_length (drag_entry_list)
       || !drag_entry_list->data) {
	// This should never happen.
	DBG("gui_drag_data_get(): no selection list\n");
        return;
    }
    en = (record_entry_t *) drag_entry_list->data;

    me = g_strdup (g_get_host_name ());
    
#if GTK_MAJOR_VERSION==2
    she = rfm_host_name (GDK_WINDOW_XID (context->dest_window));
#else
    she = rfm_host_name (GDK_WINDOW_XID (gdk_drag_context_get_dest_window (context)));
#endif
    if (me && she && strcmp(me,she)){
	DBG("DnD between clients running on different hosts is not supported.\n");
        if(dnd_data) {
            g_free (dnd_data);
            dnd_data = NULL;
        }
        return;
    }
    view_t *view_p = widgets_p->view_p;
    if(view_p->en) {
        NOOP("rodent_mouse: DND send, (%s), me=%s --> she=%s\n", view_p->en->path, me, she);
    }

    if(en->module) {
        const gchar *fmt = rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_format");
        if(fmt)
            format = g_strdup (fmt);
        NOOP("rodent_mouse: DND send, module format=%s\n", (format ? format : "null"));
    } else
        NOOP("rodent_mouse: DND send, not module format\n");
    if(!format) {
        //drag_type = DRAG_TYPE_LOCAL;
        if(strcmp (me, she)) {
            struct passwd *pw = getpwuid (getuid ());
            if(pw) {
                format = g_strdup_printf ("file://%s@%s", pw->pw_name, me);
            } else {
                format = g_strdup_printf ("file://%s", me);
            }
        } else
            format = g_strdup ("file:");
    }
    g_free (me);
    g_free (she);
    NOOP("rodent_mouse: DND send, format=%s\n", (format ? format : "null"));

    /* prepare data for the receiver */
    switch (info) {
    case TARGET_RAW:
        DBG ("TARGET_RAW\n");
    case TARGET_UTF8:
        DBG ("TARGET_UTF8\n");
    case TARGET_URI_LIST:
        NOOP("rodent_mouse: DND send, TARGET_URI_LIST\n");
    default:
        selection_len = 0;
        if(dnd_data) {
            g_free (dnd_data);
            dnd_data = NULL;
        }
        /* count length of bytes to be allocated */
        for(tmp = drag_entry_list; tmp; tmp = tmp->next) {
            const gchar *dndpath;
            en = (record_entry_t *) tmp->data;
            if(!en || !en->path || !strlen (en->path))
                continue;
            if(en->module && rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path")) {
                dndpath = rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path");
            } else {
                dndpath = en->path;
            }
            /* 2 is added for the \r\n */
            selection_len += (strlen (dndpath) + strlen (format) + 2);
        }
        /* 1 is added for terminating null character */
        dnd_data = files = g_malloc (selection_len + 1);
	if (!dnd_data) g_error("malloc: %s", strerror(errno));
        memset (files, 0, selection_len + 1);
        for(tmp = drag_entry_list; tmp; tmp = tmp->next) {
            const gchar *dndpath;
            en = (record_entry_t *) tmp->data;
            if(!en || !en->path || !strlen (en->path))
                continue;
            if(en->module && rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path")) {
                dndpath = rfm_natural (PLUGIN_DIR, en->module, en, "get_dnd_path");
            } else {
                dndpath = en->path;
            }
            sprintf (files, "%s%s\r\n", format, dndpath);
            files += (strlen (format) + strlen (dndpath) + 2);
        }
        break;
    }
    NOOP("rodent_mouse: DND send, drag data is:%s\n", dnd_data);
    gtk_selection_data_set (selection_data, 
	    gtk_selection_data_get_selection(selection_data),
	    8, (const guchar *)dnd_data, selection_len);
    g_free (format);
}

static gboolean
gui_drag_data (widgets_t * widgets_p,
               record_entry_t * target_en,
               GdkDragContext * context,
	       gint x, gint y, 
	       GtkSelectionData * data,
	       guint info,
	       guint time) {
    int the_mode = TR_MOVE;
    int nitems, action;
    gchar *url;
    int mode = 0;
    GList *list = NULL;
    gboolean result = FALSE;
    gchar *he, *me;

    //if(!target_en || data->length < 0 || data->format != 8|| !data->data) {
    if(!target_en) {
	// We should never get this warning. If we do, something is 
	// terribly wrong.
        DBG ("gui_drag_data(): !target_en || data->length < 0 || data->format != 8 || !data->data\n");
        goto drag_over;         /* of course */
    }

    me = g_strdup (g_get_host_name ());
#if GTK_MAJOR_VERSION==2
    he = rfm_host_name (GDK_WINDOW_XID (context->source_window));
#else
    he = rfm_host_name (
	    GDK_WINDOW_XID (gdk_drag_context_get_source_window (context)));
#endif
 
    view_t *view_p = widgets_p->view_p;
    if(view_p->en) {
        NOOP("rodent_mouse: *DND receive, (%s), me=%s --> she=%s\n", view_p->en->path, me, he);
    }

    /* remote instance may have full format specification,
     * or borked specification. Here we must consider
     * both cases */


#if GTK_MAJOR_VERSION==2
    if(context->action <= GDK_ACTION_DEFAULT) {
#else
    if(gdk_drag_context_get_selected_action(context) <= GDK_ACTION_DEFAULT) {
#endif   
        if(getenv ("RFM_DRAG_DOES_MOVE")
           && strlen (getenv ("RFM_DRAG_DOES_MOVE"))) {
            action = GDK_ACTION_MOVE;
        } else {
            action = GDK_ACTION_COPY;
        }
    } else {
#if GTK_MAJOR_VERSION==2
        action = context->action;
#else
        action = gdk_drag_context_get_selected_action(context);
#endif
    }

    NOOP("rodent_mouse: DND receive, info=%d (%d,%d)\n", info, TARGET_STRING, TARGET_URI_LIST);
    if(!(info == TARGET_STRING) && !(info == TARGET_URI_LIST)
       && !(info == TARGET_MOZ_URL)) {
	// Here we have something unknown in the drag.
        goto drag_over;         /* of course */
    }

    NOOP("rodent_mouse: DND receive, action=%d\n", action);
    if(action == GDK_ACTION_MOVE)
        the_mode = mode = TR_MOVE;
    else if(action == GDK_ACTION_COPY)
        the_mode = mode = TR_COPY;
    else if(action == GDK_ACTION_LINK)
        the_mode = mode = TR_LINK;
    else {
	DBG("Drag drop mode is not GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK\n");
        goto drag_over;         /* of course */
    }

    NOOP("rodent_mouse: DND receive, drag data=%s\n", (const char *)gtk_selection_data_get_data (data));

    nitems = rfm_uri_parse_list ((const char *)gtk_selection_data_get_data (data), &list);

    NOOP("rodent_mouse: DND receive, nitems=%d\n", nitems);
    if(!nitems) {
	DBG("number of items in drag is zero!\n");
        goto drag_over;         /* of course */
    }

    /***/


    /* if target is a plugin, let the plugin take care of business. */
    if(target_en->module) {
        rfm_uri_remove_file_prefix_from_list (list, he, me);
        NOOP("rodent_mouse: DND receive, en->module=%s\n", target_en->module);
        if(rfm_natural (PLUGIN_DIR, target_en->module, target_en, "valid_drop_site")) {
            NOOP("rodent_mouse: DND receive, module: valid_drop_site for %s\n", target_en->module);
            rfm_natural (PLUGIN_DIR, target_en->module, target_en, "set_drop_entry");
            if(rfm_complex (PLUGIN_DIR, target_en->module, widgets_p, target_en->path, list,  "process_drop")) {
                NOOP("rodent_mouse: DND receive, module: process_drop ok\n");
                result = TRUE;
            }
            rfm_void (PLUGIN_DIR, target_en->module, "clear_drop_entry");
            list = rfm_uri_free_list (list);
            goto drag_over;
        }
    }


    /* now determine whether cp or scp should be used 
     * (we ignore the format for this determination, 
     * because other applications may bork this)*/
    rfm_uri_remove_file_prefix_from_list (list, he, me);

    /* if target is dotdesktop type, take individual action */
    if (target_en->mimetype &&
		strcmp(target_en->mimetype, "application/x-desktop")==0){
	if (rfm_complex(PLUGIN_DIR, "dotdesktop", widgets_p, target_en->path, list, "process_drop")) {
	    NOOP("rodent_mouse: DND receive, Target is dotdesktop: %s\n", target_en->path);
	}
	list = rfm_uri_free_list (list);
	result = TRUE;
	goto drag_over;
    }


    /* local file cp/ln/mv */
    url = list->data;
    {
        /* nonsense check */
        struct stat st;
        lstat (url, &st);
	// Here we check if the file source and destination is actually 
	// the same thing, this time by stat information instead of
	// path string.
	// This is a more robust test. We must test *both* st_ino and
	// st_dev, because stuff on different devices may (and do) have
	// equal st_ino.
        if(target_en->st && 
		st.st_ino == target_en->st->st_ino &&
		st.st_dev != target_en->st->st_dev)
	{
            list = rfm_uri_free_list (list);
            rfm_diagnostics(&(view_p->widgets),"xffm/stock_dialog-warning",NULL);
            rfm_diagnostics (widgets_p, "xffm_tag/stderr", " ", strerror (EEXIST), ": ", target_en->path, "\n", NULL);
            goto drag_over;
        }
#if 0
	// This is another check. Most probably superceded by inode test above
        if(!S_ISDIR (st.st_mode) && strchr (url, '/')) {
            char *p;
            p = g_strdup (url);
            *(strrchr (p, '/')) = 0;
            if(target_en->path && strcmp (p, target_en->path) == 0) {
                list = rfm_uri_free_list (list);
                g_free (p);
                p = NULL;
                rfm_show_text(widgets_p);
                rfm_diagnostics (widgets_p, "xffm/stock_dialog-error", NULL);
                rfm_diagnostics (widgets_p, "xffm_tag/stderr", strerror (EEXIST), ": ", target_en->path, "\n", NULL);

                /*NOOP(stderr,"dbg:nonsense 2\n"); */
                goto drag_over;
            }
            g_free (p);
            p = NULL;
        }
#endif
    }

    gint type=0;
    gboolean local_target = TRUE;
    gboolean local_source = TRUE;
    read_drag_info(NULL, &type);
    if (!IS_LOCAL_TYPE(type))local_source = FALSE;
    if (!IS_LOCAL_TYPE(target_en->type))local_target = FALSE;

    NOOP("rodent_mouse: DND receive, local target = %s\n",
	    (local_target)?"TRUE":"FALSE");
    if (!local_target){
	switch (mode){
	    case TR_COPY:
		mode = TR_COPY_REMOTE;

		break;
	    case TR_MOVE:
		mode = TR_MOVE_REMOTE;
		break;
	    case TR_LINK:
		mode = TR_LINK_REMOTE;
		break;
	    case TR_RENAME:
		mode = TR_RENAME_REMOTE;
		break;
	}

    }
    gchar *text=NULL;
    const gchar *icon=NULL;
    if (!local_target){
      switch (mode){
        case TR_COPY_REMOTE:
	case TR_MOVE_REMOTE:
	    icon = "xffm/emblem_network/compositeSW/stock_go-forward";
	    text = g_strdup_printf(_("Uploading file %s"), "...");
	    break;
	default:
	    break;
      }
    } else if (!local_source){
      switch (mode){
        case TR_COPY:
	case TR_MOVE:
	    icon = "xffm/emblem_network/compositeSW/go-back";
	    text = g_strdup_printf(_("Downloading file %s..."), "");
	    break;
	default:
	    break;
      }
    } 
    if (text) {
	rfm_diagnostics(widgets_p, "xffm/emblem_network/compositeSW/go-last", NULL);
	rfm_diagnostics(widgets_p, icon, NULL);
	rfm_diagnostics(widgets_p, "xffm_tag/red", text, "\n", NULL);
	g_free(text);
    }
    rfm_complex(RFM_MODULE_DIR, "callbacks", GINT_TO_POINTER(mode), list, target_en->path, "cp"); 
    // deprecated:rodent_cp (mode, widgets_p, list, target_en->path);

    list = rfm_uri_free_list (list);
    result = TRUE;
  drag_over:
    g_free (me);
    g_free (he);
    gtk_drag_finish (context, TRUE, (the_mode & TR_MOVE) ? TRUE : FALSE, time);
    NOOP("rodent_mouse: DND receive, drag_over\n");
    return result;
}



static void setup_drag_state (view_t * view_p, GdkEventButton * event);

static void rubber_band (view_t * view_p, int x, int y, gboolean draw);

static void 
enter_drag_state (view_t * view_p) {
    NOOP("rodent_mouse: enter dragstate \n");

    if(view_p->mouse_event.dragstate) {
        NOOP("rodent_mouse: dragstate true\n");
        return;
    }
    NOOP("rodent_mouse: now entering dragstate: event=0x%lx\n", (unsigned long)(&(view_p->mouse_event.drag_event)));
    //NOOP("rodent_mouse: now entering dragstate: G_IS_OBJECT (event)=%d\n",G_IS_OBJECT (&(view_p->mouse_event.drag_event)));

#if GTK_MAJOR_VERSION>2 && GTK_MINOR_VERSION>9
    view_p->mouse_event.drag_event.context = gtk_drag_begin_with_coordinates (view_p->widgets.paper,
                   view_p->mouse_event.target_list,
                   GDK_ACTION_MOVE | GDK_ACTION_COPY | GDK_ACTION_LINK, 
	           1, //drag button
		   (GdkEvent *) (&(view_p->mouse_event.drag_event)),
                   -1, -1);

#else  
    view_p->mouse_event.drag_event.context = gtk_drag_begin (view_p->widgets.paper,
                                                 view_p->mouse_event.target_list,
                                                 GDK_ACTION_MOVE |
                                                 GDK_ACTION_COPY |
                                                 GDK_ACTION_LINK, 
						 1, //drag button
						 (GdkEvent *) (&(view_p->mouse_event.drag_event)));
#endif

    NOOP("rodent_mouse: drag begun...\n");
    if(!view_p->mouse_event.drag_event.context)
        return;
    gdk_drag_status (view_p->mouse_event.drag_event.context, view_p->mouse_event.drag_action, view_p->mouse_event.drag_event.time);
    NOOP("rodent_mouse: drag status...\n");


    gchar *plural_text=g_strdup_printf (
        ngettext ("%'u item", "%'u items",g_slist_length(view_p->selection_list)),
	g_slist_length(view_p->selection_list));
    gchar *g = g_strdup_printf ("%s: %s", _("Selection"), plural_text);
    g_free(plural_text);

    rfm_status (&(view_p->widgets), "xffm/stock_dialog-info", g, NULL);
    g_free (g);
    
    if(g_slist_length(view_p->selection_list) > 1) {
        NOOP("rodent_mouse: selection_count > 1\n");
        gtk_drag_set_icon_name (view_p->mouse_event.drag_event.context, "edit-copy", 0, 0);
        //gtk_drag_set_icon_stock (view_p->mouse_event.drag_event.context, GTK_STOCK_DND_MULTIPLE, 0, 0);
    } else if(view_p->mouse_event.dnd_pixbuf) {
        NOOP("rodent_mouse: setting view_p->mouse_event.dnd_pixbuf\n");
        gtk_drag_set_icon_pixbuf (view_p->mouse_event.drag_event.context, view_p->mouse_event.dnd_pixbuf, 0, 0);
    } else {
        gtk_drag_set_icon_name (view_p->mouse_event.drag_event.context, "edit-copy", 0, 0);
        //gtk_drag_set_icon_name (view_p->mouse_event.drag_event.context, "document", 0, 0);
        // stock icons are deprecated:
        //gtk_drag_set_icon_stock (view_p->mouse_event.drag_event.context, GTK_STOCK_DND, 0, 0);
    }
    view_p->mouse_event.dragstate = TRUE;
    NOOP("rodent_mouse: enter dragstate done\n");

}

static void
setup_drag_state (view_t * view_p, GdkEventButton * event) {
    NOOP("rodent_mouse: +DND>>setup_drag_state\n");

    NOOP("rodent_mouse: +DND>>setup_drag_state\n");
    view_p->mouse_event.drag_event.type = GDK_DRAG_ENTER;
    view_p->mouse_event.drag_event.x_root = event->x;
    view_p->mouse_event.drag_event.y_root = event->y;
    view_p->mouse_event.drag_event.time = event->time + 2;
    view_p->mouse_event.drag_event.window = event->window;
    view_p->mouse_event.drag_event.send_event = event->send_event;

    NOOP("rodent_mouse: event time=0x%x\n", event->time);
}

/*************************************************************************/
/******************   end of dnd functions *******************************/
/*************************************************************************/



static void
unsaturate_label (view_t *view_p) {
    if (view_p->mouse_event.label_p){
        population_t *p = (population_t *)view_p->mouse_event.label_p;
	p->flags &= (LABEL_SATURATED ^ 0xffffffff);
        rfm_expose_label(view_p, view_p->mouse_event.label_p);
	view_p->mouse_event.label_p = NULL;
    }
}

static void
saturate_label (view_t *view_p, population_t * population_p) {
    if(!view_p || !population_p) return;
    //overkill: if (!rodent_valid_population_p(view_p, population_p)) return;
    if (!population_p->en || !population_p->en->path) return;
    if (!rfm_population_try_read_lock(view_p, "saturate_label")) {
	NOOP (stderr, "rodent_mouse: >> rodent_label_event: !rfm_population_try_read_lock\n");
	return;
    }

    if(view_p->mouse_event.label_p != population_p) {
	if (view_p->mouse_event.label_p){
	    population_t *p = (population_t *)view_p->mouse_event.label_p;
	    // Turn unsaturated item off
	    p->flags &= (LABEL_SATURATED ^ 0xffffffff);
	    rfm_expose_label(view_p, view_p->mouse_event.label_p);
	}
	// saturate new item
	population_p->flags |= LABEL_SATURATED;
	view_p->mouse_event.label_p = population_p;

	gboolean do_label_tip =  getenv("RFM_ENABLE_LABEL_TIPS") && 
	    strlen(getenv("RFM_ENABLE_LABEL_TIPS"));
	if (do_label_tip) {
	    rodent_activate_tip(view_p, population_p, FALSE);
	}
	rfm_expose_label (view_p, view_p->mouse_event.label_p);
	view_p->flags.saturation_serial++;
    }
    rfm_population_read_unlock(view_p, "saturate_label");

    return;
}


static void
unsaturate_icon (view_t *view_p) {
    if (view_p->mouse_event.saturated_p){
	population_t *p = (population_t *)view_p->mouse_event.saturated_p;
	p->flags &= (POPULATION_SATURATED ^ 0xffffffff);
	// Label has also a different view with icon saturation.
	// This is why e expose the whole item.
	rfm_expose_item(view_p, view_p->mouse_event.saturated_p);
	view_p->mouse_event.saturated_p = NULL;
    }
}

static void
saturate_icon (view_t *view_p, population_t * population_p) {
    unsaturate_label (view_p);    
    if(population_p->pixbuf) {
	view_p->mouse_event.dnd_pixbuf=population_p->pixbuf;
    }
    if(view_p->mouse_event.saturated_p != population_p) {
	if (view_p->mouse_event.saturated_p){
	    population_t *p = (population_t *)view_p->mouse_event.saturated_p;
	    // Turn unsaturated item off
	    p->flags &= (POPULATION_SATURATED ^ 0xffffffff);
	    rfm_expose_item(view_p, view_p->mouse_event.saturated_p);
	}
	// saturate new item
	population_p->flags |= POPULATION_SATURATED;
	view_p->mouse_event.saturated_p = population_p;
	rfm_expose_item(view_p, view_p->mouse_event.saturated_p);
	view_p->flags.saturation_serial++;
    }
}


static void
unselect_all_pixbuf(view_t * view_p) {

    NOOP("rodent_mouse: >> unselect_all_pixbuf\n");
    population_t **tmp;
    GSList *list = NULL;
    for(tmp = view_p->population_pp; tmp && *tmp; tmp++) {
        population_t *population_p = *tmp;
        if(!population_p)
            continue;
        if(population_p == view_p->mouse_event.doing_drag_p)
            continue;

        if (population_p->flags  & POPULATION_SELECTED) {
	    rfm_unselect_pixbuf (view_p, population_p);
	    rfm_expose_item(view_p, population_p);
	}
        if (population_p->flags  & LABEL_SATURATED) {
	    population_p->flags  &= (LABEL_SATURATED ^ 0xffffffff);
	    rfm_expose_label(view_p, population_p);
        }
    }
    if(view_p->selection_list){
	list=view_p->selection_list;
	for (;list && list->data; list=list->next){
	    record_entry_t *en=list->data;
	    rfm_destroy_entry(en);
	}
        g_slist_free (view_p->selection_list);
    }
    view_p->selection_list = NULL;
}


static gboolean
scroll_business(view_t *view_p, gint y, GdkRectangle *area){
    GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");

    if (!GTK_IS_SCROLLED_WINDOW (scrolled_window)) return FALSE;
    double upper = gtk_adjustment_get_upper (
	    gtk_scrolled_window_get_vadjustment (scrolled_window));
    double page = gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (scrolled_window));
    double value = gtk_adjustment_get_value  (
	    gtk_scrolled_window_get_vadjustment (scrolled_window));
    gboolean set_scroll=FALSE;
    gdouble new_value = 0.0;
    if (y > value + page && value + page < upper) {
        NOOP("rodent_mouse: scrolldown: y= %d value =%lf, upper=%lf page=%lf \n",
		 y, value, upper, page);
	new_value = (y - page < upper - page)? 
	    y - page : upper - page;
	NOOP("rodent_mouse: scrolldown to %lf\n", new_value );
	set_scroll=TRUE;
    } else if (y < value) {
	new_value = y;
	NOOP("rodent_mouse: scrollup to %lf\n", new_value);
        set_scroll=TRUE;
    } else {
         NOOP("rodent_mouse: scroll noop: y= %d value =%lf, upper=%lf page=%lf \n",
		 y, value, upper, page);
   }
    if (set_scroll){
        gtk_adjustment_set_value (
		gtk_scrolled_window_get_vadjustment (
		    scrolled_window), 
		new_value);
    }
    return set_scroll;
}

static void
reset_reselect_list(view_t *view_p){
    GSList *list = view_p->reselect_list;
    for (;list && list->data; list=list->next) g_free(list->data);
    if (view_p->reselect_list) g_slist_free(view_p->reselect_list);
    view_p->reselect_list =NULL;
    list = view_p->selection_list;
    for (; list && list->data; list=list->next){
	record_entry_t *en = list->data;
	if (!en || !en->path) continue;
	view_p->reselect_list = 
	    g_slist_prepend(view_p->reselect_list, g_strdup(en->path));
    }
}


static void
rubber_band (view_t * view_p, int x, int y, gboolean draw) {
    NOOP(stderr, "rodent_mouse: >> rubber_band: %d\n", draw);
    if ((view_p->mouse_event.boxX == -1 && view_p->mouse_event.boxY == -1)
	    ||
       (view_p->mouse_event.old_X == -1 && view_p->mouse_event.old_Y == -1)){
	// disactivate lpterm.
	if (view_p->widgets.status) {
	    g_object_set_data (G_OBJECT (view_p->widgets.status), "active", NULL);
	}
	if(view_p->mouse_event.old_X == -1 && view_p->mouse_event.old_Y == -1) {
	    view_p->mouse_event.old_X = view_p->mouse_event.boxX;
	    view_p->mouse_event.old_Y = view_p->mouse_event.boxY;
	}
	gchar *string = g_strdup_printf("%s: %s", _("Selection"), _("None"));
	rfm_status(&(view_p->widgets), "xffm/stock_dialog-info", string, NULL);
	g_free(string);
	return;
    } 

    // old rectangle
    gint lowX = (view_p->mouse_event.old_X > view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint lowY = (view_p->mouse_event.old_Y > view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;
    gint highX = (view_p->mouse_event.old_X < view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint highY = (view_p->mouse_event.old_Y < view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;

    /*if(view_p->mouse_event.rubberbanding == FALSE) {
	GdkRectangle rect;
	rect.x = lowX;
	rect.y = lowY;
	rect.width = highX - lowX + 1;
	rect.height = highY - lowY + 1;
	rfm_expose_rect (&rect);
	return;
    }*/

    // new rectangle
    view_p->mouse_event.old_X = x;
    view_p->mouse_event.old_Y = y;
    gint new_lowX = (view_p->mouse_event.old_X > view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint new_lowY = (view_p->mouse_event.old_Y > view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;
    gint new_highX = (view_p->mouse_event.old_X < view_p->mouse_event.boxX) ? view_p->mouse_event.boxX : view_p->mouse_event.old_X;
    gint new_highY = (view_p->mouse_event.old_Y < view_p->mouse_event.boxY) ? view_p->mouse_event.boxY : view_p->mouse_event.old_Y;

    
    /* clean old rectangle */
    GdkEventExpose old_event;
    old_event.area.x=lowX;
    old_event.area.y=lowY;
    old_event.area.width=highX - lowX;
    old_event.area.height=highY - lowY;


    GSList *tmp;
    // find all items in old rectangle (restricted to icons)
    GSList *old_list=rodent_find_icons_in_rectangle(view_p, &(old_event.area));
    //gint old_items=g_slist_length(old_list);
   
 
    GdkEventExpose new_event;
    new_event.area.x=new_lowX;
    new_event.area.y=new_lowY;
    new_event.area.width=new_highX - new_lowX;
    new_event.area.height=new_highY - new_lowY;

	
    // find all items in new rectangle (restricted to icons)
    GSList *new_list=rodent_find_icons_in_rectangle(view_p, &(new_event.area));
    gint new_items=g_slist_length(new_list);
    NOOP(stderr, "items=%d (%d, %d, %d, %d)\n", 
	    g_slist_length(new_list),
	    new_event.area.x, new_event.area.y, new_event.area.width, new_event.area.height);

    guint64 oldsum = 0;
    guint64 newsum = 0;
    // unselect all unselected items
    for (tmp=old_list; tmp && tmp->data; tmp=tmp->next){
	if (!g_slist_find(new_list, tmp->data)){
	    population_t *population_p = tmp->data;
	    rfm_unselect_pixbuf (view_p, population_p);
	}
	if(!draw) oldsum += GPOINTER_TO_INT(tmp->data);
    }


    // this will do any scrolling
    scroll_business(view_p, y, &(new_event.area));

    // select all selected items
    NOOP(stderr, "selection items=%d\n",g_slist_length(new_list));

    for (tmp=new_list; tmp && tmp->data; tmp=tmp->next){
	population_t *pop_p = tmp->data;
	if (pop_p->en && IS_UP_TYPE(pop_p->en->type)) continue;
	rfm_select_pixbuf (view_p, pop_p);
    }
    g_slist_free(new_list);
    new_list=NULL;
    new_list=rodent_find_icons_in_rectangle(view_p, &(new_event.area));

    if(!draw) for (tmp=new_list; tmp && tmp->data; tmp=tmp->next){
	newsum += GPOINTER_TO_INT(tmp->data);
    }

    // This if() is buggy...
    if (draw || newsum != oldsum)
    {
	GSList *tmp=old_list;
	for (;tmp && tmp->data; tmp=tmp->next){
	    if(!g_slist_find(new_list, tmp->data)){
		new_list =g_slist_prepend(new_list, tmp->data);
	    }
	}
	NOOP("rodent_mouse: expose necessary: items=%d (old=%d)\n",new_items,old_items);
	NOOP("rodent_mouse: expose lists items=%d\n", g_slist_length(new_list));

	tmp=new_list;
	    
	for (;tmp && tmp->data; tmp=tmp->next){
	    population_t *population_p=tmp->data;

	    // here we need a realtime expose, but will the grab
	    // affect? is a read lock set?
	    rodent_redraw_item(view_p, population_p);
	    //rfm_expose_item(view_p, population_p);
	}
     


	gchar *string=NULL;
	if (new_items) {
	    string = g_strdup_printf (ngettext ("%'d item selected", "%'d items selected", new_items), new_items);
	} else {
	    string = g_strdup_printf("%s: %s", _("Selection"), _("None"));
	}
	rfm_status(&(view_p->widgets), "xffm/stock_dialog-info", string, NULL);
	g_free(string);

	// get new reselect list.
	reset_reselect_list(view_p);
	// disactivate lpterm.
	if (view_p->widgets.status) {
	    g_object_set_data (G_OBJECT (view_p->widgets.status), "active", NULL);
	}
    }
    g_slist_free(old_list);
    g_slist_free(new_list);
  
}

/* button press */
static void
button_popup (GdkEventButton * event, view_t * view_p, const population_t * population_p) {
    if(population_p) {
        if(!(population_p->flags  & POPULATION_SELECTED)) {
            rodent_unselect_all_pixbuf (view_p);
        }
        rfm_select_pixbuf (view_p, population_p);
	rfm_expose_item (view_p, population_p);
    }
    rodent_pop_menu (MAIN_POPUP_MENU_ID, event);
}
/*
static void
remove_ghost_selections(view_t * view_p){
    GSList *ghosts=NULL;
    GList *tmp=view_p->selection_list;
    for (; tmp && tmp->data; tmp=tmp->next){
	record_entry_t *en=tmp->data;
	if (en->path && !rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
	    ghosts = g_slist_prepend(ghosts, tmp->data);
	}
    }
    GSList *stmp = ghosts;
    for (; stmp && stmp->data; stmp=stmp->next){
	record_entry_t *en=stmp->data;
	view_p->selection_list = g_list_remove(view_p->selection_list, stmp->data);
	rfm_destroy_entry(en);
    }
    g_slist_free(ghosts);
    if (g_list_length(view_p->selection_list)==0){
	g_list_free(view_p->selection_list);
	view_p->selection_list = NULL;
    }  
}
*/

