#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2014 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

// All this runs in non main threads.

#define READ_ERROR -1

/**
 * xfdir_set_entry_tag:
 * @src_en: a valid record_entry_t pointer
 * @tama: Size in bytes.
 * 
 * Sets the entry's tag (which will show up on the status line).
 *
 **/
static gboolean 
skip_item(view_t *view_p, const gchar *pathv, record_entry_t *en){
    if (!pathv) return TRUE;
    // Not showing hidden files:
    if(!(view_p->flags.preferences & __SHOW_HIDDEN)){
	if (en && IS_UP_TYPE(en->type)) return FALSE;
	    // Skip hidden records if so configured.
	if (pathv[0] == '.' && strcmp (pathv, "..") != 0) return TRUE;
    }
    // Not showing backup files:
    if(!(view_p->flags.preferences & __SHOWS_BACKUP)){
	if(rodent_is_backup_type(pathv)) return TRUE;
    }
    return FALSE;
}

static gint
count_xfdir_population(view_t *view_p, xfdir_t *xfdir_p){
    if (!xfdir_p || !xfdir_p->gl) return 0;
    gint count = 0;
    gint i;
    for(i = 0; i < xfdir_p->pathc; i++) {
	if (!xfdir_p->gl[i].pathv) continue;
	if (skip_item(view_p, xfdir_p->gl[i].pathv, xfdir_p->gl[i].en)) continue;
	count++;
    }
    return count;
}

static void
xfdir_set_entry_tag (widgets_t *widgets_p, record_entry_t * en, off_t tama) {
    int hcount = xfdir_count_hidden_files (en->path);
    int count = xfdir_count_files (en->path);
    gchar *basename = g_path_get_basename (en->path);
    gchar *utf_string = rfm_utf_string (basename);
    g_free (basename);
    gchar *s = NULL;
    g_free (en->tag);

    if (en->module) {
	s=rfm_rational(PLUGIN_DIR, en->module, widgets_p, en, "sizetag");
    } else {
	s = rfm_sizetag (tama, count);
    }
    

    if (hcount) {
        gchar *plural_string = g_strdup_printf(ngettext ("%'u item","%'u items",hcount), hcount);
        en->tag = g_strdup_printf ("%s (%s %s: %s)", utf_string, (s)?s:"", _("Hidden"), plural_string);
        g_free(plural_string);
    } else if (s) {
        en->tag = g_strdup_printf ("%s (%s)", utf_string, s);
    } else {
        en->tag = g_strdup_printf ("%s", utf_string);
    }
    
    g_free (utf_string);
    g_free (s);
}



// check if monitor should wait:
/**
 * xfdir_monitor_skip:
 * @xfdir_p: pointer to an xfdir_t structure
 * Returns: TRUE if monitor should skip tests on current loop
 * or FALSE if monitor should go ahead with the update tests.
 * 
 * Checks whether the xfdir monitor should skip monitor tests on the current
 * loop or go ahead with monitor tests. It is an exported symbol so that modules
 * may use the same monitor algorithm as the xfdir monitor thread.
 **/
static
  gboolean
xfdir_monitor_skip (xfdir_t * xfdir_p) {    
    gboolean skip = FALSE;
    if(xfdir_p == NULL) {
        g_error ("monitor_skip: xfdir_p ==NULL!");
    }
    view_t *view_p = xfdir_p->view_p;



    if(view_p->flags.refresh) {
        skip = TRUE;
        NOOP(stdout, "thread monitor is skipping: xfdir_p->refresh\n");
    }

	    
    if (view_p->flags.thumbnailer_active){
        DBG("monitor skip on thumbnailer active\n");
        return TRUE;
    }

    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if (status == STATUS_EXIT) return TRUE;
    
    // popup active? Any popup of the popups
    // XXX this is obsolete with recursive mutex
    /*
     *     rfm_global_t *rfm_global_p = rfm_global();
     * if (rfm_global_p->popup_active) {
        skip = TRUE;
        NOOP (stderr, "thread monitor: popup visible\n");
    }*/
    /* entry window is mapped? */
    if(view_p->widgets.rename) {
        skip = TRUE;
        NOOP (stderr, "thread monitor: entry window is mapped\n");
    }
    /* tip window is mapped? This could be disabled, but better be safe. */

    else if(rodent_tip_get_active()) {
        // But is it a valid active state?
	// State should be valid here since
	// condition is taken care of in rodent_mouse.c
	// with the on_leave signal
        NOOP (stderr, "thread monitor: widgets.tooltip_active\n");
        skip = TRUE;
    }
    /* button down? */
    else if(view_p->mouse_event.boxX != -1 && view_p->mouse_event.boxY != -1) {
        skip = TRUE;
        NOOP(stdout, "thread monitor: mouse_event.boxX != -1\n");
    }
    /* dragging? */
    else if(view_p->mouse_event.doing_drag_p){
        skip = TRUE;
    } 
#ifdef DEBUG_NOOP
    static gboolean said = FALSE;
    /* saving cache? */
    if(!skip && said) {
        said = FALSE;
        NOOP ("thread monitor: active\n");
    }
    if(!skip && said) {
        said = FALSE;
        NOOP ("thread monitor: active\n");
    }
    if(skip) {
        if(!said) {
            said = TRUE;
            NOOP ("thread monitor: sleeping\n");
        }
    }
#endif
    return skip;
}

static
  gboolean
xfdir_monitor_disabled (xfdir_t * xfdir_p) {
    if(xfdir_p == NULL) {
        g_error ("monitor_skip: xfdir_p ==NULL!");
    }
    //view_t *view_p = xfdir_p->view_p;

#if 0
    // notebook page not visible?
    if (view_p->flags.type != DESKVIEW_TYPE){
        rfm_global_t *rfm_global_p = rfm_global();
	GtkNotebook *notebook = g_object_get_data(G_OBJECT(rfm_global_p->window), "notebook");
	gint current_page = gtk_notebook_get_current_page (notebook);
	GtkWidget *child =gtk_notebook_get_nth_page(notebook, current_page);
	view_t *current_view_p = g_object_get_data (G_OBJECT (child), "view_p");
	if (view_p != current_view_p) {
	    NOOP("xfdir_monitor_disabled: disabled on non visible page\n");
	    return TRUE;
	}
    }
#endif

#if 0
    // user is fooling around with the scrollbar
    if (view_p->widgets.scrolled_window) {
	GtkAdjustment *adjustment = 
	    gtk_scrolled_window_get_vadjustment(view_p->widgets.scrolled_window);
	gdouble v = floor(gtk_adjustment_get_value(adjustment));
	gint iv = v;
	gint sv = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(view_p->widgets.scrolled_window), 
		"adjustment_value"));
	if (sv != iv) {
	    g_object_set_data(G_OBJECT(view_p->widgets.scrolled_window), 
		"adjustment_value", GINT_TO_POINTER(sv));
	    return FALSE;
	}
    }
#endif
   return FALSE;
}


/**
 * xfdir_free_data:
 * @xfdir_p:  pointer to an xfdir_t structure
 *
 *  Frees allocated resources of a loaded xfdir structure
 *
 **/
static gint
xfdir_free_data (xfdir_t * xfdir_p) {

    if(!xfdir_p) {
        DBG ("xfdir_p==NULL in xfdir_free_data\n");
        return -1;
    }
    if(xfdir_p->gl != NULL && xfdir_p->pathc) {
        int i;
        for(i = 0; i < xfdir_p->pathc; i++) {
            NOOP ("thread monitor: will free xfdir_p->gl[i].pathv=0x%lx\n", (long unsigned)xfdir_p->gl[i].pathv);
            g_free (xfdir_p->gl[i].pathv);
            NOOP ("thread monitor: will destroy xfdir_p->gl[i].en=0x%lx\n", (long unsigned)xfdir_p->gl[i].en);
            rfm_destroy_entry (xfdir_p->gl[i].en);
        }
        NOOP ("thread monitor: will free xfdir_p->gl=0x%lx\n", (long unsigned)xfdir_p->gl);
        g_free (xfdir_p->gl);
        xfdir_p->gl = NULL;
        xfdir_p->pathc = 0;
    }
    NOOP ("thread monitor: will free xfdir_p->data=0x%lx\n", (long unsigned)xfdir_p->data);
    rfm_destroy_entry (xfdir_p->en);
    g_free (xfdir_p->data);
    return 0;
}

static
  gboolean
exit_monitor (xfdir_t * xfdir_p) {
    gboolean result = FALSE;
    view_t *view_p = xfdir_p->view_p;
    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    gint monitor_id = view_p->flags.monitor_id;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if(status == STATUS_EXIT) return TRUE;
    if(monitor_id != xfdir_p->monitor_id)
        result = TRUE;
    return result;
}

/**
 * xfdir_monitor_continue:
 * @xfdir_p:  pointer to an xfdir_t structure
 * Returns: TRUE if monitor should continue alive or FALSE if monitor should 
 * finish.
 *
 * This function is a generic check whether the local file monitor thread
 * should continue alive or exit. 
 **/
static
  gboolean
xfdir_monitor_continue (xfdir_t * xfdir_p) {
    if(!xfdir_p) {
        NOOP ("thread monitor: monitor_continue !xfdir_p\n");
        return FALSE;
    }
    /* signaled exit: */
    if(exit_monitor (xfdir_p)) {
        DBG ("thread monitor: exit_monitor(xfdir_p) = 1\n");
        return FALSE;
    }
    
    view_t *view_p = xfdir_p->view_p;
    if(xfdir_p->en->module) {
        if (!view_p->en || !view_p->en->module || strcmp(view_p->en->module, xfdir_p->en->module)) {
            DBG("monitor for module %s is ending.\n", xfdir_p->en->module);
            return FALSE;
        }
        return TRUE;
    }

    /* invalid view: */
    if(!view_p)
        return FALSE;
    /* root window */
    if(!view_p->en) {
        NOOP ("thread monitor: exit on !view_p->en\n");
        return FALSE;
    }
    g_mutex_lock(view_p->mutexes.status_mutex);
    gboolean status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if(status == STATUS_EXIT) return FALSE;
    
    /* directory has vanished */
    if(xfdir_p->en->path == NULL || !rfm_g_file_test (xfdir_p->en->path, G_FILE_TEST_EXISTS)) {
	// this is the test for local files.
	NOOP ("monitor: !rfm_g_file_test(xfdir_p->en->path,G_FILE_TEST_EXISTS)\n");
	return FALSE;
    }
    return TRUE;
}

static off_t
st_sum (struct stat *st) {
    off_t sum;
    if(!st)
        sum = 0;
    else {
        sum = st->st_ino + st->st_mtime + st->st_size + st->st_mode + st->st_nlink + st->st_uid + st->st_gid;
    }
    return sum;
}

static
int update_xfdir (xfdir_t * xfdir_p, gboolean fullstat, gint *heartbeat);


typedef struct xd_t{
    gchar *d_name;
#ifdef HAVE_STRUCT_DIRENT_D_TYPE
    unsigned char d_type;
#endif
}xd_t;

static GSList *
read_files_local (xfdir_t * xfdir_p, gint *heartbeat) {
    if(xfdir_p == NULL) return  NULL;
    GSList *directory_list = NULL;

    DIR *directory = opendir(xfdir_p->en->path);
    if (!directory) {
	NOOP("read_files_local(): Cannot open %s\n", xfdir_p->en->path);
	return NULL;
    }

// http://womble.decadent.org.uk/readdir_r-advisory.html

#if 0
// this crashes on gvfs mount points....
//        bug repoerted by Liviu
#if defined(HAVE_FPATHCONF) && defined(HAVE_DIRFD)
    size_t size = offsetof(struct dirent, d_name) + 
	fpathconf(dirfd(directory), _PC_NAME_MAX) + 1;
#else
    size_t size = offsetof(struct dirent, d_name) +
	pathconf(xfdir_p->en->path, _PC_NAME_MAX) + 1;
#endif
#else
    // this should be more than enough
    size_t size = 256*256;
#endif

    struct dirent *buffer = (struct dirent *)malloc(size);
    if (!buffer) g_error("malloc: %s\n", strerror(errno));

    gint error;
    struct dirent *d;
    while ((error = readdir_r(directory, buffer, &d)) == 0 && d != NULL){
        if(strcmp (d->d_name, ".") == 0) continue;
	xd_t *xd_p = (xd_t *)malloc(sizeof(xd_t));
	xd_p->d_name = g_strdup(d->d_name);
#ifdef HAVE_STRUCT_DIRENT_D_TYPE
	xd_p->d_type = d->d_type;
#endif
	directory_list = g_slist_prepend(directory_list, xd_p);
	if (heartbeat) {
	    (*heartbeat)++;
	    NOOP("incrementing heartbeat records to %d\n", *heartbeat);
	}
    }
    if (error) {
        NOOP("readdir_r: %s\n", strerror(errno));
    }

    closedir (directory);

    g_free(buffer);

    // At least the ../ record should have been read. If this
    // is not so, then a read error occurred.
    // (not uncommon in bluetoothed obexfs)
    if (!directory_list) {
	NOOP("read_files_local(): Count failed! Directory not read!\n");
    }
    return (directory_list);
}

#if 0
static off_t
st_sum_dir (struct stat *st) {
    off_t sum;
    if(!st) return 0;

    sum = st->st_ino + st->st_mtime + st->st_ctime + st->st_size + st->st_mode + st->st_nlink + st->st_uid + st->st_gid;
    
    return sum;
}
#endif

// has to indicate if a path is in a xfdir structure.
static GHashTable *
create_population_hash(xfdir_t *xfdir_p){
    gint i;
    GHashTable *hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
    for(i = 0; i < xfdir_p->pathc; i++) {
        const gchar *path;
        if (xfdir_p->gl[i].en == NULL){
	    TRACE("item %d has null entry %s.\n", i, xfdir_p->gl[i].pathv);
	    continue;

	} else path = xfdir_p->gl[i].en->path;
	g_hash_table_replace(hash, g_strdup(path), GINT_TO_POINTER(1));
    }
    TRACE("create_population_hash() OK\n");
    return hash;    
}


// this function checks if a directory (or module) currently in 
// xfdir structure needs to be updated in the view (in other words,
// is a new xfdir structure necessary?)
static
  gboolean
reload_condition (xfdir_t * xfdir_p) {
    view_t *view_p = xfdir_p->view_p;
    record_entry_t *en = xfdir_p->en;
    if(en->module) {
	// We release mutex here to allow module to sleep (hack)
	rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));	
 	gboolean reload = GPOINTER_TO_INT(rfm_natural(PLUGIN_DIR, en->module,
		    &(view_p->widgets), "reload") > 0);
	rfm_rw_lock_writer_lock (&(xfdir_p->view_p->mutexes.monitor_lock));	
	return reload;
    }
    // Test 1. Short circuit on instruction from another thread.
    //
    // put a lock on xfdir structure and use the reload variable
    // to shortcircuit test on condition set by other threads
    gboolean instant_return = xfdir_p->reload;
    xfdir_p->reload = FALSE;
    if(instant_return){
	DBG("monitor: reload_condition requests instant refresh...\n");
        return TRUE;
    }
    return FALSE;
}


static gchar *
folder_icon_id(view_t *view_p, record_entry_t *en){
    gchar *icon_id = NULL;
    // Directories may or may not be using content emblems.
    // If mounted, we do not want content emblems, ever.
    // If remote type, we do not want content emblems, ever.
    // If up type, we do not want content emblems, ever.
    // If user disabled feature, we do not want content emblems.
    void *mounted = 
	rfm_natural(PLUGIN_DIR, "fstab", en, "entry_is_mounted");
    gboolean remote = !IS_LOCAL_TYPE(en->type);
    gboolean up_type = IS_UP_TYPE(en->type);
    gboolean content_icons_disabled =
	(!getenv("RFM_CONTENT_FOLDER_ICONS") 
	 || !strlen(getenv("RFM_CONTENT_FOLDER_ICONS")));
    if (mounted || remote || up_type || content_icons_disabled){
	icon_id = rfm_get_entry_icon_id(view_p, en, TRUE);
    } else {
	// here we go get a new content icon (taxing...)
	icon_id = rfm_content_icon_id_f (view_p, en->path);
    }
    return (icon_id)?icon_id:g_strdup("xffm/stock_directory");
}


static void 
update_icon_cache(gpointer call_data, gpointer pool_data){
    xfdir_t *xfdir_p = pool_data;
    if(exit_monitor (xfdir_p)) return;
    gint i;
    view_t *view_p = xfdir_p->view_p;

    for(i = 0; i < xfdir_p->pathc; i++) {
        if(exit_monitor (xfdir_p)) {
            return;
        }
        if (!rfm_population_try_read_lock (view_p, "update_icon_cache")) {
            return; 
        }
        record_entry_t *gl_en = xfdir_p->gl[i].en;

	// Find population item.
	population_t *population_p = g_hash_table_lookup(view_p->population_hash, 
                (gl_en)?gl_en->path:"RODENT-ROOT");
        // POPULATION_ICON_CACHE can only happen on a second pass
        // by then, item has been stat'd and icon updated.
        if (population_p && (population_p->flags & POPULATION_ICON_CACHE) ){
            rfm_save_icon_id_to_cache(gl_en, population_p->icon_id);
            population_p->flags &= (POPULATION_ICON_CACHE ^ 0xffffffff);
        }
        rfm_population_read_unlock (view_p, "update_icon_cache");
	g_thread_yield();
    }
    return;
}

static void
do_the_thumbnail(population_t *population_p){
    GdkPixbuf *pixbuf=NULL;
    if (population_p->thumbnail && G_IS_OBJECT(population_p->thumbnail)) {
	TRACE("do_the_thumbnail g_object_unref %s\n",population_p->en->path);
	g_object_unref(G_OBJECT(population_p->thumbnail));
    }
    if (population_p->flags & POPULATION_IS_IMAGE){
        pixbuf = rfm_create_preview (population_p->en->path, population_p->icon_size);//refs
        if (pixbuf) population_p->flags |= POPULATION_PIXBUF_CLEAN;
    }
    population_p->thumbnail = pixbuf;
}

static void 
stat_pool_f(gpointer call_data, gpointer pool_data){
    void **arg = (void **)call_data;
    xfdir_t *xfdir_p = pool_data;
    if(exit_monitor (xfdir_p)) return;
    
    record_entry_t *gl_en = arg[0];
    gint *something_changed = arg[1];
    g_free(arg);
    view_t *view_p = xfdir_p->view_p;


    gboolean icon_expose_condition = FALSE;
    // This has to be done on local and remote items to see if icon
    // need to be updated. Remote items my be present if we
     // are monitoring a local directory with a remote directory
    // mount point. This needs to be monitored to see if the
    // mount emblem need to be set or unset.
    // type might change, say a directory becomes a file, or viceversa...
    record_entry_t *en = 
        rfm_stat_entry (gl_en->path, gl_en->type);

    // Has item changed from local to remote?
    // This will happen for mount points of remote locations.
    // Also, monitor any other changes on filetype...
    // such as the original a symlink and the new one no...
    // such as mount status changes...
    //
    // * Mount status change will require an expose for the item.
    //
    gint old_type = gl_en->type;
    gint new_type = en->type;
    if (old_type != new_type){
        NOOP(stderr, "[%s] change of type for %s 0x%x->0x%x\n",
            xfdir_p->en->path, gl_en->path,
            old_type, new_type);
        memset(gl_en->st, 0, sizeof(struct stat));
        icon_expose_condition = TRUE;
    }   
    
    // This is the stat comparison test for each item
    // in the xfdir structure.
    if(gl_en->st->st_mtime != 0 
            && st_sum (gl_en->st) == st_sum (en->st)) {
        // Nothing to do here...
        rfm_destroy_entry(en); en = NULL;
        return;
    }


    // Stat information update is necessary.
    NOOP (stderr, "monitor [%s]:  stat differs for %s \n",
        xfdir_p->en->path, gl_en->path);

    // Step 1. Update stat information in xfdir structure.
    memcpy (gl_en->st, en->st, sizeof (struct stat));
    
    // Step 2. Update mimetype info
    //         On first pass, do a plain mimetype...
    gl_en->mimetype = MIME_type(gl_en->path, gl_en->st);
    if (!gl_en->mimetype) gl_en->mimetype = g_strdup(_("unknown"));
    gl_en->mimemagic = MIME_magic(gl_en->path); //+5 seconds hot
    if (!gl_en->mimemagic) gl_en->mimemagic = g_strdup(_("unknown"));

    // Step 3. Update entry type.
    gint keep_type = (gl_en->type) & KEEP_TYPE;
    gl_en->type = (en->type|keep_type);
    // No more use for comparison entry.
    rfm_destroy_entry(en); en = NULL;

    // Step 4. Get new icon id, if applicable
    gchar *new_icon_id = NULL;

    // We go for mimetype/mimemagic
    if(IS_BROKEN_LNK (gl_en->type)) {
        new_icon_id = g_strdup("xffm/stock_missing-image");
    } else if (!IS_SDIR(gl_en->type)){ 
        // (except for directories)
        new_icon_id = g_strdup((strcmp(gl_en->mimetype, _("unknown")))?
                gl_en->mimetype:
                gl_en->mimemagic);
    } else if (IS_NOACCESS_TYPE (gl_en->type)){
            new_icon_id = g_strdup("xffm/stock_directory/compositeC/emblem_unreadable");
    } else {
        new_icon_id = folder_icon_id (view_p, gl_en);
    }
    if (new_icon_id) {
        // change inode mimetypes to icon identifiers:
        if (strcmp(new_icon_id, "inode/directory")==0){
            g_free(new_icon_id);
            new_icon_id = g_strdup("xffm/stock_directory");
        } else if (strcmp(new_icon_id, "inode/chardevice")==0){
            g_free(new_icon_id);
            new_icon_id = g_strdup("xffm/emblem_chardevice");
        } else if (strcmp(new_icon_id, "inode/blockdevice")==0){
            g_free(new_icon_id);
            new_icon_id = g_strdup("xffm/emblem_blockdevice");
        } else if (strcmp(new_icon_id, "inode/fifo")==0){
            g_free(new_icon_id);
            new_icon_id = g_strdup("xffm/emblem_fifo");
        } else if (strcmp(new_icon_id, "inode/socket")==0){
            g_free(new_icon_id);
            new_icon_id = g_strdup("xffm/emblem_network/compositeSE/emblem_fifo");
        } 
    }

    // Step 5. Update stat information in corresponding view population item.
    //         This requires a read lock.
    if (!rfm_population_read_lock (view_p, "update_stat_info")) {
        //XXX here we should break out of the loop...
        NOOP (stderr, "!rfm_population_read_lock\n");
        return; // GINT_TO_POINTER(FALSE);
    }
    if(exit_monitor (xfdir_p)) {
        // Now, once we got the read lock, is the thread doing required
        // work or not? Conditions may have changed while obtaining
        // the read lock.
        NOOP("update_stat(): exit_monitor condition\n");
        rfm_population_read_unlock (view_p, "update_stat_info");
        return; // GINT_TO_POINTER(FALSE);
    }

    // Do the update on current view population.
    
    // Find population item.
    population_t *population_p = g_hash_table_lookup(view_p->population_hash, 
            (gl_en)?gl_en->path:"RODENT-ROOT");

    // Item not found means that population_pp belong to a different
    // monitor thread...
    if (!population_p) {
        rfm_population_read_unlock (view_p, "update_stat_info");
        return; // GINT_TO_POINTER(FALSE);
    }
    if (new_icon_id) {
        // This we save for later. Parallel write access on dbh file
        // will slow things down.
        // rfm_save_icon_id_to_cache(gl_en, new_icon_id);
        // Mark item to update cache:
        if (!population_p->icon_id || strcmp(new_icon_id, population_p->icon_id)){
            population_p->flags |= POPULATION_ICON_CACHE;
        }
    }

    *something_changed=TRUE; 
    NOOP (stderr, "updating stat record for %s\n", gl_en->path);
    // Update current population item stat information.
    memcpy (population_p->en->st, gl_en->st, sizeof (struct stat));

    // Update mimetype information
    population_p->en->type = gl_en->type;
    g_free(population_p->en->mimetype);
    population_p->en->mimetype = g_strdup((gl_en->mimetype)?
        gl_en->mimetype:_("unknown"));
    g_free(population_p->en->mimemagic);
    population_p->en->mimemagic = g_strdup((gl_en->mimemagic)?
        gl_en->mimemagic:_("unknown"));

    gchar *old_icon_id = NULL;
    if (population_p->icon_id) old_icon_id = g_strdup(population_p->icon_id);


    // Update content icon id, if applicable.
    if (new_icon_id) {
        if (!population_p->icon_id || strcmp(new_icon_id, population_p->icon_id)){
            g_free(population_p->icon_id);
            population_p->icon_id = new_icon_id;
            // Mark item to update cache:
            population_p->flags |= POPULATION_ICON_CACHE;
        }
    } else {
        if (strcmp(population_p->en->mimetype, _("unknown"))){
            g_free(population_p->icon_id);
            population_p->icon_id = g_strdup(population_p->en->mimetype);
        } else if (strcmp(population_p->en->mimemagic, _("unknown"))){
            g_free(population_p->icon_id);
            population_p->icon_id = g_strdup(population_p->en->mimetype);
        }
    }

    // Set icon status to dirty for regeneration on expose
    population_p->flags &= (POPULATION_PIXBUF_CLEAN ^ 0xffffffff);


    // update cut/copied icons //
    //1. Eliminate obsolete (perchance) cut/copy flag
    if (population_p->flags & (POPULATION_COPIED|POPULATION_CUT)) {
        // Item is marked.
        if (!rfm_in_pasteboard(view_p, population_p->en)) {
            // Item should not be marked.
            population_p->flags &= (POPULATION_COPIED ^ 0xffffffff);
            population_p->flags &= (POPULATION_CUT ^ 0xffffffff);
            icon_expose_condition = TRUE;
        }
    }
    //2. Mark and expose items //+3
    rodent_update_cut_icons(view_p);        
    
    // remove any previous popup preview image, including the thumbnail file.
    // (see rodent_population.i:265)
    if (population_p->preview_pixbuf) {
        icon_expose_condition = TRUE;
        TRACE ("removing preview pixbuf at update_stat: g_object_unref %s\n",population_p->en->path); 
        if (G_IS_OBJECT(population_p->preview_pixbuf)) 
            g_object_unref(G_OBJECT(population_p->preview_pixbuf));
        population_p->preview_pixbuf = NULL;
        gchar *thumbnail = 
            rfm_get_thumbnail_path (population_p->en->path, PREVIEW_IMAGE_SIZE);
        if (g_file_test(thumbnail, G_FILE_TEST_EXISTS) && unlink(thumbnail)<0)
            DBG("unlink(%s): %s\n", thumbnail, strerror(errno));
            
        g_free(thumbnail);
    }
    // Are we in the list view?
    // If so we will need to update the displayed stat information.
    if (ICON_SIZE_ID(view_p)==0) {
        // get new layout.
        if (population_p->layout) {
            if (G_IS_OBJECT(population_p->layout))
                g_object_unref(population_p->layout);
            population_p->layout=NULL;
        }
        rfm_layout_pango_layout_setup(view_p);
        icon_expose_condition = TRUE;
        NOOP(stderr, "layout updated for %s\n", 
                population_p->en?population_p->en->path:"null entry");
    }
    // Update the icon
    // If stat information was updated, update icon as well.
    // This may imply doing magic on the file...
    // This does the actual greenball update.
    NOOP(stderr, "xfdir monitor: updating icon now: %d\n",
            (population_p->en->st)?population_p->en->st->st_uid:0);


    if (!old_icon_id ||strcmp(old_icon_id, population_p->icon_id)){
        NOOP(stderr, "icon updated for %s: %s -> %s\n", 
                population_p->en->path, old_icon_id, population_p->icon_id);
        icon_expose_condition = TRUE;
    } else {
        NOOP(stderr, "skipping icon expose for %s (not changed)\n",
                population_p->en?population_p->en->path:"null entry");
    }
    g_free(old_icon_id);
    // Should the referred item have a thumbnail displayed?
    // If item is thumbnailed, we must get a new thumbnail and expose.
    //
    if (IS_LOCAL_TYPE(population_p->en->type)) {
        if (rfm_entry_is_image(population_p->en)){
            icon_expose_condition = TRUE;
            NOOP(stderr, "setting up %s for thumbnail\n",population_p->en->path);
            population_p->flags &= (POPULATION_PIXBUF_CLEAN ^ 0xffffffff);
            population_p->flags |= POPULATION_IS_IMAGE;
            do_the_thumbnail(population_p);
        }
        else NOOP(stderr, "%s no thumbnail\n",population_p->en->path);
    }

    if (icon_expose_condition)
    {
        NOOP(stderr, "icon_expose_condition for %s\n", 
                population_p->en?population_p->en->path:"null entry");
        rfm_expose_item(view_p,population_p);
        /*rodent_redraw_item(view_p,population_p);*/
    }
//escape:
   // Release the population read lock.
   rfm_population_read_unlock (view_p, "update_stat_info");
   return;
}

#if DEBUG_TRACE
static gboolean
find_in_xfdir_p(view_t *view_p, population_t *p, xfdir_t *xfdir_p){
    gint i;
    // This is an extra read lock.
    if (!rfm_population_read_lock (view_p, "find_in_xfdir_p")) {
	TRACE("find_in_xfdir_p(): !rfm_population_read_lock\n");
	return FALSE;
    }
    gboolean retval = FALSE;
    for(i = 0; i < xfdir_p->pathc; i++) {
	if (!p->en && !xfdir_p->gl[i].en) {retval = TRUE; break;}
	if (!p->en || !xfdir_p->gl[i].en) continue;
	if (!p->en->path  || !xfdir_p->gl[i].en->path) continue;
	if (strcmp (p->en->path, xfdir_p->gl[i].en->path) == 0) {retval = TRUE; break;}
    }
    rfm_population_read_unlock (view_p, "find_in_xfdir_p");
    return retval;
}
#endif

// If a full reload is not necessary, this function just updates
// stat information, mimetype and thumbnails.
//
// This should be thread pooled....
static size_t item_count(xfdir_t *xfdir_p, GHashTable *pop_hash);
static gboolean
update_stat_info (xfdir_t * xfdir_p, gboolean initial) {

    if (xfdir_p->en->module) return FALSE;
    
    gint i;
    view_t *view_p = xfdir_p->view_p;

    NOOP("* update_stat_info[%s]\n", xfdir_p->en->path );
   

    TRACE("* update_stat_info[%s]\n", xfdir_p->en->path );
    // By now, we have entered with a write lock in place, all items in xfdir are in population_pp
    // and viceversa. We are ready to threadpool

    GError *error=NULL;
    rfm_global_t *rfm_global_p = rfm_global();

    // 2*cores to takes advantage of pipeline architecture.
    GThreadPool *stat_pool = g_thread_pool_new (stat_pool_f, xfdir_p, 2*rfm_global_p->cores, TRUE, &error);
    //gint cores = (rfm_global_p->cores <= 1)?1:rfm_global_p->cores-1;
    //GThreadPool *stat_pool = g_thread_pool_new (stat_pool_f, xfdir_p, cores, TRUE, &error);
    if (error){
	g_warning("update_stat_info(): %s\n", error->message);
	g_error_free(error);
	return FALSE;
    }
    
    TRACE("starting population sweep\n"); //time_t start=time(NULL);
    gint something_changed = FALSE;
    for(i = 0; i < xfdir_p->pathc; i++) {
        if(xfdir_p->gl[i].en == NULL || !xfdir_p->gl[i].en->path){
            continue;
	}
	if (xfdir_p->gl[i].en->st == NULL) continue;
	// Do directories on second pass (since they are more time consuming)
	if (initial && IS_SDIR(xfdir_p->gl[i].en->type)) {
	    NOOP(stderr, "initial skip for %s\n", xfdir_p->gl[i].pathv);
	    continue;
	}
	if (skip_item(view_p, xfdir_p->gl[i].pathv, xfdir_p->gl[i].en)) {
	    NOOP(stderr, "skip_item %s\n", xfdir_p->gl[i].pathv);
	    continue;
	}
	//
	// Is the work still necessary or has the thread met an exit condition?
        if(exit_monitor (xfdir_p)){
	    //rfm_population_read_unlock (view_p);
	    g_thread_pool_free (stat_pool, TRUE, TRUE);
            return FALSE;
	}
	NOOP("sending %d %p to threadpool\n", i, xfdir_p);
	void **arg=malloc(2*sizeof(void *));
        if (!arg) g_error("malloc: %s\n", strerror(errno));
	arg[0]=xfdir_p->gl[i].en;
        arg[1]=&something_changed;
	//void *arg[]={xfdir_p->gl[i].en, &something_changed};
        g_thread_pool_push(stat_pool, arg, NULL);


	//stat_pool_f(xfdir_p->gl[i].en, xfdir_p); +10 secs on /usr/bin
      
      // Directory is updated
      // This is the end of the stat comparison test.
    } // This is the end of the stat comparison loop on all items of xfdir structure

	
    // We are done with the thread pool now.
    g_thread_yield();
    g_thread_pool_free (stat_pool, FALSE, TRUE);
  
#if 0
    // FIXME: this causes constant reloading when sort set to date
    if (something_changed && 
	    !(xfdir_p->sort_column == NAME_SORT || xfdir_p->sort_column == TYPE_SORT)) 
    {
	// We need to resort the icons, 
	// so we indicate that a transfer will be necessary.
	    TRACE("update_stat_info(): something_changed (%p) && sort_column(%d)\n",
                    something_changed,
                    !(xfdir_p->sort_column == NAME_SORT ||xfdir_p->sort_column == TYPE_SORT) );
	return TRUE;
    }
#endif

    
    TRACE("population sweep DONE: %ld secs.\n", (long)(time(NULL)-start));
    // update the icon cache now. (after directory contents...)
    if (!initial) update_icon_cache(xfdir_p, xfdir_p);
    
    return FALSE;
}

static gint
pathv_compare_up (const gchar * a, const gchar * b) {
    if(!a && !b) return 0;
    if(!a) return -1;
    if(!b) return 1;
    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    return strcmp (a, b);
}

static gint
pathv_compare_down (const gchar * a, const gchar * b) {
    if(!a && !b) return 0;
    if(!a) return 1;
    if(!b) return -1;
    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    return strcmp (b, a);
}

static gint
pathv_compare_case_up (const gchar * a, const gchar * b) {
    if(!a && !b) return 0;
    if(!a) return -1;
    if(!b) return 1;
    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    return strcasecmp (a, b);
}

static gint
pathv_compare_case_down (const gchar * a, const gchar * b) {
    if(!a && !b) return 0;
    if(!a) return 1;
    if(!b) return -1;
    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    return strcasecmp (b, a);
}

static
  gint
entry_compare_up (int caso, record_entry_t * en_a, record_entry_t * en_b, gboolean case_insensitive) {
    if(!en_a && !en_b) return 0;
    if(!en_a) return -1;
    if(!en_b) return 1;

    gchar *a = en_a->path;
    gchar *b = en_b->path;
    int (*cmp)(const char *, const char *);
    if (case_insensitive) cmp = strcasecmp; else cmp = strcmp;
    if(caso == NAME_SORT || caso == TYPE_SORT) {
	if(!a && !b) return 0;
	if(!a) return -1;
	if(!b) return 1;
	if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
	if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    }
    gint result=0;
    switch (caso) {
	case TYPE_SORT:{
	    gboolean a_is_bak = (a[strlen(a)-1] == '~');
	    gboolean b_is_bak = (b[strlen(b)-1] == '~');
	    gboolean a_is_dot = FALSE;
	    gboolean b_is_dot = FALSE;
	    if (strchr(a, '.')){
		if (strcmp(strrchr(a, '.')+1,"bak")==0) a_is_bak = TRUE;
		if (strcmp(strrchr(a, '.')+1,"old")==0) a_is_bak = TRUE;
		if (strcmp(strrchr(a, '.')+1,"sik")==0) a_is_bak = TRUE;
		a = strrchr(a, '.');
		a_is_dot = TRUE;
	    }
	    if (strchr(b, '.')){
		if (strcmp(strrchr(b, '.')+1,"bak")==0) b_is_bak = TRUE;
		if (strcmp(strrchr(b, '.')+1,"old")==0) b_is_bak = TRUE;
		if (strcmp(strrchr(b, '.')+1,"sik")==0) b_is_bak = TRUE;
		b = strrchr(b, '.');
		b_is_dot = TRUE;
	    }
	    if (a_is_bak && b_is_bak) return 0;
	    if (!a_is_bak && b_is_bak) return -1;
	    if (a_is_bak && !b_is_bak) return 1;
	    if (a_is_dot && !b_is_dot) return -1;
	    if (!a_is_dot && b_is_dot) return 1;
	}
	case NAME_SORT: {
	    return cmp (a, b);
	}
	default:
	    if(!en_a->st && !en_b->st) return 0;
	    if(!en_a->st) return -1;
	    if(!en_b->st) return 1;
	    switch (caso) {
		case SIZE_SORT:
		    result = en_a->st->st_size - en_b->st->st_size; break;
		case DATE_SORT:
		    result =  en_a->st->st_mtime - en_b->st->st_mtime; break;
		case OWNER_SORT:
		    result =  en_a->st->st_uid - en_b->st->st_uid; break;
		case GROUP_SORT:
		    result =  en_a->st->st_gid - en_b->st->st_gid; break;
		case MODE_SORT:
		    result =  en_a->st->st_mode - en_b->st->st_mode; break;
	    }
    }
#if 0
    if (result == 0){
	    gchar *a = en_a->path;
	    gchar *b = en_b->path;
	    if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
	    if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
	    return cmp (a, b);
    }
#endif
    return result;
}

static gint
entry_compare_down (gint caso, record_entry_t * en_a, record_entry_t * en_b, gboolean case_insensitive) {
    if(!en_a && !en_b) return 0;
    if(!en_a) return 1;
    if(!en_b) return -1;

    gchar *a = en_a->path;
    gchar *b = en_b->path;
    if(caso == NAME_SORT || caso == TYPE_SORT) {
	if(!a && !a) return 0;
	if(!a) return 1;
	if(!b) return -1;
	if(strchr (a, G_DIR_SEPARATOR)) a = strrchr (a, G_DIR_SEPARATOR) + 1;
	if(strchr (b, G_DIR_SEPARATOR)) b = strrchr (b, G_DIR_SEPARATOR) + 1;
    }
 
    int (*cmp)(const char *, const char *);
    if (case_insensitive) cmp = strcasecmp; else cmp = strcmp;
    switch (caso) {
	case TYPE_SORT: {
	    gboolean a_is_bak = (a[strlen(a)-1] == '~');
	    gboolean b_is_bak = (b[strlen(b)-1] == '~');
	    gboolean a_is_dot = FALSE;
	    gboolean b_is_dot = FALSE;
	    if (strchr(a, '.')){
		if (strcmp(strrchr(a, '.')+1,"bak")==0) a_is_bak = TRUE;
		if (strcmp(strrchr(a, '.')+1,"old")==0) a_is_bak = TRUE;
		if (strcmp(strrchr(a, '.')+1,"sik")==0) a_is_bak = TRUE;
		a = strrchr(a, '.');
		a_is_dot = TRUE;
	    }
	    if (strchr(b, '.')){
		if (strcmp(strrchr(b, '.')+1,"bak")==0) b_is_bak = TRUE;
		if (strcmp(strrchr(b, '.')+1,"old")==0) b_is_bak = TRUE;
		if (strcmp(strrchr(b, '.')+1,"sik")==0) b_is_bak = TRUE;
		b = strrchr(b, '.');
		b_is_dot = TRUE;
	    }
	    if (a_is_bak && b_is_bak) return 0;
	    if (!a_is_bak && b_is_bak) return 1;
	    if (a_is_bak && !b_is_bak) return -1;
	    if (a_is_dot && !b_is_dot) return 1;
	    if (!a_is_dot && b_is_dot) return -1;
	}
	case NAME_SORT:{
	    return cmp (b, a);
	}
	default:
	    if(!en_a->st && !en_b->st) return 0;
	    if(!en_a->st) return 1;
	    if(!en_b->st) return -1;
	    switch (caso) {
		case SIZE_SORT:
		    return en_b->st->st_size - en_a->st->st_size;
		case DATE_SORT:
		    return en_b->st->st_mtime - en_a->st->st_mtime;
		case OWNER_SORT:
		    return en_b->st->st_uid - en_a->st->st_uid;
		case GROUP_SORT:
		    return en_b->st->st_gid - en_a->st->st_gid;
		case MODE_SORT:
		    return en_b->st->st_mode - en_a->st->st_mode;
	    }
    }
    return 0;
}

static gint compare_name_up (const void *a, const void *b);
static gint compare_name_down (const void *a, const void *b);

static gint compare_name_case_up (const void *a, const void *b);
static gint compare_name_case_down (const void *a, const void *b);

static gint
null_test (const dir_t *a, const dir_t *b) {
    // Null entry is always on top
    if(!a && !b) return 0;
    if(!a) return -1;
    if(!b) return 1;
    return 2;
}

static gint
dummy_test (const dir_t *d1, const dir_t *d2) {
    if(!d1->en || IS_DUMMY_TYPE (d1->en->type))	return -1;
    if(!d2->en || IS_DUMMY_TYPE (d2->en->type))	return 1;
    return 2;
}
static gint
directory_test (const dir_t *d1, const dir_t *d2) {
    gboolean d1_is_dir = IS_SDIR (d1->en->type);
    gboolean d2_is_dir = IS_SDIR (d2->en->type);
    if(d1_is_dir && !d2_is_dir)  return -1;
    if(!d1_is_dir && d2_is_dir)  return 1;
    return 2;
}

static gint
standard_test (const dir_t *d1, const dir_t *d2){
    // Null entries on top
    gint result = null_test(d1, d2);
    if (result < 2) return result;
    // Dummies go next
    result = dummy_test(d1, d2);
    if (result < 2) return result;
    // Directories follow dummy types.
    result = directory_test(d1, d2);
    if (result < 2) return result;
    return 2;
}


// Date comparison functions.//////////////////////////////////////////


static gint
compare_date_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (DATE_SORT, d1->en, d2->en, FALSE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_date_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (DATE_SORT, d1->en, d2->en, FALSE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

static gint
compare_date_case_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (DATE_SORT, d1->en, d2->en, TRUE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_date_case_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (DATE_SORT, d1->en, d2->en, TRUE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

// SIZE_SORT comparison functions.//////////////////////////////////////////
static gint
compare_size_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (SIZE_SORT, d1->en, d2->en, FALSE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_size_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (SIZE_SORT, d1->en, d2->en, FALSE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

static gint
compare_size_case_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (SIZE_SORT, d1->en, d2->en, TRUE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_size_case_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (SIZE_SORT, d1->en, d2->en, TRUE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}




// OWNER_SORT comparison functions.//////////////////////////////////////////
static gint
compare_owner_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (OWNER_SORT, d1->en, d2->en, FALSE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_owner_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (OWNER_SORT, d1->en, d2->en, FALSE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}
static gint
compare_owner_case_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (OWNER_SORT, d1->en, d2->en, TRUE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_owner_case_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (OWNER_SORT, d1->en, d2->en, TRUE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

// GROUP_SORT comparison functions.//////////////////////////////////////////
static gint
compare_group_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (GROUP_SORT, d1->en, d2->en, FALSE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_group_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (GROUP_SORT, d1->en, d2->en, FALSE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}
static gint
compare_group_case_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (GROUP_SORT, d1->en, d2->en, TRUE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_group_case_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (GROUP_SORT, d1->en, d2->en, TRUE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

// MODE_SORT comparison functions.//////////////////////////////////////////
static gint
compare_mode_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (MODE_SORT, d1->en, d2->en, FALSE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_mode_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (MODE_SORT, d1->en, d2->en, FALSE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

static gint
compare_mode_case_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_up (MODE_SORT, d1->en, d2->en, TRUE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_mode_case_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method

    result = entry_compare_down (MODE_SORT, d1->en, d2->en, TRUE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}



// Name comparison functions.//////////////////////////////////////////
static gint
compare_name_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method
    result = pathv_compare_up (d1->pathv, d2->pathv);
    return result;
    // If we try to break ties with date comparison,
    // we may run into a circular dependency.
    // return (compare_date_up (d1, d2));
 }

static gint
compare_name_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method 
    result = pathv_compare_down (d1->pathv, d2->pathv);
    return result;
    // If we try to break ties with date comparison,
    // we may run into a circular dependency.
    // return (compare_date_down (d1, d2));
}
static gint
compare_name_case_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method
    result = pathv_compare_case_up (d1->pathv, d2->pathv);
    return result;
    // If we try to break ties with date comparison,
    // we may run into a circular dependency.
    // return (compare_date_up (d1, d2));
 }

static gint
compare_name_case_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    // Requested sort method 
    result = pathv_compare_case_down (d1->pathv, d2->pathv);
    return result;
    // If we try to break ties with date comparison,
    // we may run into a circular dependency.
    // return (compare_date_down (d1, d2));
}

// TYPE_SORT comparison functions.//////////////////////////////////////////
static gint
compare_type_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    result = entry_compare_up (TYPE_SORT, d1->en, d2->en, FALSE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_up (d1, d2);

}



static gint
compare_type_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    result = entry_compare_down (TYPE_SORT, d1->en, d2->en, FALSE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_down (d1, d2);
}

static gint
compare_type_case_up (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    result = entry_compare_up (TYPE_SORT, d1->en, d2->en, TRUE);
    if(result)
        return result;
    // Break ties with name comparison
    return compare_name_case_up (d1, d2);

}



static gint
compare_type_case_down (const void *a, const void *b) {
    const dir_t *d1 = (const dir_t *)a;
    const dir_t *d2 = (const dir_t *)b;
    gint result = standard_test(d1, d2);
    if (result < 2) return result;

    result = entry_compare_down (TYPE_SORT, d1->en, d2->en, TRUE);
    if(result) return result;
    // Break ties with name comparison
    return compare_name_case_down (d1, d2);
}


///////////////////////////////////////////////////////////////////////////////////

/**
 * xfdir_sort:
 * @xfdir_p: pointer to an xfdir_t structure
 *
 * Sorts an xfdir_t structure according to the settings in xfdir_p->preferences 
 * and xfdir_p->sort_column. Ascending is TRUE or FALSE and sort_column may be: 
 * 
 * TYPE_SORT, NAME_SORT, SIZE_SORT,
 * DATE_SORT, OWNER_SORT, GROUP_SORT, MODE_SORT.
 * <para>
 * qsort library routine is used as a sorting method.
 * </para>
 * 
 **/
static void
xfdir_sort (xfdir_t * xfdir_p) {
    NOOP("reload_iconview: do quicksort here: sortcolumn is %d, preferences=%d\n",
	    xfdir_p->sort_column, xfdir_p->preferences);
    NOOP ("xfdir_p=0x%x, pathc is %d\n", 
	    GPOINTER_TO_INT(xfdir_p), xfdir_p->pathc);

//    set_sort_column (xfdir_p->sort_column);

    gint (*sort_f)(const void *, const void *) = NULL;
    if (SHOWS_ASCENDING(xfdir_p->preferences)) {
	switch (xfdir_p->sort_column) {
	    case NAME_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_name_case_up;
		} else {
		    sort_f = compare_name_up;
		}
		break;
	    case SIZE_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_size_case_up;
		} else {
		    sort_f = compare_size_up;
		}
		break;
	    case DATE_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_date_case_up;
		} else {
		    sort_f = compare_date_up;
		}
		break;
	    case OWNER_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_owner_case_up;
		} else {
		    sort_f = compare_owner_up;
		}
		break;
	    case GROUP_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_group_case_up;
		} else {
		    sort_f = compare_group_up;
		}
		break;
	    case MODE_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_mode_case_up;
		} else {
		    sort_f = compare_mode_up;
		}
		break;
	    default: // TYPE_SORT
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_type_case_up;
		} else {
		    sort_f = compare_type_up;
		}
		break;
	}
    } else {
	switch (xfdir_p->sort_column) {
	    case NAME_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_name_case_down;
		} else {
		    sort_f = compare_name_down;
		}
		break;
	    case SIZE_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_size_case_down;
		} else {
		    sort_f = compare_size_down;
		}
		break;
	    case DATE_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		   sort_f = compare_date_case_down;
		} else {
		   sort_f = compare_date_down;
		}
		break;
	    case OWNER_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_owner_case_down;
		} else {
		    sort_f = compare_owner_down;
		}
		break;
	    case GROUP_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_group_case_down;
		} else {
		    sort_f = compare_group_down;
		}
		break;
	    case MODE_SORT:
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_mode_case_down;
		} else {
		    sort_f = compare_mode_down;
		}
		break;
	    default: // TYPE_SORT
		if (IS_CASE_INSENSITIVE(xfdir_p->preferences)) {
		    sort_f = compare_type_case_down;
		} else {
		    sort_f = compare_type_down;  
		}
		break;
	}
    }
    
    qsort ((void *)xfdir_p->gl, xfdir_p->pathc, sizeof (dir_t),
#ifdef __COMPAR_FN_T
               (__compar_fn_t)
#endif
               sort_f);

}


// function to do a background reload and set things up for the update
static xfdir_t *
get_new_xfdir (xfdir_t *xfdir_p){
    // allocate memory for reloaded xfdir structure
    xfdir_t *new_xfdir_p = 
	new_xfdir_p = (xfdir_t *) malloc (sizeof (xfdir_t));

    NOOP (stderr, "monitor new_xfdir_p=0x%lx\n", (long unsigned)new_xfdir_p);

    // initialize new xfdir structure with current xfdir structure information
    memcpy (new_xfdir_p, xfdir_p, sizeof (xfdir_t));

    // new xfdir structure specific initializations:
    new_xfdir_p->pathc = 0;
    new_xfdir_p->gl = NULL;
    new_xfdir_p->en = rfm_copy_entry(xfdir_p->en);
    // reload directory (or module) into new xfdir structure
    // Since this is background thread, fullstat should be done here.
    if (update_xfdir (new_xfdir_p, TRUE, NULL) < 0){
	// Error in read step. count is <= 0, but should at least be 1, 
	// where that one would be the "../" record...
	g_free(new_xfdir_p);
	return NULL;
    }

    // sort directory (or module) using current user settings
    xfdir_sort (new_xfdir_p);

    return new_xfdir_p;
}


static gboolean
get_current_stat(record_entry_t *en, struct stat *st){
    if (en->module) return TRUE;
    memset(st, 0, sizeof(struct stat));
    if (stat(en->path, st) < 0){
	DBG("unable to stat %s\n", en->path);
	return FALSE; 
    }
    return TRUE;
}

static size_t
item_count(xfdir_t *xfdir_p, GHashTable *pop_hash){
    record_entry_t *en = xfdir_p->en;
    size_t count = 1;
    if (!en->module) {
        GDir *dir = g_dir_open(en->path, 0, NULL);
        if (!dir) return 0;
        while (g_dir_read_name (dir) != NULL) count++;
	if (count==xfdir_p->pathc){
	    // test coherence of count
	    g_dir_rewind(dir);
	    const gchar *name;
	    while ((name=g_dir_read_name (dir)) != NULL){
		gchar *path = g_build_filename(en->path, name, NULL);
		if (!g_hash_table_lookup(pop_hash, path)) {
		    // incoherent count
		    count = -1;
		    g_free(path);
		    break;
		}
		g_free(path);
	    }
	}

        g_dir_close (dir);
        
	// 

        TRACE("item count=%ld pathc=%ld\n", (long)count, (long)xfdir_p->pathc);
        return count;
    }
    // Modules
    xfdir_t *reloaded_xfdir_p = get_new_xfdir(xfdir_p);
    count = reloaded_xfdir_p->pathc;
    // release reloaded_xfdir_p
    xfdir_free_data (reloaded_xfdir_p);
    return count;
}

// thread_monitor_f is in charge of xfdir_p cleanup!
// AND view_p cleanup!
static gpointer
thread_monitor_f (gpointer data) {
    //return NULL;
    xfdir_t *xfdir_p = (xfdir_t *) data;
    view_t *view_p = xfdir_p->view_p;

    // signal monitor birth
    g_mutex_lock(view_p->mutexes.monitor_run_mutex);
    view_p->monitor_running = TRUE;
    g_mutex_unlock(view_p->mutexes.monitor_run_mutex);

    // memory structure should be received from parent thread
    gboolean initial = TRUE; // Don't do content icons on first pass
    //gboolean initial = FALSE; // Do content icons on first pass
    if(xfdir_p == NULL || !xfdir_p->en || !xfdir_p->en->path) {
	g_error("thread_monitor_f(): xfdir_p == NULL || !xfdir_p->path");
    }
#if 0
    if (view_p->widgets.scrolled_window) {
	GtkAdjustment *adjustment = 
	    gtk_scrolled_window_get_vadjustment(view_p->widgets.scrolled_window);
	gdouble v = floor(gtk_adjustment_get_value(adjustment));
	gint iv = v;
	g_object_set_data(G_OBJECT(view_p->widgets.scrolled_window), 
		"adjustment_value", GINT_TO_POINTER(iv));
    }
#endif


    // We get a copy of the entry, since view entry will become
    // invalid after a reload.
    //xfdir_p->en = rfm_copy_entry(view_p->en);
    NOOP("monitor increments view ref: 0x%x\n", GPOINTER_TO_INT(view_p));
	
    // count items present
    // gint current_count = count_xfdir_population(view_p, xfdir_p);
    // create population hash
    GHashTable *pop_hash = create_population_hash(xfdir_p);

    gboolean refresh_action = FALSE;


    // check whether stat condition has changed on directory 
    struct stat current_st;
    if (!get_current_stat(xfdir_p->en, &current_st)) {
	xfdir_exit_monitor(view_p);
    }

    // Setup to enter inner loop is complete. For this we have a count of 
    // items which should be present (inode count) and a stat structure 
    // with current information.
    DBG( "serial=%d monitor %d setup for %s\n", 
	    view_p->flags.monitor_id, xfdir_p->monitor_id, xfdir_p->en->path);
    rfm_global_t *rfm_global_p = rfm_global();

    do { // inner loop 
        TRACE("monitor: global mutex lock\n");
	g_mutex_lock(rfm_global_p->status_mutex);
	gint status = rfm_global_p->status;
	g_mutex_unlock(rfm_global_p->status_mutex);
	if (status == STATUS_EXIT) break; 

        TRACE("monitor: view mutex lock\n");
	g_mutex_lock(xfdir_p->view_p->mutexes.status_mutex);
	status = xfdir_p->view_p->flags.status;
	g_mutex_unlock(xfdir_p->view_p->mutexes.status_mutex);
	if (status == STATUS_EXIT)  break;

	if(exit_monitor (xfdir_p)) break;

	// check if the inner event loop must be broken
	// this happens if monitor thread will exit or a reload condition
	// has been detected
	if(!xfdir_monitor_continue (xfdir_p)) {
	    NOOP(stderr, "xfdir monitor: !xfdir_monitor_continue\n");
	    //rfm_threadwait ();
	    break;
	}
	TRACE("monitor: xfdir_monitor_continue... \n");

	// Check if the event loop must be placed in suspend mode.
	// Skip mode will bypass control thread wait.
	gboolean skip = xfdir_monitor_skip (xfdir_p);

	if(skip) {
	    TRACE("monitor(0x%x): skip = %d\n", GPOINTER_TO_INT(g_thread_self()), skip);
	    if(exit_monitor (xfdir_p)) break;
	    rfm_threadwait ();  
	    continue;
	} 

	gboolean disabled = xfdir_monitor_disabled (xfdir_p);
	if (disabled){
	    TRACE("monitor: disabled=%d \n", disabled);
	    if(exit_monitor (xfdir_p)) break;
	    goto breath;
	}
	
	TRACE( "mutexes.monitor_lock requested...\n" );
	rfm_rw_lock_writer_lock (&(xfdir_p->view_p->mutexes.monitor_lock));	
	TRACE("mutexes.monitor_lock obtained.\n" );


        // Fundamental reload condition: number of items has changed.
        size_t n_items = item_count(xfdir_p, pop_hash);
        if (n_items != xfdir_p->pathc) {
            DBG("monitor: item count differs (%ld != %ld), scheduling refresh...\n", n_items, xfdir_p->pathc);
            refresh_action = TRUE;
        } 
        //               
	// check for an immediate reload condition which requires
	// immediate reload action (gboolean refresh_action)
	if(
		refresh_action || 
		reload_condition (xfdir_p))
	{

	    // Get stat of directory corresponding to the
	    // upcoming reload.
	    if (!get_current_stat(xfdir_p->en, &current_st)){
		xfdir_exit_monitor(view_p);
		rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
		TRACE("monitor: !get_current_stat(xfdir_p->en, &current_st)\n" );
		continue;
	    }
	    // obtain new xfdir structure
	    xfdir_t *reloaded_xfdir_p = get_new_xfdir(xfdir_p);
	    // Check if reload action was successful (non NULL return value)
	    // And check whether there was an actual reload (new xfdir structure)
	    if (reloaded_xfdir_p && reloaded_xfdir_p != xfdir_p) {
		// safeguard check: if updated xfdir structure is no longer
		// required, discard updated directory 
		if(exit_monitor (xfdir_p)) {
		    TRACE ("xfdir structure is no longer required: now exiting monitor and dumping new_xfdir_p.\n");
		    // rodent no longer needs this.
		    xfdir_free_data (reloaded_xfdir_p);
		    g_free (reloaded_xfdir_p);

		    rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
		    break;
		}
    
		// Here we should disable the refresh action flag.
		refresh_action = FALSE;
		TRACE("monitor: setting up actual refresh! reloaded_xfdir_p != xfdir_p ------------\n");
		xfdir_t *obsolete_xfdir_p = xfdir_p;
		xfdir_p = reloaded_xfdir_p;
		xfdir_free_data (obsolete_xfdir_p);
		g_free (obsolete_xfdir_p);

		//current_count = count_xfdir_population(view_p, xfdir_p);
		// RELOAD HERE
		//
		// (mount status changes: fstab plugin could make do with
		// a simple expose...)
		//
		TRACE("monitor: thread_reload_view ------------\n");
		thread_reload_view (view_p, xfdir_p);
		// now that we are reloaded, update the pop_hash.
		g_hash_table_destroy(pop_hash);
		pop_hash = create_population_hash(xfdir_p);

		g_thread_yield();
		rfm_threadwait(); 
		if (xfdir_p->view_p->en && xfdir_p->view_p->en->module){
		    rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
		    gint result = GPOINTER_TO_INT(rfm_void(PLUGIN_DIR, view_p->en->module, "monitor_pause"));
		    if (result > 0) {
			continue;
		    }
		    goto breath;
		}
		TRACE("monitor: No breath...\n");

		// No breath... This will look good but may use too much resources...
		//continue;
	    } else {
		TRACE("monitor: Ignoring failed background reload 0x%x\n", GPOINTER_TO_INT(reloaded_xfdir_p));
		rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
		continue;
	    }
	}
	else {
	    // reload condition has not occurred, but stat information
	    // or thumbnail image may have changed for an existing item
	    // without affecting directory stat information. In this case
	    // we update stat information for individual xfdir structure items. 
	    TRACE("monitor: now checking update_stat_info...\n");
	    if (update_stat_info (xfdir_p, initial))
	    {
		// refresh on update stat info is anacronic. Individual items
		// have their stat info updated. view stat is not significant
		// and reload is done on mismatch of item count.
		TRACE ("monitor: refresh_pending on update_stat_info()...\n");
		// A TRUE return value indicates that the directory
		// does not correspond to the actual files 
		// which currently exist.
		// In this case we queue a refresh.
		refresh_action = TRUE;
                DBG("monitor: update_stat_info() requests a full refresh...\n");
		rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
		continue; // this implies no wait on loop cycle.
	    
	    }
	    if (initial){
		initial = FALSE;
		rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
		continue; // this implies no wait on loop cycle.
	    }
	    if ( xfdir_p->en->module &&
		    rfm_natural(PLUGIN_DIR, xfdir_p->en->module, view_p, "monitor_skipwait") > GPOINTER_TO_INT(0))
	    {
		TRACE("monitor:  doing skipwait\n");
		rfm_threadwait ();
		rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
		continue; // this implies no wait on loop cycle.
	    } else {
		NOOP(stderr, "XXXX  monitor says skipwait is FALSE\n");
	    }

	}
	rfm_rw_lock_writer_unlock (&(xfdir_p->view_p->mutexes.monitor_lock));
breath:
	
	// Controlled breath condition
	// Skip mode will not enter here.

	TRACE("monitor: taking breath......\n");
	g_mutex_lock(view_p->mutexes.monitor_control);
	if (view_p->flags.monitor_go == FALSE) {
#define MONITOR_WAIT_PERIOD 3
            TRACE("-- monitor waiting for signal\n");
	    rfm_cond_timed_wait(view_p->mutexes.monitor_signal, view_p->mutexes.monitor_control, MONITOR_WAIT_PERIOD);
	    TRACE("monitor: signalled or timed out... \n");
	} else {
	    view_p->flags.monitor_go = FALSE;
	}
	    
	g_mutex_unlock(view_p->mutexes.monitor_control);
	if(exit_monitor (xfdir_p)) break;
	if(!xfdir_monitor_continue (xfdir_p))break; 

    } while (2);   // end inner loop 2

    NOOP(stderr, "xfdir monitor: monitor id=%d loop broken for %s\n", xfdir_p->monitor_id, xfdir_p->en->path);
    // inner loop has exited.
    //  * monitor thread has met an exit condition and is no longer necessary
    //
    // Here we consider case 1, monitor thread had met an exit condition:
    NOOP ("monitor: all is cool to release xfdir memory\n");
    NOOP(stdout, "monitor: id=%d ending now\n", xfdir_p->monitor_id);
    NOOP ("monitor: will free xfdir_p=0x%lx\n", (long unsigned)xfdir_p);

    // monitor thread ends here: 
    NOOP("monitor decrements view ref: 0x%x\n", GPOINTER_TO_INT(view_p));
    // current xfdir structure (no longer needed) cleanup
    xfdir_free_data (xfdir_p);
    //rfm_destroy_entry(xfdir_p->en);
    g_free (xfdir_p);
    xfdir_p = NULL;
    g_hash_table_destroy(pop_hash);

    DBG("xfdir monitor: ending monitor thread\n");

    // signal monitor death
    g_mutex_lock(view_p->mutexes.monitor_run_mutex);
    view_p->monitor_running=FALSE;
    g_cond_signal(view_p->mutexes.monitor_run_signal);
    g_mutex_unlock(view_p->mutexes.monitor_run_mutex);


    // monitor event loop ends here (end outer loop)
    return NULL;
}

static void
xfdir_start_monitor (view_t * view_p, xfdir_t * xfdir_p) {
    // (mutex lock might be overkill here)
    xfdir_p->monitor_id = view_p->flags.monitor_id;
    view_p->flags.monitor_enabled=TRUE;

    // start the background monitor thread
    xfdir_p->thread = rfm_view_thread_create(view_p, thread_monitor_f, (gpointer) xfdir_p, "thread_monitor_f");
      
    NOOP(stderr, "--- thread(0x%x): xfdir_start_monitor id=%d\n", GPOINTER_TO_INT(xfdir_p->thread), xfdir_p->monitor_id);
    if (xfdir_p->thread == NULL) {
        DBG("cannot create monitor thread at xfdir_start_monitor (xfdir.i)\n");
        view_p->flags.monitor_enabled=FALSE;
    }

}

static void
set_local_up_item(xfdir_t *xfdir_p, gint count){
    gchar *fullpath = g_path_get_dirname (xfdir_p->en->path);
    gchar *basepath = g_path_get_basename (fullpath);
    if(strcmp (xfdir_p->en->path, "/") == 0) {
	xfdir_p->gl[count].en = NULL;
	xfdir_p->gl[count].pathv = g_strdup (g_get_host_name ());
	g_free(fullpath);
	g_free(basepath);
    } else {
	xfdir_p->gl[count].en = rfm_stat_entry (fullpath, 0);
	xfdir_p->gl[count].pathv = basepath;
	g_free(fullpath);
	SET_DUMMY_TYPE (xfdir_p->gl[count].en->type);
	SET_UP_TYPE (xfdir_p->gl[count].en->type);
	NOOP(stdout, "UP item type= 0x%x\n", xfdir_p->gl[count].en->type);
    }
}


static void
set_directory_item(xfdir_t *xfdir_p, gint count, xd_t *xd_p, gboolean fullstat){
    gint type=0;
    gchar *fullpath = NULL;
	xfdir_p->gl[count].pathv = xd_p->d_name;
	fullpath = g_build_filename (xfdir_p->en->path, xd_p->d_name, NULL);
    

#ifdef HAVE_STRUCT_DIRENT_D_TYPE
    switch (xd_p->d_type){
      case DT_BLK:
	//This is a block device.
	SET_SBLK(type);
	break;
      case DT_CHR:
	//This is a character device.
	SET_SCHR(type);
	break;
      case DT_DIR:
	//This is a directory.
	SET_SDIR(type);
	break;
      case DT_FIFO:
	//This is a named pipe (FIFO).
	SET_SFIFO(type);
	break;
      case DT_LNK:
	//This is a symbolic link.
	SET_SLNK(type);
	break;
      case DT_REG:
	//This is a regular file.
	SET_SREG(type);
	break;
      case DT_SOCK:
	//This is a Unix domain socket.
	SET_SSOCK(type);
	break;
      case DT_UNKNOWN:
	//The file type is unknown.
	fullstat = TRUE;
	break;			
    }
    if (IS_LOCAL_TYPE(xfdir_p->en->type)){ 
	SET_LOCAL_TYPE(type);
    }
#else
    fullstat = TRUE;
#endif


    //if (fullstat || count < FULL_LOAD_COUNT) 
    // Let's avoid full stat unless strictly specified.
    // This will help keep the icon replacement method clean.
    if (fullstat) 
    {
	// Create en and en->st, fill in en->st.
	xfdir_p->gl[count].en = rfm_stat_entry (fullpath, type);
	xfdir_p->tama += xfdir_p->gl[count].en->st->st_size;
    } 
    else
    {
	// Create en and en->st, leave en->st memset to 0.
	xfdir_p->gl[count].en = rfm_mk_entry_path (fullpath, type);
    }
    g_free(fullpath);


}

// The basic load function...
// update xfdir structure for directory (or module)
static int
update_xfdir (xfdir_t * xfdir_p, gboolean fullstat, gint *heartbeat) {
    record_entry_t *en = xfdir_p->en;
    const gchar *module = en->module;
    if(module) {
	NOOP(stderr, "update_xfdir, heartbeat=%d\n",GPOINTER_TO_INT(heartbeat));
	gint retval = 
	    GPOINTER_TO_INT(rfm_natural(PLUGIN_DIR, 
			xfdir_p->en->module, xfdir_p, 
			"module_xfdir_get"));
	NOOP(stderr, "update_xfdir got 0x%x\n", retval);
	if (retval) return 1;
	else {
	    DBG("exported function \"module_xfdir_get\" not defined for module \"%s\"\n",
		    xfdir_p->en->module);
	    return 0;
	}
    } 
    xfdir_p->tama = 0;

    // Potential block: stat(). 
    // This would block the thread. Heartbeat controlled load would not 
    // block the program.
    //
    if (stat(en->path, en->st) < 0) return 0;
    //if (!thread_stat(en->path, en->st)) return 0;
    
    // This just does a read, not stat. May be retrieved from kernel cache 
    // and is extremely fast.
    GSList *directory_list = read_files_local (xfdir_p, heartbeat);
    if(directory_list == NULL) return 0;

    // But if sort method is not TYPE or NAME, then we must do
    // a full stat to avoid a secondary remapping of the icons.
    if (xfdir_p->sort_column != TYPE_SORT
	    && xfdir_p->sort_column != NAME_SORT) fullstat = TRUE;
 
    
    xfdir_p->pathc = g_slist_length(directory_list);
    xfdir_p->gl = (dir_t *) malloc (xfdir_p->pathc * sizeof (dir_t));
    if(xfdir_p->gl == NULL) g_error("malloc: %s\n", strerror(errno));
    memset (xfdir_p->gl, 0, xfdir_p->pathc * sizeof (dir_t));
    GSList *list = directory_list;
    gint count = 0;
    // This is not so fast. We may be doing a stat on all or first items.
    for(;list && list->data; list = list->next, count++){
	struct xd_t *xd_p = list->data;
	if (strcmp(xd_p->d_name, "..")==0) {
	    set_local_up_item(xfdir_p, count);
	    g_free(xd_p->d_name);
	} else {
	    // d_name is passed on to xfdir_p
	    if (heartbeat) NOOP("heartbeat records stat: %s...%d\n", xd_p->d_name, *heartbeat);
	    set_directory_item(xfdir_p, count, xd_p, fullstat);
	    if (heartbeat) (*heartbeat)++;
	}
	g_free(xd_p);
    }
    g_slist_free(directory_list);
    
    return 1;
}
    
static gboolean want_monitor(widgets_t *widgets_p, record_entry_t *en);

static xfdir_t *
xfdir_get (view_t * view_p, view_preferences_t *view_preferences_p,
	const record_entry_t * in_en, 
	gint *heartbeat) {
    // allocate memory for new xfdir structure and initialize values
    xfdir_t *xfdir_p = (xfdir_t *) malloc (sizeof (xfdir_t));
    if (!xfdir_p) g_error("malloc: %s", strerror(errno));
    memset ((void *)xfdir_p, 0, sizeof (xfdir_t));
    
    xfdir_p->en = rfm_copy_entry(in_en);
    xfdir_p->view_p = view_p;

    // set the preferred sort method. 
    // If sort method is not "type" or "name", then
    // a full stat must be performed on the directory entries.
    if (view_preferences_p) {
	xfdir_p->preferences = view_preferences_p->preferences;
	xfdir_p->sort_column = view_preferences_p->sortcolumn;
    } else {
	xfdir_p->preferences = DEFAULT_VIEW_PREFERENCES;
	xfdir_p->sort_column = DEFAULT_SORT_COLUMN;
    }
    
    // read local directory information and update xfdir structure
    NOOP(stderr, "xfdir monitor: starting time on update_xfdir for %s...\n", xfdir_p->en->path);


    gboolean fullstat = FALSE;
#if 0
    // doing fullstat will hamper icon replacement.
    if (xfdir_p->sort_column != TYPE_SORT && xfdir_p->sort_column != NAME_SORT){
	fullstat = TRUE;
    }
#endif
    // Monitor will be off for .gvfs mounts, so no background stat
    // will be done. This make fullstat necessary on initial read.
#if 0
    gchar *gvfs_hack_string = g_strdup("/.gvfs/");
    if (strstr(xfdir_p->en->path, gvfs_hack_string)) fullstat = TRUE;
    g_free(gvfs_hack_string);
#else
    if (!want_monitor(&(view_p->widgets), xfdir_p->en)) fullstat = TRUE;
#endif

    update_xfdir (xfdir_p, fullstat, heartbeat);
    if (heartbeat) *heartbeat = -1;

    if (xfdir_p->pathc == READ_ERROR) {
	NOOP(stderr, "xfdir_p->pathc == READ_ERROR\n");
	g_free(xfdir_p);
	return NULL;
    }


    // now we do the sorting. Stat record is not necessary for
    // name or type sort methods, so barebones should be fine.
    NOOP("xfdir monitor: ---------- sortcolumn=%d\n", xfdir_p->sort_column);

    xfdir_sort (xfdir_p);

    // set default status tag for directory or module (sizetag)
    xfdir_set_entry_tag (&(view_p->widgets), xfdir_p->en, xfdir_p->tama);

    return xfdir_p;
}


