#! /usr/bin/env python
# -*- coding: utf-8 -*-

##############################################################################
##  DendroPy Phylogenetic Computing Library.
##
##  Copyright 2010-2015 Jeet Sukumaran and Mark T. Holder.
##  All rights reserved.
##
##  See "LICENSE.rst" for terms and conditions of usage.
##
##  If you use this work or any portion thereof in published work,
##  please cite it as:
##
##     Sukumaran, J. and M. T. Holder. 2010. DendroPy: a Python library
##     for phylogenetic computing. Bioinformatics 26: 1569-1571.
##
##############################################################################

"""
Tests of parsimony scoring.
"""

import unittest
import dendropy
from dendropy.calculate import treescore
import os
import sys
sys.path.insert(0, os.path.dirname(__file__))
from support import pathmap

class ParsimonyScoringTest(unittest.TestCase):

    def test_pscores_with_standard_data_and_gaps_as_new_states(self):
        # #NEXUS
        # begin paup;
        #     set warnroot = no;
        #     exe apternodus.chars.nexus;
        #     gett file = apternodus.tre;
        #     set criterion = parsimony;
        #     pset gap = newstate;
        #     pscore;
        # end;
        expected_scores = [396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 396, 713, 715, 723, 733, 672, 719, 734, 709, 695, 686]
        # begin paup;
        #   set warnroot = no;
        #   execute apternodus.chars.nexus;
        #   gett file = apternodus.tre;
        #   set crit=parsimony;
        #   pset gap = newstate;
        #   pscore / single = all scorefile=apternodus.gaps-as-newstate.scores.txt;
        # end;
        expected_per_site_scores = [
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 4, 2, 1, 2, 3, 4, 5, 5, 3, 4, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 6, 6, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 2, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 3, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 6, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 6, 6, 6, 4, 5, 3, 2, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 2, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 6, 6, 6, 4, 5, 3, 2, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 2, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 6, 6, 6, 4, 5, 3, 2, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 2, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 4, 2, 1, 2, 3, 4, 5, 5, 3, 4, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 6, 6, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 2, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 4, 2, 1, 2, 3, 4, 5, 5, 3, 4, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 6, 6, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 2, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 5, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 6, 6, 4, 5, 3, 2, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 6, 6, 4, 5, 3, 2, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 6, 6, 4, 5, 3, 2, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 3, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 6, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 3, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 6, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 5, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 5, 6, 4, 4, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 6, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 5, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 4, 2, 1, 2, 4, 4, 5, 5, 3, 4, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 3, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 6, 5, 6, 4, 5, 3, 3, 5, 2, 1, 3, 5, 1, 4, 3, 7, 6, 3, 4, 2, 3, 3, 5, 6, 5, 3, 2, 5, 5, 1, 5, 6, 7, 3, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 3, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [8, 13, 13, 4, 9, 12, 3, 4, 2, 4, 2, 2, 5, 6, 7, 2, 2, 3, 7, 7, 6, 8, 6, 7, 10, 10, 3, 7, 10, 3, 6, 8, 7, 7, 7, 4, 3, 7, 2, 7, 2, 4, 8, 9, 10, 13, 8, 6, 2, 7, 7, 5, 1, 2, 5, 7, 8, 8, 9, 6, 8, 10, 4, 9, 2, 1, 9, 7, 2, 6, 5, 11, 10, 9, 6, 3, 7, 8, 9, 7, 7, 6, 9, 9, 9, 2, 8, 9, 13, 4, 7, 9, 3, 2, 4, 6, 12, 7, 4, 4, 3, 4, 3, 2, 3, 5, 2, 8, 4, 3, 4, 2, 2, 4, 4, 5, 3, 15,],
            [8, 12, 13, 4, 5, 12, 3, 4, 2, 4, 3, 2, 4, 6, 7, 2, 2, 3, 7, 6, 5, 8, 8, 9, 9, 10, 3, 8, 11, 3, 8, 8, 7, 7, 8, 4, 3, 7, 2, 7, 2, 4, 10, 11, 10, 13, 8, 6, 2, 6, 8, 5, 2, 3, 5, 8, 9, 8, 10, 5, 10, 10, 5, 9, 3, 1, 7, 6, 2, 7, 5, 10, 9, 8, 6, 3, 7, 6, 7, 7, 7, 6, 10, 9, 10, 2, 7, 7, 11, 4, 7, 10, 3, 3, 5, 6, 12, 7, 4, 4, 3, 4, 3, 2, 3, 5, 2, 7, 4, 3, 4, 2, 1, 3, 3, 6, 3, 16,],
            [7, 12, 11, 4, 10, 10, 3, 4, 2, 4, 3, 2, 5, 7, 8, 2, 2, 3, 7, 5, 8, 10, 9, 9, 10, 8, 3, 9, 11, 3, 7, 6, 8, 8, 9, 4, 3, 7, 2, 6, 2, 4, 9, 7, 10, 12, 8, 6, 2, 6, 9, 5, 2, 3, 5, 7, 9, 10, 9, 5, 8, 11, 5, 9, 3, 1, 10, 6, 2, 7, 5, 11, 10, 9, 6, 3, 7, 8, 7, 7, 6, 6, 10, 7, 9, 2, 8, 8, 14, 4, 7, 10, 3, 3, 5, 7, 14, 6, 3, 4, 3, 4, 1, 2, 3, 5, 2, 6, 4, 3, 4, 2, 2, 5, 4, 5, 2, 14,],
            [8, 12, 11, 4, 9, 13, 3, 4, 2, 4, 3, 2, 5, 5, 7, 2, 2, 3, 7, 6, 7, 10, 9, 9, 10, 9, 3, 8, 10, 3, 7, 7, 6, 8, 9, 4, 2, 7, 2, 7, 2, 4, 11, 9, 8, 12, 9, 5, 2, 6, 9, 5, 2, 3, 5, 7, 9, 11, 7, 6, 9, 10, 5, 8, 3, 1, 8, 7, 2, 6, 5, 9, 11, 9, 6, 3, 8, 9, 8, 6, 6, 7, 9, 9, 10, 2, 9, 9, 14, 4, 7, 10, 3, 3, 5, 6, 11, 7, 4, 4, 3, 4, 3, 2, 3, 6, 2, 7, 4, 3, 4, 2, 2, 4, 5, 6, 3, 17,],
            [5, 7, 7, 3, 8, 9, 3, 4, 2, 3, 2, 2, 4, 4, 7, 2, 2, 3, 7, 4, 8, 9, 7, 7, 8, 10, 3, 7, 9, 3, 6, 5, 7, 6, 8, 4, 3, 6, 2, 6, 2, 4, 10, 6, 7, 13, 7, 5, 2, 6, 7, 5, 2, 3, 5, 6, 9, 9, 11, 5, 7, 12, 4, 8, 3, 1, 9, 5, 2, 7, 5, 9, 9, 8, 5, 3, 5, 9, 7, 7, 7, 6, 10, 8, 10, 2, 11, 9, 14, 4, 7, 9, 2, 3, 5, 6, 8, 4, 4, 4, 3, 5, 3, 2, 3, 5, 2, 7, 5, 3, 4, 2, 2, 4, 4, 6, 4, 14,],
            [7, 11, 12, 4, 8, 13, 3, 4, 2, 4, 3, 2, 5, 7, 8, 2, 2, 3, 7, 5, 5, 8, 9, 9, 10, 9, 3, 9, 10, 3, 8, 6, 9, 6, 8, 4, 3, 6, 2, 6, 2, 4, 9, 9, 9, 12, 8, 5, 2, 7, 9, 4, 2, 3, 5, 10, 9, 9, 9, 6, 10, 9, 5, 9, 3, 1, 10, 8, 2, 7, 5, 10, 9, 10, 6, 3, 7, 7, 7, 7, 7, 7, 8, 8, 8, 2, 7, 9, 13, 4, 7, 9, 3, 3, 5, 7, 14, 6, 4, 4, 3, 3, 3, 2, 3, 5, 2, 6, 3, 3, 4, 2, 2, 4, 5, 2, 4, 15,],
            [8, 12, 13, 4, 7, 12, 3, 4, 2, 4, 3, 2, 5, 6, 7, 2, 2, 3, 6, 5, 7, 9, 8, 8, 7, 8, 3, 8, 12, 3, 8, 8, 8, 8, 9, 3, 3, 7, 2, 7, 2, 4, 10, 8, 11, 11, 7, 5, 2, 7, 8, 5, 2, 3, 5, 8, 9, 9, 10, 6, 9, 12, 5, 11, 3, 1, 11, 8, 2, 7, 5, 11, 11, 8, 6, 3, 8, 5, 7, 7, 7, 6, 11, 8, 8, 2, 11, 9, 14, 4, 6, 11, 3, 3, 5, 6, 13, 7, 4, 4, 3, 4, 3, 2, 3, 5, 2, 6, 4, 3, 4, 2, 2, 4, 5, 5, 4, 13,],
            [8, 11, 13, 3, 10, 12, 3, 4, 2, 4, 3, 2, 5, 4, 6, 2, 2, 3, 7, 6, 8, 9, 6, 6, 9, 9, 3, 8, 11, 3, 7, 8, 9, 7, 10, 4, 3, 6, 2, 6, 2, 4, 11, 9, 9, 11, 9, 6, 2, 6, 8, 4, 2, 3, 5, 8, 8, 9, 9, 6, 9, 10, 5, 8, 3, 1, 8, 7, 2, 5, 5, 9, 7, 8, 6, 3, 6, 6, 8, 7, 6, 7, 10, 9, 7, 2, 7, 9, 13, 4, 7, 10, 3, 3, 5, 7, 14, 6, 4, 4, 3, 5, 3, 2, 3, 4, 2, 6, 5, 3, 4, 2, 2, 4, 4, 3, 4, 15,],
            [6, 12, 11, 4, 9, 11, 3, 3, 2, 4, 3, 2, 3, 6, 7, 2, 2, 3, 6, 6, 9, 10, 9, 9, 10, 10, 3, 9, 11, 3, 7, 7, 6, 8, 7, 4, 3, 6, 1, 5, 2, 3, 9, 7, 8, 9, 8, 5, 2, 7, 8, 5, 2, 3, 5, 8, 9, 11, 8, 6, 8, 10, 5, 8, 3, 1, 6, 6, 2, 6, 5, 11, 8, 11, 6, 3, 8, 8, 7, 7, 7, 5, 9, 9, 10, 2, 7, 9, 11, 4, 7, 8, 3, 2, 4, 7, 13, 5, 4, 4, 2, 4, 2, 2, 2, 4, 2, 5, 4, 3, 4, 2, 2, 6, 5, 4, 3, 13,],
            [6, 9, 10, 4, 8, 12, 3, 3, 2, 4, 3, 2, 5, 6, 8, 2, 2, 3, 7, 6, 6, 8, 7, 7, 10, 6, 3, 6, 9, 3, 7, 7, 5, 8, 7, 4, 3, 6, 2, 6, 2, 3, 10, 9, 7, 11, 7, 6, 2, 6, 6, 5, 2, 3, 5, 8, 9, 8, 10, 6, 7, 11, 4, 8, 3, 1, 8, 6, 2, 7, 5, 10, 7, 10, 6, 3, 8, 8, 7, 6, 6, 7, 9, 8, 9, 2, 9, 9, 12, 4, 6, 10, 3, 3, 5, 5, 12, 6, 5, 4, 3, 5, 3, 2, 2, 4, 2, 7, 5, 3, 3, 2, 2, 5, 4, 5, 2, 16,],
        ]
        self.verify_pscores(
                trees_fname="apternodus.tre",
                chars_fname="apternodus.chars.nexus",
                matrix_type=dendropy.StandardCharacterMatrix,
                gaps_as_missing=False,
                expected_scores=expected_scores,
                expected_per_site_scores=expected_per_site_scores)

    def test_pscores_with_standard_data_and_gaps_as_missing(self):
        # #NEXUS
        # begin paup;
        #     set warnroot = no;
        #     exe apternodus.chars.nexus;
        #     gett file = apternodus.tre;
        #     set criterion = parsimony;
        #     pset gap = missing;
        #     pscore;
        # end;
        expected_scores = [ 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 671, 670, 678, 687, 633, 675, 689, 668, 652, 644]

        # begin paup;
        #   set warnroot = no;
        #   execute apternodus.chars.nexus;
        #   gett file = apternodus.tre;
        #   set crit=parsimony;
        #   pset gap = missing;
        #   pscore / single = all scorefile=apternodus.gaps-as-missing.scores.txt;
        # end;
        expected_per_site_scores = [
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 1, 2, 1, 2, 3, 4, 5, 2, 3, 1, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 5, 5, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 2, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 3, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 5, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 5, 5, 5, 3, 5, 3, 2, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 2, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 5, 5, 5, 3, 5, 3, 2, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 2, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 5, 5, 5, 3, 5, 3, 2, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 2, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 1, 2, 1, 2, 3, 4, 5, 2, 3, 1, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 5, 5, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 2, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 3, 1, 2, 1, 2, 3, 4, 5, 2, 3, 1, 5, 5, 2, 3, 6, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 7, 5, 5, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 2, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 4, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 5, 5, 3, 5, 3, 2, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 5, 5, 3, 5, 3, 2, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 5, 5, 3, 5, 3, 2, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 3, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 5, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 3, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 5, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 4, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 4, 5, 3, 4, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 5, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [4, 4, 3, 3, 5, 9, 2, 2, 1, 2, 3, 2, 3, 2, 1, 2, 1, 2, 4, 4, 5, 2, 3, 1, 5, 5, 2, 3, 7, 1, 1, 1, 3, 2, 3, 2, 1, 3, 1, 2, 1, 2, 7, 4, 3, 7, 2, 4, 1, 4, 3, 4, 1, 2, 4, 6, 5, 4, 5, 3, 5, 3, 3, 5, 2, 1, 3, 5, 1, 3, 2, 7, 6, 3, 4, 2, 3, 3, 4, 5, 4, 3, 2, 4, 5, 1, 5, 5, 6, 2, 3, 6, 2, 1, 4, 3, 5, 5, 3, 3, 2, 3, 2, 1, 2, 3, 1, 2, 3, 1, 2, 1, 2, 3, 3, 4, 1, 8,],
            [8, 13, 6, 4, 9, 12, 3, 4, 2, 4, 2, 2, 5, 6, 3, 2, 2, 3, 7, 7, 6, 3, 6, 1, 10, 10, 3, 7, 10, 3, 6, 8, 7, 7, 7, 4, 3, 7, 2, 5, 2, 4, 8, 9, 10, 13, 8, 6, 2, 7, 7, 5, 1, 2, 5, 7, 7, 7, 8, 5, 8, 10, 4, 9, 2, 1, 9, 7, 2, 4, 3, 11, 10, 9, 6, 3, 7, 8, 8, 6, 6, 6, 9, 7, 9, 2, 8, 8, 12, 3, 7, 9, 3, 2, 4, 6, 12, 7, 4, 4, 3, 4, 3, 2, 3, 5, 2, 6, 4, 3, 4, 2, 2, 4, 4, 5, 3, 15,],
            [8, 12, 5, 4, 5, 12, 3, 4, 2, 4, 3, 2, 4, 6, 3, 2, 2, 3, 7, 6, 5, 3, 8, 1, 9, 10, 3, 8, 11, 3, 8, 8, 7, 7, 8, 4, 3, 7, 2, 5, 2, 4, 10, 11, 10, 13, 8, 6, 2, 6, 8, 5, 2, 3, 5, 8, 8, 7, 9, 4, 10, 10, 5, 9, 3, 1, 7, 6, 2, 5, 3, 10, 9, 8, 6, 3, 7, 6, 6, 6, 6, 6, 10, 7, 10, 2, 7, 6, 10, 3, 7, 10, 3, 3, 5, 6, 12, 7, 4, 4, 3, 4, 3, 2, 3, 5, 2, 5, 4, 3, 4, 2, 1, 3, 3, 6, 3, 16,],
            [7, 12, 6, 4, 10, 10, 3, 4, 2, 4, 3, 2, 5, 7, 3, 2, 2, 3, 7, 5, 8, 3, 9, 1, 10, 8, 3, 9, 11, 3, 7, 6, 8, 8, 9, 4, 3, 7, 2, 4, 2, 4, 9, 7, 10, 12, 8, 6, 2, 6, 9, 5, 2, 3, 5, 7, 8, 9, 8, 4, 8, 11, 5, 9, 3, 1, 10, 6, 2, 5, 3, 11, 10, 9, 6, 3, 7, 8, 6, 6, 5, 6, 10, 5, 9, 2, 8, 7, 13, 3, 7, 10, 3, 3, 5, 7, 14, 6, 3, 4, 3, 4, 1, 2, 3, 5, 2, 4, 4, 3, 4, 2, 2, 5, 4, 5, 2, 14,],
            [8, 12, 4, 4, 9, 13, 3, 4, 2, 4, 3, 2, 5, 5, 3, 2, 2, 3, 7, 6, 7, 3, 9, 1, 10, 9, 3, 8, 10, 3, 7, 7, 6, 8, 9, 4, 2, 7, 2, 5, 2, 4, 11, 9, 8, 12, 9, 5, 2, 6, 9, 5, 2, 3, 5, 7, 8, 10, 6, 5, 9, 10, 5, 8, 3, 1, 8, 7, 2, 4, 3, 9, 11, 9, 6, 3, 8, 9, 7, 5, 5, 7, 9, 7, 10, 2, 9, 8, 13, 3, 7, 10, 3, 3, 5, 6, 11, 7, 4, 4, 3, 4, 3, 2, 3, 6, 2, 5, 4, 3, 4, 2, 2, 4, 5, 6, 3, 17,],
            [5, 7, 4, 3, 8, 9, 3, 4, 2, 3, 2, 2, 4, 4, 3, 2, 2, 3, 7, 4, 8, 3, 7, 1, 8, 10, 3, 7, 9, 3, 6, 5, 7, 6, 8, 4, 3, 6, 2, 4, 2, 4, 10, 6, 7, 13, 7, 5, 2, 6, 7, 5, 2, 3, 5, 6, 8, 8, 10, 4, 7, 12, 4, 8, 3, 1, 9, 5, 2, 5, 3, 9, 9, 8, 5, 3, 5, 9, 6, 6, 6, 6, 10, 6, 10, 2, 11, 8, 13, 3, 7, 9, 2, 3, 5, 6, 8, 4, 4, 4, 3, 5, 3, 2, 3, 5, 2, 5, 5, 3, 4, 2, 2, 4, 4, 6, 4, 14,],
            [7, 11, 6, 4, 8, 13, 3, 4, 2, 4, 3, 2, 5, 7, 3, 2, 2, 3, 7, 5, 5, 3, 9, 1, 10, 9, 3, 9, 10, 3, 8, 6, 9, 6, 8, 4, 3, 6, 2, 4, 2, 4, 9, 9, 9, 12, 8, 5, 2, 7, 9, 4, 2, 3, 5, 10, 8, 8, 8, 5, 10, 9, 5, 9, 3, 1, 10, 8, 2, 5, 3, 10, 9, 10, 6, 3, 7, 7, 6, 6, 6, 7, 8, 6, 8, 2, 7, 8, 12, 3, 7, 9, 3, 3, 5, 7, 14, 6, 4, 4, 3, 3, 3, 2, 3, 5, 2, 4, 3, 3, 4, 2, 2, 4, 5, 2, 4, 15,],
            [8, 12, 5, 4, 7, 12, 3, 4, 2, 4, 3, 2, 5, 6, 3, 2, 2, 3, 6, 5, 7, 3, 8, 1, 7, 8, 3, 8, 12, 3, 8, 8, 8, 8, 9, 3, 3, 7, 2, 5, 2, 4, 10, 8, 11, 11, 7, 5, 2, 7, 8, 5, 2, 3, 5, 8, 8, 8, 9, 5, 9, 12, 5, 11, 3, 1, 11, 8, 2, 5, 3, 11, 11, 8, 6, 3, 8, 5, 6, 6, 6, 6, 11, 6, 8, 2, 11, 8, 13, 3, 6, 11, 3, 3, 5, 6, 13, 7, 4, 4, 3, 4, 3, 2, 3, 5, 2, 4, 4, 3, 4, 2, 2, 4, 5, 5, 4, 13,],
            [8, 11, 5, 3, 10, 12, 3, 4, 2, 4, 3, 2, 5, 4, 3, 2, 2, 3, 7, 6, 8, 3, 6, 1, 9, 9, 3, 8, 11, 3, 7, 8, 9, 7, 10, 4, 3, 6, 2, 5, 2, 4, 11, 9, 9, 11, 9, 6, 2, 6, 8, 4, 2, 3, 5, 8, 7, 8, 8, 5, 9, 10, 5, 8, 3, 1, 8, 7, 2, 3, 3, 9, 7, 8, 6, 3, 6, 6, 7, 6, 5, 7, 10, 7, 7, 2, 7, 8, 12, 3, 7, 10, 3, 3, 5, 7, 14, 6, 4, 4, 3, 5, 3, 2, 3, 4, 2, 4, 5, 3, 4, 2, 2, 4, 4, 3, 4, 15,],
            [6, 12, 5, 4, 9, 11, 3, 3, 2, 4, 3, 2, 3, 6, 3, 2, 2, 3, 6, 6, 9, 3, 9, 1, 10, 10, 3, 9, 11, 3, 7, 7, 6, 8, 7, 4, 3, 6, 1, 4, 2, 3, 9, 7, 8, 9, 8, 5, 2, 7, 8, 5, 2, 3, 5, 8, 8, 10, 7, 5, 8, 10, 5, 8, 3, 1, 6, 6, 2, 4, 3, 11, 8, 11, 6, 3, 8, 8, 6, 6, 6, 5, 9, 7, 10, 2, 7, 8, 10, 3, 7, 8, 3, 2, 4, 7, 13, 5, 4, 4, 2, 4, 2, 2, 2, 4, 2, 4, 4, 3, 4, 2, 2, 6, 5, 4, 3, 13,],
            [6, 9, 4, 4, 8, 12, 3, 3, 2, 4, 3, 2, 5, 6, 3, 2, 2, 3, 7, 6, 6, 3, 7, 1, 10, 6, 3, 6, 9, 3, 7, 7, 5, 8, 7, 4, 3, 6, 2, 4, 2, 3, 10, 9, 7, 11, 7, 6, 2, 6, 6, 5, 2, 3, 5, 8, 8, 7, 9, 5, 7, 11, 4, 8, 3, 1, 8, 6, 2, 5, 3, 10, 7, 10, 6, 3, 8, 8, 6, 5, 5, 7, 9, 6, 9, 2, 9, 8, 11, 3, 6, 10, 3, 3, 5, 5, 12, 6, 5, 4, 3, 5, 3, 2, 2, 4, 2, 5, 5, 3, 3, 2, 2, 5, 4, 5, 2, 16,],
        ]
        self.verify_pscores(
                trees_fname="apternodus.tre",
                chars_fname="apternodus.chars.nexus",
                matrix_type=dendropy.StandardCharacterMatrix,
                gaps_as_missing=True,
                expected_scores=expected_scores,
                expected_per_site_scores=expected_per_site_scores)

    def test_pscores_with_dna_data_and_gaps_as_new_states(self):
        # #NEXUS
        # begin paup;
        #     set warnroot = no;
        #     exe apternodus.chars.nexus;
        #     gett file = apternodus.tre;
        #     set criterion = parsimony;
        #     pset gap = newstate;
        #     pscore;
        # end;
        expected_scores = [4703]
        # begin paup;
        #   set warnroot = no;
        #   execute pythonindae.chars.nexus;
        #   gett file = pythonidae.mle.tre;
        #   set crit=parsimony;
        #   pset gap = newstate;
        #   pscore / single = all scorefile=pythonidae.gaps-as-newstate.scores.txt;
        # end;
        expected_per_site_scores = [
            [0, 6, 5, 3, 3, 3, 6, 4, 2, 3, 3, 4, 4, 4, 4, 4, 4, 3, 3, 4, 11, 7,
                2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 3, 5, 10, 3,
                6, 0, 0, 1, 3, 1, 1, 3, 1, 3, 2, 4, 4, 4, 8, 7, 7, 5, 3, 2, 2,
                2, 5, 1, 0, 4, 3, 2, 2, 4, 5, 3, 6, 5, 7, 3, 10, 1, 2, 5, 10,
                4, 1, 6, 2, 8, 5, 5, 5, 1, 7, 2, 3, 5, 2, 2, 1, 7, 11, 3, 4, 3,
                2, 1, 1, 2, 1, 4, 10, 6, 10, 4, 10, 8, 7, 9, 11, 7, 5, 8, 3, 3,
                4, 9, 7, 5, 2, 3, 3, 5, 4, 4, 3, 3, 5, 1, 5, 1, 1, 0, 4, 0, 0,
                0, 0, 0, 0, 0, 0, 4, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 8, 1, 4, 9,
                8, 7, 9, 4, 3, 6, 8, 1, 2, 2, 7, 9, 7, 9, 5, 6, 1, 1, 1, 1, 1,
                2, 4, 9, 4, 5, 3, 6, 6, 6, 5, 10, 7, 9, 4, 7, 10, 13, 8, 4, 5,
                8, 9, 10, 12, 7, 9, 7, 6, 8, 8, 7, 5, 6, 6, 4, 7, 9, 8, 8, 8,
                6, 7, 2, 2, 6, 5, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 2, 0, 0,
                0, 0, 0, 0, 0, 1, 0, 0, 1, 3, 1, 0, 0, 0, 0, 2, 0, 0, 0, 0, 3,
                0, 0, 0, 4, 0, 1, 0, 2, 0, 4, 2, 0, 1, 0, 0, 0, 7, 0, 2, 4, 1,
                1, 5, 3, 0, 1, 0, 0, 0, 0, 0, 1, 0, 3, 4, 0, 0, 1, 1, 1, 6, 4,
                8, 1, 0, 1, 2, 1, 10, 3, 0, 0, 0, 4, 10, 5, 3, 5, 2, 5, 4, 9,
                9, 2, 4, 1, 1, 1, 7, 0, 0, 2, 0, 0, 0, 1, 5, 7, 7, 4, 2, 1, 0,
                0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 1, 1, 2, 2, 3, 11, 0, 1, 4, 4, 1,
                2, 3, 1, 3, 2, 9, 7, 1, 3, 0, 5, 0, 0, 1, 2, 6, 2, 5, 2, 6, 1,
                0, 0, 0, 8, 6, 0, 0, 0, 2, 0, 1, 1, 3, 4, 3, 1, 1, 0, 2, 1, 3,
                1, 6, 0, 0, 3, 1, 1, 5, 0, 3, 1, 4, 8, 4, 6, 0, 3, 6, 3, 5, 2,
                1, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 1, 0, 0, 0, 0,
                0, 0, 1, 3, 0, 3, 4, 3, 3, 2, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0,
                1, 0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 3, 3, 8,
                4, 2, 0, 3, 5, 2, 1, 3, 2, 0, 0, 0, 0, 4, 3, 2, 3, 7, 5, 5, 0,
                2, 3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0,
                0, 0, 1, 0, 2, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0,
                2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 1, 1,
                0, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0,
                0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 2, 0, 0,
                0, 1, 2, 0, 0, 5, 5, 2, 7, 7, 4, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0,
                0, 1, 0, 0, 0, 7, 3, 0, 0, 0, 0, 4, 6, 3, 5, 4, 1, 0, 2, 3, 0,
                0, 2, 3, 0, 0, 0, 0, 7, 5, 1, 5, 1, 0, 2, 0, 0, 0, 0, 5, 2, 1,
                0, 1, 4, 1, 3, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 3, 1, 0,
                0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 3, 1, 1, 1, 1, 1, 1, 1, 0, 0, 7,
                1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 4, 1, 0, 3, 2, 0, 6, 4, 0,
                0, 0, 1, 2, 7, 0, 3, 0, 0, 5, 0, 0, 3, 0, 0, 5, 2, 0, 4, 0, 0,
                6, 0, 0, 0, 0, 0, 2, 0, 0, 3, 0, 0, 3, 0, 0, 4, 0, 0, 1, 0, 0,
                2, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 0, 5, 0, 0, 3, 1, 0,
                0, 6, 0, 0, 6, 0, 1, 1, 0, 3, 0, 0, 11, 8, 1, 4, 2, 5, 8, 7, 0,
                2, 0, 0, 1, 2, 1, 3, 10, 1, 3, 0, 0, 6, 0, 0, 7, 0, 0, 3, 0, 2,
                5, 5, 0, 5, 1, 0, 11, 1, 0, 8, 0, 0, 4, 0, 0, 4, 0, 0, 1, 0, 0,
                2, 0, 0, 4, 0, 0, 11, 1, 0, 3, 1, 0, 6, 0, 0, 0, 0, 0, 8, 0, 1,
                2, 0, 0, 3, 0, 0, 7, 6, 0, 12, 0, 0, 3, 0, 0, 9, 2, 1, 3, 0, 0,
                4, 1, 0, 6, 0, 0, 2, 0, 0, 2, 2, 1, 7, 0, 0, 5, 0, 0, 2, 2, 1,
                2, 1, 0, 0, 0, 0, 1, 0, 0, 2, 4, 0, 9, 0, 0, 3, 0, 0, 8, 0, 0,
                1, 0, 0, 1, 0, 0, 1, 0, 0, 9, 2, 0, 0, 0, 0, 3, 0, 0, 5, 0, 0,
                10, 0, 0, 5, 1, 0, 10, 0, 0, 4, 0, 0, 11, 1, 0, 5, 0, 0, 7, 0,
                0, 2, 0, 0, 6, 0, 0, 7, 4, 0, 3, 0, 0, 1, 0, 0, 6, 0, 0, 6, 0,
                0, 10, 1, 0, 3, 2, 0, 13, 0, 0, 4, 0, 0, 0, 0, 0, 2, 0, 0, 5,
                0, 0, 2, 1, 0, 1, 0, 0, 12, 0, 0, 11, 0, 1, 11, 0, 0, 3, 1, 0,
            12, 0, 0, 6, 0, 0, 5, 0, 3, 2, 5, 0, 5, 0, 0, 3, 0, 0, 6, 0, 0, 7,
            0, 0, 7, 0, 0, 2, 0, 0, 5, 0, 0, 8, 0, 0, 9, 0, 0, 10, 0, 0, 6, 1,
            0, 1, 0, 0, 3, 0, 0, 2, 0, 0, 4, 0, 0, 3, 0, 0, 1, 0, 0, 1, 0, 0,
            1, 0, 0, 4, 0, 0, 1, 0, 0, 8, 0, 0, 2, 0, 0, 9, 0, 0, 3, 0, 0, 4,
            1, 0, 3, 2, 0, 9, 0, 0, 7, 0, 0, 8, 3, 0, 6, 0, 0, 10, 0, 0, 3, 2,
            0, 7, 0, 0, 7, 1, 0, 1, 2, 0, 8, 5, 0, 4, 0, 0, 2, 0, 0, 6, 0, 0,
            6, 7, 0, 9, 0, 0, 4, 0, 0, 7, 0, 0, 10, 0, 0, 6, 0, 0, 1, 0, 0, 2,
            0, 0, 9, 0, 0, 6, 0, 0, 10, 0, 0, 5, 3, 0, 9, 0, 0, 4, 0, 0, 3, 0,
            0, 5, 0, 0, 7, 0, 0, 4, 3, 0, 2, 0, 0, 3, 0, 0, 4, 0, 0, 6, 3, 0,
            3, 0, 0, 2, 0, 0, 1, 2, 4, 4, 0, 0, 8, 0, 0, 6, 0, 0, 10, 11, 0, 4,
            0, 0, 5, 0, 0, 1, 5, 0, 2, 0, 0, 3, 7, 0, 5, 0, 0, 8, 6, 0, 6, 0,
            0, 12, 0, 0, 6, 0, 0, 3, 0, 0, 5, 0, 0, 10, 0, 0, 10, 1, 0, 1, 0,
            0, 4, 0, 0, 3, 7, 0, 1, 0, 0, 3, 0, 0, 7, 0, 0, 3, 1, 0, 3, 0, 0,
            2, 0, 0, 5, 0, 0, 5, 0, 0, 0, 0, 0, 3, 0, 0, 3, 0, 0, 2, 0, 0, 4,
            0, 0, 14, 0, 0, 7, 0, 0, 1, 5, 0, 7, 7, 0, 2, 0, 0, 0, 0, 0, 6, 5,
            0, 12, 1, 0, 8, 7, 0, 3, 5, 0, 5, 1, 1, 3, 4, 0, 4, 1, 0, 1, 5, 3,
            9, 7, 1, 7, 7, 9, 9, 8, 0, 2, 7, 0, 7, 1, 1, 9, 2, 3, 5, 5, 2, 9,
            0, 1, 6, 0, 0, 3, 2, 1, 3, 0, 0, 5, 1, 0, 8, 0, 0, 11, 1, 0, 9, 1,
            0, 8, 0, 0, 2, 0, 0, 2, 0, 0, 6, 0, 0, 4, 0, 0, 1, 0, 0, 1, 0, 0,
            0, 0, 0, 8, 0, 0, 4, 0, 0, 11, 7, 0, 4, 0, 0, 5, 0, 0, 2, 0, 0, 2,
            0, 0, 4, 0, 0, 2, 0, 0, 8, 0, 0, 0, 0, 0, 1, 0, 0, 10, 0, 1, 8, 0,
            0, 3, 0, 0, 4, 2, 0, 2, 0, 0, 7, 0, 0, 2, 0, 0, 9, 0, 0, 11, 0, 0,
            8, 1, 0, 1, 0, 0, 1, 0, 0, 9, 0, 0, 6, 0, 1, 5, 0, 0, 8, 1, 0, 2,
            2, 0, 9, 0, 0, 4, 0, 0, 7, 0, 0, 4, 8, 0, 1, 0, 0, 6, 6, 0, 1, 2,
            1, 2, 1, 1, 0, 1, 0, 7, 0, 0, 6, 1, 0, 1, 0, 0, 8, 6, 0, 1, 2, 0,
            10, 2, 3, 8, 3, 6, 5, 0, 0, 2, 1, 0, 6, 0, 4, 0, 0, 0, 4, 0, 0, 1,
            2, 0, 3, 8, 5, 8, 4, 0, 10, 0, 0, 7, 2, 0, 11, 1, 0, 0, 0, 0, 6, 0,
            0, 3, 0, 0, 7, 0, 0, 8, 5, 0, 10, 2, 2, 2, 0, 0, 0, 4, 0, 6, 0, 2,
            2, 0, 0, 8, 0, 0, 0, 2, 1, 1, 7, 0, 2, 9, 0, 12, 3, 0, 2, 0, 0, 1,
            0, 0, 7, 4, 6, 3, 0, 1, 6, 0, 0, 8, 0, 0, 1, 0, 0, 6, 1, 1, 5, 0,
            0, 1, 0, 1, 0, 0, 0, 2, 0, 0, 5, 0, 0, 0, 0, 0, 2, 0, 0, 8, 0, 0,
            1, 0, 5, 9, 0, 1, 8, 7, 8, 8, 0, 0, 9, 0, 0, 3, 0, 0, 1, 3, 9, 2,
            1, 0, 10, 4, 0, 0, 5, 3, 5, 4, 1, 4, 0, 0, 4, 0, 0, 2, 1, 2, 4, 0,
            1, 3, 0, 0, 5, 6, 1, 5, 3, 1, 9, 1, 7, 3, 0, 0, 7, 5, 0, 9, 4, 2,
            6, 0, 0, 10, 0, 0, 5, 8, 4, 7, 0, 0, 1, 0, 0, 7, 0, 0, 0, 0, 0, 2,
            0, 2, 0, 2, 0, 6, 4, 7, 6, 7, 4, 6, 0, 2, 1, 6, 3, 3, 3, 0, 0, 7,
            5, 3, 1, 0, 5, 8, 3, 5, 3, 2, 8, 8, 2, 8, 10, 7, 1, 1, 0, 0, 1, 1,
            1, 1, 0, 0, 0, 0, 0, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 2, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 3, 0, 0, 0, 0, 0, 0, 0,
            1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 3, 0, 0, 1, 1, 0, 0, 0,
            1, 1, 2, 1, 0, 5, 0, 0, 0, 3, 10, 7, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0,
            0, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 2, 2, 0, 5, 2,
            1, 0, 8, 0, 0, 1, 6, 8, 3, 3, 2, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 1, 6, 0, 0, 0, 1,
            2, 1, 0, 0, 0, 3, 2, 7, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 1, 1, 3, 3, 0, 0, 0, 2, 6, 0, 6, 0, 0, 2, 0, 0,
            3, 0, 0, 2, 1, 3, 5, 2, 0, 0, 0, 0, 0, 7, 9, 8, 6, 1, 3, 2, 3, 5,
            6, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 2,
            2, 2, 3, 0, 4, 0, 0, 0, 5, 0, 0, 3, 1, 0, 2, 2, 0, 3, 0, 0, 0, 0,
            1, 0, 2, 2, 2, 3, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1,
            0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 1, 5, 3, 1, 0,
            0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            1, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 2,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 0, 0, 0, 2, 0, 0, 2,
            1, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 10, 1, 1, 0,
            0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 5, 3, 1, 0, 0, 0, 1,
            0, 0, 0, 0, 1, 0, 0, 1, 1, 3, 8, 1, 1, 1, 8, 9, 5, 3, 1, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 1, 0, 0, 1, 3, 0, 1, 2, 3, 0, 1, 1, 1, 2, 1, 0, 0, 0, 1, 0,
            12, 8, 7, 0, 0, 0, 1, 0, 9, 5, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 4,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 2, 7, 1, 1, 2, 0, 3, 3, 11, 4,
            0, 10, 3, 0, 1, 0, 1, 1, 3, 2, 9, 5, 6, 10, 2, 1, 0, 0, 1, 1, 1, 1,
            0, 0, 0, 0, 1, 0, 0, 0, 5, 7, 1, 7, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
            1, 1, 8, 0, 0, 0, 0, 0, 6, 2, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0,
            0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 3, 3, 1, 0,
            0, 0, 4, 0, 0, 0, 1, 2, 1, 0, 0, 0, 0, 0, 3, 0, 0, 1, 0, 0, 0, 1,
            0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
            0, 0, 0,],
        ]
        self.verify_pscores(
                trees_fname="pythonidae.mle.nex",
                chars_fname="pythonidae.chars.nexus",
                matrix_type=dendropy.DnaCharacterMatrix,
                gaps_as_missing=False,
                expected_scores=expected_scores,
                expected_per_site_scores=expected_per_site_scores)

    def test_pscores_with_dna_data_and_gaps_as_missing(self):
        # #NEXUS
        # begin paup;
        #     set warnroot = no;
        #     exe apternodus.chars.nexus;
        #     gett file = apternodus.tre;
        #     set criterion = parsimony;
        #     pset gap = newstate;
        #     pscore;
        # end;
        expected_scores = [4271]
        # begin paup;
        #   set warnroot = no;
        #   execute pythonindae.chars.nexus;
        #   gett file = pythonidae.mle.tre;
        #   set crit=parsimony;
        #   pset gap = missing;
        #   pscore / single = all scorefile=pythonidae.gaps-as-newstate.scores.txt;
        # end;
        expected_per_site_scores = [
            [0, 6, 3, 3, 3, 2, 4, 3, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 6, 1,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 2, 8, 2,
                4, 0, 0, 1, 3, 0, 0, 2, 0, 2, 1, 3, 3, 1, 2, 2, 2, 2, 2, 1, 0,
                1, 5, 1, 0, 4, 3, 2, 2, 4, 2, 0, 4, 5, 7, 3, 10, 1, 2, 5, 10,
                4, 1, 6, 1, 6, 3, 2, 4, 1, 7, 2, 3, 5, 2, 0, 0, 6, 10, 2, 3, 3,
                2, 0, 0, 1, 0, 3, 7, 3, 7, 2, 6, 6, 4, 5, 8, 4, 2, 6, 0, 0, 1,
                4, 3, 2, 0, 1, 1, 1, 1, 1, 2, 3, 4, 1, 1, 1, 0, 0, 4, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 7,
                7, 9, 4, 1, 3, 6, 0, 2, 2, 7, 9, 7, 8, 4, 2, 0, 0, 0, 0, 0, 0,
                4, 9, 4, 5, 3, 4, 3, 4, 3, 7, 5, 7, 2, 5, 7, 11, 5, 3, 4, 5, 8,
                9, 11, 6, 8, 7, 6, 8, 8, 7, 5, 6, 6, 4, 6, 8, 7, 7, 6, 4, 3, 1,
                1, 5, 5, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0,
                0, 0, 1, 0, 0, 1, 3, 1, 0, 0, 0, 0, 2, 0, 0, 0, 0, 3, 0, 0, 0,
                4, 0, 1, 0, 2, 0, 4, 2, 0, 1, 0, 0, 0, 7, 0, 2, 4, 1, 1, 5, 3,
                0, 1, 0, 0, 0, 0, 0, 1, 0, 3, 4, 0, 0, 1, 1, 1, 6, 4, 7, 1, 0,
                1, 2, 1, 10, 1, 0, 0, 0, 4, 10, 5, 3, 5, 2, 5, 4, 8, 9, 2, 3,
                0, 0, 0, 5, 0, 0, 2, 0, 0, 0, 1, 5, 7, 6, 2, 2, 1, 0, 0, 0, 0,
                0, 0, 1, 2, 0, 0, 0, 1, 1, 2, 2, 3, 11, 0, 1, 4, 4, 0, 0, 0, 0,
                3, 2, 9, 6, 0, 3, 0, 5, 0, 0, 0, 2, 6, 2, 5, 2, 6, 1, 0, 0, 0,
                8, 6, 0, 0, 0, 2, 0, 0, 0, 3, 4, 3, 1, 0, 0, 2, 1, 3, 1, 6, 0,
                0, 3, 0, 1, 5, 0, 3, 1, 4, 8, 4, 6, 0, 3, 6, 3, 5, 0, 0, 1, 0,
                0, 1, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                3, 0, 3, 4, 3, 2, 2, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1,
                1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 3, 3, 8, 4, 0, 0,
                3, 5, 2, 0, 3, 2, 0, 0, 0, 0, 4, 3, 1, 3, 7, 5, 3, 0, 2, 3, 4,
                0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1,
                0, 2, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 3, 0,
                0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0,
                0, 0, 1, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 2,
                0, 0, 5, 4, 2, 7, 7, 4, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0,
                0, 0, 7, 3, 0, 0, 0, 0, 4, 6, 3, 5, 4, 0, 0, 2, 3, 0, 0, 2, 3,
                0, 0, 0, 0, 7, 5, 1, 5, 1, 0, 2, 0, 0, 0, 0, 0, 0, 1, 0, 1, 3,
                0, 3, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 3, 1, 0, 0, 0, 0,
                0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 1, 0, 0,
                0, 0, 0, 1, 0, 0, 0, 0, 0, 4, 1, 0, 3, 2, 0, 6, 4, 0, 0, 0, 1,
                2, 7, 0, 3, 0, 0, 5, 0, 0, 3, 0, 0, 5, 2, 0, 4, 0, 0, 6, 0, 0,
                0, 0, 0, 2, 0, 0, 3, 0, 0, 3, 0, 0, 4, 0, 0, 1, 0, 0, 2, 0, 0,
                0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 0, 5, 0, 0, 3, 1, 0, 0, 6, 0,
                0, 6, 0, 1, 1, 0, 3, 0, 0, 11, 8, 1, 4, 2, 5, 8, 7, 0, 2, 0, 0,
                1, 2, 1, 3, 10, 1, 3, 0, 0, 6, 0, 0, 7, 0, 0, 3, 0, 2, 5, 5, 0,
                5, 1, 0, 11, 1, 0, 8, 0, 0, 4, 0, 0, 4, 0, 0, 1, 0, 0, 2, 0, 0,
                4, 0, 0, 11, 1, 0, 3, 1, 0, 6, 0, 0, 0, 0, 0, 8, 0, 1, 2, 0, 0,
                3, 0, 0, 7, 6, 0, 12, 0, 0, 3, 0, 0, 9, 2, 1, 3, 0, 0, 4, 1, 0,
                6, 0, 0, 2, 0, 0, 2, 2, 1, 7, 0, 0, 5, 0, 0, 1, 2, 1, 2, 1, 0,
                0, 0, 0, 1, 0, 0, 2, 4, 0, 9, 0, 0, 3, 0, 0, 8, 0, 0, 1, 0, 0,
                1, 0, 0, 1, 0, 0, 9, 2, 0, 0, 0, 0, 3, 0, 0, 5, 0, 0, 10, 0, 0,
                5, 1, 0, 10, 0, 0, 4, 0, 0, 11, 1, 0, 5, 0, 0, 7, 0, 0, 2, 0,
                0, 6, 0, 0, 7, 4, 0, 3, 0, 0, 1, 0, 0, 6, 0, 0, 6, 0, 0, 10, 1,
                0, 3, 2, 0, 13, 0, 0, 4, 0, 0, 0, 0, 0, 2, 0, 0, 5, 0, 0, 2, 1,
                0, 1, 0, 0, 12, 0, 0, 11, 0, 1, 11, 0, 0, 3, 1, 0, 12, 0, 0, 6,
            0, 0, 5, 0, 3, 2, 5, 0, 5, 0, 0, 3, 0, 0, 6, 0, 0, 7, 0, 0, 7, 0,
            0, 2, 0, 0, 5, 0, 0, 8, 0, 0, 9, 0, 0, 10, 0, 0, 6, 1, 0, 1, 0, 0,
            3, 0, 0, 2, 0, 0, 4, 0, 0, 3, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 4,
            0, 0, 1, 0, 0, 8, 0, 0, 2, 0, 0, 9, 0, 0, 3, 0, 0, 4, 1, 0, 3, 2,
            0, 9, 0, 0, 7, 0, 0, 8, 3, 0, 6, 0, 0, 10, 0, 0, 3, 2, 0, 7, 0, 0,
            7, 1, 0, 1, 2, 0, 8, 5, 0, 4, 0, 0, 2, 0, 0, 6, 0, 0, 6, 7, 0, 9,
            0, 0, 4, 0, 0, 7, 0, 0, 10, 0, 0, 6, 0, 0, 1, 0, 0, 2, 0, 0, 9, 0,
            0, 6, 0, 0, 10, 0, 0, 5, 3, 0, 9, 0, 0, 4, 0, 0, 3, 0, 0, 5, 0, 0,
            7, 0, 0, 4, 3, 0, 2, 0, 0, 3, 0, 0, 4, 0, 0, 6, 3, 0, 3, 0, 0, 2,
            0, 0, 1, 2, 4, 4, 0, 0, 8, 0, 0, 6, 0, 0, 10, 11, 0, 4, 0, 0, 5, 0,
            0, 1, 5, 0, 2, 0, 0, 3, 7, 0, 5, 0, 0, 8, 6, 0, 6, 0, 0, 12, 0, 0,
            6, 0, 0, 3, 0, 0, 5, 0, 0, 10, 0, 0, 10, 1, 0, 1, 0, 0, 4, 0, 0, 3,
            7, 0, 1, 0, 0, 3, 0, 0, 7, 0, 0, 3, 1, 0, 3, 0, 0, 2, 0, 0, 5, 0,
            0, 5, 0, 0, 0, 0, 0, 3, 0, 0, 3, 0, 0, 2, 0, 0, 4, 0, 0, 14, 0, 0,
            7, 0, 0, 1, 5, 0, 7, 7, 0, 2, 0, 0, 0, 0, 0, 6, 5, 0, 12, 1, 0, 8,
            7, 0, 3, 5, 0, 5, 1, 1, 3, 4, 0, 4, 1, 0, 1, 5, 3, 9, 7, 1, 7, 7,
            9, 9, 8, 0, 2, 7, 0, 7, 1, 1, 9, 2, 3, 5, 5, 2, 9, 0, 1, 6, 0, 0,
            3, 2, 1, 3, 0, 0, 5, 1, 0, 8, 0, 0, 11, 1, 0, 9, 1, 0, 8, 0, 0, 2,
            0, 0, 2, 0, 0, 6, 0, 0, 4, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 8, 0,
            0, 4, 0, 0, 11, 7, 0, 4, 0, 0, 5, 0, 0, 2, 0, 0, 2, 0, 0, 4, 0, 0,
            2, 0, 0, 8, 0, 0, 0, 0, 0, 1, 0, 0, 10, 0, 1, 8, 0, 0, 3, 0, 0, 4,
            2, 0, 2, 0, 0, 7, 0, 0, 2, 0, 0, 9, 0, 0, 11, 0, 0, 8, 1, 0, 1, 0,
            0, 1, 0, 0, 9, 0, 0, 6, 0, 1, 5, 0, 0, 8, 1, 0, 2, 2, 0, 9, 0, 0,
            4, 0, 0, 7, 0, 0, 4, 8, 0, 1, 0, 0, 6, 6, 0, 1, 2, 1, 2, 1, 1, 0,
            1, 0, 7, 0, 0, 6, 1, 0, 1, 0, 0, 8, 6, 0, 1, 2, 0, 10, 2, 3, 8, 3,
            6, 5, 0, 0, 2, 1, 0, 6, 0, 4, 0, 0, 0, 4, 0, 0, 1, 2, 0, 3, 8, 5,
            8, 4, 0, 10, 0, 0, 7, 2, 0, 11, 1, 0, 0, 0, 0, 6, 0, 0, 3, 0, 0, 7,
            0, 0, 8, 5, 0, 10, 2, 2, 2, 0, 0, 0, 4, 0, 6, 0, 2, 2, 0, 0, 8, 0,
            0, 0, 2, 1, 1, 7, 0, 2, 9, 0, 12, 3, 0, 2, 0, 0, 1, 0, 0, 7, 4, 6,
            3, 0, 1, 6, 0, 0, 8, 0, 0, 1, 0, 0, 6, 1, 1, 5, 0, 0, 1, 0, 1, 0,
            0, 0, 2, 0, 0, 5, 0, 0, 0, 0, 0, 2, 0, 0, 8, 0, 0, 1, 0, 5, 9, 0,
            1, 8, 7, 8, 8, 0, 0, 9, 0, 0, 3, 0, 0, 1, 3, 9, 2, 1, 0, 10, 4, 0,
            0, 5, 3, 5, 4, 1, 4, 0, 0, 4, 0, 0, 2, 1, 2, 4, 0, 1, 3, 0, 0, 5,
            6, 1, 5, 3, 1, 9, 1, 7, 3, 0, 0, 7, 5, 0, 9, 4, 2, 6, 0, 0, 10, 0,
            0, 5, 8, 4, 7, 0, 0, 1, 0, 0, 7, 0, 0, 0, 0, 0, 2, 0, 2, 0, 2, 0,
            6, 4, 7, 6, 7, 4, 6, 0, 2, 1, 5, 3, 3, 3, 0, 0, 7, 5, 3, 1, 0, 5,
            8, 3, 5, 3, 2, 6, 2, 0, 1, 5, 6, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 2, 0, 3, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0,
            0, 0, 0, 0, 1, 0, 1, 0, 0, 3, 0, 0, 1, 1, 0, 0, 0, 1, 1, 2, 1, 0,
            5, 0, 0, 0, 3, 8, 7, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0,
            0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1,
            0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 2, 2, 0, 5, 2, 1, 0, 8, 0, 0,
            0, 6, 8, 3, 2, 1, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 1, 6, 0, 0, 0, 1, 2, 1, 0, 0, 0,
            3, 2, 7, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 1, 1, 3, 3, 0, 0, 0, 2, 6, 0, 6, 0, 0, 2, 0, 0, 3, 0, 0, 2, 1,
            3, 5, 2, 0, 0, 0, 0, 0, 7, 9, 8, 6, 1, 3, 2, 3, 5, 6, 0, 0, 0, 0,
            4, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 2, 2, 2, 3, 0, 4,
            0, 0, 0, 5, 0, 0, 3, 1, 0, 2, 2, 0, 3, 0, 0, 0, 0, 1, 0, 2, 1, 1,
            3, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 3, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 0, 5, 3, 1, 0, 0, 0, 1, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0,
            0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 0, 0, 0, 2, 0, 0, 2, 1, 2, 1, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 10, 1, 1, 0, 0, 0, 0, 0, 1,
            0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 5, 3, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1,
            0, 0, 1, 1, 3, 8, 1, 1, 1, 8, 9, 5, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
            1, 3, 0, 1, 2, 3, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 12, 8, 7, 0, 0,
            0, 1, 0, 9, 5, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 4, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 5, 0, 2, 5, 1, 1, 2, 0, 3, 3, 11, 4, 0, 10, 3, 0, 0,
            0, 1, 1, 3, 2, 9, 5, 6, 8, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1,
            0, 0, 0, 5, 7, 1, 7, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 8, 0, 0,
            0, 0, 0, 6, 2, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
            0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 3, 3, 1, 0, 0, 0, 4, 0, 0,
            0, 1, 2, 1, 0, 0, 0, 0, 0, 3, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0,
            0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,]
        ]
        self.verify_pscores(
                trees_fname="pythonidae.mle.nex",
                chars_fname="pythonidae.chars.nexus",
                matrix_type=dendropy.DnaCharacterMatrix,
                gaps_as_missing=True,
                expected_scores=expected_scores,
                expected_per_site_scores=expected_per_site_scores)



    def verify_pscores(self,
            trees_fname,
            chars_fname,
            matrix_type,
            gaps_as_missing,
            expected_scores,
            expected_per_site_scores):
        taxon_namespace = dendropy.TaxonNamespace()
        chars = matrix_type.get(
                path=pathmap.char_source_path(chars_fname),
                schema="nexus",
                taxon_namespace=taxon_namespace)
        trees = dendropy.TreeList.get(
                path=pathmap.tree_source_path(trees_fname),
                schema="nexus",
                taxon_namespace=taxon_namespace)
        self.assertEqual(len(expected_scores), len(trees))
        for tree_idx, tree in enumerate(trees):
            score_by_character_list = []
            pscore = treescore.parsimony_score(
                    tree,
                    chars,
                    gaps_as_missing=gaps_as_missing,
                    score_by_character_list=score_by_character_list)
            self.assertEqual(pscore, expected_scores[tree_idx])
            self.assertEqual(len(score_by_character_list), len(expected_per_site_scores[tree_idx]))
            for obs, exp in zip(score_by_character_list, expected_per_site_scores[tree_idx]):
                self.assertEqual(obs, exp)
            self.assertEqual(sum(score_by_character_list), pscore)

            # just to be sure it works without passing in `score_by_character_list`:
            pscore = treescore.parsimony_score(
                    tree,
                    chars,
                    gaps_as_missing=gaps_as_missing)
            self.assertEqual(pscore, expected_scores[tree_idx])

if __name__ == "__main__":
    unittest.main()


