#! /usr/bin/env python
# -*- coding: utf-8 -*-

##############################################################################
##  DendroPy Phylogenetic Computing Library.
##
##  Copyright 2010-2015 Jeet Sukumaran and Mark T. Holder.
##  All rights reserved.
##
##  See "LICENSE.rst" for terms and conditions of usage.
##
##  If you use this work or any portion thereof in published work,
##  please cite it as:
##
##     Sukumaran, J. and M. T. Holder. 2010. DendroPy: a Python library
##     for phylogenetic computing. Bioinformatics 26: 1569-1571.
##
##############################################################################

"""
Tests statistical routines.
"""

import unittest
import os
import sys
sys.path.insert(0, os.path.dirname(__file__))
from support import dendropytest
from dendropy.calculate import statistics
from dendropy.utility import messaging

_LOG = messaging.get_logger(__name__)

class HpdCalculationTests(dendropytest.ExtendedTestCase):
    """
    Test reference values as given by ``hpd.emp``, in the "TeachingDemos" package
    of R.
    """

    def testEmpiricalUnimodalHpd1(self):
        # Normal[mu=100, sd=1)
        v = [100.93356, 100.66576, 99.44097, 100.60761, 103.65723, 101.15563,
                99.09657, 100.39654, 98.77339, 101.13712, 99.33979, 99.99060,
                100.39395, 101.68240, 100.99664, 99.17798, 100.83020, 98.90373,
                100.30441, 99.49553, 100.52652, 99.76291, 99.95605, 99.63605,
                99.21535, 100.51619, 100.55036, 101.21747, 101.04181, 97.76084,
                100.19069, 99.46182, 100.47579, 99.56889, 100.23977, 101.22907,
                97.85931, 100.86051, 99.56121, 100.44109, 100.02328, 98.62446,
                100.11008, 100.12700, 99.27087, 100.72895, 99.06796, 99.38019,
                99.79908, 100.82761, 101.26901, 99.88911, 98.09761, 99.16706,
                98.98752, 100.10088, 100.58883, 99.42982, 101.90322, 101.22817,
                101.36052, 97.70629, 100.15950, 99.39458, 100.19414, 103.43317,
                100.32429, 98.90429, 101.28049, 99.82948, 100.96041, 99.46024,
                98.22509, 101.63878, 100.66998, 101.82238, 99.49847, 100.41055,
                98.71792, 99.66001, 98.53177, 99.11997, 100.14802, 98.96423,
                101.93145, 100.09478, 100.85930, 99.82181, 101.50284, 99.93301,
                99.57168, 98.19978, 100.90708, 99.25086, 101.74170, 99.86034,
                99.85785, 99.89154, 99.62313, 99.41994,]
        #==> 101.82238 97.70629
        #==> 101.90322 97.76084
        #==> 101.93145 97.85931
        #==> 103.43317 98.09761
        #==> 103.65723 98.19978
        #n= 100
        #nn= 5
        #xx= [4.11609, 4.142380000000003, 4.0721400000000045, 5.335560000000001, 5.457449999999994]
        #m= 4.07214
        #nnn= 100
        #(97.85931, 101.93145)
        c1, c2 = statistics.empirical_hpd(v)
        self.assertAlmostEqual(c1, 97.85931)
        self.assertAlmostEqual(c2, 101.93145)

    def testEmpiricalUnimodalHpd2(self):
        # Exp(rate=0.2)
        v = [3.27592276968062, 0.471030483022332, 8.69292160732502,
                5.31655522508031, 11.6689748180798, 3.74156305545426,
                0.930466635618359, 4.02394197251564, 0.0273082678142286,
                2.19627505168319, 11.2686246344702, 3.12780772801489,
                16.3526409110966, 1.03131624741206, 4.43873460812746,
                1.16054141893983, 1.37002475326881, 4.03690286425358,
                2.75003841612488, 0.247246073558927, 2.97294339863583,
                9.91361622656596, 1.40643152873963, 5.06202565485096,
                2.56700876867399, 5.10566710939115, 8.30197051456789,
                0.439280721062038, 11.3532735680448, 1.46181986900046,
                11.1246174474465, 2.24797004368156, 1.79919427493587,
                8.79207140509944, 4.81857897692776, 2.30751369846985,
                0.589188064119702, 3.36240844568238, 9.85515167894673,
                13.7341997859286, 3.04674943210557, 10.2497380129517,
                15.3365677214208, 0.322058985475451, 2.13952575810254,
                8.7431202231924, 9.48776975232077, 0.437449288806399,
                2.91444693226367, 0.234506344422698, 2.30315598892048,
                11.9319818628238, 1.30209191970236, 1.34823656175286,
                25.9922393489827, 9.88916845991366, 4.32954248951232,
                0.748464160133153, 1.30975685780868, 10.16635726164,
                12.2592059050905, 0.469188864149385, 1.23079363489524,
                40.8792947675279, 5.14233190545297, 6.33412759730077,
                4.14186116397752, 0.811017339583486, 2.73471124237403,
                9.42033216315222, 6.48358419050878, 3.18536503706127,
                3.99172384842717, 0.936779121402651, 9.05760801355255,
                5.50938969922668, 1.06717714807019, 8.42135253320348,
                3.84890870635814, 0.382886157387499, 6.31485156693912,
                0.300180648919195, 0.8748722959183, 1.82131360052153,
                3.14994857879356, 0.281196870910665, 7.2329476564647,
                2.68667792435735, 0.364864277653396, 0.757411941885948,
                5.50616672241545, 4.88404127282506, 0.167293788399547,
                8.03142971525326, 10.5768193447809, 14.8404745685177,
                1.30770040210336, 25.8265917826642, 1.59898946760222,
                4.81477369067675,]
        #  n= 100
        #  nn= 5
        #  P1= 15.33657 16.35264 25.82659 25.99224 40.87929
        #  P2= 0.02730827 0.1672938 0.2345063 0.2472461 0.2811969
        #  xx= 15.30926 16.18535 25.59209 25.74499 40.5981
        #  m= 15.30926
        #  nnn= 1
        #  FINAL =  0.02730827 15.33657
        #  [1]  0.02730827 15.33656772
        c1, c2 = statistics.empirical_hpd(v)
        self.assertAlmostEqual(c1, 0.02730827)
        self.assertAlmostEqual(c2, 15.33656772)

    def testEmpiricalUnimodalHpd3(self):
        # Gamma(shape=5.5, rate=100)
        v = [0.0380707918890542, 0.0613501755118962, 0.0349990175822757,
                0.0619137725677135, 0.07861308334536, 0.0480553266749484,
                0.062862553519841, 0.0682992193072175, 0.0498105967259703,
                0.0391473475922808, 0.0814239076471267, 0.0622768940091115,
                0.0278766690168117, 0.0317487156549716, 0.0586667309884353,
                0.0610901783693829, 0.0614272435233303, 0.0467353725415623,
                0.0235002000890242, 0.085772673901534, 0.0556214310452847,
                0.0670344572022403, 0.0544400125583679, 0.0538488966212504,
                0.05294861204774, 0.0487265157882631, 0.0319744057916896,
                0.0727994868817337, 0.0406117459082881, 0.0411118268978343,
                0.0500761307970585, 0.0308194535724841, 0.0657552493534245,
                0.100523723943228, 0.0465808929090943, 0.0460932186699443,
                0.0561270845816648, 0.0600098984413655, 0.039155736440776,
                0.0477087458173384, 0.0662527620357706, 0.0259218058062224,
                0.0240999453565313, 0.0403149976175261, 0.0397058239610132,
                0.080726331847454, 0.084357840225122, 0.0469807175107997,
                0.0629060978549567, 0.0815708340371883, 0.0662480716451838,
                0.0291424513010887, 0.0423492520899737, 0.0400760974537379,
                0.0988931209604346, 0.0334625360347498, 0.0481980311926021,
                0.0326792585090408, 0.0454491423001323, 0.020993064627905,
                0.0435735408306696, 0.0408747071998941, 0.0152619235644154,
                0.0749659776042904, 0.0568986969556779, 0.0238240850033704,
                0.0546832244279347, 0.0793788099421741, 0.0366737460311167,
                0.0122826115173667, 0.0542719395504513, 0.0426583849776426,
                0.0211571623626521, 0.097984660746214, 0.0909562231889738,
                0.0317473196033018, 0.0683970866878872, 0.0249627875602813,
                0.081633395263259, 0.050187713841904, 0.09452301382497,
                0.0832417097555666, 0.0784842034909532, 0.0329463277742707,
                0.134071786835307, 0.0672633924985841, 0.0492264776710421,
                0.0346193998786818, 0.0608703914888914, 0.0479141586724897,
                0.0653849788769291, 0.0363831431722953, 0.0978132293790966,
                0.0663011935255327, 0.0245590323775349, 0.0438722027031532,
                0.0294721189654155, 0.0482169372325583, 0.0372503987850244,
                0.049380448859450]
        #  n= 100
        #  nn= 5
        #  P1= 0.09781323 0.09798466 0.09889312 0.1005237 0.1340718
        #  P2= 0.01228261 0.01526192 0.02099306 0.02115716 0.0235002
        #  xx= 0.08553062 0.08272274 0.07790006 0.07936656 0.1105716
        #  m= 0.07790006
        #  nnn= 3
        #  FINAL =  0.02099306 0.09889312
        #  [1] 0.02099306 0.09889312
        c1, c2 = statistics.empirical_hpd(v)
        self.assertAlmostEqual(c1, 0.02099306)
        self.assertAlmostEqual(c2, 0.09889312)

class TestMedian(unittest.TestCase):
    """
    Test from: http://wiki.python.org/moin/SimplePrograms.
    """

    def testMedian(self):
        self.assertEqual(statistics.median([2, 9, 9, 7, 9, 2, 4, 5, 8]), 7)

class TestVarianceCovariance(unittest.TestCase):

    def setUp(self):
        # vlen = 30
        # ndim = 5
        # data = []
        # data.append([random.gauss(random.randint(0, 1000), 1000) for i in range(vlen)])
        # for dim in range(ndim-1):
        #     u = random.uniform(0, 1)
        #     if u > 0.8:
        #         data.append([x * 3 for x in data[0]])
        #     elif u > 0.6:
        #         data.append([x + random.gauss(0, x/100) for x in data[0]])
        #     elif u > 0.4:
        #         data.append([x + random.gauss(0, x/10) for x in data[0]])
        #     elif u > 0.2:
        #         data.append([random.gauss(0, random.randint(1, 100)) for x in data[0]])
        # print data
        self.data = [[-387.0725787021329, 931.5293654488856, 546.94301742877,
            -1308.9393303633315, 1811.352775963276, 451.3538641126903,
            562.9659558111259, 2279.483383601514, -908.0885971464554,
            -650.165321672863, -426.0293691171515, 1109.1183877724106,
            35.16260780005632, 1453.566295339867, 2838.575344737274,
            370.1667315279002, -853.2876360550472, 1171.341723677777,
            556.5194253749042, 275.59547544853393, 1936.6411793187876,
            -146.60284069858278, 227.34032009264192, 294.9677652833037,
            -230.43349352689938, 2272.2745879401054, -522.4676539536883,
            880.3216676214797, -364.3858002091546, 4.674611872631715],
            [-388.4042113486994, 927.1172167918467, 538.8506889384198,
                -1298.3326898912524, 1813.4425528719585, 449.9792508848036,
                566.6065457390224, 2275.5404727189675, -911.1644303228873,
                -637.7793275095457, -426.24850592012865, 1116.7861772441804,
                34.677032164373, 1435.260152533119, 2859.8928263857206,
                367.6125468449959, -864.1297741225864, 1170.321476247361,
                554.8054179855712, 271.657749074347, 1921.1532376931864,
                -145.65499088854472, 226.57461475353222, 298.7472447593899,
                -231.0886348421088, 2292.785060285147, -528.8141997687727,
                880.8322517812744, -367.48567832266133, 4.599758786967388],
            [-1161.2177361063987, 2794.588096346657, 1640.82905228631,
                -3926.8179910899944, 5434.0583278898275, 1354.061592338071,
                1688.8978674333778, 6838.450150804542, -2724.2657914393662,
                -1950.495965018589, -1278.0881073514545, 3327.355163317232,
                105.48782340016896, 4360.698886019602, 8515.726034211822,
                1110.5001945837007, -2559.8629081651416, 3514.0251710333314,
                1669.5582761247128, 826.7864263456017, 5809.923537956363,
                -439.80852209574834, 682.0209602779257, 884.9032958499112,
                -691.3004805806981, 6816.823763820316, -1567.402961861065,
                2640.965002864439, -1093.1574006274636, 14.023835617895145],
            [-389.0970757367656, 930.0845285660921, 546.6471157339992,
                -1325.4845049714904, 1807.0225831671107, 451.97595156242477,
                563.1808453852201, 2275.359502238535, -910.6620730299588,
                -639.8911116399893, -430.4296686299347, 1114.0085423914036,
                35.23322927194568, 1445.7289312962391, 2869.2649737548804,
                369.61107326701136, -847.7524690958505, 1180.5581555467884,
                558.6182051869164, 270.71734809287057, 1953.153593413107,
                -148.16228159208333, 228.94407431058426, 296.62535613622003,
                -232.98333550881344, 2267.1784698610254, -515.4738926823279,
                874.9725307590513, -360.1574033066678, 4.7265309567945994]]

        # import numpy
        # expected_cov = numpy.cov(numpy.array(data),rowvar=0).tolist()
        # print expected_cov
        self.expected_cov = [[149393.17790377847, -360424.96906892554, -211940.40550363238,
            505544.04928701016, -700254.966766187, -174502.80419746446,
            -217345.74407933108, -881568.5997736979, 350625.3243410929,
            252753.2580470269, 164369.7501328118, -427881.97570378595,
            -13617.530634341638, -563503.0643466829, -1093807.4471157312,
            -143274.32024934815, 329464.18420184636, -452213.78708996583,
            -215077.8426113635, -107087.45068333478, -748475.3164486786,
            56624.942130779986, -87816.4837208772, -113659.46065397991,
            88859.94740941831, -877271.1179979955, 201980.68894789586,
            -340566.08097309095, 140911.57157102908, -1808.2870504090968],
            [-360424.96906892554, 869569.9362145537, 511337.65185982996,
                -1219681.7066146561, 1689442.2338632138, 421008.1622157625,
                524366.7087900256, 2126887.1168150404, -845917.5496523898,
                -609812.9697163638, -396557.36555995565, 1032301.8444213009,
                32854.22747731646, 1359535.4531227027, 2638892.0982596218,
                345667.84863406006, -794860.8161464253, 1091011.1989148206,
                518899.4015696987, 258366.84703955788, 1805784.8802174286,
                -136613.94859184456, 211866.75720555347, 274211.2373116087,
                -214382.2903332192, 2116495.277061747, -487298.17819055147,
                821655.4956393185, -339963.90830350405, 4362.735925535438],
            [-211940.40550363238, 511337.65185982996, 300691.95320246654,
                -717234.9644576052, 993436.5475125475, 247567.57305548087,
                308338.74556175485, 1250675.074863846, -497422.68334818585,
                -358596.5629816347, -233190.50326784851, 607018.0135966156,
                19319.94031851559, 799465.1231295253, 1551742.404717372,
                203265.87160003383, -467383.8843663716, 641555.1887777617,
                305132.059903797, 151928.88274220875, 1061887.3462284554,
                -80335.43600964258, 124586.13005423953, 161240.75009037546,
                -126064.20956241484, 1244532.2694496606, -286532.5812771456,
                483153.3495236603, -199901.83344425834, 2565.5529679555852],
            [505544.04928701016, -1219681.7066146561, -717234.9644576052,
                1710856.9192995324, -2369616.693280398, -590520.9262912686,
                -735476.3899937997, -2983204.334641759, 1186502.1830001853,
                855325.270463226, 556238.3789052436, -1447924.970738861,
                -46083.75449593974, -1906929.2316904257, -3701435.8716964982,
                -484845.3566280408, 1114827.964160791, -1530318.2381011278,
                -727832.8495215549, -362379.6028049886, -2532950.829680498,
                191627.34944563507, -297177.9863681518, -384609.91910996195,
                300706.3619029368, -2968549.589068031, 683441.8508461871,
                -1152443.0012280906, 476810.7250318338, -6119.722146225432],
            [-700254.966766187, 1689442.2338632138, 993436.5475125475,
                -2369616.693280398, 3282358.894459347, 817954.7970777344,
                1018779.1665524077, 4132236.72058621, -1643501.8847934983,
                -1184764.4420874363, -770448.6531432323, 2005628.3012952602,
                63829.770800075996, 2641350.3080521864, 5127000.986694892,
                671578.037266826, -1544337.948133165, 2119663.6178804073,
                1008140.5005908067, 501967.62658711313, 3508319.811483866,
                -265416.7728446663, 411622.71723057824, 532760.6687731618,
                -416512.20772471745, 4112110.4047745992, -946778.110294905,
                1596369.7050428558, -660516.0362065202, 8475.908935267644],
            [-174502.80419746446, 421008.1622157625, 247567.57305548087,
                    -590520.9262912686, 817954.7970777344, 203834.24078224797,
                    253876.05452768996, 1029746.8036606665, -409557.69626571395,
                    -295242.9268632595, -191997.03665307324, 499797.5854784194,
                    15906.615340342569, 658226.857278107, 1277645.5563866766,
                    167357.41134353352, -384836.6886719451, 528222.8533432522,
                    251229.2922489141, 125089.06639656228, 874286.8934292952,
                    -66142.93440720538, 102577.1383801494, 132761.78158270588,
                    -103795.28152140026, 1024715.8636411378, -235927.61123554132,
                    397809.33378739585, -164595.1661642016, 2112.2592771769364],
            [-217345.74407933108, 524366.7087900256, 308338.74556175485,
                    -735476.3899937997, 1018779.1665524077, 253876.05452768996,
                    316210.3607986867, 1282561.1913416046, -510111.4859997901,
                    -367720.8262720667, -239131.93911860665, 622510.5342108728,
                    19811.239619897766, 819812.9135390642, 1591328.923455987,
                    208443.05518058766, -479336.6991379133, 657900.5923021627,
                    312905.72644413053, 155798.6535323029, 1088904.5405440978,
                    -82379.5850977837, 127759.0605202802, 165360.08034015898,
                    -129277.37186774613, 1276326.3970300478, -293863.91038333974,
                    495481.2331138214, -205012.36107524062, 2630.7156238501116],
            [-881568.5997736979, 2126887.1168150404, 1250675.074863846,
                    -2983204.334641759, 4132236.72058621, 1029746.8036606665,
                    1282561.1913416046, 5202179.297296101, -2069043.7281889115,
                    -1491539.2678336531, -969941.4300628213, 2524928.8110790555,
                    80357.76413641141, 3325284.0270826635, 6454508.230033948,
                    845470.0928448668, -1944182.937873276, 2668509.1611595503,
                    1269178.4498434912, 631941.0554721548, 4416754.270064828,
                    -334143.0030964492, 518205.1120158168, 670701.4840533712,
                    -524359.3000358283, 5176801.443597403, -1191905.4989193215,
                    2009703.8383549566, -831531.0005204134, 10670.709481456897],
            [350625.3243410929, -845917.5496523898, -497422.68334818585,
                    1186502.1830001853, -1643501.8847934983, -409557.69626571395,
                    -510111.4859997901, -2069043.7281889115, 822917.5775059358,
                    593214.284884317, 385773.9652874163, -1004238.9922932169,
                    -31960.216388136178, -1322541.9562906092, -2567157.759426228,
                    -336264.6943500905, 773258.0277370816, -1061341.708613389,
                    -504786.57990451134, -251335.81249763246, -1756661.0753439257,
                    132897.84866975705, -206104.45876818095, -266758.9838720577,
                    208553.34492990456, -2058963.6467483111, 474053.7557940749,
                    -799311.7184875482, 330722.3835950521, -4244.009956370862],
            [252753.2580470269, -609812.9697163638, -358596.5629816347,
                    855325.270463226, -1184764.4420874363, -295242.9268632595,
                    -367720.8262720667, -1491539.2678336531, 593214.284884317,
                    427669.4388326606, 278090.68951873126, -723913.6401645798,
                    -23040.250998091913, -953434.0691506112, -1850533.255967261,
                    -242411.14225233067, 557409.7886478706, -765089.5555305473,
                    -363890.3596168801, -181195.07273396023, -1266349.3650786553,
                    95803.21315757626, -148575.82840293963, -192291.63727863415,
                    150337.46468169338, -1484227.4060228057, 341730.2139528351,
                    -576210.7159965835, 238409.02615910847, -3059.4982172104096],
            [164369.7501328118, -396557.36555995565, -233190.50326784851,
                    556238.3789052436, -770448.6531432323, -191997.03665307324,
                    -239131.93911860665, -969941.4300628213, 385773.9652874163,
                    278090.68951873126, 180849.71012352195, -470774.31315405807,
                    -14982.923343675813, -619996.3704957349, -1203464.8189202165,
                    -157637.91511571038, 362482.45126124314, -497552.47194499307,
                    -236640.6363944147, -117821.50366155664, -823524.9396619514,
                    62302.80202420966, -96621.09067510017, -125052.38923511525,
                    97769.0083372963, -965199.8661808498, 222220.7239743777,
                    -374702.7345993242, 155033.09889147282, -1989.630875353818],
            [-427881.97570378595, 1032301.8444213009, 607018.0135966156,
                    -1447924.970738861, 2005628.3012952602, 499797.5854784194,
                    622510.5342108728, 2524928.8110790555, -1004238.9922932169,
                    -723913.6401645798, -470774.31315405807, 1225515.2768995222,
                    39001.91367844581, 1613935.3071920145, 3132811.410855453,
                    410354.6573258249, -943643.2096791578, 1295193.7852807571,
                    616008.64907689, 306712.0378226414, 2143707.851570392,
                    -162179.4927570954, 251516.37232037642, 325538.4205528294,
                    -254505.75555902356, 2512644.2727309032, -578510.1002880113,
                    975430.5141329551, -403595.15688305, 5179.070796881113],
            [-13617.530634341638, 32854.22747731646, 19319.94031851559,
                    -46083.75449593974, 63829.770800075996, 15906.615340342569,
                    19811.239619897766, 80357.76413641141, -31960.216388136178,
                    -23040.250998091913, -14982.923343675813, 39001.91367844581,
                    1241.3385189229937, 51366.7955789965, 99702.3108080859,
                    13060.145803019688, -30030.028192114336, 41221.07079578728,
                    19605.247834362857, 9761.597937414512, 68228.21759586644,
                    -5161.704186270724, 8004.88048572238, 10359.987223487302,
                    -8099.861329113616, 79963.00689497797, -18410.037263166676,
                    31043.267870027597, -12843.928854833162, 164.84178348727528],
            [-563503.0643466829, 1359535.4531227027, 799465.1231295253,
                    -1906929.2316904257, 2641350.3080521864, 658226.857278107,
                    819812.9135390642, 3325284.0270826635, -1322541.9562906092,
                    -953434.0691506112, -619996.3704957349, 1613935.3071920145,
                    51366.7955789965, 2125597.304548632, 4125729.5848259497,
                    540438.4500383266, -1242702.3271705636, 1705741.068189257,
                    811275.3732202249, 403950.436570662, 2823280.188805055,
                    -213590.6110219508, 331244.08755697083, 428707.7124927715,
                    -335174.29690302856, 3308990.0677973754, -761852.8239891479,
                    1284614.197931908, -531509.7065459328, 6821.053891274157],
            [-1093807.4471157312, 2638892.0982596218, 1551742.404717372,
                    -3701435.8716964982, 5127000.986694892, 1277645.5563866766,
                    1591328.923455987, 6454508.230033948, -2567157.759426228,
                    -1850533.255967261, -1203464.8189202165, 3132811.410855453,
                    99702.3108080859, 4125729.5848259497, 8008541.321542145,
                    1048998.5050313922, -2412211.78696484, 3310962.6510743233,
                    1574724.0808363736, 784040.7689848202, 5480085.136142541,
                    -414589.70319374395, 642962.9309969022, 832179.4829438967,
                    -650606.0552617743, 6423061.221356821, -1478817.680854236,
                    2493487.8598607164, -1031695.6660793469, 13239.667773888234],
            [-143274.32024934815, 345667.84863406006, 203265.87160003383,
                    -484845.3566280408, 671578.037266826, 167357.41134353352,
                    208443.05518058766, 845470.0928448668, -336264.6943500905,
                    -242411.14225233067, -157637.91511571038, 410354.6573258249,
                    13060.145803019688, 540438.4500383266, 1048998.5050313922,
                    137408.6008743094, -315966.71928906185, 433694.47866279254,
                    206270.85667665722, 102704.98019558156, 717831.0996346171,
                    -54306.42454874992, 84220.5244337459, 109002.58458148634,
                    -85220.3192824344, 841334.6894181048, -193706.59635809882,
                    326619.8854179307, -135139.83929993995, 1734.2717532579436],
            [329464.18420184636, -794860.8161464253, -467383.8843663716,
                    1114827.964160791, -1544337.948133165, -384836.6886719451,
                    -479336.6991379133, -1944182.937873276, 773258.0277370816,
                    557409.7886478706, 362482.45126124314, -943643.2096791578,
                    -30030.028192114336, -1242702.3271705636, -2412211.78696484,
                    -315966.71928906185, 726637.3819157126, -997265.430360959,
                    -474313.9488545993, -236170.86012900577, -1650566.435514354,
                    124871.17138425406, -193660.3231670929, -250666.41510243676,
                    195962.68392153853, -1934778.5608125532, 445481.2395412975,
                    -751090.921891152, 310783.5962562703, -3987.559218979412],
            [-452213.78708996583, 1091011.1989148206, 641555.1887777617,
                    -1530318.2381011278, 2119663.6178804073, 528222.8533432522,
                    657900.5923021627, 2668509.1611595503, -1061341.708613389,
                    -765089.5555305473, -497552.47194499307, 1295193.7852807571,
                    41221.07079578728, 1705741.068189257, 3310962.6510743233,
                    433694.47866279254, -997265.430360959, 1368864.31907959,
                    651045.8515231148, 324153.44444827526, 2265678.4004814955,
                    -171407.0585552743, 265823.656563293, 344043.76444693253,
                    -268981.31944869336, 2655463.6300199735, -611377.6050845169,
                    1030886.2507075757, -426529.56895785424, 5473.860147105716],
            [-215077.8426113635, 518899.4015696987, 305132.059903797,
                    -727832.8495215549, 1008140.5005908067, 251229.2922489141,
                    312905.72644413053, 1269178.4498434912, -504786.57990451134,
                    -363890.3596168801, -236640.6363944147, 616008.64907689,
                    19605.247834362857, 811275.3732202249, 1574724.0808363736,
                    206270.85667665722, -474313.9488545993, 651045.8515231148,
                    309644.9282722824, 154173.38075181373, 1077579.5867207146,
                    -81522.81306041653, 126428.52608715445, 163630.89540469993,
                    -127929.97219949249, 1262973.7207831193, -290781.4266488454,
                    490304.56733124814, -202864.38927320475, 2603.419562994833],
            [-107087.45068333478, 258366.84703955788, 151928.88274220875,
                    -362379.6028049886, 501967.62658711313, 125089.06639656228,
                    155798.6535323029, 631941.0554721548, -251335.81249763246,
                    -181195.07273396023, -117821.50366155664, 306712.0378226414,
                    9761.597937414512, 403950.436570662, 784040.7689848202,
                    102704.98019558156, -236170.86012900577, 324153.44444827526,
                    154173.38075181373, 76769.35988650756, 536522.2459911266,
                    -40589.550210622714, 62948.51597611402, 81471.77983343764,
                    -63695.14153572459, 628851.9369505029, -144789.68252842987,
                    244132.85332968459, -101012.28661090058, 1296.2207276234788],
            [-748475.3164486786, 1805784.8802174286, 1061887.3462284554,
                    -2532950.829680498, 3508319.811483866, 874286.8934292952,
                    1088904.5405440978, 4416754.270064828, -1756661.0753439257,
                    -1266349.3650786553, -823524.9396619514, 2143707.851570392,
                    68228.21759586644, 2823280.188805055, 5480085.136142541,
                    717831.0996346171, -1650566.435514354, 2265678.4004814955,
                    1077579.5867207146, 536522.2459911266, 3750088.4701662865,
                    -283708.0857056962, 439980.0171042077, 569430.5613567849,
                    -445203.54222542694, 4395081.95769238, -1011880.8642217676,
                    1706246.973139175, -705947.4037211653, 9060.324337779772],
            [56624.942130779986, -136613.94859184456, -80335.43600964258,
                    191627.34944563507, -265416.7728446663, -66142.93440720538,
                    -82379.5850977837, -334143.0030964492, 132897.84866975705,
                    95803.21315757626, 62302.80202420966, -162179.4927570954,
                    -5161.704186270724, -213590.6110219508, -414589.70319374395,
                    -54306.42454874992, 124871.17138425406, -171407.0585552743,
                    -81522.81306041653, -40589.550210622714, -283708.0857056962,
                    21463.584825583323, -33286.11965366797, -43079.587776675464,
                    33681.361394683474, -332503.7460183222, 76552.19242979104,
                    -129083.4533779376, 53407.305044565444, -685.4470597571659],
            [-87816.4837208772, 211866.75720555347, 124586.13005423953,
                    -297177.9863681518, 411622.71723057824, 102577.1383801494,
                    127759.0605202802, 518205.1120158168, -206104.45876818095,
                    -148575.82840293963, -96621.09067510017, 251516.37232037642,
                    8004.88048572238, 331244.08755697083, 642962.9309969022,
                    84220.5244337459, -193660.3231670929, 265823.656563293,
                    126428.52608715445, 62948.51597611402, 439980.0171042077,
                    -33286.11965366797, 51621.107956628206, 66810.39748020428,
                    -52234.17349903379, 515669.16612852685, -118724.19772457697,
                    200190.33198319352, -82828.47575395652, 1062.9922663705831],
            [-113659.46065397991, 274211.2373116087, 161240.75009037546,
                    -384609.91910996195, 532760.6687731618, 132761.78158270588,
                    165360.08034015898, 670701.4840533712, -266758.9838720577,
                    -192291.63727863415, -125052.38923511525, 325538.4205528294,
                    10359.987223487302, 428707.7124927715, 832179.4829438967,
                    109002.58458148634, -250666.41510243676, 344043.76444693253,
                    163630.89540469993, 81471.77983343764, 569430.5613567849,
                    -43079.587776675464, 66810.39748020428, 86474.60960771656,
                    -67604.85973949074, 667446.9213854757, -153673.90990044305,
                    259106.53605521304, -107209.4264066645, 1375.6960266945034],
            [88859.94740941831, -214382.2903332192, -126064.20956241484,
                    300706.3619029368, -416512.20772471745, -103795.28152140026,
                    -129277.37186774613, -524359.3000358283, 208553.34492990456,
                    150337.46468169338, 97769.0083372963, -254505.75555902356,
                    -8099.861329113616, -335174.29690302856, -650606.0552617743,
                    -85220.3192824344, 195962.68392153853, -268981.31944869336,
                    -127929.97219949249, -63695.14153572459, -445203.54222542694,
                    33681.361394683474, -52234.17349903379, -67604.85973949074,
                    52854.870747991714, -521798.4380423071, 120135.4345834101,
                    -202568.0459981826, 83812.77706615409, -1075.6048368400182],
            [-877271.1179979955, 2116495.277061747, 1244532.2694496606,
                    -2968549.589068031, 4112110.4047745992, 1024715.8636411378,
                    1276326.3970300478, 5176801.443597403, -2058963.6467483111,
                    -1484227.4060228057, -965199.8661808498, 2512644.2727309032,
                    79963.00689497797, 3308990.0677973754, 6423061.221356821,
                    841334.6894181048, -1934778.5608125532, 2655463.6300199735,
                    1262973.7207831193, 628851.9369505029, 4395081.95769238,
                    -332503.7460183222, 515669.16612852685, 667446.9213854757,
                    -521798.4380423071, 5151685.670941004, -1186148.933316184,
                    1999922.1689142368, -827505.67522064, 10618.107310982323],
            [201980.68894789586, -487298.17819055147, -286532.5812771456,
                    683441.8508461871, -946778.110294905, -235927.61123554132,
                    -293863.91038333974, -1191905.4989193215, 474053.7557940749,
                    341730.2139528351, 222220.7239743777, -578510.1002880113,
                    -18410.037263166676, -761852.8239891479, -1478817.680854236,
                    -193706.59635809882, 445481.2395412975, -611377.6050845169,
                    -290781.4266488454, -144789.68252842987, -1011880.8642217676,
                    76552.19242979104, -118724.19772457697, -153673.90990044305,
                    120135.4345834101, -1186148.933316184, 273114.8613477328,
                    -460469.1766300898, 190533.7237230798, -2444.5541643981],
            [-340566.08097309095, 821655.4956393185, 483153.3495236603,
                    -1152443.0012280906, 1596369.7050428558, 397809.33378739585,
                    495481.2331138214, 2009703.8383549566, -799311.7184875482,
                    -576210.7159965835, -374702.7345993242, 975430.5141329551,
                    31043.267870027597, 1284614.197931908, 2493487.8598607164,
                    326619.8854179307, -751090.921891152, 1030886.2507075757,
                    490304.56733124814, 244132.85332968459, 1706246.973139175,
                    -129083.4533779376, 200190.33198319352, 259106.53605521304,
                    -202568.0459981826, 1999922.1689142368, -460469.1766300898,
                    776393.7397663131, -321243.75384548464, 4122.182167280981],
            [140911.57157102908, -339963.90830350405, -199901.83344425834,
                    476810.7250318338, -660516.0362065202, -164595.1661642016,
                    -205012.36107524062, -831531.0005204134, 330722.3835950521,
                    238409.02615910847, 155033.09889147282, -403595.15688305,
                    -12843.928854833162, -531509.7065459328, -1031695.6660793469,
                    -135139.83929993995, 310783.5962562703, -426529.56895785424,
                    -202864.38927320475, -101012.28661090058, -705947.4037211653,
                    53407.305044565444, -82828.47575395652, -107209.4264066645,
                    83812.77706615409, -827505.67522064, 190533.7237230798,
                    -321243.75384548464, 132923.1402000635, -1705.4815430793697],
            [-1808.2870504090968, 4362.735925535438, 2565.5529679555852,
                    -6119.722146225432, 8475.908935267644, 2112.2592771769364,
                    2630.7156238501116, 10670.709481456897, -4244.009956370862,
                    -3059.4982172104096, -1989.630875353818, 5179.070796881113,
                    164.84178348727528, 6821.053891274157, 13239.667773888234,
                    1734.2717532579436, -3987.559218979412, 5473.860147105716,
                    2603.419562994833, 1296.2207276234788, 9060.324337779772,
                    -685.4470597571659, 1062.9922663705831, 1375.6960266945034,
                    -1075.6048368400182, 10618.107310982323, -2444.5541643981,
                    4122.182167280981, -1705.4815430793697, 21.890454372860443]]

    def testVarianceCovariance1(self):
        cov = statistics.variance_covariance(self.data)
        for i, x in enumerate(cov):
            for j, y in enumerate(x):
                self.assertAlmostEqual(cov[i][j], self.expected_cov[i][j])

    def testVarianceCovariance2(self):
        d = [[-323.9573971686496, 315.7388065925112, -439.6717600886799,
            -1178.099816177854, 534.643723366918, 1432.26163863866,
            -497.1855740226206, -1301.8090901379649, -757.2159368960788,
            1019.1610261218987], [-971.8721915059489, 947.2164197775336,
                -1319.0152802660396, -3534.299448533562, 1603.9311701007541,
                4296.7849159159805, -1491.5567220678618, -3905.427270413895,
                -2271.6478106882364, 3057.483078365696], [1805.2578794408694,
                    -110.00822001898491, 3403.866092254204, -7.164802263774928,
                    1090.216811454231, -1328.8920608696994, 1313.2108525086446,
                    -20191.95994589982, -2217.3808792199297,
                    -14735.400560599832], [-394.27706884544017,
                        305.72735899534194, -386.17544812632764,
                        -1312.6418677447905, 554.652417355839,
                        1429.310268304489, -479.9816099694797,
                        -1474.7086736016736, -766.5610040209765,
                        943.9067418319103], [-91.76220598952864,
                            136.88060210863958, -85.06584754543456,
                            -81.68026472600144, 26.24124439892005,
                            -0.4363035110954008, 9.600832668565268,
                            -13.610656002675592, -56.73800195687172,
                            10.062534396465715]]

        e = [[1118051.315834594, -340504.4676922032, 1930944.9443670718,
            1089311.6226191486, -702.760833078737, -1907420.4159428228,
            1043769.6960229203, -7881480.1714532245, -303512.8258235939,
            -7557633.140329254], [-340504.4676922032, 153014.53369854396,
                -566748.0815504972, -545103.1972692355, 131399.1886553155,
                811301.8312219149, -373506.9853324348, 1559586.2728247487,
                -113040.33582529923, 2056440.4519370624], [1930944.9443670718,
                    -566748.0815504972, 3350039.66106187, 1776176.7004303827,
                    62345.622800847406, -3185282.8296726383,
                    1777391.1979210868, -14030883.204065276,
                    -623124.0396947996, -13181274.97885164],
                [1089311.6226191486, -545103.1972692355, 1776176.7004303827,
                    2033255.6374553614, -594563.6823540081,
                    -2903121.0040589883, 1268999.4533419132,
                    -3864821.0472551202, 631450.3221457871,
                    -6251946.061145776], [-702.760833078737, 131399.1886553155,
                        62345.622800847406, -594563.6823540081,
                        363149.79842651676, 640030.6375209386,
                        -154836.23379737284, -2326341.9245344824,
                        -566971.7171338849, -674414.9016981383],
                    [-1907420.4159428228, 811301.8312219149,
                        -3185282.8296726383, -2903121.0040589883,
                        640030.6375209386, 4381775.401023215,
                        -2054062.3922582623, 9356357.877914747,
                        -479593.4464370444, 11710459.142730745],
                    [1043769.6960229203, -373506.9853324348,
                        1777391.1979210868, 1268999.4533419132,
                        -154836.23379737284, -2054062.3922582623,
                        1041077.3129029109, -6374584.529610114,
                        -42690.656074846236, -6773734.4112940645],
                    [-7881480.1714532245, 1559586.2728247487,
                        -14030883.204065276, -3864821.0472551202,
                        -2326341.9245344824, 9356357.877914747,
                        -6374584.529610114, 70562391.7586506,
                        5780752.083342586, 57672359.799883455],
                    [-303512.8258235939, -113040.33582529923,
                        -623124.0396947996, 631450.3221457871,
                        -566971.7171338849, -479593.4464370444,
                        -42690.656074846236, 5780752.083342586,
                        968375.1765316972, 3112971.715860271],
                    [-7557633.140329254, 2056440.4519370624,
                        -13181274.97885164, -6251946.061145776,
                        -674414.9016981383, 11710459.142730745,
                        -6773734.4112940645, 57672359.799883455,
                        3112971.715860271, 52393351.00087271]]

        cov = statistics.variance_covariance(d)
        for i, x in enumerate(cov):
            for j, y in enumerate(x):
                self.assertAlmostEqual(cov[i][j], e[i][j])

class FishersExactTests(dendropytest.ExtendedTestCase):
    """
    Fisher's exact test.
    """

    def setUp(self):
        self.table = [[12, 5], [29, 2]]

    def testLeftTail(self):
        ft = statistics.FishersExactTest(self.table)
        p = ft.left_tail_p()
        self.assertAlmostEqual(p, 0.044554737835078267)

    def testRightTail(self):
        ft = statistics.FishersExactTest(self.table)
        p = ft.right_tail_p()
        self.assertAlmostEqual(p, 0.99452520602190897)

    def testTwoTail(self):
        ft = statistics.FishersExactTest(self.table)
        p = ft.two_tail_p()
        self.assertAlmostEqual(p, 0.08026855207410688)

if __name__ == "__main__":
    unittest.main()

