/*
   Copyright (c) 2008-2009, Roger Kaufman

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

      The above copyright notice and this permission notice shall be included
      in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
  IN THE SOFTWARE.
*/

/*
   Name: bravais.cc
   Description: Generate the 14 Bravais Lattices
   Project: Antiprism - http://www.antiprism.com
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <limits.h>
#include <ctype.h>

#include <string>
#include <vector>
#include <algorithm>

#include "../base/antiprism.h"
#include "../base/rand_gen.h"
#include "lattice_grid.h"

using std::string;
using std::vector;
using std::swap;


struct bravaisItem {
   const char *crystal_system;
   const char *centering;
};

bravaisItem bravais_item_list[] = {
   {"Triclinic",  "P"},
   {"Monoclinic", "P"},
   {"Monoclinic", "C"},
   {"Orthorhombic", "P"},
   {"Orthorhombic", "C"},
   {"Orthorhombic", "F"},
   {"Orthorhombic", "I"},
   {"Tetragonal", "P"},
   {"Tetragonal", "I"},
   {"Trigonal", "P"},
   {"Hexagonal", "P"},
   {"Cubic", "P"},
   {"Cubic", "F"},
   {"Cubic", "I"},
};

class bravais
{
   private:
      int last_bravais;
      bravaisItem* bravais_items;

   public:
      bravais();
      void list_bravais(int idx, FILE *fp=stderr);
      void list_bravais(FILE *fp=stderr);
      int lookup_sym_no(string crystal_system, string centring);
      int get_last_bravais() { return last_bravais; }

      string get_crystal_system(int);
      string get_centering(int);
};

class brav_opts: public prog_opts {
   public:
      string ofile;
      string cfile;

      vector<double> vecs;
      vector<double> angles;
      vector<int> grid;
      vector<int> prim_vec_idxs;

      vector<double> strut_len;
      bool cell_struts;
      bool use_centering_for_dual;
      double radius;
      char radius_default;
      vec3d radius_by_coord;
      vec3d offset;
      char container;
      bool append_container;
      bool voronoi_cells;
      bool voronoi_central_cell;
      char auto_grid_type;
      bool grid_for_radius;
      bool convex_hull;
      bool add_hull;
      bool append_lattice;
      char color_method;
      int face_opacity;
      col_val cent_col;
      bool trans_to_origin;
      int r_lattice_type;

      bool list_bravais;
      vec3d list_radii_center;
      char list_radii_original_center;
      int list_radii;
      int list_struts;

      bool dual_lattice;
      
      bool verbose;
      double epsilon;
      
      // 0 - lattice  1 - convex hull  2 - voronoi  3 - hex overlay
      vector<col_val> vert_col;
      vector<col_val> edge_col;
      vector<col_val> face_col;
      
      string crystal_system;
      string centering;

      brav_opts(): prog_opts("bravais"),
                   cell_struts(false),
                   use_centering_for_dual(false),
                   radius(0),
                   radius_default('s'),
                   container('c'),
                   append_container(false),
                   voronoi_cells(false),
                   voronoi_central_cell(false),
                   auto_grid_type('\0'),
                   grid_for_radius(false),
                   convex_hull(false),
                   add_hull(false),
                   append_lattice(false),
                   color_method('\0'),
                   face_opacity(-1),
                   trans_to_origin(false),
                   r_lattice_type(0),
                   list_bravais(false),
                   list_radii_original_center('\0'),
                   list_radii(0),
                   list_struts(0),
                   dual_lattice(false),
                   verbose(false),
                   epsilon(0) {}

      void process_command_line(int argc, char **argv);
      void usage();
};

void extended_help()
{
   fprintf(stdout,
"\n"
"Definition: (partly from http://en.wikipedia.org/wiki/Bravais_lattice)\n"
"In geometry and crystallography, a Bravais lattice, named after Auguste\n"
"Bravais, is an infinite set of points generated by a set of discrete\n"
"translation operations. A crystal is made up of one or more atoms (the basis)\n"
"which is repeated at each lattice point. The crystal then looks the same when\n"
"viewed from any of the lattice points. In all, there are 14 possible Bravais\n"
"lattices that fill three-dimensional space.\n\n"
"August Bravais (1811-1863), a French naval officer, adventurer, and physicist\n"
"taught a course in applied mathematics for astronomy in the faculty of sciences\n"
"in Lyon from 1840. He served as the Chair of Physics, Ecole Polytechnique\n"
"between 1845 and 1856. He is best remembered for pointing out in 1845, that\n"
"there are 14 unique Bravais lattices in three dimensional crystalline systems,\n"
"adjusting the previously result (15 lattices) by Moritz Ludwig Frankenheim\n"
"obtained three years before.\n\n"
"A German Crystallographer, Frankenheim (1801-1869) is noted as the first to\n"
"enumerate the 32 crystal classes. And he also solved the symmetry systems of\n"
"the 7 crystal systems but this work went completely unnoticed at the time.\n\n"
"There is a bit of mystery surrounding what Frankenheim had as the 15th lattice.\n"
"Even today, in some texts the Hexagonal lattice with two interior points is\n"
"shown in the Trigonal class. But these two lattices use the same set of points\n"
"and it is thought that it was this duplication that was eliminated by Bravais.\n"
"However, in Bravais' paper, there is no mention of Frankenheim or the\n"
"enumeration of lattices he presented.\n\n"
"In this program, the Hexagonal cells and Trigonal cells can be seen together\n"
"by using the -R parameter.\n\n"
"Note that End Centered Cubic (would be Cubic C) does not exist but can be\n"
"produced by Tetragonal P that has cells of dimensions a,b,c = 1,1,sqrt(2)\n\n"
"Face Centered Cubic (Cubic F or FCC) is duplicated in Body Centered Tetragonal\n"
"(Tetragonal I) of dimensions a,b,c = 1,1,sqrt(2). However, the FCC embodied\n"
"would be of higher symmetry than the Tetragonal crystal system is allowed.\n\n"
"Similarly, Trigonal at 90 degrees (improper) is SC. Trigonal at 60 degrees is\n"
"FCC and Trigonal at acos(-1/3) or 109.47122063449... degrees is BCC.\n\n"
"Also there is no provision for Face Centered Tetragonal (would be Tetragonal F)\n"
"or Base Centered Tetragonal (would be Tetragonal C). These would be embodied in\n"
"Body Centered Tetragonal (Tetragonal I) and Simple Tetragonal (Tetragonal P)\n"
"respectively. This is true at any proportion other than a,b,c = 1,1,sqrt(2)\n\n"
"In Hexagonal, Orthorhombic C can be seen to occur. When Hexagonal of a=b=c is\n"
"produced, then Base Centered Tetragonal (would be Tetragonal C) occurs.\n\n"
"Hexagonal is sensitive to which unequal vector corresponds to the non-90 degree\n"
"angle. These must be in the same position or a Monoclinic lattice is produced.\n\n"
"Bravais lattices will fall into the following symmetries\n\n"
"Crystal System   Possible Symmetries (32 possible - note no 5 fold symmetries)\n"
"Triclinic        C1 Ci\n"
"Monoclinic       C2 Cs C2h\n"
"Orthorhombic     D2 C2v D2h\n"
"Tetragonal       C4 S4 C4h D4 C4v D2d D4h\n"
"Trigonal         C3 S6 D3 C3v D3d\n"
"Hexagonal        C6 C3h C6h D6 C6v D3h D6h\n"
"Cubic            T Th O Td Oh\n\n"
"Of the symbols used for cell centering:\n\n"
"P - stands for Primitive. It is a cube depicted by a vertex at eight corners\n"
"C - stands for having a point filled in on the \"C\" side of the primitive cell\n"
"    this is described in some texts as Base Centering. C is most commonly used\n"
"    A or B means use the \"A\" or \"B\" sides instead. Just a rotation of C\n"
"F - stands for Face Centering and fills all three, \"A\", \"B\" and \"C\" sides\n"
"I - (from German: innenzentriert, meaning Body Centered) is the primitive cell\n"
"    with one point filled in the center of the cell\n\n"
"The term Isometric is sometimes used for Cubic. Also allowable in this program.\n\n"
"Monoclinic is defined in this program with angles alpha = gamma = 90 <> beta\n"
"as is found in the first volume of International Tables for Crystallography.\n\n"
"Vectors a, b and c correspond to axes x, y and z (before transformations).\n"
"\n");
}

void brav_opts::usage()
{
   fprintf(stdout,
"\n"
"Usage: %s [options] crystal_system [centering]\n"
"\n"
"Generate Bravais lattices in OFF format. lattice may be specified by its index\n"
"number or (start of the) crystal_system name and centering\n\n"
"Crystal System   Centering   Vector Constraints  Angle Constraints\n"
"Triclinic        P           no constraints      any not of higher symmetries\n"
"Monoclinic       P,C         no constraints      alpha = gamma = 90 <> beta\n"
"Orthorhombic     P,C,F,I     a <> b <> c         alpha = beta = gamma = 90\n"
"Tetragonal       P,I         a = b <> c          alpha = beta = gamma = 90\n"
"Trigonal         P           a = b = c           alpha = beta = gamma <> 90\n"
"Hexagonal        P           a = b, c            alpha = beta = 90, gamma = 120\n"
"Cubic            P,F,I       a = b = c           alpha = beta = gamma = 90\n\n"
"Centering Types: P - Primitive (Simple) centering on corners (default: P)\n"
"                 C - Base (also A or B)  F - Face centering  I - Body centering\n\n"
"Synonyms: Rhombohedral = Trigonal  sc = Cubic P  fcc = Cubic F  bcc = Cubic I\n"
"\nOptions\n"
"%s"
"  -H        additional help\n"
"  -I        verbose output\n"
"  -l <lim>  minimum distance for unique vertex locations as negative exponent\n"
"               (default: %d giving %.0e)\n"
"  -o <file> write output to file (default: write to standard output)\n"
"\nLattice Options\n"
"  -v <v,n>  vector lengths, non-zero, in form \"a,b,c\" (default: calculated)\n"
"               optional fourth number, vectors taken to root n\n"
"  -a <angs> angles in the form \"alpha,beta,gamma\". Ignored for Orthorhombic,\n"
"               Tetragonal, and Cubic. For Hexagonal, any non-90 position may be\n"
"               120. Otherwise, if not supplied then random angles are chosen.\n"
"               Angles cannot be zero or 180. Angles may be negative values.\n"
"               alpha + beta + gamma must be less than 360. Each angle must be\n"
"               less than or equal to the sum of the other two angles\n"
"  -g <grid> cell grid array. one or three positive integers separated by commas\n"
"               a - automatic, of sufficient size for radius (-G required)\n"
"               one integer, NxNxN grid. (default: calculated)\n"
"               three integers, IxJxK grid. Combinations for grid center:\n"
"               even,even,even - on cell corner    odd,odd,odd - in cell body\n"
"               even,odd,even - on cell mid-edge   odd,even,odd - on face center\n"
"  -G <type> automatic grid center type (type 8 invalid for cell centering = P):\n"
"               p - corner, i - body, f - face, e - mid-edge, 8 - eighth cell\n"
"  -d <vrts> output dual of lattice based on primitive vectors\n"
"               c - use primitive vectors base on centering type\n"
"               four integers - primitive vectors are determined by four vertex\n"
"                 numbers given by non negative integers. The first vertex\n"
"                 number is the radial point and the next three vertices are the\n"
"                 primitive vectors\n"
"  -u        add cell struts. Added to cubic grid before transformation\n"
"  -s <s,n>  create struts. s is strut length taken to optional root n\n"
"               use multiple -s parameters for multiple struts\n"
"  -D <opt>  Voronoi (a.k.a Dirichlet) cells (Brillouin zones for duals)\n"
"               c - cells only, i - cell(s) touching center only\n"
"  -A        append the original lattice to the final product\n"
"\nContainer Options\n"
"  -c <type> container, c - cube (default), s - sphere (uses radius)\n"
"  -k <file> container, convex hull of off file or built in model (uses radius)\n"
"  -r <c,n>  radius. c is radius taken to optional root n. n = 2 is sqrt\n"
"               or  l - max insphere radius, s - min insphere radius (default)\n"
"               or  k - take radius from container specified by -k\n"
"  -p <xyz>  radius to lattice point \"x_val,y_val,z_val\"\n"
"  -q <vecs> center offset, in form \"a_val,b_val,c_val\" (default: none)\n"
"  -C <opt>  c - convex hull only, i - keep interior\n"
"\nColoring Options (run 'off_util -H color' for help on color formats)\n"
"  -V <col>  vertex color, (optional) elements, (optional) transparency\n"
"               elements to color are l - lattice, c - convex hull, v - voronoi\n"
"                                     h - hex relation (default elements: lcvh)\n"
"               transparency. valid range from 0 (invisible) to 255 (opaque)\n"
"  -E <col>  edge color (same format as for vertices)\n"
"  -F <col>  face color (same format as for vertices)\n"
"               lower case outputs map indexes. upper case outputs color values\n"
"               key word: s,S color by symmetry using face normals\n"
"               key word: c,C color by symmetry using face normals (chiral)\n"
"  -T <tran> face transparency for color by symmetry. valid range from 0 to 255\n"
"\nScene Options\n"
"  -R <opt>  hexagonal/cubic relation (Cubic P or Trigonal only)\n"
"               o - hex overlay, f - hex fill, O - cube overlay, F - cube fill\n"
"  -O        translate centroid of final product to origin\n"
"  -Z <col>  add centroid vertex to final product in color col\n"
"  -K        append cage of container of -k to final product\n"
"\nListing Options\n"
"  -B        display the list of Bravais lattices\n"
"  -Q <vecs> center for radius calculations in -L (default: centroid)\n"
"               c - original center, o - original center + offset in -q\n"
"  -L <opt>  list unique radial distances of points (to standard output)\n"
"               f - full report, v - values only\n"
"  -S <opt>  list every possible strut value (to standard output)\n"
"               f - full report, v - values only\n"
"\n"
"\n",prog_name(), help_ver_text, int(-log(::epsilon)/log(10) + 0.5), ::epsilon);
}

void brav_opts::process_command_line(int argc, char **argv)
{
   opterr = 0;
   char c;
   char errmsg[MSG_SZ];

   int sig_compare = INT_MAX;  
   bool radius_set = false;
   vector<double> double_parms;

   vert_col.resize(4);
   edge_col.resize(4);
   face_col.resize(4);
   
   // set some default colors first so they can be optionally unset
   // 0 - lattice  1 - convex hull  2 - voronoi  3 - hex overlay
   // voronoi cells  vcol = gold; ecol = lightgrey; fcol = transparent yellow
   vert_col[2] = col_val(255,215,0);
   edge_col[2] = col_val(211,211,211);
   face_col[2] = col_val(255,255,0,128);
   
   vert_col[3] = col_val(255,215,0);
   edge_col[3] = col_val(211,211,211);

   handle_long_opts(argc, argv);

   while( (c = getopt(argc, argv, ":hHIc:k:r:p:q:s:uv:a:g:G:d:l:D:C:AV:E:F:T:Z:KOR:BQ:L:S:o:")) != -1 ) {
      if(common_opts(c, optopt))
         continue;

      switch(c) {
         case 'H':
            extended_help();
            exit(0);

         case 'I':
            verbose = true;
            break;

         case 'c':
            if(strlen(optarg) > 1 || !strchr("cs", *optarg))
               error("method is '"+string(optarg)+"' must be c or s", c);
            container = *optarg;
            break;

         case 'k':
            cfile = optarg;
            break;
            
         case 'r':
            radius_set = true;
            if(strlen(optarg) == 1 && strchr("lsk", *optarg))
               radius_default = *optarg;
            else {
               if(!read_double_list(optarg, double_parms, errmsg, 2))
                  error(errmsg, c);
               radius = double_parms[0];
               if(double_parms.size() == 2) {
                  if(double_parms[1] == 0)
                     error("root for radius must be non-zero", c);
                  radius = pow(radius, 1/double_parms[1]);
               }
               if(radius <= 0)
                  error("radius cannot be negative or zero", c);
            }
            break;

         case 'p':
            if(!radius_by_coord.read(optarg, errmsg))
               error(errmsg, c);
            break;

         case 'q':
            if(!offset.read(optarg, errmsg))
               error(errmsg, c);
            break;

         case 's': {
            double strut_tmp;
            if(!read_double_list(optarg, double_parms, errmsg, 2))
               error(errmsg, c);
            strut_tmp = double_parms[0];
            if(double_parms.size() == 2) {
               if(double_parms[1] == 0)
                  error("root for strut must be non-zero", c);
               strut_tmp = pow(strut_tmp, 1/double_parms[1]);
            }
            if(strut_tmp <= 0)
               error("strut lengths cannot be negative", "s", c);
            strut_len.push_back(strut_tmp);
            break;
         }

         case 'u':
            cell_struts = true;
            break;

         case 'v':
            if(!read_double_list(optarg, vecs, errmsg, 4))
               error(errmsg, c);
            if(vecs.size() < 3)
               error(msg_str("three vector lengths needed (%lu were given)",
                        (unsigned long)vecs.size()), c);
            if(vecs[0] == 0 || vecs[1] == 0 || vecs[2] == 0)
               error("vector lengths need to be non-zero", c);
            if(vecs.size() == 4) {
               if(vecs[3] == 0)
                  error("root for vector must be non-zero", c);
               for(unsigned int i=0; i<3; i++)
                  vecs[i] = pow(vecs[i], 1/vecs[3]);
            }
            break;

         case 'a': {
            if(!read_double_list(optarg, angles, errmsg, 3))
               error(errmsg, c);
            if(angles.size() < 3)
               error(msg_str("three angles needed (%lu were given)",
                        (unsigned long)angles.size()), c);

            double alpha = fabs(angles[0]);
            double beta  = fabs(angles[1]);
            double gamma = fabs(angles[2]);

            if(fmod(alpha,180.0) == 0 || fmod(beta,180.0) == 0 || fmod(gamma,180.0) == 0)
               error("angles cannot be zero or 180", c);
            if(alpha + beta + gamma >= 360.0)
               error("alpha + beta + gamma must be less than 360.0", c);
            if(alpha >= beta + gamma)
               error("alpha must be less than beta + gamma", c);
            if(beta >= alpha + gamma)
               error("beta must be less than alpha + gamma", c);
            if(gamma >= alpha + beta)
               error("gamma must be less than alpha + beta", c);
            break;
         }

         case 'g':
            if(strchr("a", *optarg))
               grid_for_radius = true;
            else {
               if(!read_int_list(optarg, grid, errmsg, true, 3))
                  error(errmsg, c);
               if(grid.size() == 1) {
                  grid.push_back(grid[0]);
                  grid.push_back(grid[0]);
               }
               else if(grid.size() != 3)
                  error(msg_str("must give one or three numbers (%lu were "
                           "given)", (unsigned long)grid.size()), c);
               if(grid[0] == 0 || grid[1] == 0 || grid[2] == 0)
                  error("grid requires positive integer(s)", c);
               //if(grid[0] != grid[1] || grid[0] != grid[2] || grid[1] != grid[2])
               //   warning("grid symmetry may be less than cells symmetry", c);
            }
            break;

         case 'G':
            if(strlen(optarg) > 1 || !strchr("pfie8", *optarg))
               error("grid type is '"+string(optarg)+"' must be p, f, i, e, or 8", c);
            auto_grid_type = *optarg;
            break;

         case 'd':
            if(strlen(optarg)==1) {
               if(strchr("c", *optarg))
                  use_centering_for_dual = true;
               else
                  error("invalid option", c);
            }
            else { 
               if(!read_int_list(optarg, prim_vec_idxs, errmsg, true, 4))
                  error(errmsg, c);
               if(prim_vec_idxs.size() != 4)
                  error(msg_str("four lattice vertex indices needed (%lu were"
                           "given)", (unsigned long)prim_vec_idxs.size()), c);
               else {
                  for(unsigned int i=0; i<3; i++) {
                     for(unsigned int j=i+1; j<4; j++) {
                        if (prim_vec_idxs[i] == prim_vec_idxs[j])
                           error(msg_str("lattice vertex indices used twice. "
                                    "pos: %d & %d", i+1, j+1), c);
                     }
                  }
               }
            }
            break;

         case 'l':
            if(!read_int(optarg, &sig_compare, errmsg))
               error(errmsg, c);
            if(sig_compare < 0) {
               warning("limit is negative, and so ignored", c);
            }
            if(sig_compare > DEF_SIG_DGTS) {
               warning("limit is very small, may not be attainable", c);
            }
            break;

         case 'D':
            if(strlen(optarg) > 1 || !strchr("ci", *optarg))
               error("Voronoi cells arg is '"+string(optarg)+"' must be c, i", c);
            voronoi_cells = true;
            if(strchr("i", *optarg))
               voronoi_central_cell = true;
            break;

         case 'C':
            if(strlen(optarg) > 1 || !strchr("ci", *optarg))
               error("convex hull arg is '"+string(optarg)+"' must be c, i", c);
            convex_hull = true;
            if(strchr("i", *optarg))
               add_hull = true;
            break;
            
         case 'A':
            append_lattice = true;
            break;

         case 'V': {
            vector<char *> parts;
            int parts_sz = split_line(optarg, parts, ",");
            if(parts_sz>6)
               error("the argument has more than 6 parts", c);
            
            col_val col;
            bool valid_color = false;
            int next_parms_idx = 1;

            if (parts_sz >= 3) {
               char parts_test[MSG_SZ];
               parts_test[0] = '\0';
               strcat(parts_test,parts[0]);
               strcat(parts_test,",");
               strcat(parts_test,parts[1]);
               strcat(parts_test,",");
               strcat(parts_test,parts[2]);
               if(col.read(parts_test, errmsg)) {
                  valid_color = true;
                  next_parms_idx = 3;
               }
               col_val col_save = col;
               if (parts_sz >=4) {
                  strcat(parts_test,",");
                  strcat(parts_test,parts[3]);
                  if(col.read(parts_test, errmsg)) {
                     valid_color = true;
                     next_parms_idx = 4;
                  }
                  else
                     col = col_save;
               }
            }
            
            if (!valid_color) {
               if(!col.read(parts[0], errmsg))
                  error(errmsg, c);
            }

            unsigned int conv_elems = 15;
            if(parts_sz>next_parms_idx) {
               if(strspn(parts[next_parms_idx], "lcvh") !=
                                              strlen(parts[next_parms_idx]))
                  error(msg_str("elements to map are '%s' must be from "
                           "l, c, v, h", parts[next_parms_idx]), c);
               conv_elems = 8*(strchr(parts[next_parms_idx], 'h')!=0) +
                            4*(strchr(parts[next_parms_idx], 'v')!=0) +
                            2*(strchr(parts[next_parms_idx], 'c')!=0) +
                            1*(strchr(parts[next_parms_idx], 'l')!=0);
            }
            
            int opq = col[3];
            if(parts_sz>next_parms_idx+1)
               opq = atoi(parts[next_parms_idx+1]);
               
            if (opq < 0 || opq > 255)
               error("transparency value must be between 0 and 255", c);

            for(int i=0; i<4; i++) {
              if(conv_elems & (1<<i)) {
                  if (!col.is_set())
                     vert_col[i] = col_val();
                  else
                     vert_col[i] = col_val(col[0],col[1],col[2],opq);
               }
            }
            break;
         }
         
         case 'E': {
            vector<char *> parts;
            int parts_sz = split_line(optarg, parts, ",");
            if(parts_sz>6)
               error("the argument has more than 6 parts", c);
            
            col_val col;
            bool valid_color = false;
            int next_parms_idx = 1;

            if (parts_sz >= 3) {
               char parts_test[MSG_SZ];
               parts_test[0] = '\0';
               strcat(parts_test,parts[0]);
               strcat(parts_test,",");
               strcat(parts_test,parts[1]);
               strcat(parts_test,",");
               strcat(parts_test,parts[2]);
               if(col.read(parts_test, errmsg)) {
                  valid_color = true;
                  next_parms_idx = 3;
               }
               col_val col_save = col;
               if (parts_sz >=4) {
                  strcat(parts_test,",");
                  strcat(parts_test,parts[3]);
                  if(col.read(parts_test, errmsg)) {
                     valid_color = true;
                     next_parms_idx = 4;
                  }
                  else
                     col = col_save;
               }
            }
            
            if (!valid_color) {
               if(!col.read(parts[0], errmsg))
                  error(errmsg, c);
            }

            unsigned int conv_elems = 15;
            if(parts_sz>next_parms_idx) {
               if(strspn(parts[next_parms_idx], "lcvh") !=
                                             strlen(parts[next_parms_idx]))
                  error(msg_str("elements to map are '%s' must be l, c, v or h",
                           parts[next_parms_idx]), c);
               conv_elems = 8*(strchr(parts[next_parms_idx], 'h')!=0) +
                            4*(strchr(parts[next_parms_idx], 'v')!=0) +
                            2*(strchr(parts[next_parms_idx], 'c')!=0) +
                            1*(strchr(parts[next_parms_idx], 'l')!=0);
            }
            
            int opq = col[3];
            if(parts_sz>next_parms_idx+1)
               opq = atoi(parts[next_parms_idx+1]);
               
            if (opq < 0 || opq > 255)
               error("transparency value must be between 0 and 255", c);

            for(int i=0; i<4; i++) {
              if(conv_elems & (1<<i)) {
                  if (!col.is_set())
                     edge_col[i] = col_val();
                  else
                     edge_col[i] = col_val(col[0],col[1],col[2],opq);
               }
            }
            break;
         }
         
         case 'F': {
            vector<char *> parts;
            int parts_sz = split_line(optarg, parts, ",");
            if(parts_sz>6)
               error("the argument has more than 6 parts", c);
               
            if ((!strcasecmp(parts[0],"s")) || (!strcasecmp(parts[0],"c"))) {
               color_method = parts[0][0];
               break;
            }

            col_val col;
            bool valid_color = false;
            int next_parms_idx = 1;

            if (parts_sz >= 3) {
               char parts_test[MSG_SZ];
               parts_test[0] = '\0';
               strcat(parts_test,parts[0]);
               strcat(parts_test,",");
               strcat(parts_test,parts[1]);
               strcat(parts_test,",");
               strcat(parts_test,parts[2]);
               if(col.read(parts_test, errmsg)) {
                  valid_color = true;
                  next_parms_idx = 3;
               }
               col_val col_save = col;
               if (parts_sz >=4) {
                  strcat(parts_test,",");
                  strcat(parts_test,parts[3]);
                  if(col.read(parts_test, errmsg)) {
                     valid_color = true;
                     next_parms_idx = 4;
                  }
                  else
                     col = col_save;
               }
            }
            
            if (!valid_color) {
               if(!col.read(parts[0], errmsg))
                  error(errmsg, c);
            }

            unsigned int conv_elems = 15;
            if(parts_sz>next_parms_idx) {
               if(strspn(parts[next_parms_idx], "lcvh") !=
                                            strlen(parts[next_parms_idx]))
                  error(msg_str("elements to map are '%s' must be l, c, v or h",
                           parts[next_parms_idx]), c);
               conv_elems = 8*(strchr(parts[next_parms_idx], 'h')!=0) +
                            4*(strchr(parts[next_parms_idx], 'v')!=0) +
                            2*(strchr(parts[next_parms_idx], 'c')!=0) +
                            1*(strchr(parts[next_parms_idx], 'l')!=0);
            }
            
            int opq = col[3];
            if(parts_sz>next_parms_idx+1)
               opq = atoi(parts[next_parms_idx+1]);
               
            if (opq < 0 || opq > 255)
               error("transparency value must be between 0 and 255", c);

            for(int i=0; i<4; i++) {
               if(conv_elems & (1<<i)) {
                  if (!col.is_set())
                     face_col[i] = col_val();
                  else
                     face_col[i] = col_val(col[0],col[1],col[2],opq);
               }
            }
            break;
         }

         case 'T':
            if(!read_int(optarg, &face_opacity, errmsg))
               error(errmsg, c);
            if(face_opacity < 0 || face_opacity > 255) {
               error("face transparency must be between 0 and 255", c);
            }
            break;

         case 'Z':
            if(!cent_col.read(optarg, errmsg))
               error(errmsg, c);
            break;
            
         case 'K':
            append_container = true;
            break;

         case 'O':
            trans_to_origin = true;
            break;

         case 'R':
            if(strlen(optarg) > 1 || !strchr("ofOF", *optarg))
               error("convex hull arg is '"+string(optarg)+"' must be o, f, O or F", c);
            if(strchr("o", *optarg))
               r_lattice_type = 1;
            else
            if(strchr("f", *optarg))
               r_lattice_type = 2;
            else
            if(strchr("O", *optarg))
               r_lattice_type = 3;
            else
            if (strchr("F", *optarg))
               r_lattice_type = 4;
            break;

         case 'B':
            list_bravais = true;
            break;

         case 'Q':
            if(strlen(optarg)==1) {
               if(strchr("co", *optarg))
                  list_radii_original_center = *optarg;
               else
                  error("invalid option", c);
            }
            else
            if(!list_radii_center.read(optarg, errmsg))
               error(errmsg, c);
            break;

         case 'L':
            if(strlen(optarg) > 1 || !strchr("fv", *optarg))
               error("list radii arg is '"+string(optarg)+"' must be f or v", c);
            if(strchr("f", *optarg))
               list_radii = 1;
            else
            if(strchr("v", *optarg))
               list_radii = 2;
            break;

         case 'S':
            if(strlen(optarg) > 1 || !strchr("fv", *optarg))
               error("list struts arg is '"+string(optarg)+"' must be f or v", c);
            if(strchr("f", *optarg))
               list_struts = 1;
            else
            if(strchr("v", *optarg))
               list_struts = 2;
            break;

         case 'o':
            ofile = optarg;
            break;

         default:
            error("unknown command line error");
      }
   }
   
   if(argc==optind && !list_bravais)
      error("no lattice specified", "lattice");

   if(argc-optind > 2) {
      error("too many arguments");
      exit(1);
   }

   if(optind < argc)
      crystal_system = argv[optind++];
   if(optind < argc)
      centering = argv[optind++];

   transform(crystal_system.begin(), crystal_system.end(), crystal_system.begin(), ::tolower);
   transform(centering.begin(), centering.end(), centering.begin(), ::tolower);

   // don't let centering in when number is entered
   double dummy;
   if(read_double(crystal_system.c_str(),&dummy,errmsg)) {
      if (centering.length())
         error("too many arguments");
   }

   // have to catch this too
   if (crystal_system == "sc" || crystal_system == "fcc" || crystal_system == "bcc") {
      if (centering.length())
         error("too many arguments");
   }
   
   if (grid_for_radius && !auto_grid_type)
      error("-G required with -g a");
      
   if (radius_set and radius_by_coord.is_set())
      error("-p cannot be used with -r");

   if (container == 's' && cfile.length())
      error("-c and -k cannot be specified together");
      
   if (append_container && !cfile.length())
      error("container can only be appended if one is provided with -k", 'K');

   if (radius_default == 'k' && !cfile.length())
      error("-r k can only be used if -k container is specified",'r');

   if (container == 'c' && !cfile.length() &&
         (radius != 0 || offset.is_set() || radius_by_coord.is_set()))
      warning("cubic container in use. Radius parameters ignored", 'c');
      
   if (use_centering_for_dual || prim_vec_idxs.size())
      dual_lattice = true;

   if (dual_lattice && r_lattice_type > 0)
      error("hex relation struts cannot be done on duals", 'R');

   if (list_radii && list_struts)
      error("cannot list radii and struts at the same time");

   epsilon = (sig_compare != INT_MAX) ? pow(10, -sig_compare) : ::epsilon;
}

bravais::bravais()
{ 
   bravais_items = bravais_item_list;
   last_bravais = sizeof (bravais_item_list) / sizeof (bravais_item_list[0]);
}

string bravais::get_crystal_system(int i)
{
   return bravais_items[i].crystal_system;
}

string bravais::get_centering(int i)
{
   return bravais_items[i].centering;
}

void bravais::list_bravais(int idx, FILE *fp)
{
   fprintf(fp, "%2d)\t%-s %s\n",
      idx+1, bravais_items[idx].crystal_system, bravais_items[idx].centering);
}

void bravais::list_bravais(FILE *fp)
{
   for(int i=0; i<last_bravais; i++)
      list_bravais(i, fp);
}

int bravais::lookup_sym_no(string crystal_system, string centering)
{  
   // everything expects lower case
   transform(crystal_system.begin(), crystal_system.end(), crystal_system.begin(), ::tolower);
   transform(centering.begin(), centering.end(), centering.begin(), ::tolower);

   // is it blank
   if(!crystal_system.length())
      return -1;
   
   // is it the list order number
   char *endptr;
   int idx = strtol(crystal_system.c_str(), &endptr, 10);
   if(!*endptr)     // all of string is an integer
      return idx-1;

   idx= -1;

   // is it a partial name
   for(int i=0; i<last_bravais; i++) {
      string name = get_crystal_system(i);
      transform(name.begin(), name.end(), name.begin(), ::tolower);
      if(strncmp(crystal_system.c_str(), name.c_str(), crystal_system.size())==0) {
         crystal_system = get_crystal_system(i);
         break;
      }
   }

   // is it rhombohedral or isometric
   if(strncmp(crystal_system.c_str(), "rhombohedral", crystal_system.size())==0)
      crystal_system = "trigonal";
   else
   if(strncmp(crystal_system.c_str(), "isometric", crystal_system.size())==0)
      crystal_system = "cubic";

   // well known synonyms
   if(crystal_system == "sc") {
      crystal_system = "cubic";
      centering = "p";
   }
   else
   if(crystal_system == "fcc") {
      crystal_system = "cubic";
      centering = "f";
   }
   else
   if(crystal_system == "bcc") {
      crystal_system = "cubic";
      centering = "i";
   }

   // default centering is simple
   if(!centering.length())
      centering = "p";

   string full_name = crystal_system + centering;
   
   // is it a lattice name
   transform(full_name.begin(), full_name.end(), full_name.begin(), ::tolower);
   for(int i=0; i<last_bravais; i++) {
      string name = get_crystal_system(i) + get_centering(i);
      transform(name.begin(), name.end(), name.begin(), ::tolower);
      if(full_name==name)
         return i;
   }

   return idx;
}

// formulas from http://en.wikipedia.org/wiki/Bravais_lattice
// crystal_system is not changed
double bravais_volume(string crystal_system, const vector<double> &vecs, const vector<double> &angles)
{
   transform(crystal_system.begin(), crystal_system.end(), crystal_system.begin(), ::tolower);

   double volume = 0;
   // find crystal system
   if ( crystal_system == "triclinic" ) {
      // a * b * c * sqrt(1-cos(alpha)^2-cos(beta)^2-cos(gamma)^2+2*cos(alpha)*cos(beta)*cos(gamma))
      double cos_a = cos(angles[0]*M_PI/180);
      double cos_b = cos(angles[0]*M_PI/180);
      double cos_g = cos(angles[0]*M_PI/180);
      double tmp = 1-cos_a*cos_a-cos_b*cos_b-cos_g*cos_g+2*cos_a*cos_b*cos_g;
      volume = vecs[0]*vecs[1]*vecs[2] * sqrt(tmp);
   }
   else
   if ( crystal_system == "monoclinic" || crystal_system == "hexagonal" ) {
      double a = 0;
      for(unsigned int i=0; i<3; i++) {
         if (angles[i] != 90) {
            a = angles[i];
            break;
         }
      }
      volume = vecs[0]*vecs[1]*vecs[2] * sin(a*M_PI/180); // a * b * c * sin(alpha)
   }
   else
   if ( crystal_system == "orthorhombic" )
      volume = vecs[0]*vecs[1]*vecs[2]; // a * b * c
   else
   if ( crystal_system == "tetragonal" )
      volume = vecs[0]*vecs[0]*vecs[2]; // a^2 * c
   else
   if ( crystal_system == "trigonal" ) {
      // a^3 * sqrt(1-3*cos(alpha)^2+2*cos(alpha)^3)
      double cos_a = cos(angles[0]*M_PI/180);
      double tmp = 1-3*cos_a*cos_a+2*cos_a*cos_a*cos_a;
      volume = vecs[0]*vecs[0]*vecs[0] * sqrt(tmp);
      }
   else
   // incorrect
   //if ( crystal_system == "hexagonal" )
   //   volume = (3.0*pow(3.0,(1.0/3.0))*vecs[0]*vecs[0]*vecs[2])/2; // (3*cube_rt(3)*a^2*c)/2
   //else
   if ( crystal_system == "cubic" )
      volume = vecs[0]*vecs[0]*vecs[0]; // a^3
   else {
      fprintf(stderr,"error in bravais_volume: unknown crystal system: %s\n", crystal_system.c_str());
   }

   return volume;
}

// return random angle
double bravais_random_angle(rand_gen &ran, const double &max_angle)
{
   return ran.ran_in_range_exclude_end(0.001, max_angle); // avoid 0 and max_angle
}

// sort the three elements without altering original order
void sort_three(double &x, double &y, double &z, vector<double> v)
{
   sort( v.begin(), v.end() );

   x = v[0];
   y = v[1];
   z = v[2];
}

int bravais_check(string &crystal_system, string &centering, vector<double> &vecs, vector<double> &angles, const int &strictness, bool verbose)
{
// strictness = 0 any change in crystal system allowed
// strictness = 1 only upgrades in crystal system allowed
// strictness = 2 no change in crystal system allowed

// Crystal System       Vector Constraints      Angle Constraints
// Triclinic        (1) doesn't matter      (1) alpha <> beta <> gamma
//                                          (1) alpha = beta <> 90 <> gamma
//                  (2) a <> b <> c         (4) alpha = beta = gamma <> 90 (special case)
//                  (3) a = b <> c          (4) alpha = beta = gamma <> 90 (special case)
// Monoclinic       (1) doesn't matter      (2) alpha = gamma = 90 <> beta
//                  (2) a <> b <> c         (5) alpha = beta = 90, gamma = 120 (special case)
// Orthorhombic     (2) a <> b <> c         (3) alpha = beta = gamma = 90
// Tetragonal       (3) a = b <> c          (3) alpha = beta = gamma = 90
// Trigonal         (4) a = b = c           (4) alpha = beta = gamma <> 90
// Note on Hexagonal: odd vector position must line up with odd angle position, else monoclinic
// Hexagonal        (3) a = b <> c          (5) alpha = beta = 90, gamma = 120 (or 60)
//                  (4) a = b = c           (5) alpha = beta = 90, gamma = 120 (or 60)
// Cubic            (4) a = b = c           (3) alpha = beta = gamma = 90

   // load makeshift "database"
   vector<string> crystal_systems;
   crystal_systems.push_back("unknown");
   crystal_systems.push_back("triclinic");
   crystal_systems.push_back("monoclinic");
   crystal_systems.push_back("orthorhombic");
   crystal_systems.push_back("tetragonal");
   crystal_systems.push_back("trigonal");
   crystal_systems.push_back("hexagonal");
   crystal_systems.push_back("cubic");

   vector<string> vec_cases;
   vec_cases.push_back("unknown vector case");
   vec_cases.push_back("no constraints");
   vec_cases.push_back("a <> b <> c");
   vec_cases.push_back("a = b <> c");
   vec_cases.push_back("a = b = c");

   vector<string> angle_cases;
   angle_cases.push_back("unknown angle case");
   angle_cases.push_back("anything not in higher symmetries");
   angle_cases.push_back("alpha = gamma = 90 <> beta");
   angle_cases.push_back("alpha = beta = gamma = 90");
   angle_cases.push_back("alpha = beta = gamma <> 90");
   angle_cases.push_back("alpha = beta = 90, gamma = 120");

   vector<int> vec_rule;
   vec_rule.push_back(0);
   vec_rule.push_back(1);
   vec_rule.push_back(1);
   vec_rule.push_back(2);
   vec_rule.push_back(3);
   vec_rule.push_back(4);
   vec_rule.push_back(3);
   vec_rule.push_back(4);

   vector<int> angle_rule;
   angle_rule.push_back(0);
   angle_rule.push_back(1);
   angle_rule.push_back(2);
   angle_rule.push_back(3);
   angle_rule.push_back(3);
   angle_rule.push_back(4);
   angle_rule.push_back(5);
   angle_rule.push_back(3);

   // in case function is called with upper case lettering
   transform(crystal_system.begin(), crystal_system.end(), crystal_system.begin(), ::tolower);
   transform(centering.begin(), centering.end(), centering.begin(), ::tolower);

   // find chosen crystal system
   int crystal_system_index = 0;
   for(int i=1; i<=7; i++) {
      if ( crystal_system == crystal_systems[i] ) {
         crystal_system_index = i;
         break;
      }
   }

   // in case of misspellings
   if ( !crystal_system_index ) {
      fprintf(stderr,"error in bravais_check: unknown crystal system: %s\n", crystal_system.c_str());
      return 0;
   }

   double a = 0;
   double b = 0;
   double c = 0;

   // if vecs not set, give valid values
   if(!vecs.size()) {
      // hexagonal case gets bumped up to all equal vectors
      if (vec_rule[crystal_system_index] == 4 || crystal_system_index == 6) {
         a = 2;
         b = 2;
         c = 2;
      }
      else
      if (vec_rule[crystal_system_index] == 3) {
         a = 2;
         b = 2;
         c = 4;
      }
      else
      if (vec_rule[crystal_system_index] == 2 || vec_rule[crystal_system_index] == 1) {
         a = 2;
         b = 3;
         c = 4;
      }

      // if simple cubic then space them 1 apart instead
      if ( centering == "p" ) {
         a /= 2.0;
         b /= 2.0;
         c /= 2.0;
      }

      vecs.push_back(a);
      vecs.push_back(b);
      vecs.push_back(c);
   }

   if (verbose)
      fprintf(stderr,"info: vector a = %g, b = %g, c = %g\n", vecs[0],vecs[1],vecs[2]);

   double alpha = 0;
   double beta = 0;
   double gamma = 0;

   if (angles.size())
      sort_three(alpha,beta,gamma,angles);

   // fixed angle cases, otherwise if not given, random angles assigned
   if (angle_rule[crystal_system_index] == 3) {
      if ((alpha && alpha != 90.0) || (beta && beta != 90.0) || (gamma && gamma != 90.0)) 
         fprintf(stderr,"warning: for crystal system %s, alpha, beta and gamma must be 90 degrees. -a ignored\n",crystal_system.c_str());
      alpha = 90;
      beta = 90;
      gamma = 90;
      angles.clear();
      angles.push_back(alpha);
      angles.push_back(beta);
      angles.push_back(gamma);
   }
   else
   // hexagonal, 120 can be in any position (alpha, beta, gamma are sorted order)
   if (angle_rule[crystal_system_index] == 5 && !(alpha == 90 && beta == 90 && gamma == 120)) {
      fprintf(stderr,"warning: for crystal system %s, of alpha, beta and gamma, two values must be 90 degrees and one value 120 degrees. -a ignored\n",crystal_system.c_str());
      alpha = 90;
      beta = 90;
      gamma = 120;
      angles.clear();
      angles.push_back(alpha);
      angles.push_back(beta);
      angles.push_back(gamma);
   }
   else
   // angles can be set to random values for other crystal systems
   if (!angles.size()) {
      rand_gen ran;
      ran.time_seed();
      
      if (crystal_system == "monoclinic") {
         alpha = 90.0;
         beta = 90.0;
         gamma = 90.0;
         // don't let beta equal 90
         while(beta == 90.0)
            beta = bravais_random_angle(ran, 180.0);
      }
      else if (crystal_system == "trigonal") {
         alpha = bravais_random_angle(ran, 60.0);
         beta = alpha;
         gamma = alpha;
      }
      else if (crystal_system == "triclinic") {
         alpha = 360.0;
         beta = 360.0;
         gamma = 360.0;
         while(alpha+beta+gamma >= 360 || gamma >= alpha+beta || beta >= alpha+gamma || alpha >= beta+gamma) {
            alpha = bravais_random_angle(ran, 180.0);
            beta = bravais_random_angle(ran, 180.0);
            gamma = bravais_random_angle(ran, 180.0);
         }
      }

      angles.push_back(alpha);
      angles.push_back(beta);
      angles.push_back(gamma);

      fprintf(stderr,"warning: RANDOM angles generated\n");
   }

   if (verbose)
      fprintf(stderr,"info: angles alpha = %5.5g, beta = %5.5g, gamma = %5.5g\n", angles[0],angles[1],angles[2]);

   // for evaulation, sort vector lengths and angles, low to high (if parameters are set)
   sort_three(a,b,c,vecs);
   sort_three(alpha,beta,gamma,angles);

   int vec_case = 0;
   if ( a != b && b != c )
      vec_case = 2;
   else
   if ( (a == b && b != c) || (a != b && b == c) ) // e.g. 1,1,2 or 1,2,2
      vec_case = 3;
   else
   if ( a == b && b == c )
      vec_case = 4;

   int angle_case = 0;
   if ((alpha == 90 && beta == 90 && gamma == 120) || (alpha == 60 && beta == 90 && gamma == 90))
      angle_case = 5;
   else
   if (alpha == 90 && beta == 90 && gamma == 90)
      angle_case = 3;
   else
   if ((alpha == 90 && beta == 90 && gamma != 90) || (alpha != 90 && beta == 90 && gamma == 90)) // e.g. 90,90,100 or 45,90,90
      angle_case = 2;
   else
   if (alpha == beta && beta == gamma && alpha != 90 && beta != 90 && gamma != 90)
      angle_case = 4;
   else
// anything else will base case 1
      angle_case = 1;
      
   // odd vector position must line up with odd angle position, else monoclinic
   if ( ( vec_case == 3 || vec_case == 4 ) && angle_case == 5 ) {
      int odd_angle = 0;
      for(;odd_angle<3;odd_angle++) {
         if (angles[odd_angle] != 90.0)
            break;
      }
      if (vecs[(odd_angle+1)%3] != vecs[(odd_angle+2)%3]) {
         angle_case = 2;
         vec_case = 1;
      }
   }

   // find actual crystal system by vec and angle input
   int csystem = 0;
   // triclinic
   if ( ( angle_case == 1 ) || // vec_case can be any
        ( vec_case == 2 && angle_case == 4 ) ||
        ( vec_case == 3 && angle_case == 4 ) ) { // have to allow for triclinic with unequal vectors
      csystem = 1;
      vec_case = 1;
   }
   else
   // monoclinic
   if ( ( angle_case == 2 ) || // vec_case can be any
        ( vec_case == 2 && angle_case == 5 ) ) { // have to allow for hexagonal angles but three unequal vectors
     csystem = 2;
     vec_case = 1;
     angle_case = 2; // in case hex angles were present
   }
   else
   // orthorhombic
   if ( vec_case == 2 && angle_case == 3 )
     csystem = 3;
   else
   // tetragonal
   if ( vec_case == 3 && angle_case == 3 )
     csystem = 4;
   else
   // trigonal
   if ( vec_case == 4 && angle_case == 4 )
     csystem = 5;
   else
   // hexagonal
   if ( ( vec_case == 3 || vec_case == 4 ) && angle_case == 5 )
     csystem = 6;
   else
   // cubic
   if ( vec_case == 4 && angle_case == 3 )
     csystem = 7;

   if ( crystal_system_index != csystem ) {
      fprintf(stderr,"\n");
      fprintf(stderr,"warning in bravais_check: MISMATCH ...\n\n");
      fprintf(stderr,"asked for '%s' which requires:\n", crystal_systems[crystal_system_index].c_str());
      fprintf(stderr,"vector lengths: '%s'\n", vec_cases[vec_rule[crystal_system_index]].c_str());
      fprintf(stderr,"angles: '%s'\n", angle_cases[angle_rule[crystal_system_index]].c_str());
      fprintf(stderr,"\nbut instead found '%s' which has:\n", crystal_systems[csystem].c_str());
      fprintf(stderr,"vector lengths: '%s'\n", vec_cases[vec_case].c_str());
      fprintf(stderr,"angles: '%s'\n\n", angle_cases[angle_case].c_str());
   }

   // this should not be able to happen
   if ( csystem == 0 ) {
      fprintf(stderr,"error in bravais_check: unknown crystal system case\n");
      return 0;
   }

   // check if system change occured. if strict is other than zero, may return error state
   if (csystem != crystal_system_index) {
      // have to update crystal system name for other functioning
      crystal_system = crystal_systems[csystem];

      if (strictness == 1 && csystem < crystal_system_index) {
         fprintf(stderr,"error in bravais_check: crystal system downgrade disallowed. cannot continue\n");
         return 0;
      }
      else
      if (strictness == 2) {
         fprintf(stderr,"error in bravais_check: strict type matching in force. cannot continue\n");
         return 0;
      }
   }

   // only "monoclinic" or "orthorhombic" can have Base Centering (C)
   if ( csystem != 2 && csystem != 3 && centering == "c" ) {
      fprintf(stderr,"bravais_check: warning: %s cannot have Base Centering (C)\n",crystal_systems[csystem].c_str());
      if (strictness > 0) {
         fprintf(stderr,"bravais_check: error: crystal system downgrade disallowed. cannot continue\n");
         return 0;
      }
      else {
         fprintf(stderr,"bravais_check: warning: downgrade to Simple Centering (P)\n");
         centering = "p";
      }
   }

   return 1;
}

void bravais_centering_p(col_geom_v &geom)
{
   geom.add_vert(vec3d( 1,  1,  1)); // 0
   geom.add_vert(vec3d( 1,  1, -1)); // 1
   geom.add_vert(vec3d( 1, -1,  1)); // 2
   geom.add_vert(vec3d( 1, -1, -1)); // 3
   geom.add_vert(vec3d(-1,  1,  1)); // 4
   geom.add_vert(vec3d(-1,  1, -1)); // 5
   geom.add_vert(vec3d(-1, -1,  1)); // 6
   geom.add_vert(vec3d(-1, -1, -1)); // 7
}

void bravais_centering_a(col_geom_v &geom)
{
   bravais_centering_p(geom);

   geom.add_vert(vec3d( 1,  0,  0)); // 8
   geom.add_vert(vec3d(-1,  0,  0)); // 9
}

void bravais_centering_b(col_geom_v &geom)
{
   bravais_centering_p(geom);

   geom.add_vert(vec3d( 0,  1,  0)); // 8
   geom.add_vert(vec3d( 0, -1,  0)); // 9
}

void bravais_centering_c(col_geom_v &geom)
{
   bravais_centering_p(geom);

   geom.add_vert(vec3d( 0,  0,  1)); // 8
   geom.add_vert(vec3d( 0,  0, -1)); // 9
}

void bravais_centering_f(col_geom_v &geom)
{
   bravais_centering_p(geom);

   geom.add_vert(vec3d( 1,  0,  0)); // 8
   geom.add_vert(vec3d(-1,  0,  0)); // 9
   geom.add_vert(vec3d( 0,  1,  0)); // 10
   geom.add_vert(vec3d( 0, -1,  0)); // 11
   geom.add_vert(vec3d( 0,  0,  1)); // 12
   geom.add_vert(vec3d( 0,  0, -1)); // 13
}

void bravais_centering_i(col_geom_v &geom)
{
   bravais_centering_p(geom);
   geom.add_vert(vec3d( 0,  0,  0)); // 8
}

void bravais_cell_struts(col_geom_v &geom, const col_val &edge_col)
{
   int f[] = { 0,1, 0,2, 0,4, 3,1, 3,2, 3,7, 5,1, 5,4, 5,7, 6,2, 6,4, 6,7 };
   vector<int> edge(2);
   for(int i=0; i<12; i++) {
      for(int j=0; j<2; j++)
         edge[j] = f[i*2+j];
      geom.add_col_edge(edge, edge_col);
   }
}

void bravais_cell(col_geom_v &geom, const string &centering, const bool &cell_struts, const col_val &vert_col, const col_val &edge_col)
{
   if (centering == "p")
      bravais_centering_p(geom);
   else
   if (centering == "a")
      bravais_centering_a(geom);
   else
   if (centering == "b")
      bravais_centering_b(geom);
   else
   if (centering == "c")
      bravais_centering_c(geom);
   else
   if (centering == "f")
      bravais_centering_f(geom);
   else
   if (centering == "i")
      bravais_centering_i(geom);

   if (vert_col.is_set()) {
      coloring vc(&geom);
      vc.v_one_col(vert_col);
   }
   
   if (cell_struts)
      bravais_cell_struts(geom, edge_col);
}

void bravais_scale(col_geom_v &geom, const vector<double> &vecs, const bool &inverse)
{
   // divide by 2 since original scale is 2   
   mat3d m = mat3d::scale(vecs[0]/2, vecs[1]/2, vecs[2]/2);
   if (inverse)
      m.set_inverse();
      
   geom.transform(m);
}

// applies angles alpha,beta and gamma without changing scale
void bravais_warp(col_geom_v &geom, const vector<double> &angles, const bool &inverse)
{
   double yz = deg2rad(angles[0]);   // alpha
   double zx = deg2rad(angles[1]);   // beta
   double xy = deg2rad(angles[2]);   // gamma
   
   mat3d m = mat3d::trans_by_angles(yz, zx, xy);
   if (inverse)
      m.set_inverse();

   geom.transform(m);
}

// vecs is not changed
double bravais_radius(const vector<int> &grid, vector<double> vecs, const vector<double> &angles, const char &radius_default)
{
   col_geom_v tgeom;
   tgeom.read_resource("std_cube");

   // compensate for grid size
   for(unsigned int i=0; i<3; i++)
      vecs[i] *= grid[i];

   bravais_scale(tgeom, vecs, false);
   bravais_warp(tgeom, angles, false);

   return lattice_radius(tgeom,radius_default);
}

double bravais_auto_grid_size(const double &radius, const vector<double> &vecs, const vector<double> &angles)
{
   vector<int> grid(3);
   for(unsigned int i=0; i<3; i++)
      grid[i] = 1;
   return (radius / bravais_radius(grid, vecs, angles, 's'));
}

// radius_by_coord is not changed
double bravais_radius_by_coord(vec3d radius_by_coord, const vec3d &offset, const vector<double> &vecs, const vector<double> &angles)
{
   col_geom_v tgeom;
   tgeom.add_vert(radius_by_coord);

   // do same scale and warp as happened to lattice
   bravais_scale(tgeom, vecs, false);
   bravais_warp(tgeom, angles, false);

   const vector<vec3d> &tverts = tgeom.verts();
   radius_by_coord = tverts[0];
//radius_by_coord.dump("radius_by_coord");

   vec3d cent = vec3d(0,0,0);
   if (offset.is_set())
      cent += offset;

   return( (cent-radius_by_coord).mag() );
}

// makes local copy of tgeom
void geom_to_grid_translate(col_geom_v &geom, col_geom_v tgeom, const mat3d &transl_matrix)
{
      tgeom.transform(transl_matrix);
      geom.append(tgeom);
}

void geom_to_grid(col_geom_v &geom, const vector<int> &grid, const vector<double> &cell_size, const double &eps)
{
   col_geom_v tgeom = geom;
   geom.clear_all();

   for(int x=0; x<grid[0]; x++) {
      for(int y=0; y<grid[1]; y++) {
         for(int z=0; z<grid[2]; z++) {
            geom_to_grid_translate(geom, tgeom, mat3d::transl(vec3d(cell_size[0]*x, cell_size[1]*y, cell_size[2]*z))); 
         }
      }
   }

   sort_merge_elems(geom, "vef", eps);
}

void bravais_eighth_cell_grid(col_geom_v &geom)
{
   const vector<vec3d> &verts = geom.verts();
   
   vector<int> del_verts;
   for(unsigned int i=0; i<verts.size(); i++) {
      for(unsigned int j=0; j<3; j++) {
         if (verts[i][j] < 0) {
            del_verts.push_back(i);
            break;
         }
      }
   }

   if (del_verts.size())
      geom.delete_verts(del_verts);
}

mat3d r_lattice_trans_mat(const bool &inverse)
{
   mat3d trans_m;
   // transformation matrix by Adrian Rossiter
   // off_trans -X 1,1,1,0,-1,1,-1,0,1
   trans_m[0]=1;  trans_m[1]=1;  trans_m[2]=1;
   trans_m[4]=0;  trans_m[5]=-1; trans_m[6]=1;
   trans_m[8]=-1; trans_m[9]=0;  trans_m[10]=1;
   
   /*
   // inverse of above matrix will be like this below
   // off_trans -X 1/3,1/3,-2/3,1/3,-2/3,1/3,1/3,1/3,1/3
   double one_third = 1.0/3.0;
   double two_thirds = 2.0/3.0;
   trans_m[0]=one_third; trans_m[1]=one_third;  trans_m[2]=-two_thirds;
   trans_m[4]=one_third; trans_m[5]=-two_thirds; trans_m[6]=one_third;
   trans_m[8]=one_third; trans_m[9]=one_third;  trans_m[10]=one_third;
   */
   
   if (inverse)
      trans_m.set_inverse();
      
   return trans_m;
}

void r_lattice_overlay(col_geom_v &geom, const vector<int> &grid, const vector<double> &cell_size, const col_val &vert_col, const col_val &edge_col, const double &eps)
{
   col_geom_v hgeom;

   bravais_cell(hgeom, "p", true, vert_col, edge_col); // hard code struts and colors for the "R" lattice
   geom_to_grid(hgeom, grid, cell_size, eps);

   hgeom.transform(r_lattice_trans_mat(false));

   // translate hexagonal onto cube
   vec3d adjust = vec3d(0,0,0);
   if (!is_even(grid[1]))
      adjust += vec3d(0,1,0);
   if (!is_even(grid[0]))
      adjust += vec3d(0,0,1);
   hgeom.transform(mat3d::transl(-centroid(hgeom.verts())+centroid(geom.verts())-adjust));

   geom.append(hgeom);
}

void bravais_grid_type(vector<int> &grid, const char &auto_grid_type)
{
   string pattern; // 0 - even 1 - odd
   if (auto_grid_type == 'p')
      pattern = "000";
   else
   if (auto_grid_type == 'i')
      pattern = "111";
   else
   if (auto_grid_type == 'e')
      pattern = "010";
   else
   if (auto_grid_type == 'f')
      pattern = "101";

   for(unsigned int i=0; i<3; i++) {
      if ((pattern[i] == '0' && !is_even(grid[i])) || (pattern[i] == '1' && is_even(grid[i])))
         grid[i]++;
   }
}

void bravais_primitive_vectors(vector<vec3d> &primitive_vectors, const string &centering)
{
   double len = sqrt(2);
   primitive_vectors.push_back(vec3d(0,0,0));
   if(centering == "p") {
      primitive_vectors.push_back(vec3d(2,0,0));
      primitive_vectors.push_back(vec3d(0,2,0));
      primitive_vectors.push_back(vec3d(0,0,2));
   }
   else
   if (strchr("fcba", *(centering.c_str()))) {
      primitive_vectors.push_back(vec3d(len,len,0));
      primitive_vectors.push_back(vec3d(len,0,len));
      primitive_vectors.push_back(vec3d(0,len,len));
   }
   else
   if(centering == "i") {
      primitive_vectors.push_back(vec3d(len,len,-len));
      primitive_vectors.push_back(vec3d(len,-len,len));
      primitive_vectors.push_back(vec3d(-len,len,len));
   }
}

double tetrahedral_volume(const vec3d &a0, const vec3d &a1, const vec3d &a2, const vec3d &a3)
{
   return -vtriple(a1-a0, a2-a0, a3-a0)/factorial(3);
}

void bravais_dual(col_geom_v &geom, const vector<vec3d> &primitive_vectors, const vector<int> &prim_vec_idxs,
                  const vector<double> &vecs, const vector<double> &angles, const double &eps)
{
   col_geom_v tgeom;

   if (primitive_vectors.size()) {
      for(unsigned int i=0; i<4; i++)
         tgeom.add_vert(primitive_vectors[i]);
   }
   else {
      const vector<vec3d> &verts = geom.verts();
      for(unsigned int i=0; i<4; i++) {
         if (prim_vec_idxs[i] >= (int)verts.size()) {
            fprintf(stderr,"error in bravais_dual: vertex %d does not exist. cannot perform dual\n", prim_vec_idxs[i]);
            return;
         }
         tgeom.add_vert(verts[prim_vec_idxs[i]]);
      }

      //for(unsigned int i=0; i<4; i++) {
      //   tgeom.add_vert(verts[prim_vec_idxs[i]]);
      //}
   }

   // changing orientation
   //if ( vecs[0] != vecs[1] )
   //   swap( vecs[0], vecs[1] );
   //vecs[2] = sqrt( vecs[0] * vecs[1] * sin(angles[2]*M_PI/180) );

   // do same scale and warp as happened to lattice
   bravais_scale(tgeom, vecs, false);
   bravais_warp(tgeom, angles, false);

   const vector<vec3d> &tverts = tgeom.verts();
   vec3d a0 = tverts[0]; // radial vector
   vec3d a1 = tverts[1];
   vec3d a2 = tverts[2];
   vec3d a3 = tverts[3];

   double vecs_volume = tetrahedral_volume(a0,a1,a2,a3);
   if (double_eq(vecs_volume, 0, eps)) {
      fprintf(stderr,"error in bravais_dual: vectors are on one plane. cannot perform dual\n");
      return;
   }

   double det = vtriple(a1, a2, a3);
 
   vec3d b1 = vcross(a2, a3) / det;
   vec3d b2 = vcross(a3, a1) / det;
   vec3d b3 = vcross(a1, a2) / det;

   mat3d m;
   m = mat3d(b1, b2, b3);
   m.transpose();

   // if the dual lattice is desired to always have the spacing of the original
   // uncomment the code below and and at restore scale
   // store size to restore scale after dual
   //geom_info info(geom);
   //double val = 1;
   //if (info.num_iedges() > 0)
      //double val = info.iedge_lengths().sum/info.num_iedges();

   // undo angles and vectors of the input geom
   bravais_warp(geom, angles, true);   
   bravais_scale(geom, vecs, true);
   
   // apply dual transform
   geom.transform(m);

   // restore scale
   //geom.transform(mat3d::scale(1/val));
}

void do_bravais(col_geom_v &geom, col_geom_v &container, brav_opts &opts)
{      
   int strictness = 0;
   if (!bravais_check(opts.crystal_system, opts.centering, opts.vecs, opts.angles, strictness, opts.verbose))
      return;

   // if hexagonal/cubic fill option, force struts and colors for cage
   bool struts = (opts.r_lattice_type == 2 || opts.r_lattice_type == 4) ? true : opts.cell_struts;
   col_val vertex_col = (opts.r_lattice_type == 2 || opts.r_lattice_type == 4) ? opts.vert_col[3] : opts.vert_col[0];
   col_val edge_col = (opts.r_lattice_type == 2 || opts.r_lattice_type == 4) ? opts.edge_col[3] : opts.edge_col[0];

   // for duals, force to primitive centering
   string centering = opts.dual_lattice ? "p" : opts.centering;
   bravais_cell(geom, centering, struts, vertex_col, edge_col);

   // if hexagonal/cubic fill option, fill cell
   if (opts.r_lattice_type == 2 || opts.r_lattice_type == 4) {
      double one_third = 1.0/3.0;
      geom.add_col_vert(vec3d( one_third, one_third,-one_third), opts.vert_col[0]);
      geom.add_col_vert(vec3d(-one_third,-one_third, one_third), opts.vert_col[0]);
   }

   // correct for cell size for simple lattices
   if (opts.radius_by_coord.is_set()) {
      //if (opts.auto_grid_type == '8')
      //   opts.radius_by_coord *= sqrt(0.5);
      //else
      if (opts.centering == "p")
         opts.radius_by_coord *= 2.0;
   }

   // find automatic grid size
   if (opts.grid_for_radius) {
      double radius = opts.radius;
      if (opts.radius_default == 'k')
         radius = lattice_radius(container,opts.radius_default);
      else
      if (opts.radius_by_coord.is_set()) {
         vector<double> tangles(3,90);
         radius = bravais_radius_by_coord(opts.radius_by_coord, opts.offset, opts.vecs, tangles);
      }

      radius = bravais_auto_grid_size(radius, opts.vecs, opts.angles);

      // grid big enough for radius
      if (radius != 0) {
         for(unsigned int i=0; i<3; i++)
            opts.grid.push_back((int)ceil(radius));
      }
   }

   // if grid size is still not set
   if (!opts.grid.size()) {
      // make grid big enough for most default spherical grids, when radius has not been set.
      if (opts.container == 's') {
         int rep = (opts.centering == "p") ? 6 : 3;
         for(unsigned int i=0; i<3; i++)
            opts.grid.push_back(rep);
      }
      // default grid size
      else {
         for(unsigned int i=0; i<3; i++)
            opts.grid.push_back(2);
      }
   }

   if (opts.auto_grid_type)
      bravais_grid_type(opts.grid, opts.auto_grid_type);

   // original cell size is 2x2x2
   vector<double> cell_size(3,2.0);

   geom_to_grid(geom, opts.grid, cell_size, opts.epsilon);

   if (opts.r_lattice_type == 1 || opts.r_lattice_type == 3) {
      r_lattice_overlay(geom, opts.grid, cell_size, opts.vert_col[3], opts.edge_col[3], opts.epsilon);
      if (opts.r_lattice_type == 3)
         geom.transform(r_lattice_trans_mat(true));
   }

   if (opts.auto_grid_type == '8')
      bravais_eighth_cell_grid(geom);
   else
      // translate grid into the positive realm
      geom.transform(mat3d::transl(vec3d(1,1,1)));

   if (opts.verbose) {
      fprintf(stderr,"info: grid size is %d x %d x %d",opts.grid[0], opts.grid[1], opts.grid[2]);
      if (opts.auto_grid_type == '8')
         fprintf(stderr," (clipped to: %d.5 x %d.5 x %d.5)",opts.grid[0]-1, opts.grid[1]-1, opts.grid[2]-1);
      fprintf(stderr,"\n");
   }

   // if hexagonal/cubic fill option, position for true scale and warp
   if (opts.r_lattice_type == 2 || opts.r_lattice_type == 4) {
      vector<double> vecs(3);
      vecs[0] = sqrt(2);
      vecs[1] = sqrt(3);
      vecs[2] = sqrt(2);
      bravais_scale(geom, vecs, false);

      vector<double> angles(3);
      angles[0] = 90.0;
      angles[1] = 120.0;
      angles[2] = 90.0;
      bravais_warp(geom, angles, false);

      // rotate into place
      geom.transform(mat3d::rot(acos(1.0/3.0)/2.0,0,0));
      geom.transform(mat3d::rot(0,0,-45.0*(M_PI/180.0)));

      if (opts.cell_struts)
         add_color_struts(geom, 1.0, opts.edge_col[0]);
         
      if (opts.r_lattice_type == 4)
         geom.transform(r_lattice_trans_mat(true));
   }

   bravais_scale(geom, opts.vecs, false);
   bravais_warp(geom, opts.angles, false);

   // save original center
   vec3d original_center = centroid(geom.verts());
   
   // save lattice in case if adding back in end
   col_geom_v tgeom;
   if (opts.append_lattice)
      tgeom = geom;

   // if dual graph do that first
   if (opts.dual_lattice) {
      vector<vec3d> primitive_vectors;
      if (opts.use_centering_for_dual)
         bravais_primitive_vectors(primitive_vectors, opts.centering);
      bravais_dual(geom, primitive_vectors, opts.prim_vec_idxs, opts.vecs, opts.angles, opts.epsilon);
   }

   for(unsigned int i=0; i<opts.strut_len.size(); i++)
      add_color_struts(geom, opts.strut_len[i]*opts.strut_len[i], opts.edge_col[0]);
      
   // radius calculation if needed
   if (opts.radius_by_coord.is_set())
      opts.radius = bravais_radius_by_coord(opts.radius_by_coord, opts.offset, opts.vecs, opts.angles);
   else
   if (!opts.radius)
      opts.radius = bravais_radius(opts.grid, opts.vecs, opts.angles, opts.radius_default);

   // scoop
   if(opts.cfile.length() > 0) {
      geom_container_clip(geom, container, (opts.radius_default == 'k') ? lattice_radius(container,opts.radius_default) : opts.radius, opts.offset, opts.verbose, opts.epsilon);
   }
   else
   if (opts.container == 's') {
      geom_spherical_clip(geom, opts.radius, opts.offset, opts.verbose, opts.epsilon);
   }

   if(opts.voronoi_cells) {
      col_geom_v vgeom;
      if (get_voronoi_geom(geom, vgeom, opts.voronoi_central_cell, false, opts.epsilon)) {
         vgeom.color_vef(opts.vert_col[2], opts.edge_col[2], opts.face_col[2]);                      
         geom = vgeom;
      }
   }

   if (opts.convex_hull) {
      char errmsg[MSG_SZ]="";
      int ret = (opts.add_hull ? geom.add_hull("",errmsg) : geom.set_hull("",errmsg));
      if(ret < 0)
         fprintf(stderr,"%s\n",errmsg);
      else {
         geom.orient();
         if (opts.verbose)
            convex_hull_report(geom, opts.add_hull);
         geom.color_vef(opts.vert_col[1], opts.edge_col[1], opts.face_col[1]);
      }
   }
   
   if (opts.append_lattice) {
      geom.append(tgeom);
      tgeom.clear_all();
   }

   if (opts.color_method)
      color_by_symmetry_normals(geom, opts.color_method, opts.face_opacity, opts.epsilon);
      
   if (opts.trans_to_origin)
      geom.transform(mat3d::transl(-centroid(geom.verts())));

   if (opts.list_radii) {
      if (opts.list_radii_original_center) {
         opts.list_radii_center = original_center;
         if (opts.list_radii_original_center == 'o')
            opts.list_radii_center += opts.offset;
      }
      list_grid_radii(opts.ofile, geom, opts.list_radii_center, opts.list_radii, opts.epsilon);
   }
   else
   if (opts.list_struts)
      list_grid_struts(opts.ofile, geom, opts.list_struts, opts.epsilon);

   // last so not to alter listing outcomes
   if (opts.cent_col.is_set())
      color_centroid(geom, opts.cent_col, opts.epsilon);
      
   if (opts.append_container) {
      container.add_missing_impl_edges();
      container.clear_faces();
      geom.append(container);
   }

   //fprintf(stderr,"The volume of a single cell is %lf\n", bravais_volume(opts.crystal_system, opts.vecs, opts.angles));
}

int main(int argc, char **argv)
{
   brav_opts opts;
   opts.process_command_line(argc, argv);
   bravais ubravais;
   
   if(opts.list_bravais) {
      ubravais.list_bravais();
      exit(0);
   }

   // read in container file if using. Check existance
   char errmsg[MSG_SZ]="";
   col_geom_v container;
   if(opts.cfile.length() > 0) {
      if(!container.read(opts.cfile, errmsg))
         opts.error(errmsg);
      if(*errmsg)
         opts.warning(errmsg);
   }

   // list lookups are based on centering type "C"
   string centering_temp;
   if(opts.centering == "a" || opts.centering == "b") {
      centering_temp = opts.centering;
      opts.centering = "c";
   }

   int sym_no = ubravais.lookup_sym_no(opts.crystal_system, opts.centering);

   if(sym_no < 0) {
      // restore "a" or "b" if it exists
      if (centering_temp.length())
         opts.centering = centering_temp;
      opts.error("unknown lattice '"+opts.crystal_system + " " + opts.centering+"'");
   }
   else
   if(sym_no >= ubravais.get_last_bravais())
      opts.error("lattice number '"+opts.crystal_system+"' out of range");
   else {
      opts.crystal_system = ubravais.get_crystal_system(sym_no);
      // restore "a" or "b" if it exists
      if (centering_temp.length())
         opts.centering = centering_temp;
      else
         opts.centering = ubravais.get_centering(sym_no);
   }

   // from here on out everything expects lower case
   transform(opts.crystal_system.begin(), opts.crystal_system.end(), opts.crystal_system.begin(), ::tolower);
   transform(opts.centering.begin(), opts.centering.end(), opts.centering.begin(), ::tolower);

   // in place of ubravais.list_bravais(sym_no);
   string crystal_system_print_str = opts.crystal_system;
   if (opts.crystal_system == "trigonal")
      crystal_system_print_str = "trigonal(rhombohedral)";
   string centering_print_str = opts.centering;
   if (opts.centering == "a" || opts.centering == "b")
      centering_print_str = "c(" + opts.centering + ")";

   if (opts.verbose)
      fprintf(stderr,
         "%2d)\t%-s %s\n", sym_no+1, crystal_system_print_str.c_str(), centering_print_str.c_str());

   if (opts.auto_grid_type == '8' && opts.centering == "p")
      opts.error("grid type of 8 cannot be used with primitive (simple) bravais cells", 'G');

   if (opts.r_lattice_type > 0 && !(opts.crystal_system == "cubic" && opts.centering == "p") && opts.crystal_system != "trigonal")
      opts.warning("hex relation is only proper on Cubic P or Trigonal", 'R');
      
   //if (opts.dual_lattice && (!strchr("pfi", *(opts.centering.c_str()))))
   //   opts.error("dual lattices are only only allowed for centering types p, f or i", 'd');

   col_geom_v geom;
   do_bravais(geom, container, opts);

   if (!opts.list_radii && !opts.list_struts)   
      if(!geom.write(opts.ofile, errmsg))
         opts.error(errmsg);

   return 0;  
}
