/*
 * mkcf.c - modify a userprefs.h configuration file
 *    1 - enter your variables in struct _UserPrefs after * User Variables *
 *    2 - run mkcf
 *    3 - correct manually the ConfigDescTable
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <libgen.h> /* basename */

#include <pcreg_ex.h>
#include <fdbuf.h>
#include <filerec.h>
#include <descrec.h>
#include <variable.h>
#include <descelem.h>
#include <mkcf.h>
#include <strcatdup.h>
#include <dliststr.h>

int prog_debug;
MsgLogData *msglogp; /* global pointeur to MsgLogData in main */

void usage(void)
{
   fprintf( stderr,
"mkcf is a tool to help managing a userprefs file\n"
"mkcf [options] <userprefs file>\n"
"  ex: mkcf userprefs.h\n"
"  ex: mkcf -v -c -sm THIS_IS_RCFILE -desc UserPrefsTable -nouv -em rc_init_defaults rcfile.h\n"
"  -sm marker  : use marker as default start marker\n"
"  -em marker  : use marker as default end marker\n"
"  -desc name  : use name as desc name \n"
"  -n          : create a new empty userprefs.h file\n"
"  -c          : print variables with comment from desc\n"
"  -fc         : force comment from var if any\n"
"  -nouv       : start search variable after struct _UserPrefs\n"
"  -v          : verbose\n"
"  -p          : preserve files : userprefs.h is preserved, new is userprefs_tmp\n"
   );
   exit(1);
}

UserData *userData;

int main (int argc, char *argv[])
{
   int i ;
   UserData *ud;

   ud = userData = app_new0(UserData, 1);
   ud->prog = basename(argv[0]);
   msg_initlog(ud->prog, MSG_F_NO_DATE | MSG_F_SIGNAL |  MSG_F_COLOR, NULL, NULL );
   msg_set_level(2);
   
   if ( argc < 2 ) {
      usage();
   }

   for (i = 1 ; i < argc ; i++) {
      if (strcmp(argv[i], "-d") == 0) {
	 prog_debug++ ;
	 msg_set_dbg_msk(1);
      } else if (strcmp(argv[i], "-h") == 0) {
	 usage();
	 return -1;
      } else if (strcmp(argv[i], "-n") == 0) {
	 ud->nflag = 1;
      } else if (strcmp(argv[i], "-c") == 0) {
	 ud->cflag = 1;
      } else if (strcmp(argv[i], "-p") == 0) {
	 ud->pflag = 1;
      } else if (strcmp(argv[i], "-fc") == 0) {
	 ud->fcflag = 1;
      } else if (strcmp(argv[i], "-nouv") == 0) {
	 ud->nouv = 1;
      } else if (strcmp(argv[i], "-v") == 0) {
	 ud->verbose++ ;
      } else if (strcmp(argv[i], "-sm") == 0) {
	 ud->searchmarker = app_strdup(argv[++i]) ;
      } else if (strcmp(argv[i], "-em") == 0) {
	 ud->endmarker = app_strdup(argv[++i]) ;
      } else if (strcmp(argv[i], "-desc") == 0) {
	 ud->searchdesc = app_strdup(argv[++i]) ;
      } else {
	 ud->rcfile =  app_strdup(argv[i]);
      }
   }
   ud->defmarker = "THIS_IS_USERPREFS_FILE";
   if ( ! ud->searchmarker ) {
      ud->defmarkerflag = 1;
      ud->searchmarker = app_strdup(ud->defmarker) ;
   }
   ud->defdesc = "ConfigDescTable";
   if ( ! ud->searchdesc ) {
      ud->searchdesc = app_strdup(ud->defdesc) ;
   }
   if ( ! ud->upproto ) {
      ud->upproto = app_strdup("UserPrefs") ;
   }
   ud->initfuncs = array_strPtr_new( 4 );
   
   if ( ud->nflag ){
      mkcf_write_empty_file(ud);
   } else {
      /* read the user field in structure _UserPrefs */
      mkcf_extract_var_from_file( ud, ud->rcfile, ud->upproto, "up", 0 );
      dlist_iterator( ud->tinirecs, mkcf_iter_extract_var_for_tini, ud );

      if ( ud->verbose ){
	 dlist_iterator( ud->filerecs, filerec_iter_print_vars, ud );
      }
      mkcf_extract_conf_desc_table(ud);
      if ( ud->verbose ){
	 dlist_iterator( ud->descrecs, descrec_iter_print_elems, ud );
      }
      mkcf_write_new_user_prefs(ud);
   }
   
   mkcf_destroy(ud);
   return 0;
}

void mkcf_destroy(UserData *ud)
{
   array_destroy( ud->initfuncs);
   app_free(ud->searchmarker);
   app_free(ud->searchdesc);
   app_free(ud->endmarker);
   app_free(ud->rcfile);
   app_free(ud->upproto);
   dlist_delete_all( ud->filerecs );
   dlist_delete_all( ud->tinirecs );
   dlist_delete_all( ud->includes );
   
   app_free(ud);
}

void mkcf_extract_var_from_file( UserData *ud, char *rcfile, char *proto,
				 char *svarn,  char *relvar )
{
   char *line;
   FDBuf *fdbuf;
   int state = 0;
   char *prefix = NULL;
   
   /* open file */
   fdbuf = fdbuf_new ( rcfile, "r", 128);
   if ( fdbuf->status ) {
      msg_fatal("Can't open file '%s': %s", rcfile, strerror (errno));
   }
   fdbuf_set_flags(fdbuf, FDB_STRIP_CR);
   
   FileRecord *frec = filerec_new(rcfile, proto, svarn, relvar );
   ud->filerecs = dlist_add( ud->filerecs, (AppClass *) frec, NULL, NULL );
   
   char *pattern = app_strcatdup("struct\\s+_", proto, "\\s*{", NULL);
   Pcregex *startRe = pcregex_new( pattern, 0, 0 );
   app_free(pattern);
   
   pattern = "#\\s*include\\s*<([\\w.]+)>";
   Pcregex *includeRe = pcregex_new( pattern, 0, 2 );
   
   pattern = "\\*\\s*mkcf\\s+User\\s+Variables\\s+.*\\*";
   Pcregex *userVarRe = pcregex_new( pattern, 0, 0 );
   
   pattern = "prefix\\s+(\\w+)";
   Pcregex *prefixRe = pcregex_new( pattern, 0, 2 );
   
   pattern = "\\s*}\\s*[\\w\\s]*;";
   Pcregex *endStructRe = pcregex_new( pattern, 0, 0 );
   
   pattern = "\\s*([^\\s]+\\s*\\**)\\s*([^\\s]+)\\s*;";
   Pcregex *varRe = pcregex_new( pattern, 0, 3 );
   
   pattern = "/\\*\\s+(.*)\\*/";
   Pcregex *cmtRe = pcregex_new( pattern, 0, 2 );
   
   pattern = "mkcf\\s+TINI\\s+([\\w.-]+)\\s+([\\w.-]+)";
   Pcregex *tiniRe = pcregex_new( pattern, 0, 3 );
   
   pattern = "sVarn\\s+(\\w+)";
   Pcregex *svarnRe = pcregex_new( pattern, 0, 2 );
   
   pattern = app_strcatdup(proto, "\\s*\\*\\s*(\\w+)_new", NULL);
   Pcregex *functionRe = pcregex_new( pattern, 0, 2 );
   app_free(pattern);
   
   while( (line = fdbuf_get_line(fdbuf)) != NULL && state < 4 ) {
//       fprintf(stderr, "%s\n", line );
      switch (state) {
       case 0:
	 if ( relvar == NULL && pcregex_exec(includeRe, line, 0 ) >= 0 ){
            char *pstr = array_strPtr_get( includeRe->matchStr, 1 );
            frec->includes = dlist_str_add( frec->includes, pstr, NULL, NULL);
         }
	 if ( pcregex_exec(startRe, line, 0 ) >= 0 ){
	    state = 1;
	 }
	 break;
       case 1:
	 /* marker: * mkcf User Variables prefix up * */
	 if ( pcregex_exec(userVarRe, line, 0 ) >= 0 ){
	    if ( pcregex_exec(prefixRe, line, 0 ) > 0 ){
	       prefix = array_strPtr_get( prefixRe->matchStr, 1 );
	    }
	    state = 2;
	 }
	 if ( ud->nouv ) {
	    state = 2;
	 }
	 if ( state == 2) {
	    filerec_dup_prefix( frec, prefix );
	 }
	 break;
       case 2:
         /* end of structure: }; */
	 if ( pcregex_exec(endStructRe, line, 0 ) >= 0 ){
  	    state = 3;
            if ( ! relvar ){
               state = 4;
            }
	    break;
	 }
	 /* char **buttonBgColor;  * button_bg_color[state] * */
	 if ( pcregex_exec(varRe, line, 0 ) > 1 ){
	    char *matched = array_strPtr_get( varRe->matchStr, 0 );
	    char *cmt = NULL;
	    int tini = 0;
	    if ( pcregex_exec(cmtRe, line + strlen(matched), 0 ) > 0 ){
	       cmt = array_strPtr_get( cmtRe->matchStr, 1 );
	    }

	    if ( cmt && pcregex_exec(tiniRe, cmt, 0 ) > 1 ){
	       tini = 1;
	       matched = array_strPtr_get( tiniRe->matchStr, 0 );
	       char *svarn = NULL;
	       if ( pcregex_exec(svarnRe, cmt + strlen(matched), 0 ) > 0 ){
		  svarn = array_strPtr_get( svarnRe->matchStr, 1 );
	       }
               char *rcfile = array_strPtr_get( tiniRe->matchStr, 1 );
               mkcf_check_include(ud, frec, rcfile);
	       FileRecord *tinirec = filerec_new(
				rcfile,
				array_strPtr_get( tiniRe->matchStr, 2 ),
				svarn,
				array_strPtr_get( varRe->matchStr, 2 ) );
	       ud->tinirecs = dlist_add( ud->tinirecs, (AppClass *) tinirec,
					 NULL, NULL );
	    }
	    filerec_add_var( frec,
			     array_strPtr_get( varRe->matchStr, 2 ),
			     array_strPtr_get( varRe->matchStr, 1 ),
			     cmt, tini, fdbuf_get_lineno(fdbuf) );
	 }
	 break;
       case 3:
         /* ex: DdnsAccount *account_new( void ); */
         if ( pcregex_exec(functionRe, line, 0 ) > 0 ){
            char *function = array_strPtr_get( functionRe->matchStr, 1 );
            filerec_dup_funcname( frec, function);
            state = 4;
         }
	 break;
      }
   }
   pcregex_destroy(functionRe);
   pcregex_destroy(svarnRe);
   pcregex_destroy(tiniRe);
   pcregex_destroy(cmtRe);
   pcregex_destroy(varRe);
   pcregex_destroy(endStructRe);
   pcregex_destroy(prefixRe);
   pcregex_destroy(userVarRe);
   pcregex_destroy(includeRe);
   pcregex_destroy(startRe);
   fdbuf_destroy(fdbuf);
}


void mkcf_check_include(UserData *ud, FileRecord *frec, char *rcfile)
{
   int res = 1;
   
   StrObj *strobj = (StrObj *) dlist_lookup(frec->includes,
                              (AppClass *) rcfile, str_func_str_cmp );
   if ( ! strobj ) {
      ud->includes = dlist_str_add(ud->includes, rcfile, str_func_cmp, &res );
      if ( res ){
         msg_warning ( "'%s' already existing in ud->includes", rcfile );
      }
   }
}

/*
 * this function return 0 to avoid deleting the node
 */

int mkcf_iter_extract_var_for_tini( AppClass *data, void *user_data )
{
   FileRecord *frec = (FileRecord *) data;
   UserData  *ud = (UserData *) user_data;
   
   mkcf_extract_var_from_file( ud, frec->rcfile, frec->proto,
				 frec->svarn,  frec->relvar );
   return 0;
}

void mkcf_extract_conf_desc_table(UserData *ud)
{
   char *line;
   FDBuf *fdbuf;
   DBuf *dbuf = dbuf_new(128, 0);
   int state = 0;
   DescRecord *desc = NULL;
   
   /* open file */
   fdbuf = fdbuf_new ( ud->rcfile, "r", 128);
   if ( fdbuf->status ) {
      msg_fatal("Can't open file '%s': %s", ud->rcfile, strerror (errno));
   }
   fdbuf_set_flags(fdbuf, FDB_STRIP_CR);
   
   /* ConfigDescTable confDesc[] = */
   char *pattern = app_strcatdup(ud->searchdesc,
             "\\s+([\\w]+)\\s*\\[\\]\\s*=.*/\\*\\s*(\\w+)\\s+", NULL);
   Pcregex *descnameRe = pcregex_new( pattern, 0, 3 );
   app_free(pattern);
   
   pattern = "mkcfVersion\\s*=\\s*(\\d+)\\s*;";
   Pcregex *versionRe = pcregex_new( pattern, 0, 2 );

   /* up_init_xxx */
   pattern = "(up_init_[\\w_]+)\\s*(;*)";
   Pcregex *initfuncRe = pcregex_new( pattern, 0, 2 );
   
   pattern = "}";
   Pcregex *endrecordRe = pcregex_new( pattern, 0, 0 );
   
   pattern = "\\s*}\\s*[\\w\\s]*;";
   Pcregex *endStructRe = pcregex_new( pattern, 0, 0 );
   
   pattern = "\"([^\"]*)\"\\s*,\\s*([^,]+)\\s*,\\s*([^,]+),\\s*\"([^\"]*)\"";
   Pcregex *elem4Re = pcregex_new( pattern, 0, 5 );
   
   pattern = "\"([^\"]*)\"\\s*,\\s*([^,]+)\\s*,\\s*([^,]+),\\s*([^,]+),\\s*([^,]+),\\s*\"([^\"]*)\"";
   Pcregex *elem6Re = pcregex_new( pattern, 0, 7 );
   
   pattern = "0,\\s*0,\\s*0,\\s*0";
   Pcregex *zeroRe = pcregex_new( pattern, 0, 0 );
   
   while( (line = fdbuf_get_line(fdbuf)) != NULL && state < 3 ) {
//     fprintf(stderr, "%s\n", line );
      switch (state) {
       case 0:
	 if ( pcregex_exec(descnameRe, line, 0 ) > 0 ){
	    char *name = array_strPtr_get( descnameRe->matchStr, 1 );
	    char *proto = array_strPtr_get( descnameRe->matchStr, 2 );
	    if ( app_strcmp(name, "key_tk") == 0){
	       name = "confDesc";
	    }
	    FileRecord *frec = (FileRecord *) dlist_lookup( ud->filerecs,
                 (AppClass *) proto, filerec_proto_cmp );
	    if ( ! frec ){
	       msg_fatal("No FileRecord found for %s", name );
	    }
            filerec_dup_descname(frec, name);
	    desc = descrec_new(name, frec->proto, filerec_get_prefix(frec) );
	    ud->descrecs = dlist_add( ud->descrecs, (AppClass *) desc, NULL, NULL );
	    state = 1;
	 }
         /* up_init_accounts */
	 if ( pcregex_exec(initfuncRe, line, 0 ) > 1 ){
	    char *name = array_strPtr_get( initfuncRe->matchStr, 2 );
	    if ( *name != ';' ){
	       array_strPtr_add(ud->initfuncs,
			array_strPtr_get( initfuncRe->matchStr, 1 ) );
	    }
	 }
         if ( ud->mkcfVersion == 0 ) {
            if ( pcregex_exec(versionRe, line, 0 ) > 0 ){
               ud->mkcfVersion = atoi(array_strPtr_get(versionRe->matchStr, 1 ) );
	    }
	 }
	 break;
       case 1:
	 dbuf_cat( dbuf, (DBuf *) fdbuf);

	 if ( pcregex_exec(endrecordRe, dbuf->s, 0 ) < 0 ){
	    dbuf_strcat( dbuf, " ");
	    break;
	 }
	 if ( pcregex_exec(endStructRe, dbuf->s, 0 ) >= 0 ){
	    state = 0;
	    break;
	 }
	 if ( pcregex_exec(elem4Re, dbuf->s, 0 ) > 3 ){
	    descrec_add_elem( desc,
			  array_strPtr_get( elem4Re->matchStr, 1 ),
		 	  array_strPtr_get( elem4Re->matchStr, 2 ),
			  array_strPtr_get( elem4Re->matchStr, 3 ),
			  array_strPtr_get( elem4Re->matchStr, 4 ),
			      fdbuf_get_lineno(fdbuf) );
	 } else if ( pcregex_exec(elem6Re, dbuf->s, 0 ) > 5 ){
	    descrec_add_elem( desc,
			  array_strPtr_get( elem6Re->matchStr, 1 ),
		 	  array_strPtr_get( elem6Re->matchStr, 2 ),
			  array_strPtr_get( elem6Re->matchStr, 3 ),
			  array_strPtr_get( elem6Re->matchStr, 6 ),
			      fdbuf_get_lineno(fdbuf) );
	 } else if ( pcregex_exec(zeroRe, dbuf->s, 0 ) >= 0 ){
	    ;
	 } else {
	    msg_error("Bad line: '%s' at %d", dbuf->s, fdbuf_get_lineno(fdbuf) );
	 }
	    
	 dbuf_clear(dbuf);
	 break;
      }
   }
   pcregex_destroy(zeroRe);
   pcregex_destroy(elem4Re);
   pcregex_destroy(elem6Re);
   pcregex_destroy(endStructRe);
   pcregex_destroy(endrecordRe);
   pcregex_destroy(descnameRe);
   pcregex_destroy(versionRe);
   dbuf_destroy(dbuf);
   fdbuf_destroy(fdbuf);
}

int mkcf_iter_print_dyn_destroy( AppClass *data, void *user_data )
{
   Variable *var = (Variable *) data;
   UserData *ud = (UserData *) user_data;
   char *destroyName = NULL;

   if ( app_strcmp(var->vartype, "char *" ) == 0 ){
      destroyName = "app_free";
   } else if ( app_strcmp(var->vartype, "ArrayStr *" ) == 0 ||
               app_strcmp(var->vartype, "ArrayStr *" ) == 0 ){
      destroyName = "array_destroy";
   } else if ( app_strcmp(var->vartype, "DList *" ) == 0 ){
      destroyName = "dlist_delete_all";
   }
   
   if (destroyName){
      fprintf( ud->ofd, "    %s(up->%s);\n", destroyName, var->varname );
   }
   return 0;
}

void mkcf_print_dyn_destroy(UserData *ud )
{
   fprintf( ud->ofd, "\nvoid up_dyn_destroy(%s *up)\n{\n", ud->upproto );
   FileRecord *frec = (FileRecord *) dlist_lookup( ud->filerecs,
                    (AppClass *) ud->upproto, filerec_proto_cmp );
   if ( ! frec ){
      msg_fatal("No FileRecord found for %s", ud->upproto );
   }
 
   dlist_iterator( frec->vars, mkcf_iter_print_dyn_destroy, ud );
      
   fprintf( ud->ofd, "}\n\n" );
   
}

int mkcf_iter_print_include( AppClass *data, void *user_data )
{
   StrObj *strobj = (StrObj *) data;
   UserData *ud = (UserData *) user_data;

   fprintf( ud->ofd, "#include <%s>\n", strobj->pstr);
   return 0;
}

void mkcf_print_stuff(UserData *ud, int seen)
{
   if ( seen == 2 ){
      fprintf( ud->ofd, "#ifdef %s\n", ud->defmarker);
      seen = 1;
   }

   fprintf( ud->ofd, "\nstatic int mkcfVersion = %d;\n", ud->mkcfVersion + 1);

   dlist_iterator( ud->filerecs, filerec_iter_print_prototypes, ud );
   dlist_iterator( ud->filerecs, filerec_iter_print_content, ud );

   mkcf_print_dyn_destroy(ud);
   
   if ( seen ){
      fprintf( ud->ofd, "\n/* mkcf end of stuff generated by mkcf (do not remove) */\n");
   }
}

int mkcf_skip_comment( char *line, int in_comment)
{
   char *s;
   char lastc;
   int in_quote = 0;
   
   for ( s = line; *s ; s++ ) {
      if ( *s == '"' ) {
         if ( in_quote == 0 ) {
            in_quote = 1;
         } else {
            in_quote = 0;
         }
      }
      if ( lastc == '/' && *s == '*' && in_quote == 0 ) {
         in_comment = 1;
      }
      if ( lastc == '*' && *s == '/' && in_quote == 0 ) {
         if ( in_comment ) {
            in_comment = 0;
         }
      }
      lastc = *s;
   }
   return in_comment;
}

void mkcf_write_new_user_prefs(UserData *ud)
{
   char *line;
   FDBuf *fdbuf;
   int state = 0;
   FILE *ofd;
   int seenStuff = 2;
   int in_comment = 0;
   int done_include = 0;
   int ret;
   
   /* open file */
   fdbuf = fdbuf_new ( ud->rcfile, "r", 128);
   if ( fdbuf->status ) {
      msg_fatal("Can't open file '%s': %s", ud->rcfile, strerror (errno));
   }
   fdbuf_set_flags(fdbuf, FDB_STRIP_CR);

   char *ofile = app_strcatdup(ud->rcfile, "_tmp", NULL);
   ofd = fopen( ofile, "w");
   if ( ! ofd ) {
      msg_fatal("Can't open file '%s': %s", ofile, strerror (errno));
   }
   ud->ofd = ofd;

   char *pattern = "#\\s*include\\s*<([\\w.]+)>";
   Pcregex *includeRe = pcregex_new( pattern, 0, 0 );

   /* #ifdef THIS_IS_USERPREFS_FILE */
   pattern = app_strcatdup("#\\s*ifdef\\s+", ud->searchmarker, NULL);
   Pcregex *startRe = pcregex_new( pattern, 0, 0 );
   app_free(pattern);
   
   /* mkcf\s+end\s+of\s+stuff\s+generated */
   pattern = "mkcf\\s+end\\s+of\\s+stuff\\s+generated";
   Pcregex *endRe = pcregex_new( pattern, 0, 0 );
   
   /* user given end marker */
   Pcregex *userendRe = NULL;
   if ( ud->endmarker ){
      userendRe = pcregex_new( ud->endmarker, 0, 0 );
   }
   
   pattern = app_strcatdup("#\\s*endif.*", ud->searchmarker, NULL);
   Pcregex *endifRe = pcregex_new( pattern, 0, 0 );
   app_free(pattern);

   while( (line = fdbuf_get_line(fdbuf)) != NULL && state < 3 ) {
//       fprintf(stderr, "%s\n", line );

      if ( ud->defmarkerflag == 0 ) {
	 if ( pcregex_exec(endifRe, line, 0 ) >= 0 ){
	    fprintf( ofd, "#endif /*  %s */\n", ud->defmarker); /* change the marker */
	    continue;
	 }
      }
	  
      switch (state) {
       case 0:
	 in_comment = mkcf_skip_comment( line, in_comment);
	 if ( in_comment == 0  && done_include == 0 &&
              pcregex_exec(includeRe, line, 0 ) >= 0 ){
            dlist_iterator( ud->includes, mkcf_iter_print_include, ud );
            done_include = 1;
         }
	 if ( in_comment == 0 && pcregex_exec(startRe, line, 0 ) >= 0 ){
	    state = 1;
	    seenStuff--;
	    if ( ud->defmarkerflag == 0 ) {
	       fprintf( ofd, "#ifdef %s\n", ud->defmarker); /* change the marker */
	       break;
	    }
	 }
         /* fall thru */
       case 2:
	 fprintf( ofd, "%s\n", line);
	 break;
       case 1:
	 if ( pcregex_exec(endRe, line, 0 ) >= 0 ||
	      ( userendRe && (ret = pcregex_exec(userendRe, line, 0 )) >= 0)) {
	    state = 2;
	    seenStuff--;
	    mkcf_print_stuff(ud, seenStuff);
	    if  ( ret >= 0 ) {
	       fprintf( ofd, "%s\n", line);
	    }
	 }
	 break;
      }
   }
   if ( seenStuff ){
      msg_error( "Warning a line '#ifdef %s' and a line\n"
                 " /* mkcf end of stuff generated by mkcf */ are expected",
		 ud->defmarker );
      mkcf_print_stuff(ud, seenStuff);
   }
   
   fclose(ofd);
   pcregex_destroy(endifRe);
   pcregex_destroy(userendRe);
   pcregex_destroy(endRe);
   pcregex_destroy(startRe);
   pcregex_destroy(includeRe);
   fdbuf_destroy(fdbuf);

   if ( ud->pflag == 0 ) {
      char *bak = app_strcatdup(ud->rcfile, ".bak", NULL);
      rename( ud->rcfile, bak);
      rename( ofile, ud->rcfile);
      app_free(bak);
   }
   app_free(ofile);
}
