/* MF_MainWindow.cxx */

/*
  MF_MainWindow : Deals with the main window of the application.
  The menu, boxes and others objects contained in the window are 
  declared as private fields of this class.

*/

#include "MF_MainWindow.H"
#include "MenuCb.H"
#include "MF_Values.H"
#include "dijkstra.H"
#include "init.H"


#define GADD
#define GRMV
#define RADD
#define RRMV

MF_MainWindow::MF_MainWindow(int x, int y, int w, int h, const char * lbl)
  : Fl_Window(x, y, w, h, lbl)
{ 
  bestFirst_ = 1;
  myScroll_ = NULL;
  zoomer_ = NULL;
}

/* Adds the menu, buttons and several widgets in the window
*/
void MF_MainWindow::addMenu(void)
{

  static char *dirty_help_file = "lalnview.help";

#if defined( __APPLE__ )
  char *help_file = getResValue("helpfile");
#elif defined( WIN32 )
  char * help_file = (char *)"lalnview.help";
#else
  char *help_file = getSrcValue("helpfile");
  if( help_file ){
    FILE *nf = fopen(help_file, "r");
    if( !nf ){
      fclose(nf);
      help_file = NULL;
    }
  }
  if( !help_file ){
    char test[8000];
    char *p =  getenv("HOME");
    strcpy(test, p);
    strcat(test, "lalnview.help");
    FILE *nf = fopen(test, "r");
    if(nf){
      fclose(nf);
      help_file = strdup(test);
    } else {
      help_file = strdup("lalnview.help");
    }
  }
  /*
  if( help_file)
    printf("helpfile : %s\n", help_file);
  */
#endif

  Fl_Menu_Item menuitems[] = {
    { "&File", FL_ALT+'f', 0, 0, FL_SUBMENU },
    { "&Open", FL_CTRL+'o', (Fl_Callback *)open_cb, 0 },
    { "Open &Features", FL_CTRL+'f', (Fl_Callback *)open_features_cb, 0, FL_MENU_DIVIDER },
    { "Export as PDF", FL_CTRL+'p', (Fl_Callback *)export_PDF_cb },
    { "Capture as bmp", FL_CTRL+'b', (Fl_Callback *)save_bmp_cb, 0, FL_MENU_DIVIDER },
    { "&Quit", FL_CTRL+'q', (Fl_Callback *)close_cb },
    {0},
    {"&Options",0, 0, 0, FL_SUBMENU },
    {"&Save alignments", 0, (Fl_Callback *)save_aln_cb, 0, FL_MENU_DIVIDER },
    {"S&cale", 0, 0, 0, FL_SUBMENU},
    {"&Save scale", 0, (Fl_Callback *)save_scale_cb },
    {"&Load Scale", 0, (Fl_Callback *)load_scale_cb },
    {0},
    {"F&oreground blocs", 0, 0, 0, FL_SUBMENU},
    {"&Best ones", 0, (Fl_Callback *)bestScore_cb, 0, FL_MENU_RADIO},
    {"&Worse ones", 0, (Fl_Callback *)worseScore_cb, 0, FL_MENU_RADIO},
    {0},
    {"&Green list", 0, 0, 0, FL_SUBMENU},
    {"Add this segment", FL_CTRL+'a', (Fl_Callback *)greenListAdd_cb },
    {"Remove this segment", FL_CTRL+'r', (Fl_Callback *)greenListRemove_cb },
    {"Erase list", 0, (Fl_Callback *)greenListErase_cb },
    {0},
    {"&Red list", 0, 0, 0, FL_SUBMENU},
    {"Add this segment", FL_CTRL+'s', (Fl_Callback *)redListAdd_cb },
    {"Remove this segment", FL_CTRL+'t', (Fl_Callback *)redListRemove_cb },
    {"Erase list", 0, (Fl_Callback *)redListErase_cb },
    {0},
    {0},
    {"&?", FL_ALT+'?', 0, 0, FL_SUBMENU },
    {"&Help", FL_CTRL+'h', (Fl_Callback *)help_callback, (void *)help_file},
    {"&About", 0, (Fl_Callback *)about_cb, 0},
    {0},
    {0}
  };


  int NAMEWIDTH = 80; // NAMEWIDTH = default width of the area in wich the names of the sequences are displayed.
  // This value is updated when the sequences are loaded so that it fits to the longest sequence name.
  int SCROLH = 220; // SCROLH = Height of the main displaying panel (the panel in wich the sequences are displayed).

  /* Adding the menu items into the menu bar. */
  menu_ = new Fl_Menu_Bar(0, 0, this->w(), 25);
  menu_->copy(menuitems);
  this->add(menu_);

  /* Creates the scale on top of the main panel. */
  scale_ = new MF_Echelle(NAMEWIDTH, menu_->h(), this->w()-NAMEWIDTH, 10, 12, 40, 100);
  this->add(scale_);

  /* Creates the area in wich the name of the sequences or drawn */
  grpName_ = new Fl_Group( 0 , (scale_->y()) , NAMEWIDTH , SCROLH+scale_->h() );
  this->add(grpName_);
  name_ = new MF_Name( 0 , (scale_->y()) , NAMEWIDTH , SCROLH+scale_->h(), 10, 20, 12);
  grpName_->add(name_);

  /* myScroll_ is the main panel in wich the sequences are displayed. */
  myScroll_ = new MF_MyScroll( NAMEWIDTH, scale_->y()+scale_->h(), this->w()-NAMEWIDTH , SCROLH);
  this->add(myScroll_);
  myScroll_->resizable(myScroll_);

  /* pos1 : y position of the firsts widgets below the myScroll_ panel (buttons, minimum score chooser, ...)*/
  int pos1 = myScroll_->y() + myScroll_->h() + 10;

  btGrp_ = new Fl_Scroll(0 , pos1, this->w(), this->h()-pos1);
  btGrp_->type(Fl_Scroll::HORIZONTAL);
  btGrp_->resizable(NULL);
  this->add(btGrp_);

  scoMinC_ = new MF_ScoMinChooser(_MF_TYPE_SCOREMIN_, 20, pos1, "Similarity Score Threshold\0", 12);
  btGrp_->add(scoMinC_);

  int pos2 = scoMinC_->y()+scoMinC_->h()+3;
  identityMinC_ = new MF_ScoMinChooser(_MF_TYPE_IDENTITYMIN_, 20, pos2 , "Minimum identity\0", 12);
  btGrp_->add(identityMinC_);

  int pos3 = identityMinC_->y() + identityMinC_->h() + 3;
  lenghtMinC_ = new MF_ScoMinChooser(_MF_TYPE_LENGHTMIN_, 20, pos3 , "Minimum length\0", 12); 
  btGrp_->add(lenghtMinC_);

  int pos4 = lenghtMinC_->y() + lenghtMinC_->h() + 10;
  bestSeries_ = new MF_BestSeries(scoMinC_->x(), pos4, 600, 40);
  btGrp_->add(bestSeries_);

  parcBloc_ = new MF_ParcBloc(scoMinC_->x()+scoMinC_->w()+10, pos1, 100, 75);
  btGrp_->add(parcBloc_);

  zoomer_ = new MF_Zoomer(parcBloc_->x()+parcBloc_->w()+5, parcBloc_->y(), 100, 75);
  btGrp_->add(zoomer_);


  stats_ = new MF_Stats(zoomer_->x()+zoomer_->w()+15, pos1, 340, 70);
  btGrp_->add(stats_);


  pos4 = stats_->y()+stats_->h()+10;
  int h4 = bestSeries_->y()+bestSeries_->h()-pos4+2;
  featDisp_ = new MF_FeatDisplay(parcBloc_->x(), pos4, this->w()-parcBloc_->x()-10, h4);
  this->add(featDisp_);

  this->size_range(0 , h() , 2600, h() );
}

/* void MF_MainWindow::about(void)
   Displays the about panel.
 */
void MF_MainWindow::about(void)
{
  this->about_ = new MF_About(10, 10, 500, 400);
  this->about_->show();
}


void MF_MainWindow::actuName(void){ name_->draw(); } /* updates the area in wich the names of the sequences are displayed. */

/* Redraws the main panel. */
void MF_MainWindow::updatePanel(void)
{
  if(myScroll_)
    myScroll_->redraw();
}

/* Performs a full update of the main panel (with new values of the thresholds). */
void MF_MainWindow::fullUpdatePanel(void)
{
  if(myScroll_ && NBBLOC>0 && NBSEQ>0){
    if(bestSeries())
      dijkstra();
    myScroll_->fullUpdate();
  }
}

/* Sets the shown text inside the three boxes "Similarity score threshold", "Minimum identity" and "Minimum lenght". */
void MF_MainWindow::setMins(void)
{
  char val[16];

  sprintf(val, "%.1f\0", MF_Values::scoMin());
  scoMinC_->setValue(val);

  sprintf(val, "%.1f\0", MF_Values::identityMin());
  identityMinC_->setValue(val);

  sprintf(val, "%d\0", MF_Values::lenghtMin());
  lenghtMinC_->setValue(val);
}

/* void MF_MainWindow::activerParcBloc(void)
   Activates the buttons used to switch from a segment to the other when the 
   user clicks on a BLOC that belongs to several segments.
*/
void MF_MainWindow::activerParcBloc(void){ parcBloc_ ->activer(); }

int MF_MainWindow::bestFirst(void){ return bestFirst_; }/* Return a non zero value if the radio button "Best ones"  of the Option menu is checked. */

/* Sets the value of the private field 'bestFirst_' wich indicates wether the best or the worse blocs must be displayed in front. */
void MF_MainWindow::bestFirst(int val)
{
  bestFirst_ = val;
  myScroll_->fullUpdate();
}

int MF_MainWindow::bestSeries(void){ return bestSeries_->isChecked(); }/* Returns a non zero value if the check button "Display the  best series
of segments" (eg _MAINW_->bestSeries_->chk_) is checked. */

/* void MF_MainWindow::desactiverParcBloc(void)
   Deactivates the buttons used to switch from a segment to the other (see below).
*/
void MF_MainWindow::desactiverParcBloc(void){ parcBloc_->desactiver(); }

void MF_MainWindow::clearAbout(void){ delete this->about_; }/* Hides the About window. */

void MF_MainWindow::listMenuItems(void)
{
  int nb = menu_->size();
  printf("Nomre d'elements dans menu : %d\n", nb);

  const Fl_Menu_Item * it = menu_->menu();
  int i;
  for(i=0 ; i<nb ; i++){
    if( it[i].label() )
      printf("%-3d.  %s\n",i, it[i].label());
  }
}

/*****************************************************************************************/
/* Accessors to private fields. */
Fl_Menu_Bar * MF_MainWindow::getMenuBar(void){ return this->menu_; }
MF_MyScroll * MF_MainWindow::getScroll(void){ return this->myScroll_; }
MF_Name * MF_MainWindow::getName(void){ return this->name_; }
MF_Zoomer * MF_MainWindow::getZoomer(void){ return this->zoomer_; }
MF_ParcBloc * MF_MainWindow::getParcBloc(void){ return parcBloc_; }
MF_Stats * MF_MainWindow::getStats(void){ return stats_; }
const Fl_Menu_Item * MF_MainWindow::getMenuItems(void)
{ 
  const Fl_Menu_Item * it = menu_->menu();
  return it;
}


MF_FeatDisplay * MF_MainWindow::getFeatDisp(void){ return featDisp_; }
MF_BestSeries * MF_MainWindow::getBestSeries(void){ return bestSeries_; }
MF_Echelle * MF_MainWindow::getScale(void){ return scale_; }
MF_ScoMinChooser * MF_MainWindow::getScoMinC(void){ return scoMinC_; }
/*****************************************************************************************/
