package gomo;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.obo.dataadapter.OBOParseEngine;
import org.obo.dataadapter.OBOParseException;
import org.obo.datamodel.NestedValue;

public class DAGParser extends OBOParserAdapter {
	private Set<String> unusedRels;
	private List<GONode> nodes;
	private List<GOEntry> goids;
	private Map<String,String> groups;
	
	
	private boolean readingTerm;
	private boolean obsolete;
	private int termIndex;
	private GONode node;
	private GOEntry term;
	private Set<GOEntry> alts;
	
	//header info
	private String oboVersion;
	private Date oboDate;
	private String oboSavedBy;
	private String oboAutoGenBy;
	private String oboRemark;
	
	public DAGParser() {
		goids = new ArrayList<GOEntry>();
		nodes = new ArrayList<GONode>();
		alts = new TreeSet<GOEntry>();
		readingTerm = false;
		unusedRels = new TreeSet<String>();
		groups = new TreeMap<String,String>();
		groups.put("cellular_component", "c");
		groups.put("biological_process", "b");
		groups.put("molecular_function", "m");
	}
	
	private void resetValues() {
		node = null;
		term = null;
		alts.clear();
		readingTerm = false;
		termIndex = -5;
	}

	@Override
	public void readFormatVersion(String version) throws OBOParseException {
		oboVersion = version;
	}

	@Override
	public void readDate(Date date) throws OBOParseException {
		oboDate = date;
	}

	@Override
	public void readSavedBy(String savedBy) throws OBOParseException {
		oboSavedBy = savedBy;
	}

	@Override
	public void readAutogeneratedBy(String autogeneratedBy) throws OBOParseException {
		oboAutoGenBy = autogeneratedBy;
	}

	@Override
	public void readRemark(String remark) throws OBOParseException {
		oboRemark = remark;
	}

	@Override
	public boolean startStanza(String name) throws OBOParseException {
		if (readingTerm) {
			resetValues();
		}
		obsolete = false;
		if (name.equalsIgnoreCase("term")) {
			readingTerm = true;
			termIndex = nodes.size();
			node = new GONode(nodes.size());
			nodes.add(node);
			return true;
		} else {
			return false;
		}
	}

	@Override
	public void readID(String id, NestedValue val) throws OBOParseException {
		if (readingTerm) {
			term = new GOEntry(id, termIndex, true);
			goids.add(term);
		} else if (obsolete) {
			goids.add(new GOEntry(id, -1, true));
		}
	}

	@Override
	public void readAltID(String id, NestedValue val) throws OBOParseException {
		if (readingTerm) {
			GOEntry entry = new GOEntry(id, termIndex, false);
			alts.add(entry);
			goids.add(entry);
		} else if (obsolete) {
			goids.add(new GOEntry(id, -1, false));
		}
	}

	@Override
	public void readIsa(String id, String ns, boolean completes,
			boolean implied, NestedValue val) throws OBOParseException {
		if (readingTerm) {
			node.addParent(id);
		}
	}

	@Override
	public void readRelationship(String relType, String id, boolean necessary,
			boolean inverseNecessary, boolean completes, boolean implied,
			Integer minCardinality, Integer maxCardinality,
			Integer cardinality, String ns, NestedValue val, List<String> args,
			boolean parentIsProperty) throws OBOParseException {
		if (readingTerm) {
			if (relType.equalsIgnoreCase("part_of")) {
				node.addParent(id);
				return;
			}
		}
		unusedRels.add(relType);
	}

	@Override
	public void endParse() throws OBOParseException {
		link();
//		for (String str : unusedRels) {
//			System.out.println(str);
//		}
	}

	@Override
	public void readIsObsolete(NestedValue val) throws OBOParseException {
		if (readingTerm) {
			term.index = -1;
			for (GOEntry alt : alts) {
				alt.index = -1;
			}
			nodes.remove(nodes.size()-1);
			resetValues();
			obsolete = true;
		}
	}
	@Override
	public void readName(String name, NestedValue val) throws OBOParseException {
		if (readingTerm) {
			node.name = name;
		}
	}

	@Override
	public void readNamespace(String ns, NestedValue val) throws OBOParseException {
		if (readingTerm) {
			String group = groups.get(ns);
			if (group == null) throw new Error("Unknown group");
			node.group = group;
		}
	}

	private void link() {
		Collections.sort(goids);
		for (int i = 0; i < nodes.size(); ++i) {
			GONode node  = nodes.get(i);
			for (String id : node.isa) { //lookup the parent node ids
				int pindex = Collections.binarySearch(goids, id);
				if (pindex < 0) throw new RuntimeException("Couldn't find referenced node " + id);
				GONode parent = nodes.get(goids.get(pindex).index);
				node.parents.add(parent);
				parent.children.add(node);
			}
		}
	}

	public void write(BufferedWriter out) throws IOException {
		out.write("# Generated from an OBO file with the details:\n");
		out.write("# OBO Format Version: " + oboVersion + "\n");
		out.write("# OBO Date: " + oboDate.toString() + "\n");
		out.write("# OBO Saved By: " + oboSavedBy + "\n");
		out.write("# OBO Autogenerated By: " + oboAutoGenBy + "\n");
		out.write("# OBO Remark: " + oboRemark + "\n");
		out.write(Integer.toString(nodes.size()));//output number of nodes in DAG
		out.write('\n');
		for (int i = 0; i < nodes.size(); ++i) {
			GONode node = nodes.get(i);
			//output node group first, so it's easy to select by
			out.write(node.group);
			out.write('\n');
			//assume ASCII string Ie. single byte per character
			out.write(Integer.toString(node.name.length())); //output bytes length of description
			out.write('\t');
			out.write(node.name); //output description
			out.write('\n');
			//output node statistics
			out.write(Integer.toString(node.getNAbove()));
			out.write('\t');
			out.write(Integer.toString(node.getNBelow()));
			out.write('\n');
			//output parent node indexes
			out.write(Integer.toString(node.parents.size())); //output number of parent nodes for node
			for (GONode parent : node.parents) {
				out.write('\t');
				out.write(Integer.toString(parent.index)); //output a parent node index
			}
			out.write('\n');
			//output child node indexes
			out.write(Integer.toString(node.children.size())); //output number of child nodes for node
			for (GONode child : node.children) {
				out.write('\t');
				out.write(Integer.toString(child.index)); //output a child node index
			}
			out.write('\n');
		}
		out.write(Integer.toString(goids.size()));
		out.write('\n');
		for (GOEntry goid : goids) {
			if (goid.preferred) {
				out.write(">\t");
			} else {
				out.write("+\t");
			}
			out.write(goid.goid);
			out.write('\t');
			out.write(Integer.toString(goid.index + 1));
			out.write('\n');
		}
		out.flush();
	}
	
	public static void parse(File oboFile, File outFile) throws IOException, OBOParseException {
		DAGParser parser = new DAGParser();
		OBOParseEngine engine = new OBOParseEngine(parser);
		engine.setPath(oboFile.getPath());
		engine.parse();
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(outFile));
			parser.write(out);
			out.close();
			out = null;
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (IOException e){}//ignore
				out = null;
			}
		}
		System.out.println("Done");
	}
	
	public static void main(String[] args) throws IOException {
		if (args.length != 2) {
			System.err.println("Expected <GO OBO File> <Output File>");
			return;
		}
		File oboFile, outFile;
		oboFile = new File(args[0]);
		if (!oboFile.exists()) {
			System.err.println("OBO file doesn't exist");
			return;
		}
		if (!oboFile.isFile()) {
			System.err.println("OBO file isn't a normal file.");
			return;
		}
		if (!oboFile.canRead()) {
			System.err.println("OBO file can't be read.");
			return;
		}
		outFile = new File(args[1]);
		try {
			parse(oboFile, outFile);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private static class GOEntry implements Comparable<Object> {
		public final String goid;
		public int index;
		public final boolean preferred;
		
		public GOEntry(String goid, int index, boolean preferred) {
			this.goid = goid;
			this.index = index;
			this.preferred = preferred;
		}
		
		public int compareTo(Object o) {
			if (o instanceof GOEntry) {
				return this.goid.compareTo(((GOEntry)o).goid);
			} else if (o instanceof String) {
				return this.goid.compareTo(((String)o));
			}
			return 0;
		}
	}
	
	private static class GONode implements Comparable<GONode> {
		public final int index;
		public List<String> isa;
		public Set<GONode> parents;
		public Set<GONode> children;
		public String group;
		public String name;
		private int nabove;
		private int nbelow;
		private boolean mark;
		
		
		public GONode(int index) {
			this.index = index;
			isa = new ArrayList<String>();
			parents = new TreeSet<GONode>();
			children = new TreeSet<GONode>();
			nabove = -1;
			nbelow = -1;
			mark = false;
			name = "";
		}
		
		public void addParent(String name) {
			isa.add(name);
		}
		
		private int markUp() {
			if (mark) return 0;
			mark = true;
			int unmarked = 1;
			for (GONode parent : parents) {
				unmarked += parent.markUp();
			}
			return unmarked;
		}
		
		private void clearUp() {
			if (!mark) return;
			mark = false;
			for (GONode parent : parents) {
				parent.clearUp();
			}
		}
		
		private int markDown() {
			if (mark) return 0;
			mark = true;
			int unmarked = 1;
			for (GONode child : children) {
				unmarked += child.markDown();
			}
			return unmarked;
		}
		
		public void clearDown() {
			if (!mark) return;
			mark = false;
			for (GONode child : children) {
				child.clearDown();
			}
		}
		
		/*
		 * returns the number of nodes above
		 */
		public int getNAbove() {
			if (nabove == -1) {
				nabove = markUp() - 1;
				clearUp();
			}
			return nabove;
		}
		
		/*
		 * returns the number of nodes below
		 */
		public int getNBelow() {
			if (nbelow == -1) {
				nbelow = markDown() - 1;
				clearDown();
			}
			return nbelow;
		}

		@Override
		public int compareTo(GONode o) {
			return index - o.index;
		}
	}
}
