/*
 * Copyright (C) 2002-2011 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "rodent.h"
#include "rfm_modules.h"

/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "dotdesktop"
#define SUBMODULE_NAME "dotdesktop"
#define MODULE_LABEL _("Application Launcher...")
//#define MODULE_ICON_ID "xffm/places_folder-system"
#define MODULE_ENTRY_TIP _("Next-generation application launcher.")

#include "module-skeleton.h"
// Skeleton definitions

G_MODULE_EXPORT RFM_MODULE_PAUSE(NULL)
G_MODULE_EXPORT RFM_MONITOR_RELOAD(NULL)

G_MODULE_EXPORT RFM_BLOCKING_LOAD(TRUE)
G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
//G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

//Not applicable:
//G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY("RODENT-DOTD");
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(TRUE)
//Superceded:
//G_MODULE_EXPORT RFM_PLUGIN_INFO(_(""))
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(TRUE)
//
G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

#include "dotdesktop-module.i"

 
G_MODULE_EXPORT
void *
module_icon_id(void){
    // this is the module root item
    static gchar *icon=NULL;
    if (icon == NULL){
	icon = g_strdup_printf("%s/pixmaps/rodent-dotdesktop.svg", PACKAGE_DATA_DIR);
    }
    return icon;
}

////////////// General gmodule initialization function 

G_MODULE_EXPORT void *
g_module_check_init (GModule * module) {
    BIND_TEXT_DOMAIN;
    //NOOP("dotdesktop init... \n");
    if (!glob_signal) rfm_cond_init(glob_signal);

    if(!glob_mutex) {
	rfm_mutex_init(glob_mutex);
    }
	
    icon_hash = g_hash_table_new_full (g_str_hash, g_str_equal,g_free, g_free);
    icon_exec_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
    category_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
       
    reverse_string_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
    string_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);

    dotdesktop_t *qq=dotdesktop_v;
    GMutex *string_hash_mutex = get_string_hash_mutex();
    for (; qq && qq->category; qq++){
	if (qq->string != NULL) {
	    g_mutex_lock(string_hash_mutex);
	    g_hash_table_replace(string_hash, g_strdup(qq->category), g_strdup(qq->string));
	    g_mutex_unlock(string_hash_mutex);
	}
    }      
    
    rfm_view_thread_create(NULL, glob_dir_f, NULL, "glob_dir_f");
    rfm_view_thread_create(NULL, populate_icon_hash_f, NULL, "populate_icon_hash_f");
    rfm_view_thread_create(NULL, populate_mimetype_hash_f, NULL, "populate_mimetype_hash_f");
    // This thread requires a termination signal from janitor.
    rfm_view_thread_create (NULL, monitor_f, NULL, "monitor_f:dotdesktop");
    return NULL;
}

G_MODULE_EXPORT void 
g_module_unload(GModule * module){
    unload_module = TRUE;
    rfm_global_t *rfm_global_p = rfm_global();
    g_cond_signal(rfm_global_p->status_signal);
    g_hash_table_destroy(icon_hash);
    g_hash_table_destroy(icon_exec_hash);
    g_hash_table_destroy(category_hash);
    g_hash_table_destroy(reverse_string_hash);
    g_hash_table_destroy(string_hash);
}
//
//////////////  Generalized Root Module functions ///////////////////
//

// Plugin functions. Plugin functions may be specified as one of the
// following types.
//
// void:     These functions all return a void pointer
//   	     and take no argument
// natural:  These functions all return a void pointer
// 	     and take a single void pointer argument.
// 	     The space of natural functions is isomorphic
// 	     to the set of natural numbers.
// rational: These functions all return a void pointer
// 	     and take a two void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of rational numbers.
// complex:  These functions all return a void pointer
// 	     and take a three void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of complex numbers with integer
// 	     imaginary component.
	

//////////////////////////////////////////////////////////////////////
//                          void functions                          //
//////////////////////////////////////////////////////////////////////



//  gchar *  
// This function returns a newly allocated string with the general information
// of the module. Rodent uses this to construct the popup tip. Returned value
// should be freed when no longer used.
G_MODULE_EXPORT
void *
plugin_info(void){
    return g_strdup_printf("%s:\n%s\n",
	    _("Browse and run installed applications"),
	    _("This program is responsible for launching other applications and provides useful utilities."));
}


//////////////////////////////////////////////////////////////////////
//                        natural functions                         //
//////////////////////////////////////////////////////////////////////

// gint
// This function returns the count of elements in the module's
// view. This value is compared with current value to determine
// whether a reload is necessary.
// Returns READ_ERROR	 on skip unconditional reload
// Parameter p is the view's widgets_p.
G_MODULE_EXPORT 
void *
module_count (void *p) {
    static gint count=0;
    widgets_t *widgets_p = p;
    gint serial=GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widgets_p->paper), "dotdesktop_serial"));
    if (serial == desktop_serial){
	return GINT_TO_POINTER (READ_ERROR);
    }
    g_object_set_data(G_OBJECT(widgets_p->paper), "dotdesktop_serial", GINT_TO_POINTER(desktop_serial));
    return GINT_TO_POINTER (count++);
}

// gboolean  
// This function processes the double click action on the
// icon's label. If it returns FALSE, then the default
// Rodent action for label click is performed. The default
// Rodent action checks for file existance to do a rename
// entry (or symlink or duplicate).
// Parameter p is the item's entry pointer. 
G_MODULE_EXPORT 
void *label_click (void *p) {
    // Do nothing.
    record_entry_t *en=p;
    if (en && en->mimetype && strcmp(en->mimetype, "application/x-desktop")==0){
	// do default Rodent action (rename/duplicate/link)
	// nah! Problem with updating, since items are not currently
	//      monitored
	// return NULL;
    } 
    // Do nothing
    return  GINT_TO_POINTER(1);
}


// const gchar * 
// This function returns a const pointer to the entry's icon
// identifier. 
// Parameter p is the item's entry pointer. 
// Identifier may be returned as a mimetype or an absolute path.
G_MODULE_EXPORT
void *item_icon_id(void *p){
    return dotdesktop_item_icon_id(p);
}

//  gchar *  
// This function returns a newly allocated string with the general information
// of the entry (parameter p). Rodent uses this to construct the popup tip.
// Returned value should be freed when no longer used.
G_MODULE_EXPORT 
void *
item_entry_tip(void *p){
    return dotdesktop_entry_tip(p);
}

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// selectable icon or not.
/*G_MODULE_EXPORT
void *
is_selectable(void *p){
    record_entry_t *en=(record_entry_t *)p;
    if (rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
	return GINT_TO_POINTER(1);
    }
    return NULL;
}*/

// gboolean  
// This function informs Rodent if the entry (parameter p) is a 
// valid drop site. Drop may be processed by Rodent or by the
// module if process_drop() is defined.
G_MODULE_EXPORT
void *
valid_drop_site(void *p){
    record_entry_t *en=p;
    if (en && en->mimetype && strcmp(en->mimetype, "application/x-desktop")==0){
	// Dotdesktop files will be valid drop sites.
	return GINT_TO_POINTER(1);
    }
    // Anything else is not a valid drop site.
    return NULL; 
}


// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT
void *
module_xfdir_get (void *p) {
    xfdir_t *xfdir_p = p;
    return private_get_xfdir(xfdir_p);

}

// gboolean
// This is a dotdesktop specific function.
// The function informs Rodent if the execution of the dotdesktop
// file is to be done in a terminal emulator. 
// This flag is set within the dotdesktop file.
G_MODULE_EXPORT
void *exec_in_terminal(void *p){
    gchar *path=p;
    gboolean in_terminal=get_desktop_bool("Terminal", path);
    return (GINT_TO_POINTER(in_terminal));
}

// gchar *
// This is a dotdesktop specific function.
// This function returns a newly allocated string with the "Exec" command
// of the dotdesktop file. Rodent uses this command to spawn a 
// background process. 
// Returned value should be freed when no longer used.
// Parameter p is the path to the file.
// This variable is set within the dotdesktop file.
G_MODULE_EXPORT
void *item_exec(void *p){
    gchar *path=p;
    if (!path) return NULL;
    gchar *exec=get_desktop_string("Exec", path);
    return (void *) exec;
}

// gchar *
// This is a dotdesktop specific function.
// This function returns a newly allocated string with the "Name"
// of the dotdesktop file. Rodent uses this command to create
// popup menu item for files with "aaplication/x-desktop" mimetype.
// 
// Returned value should be freed when no longer used.
// Parameter p is the path to the file.
// This variable is set within the dotdesktop file.
G_MODULE_EXPORT
void *item_name(void *p){
    gchar *path=p;
    if (!path) return NULL;
    gchar *name=get_desktop_string("Name", path);
    return (void *) name;
}

// gchar *
// This is a dotdesktop specific function.
// This function returns a newly allocated string with the "TryExec" command
// of the dotdesktop file. Rodent uses this command if "Exec" is not found
// in the dotdesktop file to spawn a background process. 
// Returned value should be freed when no longer used.
// This variable is set within the dotdesktop file.
G_MODULE_EXPORT
void *item_tryexec(void *p){
    gchar *path=p;
    gchar *exec=get_desktop_string("TryExec", path);
    return (void *) exec;
}

// const gchar
// This function is used to get the icon associated to 
// a particular exec defined in any one of the globbed
// dot desktop files.
G_MODULE_EXPORT
void *
get_exec_icon(void *p)
{
    GMutex *exec_hash_mutex = get_exec_hash_mutex();
    const gchar *exec = p;
    gchar *hash_key = get_hash_key(exec);
    g_mutex_lock(exec_hash_mutex);
    const gchar *iconpath = g_hash_table_lookup (icon_exec_hash, hash_key);
    g_mutex_unlock(exec_hash_mutex);
    g_free(hash_key);
    return (void *)iconpath;
}

//////////////////////////////////////////////////////////////////////
//                        rational functions                        //
//////////////////////////////////////////////////////////////////////
// gboolean
// This function is used to hide popup menu elements of 
// Rodent's standard icon area popup menu.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT 
void *
hide_local_menu_items (void *p, void *q) {
    return dotdesktop_hide_local_menu_items(p, q);
}

// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
double_click(void *p, void *q){
    return dotdesktop_double_click(p, q);
}

// gboolean
// This function is to generate a module specific popup menu, either on
// icon click or on void space click . 
// If this function will generate a popup menu, return value should be TRUE,
// otherwise return FALSE so Rodent will generate the default popup menu.
// Popup menu for modules should be set as view data on the paper object
// identified by "private_popup_menu".
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
    if (!q) return NULL;
    // no popup menu for module items...
    record_entry_t *en = q;
    gchar *text;
    if (rfm_g_file_test(en->path, G_FILE_TEST_EXISTS)){
	gchar *name=get_desktop_string("Name",en->path);
	gchar *generic_name=get_desktop_string("GenericName",en->path);
	gchar *exec=get_desktop_string("Exec",en->path);
	gchar *comment=get_desktop_string("Comment",en->path);
	gboolean in_term=get_desktop_bool("Terminal",en->path);

	text = g_strconcat(
		"<big><b>",name,"</b></big>\n",
		(generic_name)?"<i>(":"",
		(generic_name)?generic_name:"",
		(generic_name)?")</i>\n\n":"",
		(comment)?comment:"",
		(comment)?"\n\n":"",
		_("Command to run when clicked:"), " ", exec, "\n\n", 
		_("Terminal application"), ": ", (in_term)?_("Yes"):_("No"),
		NULL);
	g_free(name);
	g_free(generic_name);
	g_free(exec);
	g_free(comment);
    } else {
        text = g_strdup_printf("<big><b>%s</b></big>\n\n%s  <b><i>%s</i></b>", 
	    MODULE_ENTRY_TIP, _("Group"), (en->tag)?en->tag:en->path);
    }
    rfm_confirm(p, GTK_MESSAGE_INFO, text, NULL, NULL);
    g_free(text);
    return GINT_TO_POINTER(1);
}

// gchar *
// This function returns a newly allocated string which shall
// be used by Rodent to complement the initial text which
// appears in the status area (lp command area).
// Returned value should be freed when no longer used.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
sizetag(void *p, void *q){
    //widgets_t *widgets_p=p;
    record_entry_t *en=q;
    if (en && en->st == NULL){
	return g_strdup(_("List of categories"));
    } else {
	return g_strdup(_("Browse and run installed applications"));
    }
}

//////////////////////////////////////////////////////////////////////
//                        complex functions                         //
//////////////////////////////////////////////////////////////////////

// gboolean  
// This function is used by Rodent to process a drop to a module defined
// icon. Drop is only processed if valid_drop_site() returned TRUE
// for the drop target entry beforehand. 
// Parameter p is the view's widgets_p pointer.
// Parameter q is path of the drop target entry.
// Parameter r is a GList * that contains the drop element paths.
G_MODULE_EXPORT
void *
process_drop(void *p, void *q, void *r){
    widgets_t *widgets_p=p;
    gchar *path=q;
    GList *list=r;
    execute_dot_desktop(widgets_p, path, list);
    return GINT_TO_POINTER(1);
}

static void *
thread_applications_menu_f(gpointer data){
    NOOP(stderr,"thread_applications_menu now\n");

    const gchar *parent_id = "applications_menu";
    const gchar *menu_id = "applications_menu_menu";
    GtkWidget *parent=rfm_get_widget(parent_id);
    GtkWidget *popup_widget=rfm_get_widget(menu_id);
    GtkWidget *old_popup= popup_widget;
    g_object_ref(old_popup);
    
    popup_widget = gtk_menu_new ();
    gtk_widget_show(popup_widget);
    rfm_set_widget(popup_widget, menu_id);
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (parent), popup_widget);
    g_object_unref(old_popup);
    GtkWidget *v = gtk_menu_item_new_with_mnemonic ("rodent-dotdesktop");
    gtk_widget_show (v);
    gtk_container_add (GTK_CONTAINER (popup_widget), v);
    gtk_widget_set_sensitive (v, FALSE);
    NOOP(stderr,"populate_submenu now...\n");
    return popup_widget;
}

static void *
thread_applications_menu(gpointer data){
    GtkWidget *w = rfm_context_function(thread_applications_menu_f, data);
    rodent_thread_add_menu_separator (w, NULL);  
    populate_submenu(w);
    return w;
}



// GtkWidget *
// This returns the menu for a nondetached menu
// Parameter p is the view's widgets_p pointer.
// Parameter q is the parent for the menu widget
//   (i.e., the spot where it will be attached)
//   

G_MODULE_EXPORT
void *
dotdesktop_nondetached_menu(void *p, void *q){
    NOOP(stderr, "dotdesktop_nondetached_menu now... desktop_serial=%d\n", desktop_serial);
    if (rfm_get_gtk_thread() != g_thread_self()){
	DBG("dotdesktop_nondetached_menu is a main callback function only\n");
	return GINT_TO_POINTER(1);
    }
    const gchar *menu_id = "applications_menu_menu";
    GtkWidget *popup_widget=rfm_get_widget(menu_id);
    
    static gint serial=-1;
    if (popup_widget==NULL){
	DBG("dotdesktop_nondetached_menu: this is wrong...\n");
	return NULL;
    }

    if (serial != desktop_serial) {
	serial = desktop_serial;
	rfm_view_thread_create(NULL, thread_applications_menu, NULL, "thread_applications_menu:dotdesktop");
    } 

    return GINT_TO_POINTER(1);
    //return popup_widget;
}

