/*
 * Copyright (C) 2014 Edscott Wilson Garcia
 * EMail: edscott.wilson.garcia@gmail.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "rodent.h"


/* this should be first 2 lines after headers: */
G_MODULE_EXPORT 
LIBRFM_MODULE


#define MODULE_NAME "pkg"
#define SUBMODULE_NAME "pkg"
#define MODULE_LABEL _("Package Manager")
#define PKG_MODULE_ICON_ID "xffm/emblem_package/compositeSW/emblem_bsd"
#define ZYPPER_MODULE_ICON_ID "xffm/emblem_package/compositeSW/emblem_opensuse"
#define YUM_MODULE_ICON_ID "xffm/emblem_package/compositeSW/emblem_centos"
#define APT_MODULE_ICON_ID "xffm/emblem_package/compositeSW/emblem_debian"
#define EMERGE_MODULE_ICON_ID "xffm/emblem_package/compositeSW/emblem_gentoo"
#define PACMAN_MODULE_ICON_ID "xffm/emblem_package/compositeSW/emblem_archlinux"
#define FALLBACK_MODULE_ICON_ID "xffm/emblem_package/compositeSW/emblem_star"
#define MODULE_ENTRY_TIP _("A package manager")

#include "module-skeleton.h"
// Skeleton definitions
//G_MODULE_EXPORT RFM_G_MODULE_CHECK_INIT

G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
//G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP
G_MODULE_EXPORT RFM_BLOCKING_LOAD(TRUE)

G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY("RODENT-PKG");
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(TRUE)
G_MODULE_EXPORT RFM_PLUGIN_INFO(_("Configure external packages"))
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(FALSE)
G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

////////////// Specific module initialization function 
static gchar *pkg_command=NULL;
#include "pkg-module.i"
void *
g_module_check_init(GModule *module){
    NOOP(stderr, "g_module_check_init pkg\n");
    pkg_command=get_defaults();
#ifdef ENABLE_NLS
    /* this binds rodent domain: */
    bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    DBG("rodent-pkgmain(): textdomain=%s (%s) locale_dir=%s (%s)\n",
	    textdomain(NULL), GETTEXT_PACKAGE,
	    bindtextdomain(textdomain(NULL), NULL), PACKAGE_LOCALE_DIR);
# ifdef HAVE_BIND_TEXTDOMAIN_CODESET
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
# endif
#endif   
    gchar *xmlfile = g_strdup_printf("%s/rfm/plugins/xml/pkg_%s.xml",LIBDIR, pkg_command);
    
    if (!parse_xml(xmlfile, pkg_command)) {
        g_free(xmlfile);
        return NULL;
    }
    g_free(xmlfile);
    rfm_view_thread_create(NULL, thread_popup, NULL, "thread_popup:pkg");
    return NULL;
}  

void *
module_icon_id(void){
    if (emerge)return EMERGE_MODULE_ICON_ID;
    else if (pkg) return PKG_MODULE_ICON_ID; 
    else if (zypper) return ZYPPER_MODULE_ICON_ID;
    else if (yum) return YUM_MODULE_ICON_ID;
    else if (apt) return APT_MODULE_ICON_ID;
    else if (pacman) return PACMAN_MODULE_ICON_ID;
    else return FALLBACK_MODULE_ICON_ID;
}
 

// gboolean
// This function is to generate a module specific popup menu, either on
// icon click or on void space click . 
// If this function will generate a popup menu, return value should be TRUE,
// otherwise return FALSE so Rodent will generate the default popup menu.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
private_popup(void *p, void *q){
    TRACE( "private_popup:pkg\n");
    return pkg_popup(p, q);
}

//////////////  Generalized Root Module functions ///////////////////

// Plugin functions. Plugin functions may be specified as one of the
// following types.
//
// void:     These functions all return a void pointer
//   	     and take no argument
// natural:  These functions all return a void pointer
// 	     and take a single void pointer argument.
// 	     The space of natural functions is isomorphic
// 	     to the set of natural numbers.
// rational: These functions all return a void pointer
// 	     and take a two void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of rational numbers.
// complex:  These functions all return a void pointer
// 	     and take a three void pointer arguments.
// 	     The space of rational functions is isomorphic
// 	     to the set of complex numbers with integer
// 	     imaginary component.
	

//////////////////////////////////////////////////////////////////////
//                          void functions                          //
//////////////////////////////////////////////////////////////////////

void *
module_argv(void *p, void *q){
    record_entry_t *en=p;
    if (!en) {
	DBG("ps-module: en == NULL\n");
	return NULL;
    }
    gchar **argv=q;
    if (!argv) {
	DBG("ps-module: argv == NULL\n");
	return NULL;
    }
    gchar *cmd = NULL;
    gchar **a = q;
    // Start with argument 2 (arg 0 would be rodent-plug, 1 pkg)
    a++; if (a) a++;
    for (;a && *a; a++){
        if (!cmd) cmd = g_strdup("search ");
        else {
            gchar *f = g_strconcat(cmd, " ", NULL);
            g_free(cmd);
            cmd = f;
        }
        gchar *g = g_strconcat(cmd, *a, NULL);
        g_free(cmd);
        cmd=g;
    }
    if (!cmd || !strlen(cmd)) {
        g_free(cmd);
        return NULL;
    }
    g_free(en->path);
    en->path = cmd;

    return GINT_TO_POINTER(1);
}    


//////////////////////////////////////////////////////////////////////
//                        natural functions                         //
// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
G_MODULE_EXPORT
void *
module_xfdir_get (void *p) {
    NOOP(stderr, "get module xfdir...\n");
    xfdir_t *xfdir_p = p;
    widgets_t *widgets_p = &(xfdir_p->view_p->widgets);

    static gboolean warned = FALSE;
    if (!warned) {
        if (pkg) {
            gchar *cmd = g_strdup_printf("%s -v", pkg);

            FILE *pp = popen(cmd, "r");
            if (!pp){
                rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-warning", NULL);
                rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", g_strdup_printf("%s: %s\n",cmd, _("unknown error")));
                g_free(cmd);
                return NULL;

            }
            g_free(cmd);
            gchar line[256];
            memset(line,0,256);
            if(fgets(line, 255, pp)){
                rfm_threaded_show_text(widgets_p);
                rfm_threaded_diagnostics(widgets_p, 
                        "xffm/emblem_package", NULL);
                rfm_threaded_diagnostics(widgets_p, 
                        "xffm_tag/blue", g_strdup_printf("pkg-%s", line));
            }
            pclose(pp);
        }
        if (geteuid() != 0){
            gchar *t="If you are behind a proxy wall, make sure sudo is configured to keep your http_proxy and ftp_proxy environment variables.\n";
            gchar *s="This is done with visudo, add \"http_proxy\" and \"ftp_proxy\" to Defaults\n";
	    rfm_threaded_show_text(widgets_p);
            rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-info", NULL);
            rfm_threaded_diagnostics(widgets_p, "xffm_tag/red", g_strdup(t));
            rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-info", NULL);
            rfm_threaded_diagnostics(widgets_p, "xffm_tag/red", g_strdup(s));
        } else {
            gchar *argv[] = {"pkg", "stats", NULL};
            rfm_thread_run_argv (widgets_p, argv, FALSE);
        }
        warned = TRUE;
    }

    GtkWidget *hold_your_horses = rfm_context_function(hold_your_horses_f, NULL);
    gint retval=pkg_xfdir_get(p);
    rfm_context_function(loose_your_horses_f, hold_your_horses);
    
    return GINT_TO_POINTER(retval);
}

G_MODULE_EXPORT
void *double_click(void *p, void *q){
    //DBG("void *double_click\n");
    widgets_t *widgets_p = p;
    record_entry_t *en = q;
    if (!en) return NULL;
    if (IS_ROOT_TYPE(en->type)) return NULL;
    if (g_path_is_absolute(en->path) && g_file_test(en->path, G_FILE_TEST_IS_DIR)) return NULL;
    
    if (strcmp(en->path, _("Search"))==0){
        pkg_command_t *c = xml_cmds;
        for (;c && c->pkg; c++){
            if (!c->cmd) continue;
            const gchar *search="search";
            if (emerge) search="--search";
            else if (pacman) search="-Ss";
            if (strcmp(c->cmd, search)==0){
                fprintf(stderr, "process command: %s\n", c->cmd);
                process_cmd(NULL, c);
                return GINT_TO_POINTER(1);
            }
        }
        //return GINT_TO_POINTER(1);
    }
    //rfm_threaded_diagnostics(widgets_p, "rodent", g_strdup("And here we could open a dialog to act upon the selected item or items (wishlist)\n"));
    pkg_popup(p, q);
    return GINT_TO_POINTER(1);
}
    

// const gchar * 
// This function returns a const pointer to the entry's icon
// identifier. 
// Parameter p is the item's entry pointer. 
// Identifier may be returned as a mimetype or an absolute path.
G_MODULE_EXPORT
void *
item_icon_id (void *p) {
    record_entry_t *en = p;
    if (strcmp(en->path, _("Search"))==0) return "xffm/emblem_find";
    if (g_path_is_absolute(en->path)) {
	if (pkg)
	    return "xffm/emblem_bsd/compositeSE/stock_directory";
        else if (emerge)
            return "xffm/emblem_gentoo/compositeSE/stock_directory";
        else if (zypper)
	    return "xffm/emblem_opensuse/compositeSE/stock_directory";
        else if (yum)
	    return "xffm/emblem_centos/compositeSE/stock_directory";
        else if (apt)
	    return "xffm/emblem_debian/compositeSE/stock_directory";
        else if (pacman)
	    return "xffm/emblem_archlinux/compositeSE/stock_directory";
        else
	    return "xffm/emblem_gentoo/compositeSE/stock_directory";
    }

    if (IS_LOCAL_TYPE(en->type)) return "xffm/emblem_package/compositeNE/emblem_greenball";
    if (!en || strcmp(en->path,MODULE_LABEL)==0) {
        if (pkg) return PKG_MODULE_ICON_ID; 
        else if (emerge) return EMERGE_MODULE_ICON_ID;
        else if (zypper) return ZYPPER_MODULE_ICON_ID;
        else if (yum) return YUM_MODULE_ICON_ID;
        else if (apt) return APT_MODULE_ICON_ID;
        else if (pacman) return PACMAN_MODULE_ICON_ID;
        else return FALLBACK_MODULE_ICON_ID;
    }
    if (pkg){
        if (IS_PLUGIN_TYPE1(en->type)) // older version installed.
            return "xffm/emblem_bsd/compositeNE/emblem_star";
        else
            return "xffm/emblem_bsd/compositeNE/emblem_package";
    }
    else if (emerge)
	return "xffm/emblem_gentoo/compositeNE/emblem_package";
    else if (yum)
	return "xffm/emblem_centos/compositeNE/emblem_package";
    else if (zypper)
	return "xffm/emblem_opensuse/compositeNE/emblem_package";
    else if (apt)
	return "xffm/emblem_debian/compositeNE/emblem_package";
    else if (pacman)
        return "xffm/emblem_archlinux/compositeNE/emblem_package";
    else 
	return "xffm/emblem_star/compositeNE/emblem_package";
}
//this should be the label tip. entry tip should cook up a preview with package info:../src/pkg-module.c: En la función ‘item_icon_id’:
//pkg query "%n %v %o %p %m %c %e %w %l %sh %a %q %k %M %t" en->path
G_MODULE_EXPORT
void *
item_entry_tip(void *p){
    if (pthread_mutex_trylock(&db_mutex)!=0){
	DBG("get_tooltip_text(): cannot lock mutex.\n");
	//return g_strdup(_("Work in progress..."));
	return NULL;
    } 
    gchar *ret = get_tooltip_text((record_entry_t *)p);
    pthread_mutex_unlock(&db_mutex);
    return ret;
}


