#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2014 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
//
//
#define POPUP_ID "pkg_menu"
#define POPUP_MENU_ID "pkg_menu_menu"
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>

static gchar *rpm;

static pthread_mutex_t db_mutex = PTHREAD_MUTEX_INITIALIZER; 
static gchar *pkg_dbdir;
static gchar *pkg_cachedir;
static gchar *portsdir;

static void do_it(widgets_t *widgets_p, const gchar *cmd, gint flags);
static GHashTable *installed_hash=NULL;
static gchar *package_name(gchar *version_name);
static void process_cmd(GtkMenuItem *m, gpointer data);
static void *pkg_confirm_f(void *data);
#include "pkg-menu.i"
#include "pkg-zypper.i"
#include "pkg-yum.i"
#include "pkg-xml.i"
#include "pkg-pkg.i"
#include "pkg-emerge.i"
#include "pkg-apt.i"
#include "pkg-pacman.i"

#define PLAIN_OPTION    0x01
#define ARGUMENT_OPTION  0x02
#define ARGUMENT_EXCLUSIVE_OPTION  0x04

static gchar *envvar = NULL;

static 
void
process_cmd(GtkMenuItem *m, gpointer data){
    pkg_command_t *c = data;
    if (!c) return;
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    gint flags = 0;
    DBG("pkg:\"%s\" cmd:\"%s\"\n", c->pkg, c->cmd);
    gchar *cmd=NULL;
    gchar *response = rfm_context_function(pkg_confirm_f, c);
    if (response){
            g_object_set_data(G_OBJECT(widgets_p->paper), "flags", NULL);
        DBG("response=%s\n", response);
        if (c->cmd && 
	    (strcmp(c->cmd, "search")==0 ||
	     strcmp(c->cmd, "--search")==0 ||
             strcmp(c->cmd, "-Ss")==0)){
            view_t *view_p = widgets_p->view_p;
            record_entry_t *en = rfm_copy_entry(view_p->en);
            g_free (en->path);
            g_strstrip(response);
            en->path = g_strdup_printf("%s", response);
            DBG("command will be \"emerge %s\" \n", en->path);
            rodent_refresh (widgets_p, en);
            // This would open a new window, which annoys me:
            // gchar * cmd = g_strdup_printf("rodent-plug pkg %s", response);
            // g_free(cmd);
            flags = c->flags;
            g_free(response);
            return;
        }
        
        const gchar *sudo;
        if (geteuid() == 0 || (c->flags & ACCESS_READ)) sudo="";
        else sudo = "sudo -A ";
        cmd = g_strdup_printf("%s%s %s %s", sudo, (envvar)?envvar:"", c->pkg, response);

        if (sudo && strlen(sudo)) {
            // sudoize pcr if necessary
            gchar *pcr=g_object_get_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response");
            gchar *g = g_strdup_printf("%s%s %s", sudo, c->pkg, pcr);
            g_free(pcr);
            pcr=g;
            g_object_set_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response", pcr);
        }

        g_object_set_data(G_OBJECT(widgets_p->paper), "flags", GINT_TO_POINTER(c->flags));


        flags = c->flags;
        rfm_diagnostics(widgets_p, "xffm_tag/blue", cmd, "\n", NULL);	
        DBG("do it --> %s\n", cmd);
        g_free(response);
    }
        
    do_it(widgets_p, cmd, flags);
    g_free(cmd);
    return;
}
/*
static 
void
process_cmd(GtkMenuItem *m, gpointer data){
    if (emerge) return emerge_process_cmd(m, data);
    else if (pkg) return pkg_process_cmd(m, data);
    return;
}*/

static gchar *package_name(gchar *version_name){
    if (!version_name) return NULL;
    gchar *name = g_strdup(version_name);
    gchar *p = strchr(name, '-');
    if (!p) {
	g_free(name);
	return NULL;
    }
    while (p[1] && strchr(p+1, '-') && isdigit(p[1])){
	p = strchr(p+1, '-');
    }
    *p=0;
    return name;
}

static gchar *get_defaults(void){
    gchar *com=NULL;      

    pkg = g_find_program_in_path("pkg");
    emerge = g_find_program_in_path("emerge");
    rpm = g_find_program_in_path("rpm");
    zypper = g_find_program_in_path("zypper");
    yum = g_find_program_in_path("yum");
    apt = g_find_program_in_path("apt-get");
    pacman = g_find_program_in_path("pacman");

    if (pkg) com = g_path_get_basename(pkg);
    else if (emerge) com = g_path_get_basename(emerge);
    else if (zypper) com = g_path_get_basename(zypper);
    else if (yum) com = g_path_get_basename(yum);
    else if (apt) com = g_path_get_basename(apt);
    else if (pacman) com = g_path_get_basename(pacman);
    
    rpm = g_find_program_in_path("rpm");
    dpkg = g_find_program_in_path("dpkg");
   
    // No program found, just load default FreeBSD menu configuration.   
    if (!com) return g_strdup("pkg");
    
    pkg_dbdir = g_strdup("/var/db/pkg");
    if (zypper) 
	pkg_cachedir = g_strdup("/var/cache/zypp");
    else
    	pkg_cachedir = g_strdup("/var/cache/pkg");

    if (!pkg) return com;
    ///////////////////////////////////////////////////////
    //  FreeBSD stuff
    ///////////////////////////////////////////////////////  
    portsdir = g_strdup("/usr/ports");
    const gchar *e = getenv("PKG_DBDIR");
    if (e && strlen(e) && g_file_test(e, G_FILE_TEST_IS_DIR)){
       g_free(pkg_dbdir);
       pkg_dbdir = g_strdup(e);
    }
    FILE *f = fopen("/usr/local/etc/pkg.conf", "r");
    if (f) {
        gchar line[256];
        memset (line, 0, 256);
        while (fgets(line, 255, f) && !feof(f)){
            g_strchug(line);
            if (*line == '#') continue;
            if (strchr(line, '\n'))*strchr(line, '\n') = 0;
            if (strncmp (line, "PKG_DB_DIR", strlen("PKG_DB_DIR"))==0){
                g_free(pkg_dbdir);
                gchar **gg = g_strsplit(line, ":", -1);
                pkg_dbdir = g_strdup(gg[1]);
                g_strstrip(pkg_dbdir);
                g_strfreev(gg);
            }
            if (strncmp (line, "PKG_CACHEDIR", strlen("PKG_CACHEDIR"))==0){
                g_free(pkg_cachedir);
                gchar **gg = g_strsplit(line, ":", -1);
                pkg_cachedir = g_strdup(gg[1]);
                g_strstrip(pkg_cachedir);
                g_strfreev(gg);
            }
            if (strncmp (line, "PORTSDIR", strlen("PORTSDIR"))==0){
                g_free(portsdir);
                gchar **gg = g_strsplit(line, ":", -1);
                portsdir = g_strdup(gg[1]);
                g_strstrip(portsdir);
                g_strfreev(gg);
            }
        }
        fclose(f);
    }
    return com;
}



static 
void *
search_dummy(void *p, const gchar *path, const gchar *module_name){
    xfdir_t *xfdir_p = p;
    xfdir_p->gl[1].en=rfm_mk_entry(0);
    xfdir_p->gl[1].en->st = NULL;
 //   xfdir_p->gl[1].en->parent_module = MODULE_NAME;
    xfdir_p->gl[1].en->module = module_name;
    xfdir_p->gl[1].en->path=g_strdup(path);
    xfdir_p->gl[1].pathv = g_strdup(path);
    SET_DUMMY_TYPE (xfdir_p->gl[1].en->type);
    return p;
}

static 
void *
pkg_dummy(void *p, const gchar *path, const gchar *module_name){
    xfdir_t *xfdir_p = p;
    xfdir_p->gl[0].en=rfm_mk_entry(0);
    xfdir_p->gl[0].en->st = NULL;
    xfdir_p->gl[0].en->parent_module = MODULE_NAME;
    xfdir_p->gl[0].en->module = MODULE_NAME;
    SET_ROOT_TYPE(xfdir_p->gl[0].en->type);
    xfdir_p->gl[0].en->path = g_strdup(MODULE_LABEL);
    xfdir_p->gl[0].pathv = g_strdup(MODULE_LABEL);   
    SET_DUMMY_TYPE (xfdir_p->gl[0].en->type);
    return p;
}

static    pthread_cond_t l_signal = PTHREAD_COND_INITIALIZER;
static    pthread_mutex_t l_mutex = PTHREAD_MUTEX_INITIALIZER;
static    int l_condition=0;
static    GSList *l_list=NULL;

static gboolean check_exit(const gchar *line){
    NOOP(stderr, "check_exit(): %s", line);
    if(strncmp (line, "Tubo-id exit:", strlen ("Tubo-id exit:")) == 0) {
        // all stdout output has been processed.
        pthread_mutex_lock(&(l_mutex));
        l_condition = 1;
        pthread_mutex_unlock(&(l_mutex));
        NOOP(stderr, "Now signalling\n");
        pthread_cond_signal(&(l_signal));
        return TRUE;
    }
    return FALSE;
}

static void io_search_stdout(void *user_data, void *line, int childFD){
    NOOP(stderr,"io_search_stdout\n");
    if (check_exit(line)){
        rfm_operate_stdout(user_data, line, childFD);
        return;
    }
    if (pkg) l_list = add_search_item(l_list, line, user_data);
    else if (emerge) l_list = add_emerge_search_item(l_list, line, user_data);
    else if (zypper) l_list = add_zypper_search_item(l_list, line, user_data);
    else if (yum) l_list = add_yum_search_item(l_list, line, user_data);
    else if (apt) l_list = add_apt_search_item(l_list, line, user_data);
    else if (pacman) l_list = add_pacman_search_item(l_list, line, user_data);
    else fprintf(stderr, "io_search_stdout(): no command process associated!\n");
    
}

static void io_thread_stdout(void *user_data, void *line, int childFD){
    if (check_exit(line)){
        rfm_operate_stdout(user_data, line, childFD);
        return;
    }
    if (pkg) l_list = add_pkg_item(l_list, line);
    else if (emerge) l_list = add_emerge_item(l_list, line);
    else if (rpm) l_list = add_rpm_item(l_list, line);
    else if (dpkg) l_list = add_apt_item(l_list, line);
    else if (pacman) l_list = add_pacman_item(l_list, line);
    else fprintf(stderr, "io_thread_stdout(): no command process associated!\n");
}

static GSList *
get_command_listing(widgets_t *widgets_p, gchar *command, gboolean search){
    if (!pkg_command) return NULL;
    if (pthread_mutex_trylock(&db_mutex)!=0){
        rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", g_strdup(_("Currently busy\n")));
        return NULL;
    }
    l_list = NULL;
    NOOP("command is \"%s\"\n", command);
    if (!command) {
        pthread_mutex_unlock(&db_mutex);
        return NULL;
    }
    // 1. create condition control structure
    //listing_control_t *l = get_listing_control();
    //l->list = &pkg_list;
    //l->search = search;
    // 2. create i/o thread
    g_strstrip(command);
    gchar **arg = g_strsplit(command, " ", -1);
    if (!arg) {
        pthread_mutex_unlock(&db_mutex);
        return NULL;
    }
#ifdef DEBUG
    gchar **p=arg;for(;p && *p;p++)NOOP(stderr, "arg=\"%s\"\n", *p);
#endif
    l_condition=0;
    if (search) rfm_thread_run_argv_full (widgets_p, arg, FALSE, NULL, io_search_stdout, NULL, NULL);
    else rfm_thread_run_argv_full (widgets_p, arg, FALSE, NULL, io_thread_stdout, NULL, NULL);
    g_strfreev(arg);
    // 3. wait on condition
    pthread_mutex_lock(&(l_mutex));
    if (!l_condition){
        NOOP(stderr, "waiting for signal\n");
        pthread_cond_wait(&(l_signal), &(l_mutex));
        NOOP(stderr,"got signal!\n");
    }
    pthread_mutex_unlock(&(l_mutex));
    pthread_mutex_unlock(&db_mutex);
    NOOP(stderr, "command listing routine is done...\n");
    return l_list;
}

static gint
pkg_xfdir_get (void *pp) {
    gboolean search = FALSE;
    xfdir_t *xfdir_p = pp;
    widgets_t *widgets_p = &(xfdir_p->view_p->widgets);
    if (!xfdir_p->en) {
	return 0;
    }

    gchar *command = NULL;

    if (xfdir_p->en->path && strcmp(xfdir_p->en->path, MODULE_LABEL)){
        search = TRUE;
        DBG("pkg path is : %s\n", xfdir_p->en->path);
        if (!xfdir_p->en->path) {
            fprintf(stderr, "Search expression should not be empty\n");
            return 0;
        }
        g_strstrip(xfdir_p->en->path);
        if (!strlen(xfdir_p->en->path)) {
            fprintf(stderr, "Search expression should not be empty\n");
            return 0;
        }
        if (apt) command = g_strdup_printf("apt-cache --names-only %s", xfdir_p->en->path);
        else command = g_strdup_printf("%s %s", pkg_command, xfdir_p->en->path);
        /*if (pkg) command = g_strdup_printf("%s %s", pkg, xfdir_p->en->path);
        else if (emerge) command = g_strdup_printf("%s %s", emerge, xfdir_p->en->path);
	else if (zypper) command = g_strdup_printf("%s %s", zypper, xfdir_p->en->path);
	else if (yum) command = g_strdup_printf("%s %s", yum, xfdir_p->en->path);
        else return 0;*/
    } else {
        if (pkg) command = g_strdup("pkg info");
        else if (emerge) command = g_strdup("fgr -R -i -a -t dir /var/db/pkg");
        // This is for both yum and zypper:
        else if (rpm) command = g_strdup("rpm -qa");
        else if (dpkg) command = g_strdup("dpkg --list");
        else if (pacman) command = g_strdup("pacman -Q");
    }

    NOOP(stderr, "command is:\"%s\"\n", command);    
    if (installed_hash) g_hash_table_destroy(installed_hash); 
    installed_hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);


    xfdir_p->pathc=0;
    rfm_threaded_diagnostics(widgets_p, "xffm/emblem_package", g_strdup_printf("%s\n",command));
    rfm_threaded_diagnostics(widgets_p, "xffm_tag/red", g_strdup_printf("%s\n",
		_("Please wait while querying the server...")));
    GSList *pkg_list = get_command_listing(widgets_p, command, search);
    g_free(command);
    gint first=0;
    NOOP(stderr, "Got command listing...\n");

    xfdir_p->pathc=(pkg_list)?g_slist_length(pkg_list):0;
    if (xfdir_p->pathc==0) {
	gchar *g = g_find_program_in_path(pkg_command);
	if (!g) g = g_strdup_printf(_("The \"%s\" utility is not installed.\nPlease install it."), pkg_command); 
	else g = g_strdup_printf("%s: %s\n", _("No Search Results"), xfdir_p->en->path);
        rfm_confirm(widgets_p, GTK_MESSAGE_INFO, g, NULL, NULL);
	g_free(g);
    }

    NOOP(stderr, "adding items...\n"); 
    
    // up item: 
    xfdir_p->pathc++;
    first++;

    // search item: 
    xfdir_p->pathc++;
    first++;
        
    // ports item
    if (g_file_test(portsdir, G_FILE_TEST_IS_DIR)) {
         xfdir_p->pathc++;
         first++;
    }
    // cache item
    if (g_file_test(pkg_cachedir, G_FILE_TEST_IS_DIR)) {
         DBG("1.testing for cache at %s... found\n", pkg_cachedir);
         xfdir_p->pathc++;
         first++;
    } else DBG("1.testing for cache at %s... NOT found\n", pkg_cachedir);

    xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
    if (!xfdir_p->gl) g_error("malloc: %s\n", strerror(errno));
    memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));

    if (!search){
        // Up is rodent root level.
        xfdir_p->gl[0].en = NULL;
        NOOP(stderr, "PS: up set to root level\n");
        xfdir_p->gl[0].pathv = g_strdup (g_get_host_name ());
    } else {
        pkg_dummy(xfdir_p, MODULE_NAME, MODULE_NAME);
    }

    search_dummy(xfdir_p, _("Search"), MODULE_NAME);
    // cache item
    if (g_file_test(pkg_cachedir, G_FILE_TEST_IS_DIR)) {
         DBG("2.testing for cache at %s... found\n", pkg_cachedir);
        xfdir_p->gl[2].pathv = g_strdup(_("Cache"));
        xfdir_p->gl[2].en = rfm_stat_entry(pkg_cachedir, 0);
        SET_LOCAL_TYPE(xfdir_p->en->type);
    } else DBG("2.testing for cache at %s... NOT found\n", pkg_cachedir);
    // ports item
    if (g_file_test(portsdir, G_FILE_TEST_IS_DIR)) {
        xfdir_p->gl[3].pathv = g_strdup(_("Ports"));
        xfdir_p->gl[3].en = rfm_stat_entry(portsdir, 0);
        SET_LOCAL_TYPE(xfdir_p->en->type);
    }
    

    
    
    // add the rest of the items to the xfdir structure.
    GSList *tmp = pkg_list;
    gint i;
    for (i=first; tmp && tmp->data; tmp=tmp->next,i++){
	xfdir_p->gl[i].en = (record_entry_t *)tmp->data;
	xfdir_p->gl[i].en->tag = g_strdup(xfdir_p->gl[i].en->path);
	xfdir_p->gl[i].pathv = g_strdup(xfdir_p->gl[i].en->path);
        xfdir_p->gl[i].en->module = MODULE_NAME;
	NOOP("%d = %s\n",xfdir_p->gl[i].en->st->st_uid, (const char *)xfdir_p->gl[i].en->path);
	//NOOP("pid=%d ppid=%d gid=%d\n",en->st->st_uid, en->st->st_gid, (gint)en->st->st_ino);
    }

    // Cleanup

    if (pkg_list) g_slist_free(pkg_list);

    return 1;
}
#define POPUP_ID "pkg_menu"
#define POPUP_MENU_ID "pkg_menu_menu"

/////////////////   FreeBSD pkg   /////////////////////////////


static void
scroll_to_top(void *user_data){
    rfm_threadwait();
    widgets_t *widgets_p = user_data;
    rfm_context_function(rfm_scroll_to_top, widgets_p);
}

static void
activate_entry (GtkEntry * entry, gpointer data) {
    GtkDialog *dialog = data;
    gtk_dialog_response (dialog,3);
}
static void hideshow(GtkButton *button, gpointer *data) {

    if (gtk_widget_get_visible(GTK_WIDGET(data))){
        gtk_widget_hide(GTK_WIDGET(data));
        gtk_widget_unrealize(GTK_WIDGET(data));

        GtkWidget *dialog = g_object_get_data(G_OBJECT(button), "dialog");
        GtkRequisition requisition;
#if GTK_MAJOR_VERSION<3
        gtk_widget_size_request(GTK_WIDGET(dialog), &requisition);
#else
        GtkRequisition natural;
        gtk_widget_get_preferred_size(GTK_WIDGET(dialog), &requisition, &natural);
#endif
        NOOP("w= %d h= %d\n", requisition.width, requisition.height);

        gtk_window_resize(GTK_WINDOW(dialog), requisition.width, requisition.height);
	gtk_window_set_resizable(GTK_WINDOW(dialog), TRUE);
    } else {
        gtk_widget_show_all(GTK_WIDGET(data));
    }
}

static void sensitivize_checks(GtkButton *check, gpointer data){
    if (!GTK_IS_TOGGLE_BUTTON(check)) return;
    GtkWidget *entry = g_object_get_data(G_OBJECT(check), "entry");
    gboolean state = FALSE;
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))){
        state = TRUE;
    } 
    if (entry) gtk_widget_set_sensitive(entry, state);
}

static const gchar *get_action_text(GtkButton *button);
static void content_add_options(const view_t *view_p, GtkBox *content_box, pkg_option_t *options, const gchar *tag);
    
static void
update_action(GtkButton *button, gpointer data){
    GtkWidget *dialog = g_object_get_data(G_OBJECT(button),  "dialog");
    g_object_set_data(G_OBJECT(dialog),  "action_button", button);
    NOOP("setting action button to %p\n", button);
    const gchar *cmd = get_action_text(button);
    GtkLabel *tit = data;
    gchar *markup = g_strdup_printf("<span color=\"red\">%s</span>", cmd);
    gtk_label_set_markup(tit, markup);
    g_free(markup);
    // clear all cmd_options_box
    DBG("clear all cmd_options_box\n");
    GtkWidget *cmd_options_box = g_object_get_data(G_OBJECT(dialog), "cmd_options_box");
    if (cmd_options_box){
        GList *children = gtk_container_get_children(GTK_CONTAINER(cmd_options_box));
        GList *list = children;
        for (; list && list->data; list = list->next){
            gtk_widget_destroy(GTK_WIDGET(list->data));
        }
        g_list_free(children);
    }
    
    // get c record
    DBG("get c record\n");
    pkg_command_t *c = g_object_get_data(G_OBJECT(button), "pkg_command_p");
    if (!c){
        DBG("no pkg_command_p associated to action button: %s\n", cmd);
        return;
    }
    // if c->cmd_options, add to cmd_options_box
    markup =g_strdup_printf("<b>%s</b> %s", c->cmd, _("options:"));
    widgets_t *widgets_p = rfm_get_widget("widgets_p");
    view_t *view_p = widgets_p->view_p;
    content_add_options(view_p, GTK_BOX(cmd_options_box), c->cmd_options, markup);
    g_free(markup);
    //
}

static gboolean
update_action_entry (GtkWidget *widget,
               GdkEvent  *event,
               gpointer   data){
    GtkButton *button = g_object_get_data(G_OBJECT(widget), "check");

    update_action(button, data);
    return FALSE;
}


static void
show_help(GtkButton *button, gpointer data){
    gchar *help = data;
    GtkWidget *dialog = gtk_dialog_new ();
    gtk_window_set_type_hint(GTK_WINDOW(dialog), GDK_WINDOW_TYPE_HINT_DIALOG);
    GtkWidget *label =gtk_label_new("");
    gtk_label_set_markup(GTK_LABEL(label), help);
    gtk_widget_show(label);
    GtkWidget *vbox = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
    GtkWidget *b = rfm_dialog_button ("xffm/emblem_grayball", _("Accept"));

    gtk_widget_show(b);
    gtk_dialog_add_action_widget (GTK_DIALOG(dialog),b,GTK_RESPONSE_YES);
    gtk_window_set_position (GTK_WINDOW(dialog),GTK_WIN_POS_MOUSE);

    gtk_widget_show(dialog);
    /*g_signal_connect_swapped (dialog, "response",
                          G_CALLBACK (gtk_widget_destroy),
                          dialog);*/
   
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
}


static gboolean
update_argument_entry (GtkWidget *widget,pkg_command_t *c){
    GtkDialog *dialog = g_object_get_data(G_OBJECT(widget), "dialog");
    GtkLabel *label = g_object_get_data(G_OBJECT(dialog), "top");
    const gchar *text = gtk_entry_get_text(GTK_ENTRY(widget));
    gchar *markup = g_strdup_printf("<big><b>%s<i> %s %s</i></b></big>",c->pkg, (c->cmd)?c->cmd:"", text);
    //gchar *markup = g_strdup_printf("<big><b>%s <i>%s</i></b></big>",cmd, text);
    gtk_label_set_markup(label, markup);
    g_free(markup);
    return FALSE;
}

static gboolean
update_argument_entry_sig (GtkWidget *widget,
               GdkEvent  *event,
               gpointer   data){
    return update_argument_entry(widget, (pkg_command_t *)data);
}


static gchar *content_get_options(GtkDialog *dialog, gchar *in_string, pkg_option_t *options);

static void
update_option(GtkButton *button, gpointer data){
    GtkDialog *dialog = g_object_get_data(G_OBJECT(button), "dialog");
    gchar *text=g_strdup("");
    pkg_option_t *options = g_object_get_data(G_OBJECT(button), "options");
    if (options != xml_options){
        g_object_set_data(G_OBJECT(dialog), "cmd_options", options);
    }

    text = content_get_options(dialog, text, options);

    GtkLabel *tit = data;
    gchar *markup = g_strdup_printf("<span color=\"red\">%s</span>", text);
    gtk_label_set_markup(tit, markup);
    g_free(markup);
}

static gboolean
update_option_entry (GtkWidget *widget,
               GdkEvent  *event,
               gpointer   data){
    GtkButton *button = g_object_get_data(G_OBJECT(widget), "check");

    update_option(button, data);
    return FALSE;
}

static const gchar *
get_action_text(GtkButton *button){
    NOOP("processing action button: %p\n", button);
    GList *children = gtk_container_get_children (GTK_CONTAINER(button));
    GList *l=children;
    const gchar *cmd=NULL;
    for (;l && l->data; l=l->next){
        GtkWidget *w = l->data;
        if (GTK_IS_LABEL(w)){
	    cmd = gtk_label_get_text (GTK_LABEL(w));
            break;
        }
    }
    g_list_free(children);
    GtkEntry *entry = g_object_get_data(G_OBJECT(button), "entry");
    DBG("%s: entry is %p\n", cmd, entry);
    if (entry){
        const gchar *text = gtk_entry_get_text(entry);
        DBG("%s: entry text is %s\n", cmd, text);
        if (text && strlen(text)){
            static gchar *g = NULL;
            g_free(g);
            g=g_strdup_printf("%s=%s", cmd, text);
            return (const gchar *)g;
        }
    }
    return cmd;
}

static const gchar *
content_get_action(GtkWidget *dialog){
    GtkButton *action_button = g_object_get_data(G_OBJECT(dialog), "action_button");
    NOOP("retrieving action button --> %p\n", action_button);
    return get_action_text(action_button);

}

static gchar *
content_get_options(GtkDialog *dialog, gchar *in_string, pkg_option_t *options){
    if (!options) return in_string;
    pkg_option_t *p = options;
    gchar *response_string = g_strdup(in_string);
    g_free(in_string);
    for (; p && p->loption; p++){
        gchar *label = g_strdup_printf ("--%s", p->loption);
        GtkWidget *check =g_object_get_data(G_OBJECT(dialog), label);
   
        NOOP("check %s: %p\n", label, check);
        if (!GTK_IS_TOGGLE_BUTTON(check)) continue;

        if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))){
            gchar *g = g_strconcat(response_string, " ", label, NULL);
            g_free(response_string);
            response_string = g;
            gint type = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(check), "type"));
            if (type & ARGUMENT_OPTION){
                gchar *elabel = g_strconcat(label, "-entry", NULL);
                GtkWidget *entry =g_object_get_data(G_OBJECT(dialog), elabel);
                if (entry) {
                    const gchar *etext = gtk_entry_get_text(GTK_ENTRY(entry));
                    if (etext && *etext && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check))){
                        gchar *g = g_strconcat(response_string, "=", etext, NULL);
                        g_free(response_string);
                        response_string = g;
                    }
                } 
                g_free(elabel);
            }
        }

        g_free(label);
    }
    return response_string;
}

static GtkWidget *
create_option_entry(GtkWidget *obox, GtkWidget *dialog, GtkWidget *check, 
        const gchar *parameter, const gchar *label){
    GtkWidget *entry = gtk_entry_new();
            
    g_object_set_data(G_OBJECT(entry), "check", check);
#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=2
    gtk_entry_set_placeholder_text (GTK_ENTRY(entry),  parameter);
#endif
    g_object_set_data(G_OBJECT(entry), "dialog", dialog);
    
    g_object_set_data(G_OBJECT(check), "entry", entry);
    gchar *elabel = g_strconcat(label, "-entry", NULL);
    g_object_set_data(G_OBJECT(dialog), elabel, entry);
    g_free(elabel);
    GtkWidget *lab = gtk_label_new("=");
    gtk_box_pack_start(GTK_BOX(obox), GTK_WIDGET(lab), FALSE, FALSE, 1);
    gtk_box_pack_end(GTK_BOX(obox), GTK_WIDGET(entry), TRUE, TRUE, 1);
    gtk_widget_show_all(obox);
    gtk_widget_set_sensitive(entry, FALSE);
    g_object_set_data(G_OBJECT(check), "type", GINT_TO_POINTER(ARGUMENT_OPTION)); 
    return entry;
}


static void 
content_add_actions(const view_t *view_p, GtkBox *content_box, const gchar *tag){
    if (!xml_cmds) return;
    gboolean first=TRUE;
    GtkWidget *dialog = g_object_get_data(G_OBJECT(content_box), "dialog");

    GtkWidget *obox = rfm_hbox_new(FALSE, 1);
    GtkWidget *button = gtk_toggle_button_new_with_label(tag);
    g_object_set_data(G_OBJECT(button), "dialog", dialog);
    gtk_box_pack_start(GTK_BOX(content_box), obox, FALSE, FALSE, 1);
    gtk_box_pack_start(GTK_BOX(obox), button, FALSE, FALSE, 1);
        
    GtkWidget *tit = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(obox), tit, FALSE, FALSE, 1);
    gtk_widget_show(tit);
    
    gtk_widget_show(obox);
    gtk_widget_show(button);

    GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_box_pack_start(GTK_BOX(content_box), scrolled_window, FALSE, FALSE, 1);
    
    obox = rfm_hbox_new(FALSE, 1);
#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=8
    gtk_container_add(GTK_CONTAINER(scrolled_window), obox);
#else
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW(scrolled_window),
                                       obox);
#endif
    gtk_widget_show(obox);

    GtkBox *vbox = GTK_BOX(rfm_vbox_new(FALSE, 1));
    gtk_box_pack_start(GTK_BOX(obox), GTK_WIDGET(vbox), FALSE, FALSE, 1);
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(hideshow), scrolled_window); 
    gtk_widget_realize(obox);


    obox = rfm_hbox_new(FALSE, 1);
    gtk_box_pack_start(GTK_BOX(vbox), obox, FALSE, FALSE, 1);


    gint width = -1;
    GtkWidget *check;
    pkg_command_t *pp;
    GSList *radio_group=NULL;
    for (pp=xml_cmds; pp && pp->pkg; pp++){
        const gchar *cmd = pp->cmd;
	if (!pp->cmd) cmd = "";
//        else if (!pp->string) continue; // dont show stuff with string not defined in XML
        obox = rfm_hbox_new(FALSE, 1);
        gtk_box_pack_start(vbox, obox, FALSE, FALSE, 1);
        gchar *label = g_strdup_printf ("%s", cmd);
        NOOP("adding \"%s\"\n", label);
        check = gtk_radio_button_new_with_label(radio_group,label);
        g_object_set_data(G_OBJECT(check), "pkg_command_p", pp);
        if (first){
            update_action(GTK_BUTTON(check), tit);
            first=FALSE;
        }
            
	radio_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(check));
        g_object_set_data(G_OBJECT(check), "dialog", dialog);
        g_object_set_data(G_OBJECT(dialog), label, check);
        gtk_box_pack_start(GTK_BOX(obox), check, FALSE, FALSE, 1);

        /*GtkWidget *h_button = rfm_dialog_button(NULL,"?");
        gtk_box_pack_end(GTK_BOX(obox), GTK_WIDGET(h_button), FALSE, FALSE, 1);
        g_signal_connect(G_OBJECT(h_button), "clicked", G_CALLBACK(show_help), pp->hlp);*/
	// radio types
        if (pp->parameter) {
            GtkWidget *entry = create_option_entry(obox, dialog, check, pp->parameter, label);
            g_signal_connect(G_OBJECT (entry), "key-release-event", G_CALLBACK (update_action_entry), tit);
            g_signal_connect(G_OBJECT (entry), "button-press-event", G_CALLBACK (update_action_entry), tit);
            
        } else {
            g_object_set_data(G_OBJECT(check), "type", GINT_TO_POINTER(PLAIN_OPTION));
        }
        g_signal_connect(G_OBJECT (check), "clicked", G_CALLBACK (update_action), tit);

        g_signal_connect (G_OBJECT (check), "clicked", G_CALLBACK (sensitivize_checks), NULL);
	g_free(label);
        GdkPixbuf *pixbuf = rfm_get_pixbuf("xffm/emblem_about", 24);
        rfm_add_custom_tooltip(check, pixbuf, pp->hlp);

    }
    gtk_widget_realize(GTK_WIDGET(vbox));

}

static void 
content_add_options(const view_t *view_p, GtkBox *content_box, pkg_option_t *options, const gchar *tag){
    if (!options) return;
    GtkWidget *dialog = g_object_get_data(G_OBJECT(content_box), "dialog");

    GtkWidget *obox = rfm_hbox_new(FALSE, 1);
    GtkWidget *button = gtk_toggle_button_new_with_label(tag);
    GtkWidget *label_widget = gtk_bin_get_child(GTK_BIN(button));
    gtk_label_set_markup(GTK_LABEL(label_widget), tag);
    g_object_set_data(G_OBJECT(button), "dialog", dialog);
    gtk_box_pack_start(GTK_BOX(content_box), obox, FALSE, FALSE, 1);
    gtk_box_pack_start(GTK_BOX(obox), button, FALSE, FALSE, 1);
        
    GtkWidget *tit = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(obox), tit, FALSE, FALSE, 1);
    gtk_widget_show(tit);
    
    gtk_widget_show(obox);
    gtk_widget_show(button);

    GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_box_pack_start(GTK_BOX(content_box), scrolled_window, FALSE, FALSE, 1);
    
    obox = rfm_hbox_new(FALSE, 1);
#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=8
    gtk_container_add(GTK_CONTAINER(scrolled_window), obox);
#else
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW(scrolled_window),
                                       obox);
#endif
    gtk_widget_show(obox);

    GtkBox *vbox = GTK_BOX(rfm_vbox_new(FALSE, 1));
    gtk_box_pack_start(GTK_BOX(obox), GTK_WIDGET(vbox), FALSE, FALSE, 1);
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(hideshow), scrolled_window); 
    gtk_widget_realize(obox);


    gint width = -1;
    GtkWidget *check;
    pkg_option_t *pp;
    for (pp=options; pp && pp->loption; pp++){
        obox = rfm_hbox_new(FALSE, 1);
        gtk_box_pack_start(vbox, obox, FALSE, FALSE, 1);
        gchar *label = g_strdup_printf ("--%s", pp->loption);
        check = gtk_check_button_new_with_label(label);
        g_object_set_data(G_OBJECT(dialog), label, check);
        NOOP("adding %s -> %p\n", label, check);
        GdkPixbuf *pixbuf = rfm_get_pixbuf("xffm/emblem_about", 24);
        rfm_add_custom_tooltip(check, pixbuf, pp->hlp);

        g_object_set_data(G_OBJECT(check), "dialog", dialog);
        g_object_set_data(G_OBJECT(check), "options", options);
        gtk_box_pack_start(GTK_BOX(obox), check, FALSE, FALSE, 1);
        GtkRequisition minimum;
        if (pp->parameter) {
            GtkWidget *entry = create_option_entry(obox, dialog, check, pp->parameter, label);
#if GTK_MAJOR_VERSION<3
        gtk_widget_size_request(GTK_WIDGET(dialog), &minimum);
#else
            gtk_widget_get_preferred_size(GTK_WIDGET(obox), &minimum, NULL);
#endif
            if (minimum.width > width) width = minimum.width;
            g_object_set_data(G_OBJECT(check), "type", GINT_TO_POINTER(ARGUMENT_OPTION));
            g_object_set_data(G_OBJECT(entry), "options", options);
            g_signal_connect(G_OBJECT (entry), "key-release-event", G_CALLBACK (update_option_entry), tit);
            g_signal_connect(G_OBJECT (entry), "button-press-event", G_CALLBACK (update_option_entry), tit);
            if (pp->active) {
                gtk_entry_set_text(GTK_ENTRY(entry), pp->active);
            }

        } else {
            g_object_set_data(G_OBJECT(check), "type", GINT_TO_POINTER(PLAIN_OPTION));
        }
        if (pp->active) {
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), TRUE);
        }
        g_signal_connect(G_OBJECT (check), "clicked", G_CALLBACK (update_option), tit);
        g_signal_connect (G_OBJECT (check), "clicked", G_CALLBACK (sensitivize_checks), NULL);
	g_free(label);
    }
    // update option text after last button is created, not before.
    update_option(GTK_BUTTON(check), tit);

    if (width > 0) width +=30; // hack
    gtk_widget_set_size_request(scrolled_window, width, 100);
    gtk_widget_realize(GTK_WIDGET(vbox));
}


static GtkWidget *
content_add_argument(const view_t *view_p, GtkBox *content_box, const pkg_command_t *cmd){
    GtkWidget *box = NULL;
    GtkWidget *argument = NULL;
    GtkWidget *dialog = g_object_get_data(G_OBJECT(content_box), "dialog");
    if (cmd->argument && strlen(cmd->argument)) {
        box = rfm_hbox_new(FALSE, 1);
        argument = gtk_entry_new();
        g_object_set_data(G_OBJECT(argument), "dialog", dialog); 
        GtkWidget *a = gtk_label_new("");
        gchar *at;
#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=2
	gtk_entry_set_placeholder_text (GTK_ENTRY(argument),  cmd->argument);
        at = g_strdup_printf("<i>%s</i>:", _("argument")) ;
#else
        at = g_strdup_printf("<i>%s</i>:", cmd->argument) ;
#endif
        gtk_label_set_markup(GTK_LABEL(a), at);
        g_free(at);
        gtk_widget_show(a);
        gtk_box_pack_start(GTK_BOX(box), a, FALSE, FALSE, 1);

        g_signal_connect (G_OBJECT (argument), "activate", G_CALLBACK (activate_entry), dialog);
        if (view_p->selection_list && g_slist_length(view_p->selection_list)==1){
            // XXX little hack here to avoid search as argument.
	    record_entry_t *en = view_p->selection_list->data;
            if (strcmp(en->path, _("Search"))){
                gchar *name = NULL;
                DBG("flags=0x%x (0x%x)\n",cmd->flags, NO_VERSION);
                if (cmd->flags & NO_VERSION) {
                    name = package_name(en->path);
                    DBG("no version for %s -> %s\n", en->path, name);
                }
                gtk_entry_set_text(GTK_ENTRY(argument), (name)?name:en->path);
                g_free(name);
                gtk_widget_set_sensitive(argument, FALSE);
            }
        }
        
        gtk_widget_show(argument);
        gtk_box_pack_start(GTK_BOX(box), argument, FALSE, FALSE, 1);
        gtk_widget_show(box);
        gtk_box_pack_start(content_box, box, FALSE, FALSE, 1);
        g_signal_connect(G_OBJECT(argument), "key-release-event", 
                G_CALLBACK(update_argument_entry_sig), (gpointer)cmd);
        g_signal_connect(G_OBJECT(argument), "button-press-event", 
                G_CALLBACK(update_argument_entry_sig), (gpointer)cmd);
        update_argument_entry(argument, (gpointer)cmd);

    }
    return argument;
}
   
static void
dump_stderr (void *user_data, void *stream, int childFD){
    return;
}

static void *
pkg_confirm_f(void *data){
    pkg_command_t *c = data;

    GtkDialog *dialog = GTK_DIALOG(gtk_dialog_new());
    gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);
    gtk_window_set_resizable(GTK_WINDOW(dialog), TRUE);
    rfm_global_t *rfm_global_p = rfm_global();

    gtk_window_set_transient_for(GTK_WINDOW(dialog), GTK_WINDOW(rfm_global_p->window));
    //gtk_window_set_keep_above (GTK_WINDOW(dialog), TRUE);
    gtk_window_set_title(GTK_WINDOW(dialog), _("Package Manager"));
    gtk_window_set_type_hint(GTK_WINDOW(dialog), GDK_WINDOW_TYPE_HINT_DIALOG);
    
    GtkWidget *button = rfm_dialog_button ("xffm/emblem_help", _("Help"));
    gtk_dialog_add_action_widget (dialog, button, 1);
    button = rfm_dialog_button ("xffm/stock_window-close", _("Cancel"));
    gtk_dialog_add_action_widget (dialog, button, 2);
    button = rfm_dialog_button ("xffm/stock_ok", _("Accept"));
    gtk_dialog_add_action_widget (dialog, button, 3);

    GtkBox *content_box = GTK_BOX(gtk_dialog_get_content_area(dialog));
    g_object_set_data(G_OBJECT(content_box),"dialog", dialog);
    GtkLabel *top = GTK_LABEL(gtk_label_new(""));
    g_object_set_data(G_OBJECT(dialog), "top", top);

    gchar *t = NULL;
    t = g_strdup_printf("<big><b>%s <i>%s</i></b></big>",c->pkg, (c->cmd)?c->cmd:"");
    gtk_label_set_markup(top, t);
    g_free(t);
    gtk_widget_show(GTK_WIDGET(top));
    gtk_box_pack_start(content_box, GTK_WIDGET(top), FALSE, FALSE, 1);
               
    if (use_custom_envar){
        GtkBox *env_box = GTK_BOX(rfm_hbox_new(FALSE, 1));
        gtk_box_pack_start(content_box, GTK_WIDGET(env_box), FALSE, FALSE, 1);
// FIXME: find available translation for "Environment options:"
        GtkWidget *env=gtk_label_new(_("Environment options:"));
        gtk_box_pack_start(env_box, GTK_WIDGET(env), FALSE, FALSE, 1);
        GtkWidget *env_entry = gtk_entry_new();
        if (envvar && strlen(envvar)) gtk_entry_set_text(GTK_ENTRY(env_entry), envvar);
        else {
#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=2
            gtk_entry_set_placeholder_text (GTK_ENTRY(env_entry), 
                    (use_custom_envar_string)?(use_custom_envar_string):"ENVAR=\"\"");
#endif
        }
        gtk_box_pack_start(env_box, GTK_WIDGET(env_entry), FALSE, FALSE, 1);
        g_object_set_data(G_OBJECT(dialog), "env_entry", env_entry);
        gtk_widget_show_all(GTK_WIDGET(env_box));
    }
        

    widgets_t *widgets_p = rfm_get_widget("widgets_p");
    view_t *view_p = widgets_p->view_p;

    gchar *markup =g_strdup_printf("<b>%s</b> %s", pkg_command, _("options:"));
    content_add_options(view_p, content_box, c->pkg_options, markup);
    g_free(markup);
    if (c->cmd==NULL){
	content_add_actions(view_p, content_box, _("action:"));
    }
    if (c->cmd_options){
        markup =g_strdup_printf("<b>%s</b> %s", c->cmd, _("options:"));
        content_add_options(view_p, content_box, c->cmd_options, markup);
        g_free(markup);
    } //else 
    {
        GtkWidget *cmd_options_box = rfm_vbox_new(FALSE, 1);
        g_object_set_data(G_OBJECT(cmd_options_box),"dialog", dialog);
        gtk_widget_show(cmd_options_box);
        gtk_box_pack_start(GTK_BOX(content_box), GTK_WIDGET(cmd_options_box), FALSE, FALSE, 1);
        g_object_set_data(G_OBJECT(dialog), "cmd_options_box", cmd_options_box);
    }
        
    GtkWidget *argument = content_add_argument(view_p, content_box, c);

    gtk_widget_grab_focus(button); 
    gtk_widget_show(GTK_WIDGET(dialog));
    gint response;
loop:
    response = gtk_dialog_run(dialog);
    if (use_custom_envar) {
        GtkEntry *entry = g_object_get_data(G_OBJECT(dialog), "env_entry");
        const gchar *g = gtk_entry_get_text(entry);
        g_free(envvar);
        if (g && strlen(g)) envvar = g_strdup(g);
        else envvar = NULL;
    }

    if (response == 1){ // Help button
        // construct specific help command 
        widgets_t *widgets_p = rfm_get_widget("widgets_p");
        // default, freebsd pkg
        gchar *arg[]={pkg_command, "help", c->cmd, NULL};
        if (emerge){ // This will probably change if emerge implements action specific help
            arg[0] = "man";
            arg[1] = "emerge";
            arg[2] = NULL;
        } else if (apt){
            arg[0] = "man";
            arg[1] = "apt-get";
            arg[2] = NULL;
        } 
        // FIXME: zypper and yum case, and apt
        //        need to consider translations, or execute in locale=C

        rfm_clear_text (widgets_p); 
        rfm_show_text (widgets_p); 
        if (emerge) {
            // dump stderr on man output...
            rfm_thread_run_argv_full (widgets_p, arg, FALSE, NULL, rfm_operate_stdout, dump_stderr, scroll_to_top);
        } else {
            rfm_thread_run_argv_full (widgets_p, arg, FALSE, NULL, rfm_operate_stdout, NULL, scroll_to_top);
        }
    }

    gtk_widget_hide(GTK_WIDGET(dialog));


    // get response string
    gchar *response_string = NULL;
    if (response == 3){
        widgets_t *widgets_p = rfm_get_widget("widgets_p");
        response_string = content_get_options(dialog, g_strdup(""), xml_options);
        gchar *pcr=g_object_get_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response");
        g_free(pcr);
        pcr = NULL;
        g_object_set_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response", NULL);
        // In zypper/yum, assume yes is a global option
        if (zypper) {
            pcr = g_strdup_printf("%s --non-interactive", response_string);
        }
        else if (yum){
            pcr = g_strdup_printf("%s --assumeyes", response_string);
        } else if (apt) {
            pcr = g_strdup_printf("%s --assume-yes", response_string);
        }

        const gchar *action = NULL;
        if (c->cmd == NULL) action = content_get_action(GTK_WIDGET(dialog));
        else action = c->cmd;
        if (action) {
            gchar *g = g_strconcat(response_string, " ", action, NULL);
            g_free(response_string);
            response_string = g; 
            if (pcr) {
                g = g_strconcat(pcr, " ", action, NULL);
                g_free(pcr);
                pcr = g;   
            }   
        }

        // In pkg, assume yes is an action option
        if (pkg) {
            pcr = g_strdup_printf("%s -y", response_string);
        }

        // If action has any other options, then get them too.
        pkg_option_t *cmd_options = g_object_get_data(G_OBJECT(dialog), "cmd_options");
        
        if (cmd_options || c->cmd_options) {
            response_string = content_get_options(dialog, response_string, 
                    (c->cmd_options)?c->cmd_options:cmd_options);
        }
    
        

        // FIXME: get and append action options
        
        if (argument) {
            const gchar *a = gtk_entry_get_text(GTK_ENTRY(argument));
            gchar *g = g_strconcat(response_string, " ", a, NULL);
            g_free(response_string);
            response_string = g;
            if (pcr) {
                g = g_strconcat(pcr, " ", a, NULL);
                g_free(pcr);
                pcr = g;   
            }   
        }
        g_object_set_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response", pcr);
    }

    gtk_widget_destroy(GTK_WIDGET(dialog));
    return (void *) response_string;

}

static void
operate_stdout (void *user_data, void *stream, int childFD) {
    if (!stream) return;
    widgets_t *widgets_p = user_data;
    const gchar *exit_token = "Tubo-id exit:";
    gchar *recur = NULL;
    gint confirm = 0;
    if (strstr(stream, exit_token) || strstr(stream, "Exiting on user Command")){
        NOOP("got exit token at:%s",(gchar *) stream); 
        if (strstr(stream, "[y/N]") || strstr(stream, "[Y/n]")) {
            confirm = 1;
            if (strstr(stream, "[y/N]")) *(strstr(stream, "[y/N]")+strlen("[y/N]")) = 0;
            else if (strstr(stream, "[Y/n]")) *(strstr(stream, "[Y/n]")+strlen("[Y/n]")) = 0;
        }
        else confirm = -1;
        // wtf: rpm/zypper/yum...
        // emerge missing...
        // if (strstr(stream, "Exiting on user Command")) *strstr(stream, "Exiting on user Command")=0;
    }
    rfm_operate_stdout(user_data, stream, childFD);
    if (confirm > 0){
        gchar *pcr=g_object_get_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response");
        g_object_set_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response", NULL);
        gchar *text = g_strdup_printf("<i>%s</i>\n\n<b>%s</b>", pcr, (gchar *)stream);
        gint flags = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widgets_p->paper), "flags"));
        if (rfm_confirm (widgets_p,  GTK_MESSAGE_QUESTION, text,
                    _("No"), _("Yes"))){
            // do it with -y (pkg) or --non-interactive (zypper)
            do_it(widgets_p, pcr, flags);
        }
	g_free(pcr);
        g_free(text);
    }
    if (confirm < 0){ 
        view_t *view_p = widgets_p->view_p;
        record_entry_t *en = rfm_copy_entry(view_p->en);
        rodent_refresh (widgets_p, en);
    }
}
static void
operate_stderr (void *user_data, void *stream, int childFD) {
    if (!stream) return;
    widgets_t *widgets_p = user_data;
    const gchar *exit_token = "\'--non-interactive\'";
    if (strstr(stream, exit_token) ){
	// From stderr condition:
	// (not for stdout, since this function may be called before
	// stdout is done processing stream)
        gchar *pcr=g_object_get_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response");
	g_object_set_data(G_OBJECT(widgets_p->paper), "pkg_confirm_response", NULL);
        gint flags = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widgets_p->paper), "flags"));
        gchar *text = g_strdup_printf("<i>%s</i>\n<span fgcolor=\"red\" bgcolor=\"white\">%s</span>\n", pcr,  _("Confirm Execution"));
        if (rfm_confirm (widgets_p,  GTK_MESSAGE_QUESTION, text,
                    _("No"), _("Yes"))){
            // do it with -y (pkg) or --non-interactive (zypper)
            do_it(widgets_p, pcr, flags);
        }
	g_free(pcr);
        g_free(text);
    }
    rfm_operate_stderr(user_data, stream, childFD);

}

static void 
do_it(widgets_t *widgets_p, const gchar *cmd, gint flags){  
    if (!cmd) return;
    gchar *shell = rfm_shell();
    gchar *argv[]={ shell, "-c", (gchar  *)cmd, NULL };
    if (!shell){ DBG("No valid shell found\n"); }
    if (flags & SCROLL_UP) {
        rfm_clear_text (widgets_p); 
    }
    rfm_show_text(widgets_p);
    if (strstr(cmd, "help") || (flags & SCROLL_UP)) {
        rfm_thread_run_argv_full (widgets_p, argv, FALSE, NULL, operate_stdout, NULL, scroll_to_top);
    } else {
        rfm_thread_run_argv_full (widgets_p, argv, (flags&IN_TERM)?TRUE:FALSE, NULL, operate_stdout, operate_stderr, NULL);
    }
    g_free(shell);
}


static void *
thread_popup(void *data){
    if (rfm_get_gtk_thread() == g_thread_self()){
	g_warning("thread_mk_popup_menu(): only to be called from non gtk thread.\n");
        return NULL;
    }
    TRACE("creating pkg popup menu...\n");
    GtkWidget *popup_widget=rodent_thread_add_submenu (NULL,  "rodent-pkg", POPUP_ID,  NULL);  

        rodent_thread_multimenu_make(NULL, xml_menu_definitions); 

    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;
    xfdir_register_popup(view_p, popup_widget);

    pkg_command_t *c = xml_cmds;

    for (;c && c->pkg; c++){
        gchar *g = g_strdup_printf("pkg_%s", (c->cmd)?c->cmd:"");
        GtkWidget *item = rfm_get_widget(g);
        if (!item){
            NOOP("item %s not available\n", g);
            g_free(g);
            continue;
        }
        if (item && ((c->flags)&INACTIVE)) {
            gtk_widget_set_sensitive(item, FALSE);
        }
        GdkPixbuf *pixbuf = rfm_get_pixbuf("xffm/emblem_about", 24);
        rfm_add_custom_tooltip(item, pixbuf, c->hlp);        
        g_free(g);
    }

    // XXX WTF here?:
    //rfm_set_widget(GINT_TO_POINTER(DUMMY_FLAG), "pkg_module_flags");
    gtk_widget_realize(popup_widget);
    TRACE("Pkg popup menu widget created.\n");
    return popup_widget;
}


static void *
pkg_popup(void *p, void *q){
    GtkWidget *popup_widget=rfm_get_widget(POPUP_MENU_ID);
     DBG( "popup_widget: pkg_popup\n");

     if (!popup_widget) {
     DBG( "!popup_widget: pkg\n");
	return GINT_TO_POINTER(1);
    }

    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;
    record_entry_t *en = q;

    // If entry not specified, use the entry specified by
    // a single selection list.
    if (!en) {
       //	&&  g_slist_length(view_p->selection_list)==0) return NULL;
	if (g_slist_length(view_p->selection_list)==1){
	    en = view_p->selection_list->data;
	    if (g_path_is_absolute(en->path)) {
                DBG( "pkg_popup: not absolute path (%s)\n", en->path);
                return NULL;
            }
	    //if (IS_DUMMY_TYPE(en->type)) return NULL;
	}
    }
 
    while (!rfm_rw_lock_reader_trylock (&(view_p->mutexes.monitor_lock))){
	gtk_main_iteration();
    }



        
    pkg_command_t *c = xml_cmds;

    for (;c && c->pkg; c++){
        if (!c->cmd) continue;
        gchar *g = g_strdup_printf("pkg_%s", c->cmd);
        NOOP("hide/show item:  %s\n", g);
        GtkWidget *item = rfm_get_widget(g);
        if (item == NULL) {
            DBG("hide/show item: unable to find menu item %s (does action tag have the \"string\" attribute?)\n", g);
            g_free(g);
            continue;
        } 
        g_free(g);
        gtk_widget_hide(item);
        if (!c->string) continue; // Don't show menu items where string is not defined.

        if (!en){
            if (c->flags & NO_SELECTION) gtk_widget_show(item);
            continue;
        }
        if (IS_DUMMY_TYPE(en->type)) {
            if (strcmp(c->cmd, "search")==0) gtk_widget_show(item);
            continue;
        }


        /*if (emerge) {
            if (c->flags & LOCAL_SELECTION) gtk_widget_show(item);
        } else */
        {    
                if (IS_LOCAL_TYPE(en->type)){
                    if (c->flags & (LOCAL_SELECTION)) gtk_widget_show(item);
                } else {
                    if (c->flags & (REMOTE_SELECTION)) {
                        gtk_widget_show(item);
                    }
                }
        }
        //if (c->flags & INACTIVE) gtk_widget_show(item);
    }

    
    GtkLabel *label_widget = g_object_get_data(G_OBJECT(popup_widget), "label");
    if (label_widget){
        gchar *menu_text = NULL; 
        if (en && !IS_DUMMY_TYPE(en->type)) {
            menu_text = g_strdup_printf("<b>%s</b>",en->tag?en->tag:en->path);
            //SHOW_IT("Segmentation fault");
        } else {
            // Set label to plugin name
            menu_text = g_strdup_printf("<b>%s</b>",_(MODULE_LABEL));
        }
        gtk_label_set_markup(label_widget, menu_text);	
        g_free(menu_text);
    }

    if (popup_widget) {
	TRACE( "popping 0x%x\n", GPOINTER_TO_INT(popup_widget));
	gtk_menu_popup(GTK_MENU(popup_widget), NULL, NULL, NULL, NULL, 3, gtk_get_current_event_time());
    }

    rfm_rw_lock_reader_unlock (&(view_p->mutexes.monitor_lock));
    return GINT_TO_POINTER(1);
}


static gchar *
get_tooltip_text(record_entry_t *en){

    if (!en) return NULL;

    if (en->path && pkg_cachedir && strcmp(en->path, pkg_cachedir)==0) {
        return g_strdup(_("Disk cache"));
    }
    if (en->path && portsdir && strcmp(en->path, portsdir)==0) {
        return g_strdup(_("FreeBSD (Ports)"));
    }
    if (en->path && strcmp(en->path, _("Search"))==0){
        return g_strdup(_("Search remote repositories for distribution packages."));
    } 
    if (en->path && strcmp(en->path, _(MODULE_LABEL))==0){
        return g_strdup(_("A package manager"));
    } 
    gchar *retval=NULL;
    if (installed_hash) retval = g_hash_table_lookup(installed_hash, en->path);
    if (retval) {
        retval = g_markup_escape_text(retval, -1);
    }
    else retval = g_strdup("whatever (FIXME)");
        

#if 10
    if (!rpm && !pkg) return retval;
    
    gchar *format;
    gchar *cmd=NULL;
    const gchar *query = "query";
    gchar line[4098];
    FILE *pp=NULL;
    if (pkg){
        gchar *date;
        memset(line,0,4098);
        if (!IS_LOCAL_TYPE(en->type)){
            query = "rquery";
            date = g_strdup(_("not installed"));
        } else {
            // Time stamp:
            cmd = g_strdup_printf("pkg %s \"%%t\" %s", query, en->path);
            pp = popen(cmd, "r");
            g_free(cmd);
            if (pp) {
                time_t t = 0;
                if (fgets(line, 256, pp)){
                    t = strtol(line, NULL, 10);
                }
                pclose(pp);
                ctime_r(&t, line);
                // Chop
                if (strchr(line, '\n')) *strchr(line, '\n') = 0;
            } else line[0] = '?';
            date = g_strdup(line);
        }
        // Categories:
        gchar *categories = g_strdup("");
        cmd = g_strdup_printf("pkg %s \"%%C\" %s", query, en->path);
        pp = popen(cmd, "r");
        g_free(cmd);
        if (pp) {
            while (!feof(pp) && fgets(line, 4097, pp)){
                // Chop
                if (strchr(line, '\n')) *strchr(line, '\n') = 0;
                gchar *g = g_strconcat(categories, line, ",", NULL);
                g_free(categories);
                categories = g;
            }
            // chop
            if (strrchr(categories, ',')) *strrchr(categories, ',') = 0;
            pclose(pp);
        }
                // Chop
                if (strchr(line, '\n')) *strchr(line, '\n') = 0;

        format = g_strdup_printf("\n%s: %%v\n%s: %%w\n%s: %s\n%s: %%sh\n%s: %s\n%s: %%m\n",
                _("Version"), 
                _("Homepage"),
                _("Categories"), categories,
                _("Size"),
                _("Installed"), date,
                _("Maintainer"));
        g_free(date);
        g_free(categories);
    }
    
    if (rpm) cmd = g_strdup_printf("rpm -qi %s", en->path);
    else if (pkg) cmd = g_strdup_printf("pkg %s \"%s\" %s", query, format, en->path);

    NOOP("gettooltip cmd: %s\n-----\n", cmd);
    pp = popen(cmd, "r");
    if (pp) {
        while (!feof(pp) && fgets(line, 4097, pp)){
            gchar *g = g_strconcat(retval, line, NULL);
            g_free(retval);
            retval = g;
        }
        pclose(pp);
    } else DBG("Cannot pipe %s\n", cmd);
    g_free(cmd);
    // Chop
    //if (strchr(line, '\n')) *strrchr(line, '\n') = 0;
   
    gchar *p=retval;
    retval = g_markup_escape_text(p, -1);
    g_free(p);
#endif
    return retval;

}

static gboolean
check_write_OK(const gchar *file){
    if (access(file, W_OK) == 0) return TRUE;
    return FALSE;
}


static gboolean
local_write_OK(void){
    gchar *file = g_strdup_printf("%s/%s", pkg_dbdir, "local.sqlite");
    gboolean result = check_write_OK(file);
    g_free(file);
    return result;
}

static gboolean
remote_write_OK(void){
    gchar *file = g_strdup_printf("%s/%s", pkg_dbdir, "repo-FreeBSD.sqlite");
    gboolean result = check_write_OK(file);
    g_free(file);
    return result;
}

static gboolean hyh=FALSE;
static void
on_destroy(GtkWidget *w, void *data){
    hyh=FALSE;
    return;
}

static void *hold_your_horses_f(void *data){
    rfm_global_t *rfm_global_p = rfm_global();
    GtkWidget *parent = rfm_global_p?rfm_global_p->window:NULL;
    GtkWidget *hold_your_horses = gtk_message_dialog_new_with_markup(
            GTK_WINDOW(parent),
            GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_INFO,
            GTK_BUTTONS_NONE,
            NULL);
    gchar *markup = g_strdup_printf("%s <b>%s</b>\n\n<span foreground=\"red\">%s</span>\n",
            _("Processing"),
            pkg_command,
            _("Please wait..."));

    gtk_message_dialog_set_markup(GTK_MESSAGE_DIALOG(hold_your_horses),markup);
    g_free(markup);
    if (parent) gtk_window_set_transient_for(GTK_WINDOW(hold_your_horses), GTK_WINDOW(parent));
#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<22
    GtkWidget *message_area = 
        gtk_dialog_get_content_area(GTK_DIALOG(hold_your_horses));
#else
    GtkWidget *message_area = 
        gtk_message_dialog_get_message_area(GTK_MESSAGE_DIALOG(hold_your_horses));
#endif
    GtkWidget *spinner = gtk_spinner_new();
    gtk_box_pack_end(GTK_BOX(message_area), spinner, 1, FALSE, FALSE);
    gtk_widget_show(spinner);
    hyh=TRUE;
    gtk_spinner_start(GTK_SPINNER(spinner));
    g_signal_connect(G_OBJECT(hold_your_horses), "destroy", G_CALLBACK(on_destroy), NULL);
    gtk_widget_show(hold_your_horses);
    while (gtk_events_pending()) gtk_main_iteration();
    return hold_your_horses;
}
            
static void *loose_your_horses_f(void *data){
    if (!hyh) return NULL;
    GtkWidget *hold_your_horses = data;
    gtk_widget_destroy(hold_your_horses);
    hyh = FALSE;
    return NULL;
}

