/*
 * Edscott Wilson Garcia Copyright 2012
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */



#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include "rodent.h"
#include "rfm_modules.h"
/* this should be first 2 lines after headers: */
G_MODULE_EXPORT LIBRFM_MODULE

#define MODULE_NAME "smb"
#define SUBMODULE_NAME "workgroup"
//#define MODULE_LABEL _("Network domains")
#define MODULE_LABEL _("SMB Browser")
//#define MODULE_LABEL _("Nearby Computers")
//#define MODULE_ICON_ID "xffm/emblem_network/compositeSE/emblem_network"

#define MODULE_ENTRY_TIP _("SMB Browser")
#define PARENT_MODULE_NAME "cifs"
#define MODULE_DATA_ID "nmb_data_p"

#include "module-skeleton.h"
// Skeleton definitions

G_MODULE_EXPORT RFM_BLOCKING_LOAD(TRUE)
G_MODULE_EXPORT RFM_MODULE_NAME
G_MODULE_EXPORT RFM_SUBMODULE_NAME
G_MODULE_EXPORT RFM_MODULE_LABEL
//G_MODULE_EXPORT RFM_MODULE_ICON_ID
G_MODULE_EXPORT RFM_MODULE_ENTRY_TIP

G_MODULE_EXPORT RFM_MODULE_PREFERENCES_KEY("RODENT-SAMBA")
G_MODULE_EXPORT RFM_IS_ROOT_MODULE(TRUE)
G_MODULE_EXPORT RFM_PLUGIN_INFO(_("SMB Browser"))
G_MODULE_EXPORT RFM_MODULE_ACTIVE(TRUE)
G_MODULE_EXPORT RFM_MODULE_MONITOR(FALSE)
G_MODULE_EXPORT RFM_IS_SELECTABLE(TRUE)

#define SAMBA_MODULE_C

#include "passfile.i"
#include "samba-common.i"

G_MODULE_EXPORT
void *
module_icon_id(void){
    static gchar *icon=NULL;
    if (icon == NULL){
	icon = g_strdup_printf("%s/pixmaps/rodent-smb.svg", PACKAGE_DATA_DIR);
    }
    return icon;
}

// GSList-mutex combo for master browsers (IP records)
// These are shared across view, so that's why they are static global
static 
GSList *mastersIP = NULL;
static
GMutex *mastersIP_mutex = NULL;

static
GHashTable *master_hash = NULL;
static
GMutex *master_mutex = NULL;

static
GHashTable *workgroup_hash = NULL;
static
GMutex *workgroup_mutex = NULL;
#include "query_masters.i"
#include "resolve_masters.i"

void *    
g_module_check_init (GModule * module) {
    BIND_TEXT_DOMAIN;
    if (!mastersIP_mutex) rfm_mutex_init(mastersIP_mutex);
    if (!master_mutex) rfm_mutex_init(master_mutex);
    if (!workgroup_mutex) rfm_mutex_init(workgroup_mutex);
    common_init(module);
    return NULL;
}

// const gchar * 
// This function returns a const pointer to the entry's icon
// identifier. 
// Parameter p is the item's entry pointer. 
// Identifier may be returned as a mimetype or an absolute path.

// This routine solve for two icons, namely,
//   1. the plugin icon shown on rodent root (or fuse-module)
//   2. the up icon shown in modules contained by this module.
G_MODULE_EXPORT
void *
item_icon_id (void *p) {
    record_entry_t *en = p;
    if (!en) return NULL;
    if (IS_UP_TYPE(en->type)) return "xffm/emblem_network/compositeSE/stock_go-up";
    return module_icon_id();
}



static 
void cache_xfdir_setup(xfdir_t *xfdir_p, smb_data_t *smb_data_p) {

    // mastersIP list is shared across views.
    g_mutex_lock(mastersIP_mutex);
    gint load_count = (mastersIP)?g_slist_length(mastersIP):0;
    NOOP("mastersip list length count load = %d\n", load_count);
    xfdir_p->pathc = load_count + 1;	
    xfdir_p->gl = (dir_t *)malloc(xfdir_p->pathc*sizeof(dir_t));
    if (!xfdir_p->gl) g_error("malloc: %s", strerror(errno));
    memset(xfdir_p->gl,0,xfdir_p->pathc*sizeof(dir_t));
    set_up_item(xfdir_p);
    if (load_count==0) {
	g_mutex_unlock(mastersIP_mutex);
	return;
    }

    GSList *tmp = mastersIP;
    gint i = 1;
    for (; tmp && tmp->data; tmp=tmp->next, i++){
	const gchar *ip = tmp->data;
	xfdir_p->gl[i].en=rfm_mk_entry(0);
	xfdir_p->gl[i].en->st = NULL;
	xfdir_p->gl[i].en->parent_module = MODULE_NAME;
	xfdir_p->gl[i].en->module = "workgroup";
	xfdir_p->gl[i].en->path = g_strdup(ip);  
    }
    g_mutex_unlock(mastersIP_mutex);

    for (i = 1; i < xfdir_p->pathc; i++){
	gchar *ip = xfdir_p->gl[i].en->path;
	const gchar *c = NULL;
	gchar *s = NULL;
	g_mutex_lock(master_mutex);
	if (master_hash) {
	    c = g_hash_table_lookup(master_hash, ip);
	    if (c) s = g_strdup(c);
	}
	g_mutex_unlock(master_mutex);

	gchar *wg = NULL;
	g_mutex_lock(workgroup_mutex);
	if (workgroup_hash) {
	    c = g_hash_table_lookup(workgroup_hash, ip);
	    if (c) wg = g_strdup(c);
	}
	g_mutex_unlock(workgroup_mutex);

	// Replace ip for netbios name, if netbios name is resolved:
	if (s) xfdir_p->gl[i].en->path = s;  


	xfdir_p->gl[i].en->tag = g_strdup_printf("%s %s (%s)",
		_("Workgroup:"),
		(wg)?wg:_("Unresolved alias"), ip);
	xfdir_p->gl[i].en->pseudo_path = g_strdup((wg)?wg:ip);

	if (s && wg) {
	    xfdir_p->gl[i].pathv = g_strdup_printf("%s (@%s)",
		wg, s);
	} else {
	    xfdir_p->gl[i].pathv = g_strdup_printf("%s %s",
		_("Workgroup:"),(wg)?wg:ip);
	}
	g_free(wg);
	// If netbios name is resolved, we no longer have any use for ip
	if (s) g_free(ip);
	NOOP(stderr, "+*+*+*+*+*+*+* %s\n", xfdir_p->gl[i].en->tag);
    } 
     
	
    set_up_item(xfdir_p);
    return;
}

// gboolean
// This function fills in previously allocated xfdir_p
// with glob records and entries of the module population.
// Records which are filled in are:
// xfdir_p->pathc: Number of icons for Rodent to display
// xfdir_p->gl[0 ... pathc-1].pathv: Labels to display with each icon
// xfdir_p->gl[0 ... pathc-1].en: Record_entry_t of each icon 
// 				  (NULL entries will point to Rodent root) 
//
// SMB Browser: This is a thread based background reload. This is done this 
// 	        way because the SMB queries are wall clock expensive. Thus,
// 	        to keep the user's eye busy, the actual load is done by the
// 	        background monitor.
//
G_MODULE_EXPORT
void *
module_xfdir_get (void *p) {
#ifdef DEBUG_TRACE
    static gint cc=1; TRACE("samba-module: module_xfdir_get %d\n", cc++);
#endif
    view_t *view_p = ((xfdir_t *)p)->view_p;
    if (!view_p) return NULL;
    g_mutex_lock(view_p->mutexes.status_mutex);
    gint status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if (status == STATUS_EXIT)return NULL;
    // Check whether necessary program is installed or not.
    if (!smb_check_program(&(view_p->widgets), "nmblookup")) {
	return NULL;
    }

    
    smb_data_t *smb_data_p = (smb_data_t *)malloc(sizeof(smb_data_t));
    if (!smb_data_p) g_error("malloc: %s", strerror(errno));
    memset(smb_data_p, 0, sizeof(smb_data_t));	
    rfm_mutex_init(smb_data_p->mutex);
    rfm_cond_init(smb_data_p->signal);
     
    smb_data_p->xfdir_p = p;
    smb_data_p->view_p = smb_data_p->xfdir_p->view_p;
    g_object_set_data(G_OBJECT(view_p->widgets.paper), MODULE_DATA_ID, smb_data_p);
    
   TRACE(  "THREAD 0x%x condition=0x%x\n",
		GPOINTER_TO_INT(g_thread_self()),
		smb_data_p->condition);


    query_masters(smb_data_p);

    g_mutex_lock(smb_data_p->mutex);
    if (!smb_data_p->condition){
	NOOP(stderr, "waiting for signal...\n");
	g_cond_wait(smb_data_p->signal, smb_data_p->mutex);
	NOOP(stderr, "got the  signal...\n");
    }
    g_mutex_unlock(smb_data_p->mutex);

    if (mastersIP == NULL){
	rfm_mutex_free(smb_data_p->mutex);
	rfm_cond_free(smb_data_p->signal);
	g_free(smb_data_p);
	g_object_set_data(G_OBJECT(view_p->widgets.paper), MODULE_DATA_ID, NULL);
	return NULL;
    }
    cache_xfdir_setup(p, smb_data_p);


    smb_data_p->condition=FALSE;
    NMBmastersResolve (smb_data_p);
    g_mutex_lock(smb_data_p->mutex);
    if (!smb_data_p->condition){
	g_cond_wait(smb_data_p->signal, smb_data_p->mutex);
    }
    g_mutex_unlock(smb_data_p->mutex);

    cache_xfdir_setup(p, smb_data_p);
    rfm_mutex_free(smb_data_p->mutex);
    rfm_cond_free(smb_data_p->signal);

    g_free(smb_data_p);
    g_object_set_data(G_OBJECT(view_p->widgets.paper), MODULE_DATA_ID, NULL);
    return p;


}


// gboolean
// This function informs Rodent by returning TRUE that the double click
// action will be processed by the module. If function returns FALSE
// (or is not defined in the module) then Rodent will attempt the default
// double click action.
// Parameter p is the view's widgets_p pointer.
// Parameter q is the icon's record entry.
G_MODULE_EXPORT
void *
double_click(void * p, void *q){
    passfile_double_click(p, q);
    return NULL;
}

