/*
// Dao Virtual Machine
// http://www.daovm.net
//
// Copyright (c) 2006-2014, Limin Fu
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
//
// * Redistributions of source code must retain the above copyright notice,
//   this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED  BY THE COPYRIGHT HOLDERS AND  CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED  WARRANTIES,  INCLUDING,  BUT NOT LIMITED TO,  THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL  THE COPYRIGHT HOLDER OR CONTRIBUTORS  BE LIABLE FOR ANY DIRECT,
// INDIRECT,  INCIDENTAL, SPECIAL,  EXEMPLARY,  OR CONSEQUENTIAL  DAMAGES (INCLUDING,
// BUT NOT LIMITED TO,  PROCUREMENT OF  SUBSTITUTE  GOODS OR  SERVICES;  LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION)  HOWEVER CAUSED  AND ON ANY THEORY OF
// LIABILITY,  WHETHER IN CONTRACT,  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
// OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include"stdlib.h"
#include"string.h"
#include"math.h"
#include"assert.h"

#include"daoConst.h"
#include"daoMap.h"
#include"daoArray.h"
#include"daoString.h"
#include"daoNumtype.h"
#include"daoValue.h"
#include"daoGC.h"

#define RB_RED    0
#define RB_BLACK  1


static DNode* DNode_New( DMap *map, int keytype, int valtype )
{
	if( map->list && map->keytype == keytype && map->valtype == valtype ){
		DNode *node = map->list;
		map->list = map->list->parent;
		node->parent = NULL;
		return node;
	}
	return (DNode*) dao_calloc( 1, sizeof(DNode) );
}
DNode* DNode_First( DNode *self )
{
	if( self ) while( self->left ) self = self->left;
	return self;
}

DNode* DNode_Next( DNode *self )
{
	DNode* p = self->right;
	if( self->right ){
		while( p->left ) p = p->left;
	}else if( self->parent ){
		if( self == self->parent->left )
			p = self->parent;
		else{
			p = self->parent;
			while( p->parent && p==p->parent->right ) p = p->parent;
			p = p->parent;
		}
	}
	return p;
}
void* DNode_GetKey( DNode *self )
{
	return self->key.pVoid;
}
void* DNode_GetValue( DNode *self )
{
	return self->value.pVoid;
}
DaoValue* DNode_Key( DNode *self )
{
	return self->key.pValue;
}
DaoValue* DNode_Value( DNode *self )
{
	return self->value.pValue;
}

DMap* DMap_New( short kt, short vt )
{
	DMap *self = (DMap*) dao_malloc( sizeof( DMap) );
	self->size = 0;
	self->tsize = 0;
	self->list = NULL;
	self->root = NULL;
	self->table = NULL;
	self->keytype = kt;
	self->valtype = vt;
	self->hashing = 0;
	self->mutating = 0;
	return self;
}
DMap* DHash_New( short kt, short vt )
{
	DMap *self = DMap_New( kt, vt );
	self->hashing = HASH_SEED;
	self->tsize = 4;
	self->table = (DNode**) dao_calloc( self->tsize, sizeof(DNode*) );
	return self;
}

unsigned int MurmurHash2 ( const void * key, int len, unsigned int seed );

static int DaoValue_Hash( DaoValue *self, unsigned int buf[], int id, int max, unsigned int seed )
{
	void *data = NULL;
	int i, len = 0;
	int id2 = id;
	unsigned int hash = 0;
	switch( self->type ){
	case DAO_INTEGER :
		data = & self->xInteger.value;  len = sizeof(daoint);  break;
	case DAO_FLOAT   :
		data = & self->xFloat.value;  len = sizeof(float);  break;
	case DAO_DOUBLE  :
		data = & self->xDouble.value;  len = sizeof(double);  break;
	case DAO_COMPLEX :
		data = & self->xComplex.value;  len = sizeof(complex16);  break;
	case DAO_LONG :
		data = self->xLong.value->data;
		len = self->xLong.value->size*sizeof(short);
		break;
	case DAO_ENUM  :
		data = self->xEnum.etype->name->mbs; /* XXX */
		len = self->xEnum.etype->name->size;
		break;
	case DAO_STRING  :
		if( self->xString.data->mbs ){
			data = self->xString.data->mbs;
			len = self->xString.data->size;
		}else{
			data = self->xString.data->wcs;
			len = self->xString.data->size * sizeof(wchar_t);
		}
		break;
	case DAO_ARRAY :
		data = self->xArray.data.p;
		len = self->xArray.size;
		switch( self->xArray.etype ){
		case DAO_INTEGER : len *= sizeof(int); break;
		case DAO_FLOAT   : len *= sizeof(float); break;
		case DAO_DOUBLE  : len *= sizeof(double); break;
		case DAO_COMPLEX : len *= sizeof(complex16); break;
		default : break;
		}
		break;
	case DAO_TUPLE :
		for(i=0; i<self->xTuple.size; i++){
			id = DaoValue_Hash( self->xTuple.items[i], buf, id, max, seed );
			if( id >= max ) break;
		}
		break;
	default : data = & self; len = sizeof(DaoValue*); break;
	}
	if( data ) hash = MurmurHash2( data, len, seed );
	if( id == id2 && id < max ){
		buf[id] = hash;
		id += 1;
	}
	return id;
}

static int DHash_HashIndex( DMap *self, void *key )
{
#define HASH_MAX  32
	DString *s;
	DArray *array;
	unsigned int buf[HASH_MAX];
	unsigned int T = self->tsize;
	unsigned id = 0;
	void *data;
	int m;

	switch( self->keytype ){
	case D_STRING :
		s = (DString*)key;
		m = s->size;
		data = NULL;
		if( s->mbs ){
			data = s->mbs;
		}else{
			data = s->wcs;
			m *= sizeof(wchar_t);
		}
		id = MurmurHash2( data, m, self->hashing ) % T;
		break;
	case D_VALUE :
	case D_VALUE2 :
	case D_VALUE3 :
		m = DaoValue_Hash( (DaoValue*) key, buf, 0, HASH_MAX, self->hashing );
		if( m ==1 ){
			id = buf[0] % T;
		}else{
			id = MurmurHash2( buf, m*sizeof(unsigned int), self->hashing ) % T;
		}
		break;
	case D_ARRAY :
		array = (DArray*)key;
		m = array->size * sizeof(void*);
		id = MurmurHash2( array->items.pVoid, m, self->hashing ) % T;
		break;
	case D_VOID2 :
		id = MurmurHash2( key, 2*sizeof(void*), self->hashing ) % T;
		break;
	case D_VMCODE :
		id = MurmurHash2( key, 4*sizeof(unsigned short), self->hashing ) % T;
		break;
	case D_VMCODE2 :
		id = MurmurHash2( key, 3*sizeof(unsigned short), self->hashing ) % T;
		break;
	default :
		id = MurmurHash2( & key, sizeof(void*), self->hashing ) % T;
		break;
	}
	return (int)id;
}
static DNode* DMap_SimpleInsert( DMap *self, DNode *node );
static void DMap_InsertNode( DMap *self, DNode *node );
static void DMap_InsertTree( DMap *self, DNode *node )
{
	DNode *left = node->left;
	DNode *right = node->right;
	node->hash = DHash_HashIndex( self, node->key.pVoid );
	node->parent = node->left = node->right = NULL;
	self->root = self->table[ node->hash ];
	if( self->root == NULL ){
		node->color = RB_BLACK;
		self->table[ node->hash ] = node;
		self->size += 1;
	}else{
		DMap_SimpleInsert( self, node );
		DMap_InsertNode( self, node );
		self->table[ node->hash ] = self->root;
	}
	if( left ) DMap_InsertTree( self, left );
	if( right ) DMap_InsertTree( self, right );
}
static int DMap_Lockable( DMap *self )
{
	int lockable = self->keytype >= D_VALUE && self->keytype <= D_VALUE3;
	lockable |= self->valtype >= D_VALUE && self->valtype <= D_VALUE3;
	return lockable;
}
static void DHash_ResetTable( DMap *self )
{
	DNode **nodes = self->table;
	int i, locked, tsize = self->tsize;

	if( self->hashing ==0 ) return;
	locked = DMap_Lockable( self ) ? DaoGC_LockMap( self ) : 0;
	self->tsize = 2 * self->size + 1;
	self->table = (DNode**)dao_calloc( self->tsize, sizeof(DNode*) );
	self->size = 0;
	for(i=0; i<tsize; i++) if( nodes[i] ) DMap_InsertTree( self, nodes[i] );
	DaoGC_UnlockMap( self, locked );
	if( nodes ) dao_free( nodes );
}
DMap* DMap_Copy( DMap *other )
{
	DMap *self = NULL;
	if( other->hashing ){
		self = DHash_New( other->keytype, other->valtype );
		self->tsize = other->tsize;
		self->table = (DNode**)dao_realloc( self->table, other->tsize*sizeof(DNode*) );
		memset( self->table, 0, other->tsize*sizeof(DNode*) );
	}else{
		self = DMap_New( other->keytype, other->valtype );
	}
	DMap_Assign( self, other );
	return self;
}
void DMap_Assign( DMap *self, DMap *other )
{
	DNode *node = DMap_First( other );
	DMap_Reset( self );
	while( node ){
		DMap_Insert( self, node->key.pVoid, node->value.pVoid );
		node = DMap_Next( other, node );
	}
}
static void DMap_DeleteNode( DMap *self, DNode *node );
void DMap_Delete( DMap *self )
{
	DNode *p, *node;
	DMap_Clear( self );
	if( self->table ) dao_free( self->table );
	node = self->list;
	while( node ){
		p = node;
		node = node->parent;
		DMap_DeleteNode( self, p );
	}
	dao_free( self );
}
static void DMap_SwapNode( DMap *self, DNode *node, DNode *extreme )
{
	void *key, *value;
	int hash = extreme->hash;
	key = extreme->key.pVoid;
	value = extreme->value.pVoid;
	extreme->hash = node->hash;
	extreme->key.pVoid = node->key.pVoid;
	extreme->value.pVoid = node->value.pVoid;
	node->hash = hash;
	node->key.pVoid = key;
	node->value.pVoid = value;
}
static void DMap_CopyItem( void **dest, void *item, short type )
{
	int n = 2*sizeof(void*);
	if( *dest == NULL ){
		switch( type ){
		case D_STRING : *dest = DString_Copy( (DString*) item ); break;
		case D_ARRAY  : *dest = DArray_Copy( (DArray*) item ); break;
		case D_MAP    : *dest = DMap_Copy( (DMap*) item ); break;
		case D_VALUE  :
		case D_VALUE2 :
		case D_VALUE3 : DaoValue_Copy( (DaoValue*)item, (DaoValue**) dest ); break;
		case D_VOID2  : *dest = dao_malloc(n); memcpy(*dest, item, n); break;
		default : *dest = item; break;
		}
	}else{
		switch( type ){
		case D_STRING : DString_Assign( (DString*)(*dest), (DString*) item ); break;
		case D_ARRAY  : DArray_Assign( (DArray*)(*dest), (DArray*) item ); break;
		case D_MAP    : DMap_Assign( (DMap*)(*dest), (DMap*) item ); break;
		case D_VALUE  :
		case D_VALUE2 :
		case D_VALUE3 : DaoValue_Copy( (DaoValue*) item, (DaoValue**) dest ); break;
		case D_VOID2  : memcpy(*dest, item, n); break;
		default : *dest = item; break;
		}
	}
}
static void DMap_DeleteItem( void *item, short type )
{
	switch( type ){
	case D_STRING : DString_Delete( (DString*) item ); break;
	case D_ARRAY  : DArray_Delete( (DArray*) item ); break;
	case D_MAP    : DMap_Delete( (DMap*) item ); break;
	case D_VALUE  :
	case D_VALUE2 :
	case D_VALUE3 : GC_DecRC( (DaoValue*) item ); break;
	case D_VOID2  : dao_free( item ); break;
	default : break;
	}
}
static void DMap_BufferNode( DMap *self, DNode *node )
{
	node->parent = node->left = node->right = NULL;
	if( self->keytype >= D_VALUE && self->keytype <= D_VALUE3 ) DaoValue_Clear( & node->key.pValue );
	if( self->valtype >= D_VALUE && self->valtype <= D_VALUE3 ) DaoValue_Clear( & node->value.pValue );
	if( self->list == NULL ){
		self->list = node;
		return;
	}
	node->parent = self->list;
	self->list = node;
}
static void DMap_BufferTree( DMap *self, DNode *node )
{
	if( node == NULL ) return;
	DMap_BufferTree( self, node->left );
	DMap_BufferTree( self, node->right );
	DMap_BufferNode( self, node );
}
static void DMap_DeleteNode( DMap *self, DNode *node )
{
	if( node->key.pVoid ) DMap_DeleteItem( node->key.pVoid, self->keytype );
	if( node->value.pVoid ) DMap_DeleteItem( node->value.pVoid, self->valtype );
	dao_free( node );
}
static void DMap_DeleteTree( DMap *self, DNode *node )
{
	if( node == NULL ) return;
	DMap_DeleteTree( self, node->left );
	DMap_DeleteTree( self, node->right );
	DMap_DeleteNode( self, node );
}
void DMap_Clear( DMap *self )
{
	daoint i;
	int locked = DMap_Lockable( self ) ? DaoGC_LockMap( self ) : 0;
	if( self->hashing ){
		for(i=0; i<self->tsize; i++) DMap_DeleteTree( self, self->table[i] );
		if( self->table ) dao_free( self->table );
		self->tsize = 4;
		self->table = (DNode**) dao_calloc( self->tsize, sizeof(DNode*) );
	}else{
		DMap_DeleteTree( self, self->root );
	}
	self->root = NULL;
	self->size = 0;
	DaoGC_UnlockMap( self, locked );
}
void DMap_Reset( DMap *self )
{
	daoint i;
	int locked = DMap_Lockable( self ) ? DaoGC_LockMap( self ) : 0;
	if( self->hashing ){
		for(i=0; i<self->tsize; i++) DMap_BufferTree( self, self->table[i] );
		memset( self->table, 0, self->tsize*sizeof(DNode*) );
	}else{
		DMap_BufferTree( self, self->root );
	}
	self->root = NULL;
	self->size = 0;
	DaoGC_UnlockMap( self, locked );
}

static void DMap_RotateLeft( DMap *self, DNode *child )
{
	DNode *grandpa = child->parent;
	DNode *parent = child->right;

	if( grandpa ){
		if( child == grandpa->right )
			grandpa->right = parent;
		else
			grandpa->left = parent;
	}else{
		self->root = parent;
	}
	parent->parent = grandpa;

	child->right = parent->left;
	if( child->right ) child->right->parent = child;

	parent->left = child;
	child->parent = parent;
}
static void DMap_RotateRight( DMap *self, DNode *parent )
{
	DNode *grandpa = parent->parent;
	DNode *child = parent->left;

	if( grandpa ){
		if( parent == grandpa->right )
			grandpa->right = child;
		else
			grandpa->left = child;
	}else{
		self->root = child;
	}
	child->parent = grandpa;

	parent->left = child->right;
	if( parent->left ) parent->left->parent = parent;

	child->right = parent;
	parent->parent = child;
}
void DMap_InsertNode( DMap *self, DNode *node )
{
	DNode *grandpa, *parent, *uncle;

	node->color = RB_RED;
	self->size ++;

	while( node->parent != NULL ){
		parent = node->parent;
		grandpa = parent->parent;
		if( parent->color == RB_RED ){ /* insert_case2() */
			/* grandpa can't be NULL, since parent is RED and can't be root. */
			uncle = ( parent == grandpa->left ? grandpa->right : grandpa->left );
			if( uncle != NULL && uncle->color == RB_RED ){ /* insert_case3(): */
				parent->color = RB_BLACK;
				uncle->color  = RB_BLACK;
				grandpa->color = RB_RED;
				node = grandpa;
				continue; /* insert_case1() */
			}else{
				if( node == parent->right && parent == grandpa->left ){
					DMap_RotateLeft( self, parent );
					node = node->left;
				}else if( node == parent->left && parent == grandpa->right ){
					/* rotate right around parent: */
					DMap_RotateRight( self, parent );
					node = node->right;
				}
				/* node changed, update parent and grandpa. */
				parent = node->parent;
				grandpa = parent->parent;
				/* insert_case5() */

				parent->color = RB_BLACK;
				grandpa->color = RB_RED;
				if( node == parent->left && parent == grandpa->left )
					DMap_RotateRight( self, grandpa );
				else
					DMap_RotateLeft( self, grandpa );
			}
		}
		break;
	}
	/* insert_case1() as in Wikipedia term: Red-black tree. */
	if( node->parent == NULL){
		self->root = node;
		node->color = RB_BLACK;
	}
}
static void DMap_EraseChild( DMap *self, DNode *node )
{
	DNode *extreme = node;
	DNode *child = 0;

	if( node == NULL ) return;
	self->size --;

	/* deletion by coping */
	if( node->left ){
		extreme = node->left;
		while( extreme->right ) extreme = extreme->right;
		child = extreme->left;
	}else if( node->right ){
		extreme = node->right;
		while( extreme->left ) extreme = extreme->left;
		child = extreme->right;
	}
	DMap_SwapNode( self, node, extreme );

	if( child ){
		/* replace node */
		child->parent = extreme->parent;
		if( extreme->parent ){
			if( extreme == extreme->parent->left )
				extreme->parent->left = child;
			else
				extreme->parent->right = child;
		}
		if( extreme->color == RB_BLACK ){
			if( child->color == RB_RED )
				child->color = RB_BLACK;
			else{
				node = child;
				while( node->parent ){ /* delete_case1() */

					DNode *parent = node->parent;
					DNode *sibling = ( node == parent->left ? parent->right : parent->left );
					if( sibling && sibling->color == RB_RED ){ /* delete_case2() */
						parent->color = RB_RED;
						sibling->color = RB_BLACK;
						if( node == parent->left )
							DMap_RotateLeft( self, parent );
						else
							DMap_RotateRight( self, parent );
					}
					/* node relationship changed, update parent and sibling: */
					parent = node->parent;
					sibling = ( node == parent->left ? parent->right : parent->left );
					if( ! sibling ) break;
					/* delete_case3() */
					if( parent->color == RB_BLACK && sibling->color == RB_BLACK
							&& ( ! sibling->left || sibling->left->color == RB_BLACK )
							&& ( ! sibling->right|| sibling->right->color == RB_BLACK)){
						sibling->color = RB_RED;
						node = node->parent;
						continue; /* delete_case1() */
					}else{
						/* delete_case4() */
						if( parent->color == RB_RED && sibling->color == RB_BLACK
								&& ( ! sibling->left || sibling->left->color == RB_BLACK )
								&& ( ! sibling->right|| sibling->right->color == RB_BLACK)){
							sibling->color = RB_RED;
							parent->color = RB_BLACK;
						}else{
							/* delete_case5() */
							if( node == parent->left && sibling->color == RB_BLACK
									&&( sibling->left && sibling->left->color == RB_RED )
									&&( !sibling->right|| sibling->right->color == RB_BLACK)){
								sibling->color = RB_RED;
								sibling->left->color = RB_BLACK;
								DMap_RotateRight( self, sibling );
							}else if( node == parent->right && sibling->color == RB_BLACK
									&&( sibling->right && sibling->right->color == RB_RED )
									&&( !sibling->left || sibling->left->color == RB_BLACK)){
								sibling->color = RB_RED;
								sibling->right->color = RB_BLACK;
								DMap_RotateLeft( self, sibling );
							}
							/* node relationship changed, update parent and sibling: */
							parent = node->parent;
							sibling = ( node==parent->left ? parent->right:parent->left );
							/* delete_case6() */
							sibling->color = parent->color;
							parent->color = RB_BLACK;
							if( node == parent->left ){
								sibling->right->color = RB_BLACK;
								DMap_RotateLeft( self, parent );
							}else{
								sibling->left->color = RB_BLACK;
								DMap_RotateRight( self, parent );
							}
						} /* end if */
					} /* end if */
				} /* end while */
			}
		}
	}else if( extreme->parent ){
		if( extreme == extreme->parent->left )
			extreme->parent->left = NULL;
		else
			extreme->parent->right = NULL;
	}else{
		self->root = NULL;
	}
	DMap_BufferNode( self, extreme );
}
void DMap_EraseNode( DMap *self, DNode *node )
{
	int locked;
	if( node == NULL ) return;
	if( self->hashing ){
		int hash = node->hash;
		self->root = self->table[ hash ];
		if( self->root == NULL ) return;
		locked = DMap_Lockable( self ) ? DaoGC_LockMap( self ) : 0;
		DMap_EraseChild( self, node );
		self->table[ hash ] = self->root;
		DaoGC_UnlockMap( self, locked );
		if( self->size < 0.25*self->tsize ) DHash_ResetTable( self );
	}else{
		locked = DMap_Lockable( self ) ? DaoGC_LockMap( self ) : 0;
		DMap_EraseChild( self, node );
		DaoGC_UnlockMap( self, locked );
	}
}
static daoint DArray_Compare( DArray *k1, DArray *k2 )
{
	daoint i = 0, n = k1->size;
	daoint *v1 = k1->items.pInt;
	daoint *v2 = k2->items.pInt;
	if( n != k2->size ) return (int)(n - k2->size);
	while( i < n && v1[i] == v2[i] ) i += 1;
	if( i < n ) return v1[i] - v2[i];
	return 0;
}
static daoint DVoid2_Compare( void **k1, void **k2 )
{
	if( k1[0] != k2[0] ) return (daoint)k1[0] - (daoint)k2[0];
	return (daoint)k1[1] - (daoint)k2[1];
}
static daoint DaoVmCode_Compare( DaoVmCode *k1, DaoVmCode *k2 )
{
	if( k1->code != k2->code ) return k1->code - k2->code;
	if( k1->a != k2->a ) return k1->a - k2->a;
	if( k1->b != k2->b ) return k1->b - k2->b;
	return k1->c - k2->c;
}
static daoint DaoVmCode_Compare2( DaoVmCode *k1, DaoVmCode *k2 )
{
	if( k1->code != k2->code ) return k1->code - k2->code;
	if( k1->a != k2->a ) return k1->a - k2->a;
	return k1->b - k2->b;
}

extern int DaoArray_Compare( DaoArray *x, DaoArray *y );
extern int DaoTuple_Compare( DaoTuple *lt, DaoTuple *rt );
extern int DaoList_Compare( DaoList *list1, DaoList *list2 );

int DaoValue_Compare2( DaoValue *left, DaoValue *right )
{
	if( left == right ) return 0;
	if( left == NULL || right == NULL ) return left < right ? -100 : 100;
	if( left->type != right->type ) return left->type < right->type ? -100 : 100;
	if( left->type == DAO_TUPLE && left->xTuple.ctype == right->xTuple.ctype ){
		return DaoTuple_Compare( (DaoTuple*) left, (DaoTuple*) right );
#ifdef DAO_WITH_NUMARRAY
	}else if( left->type == DAO_ARRAY && left->xArray.etype == right->xArray.etype ){
		return DaoArray_Compare( (DaoArray*) left, (DaoArray*) right );
#endif
	}else if( left->type == DAO_LIST && left->xList.ctype == right->xList.ctype ){
		return DaoList_Compare( (DaoList*) left, (DaoList*) right );
	}
	if( left->type <= DAO_STRING ) return DaoValue_Compare( left, right );
	return left < right ? -100 : 100;
}
int DaoValue_Compare3( DaoValue *left, DaoValue *right )
{
	if( left == right ) return 0;
	if( left == NULL || right == NULL ) return left < right ? -100 : 100;
	if( left->type != right->type ) return left->type < right->type ? -100 : 100;
	if( left->type <= DAO_STRING ) return DaoValue_Compare( left, right );
	return left < right ? -100 : 100;
}
static daoint DMap_CompareKeys( DMap *self, void *k1, void *k2 )
{
	daoint cmp = 0;
	switch( self->keytype ){
	case D_STRING : cmp = DString_Compare( (DString*) k1, (DString*) k2 );        break;
	case D_VALUE  : cmp = DaoValue_Compare( (DaoValue*) k1, (DaoValue*) k2 );     break;
	case D_VALUE2 : cmp = DaoValue_Compare2( (DaoValue*) k1, (DaoValue*) k2 );    break;
	case D_VALUE3 : cmp = DaoValue_Compare3( (DaoValue*) k1, (DaoValue*) k2 );    break;
	case D_ARRAY  : cmp = DArray_Compare( (DArray*) k1, (DArray*) k2 );           break;
	case D_VOID2  : cmp = DVoid2_Compare( (void**) k1, (void**) k2 );             break;
	case D_VMCODE : cmp = DaoVmCode_Compare( (DaoVmCode*) k1, (DaoVmCode*) k2 );  break;
	case D_VMCODE2: cmp = DaoVmCode_Compare2( (DaoVmCode*) k1, (DaoVmCode*) k2 ); break;
	default : cmp = (daoint)k1 - (daoint)k2; break;
	}
	/*
	// Note:
	// A MBS string and WCS string of the same content may be hashed into
	// different values, so there is no guarantee that they will be considered
	// equivalent when they are used as keys in hash maps.
	//
	// Because of this, it should always better to consider MBS keys are different
	// from WCS keys in HASH maps, to avoid the pitfalls that some keys are considered
	// equivalent between MBS and WCS, while others are not.
	*/
	if( self->hashing && cmp == 0 ){
		DString *s1 = NULL;
		DString *s2 = NULL;
		if( self->keytype == D_STRING ){
			s1 = (DString*) k1;
			s2 = (DString*) k2;
		}else if( self->keytype >= D_VALUE && self->keytype <= D_VALUE3 ){
			DaoValue *skv1 = (DaoValue*) k1;
			DaoValue *skv2 = (DaoValue*) k2;
			if( skv1->type == DAO_STRING ){
				s1 = skv1->xString.data;
				s2 = skv2->xString.data;
			}
		}
		if( s1 != NULL && (s1->mbs == NULL) != (s2->mbs == NULL) ) cmp = s1->mbs ? 1 : -1;
	}
	return cmp;
}

static DNode* DMap_FindChild( DMap *self, DNode *root, void *key, KeySearchType type )
{
	DNode *p = root;
	DNode *m = 0;
	daoint compare;

	if( root == NULL ) return NULL;

	for(;;){
		compare = DMap_CompareKeys( self, key, p->key.pVoid );
		if( compare == 0 ) return p;

		if( compare < 0 ){
			if( type == KEY_GE ) m = p;
			if( p->left ) p = p->left; else break;
		}else{
			if( type == KEY_LE ) m = p;
			if( p->right ) p = p->right; else break;
		}
	}
	return m;
}
static DNode* DMap_FindNode( DMap *self, void *key, KeySearchType type )
{
	DNode *root = self->root;
	int id;
	if( self->hashing ){
		id = DHash_HashIndex( self, key );
		root = self->table[id];
		if( root == NULL ) return NULL;
		/* if( DMap_CompareKeys( self, key, root->key.pVoid ) ==0 ) return root; */
	}
	return DMap_FindChild( self, root, key, type );
}
static DNode* DMap_SimpleInsert( DMap *self, DNode *node )
{
	DNode *p = self->root;
	int compare;
	node->color = RB_RED;
	if( self->root == NULL ) return node;
	for(;;){
		compare = DMap_CompareKeys( self, node->key.pVoid, p->key.pVoid );
		if( compare == 0 ){
			return p;
		}else if( compare < 0 ){
			if( p->left ){
				p = p->left;
			}else{
				p->left = node;
				node->parent = p;
				break;
			}
		}else{
			if( p->right ){
				p = p->right;
			}else{
				p->right = node;
				node->parent = p;
				break;
			}
		}
	}
	return node;
}
DNode* DMap_Insert( DMap *self, void *key, void *value )
{
	DNode *p, *node = DNode_New( self, self->keytype, self->valtype );
	void *okey = node->key.pVoid;
	void *ovalue = node->value.pVoid;
	int locked, id = 0;
	if( self->hashing ){
		id = DHash_HashIndex( self, key );
		node->hash = id;
		self->root = self->table[id];
		if( self->root ==NULL ){
			self->size += 1;
			self->table[id] = node;
			node->color = RB_BLACK;
			DMap_CopyItem( & node->key.pVoid, key, self->keytype );
			DMap_CopyItem( & node->value.pVoid, value, self->valtype );
			return node;
		}
	}
	node->key.pVoid = key;
	node->value.pVoid = value;
	p = DMap_SimpleInsert( self, node );
	node->key.pVoid = okey;
	node->value.pVoid = ovalue;
	if( p == node ){ /* key not exist: */
		DMap_CopyItem( & node->key.pVoid, key, self->keytype );
		DMap_CopyItem( & node->value.pVoid, value, self->valtype );
		locked = DMap_Lockable( self ) ? DaoGC_LockMap( self ) : 0;
		DMap_InsertNode( self, node );
		DaoGC_UnlockMap( self, locked );
		if( self->hashing ){
			self->table[id] = self->root;
			if( self->size >= self->tsize ) DHash_ResetTable( self );
		}
	}else{
		if( self->valtype != D_VALUE ){
			DMap_DeleteItem( p->value.pVoid, self->valtype );
			p->value.pVoid = NULL;
		}
		DMap_CopyItem( & p->value.pVoid, value, self->valtype );
		DMap_BufferNode( self, node );
	}
	return p;
}
void DMap_Erase( DMap *self, void *key )
{
	DMap_EraseNode( self, DMap_FindNode( self, key, KEY_EQ ) );
}
DNode* DMap_Find( DMap *self, void *key )
{
	return DMap_FindNode( self, key, KEY_EQ );
}
DNode* DMap_FindLE( DMap *self, void *key )
{
	return DMap_FindNode( self, key, KEY_LE );
}
DNode* DMap_FindGE( DMap *self, void *key )
{
	return DMap_FindNode( self, key, KEY_GE );
}
DNode* DMap_First( DMap *self )
{
	DNode *node = NULL;
	daoint i = 0;
	if( self == NULL ) return NULL;
	if( self->hashing ){
		while( i < self->tsize && self->table[i] == NULL ) i += 1;
		if( i < self->tsize ) node = DNode_First( self->table[i] );
	}
	if( node == NULL && self->root ) node = DNode_First( self->root );
	return node;
}
DNode* DMap_Next( DMap *self, DNode *node )
{
	DNode *next;
	if( node == NULL ) return NULL;
	next = DNode_Next( node );
	if( next == NULL && self->hashing ){
		daoint i = node->hash + 1;
		while( i < self->tsize && self->table[i] == NULL ) i += 1;
		if( i < self->tsize ) next = DNode_First( self->table[i] );
	}
	return next;
}


/*
//  PUBLIC DOMAIN CODES
//  http://sites.google.com/site/murmurhash/
//  http://www.burtleburtle.net/bob/hash/doobs.html
*/

/*
//  MurmurHash2, by Austin Appleby
//
//  Note - This code makes a few assumptions about how your machine behaves -
//  1. We can read a 4-byte value from any address without crashing
//  2. sizeof(int) == 4
//
//  And it has a few limitations -
//  1. It will not work incrementally.
//  2. It will not produce the same results on little-endian and big-endian
//  machines.
*/
unsigned int MurmurHash2 ( const void * key, int len, unsigned int seed )
{
	/*
	// 'm' and 'r' are mixing constants generated offline.
	// They're not really 'magic', they just happen to work well.
	*/
	const unsigned int m = 0x5bd1e995;
	const int r = 24;

	/* Initialize the hash to a 'random' value */
	unsigned int h = seed ^ len;

	/* Mix 4 bytes at a time into the hash */
	const unsigned char * data = (const unsigned char *)key;

	while(len >= 4) {
		unsigned int k = *(unsigned int *)data;

		k *= m;
		k ^= k >> r;
		k *= m;

		h *= m;
		h ^= k;

		data += 4;
		len -= 4;
	}

	/* Handle the last few bytes of the input array */
	switch(len)
	{
	case 3: h ^= data[2] << 16;
	case 2: h ^= data[1] << 8;
	case 1: h ^= data[0];
			h *= m;
	};

	/*
	// Do a few final mixes of the hash to ensure the last few
	// bytes are well-incorporated.
	*/
	h ^= h >> 13;
	h *= m;
	h ^= h >> 15;

	return h;
}
