/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.blobstore.s3.internal;

import com.amazonaws.SdkBaseException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.iterable.S3Objects;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.BucketLifecycleConfiguration;
import com.amazonaws.services.s3.model.ObjectTagging;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.SetObjectTaggingRequest;
import com.amazonaws.services.s3.model.Tag;
import com.amazonaws.services.s3.model.lifecycle.LifecycleFilter;
import com.amazonaws.services.s3.model.lifecycle.LifecycleFilterPredicate;
import com.amazonaws.services.s3.model.lifecycle.LifecycleTagPredicate;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicates;
import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.hash.HashCode;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.locks.Lock;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import org.joda.time.DateTime;
import org.sonatype.nexus.blobstore.BlobIdLocationResolver;
import org.sonatype.nexus.blobstore.BlobStoreSupport;
import org.sonatype.nexus.blobstore.BlobSupport;
import org.sonatype.nexus.blobstore.MetricsInputStream;
import org.sonatype.nexus.blobstore.StreamMetrics;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobAttributes;
import org.sonatype.nexus.blobstore.api.BlobId;
import org.sonatype.nexus.blobstore.api.BlobMetrics;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreConfiguration;
import org.sonatype.nexus.blobstore.api.BlobStoreException;
import org.sonatype.nexus.blobstore.api.BlobStoreMetrics;
import org.sonatype.nexus.blobstore.s3.internal.AmazonS3Factory;
import org.sonatype.nexus.blobstore.s3.internal.S3AttributesLocation;
import org.sonatype.nexus.blobstore.s3.internal.S3BlobAttributes;
import org.sonatype.nexus.blobstore.s3.internal.S3BlobStoreMetricsStore;
import org.sonatype.nexus.blobstore.s3.internal.S3Copier;
import org.sonatype.nexus.blobstore.s3.internal.S3PropertiesFile;
import org.sonatype.nexus.blobstore.s3.internal.S3Uploader;
import org.sonatype.nexus.common.log.DryRunPrefix;
import org.sonatype.nexus.common.stateguard.Guarded;

@Named(value="S3")
public class S3BlobStore
extends BlobStoreSupport<S3AttributesLocation> {
    public static final String TYPE = "S3";
    public static final String BLOB_CONTENT_SUFFIX = ".bytes";
    public static final String CONFIG_KEY = "s3";
    public static final String BUCKET_KEY = "bucket";
    public static final String BUCKET_PREFIX_KEY = "prefix";
    public static final String ACCESS_KEY_ID_KEY = "accessKeyId";
    public static final String SECRET_ACCESS_KEY_KEY = "secretAccessKey";
    public static final String SESSION_TOKEN_KEY = "sessionToken";
    public static final String ASSUME_ROLE_KEY = "assumeRole";
    public static final String REGION_KEY = "region";
    public static final String ENDPOINT_KEY = "endpoint";
    public static final String EXPIRATION_KEY = "expiration";
    public static final String SIGNERTYPE_KEY = "signertype";
    public static final String FORCE_PATH_STYLE_KEY = "forcepathstyle";
    public static final String BUCKET_REGEX = "^([a-z]|(\\d(?!\\d{0,2}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})))([a-z\\d]|(\\.(?!(\\.|-)))|(-(?!\\.))){1,61}[a-z\\d]$";
    public static final int DEFAULT_EXPIRATION_IN_DAYS = 3;
    public static final String METADATA_FILENAME = "metadata.properties";
    public static final String TYPE_KEY = "type";
    public static final String TYPE_V1 = "s3/1";
    private static final String CONTENT_PREFIX = "content";
    public static final String DIRECT_PATH_PREFIX = "content/directpath";
    static final Tag DELETED_TAG = new Tag("deleted", "true");
    static final String LIFECYCLE_EXPIRATION_RULE_ID = "Expire soft-deleted blobstore objects";
    private static final String FILE_V1 = "file/1";
    private final AmazonS3Factory amazonS3Factory;
    private S3Uploader uploader;
    private S3Copier copier;
    private S3BlobStoreMetricsStore storeMetrics;
    private LoadingCache<BlobId, S3Blob> liveBlobs;
    private AmazonS3 s3;

    @Inject
    public S3BlobStore(AmazonS3Factory amazonS3Factory, BlobIdLocationResolver blobIdLocationResolver, @Named(value="multipart-uploader") S3Uploader uploader, S3Copier copier, S3BlobStoreMetricsStore storeMetrics, DryRunPrefix dryRunPrefix) {
        super(blobIdLocationResolver, dryRunPrefix);
        this.amazonS3Factory = (AmazonS3Factory)((Object)Preconditions.checkNotNull((Object)((Object)amazonS3Factory)));
        this.uploader = (S3Uploader)Preconditions.checkNotNull((Object)uploader);
        this.copier = (S3Copier)Preconditions.checkNotNull((Object)copier);
        this.storeMetrics = (S3BlobStoreMetricsStore)((Object)Preconditions.checkNotNull((Object)((Object)storeMetrics)));
    }

    protected void doStart() throws Exception {
        S3PropertiesFile metadata = new S3PropertiesFile(this.s3, this.getConfiguredBucket(), this.metadataFilePath());
        if (metadata.exists()) {
            metadata.load();
            String type = metadata.getProperty(TYPE_KEY);
            Preconditions.checkState((TYPE_V1.equals(type) || FILE_V1.equals(type) ? 1 : 0) != 0, (String)"Unsupported blob store type/version: %s in %s", (Object)type, (Object)((Object)metadata));
        } else {
            metadata.setProperty(TYPE_KEY, TYPE_V1);
            metadata.store();
        }
        this.liveBlobs = CacheBuilder.newBuilder().weakValues().build(CacheLoader.from(arg_0 -> S3Blob.new(this, arg_0)));
        this.storeMetrics.setBucket(this.getConfiguredBucket());
        this.storeMetrics.setBucketPrefix(this.getBucketPrefix());
        this.storeMetrics.setS3(this.s3);
        this.storeMetrics.setBlobStore((BlobStore)this);
        this.storeMetrics.start();
    }

    protected void doStop() throws Exception {
        this.liveBlobs = null;
        this.storeMetrics.stop();
    }

    private String contentPath(BlobId id) {
        return String.valueOf(this.getLocation(id)) + BLOB_CONTENT_SUFFIX;
    }

    private String metadataFilePath() {
        return String.valueOf(this.getBucketPrefix()) + METADATA_FILENAME;
    }

    private String attributePath(BlobId id) {
        return String.valueOf(this.getLocation(id)) + ".properties";
    }

    protected String attributePathString(BlobId blobId) {
        return this.attributePath(blobId);
    }

    private String getLocation(BlobId id) {
        return String.valueOf(this.getContentPrefix()) + this.blobIdLocationResolver.getLocation(id);
    }

    protected Blob doCreate(InputStream blobData, Map<String, String> headers, @Nullable BlobId blobId) {
        return this.create(headers, destination -> {
            Throwable throwable = null;
            Object var4_5 = null;
            try (InputStream data = blobData;){
                MetricsInputStream input = new MetricsInputStream(data);
                this.uploader.upload(this.s3, this.getConfiguredBucket(), destination, (InputStream)input);
                return input.getMetrics();
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }, blobId);
    }

    @Guarded(by={"STARTED"})
    public Blob create(Path sourceFile, Map<String, String> headers, long size, HashCode sha1) {
        throw new BlobStoreException("hard links not supported", null);
    }

    private Blob create(Map<String, String> headers, BlobIngester ingester, @Nullable BlobId assignedBlobId) {
        BlobId blobId = this.getBlobId(headers, assignedBlobId);
        String blobPath = this.contentPath(blobId);
        String attributePath = this.attributePath(blobId);
        boolean isDirectPath = Boolean.parseBoolean(headers.getOrDefault("BlobStore.direct-path", "false"));
        Long existingSize = null;
        if (isDirectPath) {
            S3BlobAttributes blobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), attributePath);
            if (this.exists(blobId)) {
                existingSize = this.getContentSizeForDeletion(blobAttributes);
            }
        }
        S3Blob blob = (S3Blob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
        Lock lock = blob.lock();
        try {
            this.log.debug("Writing blob {} to {}", (Object)blobId, (Object)blobPath);
            StreamMetrics streamMetrics = ingester.ingestTo(blobPath);
            BlobMetrics metrics = new BlobMetrics(new DateTime(), streamMetrics.getSha1(), streamMetrics.getSize());
            blob.refresh(headers, metrics);
            S3BlobAttributes blobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), attributePath, headers, metrics);
            blobAttributes.store();
            if (isDirectPath && existingSize != null) {
                this.storeMetrics.recordDeletion(existingSize);
            }
            this.storeMetrics.recordAddition(blobAttributes.getMetrics().getContentSize());
            S3Blob s3Blob = blob;
            return s3Blob;
        }
        catch (IOException e) {
            this.deleteQuietly(attributePath);
            this.deleteQuietly(blobPath);
            throw new BlobStoreException((Throwable)e, blobId);
        }
        finally {
            lock.unlock();
        }
    }

    @Guarded(by={"STARTED"})
    public Blob copy(BlobId blobId, Map<String, String> headers) {
        Blob sourceBlob = (Blob)Preconditions.checkNotNull((Object)this.get(blobId));
        String sourcePath = this.contentPath(sourceBlob.getId());
        return this.create(headers, destination -> {
            this.copier.copy(this.s3, this.getConfiguredBucket(), sourcePath, destination);
            BlobMetrics metrics = sourceBlob.getMetrics();
            return new StreamMetrics(metrics.getContentSize(), metrics.getSha1Hash());
        }, null);
    }

    @Nullable
    @Guarded(by={"STARTED"})
    public Blob get(BlobId blobId) {
        return this.get(blobId, false);
    }

    @Nullable
    public Blob get(BlobId blobId, boolean includeDeleted) {
        S3Blob blob;
        block10: {
            Preconditions.checkNotNull((Object)blobId);
            blob = (S3Blob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
            if (blob.isStale()) {
                Lock lock = blob.lock();
                try {
                    if (!blob.isStale()) break block10;
                    S3BlobAttributes blobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), this.attributePath(blobId));
                    boolean loaded = blobAttributes.load();
                    if (!loaded) {
                        this.log.warn("Attempt to access non-existent blob {} ({})", (Object)blobId, (Object)blobAttributes);
                        return null;
                    }
                    if (blobAttributes.isDeleted() && !includeDeleted) {
                        this.log.warn("Attempt to access soft-deleted blob {} ({}), reason: {}", new Object[]{blobId, blobAttributes, blobAttributes.getDeletedReason()});
                        return null;
                    }
                    try {
                        blob.refresh(blobAttributes.getHeaders(), blobAttributes.getMetrics());
                    }
                    catch (IOException e) {
                        throw new BlobStoreException((Throwable)e, blobId);
                    }
                }
                finally {
                    lock.unlock();
                }
            }
        }
        this.log.debug("Accessing blob {}", (Object)blobId);
        return blob;
    }

    protected boolean doDelete(BlobId blobId, String reason) {
        S3Blob blob = (S3Blob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
        Lock lock = blob.lock();
        try {
            this.log.debug("Soft deleting blob {}", (Object)blobId);
            S3BlobAttributes blobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), this.attributePath(blobId));
            boolean loaded = blobAttributes.load();
            if (!loaded) {
                this.log.warn("Attempt to mark-for-delete non-existent blob {}", (Object)blobId);
                return false;
            }
            if (blobAttributes.isDeleted()) {
                this.log.debug("Attempt to delete already-deleted blob {}", (Object)blobId);
                return false;
            }
            blobAttributes.setDeleted(true);
            blobAttributes.setDeletedReason(reason);
            blobAttributes.store();
            this.s3.setObjectTagging(this.tagAsDeleted(this.contentPath(blobId)));
            this.s3.setObjectTagging(this.tagAsDeleted(this.attributePath(blobId)));
            blob.markStale();
            return true;
        }
        catch (Exception e) {
            throw new BlobStoreException((Throwable)e, blobId);
        }
        finally {
            lock.unlock();
        }
    }

    private SetObjectTaggingRequest tagAsDeleted(String key) {
        return new SetObjectTaggingRequest(this.getConfiguredBucket(), key, new ObjectTagging(Collections.singletonList(DELETED_TAG)));
    }

    private SetObjectTaggingRequest untagAsDeleted(String key) {
        return new SetObjectTaggingRequest(this.getConfiguredBucket(), key, new ObjectTagging(Collections.emptyList()));
    }

    protected boolean doDeleteHard(BlobId blobId) {
        S3Blob blob = (S3Blob)((Object)this.liveBlobs.getUnchecked((Object)blobId));
        Lock lock = blob.lock();
        try {
            this.log.debug("Hard deleting blob {}", (Object)blobId);
            String attributePath = this.attributePath(blobId);
            S3BlobAttributes blobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), attributePath);
            Long contentSize = this.getContentSizeForDeletion(blobAttributes);
            String blobPath = this.contentPath(blobId);
            boolean blobDeleted = this.delete(blobPath);
            this.delete(attributePath);
            if (blobDeleted && contentSize != null) {
                this.storeMetrics.recordDeletion(contentSize);
            }
            boolean bl = blobDeleted;
            return bl;
        }
        catch (IOException e) {
            throw new BlobStoreException((Throwable)e, blobId);
        }
        finally {
            lock.unlock();
            this.liveBlobs.invalidate((Object)blobId);
        }
    }

    @Nullable
    private Long getContentSizeForDeletion(S3BlobAttributes blobAttributes) {
        try {
            blobAttributes.load();
            return blobAttributes.getMetrics() != null ? Long.valueOf(blobAttributes.getMetrics().getContentSize()) : null;
        }
        catch (Exception e) {
            this.log.warn("Unable to load attributes {}, delete will not be added to metrics.", (Object)blobAttributes, (Object)e);
            return null;
        }
    }

    @Guarded(by={"STARTED"})
    public BlobStoreMetrics getMetrics() {
        return this.storeMetrics.getMetrics();
    }

    protected void doInit(BlobStoreConfiguration configuration) {
        try {
            BucketLifecycleConfiguration lifecycleConfiguration;
            this.s3 = this.amazonS3Factory.create(configuration);
            if (!this.s3.doesBucketExist(this.getConfiguredBucket())) {
                this.s3.createBucket(this.getConfiguredBucket());
                if (this.getConfiguredExpirationInDays() >= 0) {
                    this.addBucketLifecycleConfiguration(null);
                }
            } else if (this.getConfiguredExpirationInDays() >= 0 && !this.isExpirationLifecycleConfigurationPresent(lifecycleConfiguration = this.s3.getBucketLifecycleConfiguration(this.getConfiguredBucket()))) {
                this.addBucketLifecycleConfiguration(lifecycleConfiguration);
            }
            this.setConfiguredBucket(this.getConfiguredBucket());
        }
        catch (Exception e) {
            throw new BlobStoreException("Unable to initialize blob store bucket: " + this.getConfiguredBucket(), (Throwable)e, null);
        }
    }

    private boolean isExpirationLifecycleConfigurationPresent(BucketLifecycleConfiguration lifecycleConfiguration) {
        return lifecycleConfiguration != null && lifecycleConfiguration.getRules() != null && lifecycleConfiguration.getRules().stream().filter(r -> r.getExpirationInDays() == this.getConfiguredExpirationInDays()).anyMatch(r -> {
            LifecycleFilterPredicate predicate = r.getFilter().getPredicate();
            if (predicate instanceof LifecycleTagPredicate) {
                LifecycleTagPredicate tagPredicate = (LifecycleTagPredicate)predicate;
                return DELETED_TAG.equals((Object)tagPredicate.getTag());
            }
            return false;
        });
    }

    private BucketLifecycleConfiguration makeLifecycleConfiguration(BucketLifecycleConfiguration existing, int expirationInDays) {
        BucketLifecycleConfiguration.Rule rule = new BucketLifecycleConfiguration.Rule().withId(LIFECYCLE_EXPIRATION_RULE_ID).withFilter(new LifecycleFilter((LifecycleFilterPredicate)new LifecycleTagPredicate(DELETED_TAG))).withExpirationInDays(expirationInDays).withStatus("Enabled");
        if (existing != null) {
            List rules = existing.getRules().stream().filter(r -> !LIFECYCLE_EXPIRATION_RULE_ID.equals(r.getId())).collect(Collectors.toList());
            rules.add(rule);
            existing.setRules(rules);
            return existing;
        }
        return new BucketLifecycleConfiguration().withRules(new BucketLifecycleConfiguration.Rule[]{rule});
    }

    private void addBucketLifecycleConfiguration(BucketLifecycleConfiguration lifecycleConfiguration) {
        this.s3.setBucketLifecycleConfiguration(this.getConfiguredBucket(), this.makeLifecycleConfiguration(lifecycleConfiguration, this.getConfiguredExpirationInDays()));
    }

    private boolean delete(String path) throws IOException {
        this.s3.deleteObject(this.getConfiguredBucket(), path);
        return true;
    }

    private void deleteQuietly(String path) {
        this.s3.deleteObject(this.getConfiguredBucket(), path);
    }

    private void setConfiguredBucket(String bucket) {
        this.blobStoreConfiguration.attributes(CONFIG_KEY).set(BUCKET_KEY, (Object)bucket);
    }

    private String getConfiguredBucket() {
        return this.blobStoreConfiguration.attributes(CONFIG_KEY).require(BUCKET_KEY).toString();
    }

    private int getConfiguredExpirationInDays() {
        return Integer.parseInt(this.blobStoreConfiguration.attributes(CONFIG_KEY).get(EXPIRATION_KEY, (Object)3).toString());
    }

    private String getBucketPrefix() {
        return Optional.ofNullable((String)this.blobStoreConfiguration.attributes(CONFIG_KEY).get(BUCKET_PREFIX_KEY, String.class)).filter((Predicate<String>)Predicates.not(Strings::isNullOrEmpty)).map(s -> String.valueOf(s.replaceFirst("/$", "")) + "/").orElse("");
    }

    private String getContentPrefix() {
        String bucketPrefix = this.getBucketPrefix();
        if (Strings.isNullOrEmpty((String)bucketPrefix)) {
            return "content/";
        }
        return String.valueOf(bucketPrefix) + CONTENT_PREFIX + "/";
    }

    @Guarded(by={"NEW", "STOPPED", "FAILED"})
    public void remove() {
        try {
            boolean contentEmpty = this.s3.listObjects(this.getConfiguredBucket(), this.getContentPrefix()).getObjectSummaries().isEmpty();
            if (contentEmpty) {
                S3PropertiesFile metadata = new S3PropertiesFile(this.s3, this.getConfiguredBucket(), this.metadataFilePath());
                metadata.remove();
                this.storeMetrics.remove();
                this.s3.deleteBucket(this.getConfiguredBucket());
            } else {
                this.log.warn("Unable to delete non-empty blob store content directory in bucket {}", (Object)this.getConfiguredBucket());
            }
        }
        catch (AmazonS3Exception s3Exception) {
            if ("BucketNotEmpty".equals(s3Exception.getErrorCode())) {
                this.log.warn("Unable to delete non-empty blob store bucket {}", (Object)this.getConfiguredBucket());
            }
            throw new BlobStoreException((Throwable)s3Exception, null);
        }
        catch (IOException e) {
            throw new BlobStoreException((Throwable)e, null);
        }
    }

    public Stream<BlobId> getBlobIdStream() {
        S3Objects summaries = S3Objects.withPrefix((AmazonS3)this.s3, (String)this.getConfiguredBucket(), (String)this.getContentPrefix());
        return this.blobIdStream((Iterable<S3ObjectSummary>)summaries);
    }

    public Stream<BlobId> getDirectPathBlobIdStream(String prefix) {
        String subpath = String.valueOf(this.getBucketPrefix()) + String.format("%s/%s", DIRECT_PATH_PREFIX, prefix);
        S3Objects summaries = S3Objects.withPrefix((AmazonS3)this.s3, (String)this.getConfiguredBucket(), (String)subpath);
        return StreamSupport.stream(summaries.spliterator(), false).map(S3ObjectSummary::getKey).filter(key -> key.endsWith(".properties")).map(this::attributePathToDirectPathBlobId);
    }

    private Stream<BlobId> blobIdStream(Iterable<S3ObjectSummary> summaries) {
        return StreamSupport.stream(summaries.spliterator(), false).filter(o -> o.getKey().endsWith(".properties")).map(S3AttributesLocation::new).map(arg_0 -> ((S3BlobStore)this).getBlobIdFromAttributeFilePath(arg_0)).map(BlobId::new);
    }

    @Nullable
    public BlobAttributes getBlobAttributes(BlobId blobId) {
        try {
            S3BlobAttributes blobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), this.attributePath(blobId));
            return blobAttributes.load() ? blobAttributes : null;
        }
        catch (IOException e) {
            this.log.error("Unable to load S3BlobAttributes for blob id: {}", (Object)blobId, (Object)e);
            return null;
        }
    }

    public BlobAttributes getBlobAttributes(S3AttributesLocation attributesFilePath) throws IOException {
        S3BlobAttributes s3BlobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), attributesFilePath.getFullPath());
        s3BlobAttributes.load();
        return s3BlobAttributes;
    }

    public void setBlobAttributes(BlobId blobId, BlobAttributes blobAttributes) {
        try {
            S3BlobAttributes s3BlobAttributes = (S3BlobAttributes)this.getBlobAttributes(blobId);
            s3BlobAttributes.updateFrom(blobAttributes);
            s3BlobAttributes.store();
        }
        catch (Exception e) {
            this.log.error("Unable to set BlobAttributes for blob id: {}, exception: {}", new Object[]{blobId, e.getMessage(), this.log.isDebugEnabled() ? e : null});
        }
    }

    protected void doUndelete(BlobId blobId) {
        this.s3.setObjectTagging(this.untagAsDeleted(this.contentPath(blobId)));
        this.s3.setObjectTagging(this.untagAsDeleted(this.attributePath(blobId)));
    }

    public boolean isStorageAvailable() {
        try {
            return this.s3.doesBucketExistV2(this.getConfiguredBucket());
        }
        catch (SdkBaseException e) {
            this.log.warn("S3 bucket '{}' is not writable.", (Object)this.getConfiguredBucket(), (Object)e);
            return false;
        }
    }

    public boolean exists(BlobId blobId) {
        Preconditions.checkNotNull((Object)blobId);
        S3BlobAttributes blobAttributes = new S3BlobAttributes(this.s3, this.getConfiguredBucket(), this.attributePath(blobId));
        try {
            return blobAttributes.load();
        }
        catch (IOException ioe) {
            this.log.debug("Unable to load attributes {} during existence check, exception: {}", (Object)blobAttributes, (Object)ioe);
            return false;
        }
    }

    private BlobId attributePathToDirectPathBlobId(String s3Key) {
        Preconditions.checkArgument((boolean)s3Key.startsWith(String.valueOf(this.getBucketPrefix()) + DIRECT_PATH_PREFIX + "/"), (String)"Not direct path blob path: %s", (Object)s3Key);
        Preconditions.checkArgument((boolean)s3Key.endsWith(".properties"), (String)"Not blob attribute path: %s", (Object)s3Key);
        String blobName = s3Key.substring(0, s3Key.length() - ".properties".length()).substring((String.valueOf(this.getBucketPrefix()) + DIRECT_PATH_PREFIX).length() + 1);
        ImmutableMap headers = ImmutableMap.of((Object)"BlobStore.blob-name", (Object)blobName, (Object)"BlobStore.direct-path", (Object)"true");
        return this.blobIdLocationResolver.fromHeaders((Map)headers);
    }

    private static interface BlobIngester {
        public StreamMetrics ingestTo(String var1) throws IOException;
    }

    class S3Blob
    extends BlobSupport {
        S3Blob(BlobId blobId) {
            super(blobId);
        }

        public InputStream getInputStream() {
            S3Object object = S3BlobStore.this.s3.getObject(S3BlobStore.this.getConfiguredBucket(), S3BlobStore.this.contentPath(this.getId()));
            return object.getObjectContent();
        }
    }
}

